;;;
;;; auxsys - Auxiliary system interface
;;;  
;;;   Copyright (c) 2000-2007  Shiro Kawai  <shiro@acm.org>
;;;   
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;   
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;  
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;  
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;  
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;  
;;;  $Id: auxsys.scm,v 1.22 2007/09/29 07:41:34 shirok Exp $
;;;

(define-module gauche.auxsys
  (export fmod frexp modf ldexp
          sys-abort sys-realpath sys-mkfifo
          sys-fdset list->sys-fdset sys-fdset->list
          sys-setgid sys-setpgid sys-getpgid sys-getpgrp
          sys-setsid sys-setuid sys-times sys-uname sys-ctermid
          sys-gethostname sys-getdomainname
          sys-putenv sys-setenv sys-unsetenv sys-environ sys-environ->alist
          sys-chown sys-lchown sys-utime
          sys-getgroups sys-getlogin sys-localeconv
          sys-getloadavg)
  )
(select-module gauche.auxsys)

(dynamic-load "auxsys")

;; define alternatives if the platform doesn't support...

(cond-expand
 ((not gauche.sys.realpath)
  (define sys-realpath #f))             ; make autoload happy
 (else))

(cond-expand
 (gauche.os.windows
  (define sys-mkfifo #f)
  (define sys-setgid #f)
  (define sys-setpgid #f)
  (define sys-getpgid #f)
  (define sys-getpgrp #f)
  (define sys-setsid #f)
  (define sys-setuid #f)
  (define sys-getgroups #f)
  (define sys-uname #f)
  (define sys-chown #f)
  )
 (else))

(define sys-gethostname
  (if (global-variable-bound? 'gauche.auxsys '%sys-gethostname)
    %sys-gethostname
    (lambda ()
      (cond-expand
       ((not gauche.os.windows) (cadr (sys-uname))) ; nodename
       (else "localhost")))))   ; need better fallback

(define sys-getdomainname
  (if (global-variable-bound? 'gauche.auxsys '%sys-getdomainname)
    %sys-getdomainname
    (lambda () "localdomain"))) ; need better fallback

;; These are better to be in src/scmlib.scm, but right now we don't have
;; a nice way to make cond-expand work (when compiling src/scmlib.scm
;; cond-expand uses the host gosh's feature set, not the target gosh's.)
(cond-expand
 (gauche.sys.select
  (define (sys-fdset . pfs)
    (list->sys-fdset pfs))
  (define (sys-fdset->list fdset)
    (check-arg (cut is-a? <> <sys-fdset>) fdset)
    (do ((i (sys-fdset-max-fd fdset) (- i 1))
         (fds '() (if (sys-fdset-ref fdset i) (cons i fds) fds)))
        ((< i 0) fds)
      #f))
  (define (list->sys-fdset pfs)
    (let1 fdset (make <sys-fdset>)
      (dolist (pf pfs)
        (cond ((or (integer? pf) (port? pf))
               (sys-fdset-set! fdset pf #t))
              ((is-a? pf <sys-fdset>)
               (dotimes (i (+ (sys-fdset-max-fd pf) 1))
                 (when (sys-fdset-ref pf i)
                   (sys-fdset-set! fdset i #t))))
              (else (error "sys-fdset requires a port, an integer, or a <sys-fdset> object, but got:" pf))))
      fdset))    
  )
 (else
  ;; make autoload happy
  (define sys-fdset #f)
  (define sys-fdset->list #f)
  (define list->sys-fdset #f)
  ))

;; We support sys-setenv natively if the system has either
;; setenv(3) or putenv(3).  The feature symbol is gauche.sys.setenv.

(cond-expand
 (gauche.sys.setenv
  ;; We emulate putenv.  Somehow the old API was (sys-putenv name value),
  ;; which we support for backward compatibility.
  (define (sys-putenv name=value . other)
    (cond
     ((null? other)
      (check-arg string? name=value)
      (receive (name value) (string-scan name=value #\= 'both)
        (unless name
          (error "sys-putenv: argument doesn't contain '=':" name=value))
        (sys-setenv name value #t)))
     (else
      (sys-setenv name=value (car other) #t))))
  )
 (else
  ;; make autoload happy
  (define sys-putenv #t)
  (define sys-setenv #t)))

(cond-expand
 ((not gauche.sys.unsetenv) 
  (define sys-unsetenv #f))             ; make autoload happy
 (else))

(define (sys-environ->alist . envlist)
  (map (lambda (envstr)
         (receive (pre post) (string-scan envstr #\= 'both)
           (if pre (cons pre post) (cons envstr ""))))
       (get-optional envlist (sys-environ))))

(define sys-setpgrp
  (if (global-variable-bound? 'gauche.auxsys '%sys-setpgrp)
    %sys-setpgrp
    (lambda () (sys-setpgid 0 0))))

(cond-expand
 ((not gauche.sys.getpgid)
  (define sys-getpgid #f))              ;make autoload happy
 (else))

(cond-expand
 ((not gauche.sys.lchown)
  (define sys-lchown #f))                ;make autoload happy
 (else))

(cond-expand
 ((not gauche.sys.getloadavg)
  (define sys-getloadavg #f))           ;make autoload happy
 (else))

(provide "gauche/auxsys")
