/*
 * port.c - port implementation
 *
 *   Copyright (c) 2000-2004 Shiro Kawai, All rights reserved.
 * 
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 * 
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the authors nor the names of its contributors
 *      may be used to endorse or promote products derived from this
 *      software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  $Id: port.c,v 1.122 2005/10/13 08:14:13 shirok Exp $
 */

#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#define LIBGAUCHE_BODY
#include "gauche.h"
#include "gauche/class.h"
#include "gauche/port.h"

#undef MAX
#undef MIN
#define MAX(a, b) ((a)>(b)? (a) : (b))
#define MIN(a, b) ((a)<(b)? (a) : (b))

/*================================================================
 * Class stuff
 */

static void port_print(ScmObj obj, ScmPort *port, ScmWriteContext *ctx);
static void port_finalize(ScmObj obj, void* data);
static void register_buffered_port(ScmPort *port);
static void unregister_buffered_port(ScmPort *port);
static void bufport_flush(ScmPort*, int, int);
static void file_closer(ScmPort *p);

SCM_DEFINE_BASE_CLASS(Scm_PortClass,
                      ScmPort, /* instance type */
                      port_print, NULL, NULL, NULL, NULL);

static ScmClass *port_cpl[] = {
    SCM_CLASS_STATIC_PTR(Scm_PortClass),
    SCM_CLASS_STATIC_PTR(Scm_TopClass),
    NULL
};

SCM_DEFINE_BASE_CLASS(Scm_CodingAwarePortClass,
                      ScmPort, /* instance type */
                      port_print, NULL, NULL, NULL, port_cpl);

/*================================================================
 * Common
 */

/* Cleaning up:
 *   The underlying file descriptor/stream may be closed when the port
 *   is explicitly closed by close-port, or implicitly destroyed by
 *   garbage collector.  To keep consistency, Scheme ports should never
 *   share the same file descriptor.  However, C code and Scheme port
 *   may share the same file descriptor for efficiency (e.g. stdios).
 *   In such cases, it is C code's responsibility to destroy the port.
 */
static void port_cleanup(ScmPort *port)
{
    if (SCM_PORT_CLOSED_P(port)) return;
    switch (SCM_PORT_TYPE(port)) {
    case SCM_PORT_FILE:
        if (SCM_PORT_DIR(port) == SCM_PORT_OUTPUT
            && !SCM_PORT_ERROR_OCCURRED_P(port)) {
            bufport_flush(port, 0, TRUE);
        }
        if (port->ownerp && port->src.buf.closer) port->src.buf.closer(port);
        break;
    case SCM_PORT_PROC:
        if (port->src.vt.Close) port->src.vt.Close(port);
        break;
    default:
        break;
    }
    SCM_PORT_CLOSED_P(port) = TRUE;
    /* avoid unnecessary finalization */
    Scm_UnregisterFinalizer(SCM_OBJ(port));
}

/* called by GC */
static void port_finalize(ScmObj obj, void* data)
{
    port_cleanup(SCM_PORT(obj));
}

/*
 * Internal Constructor.
 *   If this port owns the underlying file descriptor/stream, 
 *   ownerp must be TRUE.
 */
static ScmPort *make_port(ScmClass *klass, int dir, int type)
{
    ScmPort *port;

    port = SCM_ALLOCATE(ScmPort, klass);
    SCM_SET_CLASS(port, klass);
    port->direction = dir;
    port->type = type;
    port->scrcnt = 0;
    port->ungotten = SCM_CHAR_INVALID;
    port->closed = FALSE;
    port->error = FALSE;
    port->ownerp = FALSE;
    port->flags = 0;
    port->name = SCM_FALSE;
    (void)SCM_INTERNAL_MUTEX_INIT(port->mutex);
    (void)SCM_INTERNAL_COND_INIT(port->cv);
    port->lockOwner = NULL;
    port->lockCount = 0;
    port->data = SCM_FALSE;
    port->line = 1;
    switch (type) {
    case SCM_PORT_FILE: /*FALLTHROUGH*/;
    case SCM_PORT_PROC:
        Scm_RegisterFinalizer(SCM_OBJ(port), port_finalize, NULL);
        break;
    default:
        break;
    }
    return port;
}

/*
 * Close
 */
void Scm_ClosePort(ScmPort *port)
{
    ScmVM *vm = Scm_VM();
    PORT_LOCK(port, vm);
    PORT_SAFE_CALL(port,
                   do {
                       if (!SCM_PORT_CLOSED_P(port)) {
                           port_cleanup(port);
                           if (SCM_PORT_TYPE(port) == SCM_PORT_FILE
                               && SCM_PORT_DIR(port) == SCM_PORT_OUTPUT) {
                               unregister_buffered_port(port);
                           }
                       }
                   } while (0));
    PORT_UNLOCK(port);
}

/*
 * External routine to access port exclusively
 */
static ScmObj with_port_locking_pre_thunk(ScmObj *args, int nargs, void *data)
{
    ScmPort *p = (ScmPort*)data;
    ScmVM *vm = Scm_VM();
    PORT_LOCK(p, vm);
    return SCM_UNDEFINED;
}

static ScmObj with_port_locking_post_thunk(ScmObj *args, int nargs, void *data)
{
    ScmPort *p = (ScmPort*)data;
    PORT_UNLOCK(p);
    return SCM_UNDEFINED;
}

ScmObj Scm_VMWithPortLocking(ScmPort *port, ScmObj closure)
{
    ScmObj before = Scm_MakeSubr(with_port_locking_pre_thunk, (void*)port,
                                 0, 0, SCM_FALSE);
    ScmObj after = Scm_MakeSubr(with_port_locking_post_thunk, (void*)port,
                                0, 0, SCM_FALSE);
    return Scm_VMDynamicWind(before, closure, after);
}

/*===============================================================
 * Getting information
 */
ScmObj Scm_PortName(ScmPort *port)
{
    return port->name;
}

int Scm_PortLine(ScmPort *port)
{
    return port->line;
}

static void port_print(ScmObj obj, ScmPort *port, ScmWriteContext *ctx)
{
    Scm_Printf(port, "#<%s%sport%s %A %p>",
               (SCM_PORT_DIR(obj)&SCM_PORT_INPUT)? "i" : "",
               (SCM_PORT_DIR(obj)&SCM_PORT_OUTPUT)? "o" : "",
               SCM_PORT_CLOSED_P(obj)? "(closed)" : "",
               Scm_PortName(SCM_PORT(obj)),
               obj);
}

/* Returns port's associated file descriptor number, if any.
   Returns -1 otherwise. */
int Scm_PortFileNo(ScmPort *port)
{
    if (SCM_PORT_TYPE(port) == SCM_PORT_FILE) {
        if (port->src.buf.filenum) return port->src.buf.filenum(port);
        else return -1;
    } else {
        /* TODO: proc port */
        return -1;
    }
}

/* Low-level function to find if the file descriptor is ready or not.
   DIR specifies SCM_PORT_INPUT or SCM_PORT_OUTPUT.
   If the system doesn't have select(), this function returns
   SCM_FD_UNKNOWN. */
int Scm_FdReady(int fd, int dir)
{
#ifdef HAVE_SELECT
    fd_set fds;
    int r;
    struct timeval tm;

    /* In case if this is called on non-file ports.*/
    if (fd < 0) return SCM_FD_READY;

    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    tm.tv_sec = tm.tv_usec = 0;
    if (dir == SCM_PORT_OUTPUT) {
        SCM_SYSCALL(r, select(fd+1, NULL, &fds, NULL, &tm));
    } else {
        SCM_SYSCALL(r, select(fd+1, &fds, NULL, NULL, &tm));
    }
    if (r < 0) Scm_SysError("select failed");
    if (r > 0) return SCM_FD_READY;
    else       return SCM_FD_WOULDBLOCK;
#else  /*!HAVE_SELECT*/
    return SCM_FD_UNKNOWN;
#endif /*!HAVE_SELECT*/
}

/*===============================================================
 * buffered Port
 *  - mainly used for buffered file I/O, but can also be used
 *    for other purpose, like character-code conversion port.
 */

/* [Buffered port protocol]
 *
 *  Legends
 *    b = port->src.buf.buffer
 *    c = port->src.buf.current
 *    e = port->src.buf.end
 *    '*' = valid data
 *    '-' = invalid data
 *
 *  Output
 *
 *    When used as output, the end pointer always points one byte past
 *    the buffer.  Initially the buffer is empty and the current pointer
 *    is the same as the beginning of the buffer.
 *
 *    port->src.buf.flusher(ScmPort* p, int cnt, int forcep) is called when
 *    the port needs to create some room in the buffer.   When the flusher
 *    is called, the buffer is like this:
 *
 *        <--------------- size ---------------->
 *       |*********************************-----|
 *        ^                                ^     ^
 *        b                                c     e
 *
 *    The flusher is supposed to output the cnt bytes of data beginning from
 *    the buffer, which is usually up to the current pointer (but the flusher
 *    doesn't need to check the current pointer; it is taken care of by the
 *    caller of the flusher). 
 *
 *    If the third argument forcep is false, the flusher may return before
 *    entire data is output, in case like underlying device is busy.
 *    The flusher must output at least one byte even in that case.
 *    On the other hand, if the forcep argument is true, the flusher must
 *    write cnt bytes; if it is not possible, the flusher must return -1 to
 *    indicate an error(*1).
 *
 *    The flusher returns the number of bytes actually written out.
 *    If an error occurs, the flusher must return -1.
 *
 *    The flusher must be aware that the port p is locked by the current
 *    thread when called.
 *
 *    The flusher shouldn't change the buffer's internal state.
 *
 *    After the flusher returns, bufport_flush shifts the unflushed data
 *    (if any), so the buffer becomes like this:
 *
 *        <--------------- size ---------------->
 *       |****----------------------------------|
 *        ^   ^                                  ^
 *        b   c                                  e
 *
 *    (*1) Why should these two mode need to be distinguished?  Suppose
 *    you implement a buffered port that does character encoding conversion.
 *    The flusher converts the content of the buffer to different character
 *    encoding and feed it to some specified port.  It is often the case
 *    that you find a few bytes at the end of the buffer which you can't
 *    convert into a whole character but have to wait for next byte(s).
 *    It is valid that you leave them in the buffer if you can expect
 *    more data to come.  However, if you know it is really the end of
 *    the stream, you can't leave any data in the buffer and you should
 *    take appropriate action, for example, raising an error.
 *
 *  Input
 *
 *    When used as input, the end pointer points to one byte past the
 *    end of the valid data, which may be before the end of the buffer.
 *
 *    port->src.buf.filler(ScmPort *p, int cnt) is called when the buffer
 *    doesn't have enough data to read.   Suppose the input routine detects
 *    the buffer doesn't have enough data when it looks like this:
 *
 *        <--------------- size ---------------->
 *       |-----------------------------****-----|
 *        ^                            ^   ^
 *        b                            c   e
 *
 *    First, bufport_fill shifts the unread data (if any) to the beginning
 *    of the buffer, so it becomes like this:
 *
 *        <--------------- size ---------------->
 *       |****----------------------------------|
 *        ^   ^ 
 *        bc  e
 *
 *    Then port->src.buf.filler is called.  It is supposed to read as many
 *    bytes as cnt, putting them after the end pointer.   It may read
 *    less if all cnt bytes of data is not available immediately.
 *    The filler returns the number of bytes actually read in.
 *    The filler should return 0 if it reaches the end of the data source.
 *    If an error occurs, the filler must return -1.
 *
 *    bufport_fill then adjust the end pointer, so the buffer becomes like
 *    this.
 *
 *        <--------------- size ---------------->
 *       |************************************--|
 *        ^                                   ^ 
 *        bc                                  e
 *
 *  Close
 *    Port is closed either explicitly (via close-port etc) or implicity
 *    (via GC -> finalizer).   In either case, the flusher is called first
 *    if there's any data remaining in the buffer.   Then, if the closer
 *    procedure (port->src.buf.closer) is not NULL, and port->owner is TRUE,
 *    the closer procedure is called which has to take care of any system-
 *    level cleanup.   The closer can assume the buffer is already flushed.
 *
 *  Ready
 *    When char-ready? is called on a buffered port, it first checks if
 *    there's any data available in the buffer.  If so, it returns true.
 *    If not, it calls port->src.buf.ready if it is not NULL to query
 *    the character is ready.   If port->src.buf.ready is NULL, bufport
 *    assumes the input is always ready.
 *    port->src.buf.ready should return either SCM_FD_READY, SCM_FD_WOULDBLOCK
 *    or SCM_FD_UNKNOWN.
 *
 *  Filenum
 *    Port->src.buf.filenum is a query procedure that should return the
 *    underlying integer file descriptor of the port, or -1 if there's
 *    no associated one.   If it is NULL, the port is assumed not to
 *    be associated to any file descriptor.
 *
 *  Buffering mode
 *    {For Output}
 *      SCM_PORT_BUFFER_FULL : Full buffering.  The buffer is flushed
 *         only when the buffer gets full, explicitly requested, or
 *         closed.   This is the default, and suitable for file I/O.
 *
 *      SCM_PORT_BUFFER_LINE : Line buffering.  The buffer is flushed
 *         when a newline character is put, other than the normal
 *         circumstances as in SCM_PORT_BUFFER_FULL.   Unlike C stdio,
 *         the buffer isn't flushed when an input is called on the same
 *         terminal device.
 *         This is natural for output of interactive communication.
 *         This is the default of stdout.
 *
 *      SCM_PORT_BUFFER_NONE : data is always passed to the flusher
 *         procedure.  The buffer is used just as a temporary storage.
 *         This slows down port operation significantly.  Should only
 *         be used when you want to guarantee what you write is always
 *         passed to the lower layer.   This is the default of stderr.
 *
 *    {For Input}
 *      SCM_PORT_BUFFER_FULL : Full buffering.  The filler procedure
 *         is called only if the buffer doesn't have enough data to
 *         satisfy the read request.   Read-block or read-string won't
 *         return until the specified bytes/characters are read from
 *         the port, except the port reaches EOF.
 *
 *      SCM_PORT_BUFFER_LINE : For input ports, this is almost the same
 *         as BUFFER_FULL, except that read-block and read-string may
 *         return shorter data than requested, if only that amount of
 *         data is immediately available.   Usually this mode is suitable
 *         for the ports that is attached to a pipe or network.
 *
 *      SCM_PORT_BUFFER_NONE : No buffering.  Every time the data is
 *         requested, the filler procedure is called with exact amount
 *         of the requested data.
 */

#define SCM_PORT_DEFAULT_BUFSIZ 8192

ScmObj Scm_MakeBufferedPort(ScmClass *klass,
                            ScmObj name,
                            int dir,     /* direction */
                            int ownerp,  /* owner flag*/
                            ScmPortBuffer *bufrec)
{
    ScmPort *p;
    int size = bufrec->size;
    char *buf = bufrec->buffer;
    
    if (size <= 0) size = SCM_PORT_DEFAULT_BUFSIZ;
    if (buf == NULL) buf = SCM_NEW_ATOMIC2(char*, size);
    p = make_port(klass, dir, SCM_PORT_FILE);
    p->name = name;
    p->ownerp = ownerp;
    p->src.buf.buffer = buf;
    if (dir == SCM_PORT_INPUT) {
        p->src.buf.current = p->src.buf.buffer;
        p->src.buf.end = p->src.buf.buffer;
    } else {
        p->src.buf.current = p->src.buf.buffer;
        p->src.buf.end = p->src.buf.buffer + size;
    }
    p->src.buf.size = size;
    p->src.buf.mode = bufrec->mode;
    p->src.buf.filler = bufrec->filler;
    p->src.buf.flusher = bufrec->flusher;
    p->src.buf.closer = bufrec->closer;
    p->src.buf.ready = bufrec->ready;
    p->src.buf.filenum = bufrec->filenum;
    p->src.buf.seeker = bufrec->seeker;
    p->src.buf.data = bufrec->data;
    if (dir == SCM_PORT_OUTPUT) register_buffered_port(p);
    return SCM_OBJ(p);
}

/* flushes the buffer, to make a room of cnt bytes.
   cnt == 0 means all the available data.   Note that, unless forcep == TRUE,
   this function only does "best effort" to make room, but doesn't
   guarantee to output cnt bytes.  */
static void bufport_flush(ScmPort *p, int cnt, int forcep)
{
    int cursiz = SCM_PORT_BUFFER_AVAIL(p);
    int nwrote, force = FALSE;
    
    if (cursiz == 0) return;
    if (cnt <= 0)  { cnt = cursiz; force = TRUE; }
    nwrote = p->src.buf.flusher(p, cnt, forcep);
    if (nwrote < 0) {
        p->src.buf.current = p->src.buf.buffer; /* for safety */
        p->error = TRUE;
        /* TODO: can we raise an error here, or should we propagate
           it to the caller? */
        Scm_PortError(p, SCM_PORT_ERROR_OUTPUT,
                      "Couldn't flush port %S due to an error", p);
    }
    if (nwrote >= 0 && nwrote < cursiz) {
        memmove(p->src.buf.buffer, p->src.buf.buffer+nwrote,
                cursiz-nwrote);
        p->src.buf.current -= nwrote;
    } else {
        p->src.buf.current = p->src.buf.buffer;
    }
}

/* Writes siz bytes in src to the buffered port.  siz may be larger than
   the port's buffer.  Won't return until entire siz bytes are written. */
static void bufport_write(ScmPort *p, const char *src, int siz)
{
    do {
        int room = (int)(p->src.buf.end - p->src.buf.current);
        if (room >= siz) {
            memcpy(p->src.buf.current, src, siz);
            p->src.buf.current += siz;
            siz = 0;
        } else {
            memcpy(p->src.buf.current, src, room);
            p->src.buf.current += room;
            siz -= room;
            src += room;
            bufport_flush(p, 0, FALSE);
        }
    } while (siz > 0);
}

/* Fills the buffer.  Reads at least MIN bytes (unless it reaches EOF).
 * If ALLOW_LESS is true, however, we allow to return before the full
 * data is read.
 * Returns the number of bytes actually read, or 0 if EOF, or -1 if error.
 */
static int bufport_fill(ScmPort *p, int min, int allow_less)
{
    int cursiz = (int)(p->src.buf.end - p->src.buf.current);
    int nread = 0, toread;
    if (cursiz > 0) {
        memmove(p->src.buf.buffer, p->src.buf.current, cursiz);
        p->src.buf.current = p->src.buf.buffer;
        p->src.buf.end = p->src.buf.current + cursiz;
    } else {
        p->src.buf.current = p->src.buf.end = p->src.buf.buffer;
    }
    if (min <= 0) min = SCM_PORT_BUFFER_ROOM(p);
    if (p->src.buf.mode != SCM_PORT_BUFFER_NONE) {
        toread = SCM_PORT_BUFFER_ROOM(p);
    } else {
        toread = min;
    }

    do {
        int r = p->src.buf.filler(p, toread-nread);
        if (r <= 0) break;
        nread += r;
        p->src.buf.end += r;
    } while (!allow_less && nread < min);
    return nread;
}

/* Reads siz bytes to dst from the buffered port.  siz may be larger
 * than the port's buffer, in which case the filler procedure is called
 * more than once.  Unless the port buffering mode is BUFFER_FULL,
 * this may read less than SIZ bytes if only that amount of data is
 * immediately available.
 * Caveat: if the filler procedure returns N where 0 < N < requested size,
 * we know less data is available; non-greedy read can return at that point.
 * However, if the filler procedure returns exactly the requested size,
 * and we need more bytes, we gotta be careful -- next call to the filler
 * procedure may or may not hang.  So we need to check the ready procedure.
 */
static int bufport_read(ScmPort *p, char *dst, int siz)
{
    int nread = 0, r, req;
    int avail = (int)(p->src.buf.end - p->src.buf.current);

    req = MIN(siz, avail);
    if (req > 0) {
        memcpy(dst, p->src.buf.current, req);
        p->src.buf.current += req;
        nread += req;
        siz -= req;
        dst += req;
    }
    while (siz > 0) {
        req = MIN(siz, p->src.buf.size);
        r = bufport_fill(p, req, TRUE);
        if (r <= 0) break; /* EOF or an error*/
        if (r >= siz) {
            memcpy(dst, p->src.buf.current, siz);
            p->src.buf.current += siz;
            nread += siz;
            break;
        } else {
            memcpy(dst, p->src.buf.current, r);
            p->src.buf.current += r;
            nread += r;
            siz -= r;
            dst += r;
        }
        if (p->src.buf.mode != SCM_PORT_BUFFER_FULL) {
            if (r < req) break;
            if (p->src.buf.ready
                && p->src.buf.ready(p) == SCM_FD_WOULDBLOCK) {
                break;
            }
        }
    }
    return nread;
}

/* Tracking buffered ports:
 *   The system doesn't automatically flush the buffered output port,
 *   as it does on FILE* structure.  So Gauche keeps track of active
 *   output buffered ports, in a weak vector.
 *   When the port is no longer used, it is collected by GC and removed
 *   from the vector.   Scm_FlushAllPorts() flushes the active ports.
 */

/*TODO: allow to extend the port vector. */

#define PORT_VECTOR_SIZE 256    /* need to be 2^n */

static struct {
    int dummy;
    ScmWeakVector   *ports;
    ScmInternalMutex mutex;
} active_buffered_ports = { 1, NULL }; /* magic to put this in .data area */

#define PORT_HASH(port)  \
    ((((SCM_WORD(port)>>3) * 2654435761UL)>>16) % PORT_VECTOR_SIZE)

static void register_buffered_port(ScmPort *port)
{
    int i, h, c;
    h = i = PORT_HASH(port);
    c = 0;
    /* search the available entry by quadratic hash */
    (void)SCM_INTERNAL_MUTEX_LOCK(active_buffered_ports.mutex);
    while (!SCM_FALSEP(Scm_WeakVectorRef(active_buffered_ports.ports, i, SCM_FALSE))) {
        i -= ++c; if (i<0) i+=PORT_VECTOR_SIZE;
        if (i == h) Scm_Panic("active buffered port table overflow");
    }
    Scm_WeakVectorSet(active_buffered_ports.ports, i, SCM_OBJ(port));
    (void)SCM_INTERNAL_MUTEX_UNLOCK(active_buffered_ports.mutex);
}

/* This should be called when the output buffered port is explicitly closed.
   The ports collected by GC are automatically unregistered. */
static void unregister_buffered_port(ScmPort *port)
{
    int i, h, c;
    ScmObj p;
    
    h = i = PORT_HASH(port);
    c = 0;
    (void)SCM_INTERNAL_MUTEX_LOCK(active_buffered_ports.mutex);
    do {
        p = Scm_WeakVectorRef(active_buffered_ports.ports, i, SCM_FALSE);
        if (!SCM_FALSEP(p) && SCM_EQ(SCM_OBJ(port), p)) {
            Scm_WeakVectorSet(active_buffered_ports.ports, i, SCM_FALSE);
            break;
        }
        i -= ++c; if (i<0) i+=PORT_VECTOR_SIZE;
    } while (i != h);
    (void)SCM_INTERNAL_MUTEX_UNLOCK(active_buffered_ports.mutex);
}

/* Flush all ports.  Note that it is possible that this routine can be
   called recursively if one of the flushing routine calls Scm_Exit.
   In order to avoid infinite loop, I have to delete the entries of already
   flushed port before calling flush, then recover them before return
   (unless exitting is true, in that case we know nobody cares the active
   port vector anymore).
   Even if more than one thread calls Scm_FlushAllPorts simultaneously,
   the flush method is called only once, from one of the calling thread.
 */
void Scm_FlushAllPorts(int exitting)
{
    ScmWeakVector *save, *ports;
    ScmObj p = SCM_FALSE;
    int i, saved = 0;

    save = SCM_WEAK_VECTOR(Scm_MakeWeakVector(PORT_VECTOR_SIZE));
    ports = active_buffered_ports.ports;
    
    for (i=0; i<PORT_VECTOR_SIZE;) {
        (void)SCM_INTERNAL_MUTEX_LOCK(active_buffered_ports.mutex);
        for (; i<PORT_VECTOR_SIZE; i++) {
            p = Scm_WeakVectorRef(ports, i, SCM_FALSE);
            if (!SCM_FALSEP(p)) {
                Scm_WeakVectorSet(save, i, p);
                Scm_WeakVectorSet(ports, i, SCM_FALSE);
                saved++;
                break;
            }
        }
        (void)SCM_INTERNAL_MUTEX_UNLOCK(active_buffered_ports.mutex);
        if (!SCM_FALSEP(p)) {
            SCM_ASSERT(SCM_PORTP(p) && SCM_PORT_TYPE(p)==SCM_PORT_FILE);
            if (!SCM_PORT_ERROR_OCCURRED_P(SCM_PORT(p))) {
                bufport_flush(SCM_PORT(p), 0, TRUE);
            }
        }
    }
    if (!exitting && saved) {
        (void)SCM_INTERNAL_MUTEX_LOCK(active_buffered_ports.mutex);
        for (i=0; i<PORT_VECTOR_SIZE; i++) {
            p = Scm_WeakVectorRef(save, i, SCM_FALSE);
            if (!SCM_FALSEP(p)) Scm_WeakVectorSet(ports, i, p);
        }
        (void)SCM_INTERNAL_MUTEX_UNLOCK(active_buffered_ports.mutex);
    }
}

/* Utility procedure to translate Scheme arg into buffering mode */
static ScmObj key_full   = SCM_UNBOUND;
static ScmObj key_modest = SCM_UNBOUND;
static ScmObj key_line   = SCM_UNBOUND;
static ScmObj key_none   = SCM_UNBOUND;

int Scm_BufferingMode(ScmObj flag, int direction, int fallback)
{
    if (SCM_EQ(flag, key_full)) return SCM_PORT_BUFFER_FULL;
    if (SCM_EQ(flag, key_none)) return SCM_PORT_BUFFER_NONE;
    if (fallback >= 0 && (SCM_UNBOUNDP(flag) || SCM_FALSEP(flag)))
        return fallback;
    if (direction == SCM_PORT_INPUT) {
        if (SCM_EQ(flag, key_modest)) return SCM_PORT_BUFFER_LINE;
        else Scm_Error("buffering mode must be one of :full, :modest or :none, but got %S", flag);
    }
    if (direction == SCM_PORT_OUTPUT) {
        if (SCM_EQ(flag, key_line)) return SCM_PORT_BUFFER_LINE;
        else Scm_Error("buffering mode must be one of :full, :line or :none, but got %S", flag);
    }
    /* if direction is none of input or output, allow both. */
    if (SCM_EQ(flag, key_line) || SCM_EQ(flag, key_modest)) {
        return SCM_PORT_BUFFER_LINE;
    }
    else Scm_Error("buffering mode must be one of :full, :modest, :line or :none, but got %S", flag);
    return -1;                  /* dummy */
}

ScmObj Scm_GetBufferingMode(ScmPort *port)
{
    if (SCM_PORT_TYPE(port) == SCM_PORT_FILE) {
        switch (port->src.buf.mode) {
        case SCM_PORT_BUFFER_FULL: return key_full;
        case SCM_PORT_BUFFER_NONE: return key_none;
        default:
            if (SCM_IPORTP(port)) return key_modest;
            else return key_line;
        }
    }
    return SCM_FALSE;
}

/*===============================================================
 * Generic procedures
 */

#define SAFE_PORT_OP
#include "portapi.c"
#undef SAFE_PORT_OP
#include "portapi.c"

/*===============================================================
 * File Port
 */

static int file_filler(ScmPort *p, int cnt)
{
    int nread = 0, r;
    int fd = (int)p->src.buf.data;
    char *datptr = p->src.buf.end;
    SCM_ASSERT(fd >= 0);
    while (nread == 0) {
        errno = 0;
        SCM_SYSCALL(r, read(fd, datptr, cnt-nread));
        if (r < 0) {
            p->error = TRUE;
            Scm_SysError("read failed on %S", p);
        } else if (r == 0) {
            /* EOF is read */
            break;
        } else {
            datptr += r;
            nread += r;
        }
    }
    return nread;
}

static int file_flusher(ScmPort *p, int cnt, int forcep)
{
    int nwrote = 0, r;
    int datsiz = SCM_PORT_BUFFER_AVAIL(p);
    int fd = (int)p->src.buf.data;
    char *datptr = p->src.buf.buffer;
    
    SCM_ASSERT(fd >= 0);
    while ((!forcep && nwrote == 0)
           || (forcep && nwrote < cnt)) {
        errno = 0;
        SCM_SYSCALL(r, write(fd, datptr, datsiz-nwrote));
        if (r < 0) {
            p->error = TRUE;
            Scm_SysError("write failed on %S", p);
        } else {
            datptr += r;
            nwrote += r;
        }
    }
    return nwrote;
}

static void file_closer(ScmPort *p)
{
    int fd = (int)p->src.buf.data;
    SCM_ASSERT(fd >= 0);
    close(fd);
}

static int file_ready(ScmPort *p)
{
    int fd = (int)p->src.buf.data;
    SCM_ASSERT(fd >= 0);
    return Scm_FdReady(fd, SCM_PORT_DIR(p));
}

static int file_filenum(ScmPort *p)
{
    return (int)p->src.buf.data;
}

static off_t file_seeker(ScmPort *p, off_t offset, int whence)
{
    return lseek((int)p->src.buf.data, offset, whence);
}

ScmObj Scm_OpenFilePort(const char *path, int flags, int buffering, int perm)
{
    int fd, dir = 0;
    ScmObj p;
    ScmPortBuffer bufrec;
    
    if ((flags & O_ACCMODE) == O_RDONLY) dir = SCM_PORT_INPUT;
    else if ((flags & O_ACCMODE) == O_WRONLY) dir = SCM_PORT_OUTPUT;
    else Scm_Error("unsupported file access mode %d to open %s", flags&O_ACCMODE, path);
    if (buffering < SCM_PORT_BUFFER_FULL || buffering > SCM_PORT_BUFFER_NONE) {
        Scm_Error("bad buffering flag: %d", buffering);
    }
#if defined(__MINGW32__)
    /* Force binary mode if not specified */
    if (!(flags & (O_TEXT|O_BINARY))) {
	flags |= O_BINARY;
    }
#endif /*__MINGW32__*/
    fd = open(path, flags, perm);
    if (fd < 0) return SCM_FALSE;
    bufrec.mode = buffering;
    bufrec.buffer = NULL;
    bufrec.size = 0;
    bufrec.filler = file_filler;
    bufrec.flusher = file_flusher;
    bufrec.closer = file_closer;
    bufrec.ready = file_ready;
    bufrec.filenum = file_filenum;
    bufrec.seeker = file_seeker;
    bufrec.data = (void*)fd;
    p = Scm_MakeBufferedPort(SCM_CLASS_PORT, SCM_MAKE_STR_COPYING(path),
                             dir, TRUE, &bufrec);
    return p;
}

/* Create a port on specified file descriptor.
      NAME  - used for the name of the port.
      DIRECTION - either SCM_PORT_INPUT or SCM_PORT_OUTPUT
      FD - the opened file descriptor.
      BUFMODE - buffering mode (ScmPortBufferMode)
      OWNERP - if TRUE, fd will be closed when this port is closed.
 */
ScmObj Scm_MakePortWithFd(ScmObj name, int direction,
                          int fd, int bufmode, int ownerp)
{
    ScmObj p;
    ScmPortBuffer bufrec;
    
    bufrec.buffer = NULL;
    bufrec.size = 0;
    bufrec.mode = bufmode;
    bufrec.filler = file_filler;
    bufrec.flusher =file_flusher;
    bufrec.closer = file_closer;
    bufrec.ready = file_ready;
    bufrec.filenum = file_filenum;
    bufrec.seeker = NULL;
    bufrec.data = (void*)fd;
    
    p = Scm_MakeBufferedPort(SCM_CLASS_PORT, name, direction, ownerp, &bufrec);
    return p;
}

/*===============================================================
 * String port
 */

ScmObj Scm_MakeInputStringPort(ScmString *str, int privatep)
{
    ScmPort *p = make_port(SCM_CLASS_PORT, SCM_PORT_INPUT, SCM_PORT_ISTR);
    u_int size;
    const char *s = Scm_GetStringContent(str, &size, NULL, NULL);
    p->src.istr.start = s;
    p->src.istr.current = s;
    p->src.istr.end = s + size;
    SCM_PORT(p)->name = SCM_MAKE_STR("(input string port)");
    if (privatep) PORT_PRELOCK(p, Scm_VM());
    return SCM_OBJ(p);
}

ScmObj Scm_MakeOutputStringPort(int privatep)
{
    ScmPort *p = make_port(SCM_CLASS_PORT, SCM_PORT_OUTPUT, SCM_PORT_OSTR);
    Scm_DStringInit(&p->src.ostr);
    SCM_PORT(p)->name = SCM_MAKE_STR("(output string port)");
    if (privatep) PORT_PRELOCK(p, Scm_VM());
    return SCM_OBJ(p);
}

ScmObj Scm_GetOutputString(ScmPort *port)
{
    ScmObj r;
    ScmVM *vm;
    if (SCM_PORT_TYPE(port) != SCM_PORT_OSTR)
        Scm_Error("output string port required, but got %S", port);
    vm = Scm_VM();
    PORT_LOCK(port, vm);
    r = Scm_DStringGet(&SCM_PORT(port)->src.ostr, 0);
    PORT_UNLOCK(port);
    return r;
}

ScmObj Scm_GetOutputStringUnsafe(ScmPort *port)
{
    if (SCM_PORT_TYPE(port) != SCM_PORT_OSTR)
        Scm_Error("output string port required, but got %S", port);
    return Scm_DStringGet(&SCM_PORT(port)->src.ostr, 0);
}

ScmObj Scm_GetRemainingInputString(ScmPort *port)
{
    const char *cp, *ep;
    if (SCM_PORT_TYPE(port) != SCM_PORT_ISTR)
        Scm_Error("input string port required, but got %S", port);
    /* NB: we don't need to lock the port, since the string body
       the port is pointing won't be changed */
    ep = port->src.istr.end;
    cp = port->src.istr.current;
    return Scm_MakeString(cp, ep-cp, -1, 0);
}

/*===============================================================
 * Procedural port
 */

/* To create a procedural port, fill in the ScmPortVTable function
   pointers and pass it to Scm_MakeVirutalPort.  You don't need to
   provide all the functions; put NULL if you think you don't
   provide the functionality.
*/

/* default dummy procedures */
static int null_getb(ScmPort *dummy)
    /*ARGSUSED*/
{
    return SCM_CHAR_INVALID;
}

static int null_getc(ScmPort *dummy)
    /*ARGSUSED*/
{
    return SCM_CHAR_INVALID;
}

static int null_getz(char *buf, int buflen, ScmPort *dummy)
    /*ARGSUSED*/
{
    return 0;
}

static int null_ready(ScmPort *dummy, int charp)
    /*ARGSUSED*/
{
    return TRUE;
}

static void null_putb(ScmByte b, ScmPort *dummy)
    /*ARGSUSED*/
{
}

static void null_putc(ScmChar c, ScmPort *dummy)
    /*ARGSUSED*/
{
}

static void null_putz(const char *str, int len, ScmPort *dummy)
    /*ARGSUSED*/
{
}

static void null_puts(ScmString *s, ScmPort *dummy)
    /*ARGSUSED*/
{
}

static void null_flush(ScmPort *dummy)
    /*ARGSUSED*/
{
}

ScmObj Scm_MakeVirtualPort(ScmClass *klass, int direction,
                           ScmPortVTable *vtable)
{
    ScmPort *p = make_port(klass, direction, SCM_PORT_PROC);
    
    /* Copy vtable, and ensure all entries contain some ptr */
    p->src.vt = *vtable;
    if (!p->src.vt.Getb)  p->src.vt.Getb = null_getb;
    if (!p->src.vt.Getc)  p->src.vt.Getc = null_getc;
    if (!p->src.vt.Getz)  p->src.vt.Getz = null_getz;
    if (!p->src.vt.Ready) p->src.vt.Ready = null_ready;
    if (!p->src.vt.Putb)  p->src.vt.Putb = null_putb;
    if (!p->src.vt.Putc)  p->src.vt.Putc = null_putc;
    if (!p->src.vt.Putz)  p->src.vt.Putz = null_putz;
    if (!p->src.vt.Puts)  p->src.vt.Puts = null_puts;
    if (!p->src.vt.Flush) p->src.vt.Flush = null_flush;
    /* Close and Seek can be left NULL */
    return SCM_OBJ(p);
}

/*===============================================================
 * Coding-aware port
 */

/* Coding-aware port wraps an input port, and specifically recognizes
   'coding' magic comment.   It is primarily used when loading source
   code, but can be used separately. */

/* gauche.charconv sets the pointer */
ScmPort *(*Scm_CodingAwarePortHook)(ScmPort *src,
                                    const char *srcencoding)
    = NULL;

#define CODING_MAGIC_COMMENT_LINES 2 /* maximum number of lines to be
                                        looked at for the 'encoding' magic
                                        comment. */

typedef struct coding_port_data_rec {
    ScmPort *source;            /* source port */
    int state;                  /* port state; see below */
    const char *pbuf;           /* prefetched buffer.  NUL terminated.
                                   contains at most CODING_MAGIC_COMMENT_LINES
                                   newlines. */
    int pbufsize;               /* # of bytes in pbuf */
} coding_port_data;

enum {
    CODING_PORT_INIT,           /* initial state */
    CODING_PORT_RECOGNIZED,     /* prefetched up to two lines, and
                                   conversion port is set if necessary.
                                   there are buffered data in lines[]. */
    CODING_PORT_FLUSHED         /* prefetched lines are flushed. */
};

/* A hardcoded DFA to recognize #/;.*coding[:=]\s*([\w.-]+)/ */
static const char *look_for_encoding(const char *buf)
{
    const char *s;
    char *encoding;
    
  init:
    for (;;) {
        switch (*buf++) {
        case '\0': return NULL;
        case ';':  goto comment;
        }
    }
  comment:
    for (;;) {
        switch (*buf++) {
        case '\0': return NULL;
        case '\n': goto init;
        case '\r': if (*buf != '\n') goto init; break;
        case 'c' : goto coding;
        }
    }
  coding:
    if (strncmp(buf, "oding", 5) != 0) goto comment;
    buf+=5;
    if (*buf != ':' && *buf != '=') goto comment;
    for (buf++;;buf++) {
        if (*buf != ' ' && *buf != '\t') break;
    }
    if (*buf == '\0') return NULL;

    for (s = buf;*buf;buf++) {
        if (!isalnum(*buf) && *buf != '_' && *buf != '-' && *buf != '.') {
            break;
        }
    }
    if (s == buf) goto comment;

    /* Here we found a matching string, starting from s and ends at buf. */

    /* kludge: Emacs uses special suffix #/-(unix|dos|mac)$/ to distinguish
       EOL variants.  For compatibility, drop such suffix if we have one. */
    if (buf-s > 5 && (strncmp(buf-5, "-unix", 5) == 0)) {
        buf -= 5;
    } else if (buf-s > 4 && (strncmp(buf-4, "-dos", 4) == 0
                             || strncmp(buf-4, "-mac", 4) == 0)) {
        buf -= 4;
    }

    /* Copy and return the encoding string */
    encoding = SCM_NEW_ATOMIC2(char*, buf-s+1);
    memcpy(encoding, s, buf-s);
    encoding[buf-s] = '\0';
    return encoding;
}

static void coding_port_recognize_encoding(ScmPort *port,
                                           coding_port_data *data)
{
    ScmDString ds;
    int num_newlines = 0, c;
    int cr_seen = FALSE;
    const char *encoding = NULL;

    SCM_ASSERT(data->source != NULL);

    /* Prefetch up to CODING_MAGIC_COMMENT_LINES lines or the first NUL
       character.   data->pbuf ends up holding NUL terminated string. */
    Scm_DStringInit(&ds);
    for (;;) {
        c = Scm_GetbUnsafe(data->source);
        if (c == EOF) break;
        if (c == 0) {
            /* take extra care not to lose '\0' */
            Scm_UngetbUnsafe(c, data->source);
            break;
        }
        SCM_DSTRING_PUTB(&ds, c);
        if (c == '\r') {   /* for the source that only uses '\r' */
            cr_seen = TRUE;
        } else if (c == '\n' || cr_seen) {
            if (++num_newlines >= CODING_MAGIC_COMMENT_LINES) {
                break;
            }
        } else {
            cr_seen = FALSE;
        }
    }
    data->pbuf = Scm_DStringGetz(&ds);
    data->pbufsize = strlen(data->pbuf);
    
    /* Look for the magic comment */
    encoding = look_for_encoding(data->pbuf);

    /* Wrap the source port by conversion port, if necessary. */
    if (encoding == NULL || Scm_SupportedCharacterEncodingP(encoding)) {
        return;
    }

    if (Scm_CodingAwarePortHook == NULL) {
        /* Require gauche.charconv.
           NB: we don't need mutex here, for loading the module is
           serialized in Scm_Require. */
        Scm_Require(SCM_MAKE_STR("gauche/charconv"));
        if (Scm_CodingAwarePortHook == NULL) {
            Scm_PortError(port, SCM_PORT_ERROR_OTHER,
                          "couldn't load gauche.charconv module");
        }
    }
    data->source = Scm_CodingAwarePortHook(data->source, encoding);
}

static int coding_filler(ScmPort *p, int cnt)
{
    int nread = 0;
    coding_port_data *data = (coding_port_data*)p->src.buf.data;
    char *datptr = p->src.buf.end;

    SCM_ASSERT(data->source);

    /* deals with the most frequent case */
    if (data->state == CODING_PORT_FLUSHED) {
        return Scm_GetzUnsafe(datptr, cnt, data->source);
    }
    
    if (data->state == CODING_PORT_INIT) {
        coding_port_recognize_encoding(p, data);
        data->state = CODING_PORT_RECOGNIZED;
    }

    /* Here, we have data->state == CODING_PORT_RECOGNIZED */
    if (data->pbufsize > 0) {
        if (data->pbufsize <= cnt) {
            memcpy(datptr, data->pbuf, data->pbufsize);
            nread = data->pbufsize;
            data->pbuf = NULL;
            data->pbufsize = 0;
            data->state = CODING_PORT_FLUSHED;
        } else {
            memcpy(datptr, data->pbuf, cnt);
            nread = cnt;
            data->pbuf += cnt;
            data->pbufsize -= cnt;
        }
        return nread;
    } else {
        data->state = CODING_PORT_FLUSHED;
        return Scm_GetzUnsafe(datptr, cnt, data->source);
    }
}

static void coding_closer(ScmPort *p)
{
    coding_port_data *data = (coding_port_data*)p->src.buf.data;
    if (data->source) {
        Scm_ClosePort(data->source);
        data->source = NULL;
    }
}

static int coding_ready(ScmPort *p)
{
    coding_port_data *data = (coding_port_data*)p->src.buf.data;
    if (data->source == NULL) return TRUE;
    if (data->state == CODING_PORT_RECOGNIZED) {
        return SCM_FD_READY;
    } else {
        return Scm_ByteReadyUnsafe(p);
    }
}

static int coding_filenum(ScmPort *p)
{
    coding_port_data *data = (coding_port_data*)p->src.buf.data;
    if (data->source == NULL) return -1;
    return Scm_PortFileNo(data->source);
}

ScmObj Scm_MakeCodingAwarePort(ScmPort *iport)
{
    ScmObj p;
    ScmPortBuffer bufrec;
    coding_port_data *data;

    if (!SCM_IPORTP(iport)) {
        Scm_Error("open-coding-aware-port requires an input port, but got %S", iport);
    }
    data = SCM_NEW(coding_port_data);
    data->source = iport;
    data->state = CODING_PORT_INIT;
    data->pbuf = NULL;
    data->pbufsize = 0;

    bufrec.mode = SCM_PORT_BUFFER_FULL;
    bufrec.buffer = NULL;
    bufrec.size = 0;
    bufrec.filler = coding_filler;
    bufrec.flusher = NULL;
    bufrec.closer = coding_closer;
    bufrec.ready = coding_ready;
    bufrec.filenum = coding_filenum;
    bufrec.seeker = NULL;
    bufrec.data = (void*)data;
    p = Scm_MakeBufferedPort(SCM_CLASS_CODING_AWARE_PORT,
                             Scm_PortName(iport), SCM_PORT_INPUT,
                             TRUE, &bufrec);
    return p;
}


/*===============================================================
 * with-port
 */
struct with_port_packet {
    ScmPort *origport[3];
    int mask;
    int closep;
};

static ScmObj port_restorer(ScmObj *args, int nargs, void *data)
{
    struct with_port_packet *p = (struct with_port_packet*)data;
    int pcnt = 0;
    ScmPort *curport;

    if (p->mask & SCM_PORT_CURIN) {
        curport = SCM_CURIN;
        SCM_CURIN = p->origport[pcnt++];
        if (p->closep) Scm_ClosePort(curport);
    }
    if (p->mask & SCM_PORT_CUROUT) {
        curport = SCM_CUROUT;
        SCM_CUROUT = p->origport[pcnt++];
        if (p->closep) Scm_ClosePort(curport);
    }
    if (p->mask & SCM_PORT_CURERR) {
        curport = SCM_CURERR;
        SCM_CURERR = p->origport[pcnt++];
        if (p->closep) Scm_ClosePort(curport);
    }
    return SCM_UNDEFINED;
}

ScmObj Scm_WithPort(ScmPort *port[], ScmObj thunk, int mask, int closep)
{
    ScmObj finalizer;
    struct with_port_packet *packet;
    int pcnt = 0;

    packet = SCM_NEW(struct with_port_packet);
    if (mask & SCM_PORT_CURIN) {
        packet->origport[pcnt] = SCM_CURIN;
        SCM_CURIN = port[pcnt++];
    }
    if (mask & SCM_PORT_CUROUT) {
        packet->origport[pcnt] = SCM_CUROUT;
        SCM_CUROUT = port[pcnt++];
    }
    if (mask & SCM_PORT_CURERR) {
        packet->origport[pcnt] = SCM_CURERR;
        SCM_CURERR = port[pcnt++];
    }
    packet->mask = mask;
    packet->closep = closep;
    finalizer = Scm_MakeSubr(port_restorer, (void*)packet,
                             0, 0, SCM_FALSE);
    return Scm_VMDynamicWind(Scm_NullProc(), SCM_OBJ(thunk), finalizer);
}

/*===============================================================
 * Standard ports
 */

static ScmObj scm_stdin  = SCM_UNBOUND;
static ScmObj scm_stdout = SCM_UNBOUND;
static ScmObj scm_stderr = SCM_UNBOUND;

ScmObj Scm_Stdin(void)
{
    return scm_stdin;
}

ScmObj Scm_Stdout(void)
{
    return scm_stdout;
}

ScmObj Scm_Stderr(void)
{
    return scm_stderr;
}

/*===============================================================
 * Initialization
 */

void Scm__InitPort(void)
{
    (void)SCM_INTERNAL_MUTEX_INIT(active_buffered_ports.mutex);
    active_buffered_ports.ports = SCM_WEAK_VECTOR(Scm_MakeWeakVector(PORT_VECTOR_SIZE));

    Scm_InitStaticClass(&Scm_PortClass, "<port>",
                        Scm_GaucheModule(), NULL, 0);
    Scm_InitStaticClass(&Scm_CodingAwarePortClass, "<coding-aware-port>",
                        Scm_GaucheModule(), NULL, 0);

    scm_stdin  = Scm_MakePortWithFd(SCM_MAKE_STR("(stdin)"),
                                    SCM_PORT_INPUT, 0,
                                    SCM_PORT_BUFFER_FULL, TRUE);
    scm_stdout = Scm_MakePortWithFd(SCM_MAKE_STR("(stdout)"),
                                    SCM_PORT_OUTPUT, 1,
                                    SCM_PORT_BUFFER_LINE, TRUE);
    scm_stderr = Scm_MakePortWithFd(SCM_MAKE_STR("(stderr)"),
                                    SCM_PORT_OUTPUT, 2,
                                    SCM_PORT_BUFFER_NONE, TRUE);
    key_full   = Scm_MakeKeyword(SCM_STRING(SCM_MAKE_STR("full")));
    key_modest = Scm_MakeKeyword(SCM_STRING(SCM_MAKE_STR("modest")));
    key_line   = Scm_MakeKeyword(SCM_STRING(SCM_MAKE_STR("line")));
    key_none   = Scm_MakeKeyword(SCM_STRING(SCM_MAKE_STR("none")));
}
