/* Copyright (C) 2007  Free Software Foundation, Inc.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file into combinations with other programs,
and to distribute those combinations without any restriction coming
from the use of this file.  (The General Public License restrictions
do apply in other respects; for example, they cover modification of
the file, and distribution when not linked into a combine
executable.)

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.  */

#define BID_128RES
#include "bid_internal.h"

/*****************************************************************************
 *  BID128 minimum number
 *****************************************************************************/

#if DECIMAL_CALL_BY_REFERENCE
void
__bid128_minnum (UINT128 * pres, UINT128 * px, UINT128 * py) {
  UINT128 x = *px;
  UINT128 y = *py;
#else
UINT128
__bid128_minnum (UINT128 x, UINT128 y) {
#endif

  UINT128 res;
  int exp_x, exp_y;
  int diff;
  UINT128 sig_x, sig_y;
  UINT192 sig_n_prime192;
  UINT256 sig_n_prime256;
  char x_is_zero = 0, y_is_zero = 0, non_canon_x, non_canon_y;

  BID_SWAP128(x);
  BID_SWAP128(y);
  // NaN (CASE1)
  // if x is NAN, then return y
  if ((x.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    if ((x.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    if ((y.w[1] & MASK_SNAN) == MASK_SNAN) { // y is SNAN 
      // set invalid flag 
      ; // *pfpsf |= INVALID_EXCEPTION; 
      // return quiet (y) 
      ; // y.w[1] = y.w[1] & 0xfdffffffffffffffull; 
    }
    res = y;
    BID_RETURN (res);
  }
  // if y is NAN, then return x
  else if ((y.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    if ((y.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    res = x;
    BID_RETURN (res);
  }
  // SIMPLE (CASE2)
  // if all the bits are the same, these numbers are equal (not Greater).
  if (x.w[0] == y.w[0] && x.w[1] == y.w[1]) {
    res = x;
    BID_RETURN (res);
  }
  // INFINITY (CASE3)
  if ((x.w[1] & MASK_INF) == MASK_INF) {
    // if x is neg infinity, there is no way it is greater than y, return 0
    res = (((x.w[1] & MASK_SIGN) == MASK_SIGN)) ? x : y;
    BID_RETURN (res);
  } else if ((y.w[1] & MASK_INF) == MASK_INF) {
    // x is finite, so if y is positive infinity, then x is less, return 0
    //                 if y is negative infinity, then x is greater, return 1
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }
  // CONVERT X
  sig_x.w[1] = x.w[1] & 0x0001ffffffffffffull;
  sig_x.w[0] = x.w[0];
  exp_x = (x.w[1] >> 49) & 0x000000000003fffull;

  // CHECK IF X IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_x.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_x.w[1] == 0x0001ed09bead87c0ull)
          && (sig_x.w[0] > 0x378d8e63ffffffffull))
      || ((x.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_x = 1;
  else
    non_canon_x = 0;

  // CONVERT Y
  exp_y = (y.w[1] >> 49) & 0x0000000000003fffull;
  sig_y.w[1] = y.w[1] & 0x0001ffffffffffffull;
  sig_y.w[0] = y.w[0];

  // CHECK IF Y IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_y.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_y.w[1] == 0x0001ed09bead87c0ull)
          && (sig_y.w[0] > 0x378d8e63ffffffffull))
      || ((y.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_y = 1;
  else
    non_canon_y = 0;

  // ZERO (CASE4)
  // some properties:
  //    (+ZERO == -ZERO) => therefore ignore the sign
  //    (ZERO x 10^A == ZERO x 10^B) for any valid A, B => ignore the exponent 
  //    field
  //    (Any non-canonical # is considered 0)
  if (non_canon_x || ((sig_x.w[1] == 0) && (sig_x.w[0] == 0))) {
    x_is_zero = 1;
  }
  if (non_canon_y || ((sig_y.w[1] == 0) && (sig_y.w[0] == 0))) {
    y_is_zero = 1;
  }

  if (x_is_zero && y_is_zero) {
    // if both numbers are zero, neither is greater => return either number
    res = x;
    BID_RETURN (res);
  } else if (x_is_zero) {
    // is x is zero, it is greater if Y is negative
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  } else if (y_is_zero) {
    // is y is zero, X is greater if it is positive
    res = ((x.w[1] & MASK_SIGN) != MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }
  // OPPOSITE SIGN (CASE5)
  // now, if the sign bits differ, x is greater if y is negative
  if (((x.w[1] ^ y.w[1]) & MASK_SIGN) == MASK_SIGN) {
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }
  // REDUNDANT REPRESENTATIONS (CASE6)
  // if exponents are the same, then we have a simple comparison of 
  //    the significands
  if (exp_y == exp_x) {
    res = (((sig_x.w[1] > sig_y.w[1])
            || (sig_x.w[1] == sig_y.w[1]
                && sig_x.w[0] >= sig_y.w[0])) ^ ((x.w[1] & MASK_SIGN) ==
                                                 MASK_SIGN)) ? y : x;
    BID_RETURN (res);
  }
  // if both components are either bigger or smaller, it is clear what 
  //    needs to be done
  if (sig_x.w[1] >= sig_y.w[1] && sig_x.w[0] >= sig_y.w[0]
      && exp_x > exp_y) {
    res = ((x.w[1] & MASK_SIGN) != MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }
  if (sig_x.w[1] <= sig_y.w[1] && sig_x.w[0] <= sig_y.w[0]
      && exp_x < exp_y) {
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }

  diff = exp_x - exp_y;

  // if |exp_x - exp_y| < 33, it comes down to the compensated significand
  if (diff > 0) { // to simplify the loop below,
    // if exp_x is 33 greater than exp_y, no need for compensation
    if (diff > 33) {
      // difference cannot be greater than 10^33
      res = ((x.w[1] & MASK_SIGN) != MASK_SIGN) ? y : x;
      BID_RETURN (res);
    }
    if (diff > 19) { //128 by 128 bit multiply -> 256 bits
      __mul_128x128_to_256 (sig_n_prime256, sig_x, __bid_ten2k128[diff - 20]);
      // if postitive, return whichever significand is larger 
      // (converse if negative)
      res = ((((sig_n_prime256.w[3] > 0) || sig_n_prime256.w[2] > 0)
              || (sig_n_prime256.w[1] > sig_y.w[1])
              || (sig_n_prime256.w[1] == sig_y.w[1]
                  && sig_n_prime256.w[0] >
                  sig_y.w[0])) ^ ((y.w[1] & MASK_SIGN) ==
                                  MASK_SIGN)) ? y : x;
      BID_RETURN (res);
    }
    __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_x);
    // if postitive, return whichever significand is larger 
    // (converse if negative)
    res =
      (((sig_n_prime192.w[2] > 0) || (sig_n_prime192.w[1] > sig_y.w[1])
        || (sig_n_prime192.w[1] == sig_y.w[1]
            && sig_n_prime192.w[0] >
            sig_y.w[0])) ^ ((y.w[1] & MASK_SIGN) == MASK_SIGN)) ? y : x;
    BID_RETURN (res);
  }
  diff = exp_y - exp_x;
  // if exp_x is 33 less than exp_y, no need for compensation
  if (diff > 33) {
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }
  if (diff > 19) { //128 by 128 bit multiply -> 256 bits
    // adjust the y significand upwards
    __mul_128x128_to_256 (sig_n_prime256, sig_y, __bid_ten2k128[diff - 20]);
    // if postitive, return whichever significand is larger 
    // (converse if negative)
    res =
      ((sig_n_prime256.w[3] != 0 || sig_n_prime256.w[2] != 0
        || (sig_n_prime256.w[1] > sig_x.w[1]
            || (sig_n_prime256.w[1] == sig_x.w[1]
                && sig_n_prime256.w[0] >
                sig_x.w[0]))) ^ ((x.w[1] & MASK_SIGN) ==
                                 MASK_SIGN)) ? x : y;
    BID_RETURN (res);
  }
  // adjust the y significand upwards
  __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_y);
  // if postitive, return whichever significand is larger (converse if negative)
  res = ((sig_n_prime192.w[2] != 0 || (sig_n_prime192.w[1] > sig_x.w[1] || 
      (sig_n_prime192.w[1] == sig_x.w[1] && sig_n_prime192.w[0] > sig_x.w[0])))
       ^ ((y.w[1] & MASK_SIGN) == MASK_SIGN)) ? x : y;
  BID_RETURN (res);
}

/*****************************************************************************
 *  BID128 minimum magnitude function - returns greater of two numbers
 *****************************************************************************/

#if DECIMAL_CALL_BY_REFERENCE
void
__bid128_minnum_mag (UINT128 * pres, UINT128 * px, UINT128 * py) {
  UINT128 x = *px;
  UINT128 y = *py;
#else
UINT128
__bid128_minnum_mag (UINT128 x, UINT128 y) {
#endif

  UINT128 res;
  int exp_x, exp_y;
  int diff;
  UINT128 sig_x, sig_y;
  UINT192 sig_n_prime192;
  UINT256 sig_n_prime256;
  char non_canon_x, non_canon_y;

  BID_SWAP128(x);
  BID_SWAP128(y);
  // NaN (CASE1)
  // if x is NAN, then return y
  if ((x.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    if ((x.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    if ((y.w[1] & MASK_SNAN) == MASK_SNAN) { // y is SNAN
      // set invalid flag
      ; // *pfpsf |= INVALID_EXCEPTION;
      // return quiet (y)
      ; // y.w[1] = y.w[1] & 0xfdffffffffffffffull;
    }
    res = y;
    BID_RETURN (res);
  } else if ((y.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    // if y is NAN, then return x
    if ((y.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    res = x;
    BID_RETURN (res);
  }
  // SIMPLE (CASE2)
  // if all the bits are the same, these numbers are equal (not Greater).
  if (x.w[0] == y.w[0] && x.w[1] == y.w[1]) {
    res = y;
    BID_RETURN (res);
  }
  // INFINITY (CASE3)
  if ((x.w[1] & MASK_INF) == MASK_INF) {
    // if x infinity, it has maximum magnitude.
    // Check if magnitudes are equal.  If x is negative, return it.
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN
           && (y.w[1] & MASK_INF) == MASK_INF) ? x : y;
    BID_RETURN (res);
  } else if ((y.w[1] & MASK_INF) == MASK_INF) {
    // x is finite, so if y is infinity, then x is less in magnitude
    res = x;
    BID_RETURN (res);
  }
  // CONVERT X
  sig_x.w[1] = x.w[1] & 0x0001ffffffffffffull;
  sig_x.w[0] = x.w[0];
  exp_x = (x.w[1] >> 49) & 0x000000000003fffull;

  // CHECK IF X IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_x.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_x.w[1] == 0x0001ed09bead87c0ull)
          && (sig_x.w[0] > 0x378d8e63ffffffffull))
      || ((x.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_x = 1;
  else
    non_canon_x = 0;

  // CONVERT Y
  exp_y = (y.w[1] >> 49) & 0x0000000000003fffull;
  sig_y.w[1] = y.w[1] & 0x0001ffffffffffffull;
  sig_y.w[0] = y.w[0];

  // CHECK IF Y IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_y.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_y.w[1] == 0x0001ed09bead87c0ull)
          && (sig_y.w[0] > 0x378d8e63ffffffffull))
      || ((y.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_y = 1;
  else
    non_canon_y = 0;

  // ZERO (CASE4)
  // some properties:
  //    (+ZERO == -ZERO) => therefore ignore the sign
  //    (ZERO x 10^A == ZERO x 10^B) for any valid A, B => 
  //        therefore ignore the exponent field
  //    (Any non-canonical # is considered 0)
  if (non_canon_x || ((sig_x.w[1] == 0) && (sig_x.w[0] == 0))) {
    res = x;
    BID_RETURN (res);
  }
  if (non_canon_y || ((sig_y.w[1] == 0) && (sig_y.w[0] == 0))) {
    res = y;
    BID_RETURN (res);
  }
  // REDUNDANT REPRESENTATIONS (CASE6)
  // check if exponents are the same and significands are the same
  if (exp_y == exp_x && sig_x.w[1] == sig_y.w[1]
      && sig_x.w[0] == sig_y.w[0]) {
    if (x.w[1] & 0x8000000000000000ull) { // x is negative
      res = x;
      BID_RETURN (res);
    } else {
      res = y;
      BID_RETURN (res);
    }
  } else if (((sig_x.w[1] > sig_y.w[1] || (sig_x.w[1] == sig_y.w[1]
               && sig_x.w[0] > sig_y.w[0])) && exp_x == exp_y)
             || ((sig_x.w[1] > sig_y.w[1]
                  || (sig_x.w[1] == sig_y.w[1]
                  && sig_x.w[0] >= sig_y.w[0]))
                 && exp_x > exp_y)) {
    // if both components are either bigger or smaller, it is clear what 
    // needs to be done; also if the magnitudes are equal
    res = y;
    BID_RETURN (res);
  } else if (((sig_y.w[1] > sig_x.w[1] || (sig_y.w[1] == sig_x.w[1]
               && sig_y.w[0] > sig_x.w[0])) && exp_y == exp_x)
             || ((sig_y.w[1] > sig_x.w[1]
                  || (sig_y.w[1] == sig_x.w[1]
                  && sig_y.w[0] >= sig_x.w[0]))
                 && exp_y > exp_x)) {
    res = x;
    BID_RETURN (res);
  } else {
    ; // continue
  }
  diff = exp_x - exp_y;
  // if |exp_x - exp_y| < 33, it comes down to the compensated significand
  if (diff > 0) { // to simplify the loop below,
    // if exp_x is 33 greater than exp_y, no need for compensation
    if (diff > 33) {
      res = y; // difference cannot be greater than 10^33
      BID_RETURN (res);
    }
    if (diff > 19) { //128 by 128 bit multiply -> 256 bits
      __mul_128x128_to_256 (sig_n_prime256, sig_x, __bid_ten2k128[diff - 20]);
      // if positive, return whichever significand is larger 
      // (converse if negative)
      if (sig_n_prime256.w[3] == 0 && (sig_n_prime256.w[2] == 0)
          && sig_n_prime256.w[1] == sig_y.w[1]
          && (sig_n_prime256.w[0] == sig_y.w[0])) {
        res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x; // if equal
        BID_RETURN (res);
      }
      res = (((sig_n_prime256.w[3] > 0) || sig_n_prime256.w[2] > 0)
             || (sig_n_prime256.w[1] > sig_y.w[1])
             || (sig_n_prime256.w[1] == sig_y.w[1]
                 && sig_n_prime256.w[0] > sig_y.w[0])) ? y : x;
      BID_RETURN (res);
    }
    __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_x);
    // if positive, return whichever significand is larger 
    // (converse if negative)
    if ((sig_n_prime192.w[2] == 0) && sig_n_prime192.w[1] == sig_y.w[1]
        && (sig_n_prime192.w[0] == sig_y.w[0])) {
      // if = in magnitude, return +, (if possible)
      res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
      BID_RETURN (res);
    }
    res = ((sig_n_prime192.w[2] > 0)
           || (sig_n_prime192.w[1] > sig_y.w[1])
           || (sig_n_prime192.w[1] == sig_y.w[1]
               && sig_n_prime192.w[0] > sig_y.w[0])) ? y : x;
    BID_RETURN (res);
  }
  diff = exp_y - exp_x;
  // if exp_x is 33 less than exp_y, no need for compensation
  if (diff > 33) {
    res = x;
    BID_RETURN (res);
  }
  if (diff > 19) { //128 by 128 bit multiply -> 256 bits
    // adjust the y significand upwards
    __mul_128x128_to_256 (sig_n_prime256, sig_y, __bid_ten2k128[diff - 20]);
    // if positive, return whichever significand is larger 
    // (converse if negative)
    if (sig_n_prime256.w[3] == 0 && (sig_n_prime256.w[2] == 0)
        && sig_n_prime256.w[1] == sig_x.w[1]
        && (sig_n_prime256.w[0] == sig_x.w[0])) {
      // if = in magnitude, return +, (if possible)
      res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
      BID_RETURN (res);
    }
    res = (sig_n_prime256.w[3] == 0 && sig_n_prime256.w[2] == 0
           && (sig_n_prime256.w[1] < sig_x.w[1]
               || (sig_n_prime256.w[1] == sig_x.w[1]
                   && sig_n_prime256.w[0] < sig_x.w[0]))) ? y : x;
    BID_RETURN (res);
  }
  // adjust the y significand upwards
  __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_y);
  // if positive, return whichever significand is larger (converse if negative)
  if ((sig_n_prime192.w[2] == 0) && sig_n_prime192.w[1] == sig_x.w[1]
      && (sig_n_prime192.w[0] == sig_x.w[0])) {
    // if = in magnitude, return +, if possible)
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  }
  res = (sig_n_prime192.w[2] == 0 && (sig_n_prime192.w[1] < sig_x.w[1] || 
      (sig_n_prime192.w[1] == sig_x.w[1] && 
      sig_n_prime192.w[0] < sig_x.w[0]))) ? y : x;
  BID_RETURN (res);
}

/*****************************************************************************
 *  BID128 maximum function - returns greater of two numbers
 *****************************************************************************/

#if DECIMAL_CALL_BY_REFERENCE
void
__bid128_maxnum (UINT128 * pres, UINT128 * px, UINT128 * py) {
  UINT128 x = *px;
  UINT128 y = *py;
#else
UINT128
__bid128_maxnum (UINT128 x, UINT128 y) {
#endif

  UINT128 res;
  int exp_x, exp_y;
  int diff;
  UINT128 sig_x, sig_y;
  UINT192 sig_n_prime192;
  UINT256 sig_n_prime256;
  char x_is_zero = 0, y_is_zero = 0, non_canon_x, non_canon_y;

  BID_SWAP128(x);
  BID_SWAP128(y);
  // NaN (CASE1)
  if ((x.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    // if x is NAN, then return y
    if ((x.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    if ((y.w[1] & MASK_SNAN) == MASK_SNAN) { // y is SNAN
      // set invalid flag
      ; // *pfpsf |= INVALID_EXCEPTION;
      // return quiet (y)
      ; // y.w[1] = y.w[1] & 0xfdffffffffffffffull;
    }
    res = y;
    BID_RETURN (res);
  } else if ((y.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    // if y is NAN, then return x
    if ((y.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    res = x;
    BID_RETURN (res);
  }
  // SIMPLE (CASE2)
  // if all the bits are the same, these numbers are equal (not Greater).
  if (x.w[0] == y.w[0] && x.w[1] == y.w[1]) {
    res = x;
    BID_RETURN (res);
  }
  // INFINITY (CASE3)
  if ((x.w[1] & MASK_INF) == MASK_INF) {
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN) ? y : x;
    BID_RETURN (res);
  } else if ((y.w[1] & MASK_INF) == MASK_INF) {
    // x is finite, so if y is positive infinity, then x is less, return 0
    //                 if y is negative infinity, then x is greater, return 1
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  // CONVERT X
  sig_x.w[1] = x.w[1] & 0x0001ffffffffffffull;
  sig_x.w[0] = x.w[0];
  exp_x = (x.w[1] >> 49) & 0x000000000003fffull;

  // CHECK IF X IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_x.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_x.w[1] == 0x0001ed09bead87c0ull)
          && (sig_x.w[0] > 0x378d8e63ffffffffull))
      || ((x.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_x = 1;
  else
    non_canon_x = 0;

  // CONVERT Y
  exp_y = (y.w[1] >> 49) & 0x0000000000003fffull;
  sig_y.w[1] = y.w[1] & 0x0001ffffffffffffull;
  sig_y.w[0] = y.w[0];

  // CHECK IF Y IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_y.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_y.w[1] == 0x0001ed09bead87c0ull)
          && (sig_y.w[0] > 0x378d8e63ffffffffull))
      || ((y.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_y = 1;
  else
    non_canon_y = 0;

  // ZERO (CASE4)
  // some properties:
  //    (+ZERO == -ZERO) => therefore ignore the sign
  //    (ZERO x 10^A == ZERO x 10^B) for any valid A, B => 
  //        therefore ignore the exponent field
  //    (Any non-canonical # is considered 0)
  if (non_canon_x || ((sig_x.w[1] == 0) && (sig_x.w[0] == 0))) {
    x_is_zero = 1;
  }
  if (non_canon_y || ((sig_y.w[1] == 0) && (sig_y.w[0] == 0))) {
    y_is_zero = 1;
  }

  if (x_is_zero && y_is_zero) {
    // if both numbers are zero, neither is greater => return either number
    res = x;
    BID_RETURN (res);
  } else if (x_is_zero) {
    // is x is zero, it is greater if Y is negative
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
    BID_RETURN (res);
  } else if (y_is_zero) {
    // is y is zero, X is greater if it is positive
    res = ((x.w[1] & MASK_SIGN) != MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  // OPPOSITE SIGN (CASE5)
  // now, if the sign bits differ, x is greater if y is negative
  if (((x.w[1] ^ y.w[1]) & MASK_SIGN) == MASK_SIGN) {
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  // REDUNDANT REPRESENTATIONS (CASE6)
  // if exponents are the same, then we have a simple comparison of 
  // the significands
  if (exp_y == exp_x) {
    res = (((sig_x.w[1] > sig_y.w[1]) || (sig_x.w[1] == sig_y.w[1] && 
        sig_x.w[0] >= sig_y.w[0])) ^ 
        ((x.w[1] & MASK_SIGN) == MASK_SIGN)) ? x : y;
    BID_RETURN (res);
  }
  // if both components are either bigger or smaller, it is clear what 
  // needs to be done
  if ((sig_x.w[1] > sig_y.w[1]
       || (sig_x.w[1] == sig_y.w[1] && sig_x.w[0] > sig_y.w[0]))
      && exp_x >= exp_y) {
    res = ((x.w[1] & MASK_SIGN) != MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  if ((sig_x.w[1] < sig_y.w[1]
       || (sig_x.w[1] == sig_y.w[1] && sig_x.w[0] < sig_y.w[0]))
      && exp_x <= exp_y) {
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  diff = exp_x - exp_y;
  // if |exp_x - exp_y| < 33, it comes down to the compensated significand
  if (diff > 0) { // to simplify the loop below,
    // if exp_x is 33 greater than exp_y, no need for compensation
    if (diff > 33) {
      // difference cannot be greater than 10^33
      res = ((x.w[1] & MASK_SIGN) != MASK_SIGN) ? x : y;
      BID_RETURN (res);
    }
    if (diff > 19) { //128 by 128 bit multiply -> 256 bits
      __mul_128x128_to_256 (sig_n_prime256, sig_x, __bid_ten2k128[diff - 20]);
      // if postitive, return whichever significand is larger 
      // (converse if negative)
      res = ((((sig_n_prime256.w[3] > 0) || sig_n_prime256.w[2] > 0)
              || (sig_n_prime256.w[1] > sig_y.w[1])
              || (sig_n_prime256.w[1] == sig_y.w[1]
                  && sig_n_prime256.w[0] >
                  sig_y.w[0])) ^ ((y.w[1] & MASK_SIGN) ==
                                  MASK_SIGN)) ? x : y;
      BID_RETURN (res);
    }
    __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_x);
    // if postitive, return whichever significand is larger 
    // (converse if negative)
    res =
      (((sig_n_prime192.w[2] > 0) || (sig_n_prime192.w[1] > sig_y.w[1])
        || (sig_n_prime192.w[1] == sig_y.w[1]
            && sig_n_prime192.w[0] >
            sig_y.w[0])) ^ ((y.w[1] & MASK_SIGN) == MASK_SIGN)) ? x : y;
    BID_RETURN (res);
  }
  diff = exp_y - exp_x;
  // if exp_x is 33 less than exp_y, no need for compensation
  if (diff > 33) {
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  if (diff > 19) { //128 by 128 bit multiply -> 256 bits
    // adjust the y significand upwards
    __mul_128x128_to_256 (sig_n_prime256, sig_y, __bid_ten2k128[diff - 20]);
    // if postitive, return whichever significand is larger 
    // (converse if negative)
    res =
      ((sig_n_prime256.w[3] != 0 || sig_n_prime256.w[2] != 0
        || (sig_n_prime256.w[1] > sig_x.w[1]
            || (sig_n_prime256.w[1] == sig_x.w[1]
                && sig_n_prime256.w[0] >
                sig_x.w[0]))) ^ ((x.w[1] & MASK_SIGN) !=
                                 MASK_SIGN)) ? x : y;
    BID_RETURN (res);
  }
  // adjust the y significand upwards
  __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_y);
  // if postitive, return whichever significand is larger (converse if negative)
  res = ((sig_n_prime192.w[2] != 0 || (sig_n_prime192.w[1] > sig_x.w[1] || 
      (sig_n_prime192.w[1] == sig_x.w[1] && 
      sig_n_prime192.w[0] > sig_x.w[0]))) ^ 
      ((y.w[1] & MASK_SIGN) != MASK_SIGN)) ? x : y;
  BID_RETURN (res);
}

/*****************************************************************************
 *  BID128 maximum magnitude function - returns greater of two numbers
 *****************************************************************************/

#if DECIMAL_CALL_BY_REFERENCE
void
__bid128_maxnum_mag (UINT128 * pres, UINT128 * px, UINT128 * py) {
  UINT128 x = *px;
  UINT128 y = *py;
#else
UINT128
__bid128_maxnum_mag (UINT128 x, UINT128 y) {
#endif

  UINT128 res;
  int exp_x, exp_y;
  int diff;
  UINT128 sig_x, sig_y;
  UINT192 sig_n_prime192;
  UINT256 sig_n_prime256;
  char non_canon_x, non_canon_y;

  BID_SWAP128(x);
  BID_SWAP128(y);
  // NaN (CASE1)
  if ((x.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    // if x is NAN, then return y
    if ((x.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    if ((y.w[1] & MASK_SNAN) == MASK_SNAN) { // y is SNAN
      // set invalid flag
      ; // *pfpsf |= INVALID_EXCEPTION;
      // return quiet (y)
      ; // y.w[1] = y.w[1] & 0xfdffffffffffffffull;
    }
    res = y;
    BID_RETURN (res);
  } else if ((y.w[1] & 0x7c00000000000000ull) == 0x7c00000000000000ull) {
    // if y is NAN, then return x
    if ((y.w[1] & 0x0200000000000000ull) == 0x0200000000000000ull) {
      ; // *pfpsf |= INVALID_EXCEPTION;   // set exception if sNaN
    }
    res = x;
    BID_RETURN (res);
  }
  // SIMPLE (CASE2)
  // if all the bits are the same, these numbers are equal (not Greater).
  if (x.w[0] == y.w[0] && x.w[1] == y.w[1]) {
    res = y;
    BID_RETURN (res);
  }
  // INFINITY (CASE3)
  if ((x.w[1] & MASK_INF) == MASK_INF) {
    // if x infinity, it has maximum magnitude
    res = ((x.w[1] & MASK_SIGN) == MASK_SIGN
           && (y.w[1] & MASK_INF) == MASK_INF) ? y : x;
    BID_RETURN (res);
  } else if ((y.w[1] & MASK_INF) == MASK_INF) {
    // x is finite, so if y is positive infinity, then x is less, return 0
    //                 if y is negative infinity, then x is greater, return 1
    res = y;
    BID_RETURN (res);
  }
  // CONVERT X
  sig_x.w[1] = x.w[1] & 0x0001ffffffffffffull;
  sig_x.w[0] = x.w[0];
  exp_x = (x.w[1] >> 49) & 0x000000000003fffull;

  // CHECK IF X IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_x.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_x.w[1] == 0x0001ed09bead87c0ull)
          && (sig_x.w[0] > 0x378d8e63ffffffffull))
      || ((x.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_x = 1;
  else
    non_canon_x = 0;

  // CONVERT Y
  exp_y = (y.w[1] >> 49) & 0x0000000000003fffull;
  sig_y.w[1] = y.w[1] & 0x0001ffffffffffffull;
  sig_y.w[0] = y.w[0];

  // CHECK IF Y IS CANONICAL
  // 9999999999999999999999999999999999(decimal) = 
  //     1ed09_bead87c0_378d8e63_ffffffff(hexadecimal)
  // [0, 10^34) is the 754r supported canonical range.  
  // If the value exceeds that, it is interpreted as 0.
  if ((sig_y.w[1] > 0x0001ed09bead87c0ull)
      || ((sig_y.w[1] == 0x0001ed09bead87c0ull)
          && (sig_y.w[0] > 0x378d8e63ffffffffull))
      || ((y.w[1] & 0x6000000000000000ull) == 0x6000000000000000ull))
    non_canon_y = 1;
  else
    non_canon_y = 0;

  // ZERO (CASE4)
  // some properties:
  //    (+ZERO == -ZERO) => therefore ignore the sign
  //    (ZERO x 10^A == ZERO x 10^B) for any valid A, B => 
  //         therefore ignore the exponent field
  //    (Any non-canonical # is considered 0)
  if (non_canon_x || ((sig_x.w[1] == 0) && (sig_x.w[0] == 0))) {
    res = y;
    BID_RETURN (res);
  }
  if (non_canon_y || ((sig_y.w[1] == 0) && (sig_y.w[0] == 0))) {
    res = x;
    BID_RETURN (res);
  }
  // REDUNDANT REPRESENTATIONS (CASE6)
  if (exp_y == exp_x && sig_x.w[1] == sig_y.w[1]
      && sig_x.w[0] == sig_y.w[0]) {
    // check if exponents are the same and significands are the same
    if (x.w[1] & 0x8000000000000000ull) { // x is negative
      res = y;
      BID_RETURN (res);
    } else {
      res = x;
      BID_RETURN (res);
    }
  } else if (((sig_x.w[1] > sig_y.w[1] || (sig_x.w[1] == sig_y.w[1]
               && sig_x.w[0] > sig_y.w[0])) && exp_x == exp_y)
             || ((sig_x.w[1] > sig_y.w[1]
                  || (sig_x.w[1] == sig_y.w[1]
                  && sig_x.w[0] >= sig_y.w[0]))
                 && exp_x > exp_y)) {
    // if both components are either bigger or smaller, it is clear what 
    // needs to be done; also if the magnitudes are equal
    res = x;
    BID_RETURN (res);
  } else if (((sig_y.w[1] > sig_x.w[1] || (sig_y.w[1] == sig_x.w[1]
               && sig_y.w[0] > sig_x.w[0])) && exp_y == exp_x)
             || ((sig_y.w[1] > sig_x.w[1]
                  || (sig_y.w[1] == sig_x.w[1]
                  && sig_y.w[0] >= sig_x.w[0]))
                 && exp_y > exp_x)) {
    res = y;
    BID_RETURN (res);
  } else {
    ;        // continue
  }
  diff = exp_x - exp_y;
  // if |exp_x - exp_y| < 33, it comes down to the compensated significand
  if (diff > 0) { // to simplify the loop below,
    // if exp_x is 33 greater than exp_y, no need for compensation
    if (diff > 33) {
      res = x; // difference cannot be greater than 10^33
      BID_RETURN (res);
    }
    if (diff > 19) { //128 by 128 bit multiply -> 256 bits
      __mul_128x128_to_256 (sig_n_prime256, sig_x, __bid_ten2k128[diff - 20]);
      // if postitive, return whichever significand is larger 
      // (converse if negative)
      if (sig_n_prime256.w[3] == 0 && (sig_n_prime256.w[2] == 0)
          && sig_n_prime256.w[1] == sig_y.w[1]
          && (sig_n_prime256.w[0] == sig_y.w[0])) {
        res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y; // if equal
        BID_RETURN (res);
      }
      res = (((sig_n_prime256.w[3] > 0) || sig_n_prime256.w[2] > 0)
             || (sig_n_prime256.w[1] > sig_y.w[1])
             || (sig_n_prime256.w[1] == sig_y.w[1]
                 && sig_n_prime256.w[0] > sig_y.w[0])) ? x : y;
      BID_RETURN (res);
    }
    __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_x);
    // if postitive, return whichever significand is larger (converse if negative)
    if ((sig_n_prime192.w[2] == 0) && sig_n_prime192.w[1] == sig_y.w[1]
        && (sig_n_prime192.w[0] == sig_y.w[0])) {
      // if equal, return positive magnitude
      res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
      BID_RETURN (res);
    }
    res = ((sig_n_prime192.w[2] > 0)
           || (sig_n_prime192.w[1] > sig_y.w[1])
           || (sig_n_prime192.w[1] == sig_y.w[1]
               && sig_n_prime192.w[0] > sig_y.w[0])) ? x : y;
    BID_RETURN (res);
  }
  diff = exp_y - exp_x;
  // if exp_x is 33 less than exp_y, no need for compensation
  if (diff > 33) {
    res = y;
    BID_RETURN (res);
  }
  if (diff > 19) { //128 by 128 bit multiply -> 256 bits
    // adjust the y significand upwards
    __mul_128x128_to_256 (sig_n_prime256, sig_y, __bid_ten2k128[diff - 20]);
    // if postitive, return whichever significand is larger 
    // (converse if negative)
    if (sig_n_prime256.w[3] == 0 && (sig_n_prime256.w[2] == 0)
        && sig_n_prime256.w[1] == sig_x.w[1]
        && (sig_n_prime256.w[0] == sig_x.w[0])) {
      // if equal, return positive (if possible)
      res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
      BID_RETURN (res);
    }
    res = (sig_n_prime256.w[3] == 0 && sig_n_prime256.w[2] == 0
           && (sig_n_prime256.w[1] < sig_x.w[1]
               || (sig_n_prime256.w[1] == sig_x.w[1]
                   && sig_n_prime256.w[0] < sig_x.w[0]))) ? x : y;
    BID_RETURN (res);
  }
  // adjust the y significand upwards
  __mul_64x128_to_192 (sig_n_prime192, __bid_ten2k64[diff], sig_y);
  // if postitive, return whichever significand is larger (converse if negative)
  if ((sig_n_prime192.w[2] == 0) && sig_n_prime192.w[1] == sig_x.w[1]
      && (sig_n_prime192.w[0] == sig_x.w[0])) {
    // if equal, return positive (if possible)
    res = ((y.w[1] & MASK_SIGN) == MASK_SIGN) ? x : y;
    BID_RETURN (res);
  }
  res = (sig_n_prime192.w[2] == 0 && (sig_n_prime192.w[1] < sig_x.w[1] || 
      (sig_n_prime192.w[1] == sig_x.w[1] && 
      sig_n_prime192.w[0] < sig_x.w[0]))) ? x : y;
  BID_RETURN (res);
}
