package GCMainWindow;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use utf8;
    
use strict;
use Gtk2;

my $VERSION = '5.3';

{
    package GCMovieWindow;
    use base "Gtk2::Dialog";
    use GCGraphicComponents;

    sub new
    {
        my ($proto, $parent, $title) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new('',
                              undef,
                              [qw/no-separator destroy-with-parent/]);
        bless ($self, $class);
        
        $self->set_modal(0);
        $self->set_icon_from_file($ENV{GCF_SHARE_DIR}.'/icons/gcfilms_logo32.png');
        $self->set_title('GCfilms - '.$title);

        $self->{panel} = new GCMoviePanel($parent, $parent->{options}, 1);

        my $scrollPanelMovie = new Gtk2::ScrolledWindow;
        $scrollPanelMovie->set_policy ('automatic', 'automatic');
        $scrollPanelMovie->set_shadow_type('etched-in');
        $scrollPanelMovie->add_with_viewport($self->{panel});

        $self->vbox->add($scrollPanelMovie);

        $self->vbox->show_all;
        $self->{panel}->setShowOption($parent->{displayOptionsDialog}->{show});

        $self->set_default_size(600,500);
        
        return $self;
    }
}

{
    package GCRandomMovieWindow;
    use base "Gtk2::Dialog";
    use GCGraphicComponents;

    sub show
    {
        my $self = shift;

        $self->SUPER::show();
        $self->show_all;
        my $code = $self->run;
        $self->destroy;
        return $code;
    }

    sub new
    {
        my ($proto, $parent, $title) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new('',
                              undef,
                              [qw/no-separator destroy-with-parent/],
                              @GCDialogs::okCancelButtons);
        bless ($self, $class);
        
        $self->set_modal(0);
        $self->set_icon_from_file($ENV{GCF_SHARE_DIR}.'/icons/gcfilms_logo32.png');
        $self->set_title('GCfilms - '.$title);

        $self->{panel} = new GCMoviePanel($parent, $parent->{options}, 1);

        my $scrollPanelMovie = new Gtk2::ScrolledWindow;
        $scrollPanelMovie->set_policy ('automatic', 'automatic');
        $scrollPanelMovie->set_shadow_type('etched-in');
        $scrollPanelMovie->add_with_viewport($self->{panel});

        $self->{nextButton} = Gtk2::Button->new_from_stock('gtk-go-forward');
        $parent->{tooltips}->set_tip(($self->action_area->get_children)[0],
                                     $parent->{lang}->{RandomOkTip});
        $parent->{tooltips}->set_tip($self->{nextButton},
                                     $parent->{lang}->{RandomNextTip});

        $self->add_action_widget($self->{nextButton}, 'no');
        $self->vbox->add($scrollPanelMovie);

        $self->vbox->show_all;
        $self->{panel}->setShowOption($parent->{displayOptionsDialog}->{show});

        $self->set_default_size(600,500);
        $self->set_position('center');
        
        return $self;
    }
}

{
    package GCFrame;
    use base "Gtk2::Window";

    if ($^O =~ /win32/i)
    {
        use File::DosGlob 'glob';
    }

    use GCMoviesLists;
    use GCPlugins;
    use GCGraphicComponents;
    use GCUtils;
    use GCOptions;
    use GCBorrowings;
    use GCDialogs;
    use GCExport;
    use GCImport;
    use GCSplash;
    use GCLang;
    use GCStyle;
    use GCDisplay;
    use GCData;
    use GCGenres;

    use LWP;
    use URI::Escape;
    use File::Basename;
    use IO::Handle;
    use Storable qw(store_fd fd_retrieve);

    sub beforeDestroy
    {
		my $self = shift;
		$self->leave;
		return 1;
    }

    sub leave
    {
        my $self = shift;

        $self->saveList if ($self->{options}->autosave) && ($self->{movies}->{nbMovies});
	
		my ($width, $height) = $self->get_size;
	
		$self->{options}->width($width);
		$self->{options}->height($height);
		$self->{options}->split($self->{pane}->get_position) if ($self->{pane});
	
		$self->{options}->save;

        $self->{movies}->clean;

        if ($^O !~ /win32/i)
        {
            close $self->{searchJob}->{reader};
            close $self->{searchJob}->{writer};
            kill 9, $self->{searchJob}->{pid};
            wait;
        }
        
        $self->destroy;
    }

    sub deleteCurrentMovie
    {
        my $self = shift;
		my $response = 'yes';
	
		if ($self->{options}->confirm)
		{
		    my  $dialog = Gtk2::MessageDialog->new($self,
							   [qw/modal destroy-with-parent/],
							   'question',
							   'yes-no',
							   $self->{lang}->{RemoveConfirm});
		    $response = $dialog->run;
		    $dialog->destroy;
		}
	
		if ($response eq 'yes')
		{
		    $self->{movies}->removeCurrentMovie;
		}
    }

    sub newMovie
    {
			
        my ($self, $self2) = @_;
        $self = $self2 if $self2;
		
		#deverrouille l'interface a la création d'une nouvelle fiche
		$self->{options}->lockPanel(0);
		# Question métaphysique : je mémorise ou pas ?, décommenté pour mémoriser
		#$self->{panel}->change_state($self->{panel}, 0);
		$self->{menubar}->{menuEditLockMoviesItem}->set_active(0);

		my $info = {};
		$info->{title} = $self->{lang}->{NewMovie};
		$info->{seen} = 0;
		$self->addMovie($info, 1);
    }

    sub loadUrl
    {
        my ($self, $url) = @_;
        
        my $baseUrl;
        my $plugin;
        foreach (values %GCPlugins::pluginsMap)
        {
            ($baseUrl = $_->getMovieUrl) =~ s/http:\/\/(w{3})?//;
            $plugin = $_;
            last if $url =~ m/$baseUrl/;
        }
        return unless $url =~ m/$baseUrl/;
        $self->setWaitCursor($self->{lang}->{StatusGetInfo});
        $self->newMovie;
        my $info = $plugin->loadUrl($url);
        $self->restoreCursor;
        return if ! $info->{title};        
        $self->addMovie($info, 0);
    }
    
    sub searchMovie
    {
        my ($self, $title, $forceask, $currentPlugin) = @_;

        my $plugin;
        
        if (($self->{options}->plugin eq 'multi') && !$forceask)
        {
            my $pluginName = $self->{multiSiteDialog}->getPlugin($currentPlugin);
            $plugin = $GCPlugins::pluginsMap{$pluginName};
            if (!$plugin)
            {
              my $dialog = Gtk2::MessageDialog->new($self,
							   [qw/modal destroy-with-parent/],
							   'error',
							   'ok',
							   $self->{lang}->{MultiSiteEmptyError});
		      my $response = $dialog->run;
		      $dialog->destroy;
		      return;
            
            }
        }
        elsif ($self->{options}->plugin eq 'multiask')
        {
            $self->{pluginsAskDialog}->setPluginsList($self->{options}->multisite);
            $self->{pluginsAskDialog}->query($title);
            $self->{pluginsAskDialog}->show;
            $plugin = $self->{pluginsAskDialog}->plugin;
#            $plugin->setProxy($self->{options}->proxy);
            return if !$plugin;
            $title = $self->{pluginsAskDialog}->query;
        }
        elsif ($forceask || ! $self->{plugin})
        {
            $self->{pluginsDialog}->query($title);
            $self->{pluginsDialog}->show;
            $plugin = $self->{pluginsDialog}->plugin;
            return if !$plugin;
#            $plugin->setProxy($self->{options}->proxy);
            $title = $self->{pluginsDialog}->query;
        }
        else
        {
            $plugin = $self->{plugin};
        }

        $plugin->setProxy($self->{options}->proxy);

        $plugin->{title} = $title;
        $plugin->{type} = 'load';
        
        if (! $self->{options}->searchStop)
        {
            $self->setWaitCursor($self->{lang}->{StatusSearch}.' ('.$plugin->getName.')');
            $plugin->load;
            $self->restoreCursor;
        }
        else
        {
            store_fd $plugin, $self->{searchJob}->{command};
            my $getInfo = 0;
            my $dialog = Gtk2::MessageDialog->new($self,
                [qw/modal destroy-with-parent/],
                'info',
                'cancel',
                $self->{lang}->{StatusSearch} . "\n" . $plugin->getName);
                
            my $progress = new Gtk2::ProgressBar;
            $dialog->vbox->pack_start($progress,1,1,10);
            $progress->set_pulse_step(0.05);
            $progress->show_all;
            my $pulseTimeout = Glib::Timeout->add(50 , sub {
                return 0 if $getInfo;
                $progress->pulse;
                return 1;
            });    
                
            my $watch = Glib::IO->add_watch(fileno($self->{searchJob}->{data}),
                                'in',
                                sub {
                                    return if !$dialog;
                                    $dialog->response('cancel');
                                    $getInfo = 1;
                                    return 0;
                                });
            $dialog->run if !$getInfo;
            Glib::Source->remove($watch);
            Glib::Source->remove($pulseTimeout);
            $dialog->destroy;
            my $command = $self->{searchJob}->{command};
            if ($getInfo)
            {
                $plugin = fd_retrieve($self->{searchJob}->{data});
                print $command "OK\n";
            }
            else
            {
                print $command "STOP\n";
                return;
            }
        }
            
		my $movieNumber = $plugin->getMoviesNumber();
	
		if ($movieNumber == 0)
		{
		    my $force = 0;
		    my $idx = 0;
		    if (!$forceask && ($self->{options}->plugin eq 'multi')
		     && (($currentPlugin + 1) < $self->{multiSiteDialog}->getPluginsNumber))
		    {
		      $idx = $currentPlugin + 1;
		    }
		    else
		    {
		      my  $dialog = Gtk2::MessageDialog->new($self,
							   [qw/modal destroy-with-parent/],
							   'error',
							   'yes-no',
							   $self->{lang}->{NoMovieFound});
		      my $response = $dialog->run;
		      $dialog->destroy;
		      return if $response ne 'yes';
		      $force = 1;
		    }
		    
            $self->searchMovie($title,$force, $idx);
		}
		elsif ($movieNumber == 1)
		{
		    $self->getMovieInfo($plugin, 0);
		}
		else
		{
		    my $withNext = 0;
            $withNext = 1 if ($self->{options}->plugin eq 'multi')
		        && (($currentPlugin + 1) < $self->{multiSiteDialog}->getPluginsNumber);
		    
		    my @movies = $plugin->getMovies();
            $self->{resultsDialog}->setWithNext($withNext);
            $self->{resultsDialog}->setSearchPlugin($plugin);
		    $self->{resultsDialog}->setList('', @movies);
		    my $next = $self->{resultsDialog}->show;
		    if ($self->{resultsDialog}->{validated})
		    {
				my $idx = $self->{resultsDialog}->{movieIndex};
				$self->getMovieInfo($plugin, $idx);
		    }
		    elsif ($next)
		    {
		      $self->searchMovie($title,0, $currentPlugin + 1);
		    }
		}
    }

    sub getMovieInfo
    {
        my ($self, $plugin, $idx) = @_;
        
        $plugin->{wantedIdx} = $idx;
        $plugin->{type} = 'info';
        my $info;
        
        if (! $self->{options}->searchStop)
        {
            $self->setWaitCursor($self->{lang}->{StatusGetInfo});
            $info = $plugin->getMovieInfo;
            $self->restoreCursor;
        }
        else
        {
            store_fd $plugin, $self->{searchJob}->{command};
            my $getInfo = 0;
            my $dialogGet = Gtk2::MessageDialog->new($self,
                [qw/modal destroy-with-parent/],
                'info',
                'cancel',
                $self->{lang}->{StatusGetInfo});
                
            my $progress = new Gtk2::ProgressBar;
            $dialogGet->vbox->pack_start($progress,1,1,10);
            $progress->set_pulse_step(0.05);
            $progress->show_all;
            my $pulseTimeout = Glib::Timeout->add(50 , sub {
                return 0 if $getInfo;
                $progress->pulse;
                return 1;
            });
                
            my $watch = Glib::IO->add_watch(fileno($self->{searchJob}->{data}),
                                'in',
                                sub {
                                    return if !$dialogGet;
                                    $dialogGet->response('cancel');
                                    $getInfo = 1;
                                    return 0;
                                });
            $dialogGet->run if !$getInfo;
            Glib::Source->remove($watch);
            Glib::Source->remove($pulseTimeout);
            $dialogGet->destroy;
            my $command = $self->{searchJob}->{command};
            if ($getInfo)
            {
                $info = fd_retrieve($self->{searchJob}->{data});
                print $command "OK\n";
            }
            else
            {
                print $command "STOP\n";
                return;
            }
        }

        $info->{orig} = $info->{title} if (!$info->{orig}) && ($self->{options}->alwaysOriginal);
        $info->{image} = '' if $info->{image} eq 'empty';        
        
        if ($self->{options}->askImport)
        {
            $self->{importFieldsDialog}->info($info);
            return if ! $self->{importFieldsDialog}->show;
            $info = $self->{importFieldsDialog}->info;
        }
        $self->addMovie($info, 0);
    }
    
    sub getUniqueImageFileName
    {
        my ($self, $suffix) = @_;
    
        my $tmpImagesDir = $self->{options}->images;
    
        if ( ! -e $tmpImagesDir)
        {
            mkdir($tmpImagesDir);
        }

        my $filePrefix = $tmpImagesDir.$self->{imagePrefix};
        my @tmp_filenames;
        @tmp_filenames = glob $filePrefix.'*.*';
        my %modtimes;
        my @sorted = sort { ($modtimes{$a} ||= -M $a) <=>
                         ($modtimes{$b} ||= -M $b)
                       } @tmp_filenames;
        my $i = 0;
        my $mostRecent = $sorted[$i];
        while (! $mostRecent =~ /^[0-9]*\./ )
	    {
			$mostRecent = $sorted[$i];
			$i++;
	    }
        my $sysPrefix = $filePrefix;
        $sysPrefix =~ s/\\/\\\\/g if ($^O =~ /win32/i);
        $mostRecent =~ s/$sysPrefix([0-9]*)\..*/$1/;

        $mostRecent = 0 if !$mostRecent;

        #my ($name,$path,$suffix) = fileparse($info->{image}, "\.gif", "\.jpg", "\.png");

        my $picture = $filePrefix.$mostRecent.$suffix;

        while (-e $picture)
        {
            $mostRecent++;
            $picture = $filePrefix.$mostRecent.$suffix;
        }
        
        return $picture;
    }
    
    sub changeInfo
    {
        my ($self, $info) = @_;

        my @genres = split /,/, $info->{type};
        my $newGenres = '';
        
        foreach (@genres)
        {
            $newGenres .= $self->{genresDialog}->{convertor}->convert($_).',';
        }
        $newGenres =~ s/.$//;
        
        $info->{type} = $newGenres;
    }
    
    sub addMovie
    {
        my ($self, $info, $newMovie) = @_;

        $self->{movies}->updateCurrent if $newMovie;
        
        $self->changeInfo($info);

        my $ignore = $self->{ignoreString};
        $self->{panel}->clear if $newMovie;
        
        $self->{panel}->title($info->{title}) if $info->{title} ne $ignore;
        $self->{panel}->date($info->{date}) if $info->{date} ne $ignore;
        $self->{panel}->time($info->{time}) if $info->{time} ne $ignore;
        $self->{panel}->director($info->{director}) if $info->{director} ne $ignore;
        $self->{panel}->nationality($info->{nat}) if $info->{nat} ne $ignore;
        $self->{panel}->orig($info->{orig}) if $info->{orig} ne $ignore;
        $self->{panel}->actors($info->{actors}) if $info->{actors} ne $ignore;
        $self->{panel}->synopsis($info->{synopsis}) if $info->{synopsis} ne $ignore;
        $self->{panel}->url($info->{url}) if $info->{url} ne $ignore;
        $self->{panel}->age($info->{age}) if $info->{age} ne $ignore;
        $self->{panel}->audio($info->{audio}) if $info->{audio} ne $ignore;
        $self->{panel}->subt($info->{subt}) if $info->{subt} ne $ignore;
        
        $self->{panel}->selectTitle if $newMovie;

        if ($info->{image} && ($info->{image} ne $ignore))
        {
            #$info->{image} = $self->downloadPicture($info->{image});
            #$self->{moviesList}->changeCurrent($info->{image}, $info->{borrower})
            #    if ($info->{image} ne $self->{panel}->image) ||
            #       ($info->{borrower} ne $self->{panel}->borrower);
            #$self->{panel}->image($info->{image});
            #$self->{movies}->markToBeAdded($info->{image});
            
            
                        $self->setWaitCursor($self->{lang}->{StatusGetImage});
            my $browser = LWP::UserAgent->new;
            $browser->proxy(['http'], $self->{options}->proxy);
            $browser->agent($self->{agent});
  
            my ($name,$path,$suffix) = fileparse($info->{image}, "\.gif", "\.jpg", "\.png");
            my $picture = $self->getUniqueImageFileName($suffix);

            $browser->get($info->{image}, ':content_file' => $picture);

            $info->{image} = $picture;
            $self->{moviesList}->changeCurrent($info->{image}, $info->{borrower})
                if ($info->{image} ne $self->{panel}->image) ||
                   ($info->{borrower} ne $self->{panel}->borrower);
            $self->{panel}->image($info->{image});
            $self->{movies}->markToBeAdded($picture);
            $self->restoreCursor;
            
        }

        if ($info->{type} ne $ignore)
        {
            #$self->{panel}->setTypes($self->{movies}->getTypes);
            #$self->{menubar}->setTypes($self->{movies}->getTypes);
            #$self->{menubar}->setTypes($self->{movies}->getTypes,
            #                           $self->{context}->{menuCategories});
            
            $self->{panel}->type($info->{type});
        }

        if ($newMovie)
        {
            my $id = $self->{movies}->addMovie($info);
            $self->{panel}->id($id);
        }
        else
        {
            $self->{movies}->updateCurrent;
            $self->{moviesList}->showCurrent;
        }
    }
    
    sub downloadPicture
    {
        my ($self, $pictureURL) = @_;
    
        $self->setWaitCursor($self->{lang}->{StatusGetImage});
        my $browser = LWP::UserAgent->new;
        $browser->proxy(['http'], $self->{options}->proxy);
        $browser->agent($self->{agent});
  
        my ($name,$path,$suffix) = fileparse($pictureURL, "\.gif", "\.jpg", "\.png");
        my $picture = $self->getUniqueImageFileName($suffix);

        $browser->get($pictureURL, ':content_file' => $picture);
        $self->restoreCursor;
            
        return $picture;
    }

    sub openMovie
    {
        my ($self, $filename) = @_;

        $self->saveList if ($self->{options}->autosave) && ($self->{movies}->{nbMovies});
        
        (my $history = $self->{options}->history) =~ s/(.*\|)?($filename\|)(.*\|)?/$2$1$3/;
        $history =~ s/(.*?)\|[^\|]*?$/$filename\|$1/ if $history !~ /^$filename/;
        $self->{options}->history($history);
        $self->{menubar}->{menuHistoryItem}->remove_submenu();
        $self->{menubar}->{menuHistory} = Gtk2::Menu->new();
        $self->{menubar}->addHistoryMenu();
        $self->{options}->file($filename);
        $self->{options}->save;
        $self->{movies}->load($self->{options}->file, $self, 1);
        $self->{movies}->display(0);
        $self->{movies}->select(0);
        $self->set_title('GCfilms - ['.basename($filename).']');			
    }

    sub openList
    {
        my $self = shift;
        my $fileDialog = new GCFileChooserDialog($self->{lang}->{OpenList}, $self, 'open');

        $fileDialog->set_filename($self->{options}->file);
        my $response = $fileDialog->run;
        if ($response eq 'ok')
        {
            my $fileName = $fileDialog->get_filename;
            $fileDialog->destroy;
            $self->openMovie($fileName);
        }
        else
        {
            $fileDialog->destroy;
        }
    }

    sub newList
    {
        my $self = shift;

        $self->saveList if ($self->{options}->autosave) && ($self->{movies}->{nbMovies});

        $self->{movies}->clearList;
        $self->{options}->file('');
        $self->set_title('GCfilms - []');
    }

    sub saveAs
    {
        my $self = shift;
        my $fileDialog = new GCFileChooserDialog($self->{lang}->{SaveList}, $self, 'save');
        $fileDialog->set_filename($self->{options}->file);
            
        while (1)
        {
    	   	my $response = $fileDialog->run;
            if ($response eq 'ok')
            {
                my $filename = $fileDialog->get_filename;
                my $file = $self->{options}->file;
                (my $history = $self->{options}->history) =~ s/(.*\|)?$file\|(.*\|)?/$filename\|$1$2/;
                $self->{options}->history($history);
                $self->{menubar}->{menuHistoryItem}->remove_submenu();
                $self->{menubar}->{menuHistory} = Gtk2::Menu->new();
                $self->{menubar}->addHistoryMenu();
                my $previousFile = $self->{options}->file;
                $self->{options}->file($filename);
                if ($self->saveList())
                {
                    $self->{options}->save;
                    $self->set_title('GCfilms - ['.basename($filename).']');
                    last;
                }
                else
                {
                    $self->{options}->file($previousFile);
                }            
    		}
    		last if ($response eq 'cancel')
        }
        $fileDialog->destroy;
    } 

    sub saveList
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;
				my $response = 'yes';

        if ($self->{options}->file)
        {
            return $self->{movies}->save;
        }
        else
        {
	       my $dialog = Gtk2::MessageDialog->new($self,
                                                  [qw/modal destroy-with-parent/],
                                                  'info',
                                                  'yes-no',
                                                  $self->{lang}->{SaveCurrentList});
	       $response = $dialog->run();
	       $dialog->destroy ;
		   $self->saveAs if ($response eq 'yes');
        }
    }

    sub removeSearch
    {
        my $self = shift;
        
        $self->setFilter('search', 'all');
    }
    
    sub search
    {
        my ($self, $self2, $value) = @_;
        $self = $self2 if ($self2 ne 'all') && ($self2 ne 'displayed');
        my $type = 'all';
        $type = $value if ($self != $self2);

        $self->{searchDialog}->show;
        
        my $info = $self->{searchDialog}->search;
        return if ! $info;
        
        $self->{menubar}->selectAll if $type eq 'all';
        $self->setFilter('search', $info);
    }
    
    sub showBorrowed
    {
        my $self = shift;
        $self->{movies}->updateCurrent;
        $self->{borrowedDialog}->setList($self->{movies}->{movieArray});
        $self->{borrowedDialog}->show;
    }
    
    sub export
    {
        my ($self, $exporter) = @_;
        $self->{exportDialog}->setModule($exporter);
        $self->{exportDialog}->show;
    }
    sub import
    {
        my ($self, $importer) = @_;
        $self->{importDialog}->setModule($importer);
        $self->{importDialog}->show;
    }

    sub importWithDetect
    {
        my ($self, $file) = @_;
  
        $file =~ s/^file:\/\/(.*)\W*$/$1/;
        $file =~ s/.$//ms;
        (my $suffix = $file) =~ s/.*\.(\w+)\W*$/$1/;
        
        foreach my $importer(@GCImport::importersArray)
        {
            my $current = $importer->getSuffix;
            if ($current eq $suffix)
            {
                $self->setWaitCursor($self->{lang}->{StatusGetInfo});
                my %options;
                $options{parent} = $self;
                $options{newList} = 0;
                $options{file} = $file;
                $options{lang} = $self->{lang};
                $importer->process(\%options);
                $self->restoreCursor;
                return;
            }
        }
    }

    sub optionsError
    {
        my ($self, $type) = shift;

        my $msg;
        if ($type eq 'open')
        {
            $msg = $self->{lang}->{OptionsOpenError};
        }
        elsif ($type eq 'create')
        {
            $msg = $self->{lang}->{OptionsCreateError};
        }
        else
        {
            $msg = $self->{lang}->{OptionsSaveError};
        }
            
        my $dialog = Gtk2::MessageDialog->new($self,
                                              [qw/modal destroy-with-parent/],
                                              'error',
                                              'ok',
                                              $msg.$self->{options}->{file});
        $dialog->run();
        $dialog->destroy ;

        $self->destroy;
    }

    sub checkImagesDirectory
    {
        my ($self, $withDialog) = @_;
        my $error = 0;
        if ( ! -e $self->{options}->images)
        {
            $! = '';
            $error = 1 if (!mkdir($self->{options}->images));
            rmdir $self->{options}->images if ! $!;
        }
        else
        {
            $error = 1 if !( -d _ && -r _ && -w _ && -x _ )
        }
        return $error if !$withDialog;
        if ($error)
        {
            my $fileDialog = new GCFileChooserDialog($self->{lang}->{OptionsImagesTitle}, $self, 'select-folder');
            my  $errorDialog = Gtk2::MessageDialog->new($self,
						   [qw/modal destroy-with-parent/],
						   'error',
						   'ok',
						   $self->{lang}->{ImageError});
            $fileDialog->set_filename($self->{options}->images);
            my $response;
            do
            {
                $errorDialog->run();
                $errorDialog->hide();
                $response = '';
                $response = $fileDialog->run while ($response ne 'ok');
                $self->{options}->images($fileDialog->get_filename);
            } while ($self->checkImagesDirectory(0));
            $errorDialog->destroy;
            $fileDialog->destroy;
        }
        return $error;
    }

    sub setWaitCursor
    {
		my ($self, $message) = @_;
        $self->setStatus($message);
		$self->window->set_cursor(Gtk2::Gdk::Cursor->new('watch'));
		Gtk2->main_iteration while (Gtk2->events_pending);
    }
    sub restoreCursor
    {
		my $self = shift;
        $self->restoreStatus;
		$self->window->set_cursor(Gtk2::Gdk::Cursor->new('left_ptr'));
		Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub setFilter
    {
        my ($self, $filter, $parameter) = @_;
        $self->{movies}->updateCurrent;
        $self->{movies}->setFilter($filter, $parameter);
    }

    sub onSortClick
    {
        my ($widget, $self) = @_;
        $self->{moviesList}->getSortButton()->signal_connect('clicked', \&onSortClick, $self)
            if ($self->{moviesList}->setSortOrder($self->{movies}->changeOrder));
    }

    sub selectMultiPluginsList
    {
        my $self = shift;
        
        $self->{multiSiteDialog}->show;
    }

    sub options
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        my $transform = $self->{options}->transform;
        my $articles = $self->{options}->articles;
        my $formats = $self->{options}->formats;
        my $urlPicture = $self->{options}->urlPicture;

        $self->{optionsDialog}->show;
        #$self->{multiSiteDialog}->show;

        $self->checkTransform
            if ($self->{options}->articles ne $articles) || ($self->{options}->transform != $transform);
            
        $self->checkFormats
            if ($self->{options}->formats ne $formats);

        $self->checkUrlPicture
            if ($self->{options}->urlPicture ne $urlPicture);
       
        $self->checkDisplayed;
        $self->checkPlugin;
        $self->checkView;
    }
    
    sub displayOptions
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        my $displayed = $self->{options}->displayed;
        $self->{displayOptionsDialog}->show;
        $self->checkPanel
            if ($self->{options}->displayed ne $displayed);
    }

    sub borrowers
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->{borrowersDialog}->show;
        
        $self->checkBorrowers;
    }

    sub genresConversion
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->{genresDialog}->show;
        
        #my ($old, $new) = $self->{renameDialog}->names;
        #return if ! $old;
        #$self->{movies}->rename($old,$new);
    }
    
    sub rename
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->{renameDialog}->show;
        
        my ($old, $new) = $self->{renameDialog}->names;
        return if ! $old;
        $self->{movies}->rename($old,$new);
    }

    sub about
    {
        my $self = shift;
        
        my $about = new GCAboutDialog($self, $VERSION);
        $about->show;
    }

    sub help
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->openUrl('http://home.gna.org/gcfilms/doc');
    }

    sub openUrl
    {
        my ($self, $url) = @_;

        my $command = $self->{options}->browser.' '.$url;
        if ($^O =~ /win32/i)
        {
            system "start $command";
        }
        else
        {
            $command =~ s/&/\\&/g;
            system "$command &";
        }
    }

    sub setStatus
    {
		my ($self, $status) = @_;
		$self->{status}->push(1, $status) if ($self->{status});
    }
    
    sub restoreStatus
    {
        my $self = shift;
        
        $self->{status}->pop(1);
    }

    sub display
    {
        my ($self, $idx) = @_;
        
        $self->{movies}->display($idx);
    }
    
    sub displayInWindow
     {
        my ($self, $idx, $random) = @_;

        my $title = $self->{movies}->{movieArray}[$idx]->{title};
        $title = $self->{movies}->currentTitle if $idx == undef;

        my $window;
        if ($random)
        {
	        $window = new GCRandomMovieWindow($self, $title);
        }
        else
        {
	        $window = new GCMovieWindow($self, $title);			
        }

        my @borrowers = split m/\|/, $self->{options}->borrowers;
        $window->{panel}->setBorrowers(\@borrowers);

        $self->{movies}->displayInPanel($window->{panel}, $idx);
        
        my $code = $window->show;

        return $code;
    }
   
    sub randomMovie
    {
        my ($self) = @_;
        my @tmpArray = undef;
        $self->{movies}->updateCurrent;
        
        my $message = '';
        
        #Initialize movies array.
        $self->{randomPool} = [];
        my $realId = 0;
        foreach (@{$self->{movies}->{movieArray}})
        {
            if (!$_->{seen})
            {
                $_->{realId} = $realId;
                push @{$self->{randomPool}}, $_;
            }
            $realId++;
        }
 
        if (scalar @{$self->{randomPool}} > 0)
        {
            my $code = 'no';
            my $idx = 0;
            while ($code eq 'no')
            {
        	   $idx = int rand(scalar @{$self->{randomPool}});
        	   $realId = $self->{randomPool}->[$idx]->{realId};
        	   $code = $self->displayInWindow($realId, 1);
        	   splice @{$self->{randomPool}}, $idx, 1;
        	   last if ! @{$self->{randomPool}};
            }
            $message = $self->{lang}->{RandomEnd} if $code eq 'no';
            if ($code eq 'ok')
            {
                $self->{movies}->{movieArray}[$realId]->{seen} = 1;
                $self->{panel}->seen(1)
                    if $self->{movies}->{currentMovie} == $realId;
            }
        }
    	else
    	{
    	   $message = $self->{lang}->{RandomError};
    	}

        if ($message)
        {
            my $dialog = Gtk2::MessageDialog->new($self->{parent},
    		  	                                 [qw/modal destroy-with-parent/],
                                                'info',
                                                  'ok',
                                                  $message);
    		$dialog->run();
	       	$dialog->destroy;
        }
    }
    
    sub initProgress
    {
        my $self = shift;
 
        $self->setProgress(0.0);
        $self->set_sensitive(0);
    }
    
    sub endProgress
    {
        my $self = shift;
        $self->setProgress(1.0);
        Glib::Timeout->add(500 , sub {
            $self->setProgress(0.0);
            return 0;
        });
        $self->set_sensitive(1);
    }
    
    sub setProgress
    {
        my ($self, $current) = @_;
        $self->{progress}->set_fraction($current);
        Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub setProgressForMoviesLoad
    {
    }

    sub setProgressForMoviesDisplay
    {
        my ($self, $current, $total) = @_;
        return if ($current % 10);
        $self->{progress}->set_fraction($current / $total) if $total;
        $self->{progress}->set_fraction(0.5) if ! $total;
        Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub blockListUpdates
    {
        my ($self, $value) = @_;
        
        $self->{movies}->{block} = $value;
    }

    sub reloadList
    {
        my $self = shift;
        
        $self->{movies}->reloadList($self, 1);
    }

    sub setMoviesList
    {
        my ($self, $view, $columns, $init) = @_;
          
        $self->{pane}->remove($self->{moviesList}) if $self->{pane} && $self->{moviesList};
        $self->{moviesList}->destroy if $self->{moviesList};
        
        $self->{moviesList} = new GCTextList($self, $self->{lang}->{Movies}) if $view == 0;
        $self->{moviesList} = new GCImageList($self, $self->{lang}->{Movies}, $columns) if $view == 1;
        $self->{moviesList} = new GCDetailledList($self, $self->{lang}->{Movies}) if $view == 2;

        $self->{moviesList}->setSortOrder if $self->{movies};
        $self->{pane}->pack1($self->{moviesList},1,0) if $self->{pane};
        $self->{moviesList}->show_all;
        if ($self->{movies})
        {
            $self->{movies}->resetSortField;
            $self->reloadList if ! $self->{initializing};
            Glib::Timeout->add(100 ,\&showCurrent, $self);
            #$self->{moviesList}->showCurrent;
        }
        
        #Assign context menu to movies list that will be in charge of displaying it.
        $self->{moviesList}->{context} = $self->{context};
    }
    
    sub createContextMenu
    {
        my $self = shift;
      
        # Context menu creation. It is displayed when right clicking on a list item.
        $self->{context} = new Gtk2::Menu;
        
        $self->{context}->append(Gtk2::TearoffMenuItem->new());
        
        my $itemDelete = Gtk2::ImageMenuItem->new_from_stock('gtk-delete',undef);
        $itemDelete->signal_connect("activate" , sub {
            $self->deleteCurrentMovie;
        });
        $self->{context}->append($itemDelete);
        
        my $newWindow = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuNewWindow});
        $newWindow->signal_connect("activate" , sub {
                $self->displayInWindow;
        });
        $self->{context}->append($newWindow);
        
        $self->{context}->append(Gtk2::SeparatorMenuItem->new);
        
        $self->{context}->{menuDisplayType} = new Gtk2::Menu;
        my %views = %{$self->{lang}->{OptionsViews}};
        my $displayGroup = undef;
		foreach (0..(scalar(keys %views) - 1))
		{
		    my $item = Gtk2::RadioMenuItem->new_with_mnemonic($displayGroup, $views{$_});
		    $item->signal_connect('activate', sub {
		        my ($widget, $self) = @_;
                if ($widget->get_active)
                {
                    my $group = $widget->get_group;
                    my $i = 0;
                    $i++ while ($views{$i} ne $widget->child->get_label);
                    $self->{options}->view($i);
	                $self->setMoviesList($self->{options}->view,
                                         $self->{options}->columns);
                    $self->checkView;
                    
                }
		    }, $self);
		    $self->{context}->{menuDisplayType}->append($item);
		    $displayGroup = $item->get_group;
		}
		
        $self->{context}->{itemDisplayType} = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{OptionsView});
        $self->{context}->{itemDisplayType}->set_submenu($self->{context}->{menuDisplayType});
        $self->{context}->append($self->{context}->{itemDisplayType});
        
        $self->{context}->append(Gtk2::SeparatorMenuItem->new);
        
        # Filters selection
                
        my $menuDisplay = Gtk2::Menu->new();
        my $displayItem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuDisplay});
        
        my $allMovies = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuViewAllMovies});
        $allMovies->signal_connect("activate" , sub {
                $self->{menubar}->selectAll;
        });
        $menuDisplay->append($allMovies);
        
        #my $searchSelectedMovies = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuSearchSelectedMovies});
        my $searchSelectedMovies = Gtk2::ImageMenuItem->new_from_stock('gtk-find',undef);
        $searchSelectedMovies->signal_connect("activate" , sub {
                $self->search('displayed');
        });
        $menuDisplay->append($searchSelectedMovies);
        
        $menuDisplay->append(Gtk2::SeparatorMenuItem->new);
        
        $self->{contextMenuSeen} = Gtk2::Menu->new();
        my $allSeen = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$self->{lang}->{MenuAllMovies});
        $self->{contextMenuSeen}->append($allSeen);
        $allSeen->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuSeen', 0);
        });
        my $seenGroup = $allSeen->get_group();
        my $notseen = Gtk2::RadioMenuItem->new_with_mnemonic($seenGroup,$self->{lang}->{MenuNotSeen});
        $self->{contextMenuSeen}->append($notseen);
        $notseen->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuSeen', 1);
        });
        my $seenItem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuSeenOrNot});
        $seenItem->set_submenu($self->{contextMenuSeen});
        $menuDisplay->append($seenItem);

        $self->{contextMenuBorrowing} = Gtk2::Menu->new();
        my $allBorrowed = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$self->{lang}->{MenuAllMovies});
        $self->{contextMenuBorrowing}->append($allBorrowed);
        $allBorrowed->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuBorrowing', 0);
        });
        my $borrowedGroup = $allBorrowed->get_group();
        my $notBorrowed = Gtk2::RadioMenuItem->new_with_mnemonic($borrowedGroup,$self->{lang}->{MenuNotBorrowed});
        $self->{contextMenuBorrowing}->append($notBorrowed);
        $notBorrowed->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuBorrowing', 1);
        });
        my $isBorrowed = Gtk2::RadioMenuItem->new_with_mnemonic($borrowedGroup,$self->{lang}->{MenuBorrowed});
        $self->{contextMenuBorrowing}->append($isBorrowed);
        $isBorrowed->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuBorrowing', 2);
        });
        my $borrowedItem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuBorrowing});
        $borrowedItem->set_submenu($self->{contextMenuBorrowing});
        $menuDisplay->append($borrowedItem);

        $self->{contextMenuCategories} = Gtk2::Menu->new();
        my $allCat = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$self->{lang}->{MenuAllCategories});
        $self->{contextMenuCategories}->append($allCat);
        $allCat->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuCategories', 0);
        });
        my $catGroup = $allCat->get_group();
        my $catitem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuCategories});
        $catitem->set_submenu($self->{contextMenuCategories});
        $menuDisplay->append($catitem);
        $self->{contextMenuCategories}->{group} = $catGroup;
        $self->{context}->{menuCategories} = $self->{contextMenuCategories};

        $self->{contextMenuAge} = Gtk2::Menu->new();
        my $allMoviesAge = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$self->{lang}->{MenuAllMovies});
        $self->{contextMenuAge}->append($allMoviesAge);
        $allMoviesAge->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuAge', 0);
        });
        my $ageGroup = $allMoviesAge->get_group();
        my $noneAge = Gtk2::RadioMenuItem->new_with_mnemonic($ageGroup,$self->{lang}->{MenuAgeNone});
        $self->{contextMenuAge}->append($noneAge);
        $noneAge->signal_connect("activate" , sub {
            $self->contextDisplayChange(shift, 'menuAge', 1);
        });
        my $parentAge = Gtk2::RadioMenuItem->new_with_mnemonic($ageGroup,$self->{lang}->{MenuAgeParent});
        $self->{contextMenuAge}->append($parentAge);
        $parentAge->signal_connect("activate" , sub {
            $self->contextDisplayChange(shift, 'menuAge', 2);
        });
        my $chooseAge = Gtk2::RadioMenuItem->new_with_mnemonic($ageGroup,$self->{lang}->{MenuAgeLess});
        $self->{contextMenuAge}->append($chooseAge);
        $chooseAge->signal_connect("activate" , sub {
            $self->contextDisplayChange(shift, 'menuAge', 3);
        });
        my $ageItem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuAge});
        $ageItem->set_submenu($self->{contextMenuAge});
        $menuDisplay->append($ageItem);

        $self->{contextMenuRating} = Gtk2::Menu->new();
        my $allRating = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$self->{lang}->{MenuAllRating});
        $self->{contextMenuRating}->append($allRating);
        $allRating->signal_connect("activate", sub {
            $self->contextDisplayChange(shift, 'menuRating', 0);
        });
        my $ratingGroup = $allRating->get_group();
        my $chooseRating = Gtk2::RadioMenuItem->new_with_mnemonic($ratingGroup,$self->{lang}->{MenuChooseRating});
        $self->{contextMenuRating}->append($chooseRating);
        $chooseRating->signal_connect("activate" , sub {
            $self->contextDisplayChange(shift, 'menuRating', 1);
        });
        my $ratingItem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuRating});
        $ratingItem->set_submenu($self->{contextMenuRating});
        $menuDisplay->append($ratingItem);
        
        $displayItem->set_submenu($menuDisplay);
        $self->{context}->append($displayItem);
        
        $self->{context}->show_all;
      
    }
    
    sub contextDisplayChange
    {
        my ($self, $widget, $menuName, $number) = @_;
        
        return if $widget && ! $widget->get_active;
        return if $self->{menubar}->{contextUpdating};

        $self->{menubar}->{contextUpdating} = 1;
        $self->{menubar}->{$menuName}->set_active($number);
        $self->{menubar}->{$menuName}->get_active->activate;
        $self->{menubar}->{contextUpdating} = 0;
    }
    
    sub showCurrent
    {
        my $self = shift;
        $self->{moviesList}->showCurrent;
    }
    
    sub allMovies
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->{menubar}->selectAll;        
    }
    
    sub transformTitle
    {
        my ($self, $title) = @_;

        return $title if ! $self->{options}->transform;
        
        my $tmpTitle = $title;
        foreach (@{$self->{articles}})
        {
            return $tmpTitle if ($tmpTitle =~ s/^($_)(\s+|('))(.)(.*)/\U$4\E$5 ($1$3)/i); #'
        }
        return $tmpTitle;
    }
    
    sub new
    {
        my ($proto, $options, $searchJob) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new('toplevel');
        bless ($self, $class);

        $options->setParent($self);
        $self->{options} = $options;

        $self->{searchJob} = $searchJob;

        $self->{lang} = $GCLang::langs{$self->{options}->lang};

        $options->style('Gtk') if ! $options->exists('style');
        my $style = $GCStyle::styles{$options->style};

        $self->{style} = $style;
        
        $self->{initializing} = 1;
        $self->set_sensitive(0);

        if (($self->{options}->splash) || (! $self->{options}->exists('splash')))
        {
            my $splash = new GCSplashWindow($self);
        }
        else
        {            
            $self->init;
            $self->loadPrevious;
            $self->{initializing} = 0;
            $self->set_sensitive(1);
        }

        return $self;
    }

    sub createStockItems
    {
        my $self = shift;
        
        my $baseStock;
        $baseStock->{translation_domain} = 'gtk20';
        $baseStock->{keyval} = 0;
        $baseStock->{modifier} = [  ];
        
        $baseStock->{stock_id} = 'gtk-execute';
        $baseStock->{label} = $self->{lang}->{ToolbarRandom};                
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-refresh';
        $baseStock->{label} = $self->{lang}->{ToolbarAll};                
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-convert';
        $baseStock->{label} = $self->{lang}->{MenuImport};                
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-paste';
        $baseStock->{label} = $self->{lang}->{MenuExport};                
        Gtk2::Stock->add($baseStock);

        $baseStock->{stock_id} = 'gtk-jump-to';
        $baseStock->{label} = $self->{lang}->{PanelSearchButton};                
        Gtk2::Stock->add($baseStock);
        
        my $addStock = Gtk2::Stock->lookup('gtk-add');
        $addStock->{keyval} = 65;
        $addStock->{modifier} = [ 'control-mask' ];
        Gtk2::Stock->add($addStock);
    }

    sub init
    {
        my $self = shift;
        my $splash = shift;

        $self->{options}->save if $self->checkImagesDirectory(1);
        
        $self->createStockItems;
        
        $splash->setProgress(0.01) if $splash;

        Gtk2::Rc->parse($self->{style}->{rcFile});
        
        $self->{agent} = 'Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.7.5) Gecko/20041111 Firefox/1.0';
        $self->{ignoreString} = 'gcf_000_ignore';
        $self->{imagePrefix} = 'gcfilms_';
        $self->{defaultImage} = $ENV{GCF_SHARE_DIR}.'/no.jpg';
        
        $self->{tooltips} = Gtk2::Tooltips->new();
        
        $self->{searchJob}->{command}->autoflush(1);
        $self->{searchJob}->{data}->autoflush(1);

        my $tmpTitle = $self->{options}->file ? basename($self->{options}->file) : '';
        $self->set_title('GCfilms - ['.$tmpTitle.']');
        $self->set_icon_from_file($ENV{GCF_SHARE_DIR}.'/icons/gcfilms_logo32.png');

        $self->signal_connect(delete_event => \&beforeDestroy, $self);
        $self->signal_connect(destroy => sub { Gtk2->main_quit; });
        
        $splash->setProgress(0.02) if $splash;
        
        $self->createContextMenu;
        $self->setMoviesList($self->{options}->view, $self->{options}->columns, 1);

        $splash->setProgress(0.03) if $splash;

        $self->{panel} = new GCMoviePanel($self, $self->{options}, 0);
        $self->{scrollPanelMovie} = new Gtk2::ScrolledWindow;
        $self->{scrollPanelMovie}->set_policy ('automatic', 'automatic');
        $self->{scrollPanelMovie}->set_shadow_type('etched-in');
        $self->{scrollPanelMovie}->add_with_viewport($self->{panel});

        $self->{movies} = new GCMovies($self);
        $self->{movies}->setPanel($self->{panel});

        $self->{menubar} = new GCMenuBar($self);
        $self->{menubar}->set_name('GCMenubar');

        $splash->setProgress(0.04) if $splash;

        $self->{toolbar} = GCToolBar->new($self);
        $self->{hboxToolbar} = new Gtk2::HBox(0,0);
        $self->{hboxToolbar}->pack_start($self->{toolbar},1,1,0);
        
        if (-e $ENV{GCF_SHARE_DIR}.'/icons/gcfilms_logo48.png')
        {
            my $logoButton = Gtk2::Button->new;
            $logoButton->set_relief('half');
            $logoButton->signal_connect('clicked' , sub {
                $self->about;
            });
            my $logo = Gtk2::Image->new_from_file($ENV{GCF_SHARE_DIR}.'/icons/gcfilms_logo48.png');
            $logoButton->add($logo);
            $self->{hboxToolbar}->pack_start($logoButton,0,0,0);
        }

        my $vbox = new Gtk2::VBox(0, 0);
        $self->{pane} = new Gtk2::HPaned;

        $self->{pane}->pack1($self->{moviesList},1,0);
        $self->{pane}->pack2($self->{scrollPanelMovie},1,1);
        $self->{pane}->set_position($self->{options}->split);

        $vbox->pack_start($self->{menubar}, 0, 0, 1);
        $vbox->pack_start($self->{hboxToolbar},0,0,0 ) ;
        $vbox->pack_start($self->{pane}, 1, 1, 0);

        $self->{status} = Gtk2::Statusbar->new;
        $self->{status}->set_has_resize_grip(1);
        $self->{progress} = new Gtk2::ProgressBar;
        $self->{progress}->set_size_request(100,-1);
        $self->{status}->pack_start($self->{progress}, 0, 0, 5);
        $vbox->pack_start($self->{status},0,0,0);

        $self->add($vbox);

        $splash->setProgress(0.05) if $splash;

        $self->set_default_size($self->{options}->width,$self->{options}->height);

        $self->{resultsDialog} = new GCResultsDialog($self);
        $self->{optionsDialog} = new GCOptionsDialog($self, $self->{options});
        $self->{displayOptionsDialog} = new GCDisplayOptionsDialog($self, 'displayed');
        $self->{borrowersDialog} = new GCBorrowersDialog($self);
        $self->{borrowedDialog} = new GCBorrowedDialog($self);
        
        $self->{exportDialog} = new GCExportDialog($self);
        $self->{importDialog} = new GCImportDialog($self);
        
        $splash->setProgress(0.06) if $splash;
        
        $self->{searchDialog} = new GCSearchDialog($self);
        $self->{renameDialog} = new GCRenameDialog($self);
        $self->{importFieldsDialog} = new GCImportFieldsDialog($self);
        $self->{pluginsDialog} = new GCPluginsDialog($self);
        $self->{pluginsAskDialog} = new GCPluginsDialog($self);
        $self->{multiSiteDialog} = new GCMultiSiteDialog($self);
        $self->{genresDialog} = new GCGenresGroupsDialog($self);
        
        $splash->setProgress(0.07) if $splash;
        
        $self->drag_dest_set('all', ["copy","private","default","move","link","ask"]);

        $self->signal_connect(drag_data_received => \&drop_handler);

        my $target_list = Gtk2::TargetList->new();
        my $atom1 = Gtk2::Gdk::Atom->new("text/uri-list");
        my $atom2 = Gtk2::Gdk::Atom->new("text/plain");
        $target_list->add($atom1, 0, 0);
        $target_list->add($atom2, 0, 0);

        $self->drag_dest_set_target_list($target_list);

        sub drop_handler {
            my ($treeview, $context, $widget_x, $widget_y, $data, $info,$time) = @_;
            my $type = $data->type->name;
            if ($type eq 'text/uri-list')
            {
                $self->newList;
                my @files = split /\n/, $data->data;
                $self->importWithDetect($_) foreach @files;
                $self->{movies}->setStatus;
            }
            elsif ((my $url = $data->data) =~ m/^http:\/\//)
            {
                $self->loadUrl($url);
            }
        }

        $splash->setProgress(0.08) if $splash;
        
        $self->show_all;

        $splash->setProgress(0.09) if $splash;

        $self->checkDisplayed;
        $self->checkPlugin;
        $self->checkBorrowers;
        $self->checkView;
        $self->checkTransform;
        $self->checkPanel;
        
        $self->{options}->searchStop(0) if ($^O =~ /win32/i);

    }

    sub checkDisplayed
    {
        my $self = shift;
        
        if ($self->{options}->status)
        {
            $self->{status}->show;
        }
        else
        {
            $self->{status}->hide;
        }

        if ($self->{options}->toolbar)
        {
            $self->{hboxToolbar}->show;
            $self->{toolbar}->set_icon_size('small-toolbar') if $self->{options}->toolbar == 1;
            $self->{toolbar}->set_icon_size('large-toolbar') if $self->{options}->toolbar == 2;
        }
        else
        {
            $self->{hboxToolbar}->hide;
        }

    }

    sub checkProxy
    {
		my $self= shift;
		
		$self->{plugin}->setProxy($self->{options}->proxy) if ($self->{plugin});
    }
    
    sub checkPlugin
    {
        my $self = shift;
        $self->{plugin} = undef;
        $self->{plugin} = $GCPlugins::pluginsMap{$self->{options}->plugin};

        $self->checkProxy;
    }
    
    sub checkBorrowers
    {
        my $self = shift;
        
        my @borrowers = split m/\|/, $self->{options}->borrowers;
        $self->{panel}->setBorrowers(\@borrowers);
    }
    
    sub checkView
    {
        my $self = shift;
		my %views = %{$self->{lang}->{OptionsViews}};
        my $newLabel = $views{$self->{options}->view};
		if(scalar (keys %views) != 2) # inactif quand le toggle btn est utilise
		{ 
		    my $currentLabel = $self->{toolbar}->{viewOption}->child->get_label;
            return if ($currentLabel eq $newLabel);
        	$self->{toolbar}->{view}->set_active($self->{options}->view);
        	$self->{toolbar}->{viewOption}->remove_menu;
        	$self->{toolbar}->{viewOption}->set_menu($self->{toolbar}->{view});
		}
		else
		{
		    my $currentLabel = $self->{toolbar}->{view}->get_label;
            return if ($currentLabel eq $newLabel);
    		$self->{toolbar}->{view}->set_label($newLabel);
    		$self->{toolbar}->{view}->set_active(!$self->{options}->view);
		}
        $self->{context}->{menuDisplayType}->set_active($self->{options}->view);
        $self->{context}->{menuDisplayType}->get_active->set_active(1);
    }
    
    sub checkTransform
    {
        my $self = shift;
 
        my @array = split m/,/, $self->{options}->articles;
        
        foreach (@array)
        {
            s/^\s*//;
            s/\s*$//;
        }
        
        $self->{articles} = \@array;
        
        $self->reloadList if ! $self->{initializing};
    }
    
    sub checkFormats
    {
        my $self = shift;
        
        my @formats = split m/,/, $self->{options}->formats;
        $self->{panel}->setFormats(\@formats);
    }
    
    sub checkPanel
    {
        my $self = shift;

        $self->{panel}->setShowOption($self->{displayOptionsDialog}->{show});
    }
    
    sub checkUrlPicture
    {
        my $self = shift;
        
        $self->{panel}->setUrlType($self->{options}->urlPicture ? 'image' : 'text');
    }
    
    sub loadPrevious
    {
        my $self = shift;
        my $splash = shift;

        $self->{movies}->setOptions($self->{options}, $splash);
        $self->{panel}->setTypes($self->{movies}->getTypes);
        $splash->setProgress(0.91) if $splash;
        $self->{panel}->setPlaces($self->{movies}->getPlaces);
        $splash->setProgress(0.93) if $splash;
        $self->{panel}->setAudios($self->{movies}->getAudios);
        $splash->setProgress(0.95) if $splash;
        $self->{panel}->setSubs($self->{movies}->getSubs);
        $splash->setProgress(0.97) if $splash;
        $self->{menubar}->setTypes($self->{movies}->getTypes);
        $splash->setProgress(0.99) if $splash;
        $self->{menubar}->setTypes($self->{movies}->getTypes,
                                   $self->{context}->{menuCategories});
        
        $self->{movies}->display(0);
        $self->{movies}->select(0);
        
        # Windows program starts iconify to hide console window.
        # Here we make only main window become visible.
        $self->deiconify if ($^O =~ /win32/i);
    }

}

1;
