# -*- coding: utf-8 -*-

from sensor.Sensor import Sensor
from utils.datatypes import *
from utils import i18n

import re, time, urllib, feedparser

class BibleSensor(Sensor):

    VOTD_URL   = "http://www.biblegateway.com/usage/votd/rss/votd.rdf?%(version_id)s"
    JUNK_START = "<br />>"

    # The amount of time (in minutes) between updates of the verse of the day.
    # The default is 60 minutes because, well, the verse of the day doesn't
    # change very often. :)

    UPDATE_INTERVAL = 60


    def __init__(self):

        global _; _ = i18n.Translator("bible-sensor")

        self.__citation = ""
        self.__text = _("Loading...")
        self.__copyright = ""
        self.__verse_url = ""
        self.__has_error = "false"
        self.__has_text = "true"

        Sensor.__init__(self)

        self._set_config_type("version_id", TYPE_STRING, "31")

        self._add_thread(self.__load_passage_thread)
        self._add_timer(1000, self.__show_passage_timer)


    def get_configurator(self):
        """Returns the configurator for the Bible sensor."""

        configurator = self._new_configurator()
        configurator.set_name(_("Bible"))
        configurator.add_title(_("Bible Settings"))
        configurator.add_option(
            _("Version:"), "version_id",
            _("The Bible version in which the verse will be displayed"),
            [
                 ("28", "العربية (AR)"),
                ("    Arabic Life Application Bible", "28"),
                ("Български (BG)", "21"),
                ("    Bulgarian Bible", "21"),
                ("Kreyol (CPF)", "23"),
                ("    Haitian Creole Version", "23"),
                ("Čeština (CS)", "29"),
                ("    Slovo na cestu", "29"),
                ("Dansk (DA)", "11"),
                ("    Dette er Biblen pÃ¥ dansk", "11"),
                ("Deutsch (DE)", "10"),
                ("    Luther Bibel 1545", "10"),
                ("English (EN)", "31"),
                ("    Amplified Bible", "45"),
                ("    American Standard Version", "8"),
                ("    Contemporary English Version", "46"),
                ("    Darby Translation", "16"),
                ("    Douay-Rheims 1899 American Edition", "63"),
                ("    English Standard Version", "47"),
                ("    Holman Christian Standard Bible", "77"),
                ("    21st Century King James Version", "48"),
                ("    King James Version", "9"),
                ("    The Message", "65"),
                ("    New American Standard Bible", "49"),
                ("    New Century Version", "78"),
                ("    New International Reader's Version", "76"),
                ("    New International Version", "31"),
                ("    New International Version - UK", "64"),
                ("    New King James Version", "50"),
                ("    New Living Translation", "51"),
                ("    New Life Version", "74"),
                ("    Worldwide English (New Testament)", "73"),
                ("    Wycliffe New Testament", "53"),
                ("    Young's Literal Translation", "15"),
                ("Español (ES)", "60"),
                ("    Biblia en Lenguaje Sencillo", "57"),
                ("    Castillian", "41"),
                ("    Dios Habla Hoy", "58"),
                ("    La Biblia de las Américas", "59"),
                ("    Nueva Versión Internacional", "42"),
                ("    Reina-Valera Antigua", "6"),
                ("    Reina-Valera 1960", "60"),
                ("    Reina-Valera 1995", "61"),
                ("Français (FR)", "2"),
                ("    La Bible du Semeur", "32"),
                ("    Louis Segond", "2"),
                ("Κοινη (GRC)", "69"),
                ("    1550 Stephanus New Testament", "69"),
                ("    1894 Scrivener New Testamant", "70"),
                ("    1881 Westcott-Hort New Testament", "68"),
                ("Ilonggo (HIL)", "71"),
                ("    Hiligaynon Bible", "71"),
                ("Hrvatski (HR)", "62"),
                ("    Croatian Bible", "62"),
                ("Magyar (HU)", "17"),
                ("    Hungarian KÃ¡roli", "17"),
                ("Íslenska (IS)", "18"),
                ("    Icelandic Bible", "18"),
                ("Italiano (IT)", "3"),
                ("    Conferenza Episcopale Italiana", "3"),
                ("    Il Nuovo Testamento in Lingua Moderna", "34"),
                ("    La Nuaova Diodati", "55"),
                ("한국 (KO)", "20"),
                ("    Korean Bible", "20"),
                ("Latina (LA)", "4"),
                ("    Biblia Sacra Vulgata", "4"),
                ("Maori (MI)", "24"),
                ("    Maori Bible", "24"),
                ("Plautdietsch (NDS)", "56"),
                ("    Reimer 2001", "56"),
                ("Nederlands (NL)", "30"),
                ("    Het Boek", "30"),
                ("Norsk (NO)", "5"),
                ("    Det Norsk Bibelselskap 1930", "5"),
                ("    Levande Bibeln", "35"),
                ("Polski (PL)", "12"),
                ("    Biblia Tysiaclecia", "12"),
                ("Portugues (PT)", "36"),
                ("    João Ferreira de Almeida Atualizada", "25"),
                ("    Nova Versão Internacional", "36"),
                ("    O Livro", "37"),
                ("Româna (RO)", "14"),
                ("    Romanian Cornilescu Version", "26"),
                ("    Romanian", "14"),
                ("    Romanian", "38"),
                ("Русский (RU)", "13"),
                ("    Russian Synodal Version", "13"),
                ("    Slovo Zhizny", "39"),
                ("Slovencina (SK)", "40"),
                ("    Nádej pre kazdého", "40"),
                ("Shqip (SQ)", "1"),
                ("    Albanian Bible", "1"),
                ("Svenska (SV)", "44"),
                ("    Svenska 1917", "7"),
                ("    Levande Bibeln", "44"),
                ("Kiswahili (SW)", "75"),
                ("    Swahili New Testament", "75"),
                ("Tagalog (TL)", "43"),
                ("    Ang Salita ng Diyos", "43"),
                ("Українська (UK)", "27"),
                ("    Ukranian Bible", "27"),
                ("Tiêng Viêt (VI)", "19"),
                ("    1934 Vietnamese Bible", "19"),
                ("汉语 (ZH)", "22"),
                ("    Chinese Union Version", "22")
            ])
        return configurator


    def __show_passage_timer(self):
        """Sends the citation and text to the display."""

        output = self._new_output()
        output.set("citation", self.__citation)
        output.set("text", self.__text)
        output.set("copyright", self.__copyright)
        output.set("verse_url", self.__verse_url)
        output.set("has_error", self.__has_error)
        output.set("has_text", self.__has_text)
        self._send_output(output)

        return 1


    def __load_passage_thread(self):
        """Loads the passage of the day and text in a background thread, and
           reloads it every time the version changes or a certain interval
           passes."""

        last_version_id = ""
        last_update = int(time.time())

        while (1):
            if (self._is_stopped()):
                break

            version_id = self._get_config("version_id")
            if (version_id != last_version_id or \
                int(time.time()) - last_update >= self.UPDATE_INTERVAL * 60):
                self.__load_passage()
                last_update = int(time.time())
                last_version_id = version_id

            time.sleep(1)


    def __load_passage(self):
        """Loads the verse of the day from biblegateway.com."""

        version_id = self._get_config("version_id")
        url = (self.VOTD_URL % vars())

        try:
            rss = feedparser.parse(url)
            if (len(rss["entries"]) == 1):
                item = rss["entries"][0]
            else:
                raise Exception, "There should be only one RSS item."
            self.__citation = item.title
            self.__copyright = rss.channel.copyright
            self.__text = self.__convert_for_display(item.content[0].value)

            # If this is an "Error" item, it will have no GUID.

            if (item.has_key("link")):
                self.__verse_url = item.link
            else:
                self.__verse_url = ""

            if (self.__citation.startswith("Error:")):
                self.__has_error = "true"
                self.__has_text = "false"
            else:
                self.__has_error = "false"
                self.__has_text = "true"

        except Exception, inst:

            # Sorry, I'm a Python newbie.  I could probably do this in a
            # shorter manner with tuples, but this works and is easy for my
            # puny brain to understand.

            self.__citation = ""
            self.__text = _("Could not read verse of the day:") + "\n" + str(inst)
            self.__copyright = ""
            self.__verse_url = ""
            self.__has_error = "true"
            self.__has_text = "false"
            print "Could not read verse of the day:"
            print inst


    def __convert_for_display(self, text):
        """Converts the given text for display in a label."""

        print "Bible text before munging:\n" + text

        # Get rid of double spaces.

        text = text.replace("  ", " ")

        # Get rid of common HTML character entities.

        text = text.replace("&ldquo;", "“")
        text = text.replace("&rdquo;", "”")
        text = text.replace("&lsquo;", "‘")
        text = text.replace("&rsquo;", "’")

        # Get rid of that annoying promotional message at the end of the
        # text.

        junkPos = text.find(self.JUNK_START)
        if (junkPos >= 0):
            text = text[0:junkPos]

        print "Bible text after munging:\n" + text

        return text


def new_sensor(args): return apply(BibleSensor, args)
