from utils.Observable import Observable
from utils.TypeConverter import TypeConverter
from utils.datatypes import *
from utils import actionparser


#
# Abstract class for data targets.
#
class DataTarget(Observable):

    def __init__(self, parent, display):

        # the actions and their associated calls
        self.__actions = {}

        # the index number of this target or -1
        self.__index = -1

        # the parent of this target
        self.__parent = parent

        # the display of this target
        self.__display = display

        # converter for data types
        self.__type_converter = TypeConverter()
        

        self._set_property_type("watch", TYPE_LIST)


    #
    # Returns the path of the .display file.
    #
    def get_path(self):  return self._get_display().get_path()



    #
    # Returns the parent of this target.
    #
    def _get_parent(self): return self.__parent



    def delete(self):

        self.drop_observers()
        del self.__actions
        del self.__parent
        del self.__display


    #
    # Returns the display of this target.
    #
    def _get_display(self): return self.__display



    #
    # Returns the sensor watch propagator of this target. This is either the
    # display window or an array.
    #
    def _get_propagator(self):

        parent = self._get_parent()
        while (parent and not hasattr(parent, "add_mapping")):
            parent = parent._get_parent()

        if (not parent): parent = self._get_display()

        return parent


    #
    # Sets the index of this target.
    #
    def set_index(self, value): self.__index = value


    #
    # Returns the index of this target.
    #
    def get_index(self): return self.__index



    #
    # Configures this target.
    #
    def set_config(self, key, value):

        if (key == "watch"):
            entries = value
            for e in entries:
                prop, sensorplug = e.split("=")
                self._get_propagator().add_mapping(sensorplug.strip(),
                                                self, prop.strip())
            #end for



    #
    # Associates the given action woth the given sensor call.
    #
    def set_action_call(self, action, call):

        self.__actions[action] = actionparser.parse(call)



    #
    # Emits the given action if the pointer is over this target.
    #
    def get_action_call(self, action): return self.__actions.get(action)



    def has_actions(self): return self.__actions



    #
    # Returns whether this target supports the given action.
    #
    def has_action(self, action): return self.__actions.has_key(action)


    #
    # Handles the given action.
    #
    def handle_action(self, action, px, py, path, args):

        if (self.get_index() != -1): path.append(self.get_index())
        if (self.has_action(action)):
            call = self.get_action_call(action)
            self._get_display().call_sensor( * [call, path] + args )


    def notify_handle_action(self, value, path): pass


    #
    # Sets the data type for the given property.
    #
    def _set_property_type(self, key, dtype):

        self.__type_converter.add_type(key, dtype)



    #
    # Converts a value to the data type of the given property. This function
    # is idem-potent.
    #
    def _convert_type(self, key, v):

        return self.__type_converter.str2type(key, v)
