from DisplayTarget import DisplayTarget
from utils.datatypes import *

import gtk


#
# Target where sensors can draw on.
# Transparent colors are not supported due to missing support in GDK.
#
class TargetCanvas(DisplayTarget):

    def __init__(self, parent, display):

        self.__buffer = None
        self.__buffer_gc = None
        self.__size = (0, 0)


        DisplayTarget.__init__(self, parent, display)

        self.__widget = gtk.Image()
        self.__widget.show()
        self.add(self.__widget)

        self._set_property_type("graphics", TYPE_LIST)



    #
    # Initializes the offscreen drawing buffer. The size is twice as big
    # because of antialiasing.
    #
    def __init_buffer(self, width, height):

        self.__size = (width, height)

        width = max(1, width) * 2
        height = max(1, height) * 2
        
        self.__buffer = gtk.gdk.Pixmap(gtk.gdk.get_default_root_window(),
                                       width, height)
        self.__buffer_gc = self.__buffer.new_gc()


        gdkcolor = gtk.gdk.color_parse("#000000")
        self.__buffer_gc.set_rgb_fg_color(gdkcolor)
        self.__buffer.draw_rectangle(self.__buffer_gc, 1,
                                     0, 0, width, height)



    #
    # Transforms the given coordinates into buffer space.
    #
    def __transform_coords(self, x, y):

        width, height = self.__buffer.get_size()
        tx = int( (width / 2.0)  * (1.0 + float(x)) )
        ty = int( (height / 2.0) * (1.0 - float(y)) )

        return (tx, ty)



    #
    # Performs the given drawing operations.
    # TODO: implement ellipses and pie slices.
    #
    def __draw(self, commands):

        w, h = self.get_geometry()[2:]
        if (not self.__buffer or (w, h) != self.__size):
            if (self.__buffer): print (w * 2, h * 2), self.__buffer.get_size()
            self.__init_buffer(w, h)
            self.__draw("clear")

        width, height = self.__buffer.get_size()

        for c in commands:
            if (not c): continue
            parts = c.split()
            cmd = parts[0]
            args = parts[1:]

            if (cmd == "clear"):
                gdkcolor = gtk.gdk.color_parse("black")
                self.__buffer_gc.set_rgb_fg_color(gdkcolor)
                self.__buffer.draw_rectangle(self.__buffer_gc, gtk.TRUE,
                                             0, 0, width, height)

            elif (cmd == "color"):
                color = args[0]
                gdkcolor = gtk.gdk.color_parse(color)
                if (gdkcolor.red == gdkcolor.green == gdkcolor.blue == 0):
                    gdkcolor = gtk.gdk.color_parse("#101010")
                self.__buffer_gc.set_rgb_fg_color(gdkcolor)

            elif (cmd == "line"):
                x1, y1, x2, y2 = args
                x1, y1 = self.__transform_coords(x1, y1)
                x2, y2 = self.__transform_coords(x2, y2)
                self.__buffer.draw_line(self.__buffer_gc, x1, y1, x2, y2)

            elif (cmd == "pie"):
                angle1, angle2, filled = args
                x, y = self.__transform_coords(-1.0, 1.0)
                w, h = self.__transform_coords(1.0, -1.0)
                self.__buffer.draw_arc(self.__buffer_gc, int(filled),
                                       x, y, w, h,
                                       long(angle1) * 64,
                                       long(angle2) * 64)

            elif (cmd == "polygon"):
                fill = args.pop()
                points = []
                while (args):
                    x = args.pop(0)
                    y = args.pop(0)
                    points.append(self.__transform_coords(x, y))
                #end while

                self.__buffer.draw_polygon(self.__buffer_gc, int(fill), points)

            elif (cmd == "rectangle"):
                x1, y1, x2, y2, fill = args
                x1, y1 = self.__transform_coords(x1, y1)
                x2, y2 = self.__transform_coords(x2, y2)
                w = x2 - x1
                h = y2 - y1
                self.__buffer.draw_rectangle(self.__buffer_gc, int(fill),
                                             x1, y1, w, h)

                
        #end for

        pbuf = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB, gtk.TRUE, 8,
                              width, height)
        pbuf.get_from_drawable(self.__buffer, self.get_colormap(),
                               0, 0, 0, 0, width, height)
        pbuf = pbuf.add_alpha(gtk. TRUE, chr(0), chr(0), chr(0))
        pbuf = pbuf.scale_simple(width / 2, height / 2, gtk.gdk.INTERP_HYPER)
        self.__widget.set_from_pixbuf(pbuf)
    
        

    def set_config(self, key, value):

        value = self._convert_type(key, value)

        if (key == "graphics"):
            self.__draw(value)

        else:
            DisplayTarget.set_config(self, key, value)
