import polling
import glibtop

import time
import os


class Network:

    def __init__(self):

        self.__time = 0
        self.__in   = 0
        self.__out  = 0


        self.__get_in_out = polling.wrap(self.__poll_net_info, .5, 0)
        self.__get_errors = polling.wrap(self.__poll_net_info, .5, 1)
        self.__get_ipaddr = polling.wrap(self.__poll_net_info, 10, 2)
        self.__get_subnet = polling.wrap(self.__poll_net_info, 10, 3)
        self.__get_flags  = polling.wrap(self.__poll_net_info, 10, 4)
        self.__get_mtu    = polling.wrap(self.__poll_net_info, 10, 5)
        self.get_devices  = polling.wrap(self.__poll_devices, 5)



    #
    #:function get_bytes_in_out | dev | Returns incoming and outgoing bytes
    #                                   for the given network device.
    #:  param dev | string | Network device
    #:  return Two int values: Incoming bytes and outgoing bytes.
    #:/function
    #
    def get_bytes_in_out(self, dev): return self.__get_in_out(dev)[0:2]



    #
    #:function get_pack_in_out | dev | Returns incoming and outgoing
    #                                  packages for the given network device.
    #:  param dev | string | Network device
    #:  return Two int values: Incoming packages and outgoing packages.
    #:/function
    #
    def get_pack_in_out(self, dev): return self.__get_in_out(dev)[2:4]



    #
    #:function get_speed_in_out | dev | Returns incoming and outgoing
    #                                   transfer rate for the given network
    #                                   device.
    #:  param dev | string | Network device
    #:  return Two int values: Incoming transfer rate and outgoing transfer
    #                          rate.
    #:/function
    #
    def get_speed_in_out(self, dev): return self.__get_in_out(dev)[4:6]



    #
    #:function get_errors_in_out | dev | Returns incoming and outgoing
    #                                    transfer errors for the given
    #                                    network device.
    #:  param dev | string | Network device
    #:  return Two int values: Incoming transfer errors and outgoing
    #                          transfer errors.
    #:/function
    #
    def get_errors_in_out(self, dev): return self.__get_errors(dev)[0:2]



    #
    #:function get_errors_total | dev | Returns total transfer errors
    #                                   for the given network device.
    #:  param dev | string | Network device
    #:  return One int values: Total transfer errors.
    #:/function
    #
    def get_errors_total(self, dev): return self.__get_errors(dev)[2]



    #
    #:function get_ipaddr | dev | Returns the IP address for the given
    #                             network device.
    #:  param dev | string | Network device
    #:  return IP address as string.
    #:/function
    #
    def get_ipaddr(self, dev): return self.__get_ipaddr(dev)



    #
    #:function get_subnet | dev | Returns the subnet address for the given
    #                             network device.
    #:  param dev | string | Network device
    #:  return Subnet address as string.
    #:/function
    #
    def get_subnet(self, dev): return self.__get_subnet(dev)


    #
    #:function get_flags | dev | Returns the flags for the given
    #                            network device.
    #:  param dev | string | Network device
    #:  return Flags as string.
    #:/function
    #
    def get_flags(self, dev): return self.__get_flags(dev)



    #
    #:function get_mtu | dev | Returns the MTU for the given
    #                          network device.
    #:  param dev | string | Network device
    #:  return MTU as int.
    #:/function
    #
    def get_mtu(self, dev): return self.__get_mtu(dev)



    #
    #:function get_devices | | Returns all available network devices.
    #                          Currently just works for FreeBSD and Linux.
    #:  param | |
    #:  return A list of available network devices.
    #:/function
    #
    def __poll_devices(self):

        import libdesklets as lib

        devices = []
        platform = lib.sys.get_os()
        
        if ("FreeBSD" == platform or "NetBSD" == platform):
            fd = os.popen("ifconfig -a | grep mtu", "r")
            data = fs.readlines()
            fd.close()

            for lines in data:
                fields = lines.strip().split(":")
                if (fields[0][:2] == "lo"):
                    continue
                else:
                    device = fields[0]
                    devices.append(device)

        elif ("Linux" == platform):
            fd = open("/proc/net/dev", "r")
            data = fd.readlines()
            fd.close()

            for lines in data[2:]:
                fields = lines.strip().split(":")
                if (fields[0][:2] == "lo"):
                    continue
                else:
                    device = fields[0]
                    devices.append(device)

        else: # (what about OpenBSD?)
            pass

        return devices



    def __poll_net_info(self, dev, mode):

        flags, mtu, subnet, address, pack_in, pack_out, pack_total, \
        bytes_in, bytes_out, bytes_total, err_in, err_out, err_total, \
        collisions = glibtop.get_netload(dev)

        if (mode == 0):
            speed_in    = 0
            speed_out   = 0
            t           = time.time()
            interval    = t - self.__time
            self.__time = t

            if ((self.__in == 0) and (self.__out == 0)):
                self.__in, self.__out = bytes_in, bytes_out
            else:
                in_diff  = bytes_in  - self.__in
                out_diff = bytes_out - self.__out
                speed_in = int(in_diff / interval)
                speed_out = int(out_diff / interval)

                self.__in, self.__out = bytes_in, bytes_out

            return (bytes_in, bytes_out, pack_in, pack_out, speed_in, speed_out)

        elif (mode == 1):
            return (err_in, err_out, err_total, collisions)

        elif (mode == 2):
            ip1 = address % 0x100; address /= 0x100
            ip2 = address % 0x100; address /= 0x100
            ip3 = address % 0x100; address /= 0x100
            ip4 = address

            return (str(ip1) + "." + str(ip2) + "." + str(ip3) + "." + str(ip4))

        elif (mode == 3):
            sub1 = subnet % 0x100; subnet /= 0x100
            sub2 = subnet % 0x100; subnet /= 0x100
            sub3 = subnet % 0x100; subnet /= 0x100
            sub4 = subnet

            return (str(sub1) + "." + str(sub2) + "." + str(sub3) + "." + str(sub4))

        elif (mode == 4):
            return flags

        elif (mode == 5):
            return mtu
