/******************************************************************************/
/* Copyright (C) 2003 Benoît Dejean <bnet@ifrance.com>			      */
/* 									      */
/* This program is free software; you can redistribute it and/or modify	      */
/* it under the terms of the GNU General Public License as published by	      */
/* the Free Software Foundation; either version 2 of the License, or          */
/* (at your option) any later version.					      */
/* 									      */
/* This program is distributed in the hope that it will be useful,	      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of	      */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	      */
/* GNU General Public License for more details.				      */
/* 									      */
/* You should have received a copy of the GNU General Public License	      */
/* along with this program; if not, write to the Free Software		      */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  */
/* 									      */
/* http:www.gnu.org/licenses/gpl.html					      */
/******************************************************************************/


/* Benoît Dejean bnet at ifrance dot com */

#include <stddef.h>

#include <Python.h>

#include <glibtop.h>
#include <glibtop/cpu.h>
#include <glibtop/loadavg.h>
#include <glibtop/mem.h>
#include <glibtop/swap.h>
#include <glibtop/ppp.h>
#include <glibtop/netload.h>
#include <glibtop/uptime.h>
#include <glibtop/proclist.h>
#include <glibtop/procstate.h>
#include <glibtop/procuid.h>
#include <glibtop/procmem.h>
#include <glibtop/proctime.h>
#include <glibtop/procsignal.h>
#include <glibtop/prockernel.h>
#include <glibtop/procsegment.h>
#include <glibtop/procargs.h>
#include <glibtop/procmap.h>
#include <glibtop/mountlist.h>
#include <glibtop/fsusage.h>


/* for glibtop_free 
   not compatible with 2.5 versions */

#if !defined(LIBGTOP_VERSION_CODE) || LIBGTOP_VERSION_CODE < 2005000
#include <glibtop/xmalloc.h>
#else
#include <glib.h>
#define glibtop_free g_free
#endif


/***********************************************/
/*** The following macros deal with Py_Tuple ***/
/***********************************************/

/* it is not possible to use PyBuild_Value because 
   "K" (integer) [unsigned PY_LONG_LONG] is not available in python2.2
   PyBuild_Value is used every time possible */

/* Python API sux :D */


#define FILL_TUPLE(tuple, converter, tab, len) do { \
size_t i=0; \
while(i < (len)) \
{ \
PyTuple_SetItem(tuple, i, (converter)((tab)[i])); \
++i; \
} \
} \
while(0)


#define FILL_TUPLE2(tuple, converter, ptab, len) do { \
size_t i=0; \
while(i < (len)) \
{ \
PyTuple_SetItem(tuple, i, (converter)((ptab)+i)); \
++i; \
} \
} \
while(0)


#define ADD_TUPLE(tuple, index, converter, tab, len) do { \
PyObject * const inner = PyTuple_New(len); \
FILL_TUPLE(inner, converter, tab, len); \
PyTuple_SetItem(tuple, index, inner); \
} \
while(0)


/* end macros */



/***************************/
/* macros to shorten lines */
/***************************/

#define PyL_ULL PyLong_FromUnsignedLongLong
#define PyL_UL  PyLong_FromUnsignedLong
#define PyF_D   PyFloat_FromDouble
#define PyS_S   PyString_FromString
#define PyS_SaS PyString_FromStringAndSize
#define PyI_L   PyInt_FromLong
 
/* end macros */

PyObject* get_cpu()
{
  PyObject * const t = PyTuple_New(11);

  glibtop_cpu buf;
  glibtop_get_cpu(&buf);

  PyTuple_SetItem(t,  0, PyL_ULL (buf.total));
  PyTuple_SetItem(t,  1, PyL_ULL (buf.user));
  PyTuple_SetItem(t,  2, PyL_ULL (buf.nice));
  PyTuple_SetItem(t,  3, PyL_ULL (buf.sys));
  PyTuple_SetItem(t,  4, PyL_ULL (buf.idle));
  PyTuple_SetItem(t,  5, PyL_ULL (buf.frequency));
  ADD_TUPLE(      t,  6, PyL_ULL, buf.xcpu_total, GLIBTOP_NCPU);
  ADD_TUPLE(      t,  7, PyL_ULL, buf.xcpu_user,  GLIBTOP_NCPU);
  ADD_TUPLE(      t,  8, PyL_ULL, buf.xcpu_nice,  GLIBTOP_NCPU);
  ADD_TUPLE(      t,  9, PyL_ULL, buf.xcpu_sys,   GLIBTOP_NCPU); 
  ADD_TUPLE(      t, 10, PyL_ULL, buf.xcpu_idle,  GLIBTOP_NCPU);

  return t;  
}



PyObject* get_mem()
{
  PyObject * const t = PyTuple_New(8);

  glibtop_mem buf;
  glibtop_get_mem(&buf);

  PyTuple_SetItem(t, 0, PyL_ULL(buf.total));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.used));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.free));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.shared));
  PyTuple_SetItem(t, 4, PyL_ULL(buf.buffer));
  PyTuple_SetItem(t, 5, PyL_ULL(buf.cached));
  PyTuple_SetItem(t, 6, PyL_ULL(buf.user));
  PyTuple_SetItem(t, 7, PyL_ULL(buf.locked));

  return t;
}



PyObject* get_swap()
{
  PyObject * const t = PyTuple_New(5);

  glibtop_swap buf;
  glibtop_get_swap(&buf);

  PyTuple_SetItem(t, 0, PyL_ULL(buf.total));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.used));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.free));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.pagein));
  PyTuple_SetItem(t, 4, PyL_ULL(buf.pageout));

  return t;
}



PyObject* get_uptime()
{
  glibtop_uptime buf;
  glibtop_get_uptime(&buf);

  return Py_BuildValue((char*)"(dd)", buf.uptime, buf.idletime);
}



PyObject* get_loadavg()
{
  PyObject * const t = PyTuple_New(4);

  glibtop_loadavg buf;
  glibtop_get_loadavg(&buf);  

  PyTuple_SetItem(t, 0, Py_BuildValue((char*)"(ddd)",
				      buf.loadavg[0],
				      buf.loadavg[1],
				      buf.loadavg[2]
				      ));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.nr_running));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.nr_tasks));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.last_pid));

  return t;  
}



PyObject* get_netload(const char *interface)
{
  PyObject * const t = PyTuple_New(14);
     
  glibtop_netload buf;
  glibtop_get_netload(&buf, interface);      

  PyTuple_SetItem(t,  0, PyL_ULL(buf.if_flags));
  PyTuple_SetItem(t,  1, PyL_ULL(buf.mtu));
  PyTuple_SetItem(t,  2, PyL_ULL(buf.subnet));
  PyTuple_SetItem(t,  3, PyL_ULL(buf.address));
  PyTuple_SetItem(t,  4, PyL_ULL(buf.packets_in));
  PyTuple_SetItem(t,  5, PyL_ULL(buf.packets_out));
  PyTuple_SetItem(t,  6, PyL_ULL(buf.packets_total));
  PyTuple_SetItem(t,  7, PyL_ULL(buf.bytes_in));
  PyTuple_SetItem(t,  8, PyL_ULL(buf.bytes_out));
  PyTuple_SetItem(t,  9, PyL_ULL(buf.bytes_total));
  PyTuple_SetItem(t, 10, PyL_ULL(buf.errors_in));
  PyTuple_SetItem(t, 11, PyL_ULL(buf.errors_out));
  PyTuple_SetItem(t, 12, PyL_ULL(buf.errors_total));
  PyTuple_SetItem(t, 13, PyL_ULL(buf.collisions));

  return t;       
}



PyObject* get_ppp(unsigned short device)
{
  PyObject * const t = PyTuple_New(3);

  glibtop_ppp buf;
  glibtop_get_ppp(&buf, device);
      
  PyTuple_SetItem(t, 0, PyL_ULL(buf.state));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.bytes_in));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.bytes_out));
      
  return t;
}



PyObject* get_proclist(int64_t which, long long arg)
{
  glibtop_proclist buf;
  const unsigned * const list = glibtop_get_proclist(&buf, which, arg);      
  
  {
    PyObject * const t = PyTuple_New(buf.number);
	
    FILL_TUPLE(t, PyL_UL, list, buf.number);
	
    glibtop_free((void*)list);

    return t;  
  }        
}



PyObject* get_proc_state(long pid)
{
  glibtop_proc_state buf;
  glibtop_get_proc_state(&buf, pid);

  return Py_BuildValue((char*)"(scii)", buf.cmd, buf.state, buf.uid, buf.gid);       
}



PyObject* get_proc_uid(long pid)
{
  glibtop_proc_uid buf;
  glibtop_get_proc_uid(&buf, pid);

  return Py_BuildValue((char*)"(iiiiiiiiiiii)",
		       buf.uid,
		       buf.euid,
		       buf.gid,
		       buf.egid,
		       buf.pid,
		       buf.ppid,
		       buf.pgrp,
		       buf.session,
		       buf.tty,
		       buf.tpgid,
		       buf.priority,
		       buf.nice
		       );
}



PyObject* get_proc_mem(long pid)
{
  PyObject * const t = PyTuple_New(6);

  glibtop_proc_mem buf;  
  glibtop_get_proc_mem(&buf, pid);      
 
  PyTuple_SetItem(t, 0, PyL_ULL(buf.size));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.vsize));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.resident));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.share));
  PyTuple_SetItem(t, 4, PyL_ULL(buf.rss));
  PyTuple_SetItem(t, 5, PyL_ULL(buf.rss_rlim));
  
  return t;
}



PyObject* get_proc_time(long pid)
{
  PyObject * const t = PyTuple_New(11);

  glibtop_proc_time buf;  
  glibtop_get_proc_time(&buf, pid);
 
  PyTuple_SetItem(t,  0, PyL_ULL (buf.start_time));
  PyTuple_SetItem(t,  1, PyL_ULL (buf.rtime));
  PyTuple_SetItem(t,  2, PyL_ULL (buf.utime));
  PyTuple_SetItem(t,  3, PyL_ULL (buf.stime));
  PyTuple_SetItem(t,  4, PyL_ULL (buf.cutime));
  PyTuple_SetItem(t,  5, PyL_ULL (buf.cstime));
  PyTuple_SetItem(t,  6, PyL_ULL (buf.timeout));
  PyTuple_SetItem(t,  7, PyL_ULL (buf.it_real_value));
  PyTuple_SetItem(t,  8, PyL_ULL (buf.frequency));
  ADD_TUPLE(      t,  9, PyL_ULL, buf.xcpu_utime, GLIBTOP_NCPU);
  ADD_TUPLE(      t, 10, PyL_ULL, buf.xcpu_stime, GLIBTOP_NCPU);

  return t;
}



PyObject* get_proc_signal(long pid)
{
  PyObject * const t = PyTuple_New(4);
      
  glibtop_proc_signal buf;  
  glibtop_get_proc_signal(&buf, pid);
      
  ADD_TUPLE(t, 0, PyL_ULL, buf.signal,    2);
  ADD_TUPLE(t, 1, PyL_ULL, buf.blocked,   2);
  ADD_TUPLE(t, 2, PyL_ULL, buf.sigignore, 2);
  ADD_TUPLE(t, 3, PyL_ULL, buf.sigcatch,  2);
      
  return t;
}



PyObject* get_proc_kernel(long pid)
{
  PyObject * const t = PyTuple_New(9);

  glibtop_proc_kernel buf;  
  glibtop_get_proc_kernel(&buf, pid);
 
  PyTuple_SetItem(t, 0, PyL_ULL(buf.k_flags));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.min_flt));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.maj_flt));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.cmin_flt));
  PyTuple_SetItem(t, 4, PyL_ULL(buf.cmaj_flt));
  PyTuple_SetItem(t, 5, PyL_ULL(buf.kstk_esp));
  PyTuple_SetItem(t, 6, PyL_ULL(buf.kstk_eip));
  PyTuple_SetItem(t, 7, PyL_ULL(buf.nwchan));
  PyTuple_SetItem(t, 8, PyS_S  (buf.wchan));

  return t;
}



PyObject* get_proc_segment(long pid)
{
  PyObject * const t = PyTuple_New(8);

  glibtop_proc_segment buf;  
  glibtop_get_proc_segment(&buf, pid);

  PyTuple_SetItem(t, 0, PyL_ULL(buf.text_rss));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.shlib_rss));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.data_rss));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.stack_rss));
  PyTuple_SetItem(t, 4, PyL_ULL(buf.dirty_size));
  PyTuple_SetItem(t, 5, PyL_ULL(buf.start_code));
  PyTuple_SetItem(t, 6, PyL_ULL(buf.end_code)); 
  PyTuple_SetItem(t, 7, PyL_ULL(buf.start_stack));

  return t;
}



PyObject* get_proc_args(long pid)
{
  glibtop_proc_args buf;
  const char * const args0= glibtop_get_proc_args(&buf, pid, 0u);

  /* splitting args0 */
  /* args0 [......0..   ....0.........0] */

  size_t argc = 0;
  const char **argv = PyMem_Malloc(42 * sizeof *argv);
  size_t allocated = 42;

  {
    const char *arg = NULL;

    for(arg = args0; arg != (args0 + buf.size); )
      {
	if(argc==allocated)
	  {
	    argv = PyMem_Realloc(argv, 2 * allocated * sizeof *argv);
	    allocated *= 2;
	  }

	argv[argc] = arg;
	++argc;
	arg += strlen(arg)+1;
      }
  }

  /* building tuple */
  {
    PyObject * const t = PyTuple_New(argc);

    FILL_TUPLE(t, PyS_S, argv, argc);	

    PyMem_Free(argv);
    glibtop_free((void*)args0);
  
    return t;
  }    
}



static PyObject* map_entry_to_PyTuple(const glibtop_map_entry *e)
{
  PyObject * const t = PyTuple_New(7);
 
  PyTuple_SetItem(t, 0, PyL_ULL(e->start));
  PyTuple_SetItem(t, 1, PyL_ULL(e->end));
  PyTuple_SetItem(t, 2, PyL_ULL(e->offset));
  PyTuple_SetItem(t, 3, PyL_ULL(e->perm));
  PyTuple_SetItem(t, 4, PyL_ULL(e->inode));
  PyTuple_SetItem(t, 5, PyL_ULL(e->device));
  PyTuple_SetItem(t, 6, PyS_S  (e->filename));

  return t;
}


PyObject * get_proc_map(long pid)
{
  glibtop_proc_map buf;
  const glibtop_map_entry * const entries = glibtop_get_proc_map(&buf, pid);  

  {      
    PyObject * const t = PyTuple_New(buf.number);

    FILL_TUPLE2(t, map_entry_to_PyTuple, entries, buf.number);

    glibtop_free((void*)entries);

    return t;
  }
    
}



static PyObject* mountentry_to_PyTuple(const glibtop_mountentry *e)
{
  PyObject * const t = PyTuple_New(4);

  PyTuple_SetItem(t, 0, PyL_ULL(e->dev));  
  PyTuple_SetItem(t, 1, PyS_S  (e->devname));
  PyTuple_SetItem(t, 2, PyS_S  (e->mountdir));
  PyTuple_SetItem(t, 3, PyS_S  (e->type));

  return t;
}


PyObject* get_mountlist(int allfs)
{
  glibtop_mountlist buf;
  const glibtop_mountentry * const entries = glibtop_get_mountlist(&buf, allfs);
      
  {
    PyObject * const t = PyTuple_New(buf.number);
	
    FILL_TUPLE2(t, mountentry_to_PyTuple, entries, buf.number);
	
    glibtop_free((void*)entries);
	
    return t;
  }
}



PyObject* get_fsusage(const char *mount_dir)
{
  PyObject * const t = PyTuple_New(5);
      
  glibtop_fsusage buf;  
  glibtop_get_fsusage(&buf, mount_dir);
      
  PyTuple_SetItem(t, 0, PyL_ULL(buf.blocks));
  PyTuple_SetItem(t, 1, PyL_ULL(buf.bfree));
  PyTuple_SetItem(t, 2, PyL_ULL(buf.bavail));
  PyTuple_SetItem(t, 3, PyL_ULL(buf.files));
  PyTuple_SetItem(t, 4, PyL_ULL(buf.ffree));

  return t;    
}
