from DisplayTarget import DisplayTarget
from utils import dialog
from utils.datatypes import *
from utils import Unit
from main import _

import gobject
import gtk


#
# Abstract class for targets that can contain others.
#
class ContainerTarget(DisplayTarget):

    __slots__ = ('__children', '__children_id', '__ids', '__relatives')

    def __init__(self, name, parent):

        # the list is necessary for preserving order
        self.__children = []

        # mapping: id -> child
        self.__children_id = {}

        # mapping: child -> id
        self.__ids = {}

        # the relative-relations: id_of_relative -> [child]
        self.__relatives = {}

        DisplayTarget.__init__(self, name, parent)



    #
    # Observer callback for watching children.
    #
    def child_observer(self, src, cmd):

        if (cmd == src.OBS_GEOMETRY):
            # handle relative positioning
            if (self.__relatives):
                try:
                    ident = self.get_id_by_child(src)
                    relatives = self.__relatives[ident]
                    for r in relatives:
                        r.set_relative_to(src)
                except:
                    pass



    #
    # Sets the relative of a child.
    #
    def set_relative(self, child, relative_id):

        if (not relative_id in self.__relatives):
            self.__relatives[relative_id] = []
        self.__relatives[relative_id].append(child)
        try:
            relative = self.get_child_by_id(relative_id)
            child.set_relative_to(relative)
        except:
            pass



    #
    # Creates a new child of the given type.
    #
    def new_child(self, childtype, settings, children):

        import targetregistry
        cid = settings["id"]
        try:
            child = targetregistry.create(childtype, self)
        except KeyError:
            return
        self._register_child(child, cid)
        child.get_widget().show()

        for t, s, c in children:
            child.new_child(t, s, c)

        for key, value in settings.items():
            child.set_xml_prop(key, value)

        return child



    def _register_child(self, child, cid):

        self.__children_id[cid] = child
        self.__ids[child] = cid
        self.__children.append(child)
        self._get_display().add_target_to_script(cid, child)



    def _unregister_child(self, child):

        cid = self.__ids[child]

        try:
            del self.__children_id[cid]
        except KeyError:
            pass

        try:
            del self.__ids[child]
        except KeyError:
            pass

        self.__children.remove(child)



    def _get_children(self): return self.__children[:]



    def _get_child(self):

        children = self._get_children()
        if (not children): return
        elif (len(children) > 1):
            name = "???"
            log("The %(name)s container accepts only one child." % vars())
            return children[0]

        else: return children[0]



    def get_child_by_id(self, ident): return self.__children_id.get(ident)
    def get_id_by_child(self, child): return self.__ids.get(child)



    def delete(self):

        DisplayTarget.delete(self)
        for c in self._get_children():
            c.delete()
            self._unregister_child(c)
        self.__children = []
        self.__children_id.clear()
        self.__ids.clear()



    #
    # Override this method if the container size differs from the target size.
    #
    def get_container_geometry(self):

        return self.get_geometry()



    #
    # Returns the border size of this container. Override this method to return
    # the sizes of the four borders (left, top, right, bottom).
    #
    def get_border_size(self):

        return (Unit.ZERO, Unit.ZERO, Unit.ZERO, Unit.ZERO)



    #
    # Returns the bounding box occupied by the children of this container.
    #
    def __get_children_bbox(self):

        bx1 = by1 = Unit.MAXIMUM
        bx2 = by2 = Unit.ZERO
        for c in self._get_children():
            if (c.is_standalone()): continue

            cx, cy, cw, ch = c.get_geometry()
            ux, uy, uw, uh = c.get_user_geometry()

            if (uw.get_unit() != Unit.UNIT_PERCENT):
                bx2 = max(bx2, cw)
            if (uh.get_unit() != Unit.UNIT_PERCENT):
                by2 = max(by2, ch)

            if (ux.get_unit() != Unit.UNIT_PERCENT):
                bx1 = min(bx1, cx)
                if (uw.get_unit() != Unit.UNIT_PERCENT):
                    bx2 = max(bx2, cx + cw)

            if (uy.get_unit() != Unit.UNIT_PERCENT):
                by1 = min(by1, cy)
                if (uh.get_unit() != Unit.UNIT_PERCENT):
                    by2 = max(by2, cy + ch)

        bx1 = min(bx1, bx2)
        by1 = min(by1, by2)
        bx2 = max(bx1, bx2)
        by2 = max(by1, by2)

        return (bx1, by1, bx2 - bx1, by2 - by1)



    #
    # Containers which set index values for the children override this method.
    #
    def get_next_child_index(self): return -1



    #
    # Sets the size of this container.
    #
    def set_size(self, width, height):
        assert (isinstance(width, Unit.Unit))
        assert (isinstance(height, Unit.Unit))

        DisplayTarget.set_size(self, width, height)



    #
    # Returns the geometry value that should be used for unset values.
    #
    def _get_value_for_unset(self, x = None, y = None,
                             width = None, height = None):

        bx, by, bw, bh = self.__get_children_bbox()
        bw1, bh1, bw2, bh2 = self.get_border_size()
        bw += bw1 + bw2
        bh += bh1 + bh2

        if (x): return x
        elif (y): return y
        elif (width): return bx + bw
        elif (height): return by + bh



    #
    # Tells other targets about geometry changes.
    #
    def _propagate_geometry(self):

        parent = self._get_parent()
        if (parent != self._get_display() and not self.is_standalone()):
            parent.request_adjust_geometry()

        # tell percentual children about the new size
        for c in self._get_children():
            if (c.is_standalone()): continue

            ux, uy, uw, uh = c.get_user_geometry()
            if (Unit.UNIT_PERCENT in (ux.get_unit(), uy.get_unit(),
                                      uw.get_unit(), uh.get_unit())):
                c.request_adjust_geometry()

        self.update_observer(self.OBS_GEOMETRY)



    def handle_action(self, action, px, py, args):

        DisplayTarget.handle_action(self, action, px, py, args)

        bw, bn, be, bs = self.get_border_size()
        children = self._get_children()
        px -= bw; py -= bn
        for c in children:
            if (c.is_standalone()): continue

            cx, cy, cw, ch = c.get_geometry()
            if (cx <= px <= cx + cw and cy <= py <= cy + ch):
                c.handle_action(action, px - cx, py - cy, args)
                # don't propagate True when leaving the window
                if (action != self.ACTION_LEAVE):
                    c.notify_handle_action(True)
            else:
                c.notify_handle_action(False)
        #end for



    #
    # Propagate LEAVE events to all children.
    #
    def notify_handle_action(self, value):

        DisplayTarget.notify_handle_action(self, value)
        if (value == False):
            for c in self._get_children():
                if (c.is_standalone()): continue
                c.notify_handle_action(False)
        #end if

