from config.DisplayConfigger import DisplayConfigger
from config.StateSaver import DefaultStateSaver
from DisplayTarget import DisplayTarget
from TargetGroup import TargetGroup
from main import UNSET_COORD
from scripting.Script import Script
from scripting.Scriptlet import Scriptlet
from utils import actionparser
from utils.Struct import Struct
from utils.Observable import Observable
from utils import Unit
from utils import vfs
import utils

import utils.dialog

import gobject
import gtk
import time
import random
import sys
import os
import re


# The name of the namespace where all display element are accessible.
_ROOT = "Dsp"


#
# Class for display windows.
#
class Display(gtk.HBox, Observable):

    __slots__ = ('__sensor_controls', '__arrays', '__sensor_menu',
                 '__group', '__script',
                 '__configurator', '__path', '__display_file', '__id',
                 '__last_pointer_pos', '__pointer_pos', '__is_sensitive',
                 '__menu')

    # observer commands
    OBS_CLOSE = 0
    OBS_RESTART = 1
    OBS_GEOMETRY = 2
    OBS_SHAPE = 3
    OBS_TITLE = 4
    OBS_ICON = 5
    OBS_FLAGS = 6
    OBS_CLOSED = 7


    # event types
    EVENT_MOTION = 0
    EVENT_PRESS = 1
    EVENT_RELEASE = 2
    EVENT_KEY_PRESS = 3
    EVENT_KEY_RELEASE = 4
    EVENT_LEAVE = 5
    EVENT_SCROLL = 6


    # regular expression to test whether a path is absolute
    __IS_ABSOLUTE_PATH_RE = re.compile("[a-zA-Z]+://.+")

    def __init__(self, ident, rep):

        # the display menu
        self.__DISPLAY_MENU = [
            (_("_Configure desklet"), self.__handle_configure),
            (_("_Move desklet"), self.__handle_move),
            ("", None),
            (_("_View Source"), self.__handle_source),
            ("", None),
            (_("Re_start desklet"), self.__handle_restart),
            (_("_Remove desklet"), self.__handle_remove)]


        # the controls handling the deprecated sensor stuff: id -> control
        self.__sensor_controls = {}

        # mapping of the elements to their parent arrays: child_id -> array_id
        self.__arrays = {}

        # the deprecated sensor menu
        # FIXME: remove eventually
        self.__sensor_menu = []

        # content of this display (valid until display has been initialized)
        self.__content = None

        # scriptlets of this display (valid until display has been initialized)
        self.__scriptlets = []

        # the root TargetGroup
        self.__group = None

        # the scripting environment
        self.__script = Script(ident, rep)

        # the configurator object
        self.__configurator = DisplayConfigger(ident)
        self.__configurator.set_scripting_environment(self.__script)

        # the path of the .display file
        self.__path = os.path.dirname(rep)
        self.__display_file = rep

        # the unique ID of this display
        self.__id = ident

        # the last position of the mouse pointer (used for filtering out
        # unwanted events)
        self.__last_pointer_pos = (-1, -1)

        # mapping between sensors and targets; which target watches
        # which sensor?
        # (sensor, port) -> (target, property)
        self.__mapping = {}

        # temporary data for remembering the position of the last mouse click
        self.__pointer_pos = (0, 0)

        # whether the display reacts on events
        self.__is_sensitive = True

        # the menu to open
        self.__menu = None

        gtk.HBox.__init__(self)



    #
    # Returns the unique ID of this display.
    #
    def get_id(self): return self.__id

    def get_next_child_index(self): return -1

    def get_index_path(self): return []



    #
    # Initializes the display. Call this after you have finished constructing
    # it.
    #
    def initialize(self):
        assert self.__content

        childtype, settings, children = self.__content

        # load saved positions
        positions = DefaultStateSaver().get_key("positions", {})
        lx, ly = positions.get(self.get_id(), (UNSET_COORD, UNSET_COORD))
        x = settings.get("x", lx)
        y = settings.get("y", ly)
        if (x != UNSET_COORD): settings["x"] = x
        if (y != UNSET_COORD): settings["y"] = y

        # build widget hierarchy
        self.new_child(childtype, settings, children)
        self.add(self.__group.get_widget())

        # run initial scripts
        for s in self.__scriptlets:
            self.execute_script(s)

        # load stored configuration
        self.__configurator.load_config()



    #
    # Sets the content of the display.
    #
    def set_content(self, childtype, settings, children):

        self.__content = (childtype, settings, children)



    #
    # Sets metadata.
    #
    def set_metadata(self, author = "", version = "", name= "", preview = ""):

        # a banner without the display's name would look crappy
        if (name):
            if (preview): preview = self.get_full_path(preview)
            self.__configurator.set_banner(preview,
                                  "<big>%s</big> %s\n"
                                  "<small>%s</small>" % (name, version, author))



    #
    # Adds the given scriptlet.
    #
    def add_scriptlet(self, code, filename):

        scriptlet = Scriptlet(code, filename)
        #self.execute_script(scriptlet)
        self.__scriptlets.append(scriptlet)



    #
    # Executes the given script.
    #
    def execute_script(self, code):

        self.__script.execute(code)



    #
    # Executes the given callback script.
    #
    def execute_callback_script(self, code, this):

        scriptlet = Scriptlet(code, self.__display_file)
        self.__script.add_element(None, "self", this)
        self.__script.execute(scriptlet)



    #
    # Adds the given target to the scripting environment.
    #
    def add_target_to_script(self, name, target):

        index_path = target.get_index_path()
        length = len(index_path)

        if ("#" in name): name = name[:name.find("#")]
        if (length > 0):
            self.__script.add_element_with_path(_ROOT, name, target, index_path)
        else:
            self.__script.add_element(_ROOT, name, target)



    #
    # Builds the configurator.
    #
    def build_configurator(self, items):

        self.__configurator.build(items)
        self.__configurator.set_path(self.__path)



    #
    # Sends an event to be executed to the display.
    #
    def send_event(self, etype, *args):

        if (not self.__is_sensitive): return
        w, h = self.__group.get_widget().size_request()
        lx, ly = self.__pointer_pos


        if (etype == self.EVENT_MOTION):
            x, y = args

            utils.request_call(self.__queue_motion, x, y, w, h, False)
            return


        elif (etype == self.EVENT_LEAVE):
            utils.request_call(self.__queue_motion, 0, 0, w, h, True)
            return


        elif (etype == self.EVENT_PRESS):
            button, x, y, counter = args

            self.__pointer_pos = (x, y)
            if (counter == 1):
                action = DisplayTarget.ACTION_PRESS
            else:
                action = DisplayTarget.ACTION_DOUBLECLICK

            event = Struct(button = button, _args = [button])


        elif (etype == self.EVENT_RELEASE):
            button, x, y = args

            if (button == 1):
                if (abs(lx - x) < 10 and abs(ly - y) < 10):
                    action = DisplayTarget.ACTION_CLICK
                else:
                    action = DisplayTarget.ACTION_RELEASE
            elif (button == 2):
                return
            elif (button == 3):
                action = DisplayTarget.ACTION_MENU
            else:
                return
            event = Struct(button = button, _args = [button])


        elif (etype == self.EVENT_SCROLL):
            direction, x, y = args

            if (direction == gtk.gdk.SCROLL_UP):
                direction = 0
            elif (direction == gtk.gdk.SCROLL_DOWN):
                direction = 1
            else:
                direction = -1
            action = DisplayTarget.ACTION_SCROLL
            event = Struct(direction = direction, _args = [direction])


        elif (etype == self.EVENT_KEY_PRESS):
            key, x, y = args
            action = DisplayTarget.ACTION_KEY_PRESS
            event = Struct(key = key)


        elif (etype == self.EVENT_KEY_RELEASE):
            key, x, y = args
            action = DisplayTarget.ACTION_KEY_RELEASE
            event = Struct(key = key)


        else:
            # what kind of event did we get there?
            assert (False)  # never reached

        self.__group.handle_action(action,
                                   Unit.Unit(x, Unit.UNIT_PX),
                                   Unit.Unit(y, Unit.UNIT_PX),
                                   event)
        self.__group.notify_handle_action(True)

        # extend the menu or create one if there's none
        if (action == DisplayTarget.ACTION_MENU):
            self.__make_menu(0, 0)



    #
    # Sets the sensitive flag of the display. Insensitive displays don't react
    # on user events.
    #
    def set_sensitive(self, value):

        self.__is_sensitive = value



    #
    # Returns the path of the .display file.
    #
    def get_path(self):

        return self.__path



    #
    # Returns the full path of the given path which may be relative to the
    # .display file.
    #
    def get_full_path(self, path):

        # a path is absolute iff it starts with "/" or with a protocol name
        # such as "http://", otherwise it's relative
        if (path.startswith("/") or self.__IS_ABSOLUTE_PATH_RE.match(path)):
            return path
        else:
            return os.path.join(self.__path, path)



    #
    # Returns the display.
    #
    def _get_display(self): return self



    def new_child(self, childtype, settings, children):

        import targetregistry
        # we don't catch the KeyError here, but in the DisplayFactory
        try:
            self.__group = targetregistry.create(childtype, self)
        except KeyError, exc:
            log(exc)
        self.__group.get_widget().show()
        cid = settings["id"]
        self.add_target_to_script(cid, self.__group)

        for t, s, c in children:
            self.__group.new_child(t, s, c)

        for key, value in settings.items():
            self.__group.set_xml_prop(key, value)



    #
    # Opens the configuration dialog for this display.
    #
    def __open_configurator(self):
        assert (self.__configurator)

        configurators = [s.configurator
                         for s in self.__sensor_controls.values()]

        if (configurators):
            # support old deprecated sensor stuff
            from DisplayConfigurator import DisplayConfigurator
            dconf = DisplayConfigurator(configurators)

        else:
            self.__configurator.show()



    #
    # Saves this display's position.
    #
    def __save_position(self):

        x, y, nil, nil = self.__group.get_user_geometry()
        positions = DefaultStateSaver().get_key("positions", {})
        positions[self.get_id()] = (x.as_px(), y.as_px())
        DefaultStateSaver().set_key("positions", positions)



    #
    # Removes this display.
    #
    def remove_display(self):

        # save the display's position
        self.__save_position()

        self.__configurator.destroy()
        self.update_observer(self.OBS_CLOSED, self.__id)
        self.drop_observers()

        self.__script.stop()
        for s in self.__sensor_controls.values():
            s.stop = True

        self.__group.delete()

        del self.__sensor_controls
        del self.__mapping
        del self.__group



    #
    # Purges this display.
    #
    def purge_display(self):

        # remove the display's position
        positions = DefaultStateSaver().get_key("positions", {})
        try:
            del positions[self.get_id()]
        except KeyError:
            # position wasn't stored
            pass

        # clean up scripting environment
        self.__script.remove()

        # remove the configuration
        self.__configurator.remove_config()



    #
    # Sends the given action with an event object to the display.
    #
    def send_action(self, src, action, event):

        call = src.get_action_call(action)
        if (call):

            # analyze call to see if it's a legacy call
            # FIXME: remove eventually :)
            if (re.match("\w+:.*", call)):
                log("Deprecation: Please use new style call.",
                    is_warning = True)

                try:
                    legacy_args = event._args
                except:
                    legacy_args = []

                legacy_call = actionparser.parse(call)
                path = src.get_index_path()
                self.call_sensor(legacy_call, path, *legacy_args)

            else:
                src.notify_handle_action(True)
                src._setp("event", event)
                self.execute_callback_script(call, src)



    #
    # Has to be called when a menu is to open. Do not open the menu yourself.
    #
    def queue_menu(self, menu):

        self.__menu = menu



    #
    # Builds the menu.
    #
    def __make_menu(self, button, eventtime):

        need_separator = False
        if (not self.__menu):
            ident = Display.make_id()
            self.__menu = self.__group.new_child("menu",
                                                 {"id": ident},
                                                 [])
            self.__group.set_prop("on-menu", "%s.%s.popup()" % (_ROOT, ident))

        elif (not "display" in self.__menu.get_slots()):
            # add separator
            need_separator = True
        #end if

        # FIXME: remove eventually :)
        if (self.__sensor_menu and
            not "deprecated sensor menu" in self.__menu.get_slots()):
            for entry in self.__sensor_menu:
                if (not entry):
                    label, active, submenu, h, args = \
                           "", True, None, None, []
                else:
                    label, active, submenu, h, args = entry

                item = self.__menu.new_child("menu-item",
                                             {"id": Display.make_id(),
                                              "slot": "deprecated sensor menu",
                                              "label": label},
                                             [])
                args = [item.get_widget()] + list(args)
                if (h): item.set_handler(h, *args)
                if (not active): item.set_prop("active", False)
        #end if

        if (not "display" in self.__menu.get_slots()):
            if (need_separator):
                self.__menu.new_child("menu-item",
                                      {"id": Display.make_id(),
                                       "slot": "display"}, [])

            for label, h in self.__DISPLAY_MENU:
                item = self.__menu.new_child("menu-item",
                                             {"id": Display.make_id(),
                                              "slot": "display",
                                              "label": label},
                                             [])
                if (h): item.set_handler(h)
        #end if

        self.__menu.prepare()
        self.__menu.get_widget().popup(None, None, None, button,
                                       eventtime)
        self.__menu = None



    def __queue_motion(self, px, py, w, h, is_leave):

        # some window managers send a LEAVE event for mouse clicks;
        # work around this
        if (is_leave and (px, py) == self.__last_pointer_pos
            and (0 <= px <= w) and (0 <= py <= h)):
            is_leave = False

        # don't do redundant work
        if (not is_leave and (px, py) == self.__last_pointer_pos): return
        else: self.__last_pointer_pos = (px, py)

        if (not is_leave):
            ux = Unit.Unit(px, Unit.UNIT_PX)
            uy = Unit.Unit(py, Unit.UNIT_PX)
            self.__group.handle_action(DisplayTarget.ACTION_MOTION,
                                       ux, uy, Struct())
            self.__group.notify_handle_action(True)
        else:
            self.__group.notify_handle_action(False)



    #
    # Observer for the root group.
    #
    def child_observer(self, src, cmd):

        if (cmd == src.OBS_GEOMETRY):
            x, y, w, h = self.__group.get_geometry()
            ux, uy = self.__group.get_user_geometry()[:2]

            if (ux.is_unset() or uy.is_unset()):
                utils.request_call(self.update_observer, self.OBS_GEOMETRY,
                                   UNSET_COORD, UNSET_COORD,
                                   w.as_px(), h.as_px())
            else:
                self.__save_position()
                utils.request_call(self.update_observer, self.OBS_GEOMETRY,
                                   x.as_px(), y.as_px(),
                                   w.as_px(), h.as_px())



    #
    # Sets the anchored position of the display.
    #
    def set_position(self, x = UNSET_COORD, y = UNSET_COORD):

        if (x != y != UNSET_COORD):
            x = Unit.Unit(x, Unit.UNIT_PX)
            y = Unit.Unit(y, Unit.UNIT_PX)
            cx, cy, cw, ch = self.__group.get_geometry()

            if ((cx, cy) != (x, y)):
                ax, ay = self.__group.get_anchored_coords(x, y, cw, ch)
                dx, dy= x - ax, y - ay
                new_x = x.as_px() + dx.as_px()
                new_y = y.as_px() + dy.as_px()

                self.__group.set_xml_prop("x", str(new_x))
                self.__group.set_xml_prop("y", str(new_y))



    #
    # Sets the size of the display.
    #
    def set_size(self, width, height):

        self.__group.set_xml_prop("width", str(width))
        self.__group.set_xml_prop("height", str(height))



    #
    # Sets the configuration settings.
    # FIXME: remove eventually :)
    #
    def __set_settings(self, settings, sensor):

        for key, value in settings.get_entries():
            # get all (target, property) tuples that are watching the given
            # sensor key and notify the targets
            entries = self.__mapping.get((sensor, key), [])

            # if an array does not have enough elements, create them
            if (not entries and "[" in key):
                pos = key.find("[")
                port = key[:pos]
                path = key[pos + 1:-1].split("][")
                new_length = int(path.pop()) + 1
                if (port in self.__arrays):
                    array = self.__arrays[port]
                    array.set_prop("length", new_length)
                    entries = self.__mapping.get((sensor, key), [])

            for target, prop in entries:
                target.set_xml_prop(prop, value)



    #
    # Sets the window configuration.
    #
    def set_prop(self, key, value):

        if (key == "window-flags"):
            self.update_observer(self.OBS_FLAGS, value)

        elif (key == "title"):
            self.update_observer(self.OBS_TITLE, value)

        elif (key == "icon"):
            filename = self.get_full_path(value)
            loader = gtk.gdk.PixbufLoader()
            try:
                data = vfs.read_entire_file(filename)
            except:
                return
            try:
                loader.write(data, len(data))
            except:
                log("Warning: Invalid image format.")
                return

            loader.close()
            pixbuf = loader.get_pixbuf()
            self.update_observer(self.OBS_ICON, pixbuf)

        elif (key == "shape"):
            if (value.lstrip().startswith("<")):
                from utils.DOM import DOM
                from utils import svg
                w, h, = self.size_request()
                if (not w or not h): return
                root = DOM(value).get_root()
                root["width"] = `w`
                root["height"] = `h`
                img = gtk.Image()
                svg.render(img, w, h, str(root))
                pixbuf = img.get_pixbuf()

            else:
                filename = self.get_full_path(value)
                loader = gtk.gdk.PixbufLoader()
                try:
                    data = vfs.read_entire_file(filename)
                except:
                    return
                try:
                    loader.write(data, len(data))
                except:
                    log("Warning: Invalid image format.")
                    return

                loader.close()
                pixbuf = loader.get_pixbuf()

            pix, mask = pixbuf.render_pixmap_and_mask(1)
            self.update_observer(self.OBS_SHAPE, mask)



    #
    # Adds a sensor to this display.
    # FIXME: remove eventually :)
    #
    def add_sensor(self, ident, sensor):

        def set_menu(menu): self.__sensor_menu = menu

        self.__sensor_controls[ident] = sensor
        sensor.bind("output", self.__set_settings, ident)
        sensor.bind("menu", set_menu)



    #
    # Binds a sensor's output to an element.
    # FIXME: remove eventually :)
    #
    def bind_sensor(self, sensorplug, element, prop):

        def h(value, element, prop, port):
            for k, v in value.items():
                if (k == port):
                    element.set_xml_prop(prop, v)

        ident, port = sensorplug.split(":")
        if (not (element, prop) in
              self.__mapping.setdefault((ident, port), [])):
            self.__mapping[(ident, port)].append((element, prop))



    #
    # Calls a function of a Sensor.
    # FIXME: Remove eventually
    #
    def call_sensor(self, cmd, path, *args):
        assert(cmd)

        args = list(args)
        for ident, callname, userargs in cmd:
            sensorctrl = self.__sensor_controls[ident]

            allargs = args + userargs

            # the sensor is an external module, so we make sure it cannot crash
            # the application
            try:
                sensorctrl.action = (callname, path, allargs)

            except StandardError, exc:
                log("The sensor produced an error: %s" % (exc,))



    #
    # Unbinds a sensor's output from an element.
    # FIXME: remove eventually :)
    #
    def unbind_sensor(self, sensorplug, element, prop):

        ident, port = sensorplug.split(":")
        try:
            self.__mapping[(ident, port)].remove((element, prop))
        except KeyError:
            pass



    #
    # Registers an array to be parent array of the given child.
    # FIXME: remove eventually :)
    #
    def register_array_for_port(self, array, sensorplug):

        ident, port = sensorplug.split(":")
        self.__arrays[port] = array



    #
    # Returns the anchored geometry of this display.
    #
    def get_geometry(self):

        try:
            x, y, w, h = self.__group.get_geometry()
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)
            dx, dy = x - ax, y - ay
            return (x + dx, y + dy, w, h)
        except:
            return (Unit.ZERO, Unit.ZERO, Unit.ZERO, Unit.ZERO)



    def __handle_configure(self, *args):

        self.__open_configurator()



    def __handle_move(self, *args):

        self.update_observer(self.OBS_GEOMETRY, UNSET_COORD, UNSET_COORD, 0, 0)



    def __handle_source(self, *args):

        from config import settings

        if (HAVE_WIN32):
            os.system("start notepad \"%s\"" % (self.__display_file,))
        else:
            os.system("%s \"%s\" & disown" % (settings.editor,
                                              self.__display_file))


    def __handle_restart(self, *args):

        self.update_observer(self.OBS_RESTART, self.__id)



    def __handle_remove(self, *args):

        self.close()



    def close(self):

        def remove_display(*args):
            self.update_observer(self.OBS_CLOSE, self.__id)


        utils.dialog.question(None,
                              _("Do you really want to remove this desklet ?"),
                              _("This desklet will no longer be displayed "
                                "and its configuration will be purged."),
                              (gtk.STOCK_CANCEL, Null),
                              (gtk.STOCK_DELETE, remove_display)
                             )



    #
    # Returns a unique ID string.
    #
    def make_id():
        return "id%d%d" % (int(time.time() * 100), random.randrange(0xffff))

    make_id = staticmethod(make_id)
