from utils.datatypes import *
from utils.Struct import Struct
from utils import Unit
from DataTarget import DataTarget

from urllib import unquote

import gobject
import gtk


#
# Abstract class for DisplayTargets.
#
class DisplayTarget(DataTarget):

    __slots__ = ('__has_adjust_request', '__geometry', '__old_geometry',
                 '__user_geometry', '__relative', '__is_relative', '__anchor',
                 '__had_action')

    # observer actions
    OBS_GEOMETRY = 0

    # user actions
    ACTION_CLICK = "click"
    ACTION_DOUBLECLICK = "doubleclick"
    ACTION_PRESS = "press"
    ACTION_RELEASE = "release"
    ACTION_MENU = "menu"
    ACTION_SCROLL = "scroll"
    ACTION_ENTER = "enter"
    ACTION_LEAVE = "leave"
    ACTION_MOTION = "motion"
    ACTION_FILE_DROP = "file-drop"
    ACTION_LINK_DROP = "link-drop"
    ACTION_KEY_PRESS = "key-press"
    ACTION_KEY_RELEASE = "key-release"

    # placement anchors
    ANCHOR_NW = "nw"
    ANCHOR_N = "n"
    ANCHOR_NE = "ne"
    ANCHOR_E = "e"
    ANCHOR_SE = "se"
    ANCHOR_S = "s"
    ANCHOR_SW = "sw"
    ANCHOR_W = "w"
    ANCHOR_CENTER = "center"


    # what we accept for drag & drop
    __DND_FILE = [("text/uri-list", 0, 0)]
    __DND_LINK = [("x-url/http", 0, 0), ("_NETSCAPE_URL", 0, 0)]


    def __init__(self, name, parent):

        # flag indicating whether an geometry adjust has been requested already
        self.__has_adjust_request = False

        # the real coordinates and size of this widget
        self.__geometry = (Unit.ZERO, Unit.ZERO,
                           Unit.ZERO, Unit.ZERO)
        self.__old_geometry = (Unit.UNSET, Unit.UNSET,
                               Unit.ZERO, Unit.ZERO)

        # coordinates and size of this widget as set by the user
        self.__user_geometry = (Unit.UNSET, Unit.UNSET,
                                Unit.UNSET, Unit.UNSET)

        # the widget to which this one is positioned relatively
        self.__relative = None

        # relative settings (is_rel_x, is_rel_y)
        self.__is_relative = (False, False)

        # the placement anchor
        self.__anchor = self.ANCHOR_NW

        # the value of the last notify_handle_action call
        self.__had_action = False

        DataTarget.__init__(self, name, parent)

        for prop, datatype in [("x", TYPE_UNIT),
                               ("y", TYPE_UNIT),
                               ("width", TYPE_UNIT),
                               ("height", TYPE_UNIT)]:
            self._register_property(prop, datatype,
                                    self._setp_geometry, self._getp_geometry)

        self._register_property("relative-to", TYPE_LIST,
                                self._setp_relative_to, self._getp)

        self._register_property("anchor", TYPE_STRING,
                                self._setp_anchor, self._getp)

        self._register_property("visible", TYPE_BOOL,
                                self._setp_visible, self._getp)
        self._setp("visible", True)

        for action in (self.ACTION_ENTER,
                       self.ACTION_LEAVE,
                       self.ACTION_CLICK,
                       self.ACTION_PRESS,
                       self.ACTION_RELEASE,
                       self.ACTION_SCROLL,
                       self.ACTION_FILE_DROP,
                       self.ACTION_LINK_DROP,
                       self.ACTION_DOUBLECLICK,
                       self.ACTION_MOTION,
                       self.ACTION_MENU,
                       self.ACTION_KEY_PRESS,
                       self.ACTION_KEY_RELEASE):
            self._register_action(action)



    #
    # Reacts on notifications by the relative of this widget.
    #
    def set_relative_to(self, relative):

        self.__relative = relative
        self.adjust_geometry()



    def __on_file_drop(self, widget, context, x, y, data, info, time):
        ''' catch DND events and process them to send them to them
        main display, which forwards them to the sensor '''

        # get the display
        display = self._get_display()

        # tell the main display to send files and coordinates to the sensor
        files = [unquote(uri) for uri in data.data.split("\r\n") if uri != '']
        display.send_action(self, self.ACTION_FILE_DROP,
                            Struct(files = files, _args = [files]))


    def __on_link_drop(self, widget, context, x, y, data, info, time):
        ''' catch DND events and process them to send them to them
        main display, which forwards them to the sensor '''

        # get the display
        display = self._get_display()

        # tell the main display to send link and coordinates to the sensor
        links = [unquote(data.data.split("\n")[0])]
        display.send_action(self, self.ACTION_LINK_DROP,
                            Struct(links = links, _args = [links]))


    #
    # Returns whether this target is standalone, i.e. needs no parent.
    #
    def is_standalone(self): return False


    #
    # Returns the widget of this target.
    #
    def get_widget(self): raise NotImplementedError


    #
    # Returns the true coordinates of this target when the given coordinates
    # are the hotspot.
    #
    def get_anchored_coords(self, x, y, w, h):
        assert (isinstance(x, Unit.Unit))
        assert (isinstance(y, Unit.Unit))
        assert (isinstance(w, Unit.Unit))
        assert (isinstance(h, Unit.Unit))

        if (x.is_unset() or y.is_unset()):
            return (x, y)

        anchor = self.__anchor
        if (anchor in (self.ANCHOR_NW, self.ANCHOR_W, self.ANCHOR_SW)):
            ax = x
        elif (anchor in (self.ANCHOR_N, self.ANCHOR_CENTER, self.ANCHOR_S)):
            ax = x - (w / 2)
        else:
            ax = x - w

        if (anchor in (self.ANCHOR_NW, self.ANCHOR_N, self.ANCHOR_NE)):
            ay = y
        elif (anchor in (self.ANCHOR_W, self.ANCHOR_CENTER, self.ANCHOR_E)):
            ay = y - (h / 2)
        else:
            ay = y - h

        return (ax, ay)


    #
    # Returns the position of the anchor in this target. Origin is the top
    # left corner.
    #
    def get_anchor(self):

        x, y, w, h = self.__geometry
        ax, ay = self.get_anchored_coords(x, y, w, h)
        dx = (x - ax)
        dy = (y - ay)

        return (dx, dy)



    #
    # Returns the target ID to which this target is placed relatively.
    #
    def get_relative_to(self):

        return self.__relative


    #
    # Returns the size of the parent widget.
    #
    def _get_parent_size(self):

        # get the geometry of the parent widget or the screen,
        # if there's no parent
        parent = self._get_parent()
        if (parent != self._get_display()):
            parent_width, parent_height = \
                 self._get_parent().get_container_geometry()[2:]
        else:
            parent_width = Unit.Unit(gtk.gdk.screen_width(), Unit.UNIT_PX)
            parent_height = Unit.Unit(gtk.gdk.screen_height(), Unit.UNIT_PX)

        # don't allow dangerous values for the parent's size
        parent_width = max(Unit.ONE, parent_width)
        parent_height = max(Unit.ONE, parent_height)

        return (parent_width, parent_height)


    #
    # Returns the geometry (coordinates and size) of this target.
    #
    def get_geometry(self):

        x, y, w, h = self.__geometry
        if (self.get_prop("visible")):
            return (x, y, w, h)
        else:
            return (x, y, Unit.ZERO, Unit.ZERO)



    #
    # Sets the user geometry of this target.
    #
    def set_user_geometry(self, x = None, y = None,
                          width = None, height = None):

        ox, oy, ow, oh = self.__user_geometry
        if (x == None): x = ox
        if (y == None): y = oy
        if (width == None): width = ow
        if (height == None): height = oh
        self.__user_geometry = (x, y, width, height)



    #
    # Returns the geometry from the user's point of view.
    #
    def get_user_geometry(self):

        return self.__user_geometry


    #
    # Requests a geometry adjustment. Use this to avoid redundant calls of
    # adjust_geometry().
    #
    def request_adjust_geometry(self):

        if (not self.__has_adjust_request):
            self.__has_adjust_request = True
            gobject.timeout_add(0, self.adjust_geometry)



    #
    # Adjusts the geometry of this target.
    #
    def adjust_geometry(self):

        self.__has_adjust_request = False

        x, y, w, h = self.__geometry
        ux, uy, uw, uh = self.get_user_geometry()
        anchor_x, anchor_y = self.get_anchor()

        parent = self._get_parent()
        parent_width, parent_height = self._get_parent_size()

        # adjust percentage stuff
        ux.set_100_percent(parent_width.as_px())
        uw.set_100_percent(parent_width.as_px())
        uy.set_100_percent(parent_height.as_px())
        uh.set_100_percent(parent_height.as_px())

        # compute geometry in pixels
        if (ux.is_unset()): new_x = x + anchor_x
        else: new_x = ux

        if (uy.is_unset()): new_y = y + anchor_y
        else: new_y = uy

        if (uw.is_unset()): new_w = self._get_value_for_unset(width = w)
        else: new_w = uw

        if (uh.is_unset()): new_h = self._get_value_for_unset(height = h)
        else: new_h = uh

        # handle relative positioning
        if (self.__relative):
            is_rel_x, is_rel_y = self.__is_relative

            # get the coords of the anchor of the relative
            anchor_x, anchor_y = self.__relative.get_anchor()
            rx, ry, rw, rh = self.__relative.get_geometry()

            # get the remaining amount of pixels from the anchor to the
            # bottom/right edge of the relative
            tw = rw - anchor_x
            th = rh - anchor_y

            if (not ux.is_unset()): new_x += rx + anchor_x
            else: new_x = rx + anchor_x
            if (not uy.is_unset()): new_y += ry + anchor_y
            else: new_y = ry + anchor_y
            if (is_rel_x): new_x += tw
            if (is_rel_y): new_y += th
        #end if

        # get coords of the top left corner
        new_x, new_y = self.get_anchored_coords(new_x, new_y, new_w, new_h)
        if (parent == self._get_display()
              or (new_x, new_y, new_w, new_h) != self.__old_geometry):

            self.__geometry = (new_x, new_y, new_w, new_h)
            self.__old_geometry = (new_x.copy(), new_y.copy(), new_w.copy(), new_h.copy())
            self.get_widget().set_size_request(new_w.as_px(), new_h.as_px())
            # if w and h of a widget are 0, we might not propagate the GEOMETRY
            # the problem is, if they are invisible, they have to be notified,
            # otherwise other widgets are taking their place
            self._propagate_geometry()



    #
    # Returns the geometry value that should be used for unset values.
    #
    def _get_value_for_unset(self, x = None, y = None,
                             width = None, height = None):

        if   (x != None): return x
        elif (y != None): return y
        elif (width != None): return width
        elif (height != None): return height



    #
    # Tells other targets about geometry changes.
    #
    def _propagate_geometry(self):

        parent = self._get_parent()
        if (parent):
            parent.request_adjust_geometry()
        self.update_observer(self.OBS_GEOMETRY)


    #
    # Sets the position of this target.
    #
    def set_position(self, x, y):
        assert (isinstance(x, Unit.Unit))
        assert (isinstance(y, Unit.Unit))

        ox, oy, w, h = self.__geometry
        if ((x, y) != (ox, oy)):
            self.__geometry = (x, y, w, h)
            self.adjust_geometry()



    #
    # Sets the size of this target. Use this instead of set_size_request() in
    # targets to set the size manually.
    #
    def set_size(self, width, height):
        assert (isinstance(width, Unit.Unit))
        assert (isinstance(height, Unit.Unit))

        x, y, w, h = self.__geometry
        if ((w, h) != (width, height)):
            self.__geometry = (x, y, width, height)
            self.adjust_geometry()



    def handle_action(self, action, px, py, event):
        assert (isinstance(px, Unit.Unit))
        assert (isinstance(py, Unit.Unit))

        # we need the pointer position relative to the widget, so we have to
        # setup a new event structure for some actions
        if (action in (self.ACTION_CLICK, self.ACTION_DOUBLECLICK,
                       self.ACTION_MOTION, self.ACTION_PRESS,
                       self.ACTION_RELEASE)):
            x, y = self.get_widget().get_pointer()
            nil, nil, w, h = self.get_geometry()
            ux = Unit.Unit(x, Unit.UNIT_PX)
            uy = Unit.Unit(y, Unit.UNIT_PX)
            if (w.as_px() > 0): ux.set_100_percent(w.as_px())
            if (h.as_px() > 0): uy.set_100_percent(h.as_px())
            event["x"] = ux
            event["y"] = uy
            # FIXME: remove eventually :)
            if (action == self.ACTION_MOTION): event["_args"] = [x, y]

        DataTarget.handle_action(self, action, px, py, event)



    #
    # Notifies the target whether the action occured on it or not.
    # This detects ENTER and LEAVE events.
    #
    def notify_handle_action(self, value):

        # enter notify
        if (not self.__had_action and value):
            action = self.ACTION_ENTER
            if (self.get_action_call(action)):
                self.handle_action(action, Unit.ZERO, Unit.ZERO, Struct())

        # leave notify
        elif (self.__had_action and not value):
            action = self.ACTION_LEAVE
            if (self.get_action_call(action)):
                self.handle_action(action, Unit.ZERO, Unit.ZERO, Struct())

        self.__had_action = value



    #
    # Geometry properties.
    #
    def _setp_geometry(self, key, value):
        assert (isinstance(value, Unit.Unit))

        if (key == "x"):
            self.set_user_geometry(x = value)
            self.adjust_geometry()
            self._setp(key, value)

        elif (key == "y"):
            self.set_user_geometry(y = value)
            self.adjust_geometry()
            self._setp(key, value)

        elif (key == "width"):
            self.set_user_geometry(width = value)
            self.adjust_geometry()
            self._setp(key, value)

        elif (key == "height"):
            self.set_user_geometry(height = value)
            self.adjust_geometry()
            self._setp(key, value)


    def _setp_relative_to(self, key, value):

        name, mode = value
        if (mode == "x"): self.__is_relative = (True, False)
        elif (mode == "y"): self.__is_relative = (False, True)
        elif (mode == "xy"): self.__is_relative = (True, True)

        self._get_parent().set_relative(self, name)
        self._setp(key, value)


    def _setp_anchor(self, key, value):

        x, y = self.get_user_geometry()[:2]
        self.__anchor = value
        self.adjust_geometry()
        self._setp(key, value)


    def _getp_geometry(self, key):

        x, y, w, h = self.get_geometry()
        #x, y = self.get_anchored_coords(x, y, w, h)

        parent_width, parent_height = self._get_parent_size()

        if (key == "x"):
            unit = x
            unit.set_100_percent(parent_width.as_px())
        elif (key == "y"):
            unit = y
            unit.set_100_percent(parent_height.as_px())
        elif (key == "width"):
            unit = w
            unit.set_100_percent(parent_width.as_px())
        elif (key == "height"):
            unit = h
            unit.set_100_percent(parent_height.as_px())

        return unit


    #
    # "visible" property.
    #
    def _setp_visible(self, key, value):

        if (value): self.get_widget().show()
        else: self.get_widget().hide()
        self._setp(key, value)
        self.adjust_geometry()
        self._propagate_geometry()



    #
    # Action handlers.
    #
    def _setp__action(self, key, value):

        DataTarget._setp__action(self, key, value)
        if (key == "on-file-drop"):
            self.get_widget().drag_dest_set(gtk.DEST_DEFAULT_ALL,
                                            self.__DND_FILE,
                                            gtk.gdk.ACTION_COPY)
            self.get_widget().connect("drag_data_received",
                                      self.__on_file_drop)

        elif (key == "on-link-drop"):
            self.get_widget().drag_dest_set(gtk.DEST_DEFAULT_ALL,
                                            self.__DND_LINK,
                                            gtk.gdk.ACTION_COPY)
            self.get_widget().connect("drag_data_received",
                                      self.__on_link_drop)

