from DisplayTarget import DisplayTarget
from utils.datatypes import *
from utils import Unit
from utils import vfs
from utils import tiling
import utils

import gtk
import os



#
# Class for image targets.
#
class TargetImage(DisplayTarget):

    def __init__(self, name, parent):

        # the pixbuf of the original image and its URI
        self.__original_image = None

        self.__parent_geometry = (Unit.ZERO, Unit.ZERO, Unit.ZERO, Unit.ZERO)

        # the original image size
        self.__original_width = 0
        self.__original_height = 0

        # the current image size
        self.__current_width = 0
        self.__current_height = 0

        # the current saturation value
        self.__current_saturation = 0.0

        # the current opacity value
        self.__current_opacity = 0.0
        

        DisplayTarget.__init__(self, name, parent)

        self.__widget = gtk.Image()
        self.__widget.show()

        self._register_property("uri", TYPE_STRING,
                                self._setp_uri, self._getp)
        self._register_property("saturation", TYPE_FLOAT,
                                self._setp_saturation, self._getp)
        self._register_property("scale", TYPE_FLOAT,
                                self._setp_scale, self._getp)
        self._register_property("opacity", TYPE_FLOAT,
                                self._setp_opacity, self._getp)
        self._register_property("image-width", TYPE_UNIT,
                                self._setp_image_size, self._getp_image_size)
        self._register_property("image-height", TYPE_UNIT,
                                self._setp_image_size, self._getp_image_size)

        self._setp("image-width", Unit.Unit(-1, Unit.UNIT_PX))
        self._setp("image-height", Unit.Unit(-1, Unit.UNIT_PX))
        self._setp("opacity", 1.0)
        self._setp("saturation", 1.0)
        self._setp("scale", 1.0)

        # watch the parent for geometry changes
        parent.add_observer(self.__on_observe_parent)


    def get_widget(self): return self.__widget


    #
    # Observer for the parent's geometry.
    #
    def __on_observe_parent(self, src, cmd, *args):

        if (cmd == src.OBS_GEOMETRY and
              src.get_geometry() != self.__parent_geometry):
            self.__render_image()
            self.__parent_geometry = src.get_geometry()



    #
    # Loads the image from the given URI.
    #
    def __load_image(self, uri):

        if (not uri): return

        try:
            data = vfs.read_entire_file(uri)
        except:
            return

        loader = gtk.gdk.PixbufLoader()
        try:
           loader.write(data, len(data))
        except:
            log("Invalid image format.")
            return

        loader.close()

        self.__original_image = loader.get_pixbuf()
        self.__widget.set_from_pixbuf(self.__original_image)

        if (self.__original_image):
            self.__original_width = self.__original_image.get_width()
            self.__original_height = self.__original_image.get_height()
            self.__current_width = 0
            self.__current_height = 0

        # clean up dead pixbufs
        import gc
        utils.request_call(gc.collect)


    #
    # Renders the image.
    #
    def __render_image(self):

        if (not self.__original_image): return

        nil, nil, parentwidth, parentheight = self._get_parent().get_geometry()

        if ((self._getp("image-width").get_unit() == Unit.UNIT_PERCENT and
             parentwidth < Unit.ONE) or
            (self._getp("image-height").get_unit() == Unit.UNIT_PERCENT and
             parentheight < Unit.ONE)): return

        imgwidth = self._getp("image-width")
        imgheight = self._getp("image-height")
        if (parentwidth > Unit.ZERO):
            imgwidth.set_100_percent(parentwidth.as_px())
        if (parentheight > Unit.ZERO):
            imgheight.set_100_percent(parentheight.as_px())
        imgwidth = imgwidth.as_px()
        imgheight = imgheight.as_px()

        # scale image preserving aspect ratio
        if (imgwidth <= 0 and imgheight > 0):
            imgwidth = imgheight * \
                       (self.__original_width / float(self.__original_height))
        elif (imgwidth > 0 and imgheight <= 0):
            imgheight = (imgwidth * self.__original_height) / \
                        float(self.__original_width)
        elif (imgwidth <= 0 and imgheight <= 0):
            imgwidth = self.__original_width
            imgheight = self.__original_height

        # apply the "scale" property
        scale = self.get_prop("scale")
        # some authors don't pay attention, let's catch it
        if (scale <= 0.0): scale = 1.0
        imgwidth = int(imgwidth * scale)
        imgheight = int(imgheight * scale)

        saturation = self.get_prop("saturation")
        opacity = self.get_prop("opacity")

        if ((imgwidth, imgheight, saturation, opacity) ==
            (self.__current_width, self.__current_height,
             self.__current_saturation, self.__current_opacity)): return

        # TODO: reimplement saturation again (in C)
    
        tiling.render_to_image(self.__widget, self.__original_image.copy(),
                               imgwidth, imgheight, opacity);

        self.set_size(Unit.Unit(imgwidth, Unit.UNIT_PX),
                      Unit.Unit(imgheight, Unit.UNIT_PX))
        self.__current_width = imgwidth
        self.__current_height = imgheight
        self.__current_saturation = saturation
        self.__current_opacity = opacity


    #
    # "uri" property.
    #
    def _setp_uri(self, key, value):

        path = self._get_display().get_full_path(value)
        try:
            self.__load_image(path)

        except:
            log("Failed to set URI %s for key %s." % (path, key))
            return

        self.__render_image()
        self._setp(key, value)


    #
    # "opacity" property.
    #
    def _setp_opacity(self, key, value):

        value = min(1.0, max(0.0, value))
        old_opac = self.get_prop("opacity")
        if (abs(value - old_opac) > 0.01):
            self._setp(key, value)
            self.__render_image()

    #
    # "saturation" property.
    #
    def _setp_saturation(self, key, value):

        old_sat = self.get_prop("saturation")
        if (abs(value - old_sat) > 0.01 ):
            self._setp(key, value)
            self.__render_image()



    #
    # "scale" property.
    #
    def _setp_scale(self, key, value):

        old_scale = self.get_prop("scale")
        if (abs(value - old_scale) > 0.01):
            self._setp(key, value)
            self.__render_image()



    #
    # "image-width" and "image-height" properties.
    #
    def _setp_image_size(self, key, value):
        assert (isinstance(value, Unit.Unit))

        if (key == "image-width" and
              value != self._getp("image-width")):
            self._setp(key, value)
            self.__render_image()
        elif (key == "image-height" and
              value != self._getp("image-height")):
            self._setp(key, value)
            self.__render_image()



    #
    # "image-height" property.
    #
    def _getp_image_size(self, key):

        if (key == "image-width"):
            return self._getp(key)
        elif (key == "image-height"):
            return self._getp(key)

