import gobject
import gtk
import sys
import vfs
import utils


#
# Image widget that supports tiling.
#
class TilingImage(gtk.Image):

    def __init__(self):

        self.__pixbuf = None
        self.__size = (1, 1)
        self.__is_invalidated = True

        gtk.Image.__init__(self)



    #
    # Sets the image from a color string.
    #
    def set_from_color(self, color):

        if (color[0] == "#" and len(color) == 9):
            alpha = int(color[-2:], 16)
            color = color[:-2]
        else:
            alpha = 255

        w, h = self.__size
        if (not self.__pixbuf):
            self.__pixbuf = gtk.gdk.Pixbuf(0, 1, 8, 320, 32)
        c = gtk.gdk.color_parse(color)
        fillcolor = int('0x%02x%02x%02x%02x' % (c.red / 256, c.green / 256,
                                                c.blue / 256, alpha), 16)
        self.__pixbuf.fill(fillcolor)

        self.__is_invalidated = True
        self.tile(w, h)



    #
    # Sets the image from a file.
    #
    def set_from_file(self, uri):

        if (vfs.exists(uri)):
            try:
                data = vfs.read_entire_file(uri)
            except:
                return
            loader = gtk.gdk.PixbufLoader()
            try:
                loader.write(data, len(data))
            except:
                log("Invalid image format.")
                return
            loader.close()
            self.__pixbuf = loader.get_pixbuf()
            w, h = self.__size
            self.__is_invalidated = True
            self.tile(w, h)



    #
    # Tiles the background image over the visible area.
    #
    def tile(self, width, height):

        if (not self.__is_invalidated and (width, height) == self.__size):
            return

        self.__is_invalidated = False
        self.__size = (width, height)

        if (not self.__pixbuf): return
        w = self.__pixbuf.get_width()
        h = self.__pixbuf.get_height()
        width = max(width, 1)
        height = max(height, 1)

        if (HAVE_WIN32):
            self.tile_on_image_win32(self.__pixbuf, width, height)
        else:
            import tiling
            tiling.tile_on_image (self, self.__pixbuf, width, height)


    def tile_on_image_win32(self, pixbuf, width, height):

        w = pixbuf.get_width()
        h = pixbuf.get_height()

        bg = gtk.gdk.Pixbuf(0, 1, 8, width, height)
        for x in xrange(0, width, w):
            for y in xrange(0, height, h):
                pw = min(w, width - x)
                ph = min(h, height - y)
                pixbuf.copy_area(0, 0, pw, ph, bg, x, y)

        self.set_from_pixbuf(bg)

        cache_width = int(width / w) * w
        cache_height = int(height / h) * h
        if (cache_width > 0 and cache_height > 0):
            cache = gtk.gdk.Pixbuf(0, 1, 8, cache_width, cache_height)
            bg.copy_area(0, 0, cache_width, cache_height, cache, 0, 0)
            self.__pixbuf = cache

