/* Functions for tiling pixbufs.
 * Copyright (C) 2004, 2005 Martin Grimme, Christian Meyer
 * grab_background () taken from the adesklets project and modified by the
 * gDesklets team (thanks to Sylvain from adesklets)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "utils.h"

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <X11/Xlib.h>


PyMODINIT_FUNC inittiling(void);

static inline void
copy_n_rows (gint32 n, GdkPixbuf *pbuf, gint32 row_size, gint32 offset)
{
  guchar * const pixels = gdk_pixbuf_get_pixels (pbuf);
  memcpy (pixels + offset, pixels, n * row_size);
}



static void
make_row (const GdkPixbuf *src, GdkPixbuf *dest, gint32 offset)
{
  gint32   x, y;
  guchar *in, *out;

  const gint32 src_height     = gdk_pixbuf_get_height (src);
  const gint32 dest_height    = gdk_pixbuf_get_height (dest);
  const gint32 src_rowstride  = gdk_pixbuf_get_rowstride (src);
  const gint32 dest_rowstride = gdk_pixbuf_get_rowstride (dest);

  const gint32 rstride = gdk_pixbuf_get_width (src) * (
       (gdk_pixbuf_get_n_channels (src) * gdk_pixbuf_get_bits_per_sample (src)
        + 7) / 8);
  const gint32 q = offset / dest_rowstride;

  in  = gdk_pixbuf_get_pixels (src);
  out = gdk_pixbuf_get_pixels (dest) + offset;

  for (y = 0; (y < src_height) && (y + q < dest_height); y++) {
    for (x = 0; x < dest_rowstride; x += rstride) {
      memcpy (out + x, in, MIN (src_rowstride, dest_rowstride - x) );
    }

    in  +=  src_rowstride;
    out += dest_rowstride;
  }

}



static void
tile (const GdkPixbuf *src, GdkPixbuf *dest)
{
  gint32 row, offset;

  const gint32 row_width  = gdk_pixbuf_get_rowstride (dest);
  const gint32 row_height = gdk_pixbuf_get_height (src);
  const gint32 row_size   = row_width * row_height;
  const gint32 dest_size  = row_width * gdk_pixbuf_get_height (dest);
  const gint32 max        = gdk_pixbuf_get_height (dest) / row_height;

  /* first iteration unrolled */
  offset = 0;
  row    = 0;
  make_row (src, dest, offset);
  row++;
  offset += row_size;

  while (offset < dest_size && row < max) {
      const gint32 n = MIN (row, max - row);
      copy_n_rows (n, dest, row_size, offset);
      row += n;
      offset += row_size * n;
  }

  /* last iteration unrolled */
  make_row (src, dest, offset);
}



static PyObject*
tile_on_image(PyObject* self, PyObject* args)
{
  GtkImage *image;
  const GdkPixbuf *pbuf;
  GdkPixbuf *alphaified;
  gint32 width, height;

  GdkPixbuf *bg;

  if(!PyArg_ParseTuple(args, "O&O&ii",
                       parse_gtk_image,  &image,
                       parse_gdk_pixbuf, &pbuf,
                       &width, &height))
    return NULL;

  /* alphaify */
  alphaified = gdk_pixbuf_add_alpha (pbuf, FALSE, 0, 0, 0);

  bg = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
                       TRUE,
                       8, width, height);
  tile (alphaified, bg);
  gtk_image_set_from_pixbuf (image, bg);
  g_object_unref (G_OBJECT (bg));
  g_object_unref (G_OBJECT (alphaified));

  Py_INCREF(Py_None);
  return Py_None;
}



static GdkPixbuf *
grab_background(int x, int y, int width, int height) {
  int screen;
  Display *dpy;
  XEvent ev;
  Window src;
  XSetWindowAttributes attrs = { ParentRelative, 0L, 0, 0L, 0, 0, Always, 0L,
                                 0L, False, ExposureMask, 0L, True, 0, 0 };
  GdkWindow *gdkwin;
  GdkPixbuf *pbuf;

  dpy = gdk_x11_get_default_xdisplay ();
  screen = DefaultScreen (dpy);

  src = XCreateWindow (dpy, RootWindow (dpy, screen), x, y,
                       width, height, 0, CopyFromParent, CopyFromParent,
                       CopyFromParent, CWBackPixmap | CWBackingStore |
                       CWOverrideRedirect | CWEventMask,
                       &attrs);
  XGrabServer (dpy);
  XMapRaised (dpy, src);
  XSync (dpy, False);

  /* wait until the window is visible */
  do
    XWindowEvent (dpy, src, ExposureMask, &ev);
  while (ev.type != Expose);

  gdkwin = gdk_window_foreign_new (src);
  pbuf = gdk_pixbuf_get_from_drawable (NULL, gdkwin, NULL,
                                       0, 0, 0, 0, width, height);
  g_object_unref (G_OBJECT (gdkwin));
  XUngrabServer (dpy);
  XDestroyWindow (dpy, src);

  return pbuf;
}



static PyObject*
tile_transparency(PyObject* self, PyObject* args)
{
  GtkWidget *widget;
  glong wallpaper_id;
  gint32 x, y, width, height;

  gint32       pwidth, pheight, sx, sy;
  GdkBitmap   *mask;
  GdkColormap *cmap;
  GdkPixbuf   *pbuf;
  GdkPixmap   *pix;
  GdkPixmap   *pmap;
  GdkWindow   *rootwin;
  GtkStyle    *style;

  if(!PyArg_ParseTuple(args, "O&liiii",
                       parse_gtk_widget, &widget,
                       &wallpaper_id, &x, &y, &width, &height))
    return NULL;

  pmap = gdk_pixmap_foreign_new ((GdkNativeWindow) wallpaper_id);

  gdk_drawable_get_size (GDK_DRAWABLE (pmap), &pwidth, &pheight);

  /* grab background directly if the bg pixmap is very small, since it's faster
     than tiling */
  if (pwidth * pheight < 100) {
    pbuf = grab_background (x, y, width, height);
  } else {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);

    rootwin = gdk_get_default_root_window ();
    cmap = gdk_drawable_get_colormap (GDK_DRAWABLE (rootwin));

    /* tile wallpaper over pixbuf */
    sx = - (x % pwidth);
    sy = - (y % pheight);
    for (x = sx; x < width; x += pwidth) {
      for (y = sy; y < height; y += pheight) {
        gint32 dstx = MAX (0, x);
        gint32 dsty = MAX (0, y);
        gint32 srcx = dstx - x;
        gint32 srcy = dsty - y;

        gint32 w = MIN (pwidth - srcx, width - dstx);
        gint32 h = MIN (pheight - srcy, height - dsty);

        gdk_pixbuf_get_from_drawable (pbuf, pmap, cmap, srcx, srcy,
                                      dstx, dsty, w, h);
      }
    }
  }

  /* put background onto the widget */
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &pix, &mask, 127);

  style = gtk_style_new ();
  style->bg_pixmap[GTK_STATE_NORMAL] = pix;
  gtk_widget_set_style (widget, style);

  g_object_unref (G_OBJECT (pbuf));
  g_object_unref (G_OBJECT (pmap));
  g_object_unref (G_OBJECT (style));

  Py_INCREF(Py_None);
  return Py_None;
}


static void
set_opacity (GdkPixbuf *pbuf, float opacity)
{
  guchar *data;
  guint32 x, y, rowstride, height;

  data  = gdk_pixbuf_get_pixels (pbuf);

  rowstride = gdk_pixbuf_get_rowstride (pbuf);
  height = gdk_pixbuf_get_height (pbuf);
  for (x = 3; x < rowstride; x += 4)
    for (y = 0; y < height; y++)
      data[y * rowstride + x] *= opacity;
}


static PyObject*
render_to_image (PyObject* self, PyObject* args)
{
  GtkImage *image;
  const GdkPixbuf *pbuf;
  gint32 width, height;
  gint32 srcwidth, srcheight;
  float opacity;

  GdkPixbuf *scaled;
  GdkPixbuf *alphaified;

  if(!PyArg_ParseTuple(args, "O&O&iif",
                       parse_gtk_image,  &image,
                       parse_gdk_pixbuf, &pbuf,
                       &width, &height, &opacity))
    return NULL;

  srcwidth = gdk_pixbuf_get_width (pbuf);
  srcheight = gdk_pixbuf_get_height (pbuf);

  alphaified = gdk_pixbuf_add_alpha (pbuf, FALSE, 0, 0, 0);

  /* scale pixbuf */
  if (srcwidth != width || srcheight != height) {
    scaled = gdk_pixbuf_scale_simple (alphaified, width, height,
                                      GDK_INTERP_BILINEAR);
  } else {
    scaled = alphaified;
  }

  /* set opacity */
  set_opacity (scaled, opacity);

  /* set image */
  gtk_image_set_from_pixbuf (image, scaled);
  if (srcwidth != width || srcheight != height)
    g_object_unref (G_OBJECT (scaled));

  g_object_unref (G_OBJECT (alphaified));

  Py_INCREF(Py_None);
  return Py_None;
}



PyMODINIT_FUNC
inittiling(void)
{
  static const PyMethodDef methods[] =
    {
      {"render_to_image", render_to_image, METH_VARARGS, NULL},
      {"tile_on_image", tile_on_image, METH_VARARGS, NULL},
      {"tile_transparency", tile_transparency, METH_VARARGS, NULL},
      {NULL, NULL, 0, NULL}
    };

  if(!gdesklets_get_pygobject_type())
    return;

  Py_InitModule("tiling", (PyMethodDef*) methods);
}

