/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "setoper.h" 
  /* set operation library header (March 16, 1995 version or later) */
#include "cdddef.h"
#include "cdd.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <string.h>

#include "gdis.h"
#include "morph.h"
#include "matrix.h"
#include "interface.h"

/* TODO - get rid of all n_input's - it's always 4 ie len,x,y,z*/
#define NDIM 4

/* CDD main call */
gint DDEnumerate(rowrange, colrange, rowrange, colrange, Amatrix, struct model_pak *);

long mprev;
double *beta;
long mlast;

/*********************/
/* DEBUGGING routine */
/*********************/
void print_plane_vertices(struct plane_pak *plane)
{
GSList *list1, *list2;
struct vertex_pak *v;

for (list1=plane->vertices ; list1 ; list1=g_slist_next(list1))
  {
  v = (struct vertex_pak *) list1->data;
  printf("[%p] adj", v);
  for (list2=v->adj ; list2 ; list2=g_slist_next(list2))
    {
    printf(" : %p", list2->data);
    }
  printf("\n");
  }
}

/********************/
/* free vertex data */
/********************/
void free_vertices(struct model_pak *model)
{
model->num_vertices = 0;
}

/************************/
/* transfer vertex data */
/************************/
#define DEBUG_SAVE_VERTEX 0
void save_vertex(struct model_pak *model)
{
gint j;
RayRecord *RR;
struct vertex_pak *vertex;

free_slist(model->vertices);
model->vertices = NULL;

/* setup for placing in GDIS struct */
if (VertexCount < 4)
  {
  show_text(ERROR, "Insufficient planes to define a closed polyhedron.\n");
  model->num_vertices=0;
  return;
  }

#if DEBUG_SAVE_VERTEX
printf("Vertices:\n");
#endif

RR = FirstRay;
j=0;
while (RR != NULL) 
  {
  if (RR->feasible) 
    {
    vertex = g_malloc(sizeof(struct vertex_pak));
    vertex->x[0] = RR->Ray[1]/fabs(RR->Ray[0]);
    vertex->x[1] = RR->Ray[2]/fabs(RR->Ray[0]);
    vertex->x[2] = RR->Ray[3]/fabs(RR->Ray[0]);
    vertex->adj = NULL;
    model->vertices = g_slist_prepend(model->vertices, vertex);

#if DEBUG_SAVE_VERTEX
printf("[%d]", j);
P3VEC(" : ", vertex->x);
#endif

    j++;
    }
  RR = RR->Next;
  }

/* reverse, so the adjacent indices are the same */
model->vertices = g_slist_reverse(model->vertices);
model->num_vertices = g_slist_length(model->vertices);

g_assert(j == model->num_vertices);

#if DEBUG_SAVE_VERTEX
printf("Found %d vertices.\n", model->num_vertices);
#endif
}

/*****************************/
/* get vertex adjacency data */
/*****************************/
#define DEBUG_SAVE_ADJ 0
void save_adj(struct model_pak *data, Amatrix A)
{
gint i, j, m_size, n_size;
boolean adj;
RayRecord *RayPtr1, *RayPtr2;
struct vertex_pak *v1, *v2;

m_size = data->num_planes + 1;
n_size = NDIM;

#if DEBUG_SAVE_ADJ
printf("Adjacency list:\n");
#endif

LastRay->Next=NULL;
i=0;
for (RayPtr1=FirstRay ; RayPtr1 ; RayPtr1 = RayPtr1->Next)
  {
v1 = g_slist_nth_data(data->vertices, i);
#if DEBUG_SAVE_ADJ
printf("Vertex (%2d:%p) adj:", i, v1);
#endif
  j=0;
  for (RayPtr2=FirstRay ; RayPtr2 ; RayPtr2 = RayPtr2->Next)
    {
    if (i != j)
      {
      CheckAdjacency2(m_size, n_size, A, &RayPtr1, &RayPtr2, &adj);
      if (adj) 
        {
        v2 = g_slist_nth_data(data->vertices, j);
#if DEBUG_SAVE_ADJ
printf(" (%2d:%p)", j, v2);
#endif

v1->adj = g_slist_prepend(v1->adj, v2);
        }
      }
    j++;
    }
#if DEBUG_SAVE_ADJ
printf("\n");
#endif
  i++;
  }
}

/******************************/
/* compute visible facet data */
/******************************/
#define DEBUG_SAVE_FACET 0
void save_facet(struct model_pak *data)
{
gint i, j, m;
long elem, zcar;
rowset cset;
RayRecord *RR;
GSList *list;
struct vec_pak *vertex;
struct plane_pak *plane;

m = data->num_planes + 1;

/* wipe previous (if any) */
for (list=data->planes ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;

  plane->vertices = NULL;
  plane->present = FALSE;
  }

/* construct vertex list for each facet */
RR = FirstRay;
j=0;
while (RR != NULL) 
  {
  if (RR->feasible) 
    {
    set_initialize(&cset, m);
    zcar = set_card(RR->ZeroSet);
    if (m - zcar >= zcar) 
      {

vertex = g_slist_nth_data(data->vertices, j);
g_assert(vertex != NULL);

      for (elem=1 ; elem<=RR->ZeroSet[0] ; elem++)
        {
        if (set_member(elem,RR->ZeroSet))
          {
/* facet number elem 'contains' current vertex */
/* NB: stupid pascal numbering */
          i = (gint) elem - 1;

plane = g_slist_nth_data(data->planes, i);
g_assert(plane != NULL);
  
  plane->present = TRUE;


/* get plane label (miller indices) */
  ARR3SET(plane->index, plane->m);
/* NEW cartesian normal is now calculated before */
  ARR3SET(plane->norm, plane->x);
/* planes->x is -ve inequality, remember! */
  VEC3MUL(plane->norm, -1.0);

plane->vertices = g_slist_prepend(plane->vertices, vertex);

          }
        }

      j++;
      }
    }
  RR = RR->Next;
  }

/* output summary */
#if DEBUG_SAVE_FACET
printf("-----------------------------\n");
i=0;
for (list=data->planes ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;
  if (plane->present)
    printf("plane %d has %d vertices.\n", i, g_slist_length(plane->vertices));
  i++;
  }
#endif
}

/*********************************************/
/* use adjacency data to set up vertex order */
/*********************************************/
#define DEBUG_INIT_FACET 0
void init_facet(struct model_pak *model)
{
gint n;
gdouble norm[3];
GSList *list1, *list2, *list3, *new_list;
struct plane_pak *plane;
struct vertex_pak *v0=NULL, *v1, *v2;

for (list1=model->planes ; list1 ; list1=g_slist_next(list1))
  {
  plane = (struct plane_pak *) list1->data;

  if (!plane->present)
    continue;

/* the idea is to replace the old vertex list for the facet */
/* with one that is ordered according to vertex adjacencies */
  new_list = NULL;
  list2 = plane->vertices;
  n = g_slist_length(plane->vertices);
/* skip dud vertices */
  if (n < 3)
    continue;

#if DEBUG_INIT_FACET
printf("facet %p:\n", plane);
print_plane_vertices(plane);
#endif

  while (list2)
    {
    v1 = (struct vertex_pak *) list2->data;
    list2 = g_slist_next(list2);
 
    if (!new_list)
      {
#if DEBUG_INIT_FACET
printf(" +%p\n", v1);
#endif
      new_list = g_slist_prepend(new_list, v1);
      plane->vertices = g_slist_remove(plane->vertices, v1);

      v0 = v1;
      }
    else
      {

g_assert(v0 != NULL);

/* search for an adjacency that isn't already in the new vertex list */
      for (list3=v0->adj ; list3 ; list3=g_slist_next(list3))
        {
        if (v1 == list3->data)
          {
#if DEBUG_INIT_FACET
printf(" +%p\n", v1);
#endif
/* move the vertex to the new list */
          new_list = g_slist_prepend(new_list, v1);
          plane->vertices = g_slist_remove(plane->vertices, v1);
/* rescan the list for the next adjacency */
          list2 = plane->vertices;

          v0 = v1;
          break;
          }
        }
      }

#if DEBUG_INIT_FACET
printf("current: %d : %d/%d\n", g_slist_length(new_list),
                                g_slist_length(plane->vertices), n);
#endif


    if (!list2) 
      list2 = plane->vertices;

if (g_slist_length(new_list) == n)
  break;

    }

#if DEBUG_INIT_FACET
printf("\n");
#endif

g_assert(g_slist_length(new_list) == n);

/* check the normal */
v0 = g_slist_nth_data(new_list, 0);
v1 = g_slist_nth_data(new_list, 1);
v2 = g_slist_nth_data(new_list, 2);

calc_norm(norm, v1->x, v0->x, v2->x);

if (via(norm, plane->norm, 3) < PI/2.0)
  new_list = g_slist_reverse(new_list);

g_slist_free(plane->vertices);
plane->vertices = new_list;

  }
}

/*****************************************************/
/* Main morphology call for vertex/facet enumeration */
/*****************************************************/
#define DEBUG_EXEC_CDD 0
gint exec_cdd(struct model_pak *data)
{
gint i, j, status, num_zero;
gdouble max;
rowrange m_input, m_size;
colrange n_input;
Amatrix AA;
GSList *plist;
struct plane_pak *pdata;

#if DEBUG_EXEC_CDD
printf("exec_cdd() start\n");
#endif

/* static initialization */
mprev=0;
beta = NULL;
mlast=0;

/* setup run flags */
HyperplaneOrder = LexMin;
AdjacencyTest = Combinatorial;
NondegAssumed = FALSE;
RecomputeRowOrder=TRUE;
PreOrderedRun=FALSE;
InitBasisAtBottom = FALSE;
Error=None;
CompStatus=InProgress;

m_input = data->num_planes;
n_input = NDIM;                   /* ineq -> len + h k l */
/* can be Real or Integer, makes no diff (even if input is float, strange) */
Number = Real;
/*
Number = Integer;
*/
RestrictedEnumeration = FALSE;
RelaxedEnumeration = FALSE;

/* feed planes into matrix */
max=0.0;
i=num_zero=0;
plist = data->planes;
while (plist != NULL)
  {
  pdata = (struct plane_pak *) plist->data;

/* fill out the matrix */
    AA[i]= (double *) calloc(n_input, sizeof(double));
    switch(data->morph_type)
      {
      case DHKL:
        AA[i][0] = (double) 1.0/fabs(pdata->dhkl);
        break;
      case EQUIL_UN:
        AA[i][0] = (double) fabs(pdata->esurf[0]);
        break;
      case EQUIL_RE:
        AA[i][0] = (double) fabs(pdata->esurf[1]);
        break;
      case GROWTH_UN:
        AA[i][0] = (double) fabs(pdata->eatt[0]);
        break;
      case GROWTH_RE:
        AA[i][0] = (double) fabs(pdata->eatt[1]);
        break;
      default:
        printf("Bad morph type.\n");
        return(1);
      }
/* input normalized miller indices */
    AA[i][1] = (double) pdata->x[0];
    AA[i][2] = (double) pdata->x[1];
    AA[i][3] = (double) pdata->x[2];
/* prevent 0 length faces from wiping out the hull */
    if (AA[i][0] < FRACTION_TOLERANCE)
      {
      for (j=1 ; j<n_input ; j++) 
        AA[i][j] = 0.0;
      num_zero++;
      }
/* get maximum - for later scaling */
    if (AA[i][0] > max)
      max = AA[i][0];
/* next row */
    i++;

/* next plane */
  plist = g_slist_next(plist);
  }

if (num_zero == i)
  {
#if DEBUG_EXEC_CDD
printf("No non-zero input planes found.\n");
#endif
  return(1);
  }
else
  {
/* there seem to be a lot of precision errors related to scaling */
/*
#define MORPH_SCALE 10.0
#define MORPH_SCALE 5.0
#define MORPH_SCALE 1.0
*/
/* 5 works best so far - need to find a better solution tho */
/* skipping this completely seems to work (fingers crossed) */
/*
  if (max > FRACTION_TOLERANCE)
    {
    for (j=0 ; j<i ; j++)
      AA[j][0] *= MORPH_SCALE/max;
    }
*/
  }

if (i != data->num_planes)
  {
  printf("Error: mismatched number of planes.\n");
  return(2);
  }

m_input = i;
m_size = m_input + 1;

AA[m_size-1] = (double *) calloc(n_input, sizeof(double));
/* artificial row for x_1 >= 0*/
AA[m_size-1][0] = 1.0;
for (j=1 ; j<n_input ; j++) 
  AA[m_size-1][j] = 0.0;

/* more init */
EqualityIndex = (long *) calloc(m_size+1, sizeof(EqualityIndex));
SetInequalitySets(m_size, EqualityIndex);

#if DEBUG_EXEC_CDD
printf("%d input inequalities:\n", (gint) m_size);
for (i=0 ; i<m_size ; i++)
  {
  for (j=0 ; j<n_input ; j++)
    {
    printf("%10.2f ",AA[i][j]);
    }
  printf("\n");
  }
printf("\n\n");
#endif

/* vertex/facets enumeration - and model_pak storage */
status = DDEnumerate(m_input, n_input, m_size, n_input, AA, data);

if (beta)
  g_free(beta);

return(status);
}

