/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "parse.h"
#include "sginfo.h"
#include "matrix.h"
#include "surface.h"
#include "spatial.h"
#include "task.h"
#include "numeric.h"
#include "interface.h"
#include "dialog.h"
#include "gtkshorts.h"
#include "opengl.h"

/* main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

#define FWHM_GAUSSIAN 0.4246609
/* MEH = m*e*e/2*h*h */
#define MEH 0.026629795

GtkWidget *diff_rad_type, *diff_rad_length;
GtkWidget *diff_broadening;
GtkWidget *diff_filename, *diff_plot;

enum {DIFF_XRAY, DIFF_NEUTRON, DIFF_ELECTRON,
      DIFF_GAUSSIAN, DIFF_LORENTZIAN, DIFF_PSEUDO_VOIGT};

/*****************************/
/* eliminate repeated planes */
/*****************************/
/* NB: assumes the input plane list is Dhkl sorted */
GSList *diff_get_unique_faces(GSList *list, struct model_pak *model)
{
gdouble delta, dhkl;
GSList *list1;
struct plane_pak *plane=NULL, *refplane=NULL;

dhkl = 999999999.9;
list1=list;
while (list1)
  {
  plane = (struct plane_pak *) list1->data;
  list1 = g_slist_next(list1);

/* determine if this is a repeated plane */
  delta = dhkl - plane->dhkl;
  if (delta > FRACTION_TOLERANCE)
    {
/*
printf("(%d %d %d) ", plane->index[0], plane->index[1], plane->index[2]);
    ret_list = g_slist_prepend(ret_list, plane);
*/

/* this is a new plane */
    refplane = plane;
    dhkl = refplane->dhkl;
    refplane->multiplicity = 1;
    }
  else
    {
/* related plane */
    list = g_slist_remove(list, plane);
    g_free(plane);
/*
printf("(%d %d %d) ", plane->index[0], plane->index[1], plane->index[2]);
*/
    refplane->multiplicity++;
    }
  }

return(list);
}

/**************************/
/* hkl enumeration (exp.) */
/**************************/
/* TODO - merge common functionality with get_ranked_faces() in surface.c */
#define DEBUG_DIFF_RANK 0
GSList *diff_get_ranked_faces(gdouble min, struct model_pak *model)
{
gint h, k, l, hs, ks, ls;
gdouble f1[3];
GSList *list1;
struct plane_pak *plane;

#define HKL_LIMIT 10

/* enumerate all unique hkl within defined limit */
#if DEBUG_DIFF_RANK
printf("trying:\n");
#endif
list1 = NULL;
for (hs=-1 ; hs<2 ; hs+= 2)
  {
/* start at 0 for -ve and at 1 for +ve */
  for (h=(hs+1)/2 ; h<HKL_LIMIT ; h++)
    {
    for (ks=-1 ; ks<2 ; ks+= 2)
      {
      for (k=(ks+1)/2 ; k<HKL_LIMIT ; k++)
        {
        for (ls=-1 ; ls<2 ; ls+= 2)
          {
          for (l=(ls+1)/2 ; l<HKL_LIMIT ; l++)
            {
#if DEBUG_DIFF_RANK
printf("(%d %d %d)", hs*h, ks*k, ls*l);
#endif
            if (!h && !k && !l)
              continue;
	    if (surf_sysabs(model, hs*h, ks*k, ls*l))
	      continue;
#if DEBUG_DIFF_RANK
printf("\n");
#endif
/* add the plane */
            VEC3SET(f1, hs*h, ks*k, ls*l);
            plane = fn_make_plane(f1, model);
	    if (!plane)
	      continue;

            if (plane->dhkl > min)
	      list1 = g_slist_prepend(list1, plane);
	    else
	      break;
            }
          }
        }
      }
    }
  }

/* sort via Dhkl */
list1 = g_slist_sort(list1, (gpointer) dhkl_compare);
/* remove faces with same Dhkl (NB: assumes they are sorted!) */
list1 = diff_get_unique_faces(list1, model);

return(list1);
}

/************************************/
/* compute atomic scattering factor */
/************************************/
/* FIXME - gain speed by prefeching for unique_atom_list */
gdouble diff_get_sfc(gint type, gdouble sol, struct core_pak *core)
{
gint i;
gboolean flag;
gdouble a, b, sfc, sol2;
GSList *list;

/* FIXME - CU will not match Cu in hash table lookup */
/*
list = g_hash_table_lookup(sysenv.sfc_table, core->label);
*/
list = g_hash_table_lookup(sysenv.sfc_table, elements[core->atom_code].symbol);

if (!list)
  {
  printf("No sfc found for [%s], fudging...\n", core->label);
  return((gdouble) core->atom_code);
  }

/* eforce list length */
g_assert(g_slist_length(list) > 8);
flag = FALSE;
switch (type)
  {
  case DIFF_ELECTRON:
    flag = TRUE;
  case DIFF_XRAY:
/* wave form factor */
    sfc = *((gdouble *) g_slist_nth_data(list, 8));
    sol2 = sol*sol;
    for (i=0 ; i<4 ; i++)
      {
      a = *((gdouble *) g_slist_nth_data(list, 2*i));
      b = *((gdouble *) g_slist_nth_data(list, 2*i+1));
      sfc += a*exp(-b*sol2);
      }
/* NEW - electron correction */
    if (flag)
      sfc = MEH * ((gdouble) core->atom_code - sfc) / sol;
    break;

  case DIFF_NEUTRON:
    sfc = *((gdouble *) g_slist_nth_data(list, 9));
    break;

  default:
    printf("Unsuported radiation type, fudging...\n");
    sfc = core->atom_code;
    break;
  }

/*
printf("[%s] %d : %f\n", core->label, core->atom_code, sfc);
*/

return(sfc);
}

/******************************/
/* diffraction output routine */
/******************************/
#define DEBUG_CALC_SPECTRUM 0
void diff_calc_spectrum(GSList *list, struct model_pak *model)
{
gint i, j, n;
gchar *cmd, *temp;
gdouble angle, sol, f, g, lpf;
gdouble dr, g2, dwf, fwhm, bf, s;
gdouble rdv[3];
gdouble *spectrum;
GSList *item;
struct plane_pak *pdata;
FILE *fp_plot, *fp_raw;

/* init spectrum sampling */
n = model->diffract.theta[1]/model->diffract.theta[2] + 1;
spectrum = g_malloc(n * sizeof(gdouble));

/* loop - spectrum boxes */
for (i=0 ; i<n ; i++)
  {
  spectrum[i] = 0.0;

/* loop - sum contribution from peaks */
  for (item=list ; item ; item=g_slist_next(item))
    {
    pdata = (struct plane_pak *) item->data;

/* structure factor squared */
    f = pdata->f[0]*pdata->f[0] + pdata->f[1]*pdata->f[1];

/* sin(angle) / lambda */
    sol = 0.5/pdata->dhkl;
/* get 2theta value for printout */
    angle = 2.0*asin(sol*model->diffract.wavelength);

/* lorentz polarization factor (unpolarized incident beam) */
    if (model->diffract.radiation == DIFF_XRAY)
      {
      lpf = 0.5*(1.0 + tbl_cos(angle)*tbl_cos(angle));
      lpf /= (tbl_sin(angle) * tbl_sin(angle/2.0));
      }
    else
      lpf = 1.0;

/* TODO - incorporate the debye-waller factor correctly */
    ARR3SET(rdv, pdata->index);
    vecmat(model->rlatmat, rdv);
    g2 = VEC3MAGSQ(rdv);
    dwf = exp(-1.0*g2/3.0);
/* FIXME - fudge until the above is corrected */
dwf = 1.0;

/* current spectrum interval */
    j = 0.5 + (R2D*angle)/model->diffract.theta[2];

/* distance to the peak maximum (NB: whole units of THETA_SI) */
    dr = (j - i) * model->diffract.theta[2];

/* broadening */
    fwhm = sqrt(model->diffract.u + model->diffract.v*tan(angle/2.0)
                           + model->diffract.w*tan(angle/2.0)*tan(angle/2.0));

/* broadening factor */
    bf = 1.0;
    s = fwhm;
    if (s)
      {
      switch (model->diffract.broadening)
        {
        case DIFF_GAUSSIAN:
          s *= FWHM_GAUSSIAN;
          bf = exp(-dr*dr/(2.0*s*s))/(s*sqrt(2.0*PI));
          break;
        case DIFF_LORENTZIAN:
          bf = fwhm/(2.0*PI*(dr*dr + 0.25*fwhm*fwhm));
          break;

        case DIFF_PSEUDO_VOIGT:
#define C1 4*log(2)
/* mixing parameter */
          g = model->diffract.asym;
/* lorentzian */
          bf = g * 2.0 / (fwhm * PI * (1.0 + 4.0*(dr*dr/(fwhm*fwhm))));
/* gaussian */
          bf +=  (1-g) * sqrt(C1) * exp(-C1*dr*dr/(fwhm*fwhm)) / (fwhm*PI);
          break;

        default:
          g_assert_not_reached();
        }
      }
    else
      {
      if (dr)
        bf = 0.0;
      }
/* sum the intensity */
    spectrum[i] += lpf*f*bf*dwf;

#if DEBUG_CALC_SPECTRUM
  if (!dr)
  printf("[%d %d %d] : 2theta = %f : F2 = %f\n",
          pdata->index[0], pdata->index[1], pdata->index[2], R2D*angle, f);
#endif

    }
  }

/* save the spectrum */
temp = gun("dat");
fp_plot = fopen(temp, "wt");
fp_raw = fopen(gtk_entry_get_text(GTK_ENTRY(diff_filename)), "wt");
if (!fp_plot || !fp_raw)
  {
  show_text(ERROR, "Failed to save spectrum data.\n");
  return;
  }

/* TODO - legend showing wavelength */
/* radiation dependant title */
switch (model->diffract.radiation)
  {
  case DIFF_ELECTRON:
    fprintf(fp_plot, "@ title \"Electron powder pattern for %s\"\n", model->basename);
    break;
  case DIFF_NEUTRON:
    fprintf(fp_plot, "@ title \"Neutron powder pattern for %s\"\n", model->basename);
    break;
  case DIFF_XRAY:
    fprintf(fp_plot, "@ title \"XRay powder pattern for %s\"\n", model->basename);
    break;
  }
fprintf(fp_plot, "@ xaxis label \"2theta\"\n");
fprintf(fp_plot, "@ yaxis label \"Intensity\"\n");

for (i=0 ; i<n ; i++)
  {
/* minimum value check */
  if (i*model->diffract.theta[2] > model->diffract.theta[0])
    {
    fprintf(fp_raw, "%5.4f %f\n", i*model->diffract.theta[2], spectrum[i]);
    fprintf(fp_plot, "%5.4f %f\n", i*model->diffract.theta[2], spectrum[i]);
    }
  }
fclose(fp_raw);
fclose(fp_plot);

/* send to grace */
if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(diff_plot)))
  {
/* the free effectively fits the graph to the window */
  cmd = g_strdup_printf("%s -free \"%s\"", sysenv.grace_path, temp);
  g_spawn_command_line_async(cmd, NULL);

  g_free(temp);
  g_free(cmd);
  }

/* cleanup */
g_free(spectrum);
}

/****************************/
/* main diffraction routine */
/****************************/
#define DEBUG_DIFF_CALC 0
gint diff_calc(void)
{
gdouble tmp, sfc, sol, tc, ts;
gdouble vec[3], rdv[3];
const gchar *text;
GSList *item, *list, *clist;
struct model_pak *model;
struct plane_pak *pdata;
struct core_pak *core;

model = model_ptr(sysenv.active, RECALL);
if (!model)
  return(1);
if (model->periodic != 3)
  {
  show_text(WARNING, "Your model is not 3D periodic.\n");
  return(1);
  }

text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(diff_rad_type)->entry));
if (g_ascii_strncasecmp(text, "Electron", 8) == 0)
  model->diffract.radiation = DIFF_ELECTRON;
if (g_ascii_strncasecmp(text, "Neutron", 7) == 0)
  model->diffract.radiation = DIFF_NEUTRON;
if (g_ascii_strncasecmp(text, "X-Ray", 5) == 0)
  model->diffract.radiation = DIFF_XRAY;

text = gtk_entry_get_text(GTK_ENTRY(diff_rad_length));
model->diffract.wavelength = str_to_float(text);

text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(diff_broadening)->entry));

if (g_ascii_strncasecmp(text, "Gaussian", 8) == 0)
  model->diffract.broadening = DIFF_GAUSSIAN;
if (g_ascii_strncasecmp(text, "Lorentzian", 10) == 0)
  model->diffract.broadening = DIFF_LORENTZIAN;
if (g_ascii_strncasecmp(text, "Pseudo-Voigt", 13) == 0)
  model->diffract.broadening = DIFF_PSEUDO_VOIGT;

/* TODO - enforce theta min < max */

#if DEBUG_DIFF_CALC
printf(" radiation = %d\n", model->diffract.radiation);
printf("wavelength = %f\n", model->diffract.wavelength);
printf("     theta = %f %f %f\n", model->diffract.theta[0], model->diffract.theta[1], model->diffract.theta[2]);
printf("      fwhm = %f %f %f\n", model->diffract.u, model->diffract.v, model->diffract.w);
#endif

/* get min Dhkl for (specified) 2theta max */
tmp = 0.5*model->diffract.wavelength/tbl_sin(0.5*D2R*model->diffract.theta[1]);

/* calculate and loop over all planes satisfying the above */
list = diff_get_ranked_faces(tmp, model);

for (item=list ; item ; item=g_slist_next(item))
  {
  pdata = (struct plane_pak *) item->data;

/* sin(angle) / lambda */
  sol = 0.5/pdata->dhkl;

/* valid range for scattering coefficients */
g_assert(sol < 2.0);

  pdata->f[0] = 0.0;
  pdata->f[1] = 0.0;
  for (clist=model->cores ; clist ; clist=g_slist_next(clist))
    {
    core = (struct core_pak *) clist->data;

/* reciprocal lattice vector */
    ARR3SET(rdv, pdata->index);
    vecmat(model->rlatmat, rdv);
/* core position */
    ARR3SET(vec, core->x);
    vecmat(model->latmat, vec);
/* dot product */    
    ARR3MUL(rdv, vec);

    tmp = 2.0 * PI * (rdv[0] + rdv[1] + rdv[2]);
    tc = tbl_cos(tmp);
    ts = tbl_sin(tmp);

/* get the atomic form factor */
    sfc = diff_get_sfc(model->diffract.radiation, sol, core);

/* structure factor sum */
    pdata->f[0] += sfc*tc;
    pdata->f[1] += sfc*ts;
    }
  }

/* compute/plot intensities */
diff_calc_spectrum(list, model);

free_slist(list);

return(0);
}

/****************************/
/* diffraction setup dialog */
/****************************/
void diffract_dialog(void)
{
gint id;
GtkWidget *frame, *hbox, *hbox2, *vbox, *label;
GList *list;
struct dialog_pak *dialog;
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;
if (data->periodic != 3)
  {
  show_text(WARNING, "Your model is not 3D periodic.\n");
  return;
  }

/* request an energetics dialog */
if ((id = request_dialog(data->number, DIFFAX)) < 0)
  return;
dialog = &sysenv.dialog[id];

/* create the dialog widget */
dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW(dialog->win), "Powder Diffraction");
g_signal_connect(GTK_OBJECT(dialog->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* radiation details */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

label = gtk_label_new(g_strdup_printf(" Radiation "));
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
list = NULL;
list = g_list_prepend(list, "Electrons");
list = g_list_prepend(list, "Neutrons");
list = g_list_prepend(list, "X-Rays");
diff_rad_type = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(diff_rad_type)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(diff_rad_type), list);
gtk_box_pack_end(GTK_BOX(hbox), diff_rad_type, FALSE, FALSE, PANEL_SPACING);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

label = gtk_label_new(g_strdup_printf(" Wavelength "));
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

diff_rad_length = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(diff_rad_length),
                   g_strdup_printf("%9.6f", data->diffract.wavelength));
gtk_box_pack_end(GTK_BOX(hbox), diff_rad_length, FALSE, FALSE, 0);

/* peak broadening */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

label = gtk_label_new(g_strdup_printf(" Broadening function "));
gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

list = NULL;
list = g_list_prepend(list, "Pseudo-Voigt");
list = g_list_prepend(list, "Lorentzian");
list = g_list_prepend(list, "Gaussian");
diff_broadening = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(diff_broadening)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(diff_broadening), list);
gtk_box_pack_end(GTK_BOX(hbox), diff_broadening, FALSE, FALSE, PANEL_SPACING);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);
gtksh_auto_spin(" Asymmetric mixing component ", &data->diffract.asym,
                                  0.0, 1.0, 0.01, NULL, NULL, hbox);

/* split pane */
hbox2 = gtk_hbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->vbox), hbox2, TRUE, TRUE, 0);

/* angle range */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, PANEL_SPACING);
gtksh_auto_spin(" 2Theta min", &data->diffract.theta[0], 0.0, 160.0, 5.0, NULL, NULL, hbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, PANEL_SPACING);
gtksh_auto_spin(" 2Theta max", &data->diffract.theta[1], 0.0, 170.0, 5.0, NULL, NULL, hbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, PANEL_SPACING);
gtksh_auto_spin(" 2Theta step", &data->diffract.theta[2], 0.1, 5.0, 0.1, NULL, NULL, hbox);


/* U V W broadening parameters */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);
gtksh_auto_spin(" U ", &data->diffract.u, -9.9, 9.9, 0.05, NULL, NULL, hbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);
gtksh_auto_spin(" V ", &data->diffract.v, -9.9, 9.9, 0.05, NULL, NULL, hbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);
gtksh_auto_spin(" W ", &data->diffract.w, -9.9, 9.9, 0.05, NULL, NULL, hbox);


/* output stuff */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

label = gtk_label_new(g_strdup_printf(" Raw data save file "));
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

diff_filename = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(diff_filename), data->basename);
gtk_box_pack_end(GTK_BOX(hbox), diff_filename, FALSE, FALSE, 0);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);
if (sysenv.grace_path)
  diff_plot = new_check_button("Plot spectrum ", NULL, NULL, 1, hbox);
else
  {
  diff_plot = new_check_button("Plot spectrum ", NULL, NULL, 0, hbox);
  gtk_widget_set_sensitive(GTK_WIDGET(diff_plot), FALSE);
  }

/* terminating buttons */
gtksh_stock_button(GTK_STOCK_EXECUTE,
                   diff_calc, NULL, 
                   GTK_DIALOG(dialog->win)->action_area);

gtksh_stock_button(GTK_STOCK_CLOSE,
                   event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(dialog->win)->action_area);

gtk_widget_show_all(dialog->win);
}

