/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __sun
#include <sys/dirent.h>
#else
#ifdef __WIN32
#include <dirent.h>
#else
#include <sys/dir.h>
#include <sys/param.h>
#endif
#endif

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "interface.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/******************/
/* Biosym writing */
/******************/
gint write_arc(gchar *filename, struct model_pak *data)
{
gint m, flag;
gdouble x[3];
GSList *clist, *mlist;
struct core_pak *core;
struct mol_pak *mol;
time_t t1;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 2);

/* open the file */
fp = fopen(filename,"wt");
if (!fp)
  return(3);

/* TODO - write multiple frames if it has them */

/* print header */
fprintf(fp,"!BIOSYM archive 3\n");
if (data->periodic)
  fprintf(fp,"PBC=ON\n");
else
  fprintf(fp,"PBC=OFF\n");
gdis_blurb(fp);

/* get & print the date (streamlined slightly by sxm) */
t1 = time(NULL);
fprintf(fp,"!DATE %s",ctime(&t1));

if (data->periodic)
  {
  if (data->periodic == 2)
    {
/* saving a surface as a 3D model - make depth large enough to fit everything */
    fprintf(fp,"PBC %9.4f %9.4f %9.4f %9.4f %9.4f %9.4f (P1)\n",
                data->pbc[0], data->pbc[1], 2.0*data->rmax,
                180.0*data->pbc[3]/PI,
                180.0*data->pbc[4]/PI,
                180.0*data->pbc[5]/PI);
    }
  else
    {
    fprintf(fp,"PBC %9.4f %9.4f %9.4f %9.4f %9.4f %9.4f (P1)\n",
                data->pbc[0], data->pbc[1], data->pbc[2],
                180.0*data->pbc[3]/PI,
                180.0*data->pbc[4]/PI,
                180.0*data->pbc[5]/PI);
    }
  }



m=0;
for (mlist=data->moles ; mlist ; mlist=g_slist_next(mlist))
  {
  mol = (struct mol_pak *) mlist->data;

/* flag - atom written to file? (ie if UNDELETED) */
  flag=0;
/* m - number of atoms in molecule */
  for (clist=mol->cores ; clist ; clist=g_slist_next(clist))
    {
    core = (struct core_pak *) clist->data;
    if (core->status & DELETED)
      continue;

/* everything is cartesian after latmat mult */
    ARR3SET(x, core->x);
    vecmat(data->latmat, x);
/* unique molecule numbering for BIOSYM files (>0) */
    fprintf(fp, "%-4s  %14.9f %14.9f %14.9f CORE %-6d ?       %-2s ",
                core->label,x[0],x[1],x[2],m+1, 
                elements[core->atom_code].symbol);

if (core->charge < 0.0)
  fprintf(fp, "%5.3f\n", core->charge);
else
  fprintf(fp, " %4.3f\n", core->charge);

/* indicate we've written at least one atom */
    flag++;
    }
/* if at least one atom in the molecule was written - inc mol numbering */
  if (flag)
    {
    fprintf(fp,"end\n");
    m++;
    }
  }
fprintf(fp,"end\n");

fclose(fp);
return(0);
}

/*************************************************/
/* read a car/arc block into the model structure */
/*************************************************/
/* NB: assumes fp is at a !DATE line */
void read_arc_block(FILE *fp, struct model_pak *data)
{
gint n, end_count, num_tokens;
gchar **buff;
GSList *clist;
struct core_pak *core;

g_assert(fp != NULL);
g_assert(data != NULL);

/* FIXME - this breaks measurement updates */
/*
free_core_list(data);
*/

/* init */
clist = data->cores;
n=end_count=0;

/* loop while there's data */
for (;;)
  {
  buff = get_tokenized_line(fp, &num_tokens);
  if (!buff)
    break;

/* increment/reset end counter according to input line */
  if (g_ascii_strncasecmp("end", *buff, 3) == 0)
    end_count++;
  else
    end_count = 0;

/* skip single end, terminate on double end */
  if (end_count == 1)
    continue;
  if (end_count == 2)
    break;

/* cell dimension search */
  if (g_ascii_strncasecmp("pbc", *buff, 3) == 0)
    {
    if (num_tokens > 6)
      {
      data->pbc[0] = str_to_float(*(buff+1));
      data->pbc[1] = str_to_float(*(buff+2));
      data->pbc[2] = str_to_float(*(buff+3));
      data->pbc[3] = PI*str_to_float(*(buff+4))/180.0;
      data->pbc[4] = PI*str_to_float(*(buff+5))/180.0;
      data->pbc[5] = PI*str_to_float(*(buff+6))/180.0;
      }
/* cope with 2D */
    if (data->pbc[2] == 0)
      data->periodic = 2;
    continue;
    }

/* process a single coord line */
  if (num_tokens < 8) {
    show_text(ERROR, "unexpected end of file reading arc file\n");
    return;
  }
  if (elem_type(*(buff+7)) && num_tokens > 8)
    {
/* overwrite existing core (if any) */
    if (clist)
      {
      core = (struct core_pak *) clist->data;
      clist = g_slist_next(clist);
      }
    else
      {
/* otherwise, add a new core */
/* NB: must append (not prepend) since we may overwrite & the order must be the same */
/* TODO - we could prepend cores (it's faster) and then reverse the list at the end */
/* but it's more complicated as we should only reverse the newly added cores */
      core = new_core(*(buff+7), data);
      data->cores = g_slist_append(data->cores, core);
      }

    core->x[0] = str_to_float(*(buff+1));
    core->x[1] = str_to_float(*(buff+2));
    core->x[2] = str_to_float(*(buff+3));
    core->charge = str_to_float(*(buff+8));
    core->lookup_charge = FALSE;
    }

  g_strfreev(buff);
  }
}

/*********************/
/* biosym frame read */
/*********************/
gint read_arc_frame(FILE *fp, struct model_pak *data)
{
/* frame overwrite */
read_arc_block(fp, data);
return(0);
}

/******************/
/* Biosym reading */
/******************/
#define DEBUG_READ_ARC 0
gint read_arc(gchar *filename, struct model_pak *data)
{
gint frame=0, num_tokens;
gchar **buff;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 2);

fp = fopen(filename, "rt");
if (!fp)
  return(3);

/* loop while there's data */
for (;;)
  {
  buff = get_tokenized_line(fp, &num_tokens);
  if (!buff)
    break;

/* pbc on/off search */
  if (g_ascii_strncasecmp("pbc=on", *buff, 6) == 0)
    data->periodic = 3;
  if (g_ascii_strncasecmp("pbc=off", *buff, 7) == 0)
    data->periodic = 0;

/* coords search */
  if (g_ascii_strncasecmp("!date", *buff, 5) == 0)
    {
/* NEW */
    add_frame_offset(fp, data);

/* only load the required frame */
    if (frame == data->cur_frame)
      read_arc_block(fp, data);

/* increment counter */
    frame++;
    }
  g_strfreev(buff);
  }

/* got everything */
data->num_frames = frame;

/* model setup */
strcpy(data->filename, filename);
g_free(data->basename);
data->basename = strdup_basename(filename);

prep_model(data);

return(0);
}

