/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __sun
#include <sys/dirent.h>
#else
#ifdef __WIN32
#include <dirent.h>
#else
#include <sys/dir.h>
#include <sys/param.h>
#endif
#endif

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "interface.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/*******************************/
/* MARVIN specific file saving */
/*******************************/
gint write_marvin(gchar *filename, struct model_pak *data)
{
gint mol, region;
gdouble vec[3];
GSList *list;
struct core_pak *core;
struct shel_pak *shel;
FILE *fp;

/* open the file */
fp = fopen(filename,"w");
if (!fp)
  return(1);

show_text(STANDARD, "Saving file in MARVIN restart format!\n");

/* save via the number of molecules in the */
/* model - since this allows mol numbering */
mol=0;

/* four region types printing */
for (region=REGION1A ; region<REGION2B ; region++)
  {
  if (data->region_empty[region])
    continue;
  switch(region)
    {
    case REGION1A:
      fprintf(fp,"coordinates 1 A\n");
      break;
    case REGION2A:
      fprintf(fp,"coordinates 2 A\n");
      break;
    case REGION1B:
      fprintf(fp,"coordinates 1 B\n");
      break;
    case REGION2B:
      fprintf(fp,"coordinates 2 B\n");
      break;
    }

/* do atoms */
  for (list=data->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;

/* don't print checks */
    if (core->region != region)
      continue;
    if (core->status & DELETED)
      continue;

/* NB: x,y,z have been fractionalized */
    ARR3SET(vec, core->x);
    vecmat(data->latmat, vec);
    fprintf(fp,"%-7s   core   %14.9f  %14.9f  %14.9f   %14.9f\n",
               core->label, vec[0], vec[1], vec[2], core->charge);
/* FIXME = how to get proper rotated coords? */
/* shell? */
    if (core->shell)
      {
      shel = (struct shel_pak *) core->shell;

      ARR3SET(vec, shel->x);
      vecmat(data->latmat, vec);
      fprintf(fp,"%-7s   shel   %14.9f  %14.9f  %14.9f\n",
             shel->label, vec[0], vec[1], vec[2]);
      }
    }
/* done region */
  fprintf(fp,"end\n");
  }

/* done */
fclose(fp);
return(0);
}

/****************************/
/* marvin output file parse */
/****************************/
#define DEBUG_LOAD_MVNOUT 0
gint read_mvnout(gchar *filename, struct model_pak *data)
{
gint i, j, k, current_block, region, status, offset, model;
gint block[4];
gchar line[LINELEN], **key, **buff;
gdouble veca[2], vecb[2];
gdouble version;
struct core_pak *core;
struct shel_pak *shel;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 1);

fp = fopen(filename, "rt");
if (!fp)
  return(1);

model = data->number;

/* setup */
data->id = MVNOUT;
data->periodic = 2;
strcpy(data->filename, filename);
g_free(data->basename);
data->basename = strdup_basename(filename);

/* starting locations */
block[0] = -1;
block[1] = -1;
block[2] = -1;
block[3] = -1;
current_block = 0;

/*****************************************/
/* parse 1 - get location of data blocks */
/*****************************************/
/* scheme - ignore all but the last block of a particular region */
/* read until EOF all coordinate BLOCK data */
status=offset=0;
while (!fgetline(fp,line))
  {
  key = get_tokens(line, 7);
/* YAMH - get version to determine a parsing parameter */
  if (g_ascii_strncasecmp(line,"                    Version",27) == 0)
    {
    sscanf(*(key+1),"%lf",&version);
    offset = (version < 1.99) ? 0 : 1;
#if DEBUG_LOAD_MVNOUT
printf("Marvin version %lf, offset = %d\n",version,offset);
#endif 
    }
/* get surface vectors */
  if (g_ascii_strncasecmp(line,"Surface Vectors:",16) == 0)
    {
/* read them in */
    sscanf(*(key+2),"%lf",&veca[0]);
    sscanf(*(key+3),"%lf",&veca[1]);

    if (fgetline(fp,line))
      return(3);
    buff = get_tokens(line, 2);
    sscanf(*buff,"%lf",&vecb[0]);
    sscanf(*(buff+1),"%lf",&vecb[1]);
    g_strfreev(buff);
/* NB: gdis stores vectors in reverse order */
    VEC3SET(&(data->latmat[0]), veca[0], vecb[0], 0.0);
    VEC3SET(&(data->latmat[3]), veca[1], vecb[1], 0.0);
    VEC3SET(&(data->latmat[6]), 0.0, 0.0, 1.0);
    data->construct_pbc = TRUE;
    data->periodic = 2;
/*
    make_latmat(data);
*/

    status++;
    continue;
    }
/* NB: disallow image creation in this direction */
data->pbc[2] = 0.0;

/* else, BLOCK (data) keyword check */
/* NB: marvin has block and BLOCK littered about, want BLOCK only */
  if (strncmp(*key,"BLOCK",5) == 0)
    {
    current_block++;
/* is it region data? */
    if (g_ascii_strncasecmp(*(key+2),"region",5) != 0)
      continue;
/* get region type */
    if (g_ascii_strncasecmp(*(key+5),"1a",2) == 0)
      region = REGION1A;
    else if (g_ascii_strncasecmp(*(key+5),"1b",2) == 0)
      region = REGION1B;
    else if (g_ascii_strncasecmp(*(key+5),"2a",2) == 0)
      region = REGION2A;
    else if (g_ascii_strncasecmp(*(key+5),"2b",2) == 0)
      region = REGION2B;
    else
      continue;
/* is it coordinate data? (don't want gradients) */
    fgetline(fp,line);
    fgetline(fp,line);

    buff = get_tokens(line, 4);
    if (g_ascii_strncasecmp(*(buff+3),"coordinates",11) != 0)
      continue;
    block[region] = current_block;
    g_strfreev(buff);

#if DEBUG_LOAD_MVNOUT
printf("Found region type: [%d], block id: %d\n",region,current_block);
#endif
    data->region_empty[region] = FALSE;
    status++;
    }
  g_strfreev(key);
  }

/* checks, return bad status if a key item was missing */
if (status < 2)
  return(status);

/**************************/
/* parse 2 - get the data */
/**************************/
rewind(fp);
current_block = 0;
/* core/shell counters */
i=j=0;
while (!fgetline(fp,line))
  {
/* go through the regions and see which block match should be read */
  if (strncmp(line,"BLOCK",5) == 0)
    {
    current_block++;
    region = -1;
    for (k=REGION1A ; k<REGION2B ; k++)
      if (block[k] == current_block)
        region = k;
    if (region == -1)
      continue;
/* skip straight to data */
    fgetline(fp,line);
    fgetline(fp,line);
    fgetline(fp,line);
    fgetline(fp,line);
/* read it in */
#if DEBUG_LOAD_MVNOUT
printf("Block id: %d, reading region type: %d\n",current_block,region);
#endif
    while(g_ascii_strncasecmp(line,"-------",7) != 0)
      {
      buff = get_tokens(line, 7);
/* NB: marvin 1.8 & 2.0 have this at different positions! */
      if (g_ascii_strncasecmp(*(buff+offset+2),"cor",3) == 0)
        {
        core = new_core(*(buff+offset+1), data);
        data->cores = g_slist_prepend(data->cores, core);

        core->region = region;
        core->x[0] = str_to_float(*(buff+offset+3));
        core->x[1] = str_to_float(*(buff+offset+4));
        core->x[2] = str_to_float(*(buff+offset+5));
        core->charge = str_to_float(*(buff+offset+6));
        core->lookup_charge = FALSE;
        }
      else if (g_ascii_strncasecmp(*(buff+offset+2),"she",3) == 0)
        {
        shel = new_shell(*(buff+offset+1), data);
        data->shels = g_slist_prepend(data->shels, shel);

        shel->region = region;
        shel->x[0] = str_to_float(*(buff+offset+3));
        shel->x[1] = str_to_float(*(buff+offset+4));
        shel->x[2] = str_to_float(*(buff+offset+5));
        shel->charge = str_to_float(*(buff+offset+6));
        shel->lookup_charge = FALSE;
        }
      else
        {
        printf("load_mvnout(): parse error.\n");
printf("[%s]\n",line);
        break;
        }
/* get next line, terminate on EOF */
      if (fgetline(fp,line))
        {
        printf("Warning: EOF in data block.\n");
        break;
        }
      g_strfreev(buff);
      }
    }

/* energy searches */
  if (strncmp(line,"Attachment Energy",17) == 0)
    {
    buff = get_tokens(line, 5);
    if (data->gulp.eatt[0] == 0.0)
      data->gulp.eatt[0] = str_to_float(*(buff+3));
    else
      data->gulp.eatt[1] = str_to_float(*(buff+3));
    g_free(data->gulp.eatt_units);
    data->gulp.eatt_units = g_strdup(*(buff+4));
    g_strfreev(buff);
    }
  if (strncmp(line,"Surface Energy",14) == 0)
    {
    buff = get_tokens(line, 6);
    if (data->gulp.esurf[0] == 0.0)
      data->gulp.esurf[0] = str_to_float(*(buff+3));
    else
      data->gulp.esurf[1] = str_to_float(*(buff+3));
    g_free(data->gulp.esurf_units);
    data->gulp.esurf_units = g_strdup(*(buff+5));
    g_strfreev(buff);
    }
  }

/* init lighting */
j=0;
for (i=REGION1A ; i<REGION2B ; i++)
  {
  if (!data->region_empty[i] && !j)
    {
    data->region[i] = TRUE;
    j++;
    }
  else
    data->region[i] = FALSE;
  }
   
#if DEBUG_LOAD_MVNOUT
printf("Surface vectors: %lf x %lf\n",data->pbc[0],data->pbc[1]);
#endif

/* force non periodic in z (marvin is 2D) */
data->pbc[2] = 0;
prep_model(data);

fclose(fp);
return(0);
}

/****************/
/* file reading */
/****************/
#define DEBUG_READ_MARVIN 0
gint read_marvin(gchar *filename, struct model_pak *data)
{
gint num_tokens, keyword, region;
gchar **buff, *tmp, line[LINELEN];
GSList *list;
struct core_pak *core;
struct shel_pak *shel;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 1);

fp = fopen(filename, "rt");
if (!fp)
  return(1);

/* defaults */
/*
data->periodic = 2;
*/

data->region_empty[REGION1A] = TRUE;
data->region_empty[REGION2A] = TRUE;
data->region_empty[REGION1B] = TRUE;
data->region_empty[REGION2B] = TRUE;

keyword = COMPARE;
region = REGION1A;

while(!fgetline(fp, line))
  {
  list = get_keywords(line);
  if (list != NULL)
    {
    keyword = GPOINTER_TO_INT(list->data);
    switch(keyword)
      {
      case MARVIN_COORD:
        tmp = g_strstrip(g_strdup(get_token_pos(line, 1)));

        if (g_ascii_strncasecmp("1 a", tmp, 3) == 0)
          region = REGION1A;
        if (g_ascii_strncasecmp("2 a", tmp, 3) == 0)
          region = REGION2A;
        if (g_ascii_strncasecmp("1 b", tmp, 3) == 0)
          region = REGION1B;
        if (g_ascii_strncasecmp("2 b", tmp, 3) == 0)
          region = REGION2B;

        data->region_empty[region] = FALSE;

        g_free(tmp);
        break;
      }
    }
  else
    {
    buff = tokenize(line, &num_tokens);
/* have we got a valid element to read */
    if (buff)
      {
    if (elem_type(*buff))
      {
      switch(keyword)
        {
        case MARVIN_COORD:
        case MARVIN_BASIS:
          if (num_tokens > 4)
            {
            if (g_ascii_strncasecmp("cor", *(buff+1), 3) == 0)
              {
              core = new_core(*buff, data);
              data->cores = g_slist_prepend(data->cores, core);

              core->x[0] = str_to_float(*(buff+2));
              core->x[1] = str_to_float(*(buff+3));
              core->x[2] = str_to_float(*(buff+4));
              }
            if (g_ascii_strncasecmp("shel", *(buff+1), 4) == 0)
              {
              shel = new_shell(*buff, data);
              data->shels = g_slist_prepend(data->shels, shel);

              shel->region = region;
              shel->x[0] = str_to_float(*(buff+2));
              shel->x[1] = str_to_float(*(buff+3));
              shel->x[2] = str_to_float(*(buff+4));
              }
            }
          break;
        }
      }
    else
      {
      if (g_ascii_strncasecmp(line, "latvec", 6) == 0)
        {
        data->periodic = 2; 
        if (!fgetline(fp, line))
          {
          g_strfreev(buff);
          buff = tokenize(line, &num_tokens);
          data->latmat[0] = str_to_float(*(buff+0));
          data->latmat[1] = str_to_float(*(buff+1));
          data->latmat[2] = str_to_float(*(buff+2));
          }
        if (!fgetline(fp, line))
          {
          g_strfreev(buff);
          buff = tokenize(line, &num_tokens);
          data->latmat[3] = str_to_float(*(buff+0));
          data->latmat[4] = str_to_float(*(buff+1));
          data->latmat[5] = str_to_float(*(buff+2));
          }
        if (!fgetline(fp, line))
          {
          g_strfreev(buff);
          buff = tokenize(line, &num_tokens);
          data->latmat[6] = str_to_float(*(buff+0));
          data->latmat[7] = str_to_float(*(buff+1));
          data->latmat[8] = str_to_float(*(buff+2));
          }
        data->construct_pbc = TRUE;
        }
      }
      }
    g_strfreev(buff);
    }
  }

/* model setup */
strcpy(data->filename, filename);
g_free(data->basename);
data->basename = strdup_basename(filename);

prep_model(data);

fclose(fp);
return(0);
}

