/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __sun
#include <sys/dirent.h>
#else
#ifdef __WIN32
#include <dirent.h>
#else
#include <sys/dir.h>
#include <sys/param.h>
#endif
#endif

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "interface.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/**********************/
/* save in XTL format */
/**********************/
gint write_xtl(gchar *filename, struct model_pak *data)
{
gint invert=FALSE;
gdouble vec[3], tmat[9];
GSList *list;
struct core_pak *core;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(data->periodic, 1);

/* open the file */
fp = fopen(filename,"w");
if (!fp)
  return(1);

show_text(STANDARD, "Saving file in XTL format!\n");

/* header */
fprintf(fp,"TITLE %s\n",data->basename);
fprintf(fp,"DIMENSION %d\n",data->periodic);
fprintf(fp,"CELL\n");
if (data->periodic == 3)
  fprintf(fp,"%f  %f  %f  %f  %f  %f\n",data->pbc[0],data->pbc[1],data->pbc[2],
          180.0*data->pbc[3]/PI, 180.0*data->pbc[4]/PI, 180.0*data->pbc[5]/PI);
else
  fprintf(fp,"%f  %f  %f\n",data->pbc[0],data->pbc[1], 180.0*data->pbc[5]/PI);

fprintf(fp,"SYMMETRY NUMBER %d\n", data->sginfo.spacenum);
if (data->sginfo.cellchoice)
  fprintf(fp,"SYMMETRY QUALIFIER ORIGIN_%d\n", data->sginfo.cellchoice);
/* coords */
fprintf(fp,"ATOMS\n");
fprintf(fp,"NAME       X          Y          Z     CHARGE   TEMP    OCCUP   SCAT\n");

/* Cerius2 expects the opposite coordinate ordering for surfaces */
/* is it a surface with negative z? */
if (data->periodic == 2)
  {
  if (g_slist_length(data->cores))
    {
    core = (struct core_pak *) g_slist_nth_data(data->cores, 0);
    if (core->x[2] < 0.0)
      invert=TRUE;
    }
  }

/* rot to make z positive */
make_rotmat(&tmat[0], PI, PITCH);

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  if (core->status & DELETED)
    continue;
  if (core->primary)
    {
/* retrieve */
    ARR3SET(vec, core->x);
    if (invert)
      vecmat(tmat, vec);

    fprintf(fp,"%4s %10.5f %10.5f %10.5f  0.0000  0.0000  1.0000   C 0+\n",
            core->label, vec[0], vec[1], vec[2]);
    }
  }
fprintf(fp,"EOF\n");

/* done */
fclose(fp);
return(0);
}

/****************/
/* read routine */
/****************/
#define DEBUG_READ_XYZ 0
gint read_xtl(gchar *filename, struct model_pak *data)
{
gint i, num_tokens;
gchar **buff;
struct core_pak *core;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 1);

fp = fopen(filename, "rt");
if (!fp)
  return(1);

/* defaults */
data->fractional = TRUE;
data->periodic = 3;

/*
while(!fgetline(fp, line))
*/
for (;;)
  {
  buff = get_tokenized_line(fp, &num_tokens);
  if (!buff)
    break;

/* periodicity search */
  if (g_ascii_strncasecmp("dimension", *buff, 9) == 0)
    {
    if (num_tokens > 1)
      data->periodic = (gint) str_to_float(*(buff+1));
    }

/* space group search */
  if (g_ascii_strncasecmp("symmetry", *buff, 8) == 0)
    {
    for (i=1 ; i<num_tokens-1 ; i++)
      {
/* seek space group label */
      if (g_ascii_strncasecmp("label", *(buff+i), 5) == 0)
        {
        g_free(data->sginfo.spacename);
        data->sginfo.spacename = g_strdup(*(buff+i+1));
/* indicate that name should used in lookup */
        data->sginfo.spacenum = -1;
        }
/* seek space group number */
/* TODO - origin qualifier (cellchoice) */
      if (g_ascii_strncasecmp("number", *(buff+i), 6) == 0)
        data->sginfo.spacenum = (gint) str_to_float(*(buff+i+1));
      }
    }

/* pbc search */
  if (g_ascii_strncasecmp("cell", *buff, 4) == 0)
    {
    g_strfreev(buff);
/* get next line (if possible) */
    buff = get_tokenized_line(fp, &num_tokens);
    if (!buff)
      return(2);
    if (num_tokens > 1)
      {
      data->pbc[0] = str_to_float(*(buff+0));
      data->pbc[1] = str_to_float(*(buff+1));
      }
    else
      printf("insufficient tokens.\n");

    switch(data->periodic)
      {
      case 2:
        if (num_tokens > 2)
          data->pbc[5] = D2R * str_to_float(*(buff+2));
        else
          printf("insufficient tokens.\n");
        break;
      case 3:
        if (num_tokens > 5)
          {
          data->pbc[2] = str_to_float(*(buff+2));
          data->pbc[3] = D2R * str_to_float(*(buff+3));
          data->pbc[4] = D2R * str_to_float(*(buff+4));
          data->pbc[5] = D2R * str_to_float(*(buff+5));
          }
        else
          printf("insufficient tokens.\n");
        break;
      }
    }

/* coordinate search */
  if (g_ascii_strncasecmp("name", *buff, 4) == 0)
    {
    for (;;)
      {
      buff = get_tokenized_line(fp, &num_tokens);
      if (!buff)
        break;
      if (g_ascii_strncasecmp("eof", *buff, 3) == 0)
        break;

/* enough tokens */
      if (num_tokens > 3)
        {
/* add new atom if first item is a valid atom type */
        if (elem_type(*buff))
          {
          core = new_core(*buff, data);
          data->cores = g_slist_prepend(data->cores, core);

          core->x[0] = str_to_float(*(buff+1));
          core->x[1] = str_to_float(*(buff+2));
          core->x[2] = str_to_float(*(buff+3));
          }
        }
      }
    }
  g_strfreev(buff);
  }

/* model setup */
strcpy(data->filename, filename);
g_free(data->basename);
data->basename = strdup_basename(filename);

prep_model(data);

return(0);
}


