/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "task.h"
#include "morph.h"
#include "select.h"
#include "matrix.h"
#include "space.h"
#include "spatial.h"
#include "opengl.h"
#include "surface.h"
#include "gtkshorts.h"
#include "interface.h"

#include "logo_left.xpm"
#include "logo_right.xpm"
#include "folder.xpm"
#include "disk.xpm"
#include "arrow.xpm"
/*
#include "split_none.xpm"
#include "split_vert.xpm"
#include "split_horz.xpm"
#include "split_both.xpm"
#include "matrix.xpm"
#include "render_setup.xpm"
*/

#include "tools.xpm"
#include "palette.xpm"
#include "cross.xpm"
#include "element.xpm"
#include "geom.xpm"
#include "axes.xpm"
#include "methane.xpm"
#include "box.xpm"
#include "surface.xpm"
#include "polymer.xpm"
#include "diamond2.xpm"
/*
#include "left_arrow1.xpm"
#include "right_arrow1.xpm"
*/

/* top level data structure */
extern struct sysenv_pak sysenv;

extern GMutex *gdk_threads_mutex;

/* bit messy - put some of these in sysenv? */
/* main window */
GtkWidget *window;
/* backing pixmap */
GdkPixmap *pixmap = NULL;
/* model tree (put in sysenv?) */
GtkWidget *tree;
/* model pane stuff */
GtkWidget *scrolled_window, *clist;
/* gdisrc? */
gint pane_width=65;
/* current viewing mode combo box */
GtkWidget *viewing_mode;

/************************/
/* EVENT - button press */
/************************/
#define DEBUG_BUTTON_PRESS_EVENT 0
gint button_press_event(GtkWidget *w, GdkEventButton *event)
{
gint refresh=0, x, y;
gint shift=FALSE, ctrl=FALSE;
gdouble r[3];
GdkModifierType state;
GSList *list;
struct model_pak *data;
struct bond_pak *bond;
struct core_pak *core;

/* get model */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return(FALSE);

/* event info */
x = event->x;
y = event->y;
sysenv.moving = FALSE;

/* analyse the current state */
state = (GdkModifierType) event->state;
if ((state & GDK_SHIFT_MASK))
  shift = TRUE;
if ((state & GDK_CONTROL_MASK))
  ctrl = TRUE;

/* only want button 1 (for now) */
if (event->button != 1)
  return(FALSE);

/* can we assoc. with a single atom */
/* TODO - maybe return a list? (could be more than one) */
pixel_coord_map(x, y, r, data);
core = seek_coord2d(r, data);

/* allow shift+click to add/remove single atoms in the selection */
/* TODO - depending on mode add/remove objects in selection eg atom/mols */
if (shift)
  {
  switch(data->mode)
    {
    default:
      if (core)
        {
        switch (sysenv.select_mode)
          {
          case ATOM_LABEL:
            select_all_labels(core, data);
            break;
          case CORE:
/* if atom is already in the selection - remove it */
            if (select_add_core(core, data))
              select_del_core(core, data);
            break;
          case ELEM:
            select_all_elem(core->atom_code, data);
            break;
          case MOL:
            select_add_mol(core->mol, data);
            break;
          case REGION:
            select_add_region(core, data);
            break;

          default:
            g_assert_not_reached();
          }
        refresh++;
        }
      else
        {
/* otherwise start a box selection */
        update_box(x,y,data,START);
        }
      break;
    }
  }
else
  {
/* determine the type of action required */
  switch(data->mode)
    {
    case BOND_INFO:
      gl_info_bond(w,x,y,data);
      break;

    case BOND_DELETE:
/* delete via constituent atoms, */
      if (core)
        {
        make_bond(core, data->mode, data); 
        refresh++;
        }
      else
        {
/* or by bond midpoints */
        list = gl_seek_bond(w,x,y,data);
        if (list)
          {
          bond = (struct bond_pak *) list->data;
          user_bond(bond->atom1, bond->atom2, data->mode, data);
          refresh++;
          }
        }
      break;

    case BOND_SINGLE:
    case BOND_DOUBLE:
    case BOND_TRIPLE:
      if (core)
        {
        make_bond(core, data->mode, data); 
        refresh++;
        }
      break;

    case ATOM_ADD:
      add_atom(r, data);
      refresh++; 
      break;
    case ATOM_MOVE:
      move_atom(r, START);
      refresh++;
      break;
    case ATOM_DELETE:
      delete_atom_at(r, data);
      calc_bonds(data);
      calc_mols(data);
      refresh++;
      break;

    case DELETE_VECTOR:
      delete_vector_at(r, data);
      refresh++;
      break;

    case MOL_MOVE:
      move_mol(x,y,START);
      refresh++;
      break;

    case MOL_DELETE:
      if (core)
        {
        delete_mol(core->mol, data);
        refresh++;
        }
      break;

    case DEFINE_RIBBON:
      if (core)
        construct_backbone(core, data);
      break;

    case DEFINE_VECTOR:
    case DEFINE_PLANE:
      if (core)
        new_spatial_point(data->mode, core, data);
      break;

/* selection stuff */
    default:
      select_clear();
/* NEW - don't select if a core is under the mouse */
/* instead we allow the selection box to be drawn */
      update_box(x,y,data,START);
      refresh++;
      break; 
    }
  }
return(FALSE);
}

/***************************/
/* EVENT - button released */
/***************************/
gint button_release_event(GtkWidget *w, GdkEventButton *event)
{
gint x, y;
struct model_pak *data;
GdkModifierType state;

/* get model */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return(FALSE);

/* get event info */
x = event->x;
y = event->y;
state = (GdkModifierType) event->state;

/* NEW - motion flag */
sysenv.moving = FALSE;
refresh_view();

/* first mouse button */
if (event->button == 1) 
  {
/* clean up after move operations */
  switch(data->mode)
    {
    case DIST_INFO:
      info_dist(x,y,data);
      break;
    case ANGLE_INFO:
      info_angle(x,y,data);
      break;
    case DIHEDRAL_INFO:
      info_torsion(x,y,data);
      break;
    case ATOM_MOVE:
      move_atom(NULL, STOP);
      break;
    case MOL_MOVE:
      move_mol(0,0,STOP);
      break;
    default:
/* commit the box contents to the selection */
/* if shift pressed - append selection, otherwise create new */
      if (data->box_on)
        gl_select_box(w);
/* turn the box off */
      data->box_on = FALSE;
      break;
    }
  }

redraw_canvas(SINGLE);
return(FALSE);
}

/************************/
/* EVENT - mouse motion */
/************************/
#define DEBUG_MOTION 0
gint motion_notify_event(GtkWidget *w, GdkEventMotion *event)
{
gint x, y, dx, dy, fx, fy, ds, da, refresh;
gint shift=FALSE, ctrl=FALSE;
gdouble r[3], mat[9], vec[3], scale;
GSList *glist;
GdkModifierType state;
static gint ox=0, oy=0;
struct model_pak *data;

/* get model */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return(FALSE);

/* get mouse data */
if (event->is_hint)
  gdk_window_get_pointer(event->window, &x, &y, &state);
else
  {
/* windows apparently reaches this */
  x = event->x;
  y = event->y;
  state = (GdkModifierType) event->state;
  }

/* discard discontinuous jumps (ie ox,oy are invalid on 1st call) */
if (!sysenv.moving)
  {
  ox = x;
  oy = y;
  sysenv.moving = TRUE;
  return(TRUE);
  }

/* convert relative mouse motion to an increment */
dx = x-ox;
dy = oy-y;        /* inverted y */

/* single update */
refresh=0;

/* analyse the current state */
if ((state & GDK_SHIFT_MASK))
  shift = TRUE;
if ((state & GDK_CONTROL_MASK))
  ctrl = TRUE;

pixel_coord_map(x, y, r, data);

/* first mouse button - mostly selection stuff */
if (state & GDK_BUTTON1_MASK)
  {
  switch(data->mode)
    {
    case ATOM_MOVE:
      move_atom(r, UPDATE);
      refresh++;
      break;
    case MOL_MOVE:
      move_mol(x, y, UPDATE);
      refresh++;
      break;
    case ATOM_ADD:
      break;
    case FREE:
      update_box(x, y, data, UPDATE);
      refresh++;
      break;
    default:
      break;
    }
/* don't switch to low quality drawing */
  sysenv.moving = FALSE;
  if (refresh)
    redraw_canvas(SINGLE);
  ox=x;
  oy=y;
  return(TRUE);
  }

/* second mouse button */
if (state & GDK_BUTTON2_MASK)
  {
/* shift - zoom */
  if (shift)
    {
/* zoom */
    data->scale += PIX2SCALE * (gdouble) (dx+dy);
/* lower limit check (upper? or leave to user discretion?) */
    if (data->scale < 0.1)
      data->scale = 0.1;
/* update */
    init_objs(REDO_COORDS, data);
    refresh++;
    sysenv.moving = TRUE;
    }
/* no shift - translate */
  else
    {
/* selection only translation */
    if (ctrl)
      {
/* FIXME - how does the 500 fit in ??? */
      scale = data->rmax/500.0;
/* convert pixel translation to real space translation */
      VEC3SET(vec, scale*dx, -scale*dy, 0.0);

      vecmat(data->irotmat, vec);
      vecmat(data->ilatmat, vec);
/* apply to all */
      xlat_select(vec, data);

/* recalc coords */
      calc_coords(REFRESH, data);
      refresh++;
      sysenv.moving = TRUE;
      }
    else
      {
/* translate everything */
      data->offset[0] += x-ox;
      data->offset[1] += y-oy;
      glist = data->ghosts;
      while (glist != NULL)
        {
        data = model_ptr(GPOINTER_TO_INT(glist->data), RECALL);
        data->offset[0] += x-ox;
        data->offset[1] += y-oy;
        glist = g_slist_next(glist);
        }
      refresh++;
      sysenv.moving = TRUE;
      }
    }
  }

/* third mouse button clicked? */
if (state & GDK_BUTTON3_MASK)
  {
/* shift clicked? */
  if (shift)
    {
/* yaw */
    if (dx || dy)
      {
/* rotation amount */
      da = abs(dx) + abs(dy);
/* vector from center to mouse pointer (different for OpenGL window) */
/* FIXME - if ctrl is pressed, the center should be the selection */
/* centroid & not the middle of the window */
      fx = x - data->offset[0] - (sysenv.x + sysenv.width/2);
      fy = data->offset[1] + (sysenv.y + sysenv.height/2 - y);

/* rotation direction via z component of cross product (+=clock, -=anti) */
      if ((fx*dy - dx*fy) > 0)
        ds = -1;
      else
        ds = 1;

#if DEBUG_MOTION
printf("(%d,%d) x (%d,%d) : %d\n",dx,dy,fx,fy, ds);
#endif

/* assign and calculate */
      data->da = (gdouble) (ds*da);
      data->da *= ROTATE_SCALE * PI / 180.0;
/* NEW - selection */
      if (ctrl)
        {
        make_rotmat(mat, data->da, ROLL);
        rotate_select(data, mat);
        data->da = 0.0;
        }
/* update coords */
      calc_coords(ROLL, data);
      data->da = 0.0;
      }
    refresh++;
    sysenv.moving = TRUE;
    }
  else
    {
#if DEBUG_MOTION
printf("(%d,%d)\n",dx,dy);
#endif
/* pitch and roll */
    dy *= -1;
    if (dy)
      {
      data->da = (gdouble) ROTATE_SCALE * dy * PI / 180.0;
/* NEW - selection */
      if (ctrl)
        {
        make_rotmat(mat, data->da, PITCH);
        rotate_select(data, mat);
        data->da = 0.0;
        }
/* update coords */
      calc_coords(PITCH, data);
      data->da = 0.0;
      refresh++;
      sysenv.moving = TRUE;
      }
    if (dx)
      {
      data->da = (gdouble) ROTATE_SCALE * dx * PI / 180.0;
/* NEW - selection */
      if (ctrl)
        {
        make_rotmat(mat, data->da, YAW);
        rotate_select(data, mat);
        data->da = 0.0;
        }
/* update coords */
      calc_coords(YAW, data);
      data->da = 0.0;
      refresh++;
      sysenv.moving = TRUE;
      }
    }
  }

/* save old values */
ox=x;
oy=y;

/* redraw? */
if (refresh)
  redraw_canvas(SINGLE);

return(TRUE);
}

/***************************/
/* expose all hidden atoms */
/***************************/
void unhide_atoms(void)
{
GSList *list;
struct model_pak *data;
struct core_pak *core;

/* exit if nothing is loaded */
if (!sysenv.num_models)
  return;
/* deletion for the active model only */
data = model_ptr(sysenv.active, RECALL);

/* unhide */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  core->status &= ~HIDDEN;
  }

/* update */
redraw_canvas(SINGLE);
}

/****************/
/* change mode */
/***************/
void switch_mode(gint new_mode)
{
GSList *list;
struct model_pak *data;
struct core_pak *core1;

if (!sysenv.num_models)
  return;

/* get selected model */
data = model_ptr(sysenv.active, RECALL);

/* NEW - is the model change allowed? */
switch (data->id)
  {
  case MORPH:
    if (new_mode != FREE)
      {
      show_text(WARNING, "Disallowed mode.\n");
      return;
      }
  }

/* clean up (if necessary) from previous mode */
switch (data->mode)
  {
  case ATOM_MOVE:
    move_atom(NULL, STOP);
    break;

  case MOL_MOVE:
    move_mol(0,0,STOP);
    break;

  case DIST_INFO:
  case BOND_INFO:
    for (list=data->cores ; list ; list=g_slist_next(list))
      {
      core1 = (struct core_pak *) list->data;
      core1->status &= ~SELECT;
      }
    break;
  }

/* set the new mode */
data->mode = new_mode;
data->state = 0;
redraw_canvas(SINGLE);
}

/* Need this for the menu item factory widget */
void gtk_switch_mode(GtkWidget *w, gint new_mode)
{
switch_mode(new_mode);
}

/***************/
/* change view */
/***************/
void switch_view(gint new_mode)
{
gint i,j;
gint inc=0, count;

/* all modes */
switch (new_mode)
  {
  case PREV_MODEL:
    inc=-2;
  case NEXT_MODEL:
    inc++;
/* don't do if equal or more display slots than models */
    if ((sysenv.num_displayed - sysenv.num_models) >= 0)
      break;
/* find active model number and apply change */
    for (i=0 ; i<sysenv.num_displayed ; i++)
      if (sysenv.displayed[i] == sysenv.active)
        {
        sysenv.displayed[i] += inc;
        break;
        }
/* wrap around */
    if (sysenv.displayed[i] < 0)
      sysenv.displayed[i] += sysenv.num_models;
    if (sysenv.displayed[i] >= sysenv.num_models)
      sysenv.displayed[i] -= sysenv.num_models;
    sysenv.active = sysenv.displayed[i];
/* avoid repeat displays of the active model */
    do
      {
      count=0;
      for (j=0 ; j<sysenv.num_displayed ; j++)
        if (sysenv.displayed[j] == sysenv.active)
          count++;
/* found a duplicate - try the next */
      if (count == 2)
        sysenv.displayed[i] += inc;
/* wrap around (need to redo count if wrap occurs) */
      if (sysenv.displayed[i] < 0)
        {
        sysenv.displayed[i] += sysenv.num_models;
        count=2;
        }
      if (sysenv.displayed[i] >= sysenv.num_models)
        {
        sysenv.displayed[i] -= sysenv.num_models;
        count=2;
        }
      sysenv.active = sysenv.displayed[i];
      }
    while (count == 2);
/* highlight the new active model */
/*
    tree_select(sysenv.active);
*/
    break;
  }

/* update the screen */
redraw_canvas(ALL);
}

/************/
/* GTK hook */
/************/
void gtk_switch_view(GtkWidget *w, gint new_mode)
{
switch_view(new_mode);
}

/**********************************/
/* update the model/canvas layout */
/**********************************/
#define DEBUG_CANVAS_SHUFFLE 0
void canvas_shuffle(void)
{
GSList *list, *mlist;
struct model_pak *model;
struct canvas_pak *canvas;

model = model_ptr(sysenv.active, RECALL);

/* this becomes non NULL when we find the first */
/* available spot for the active model */
mlist = NULL;

for (list=sysenv.canvas_list ; list ; list=g_slist_next(list))
  {
  canvas = (struct canvas_pak *) list->data;
  if (model)
    {
    if (!canvas->model)
      {
/* first time exception - the currently active model */
      if (!mlist)
        {
        canvas->active = TRUE;
        mlist = g_slist_find(sysenv.mal, model);
        }
      }
/* at this stage we've found a candidate spot for the active model */
    if (mlist)
      {
      model = (struct model_pak *) mlist->data;
      canvas->model = model;

#if DEBUG_CANVAS_SHUFFLE 
printf("assigning model: %p to canvas: %p\n", model, canvas);
#endif

/* if no more models in the current sequence - fill out the */
/* rest of the canvas model slots with NULL */
      mlist = g_slist_next(mlist);
      if (!mlist)
        model = NULL;
      }
    }
  else
    canvas->model = NULL;
  }

/* if no NULL spots to put the active model - force it somewhere */
if (model)
  {
/* FIXME - single canvas hack */
  list = sysenv.canvas_list;
  canvas = (struct canvas_pak *) list->data;
  
  canvas->active = TRUE;
  canvas->model = model;

#if DEBUG_CANVAS_SHUFFLE 
printf("assigning model: %p to canvas: %p\n", model, canvas);
#endif
  }
}

/************************************/
/* pick a (loaded) model to display */
/************************************/
#define DEBUG_PICK_MODEL 0
void pick_model(gint model)
{
gint i;
struct model_pak *data;

/* checks */
data = model_ptr(model, RECALL);
if (!data)
  return;

/* make the model active */
sysenv.active = model;

/* updates */
canvas_shuffle();
refresh_view();
update_creator();

if (g_slist_length(data->selection) == 1)
  atom_properties_update((data->selection)->data, data);

/* NEW - update using widget <-> variable relation list */
gtksh_relation_update(data);

/* deprec. */
/* cycle through list of dialogs & update accordingly */
for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (!sysenv.dialog[i].active)
    continue;
/* dialog dependent refresh */
  switch(sysenv.dialog[i].type)
    {
    case SPATIAL:
printf("TODO - spatial dialog updates.\n");
      break;
    case OPENGL_OPTIONS:
/* TODO - update dialog with new model's opengl options */
printf("TODO - update opengl_options dialog\n");
      break;
    case POVRAY:
      update_render_dialog();
      break;
    default:;
/* dialogs that don't need refreshing */
    }
  }
redraw_canvas(ALL);
}

/****************************/
/* event hook for the above */
/****************************/
void event_pick_model(GtkWidget *w, struct model_pak *data)
{
g_return_if_fail(data != NULL);
g_return_if_fail(data->id != NODATA);
pick_model(data->number);
}

/*****************************************/
/* force the selection of main tree item */
/*****************************************/
/*
void tree_select(gint model)
{
if (GTK_IS_TREE(tree))
  gtk_tree_select_item(GTK_TREE(tree), model);
}
*/

/*****************************/
/* view panel button handler */
/*****************************/
void reset_view(void)
{
struct model_pak *data;

/* do we have a loaded model? */
data = model_ptr(sysenv.active, RECALL);
if (data == NULL)
  return;

/* make the changes */
VEC3SET(data->angle, 0.0, 0.0, 0.0);
data->da = 0.0;
data->scale = 1.0;
if (sysenv.rab_on)
  {
  VEC3SET(sysenv.angle, 0.0, 0.0, 0.0);
  }

/* update */
calc_coords(INITIAL, data);
cent_coords(1, data);
calc_coords(REFRESH, data);
 /*
init_objs(INIT_COORDS, data);
*/
/* this'll trigger any dialog updates */
pick_model(data->number);
}

/*******************************/
/* refresh spin buttons values */
/*******************************/
void refresh_view()
{
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
if (!model)
  return;

ARR3SET(sysenv.angle, model->angle);
gtksh_relation_update(NULL);
}

/******************************/
/* The omnipresent view panel */
/******************************/
void change_view(GtkWidget *w, gpointer dummy)
{
gint i;
gdouble da[3];
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
if (!model)
  {
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), 0.0);
  return;
  }

/* get angle difference */
ARR3SET(da, sysenv.angle);
ARR3SUB(da, model->angle);

/* apply the appropriate rotation matrix */
for (i=3 ; i-- ; )
  {
  if (da[i])
    {
    model->da = D2R*da[i];
    switch (i)
      {
      case 0:
        calc_coords(YAW, model);
        break;
      case 1:
        calc_coords(PITCH, model);
        break;
      case 2:
        calc_coords(ROLL, model);
        break;
      }
    model->da = 0.0;
    model->angle[i] = sysenv.angle[i];
    }
  }
redraw_canvas(SINGLE);
}

/***************************/
/* create/update text tray */
/***************************/
void show_text(gint type, gchar *message)
{
GtkWidget *view;
static GtkTextBuffer *buffer=NULL;
static GtkTextIter iter;

/* checks */
if (!message)
  return;

if (sysenv.canvas)
  {
  if (!buffer)
    {
/* first time init */
    view = gtk_text_view_new();
    gtk_text_view_set_editable(GTK_TEXT_VIEW(view), FALSE);
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view));
    gtk_container_add(GTK_CONTAINER(sysenv.tpane), view);
    gtk_widget_show(view);
/* tags */
    gtk_text_buffer_create_tag(buffer, "fg_blue", "foreground", "blue", NULL);  
    gtk_text_buffer_create_tag(buffer, "fg_red", "foreground", "red", NULL);
/* position iterator */
    gtk_text_buffer_get_iter_at_line(buffer, &iter, 0);
    }

/* assign colour via message type */
  switch(type)
    {
    case ERROR:
      gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, g_strdup(message), -1, "fg_red", NULL); 
      break;
    case WARNING:
      gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, g_strdup(message), -1, "fg_blue", NULL); 
      break;
    default:
      gtk_text_buffer_insert(buffer, &iter, g_strdup(message), -1);
    }
/* TODO - delete items from buffer when line gets too big */
  }
else
  printf("[%s]\n", message);
}

/*********************************/
/* atom proerties display toggle */
/*********************************/
void apb_toggle(void)
{
sysenv.apb_on ^= 1;
sysenv.write_gdisrc = TRUE;

if (sysenv.apb_on)
  gtk_widget_show(sysenv.apb_frame);
else
  gtk_widget_hide(sysenv.apb_frame);
}

/***********************************/
/* rotation spinner display toggle */
/***********************************/
void rab_toggle(void)
{
sysenv.rab_on ^= 1;
sysenv.write_gdisrc = TRUE;

if (sysenv.rab_on)
  gtk_widget_show(sysenv.rab_frame);
else
  gtk_widget_hide(sysenv.rab_frame);
}

/********************************/
/* rotation adjustment spinners */
/********************************/
void rot_adjust_box(GtkWidget *box)
{
GtkWidget *frame, *hbox, *vbox, *label;

/* rotation frame */
frame = gtk_frame_new(NULL);
gtk_container_add(GTK_CONTAINER(box), frame);
hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);
label = gtk_label_new ("Yaw");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
gtksh_direct_spin(NULL, &sysenv.angle[0],
                  0.0, 360.0, 5.0,
                  change_view, GINT_TO_POINTER(0), vbox);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);
label = gtk_label_new ("Pitch");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
gtksh_direct_spin(NULL, &sysenv.angle[1],
                  0.0, 360.0, 5.0,
                  change_view, GINT_TO_POINTER(1), vbox);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);
label = gtk_label_new ("Roll");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
gtksh_direct_spin(NULL, &sysenv.angle[2],
                  0.0, 360.0, 5.0,
                  change_view, GINT_TO_POINTER(2), vbox);
}

/**********************/
/* model tree globals */
/**********************/
enum
{
TREE_PIXMAP,
TREE_NAME,
TREE_FUNCTION,
TREE_POINTER,
TREE_NCOLS
};

/*****************************/
/* delete the selected model */
/*****************************/
void del_tree_selected(void)
{
GtkTreeIter iter, parent;
GtkTreeSelection *selection;
GtkTreeModel *model;
struct model_pak *mdata;

selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(sysenv.tree)); 
if (selection)
  {
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
/* the selection is the parent -> just do a pick model */
    gtk_tree_model_get(model, &iter, TREE_POINTER, &mdata, -1);
    g_assert(mdata != NULL);
/* NEW */
    if (mdata->locked)
      {
      show_text(ERROR, "Model is locked.\n");
      return;
      }
/* remove the main model tree item */
    if (gtk_tree_model_iter_parent(model, &parent, &iter))
      gtk_tree_store_remove(sysenv.tree_store, &parent);
    else
      gtk_tree_store_remove(sysenv.tree_store, &iter);

    delete_model(mdata->number);
    }
  }
}

/**************************/
/* tree traverse function */
/**************************/
gint func(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
GtkTreeSelection *selection;
struct model_pak *mdata;

/* get the model pointer from the tree store */
gtk_tree_model_get(model, iter, TREE_POINTER, &mdata, -1);
/* if it matches the model supplied, select it */
if (data == mdata)
  {
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(sysenv.tree)); 
  gtk_tree_selection_select_iter(selection, iter);
/* return TRUE to indicate we're done  */
  return(TRUE);
  }
/* no match, keep traversing */
return(FALSE);
}

/******************************/
/* select the specified model */
/******************************/
void tree_select(gint n)
{
struct model_pak *model;

model = model_ptr(n, RECALL);
if (!model)
  return;

gtk_tree_model_foreach(GTK_TREE_MODEL(sysenv.tree_store), &func, model);
}

/***************************/
/* select the active model */
/***************************/
void tree_select_active(void)
{
tree_select(sysenv.active);
}

/*******************************/
/* add a new model to the tree */
/*******************************/
void new_tree_item(gint model, gint mode)
{
gint i, n, expand=FALSE;
GtkTreeIter root, branch;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
GtkTreePath *treepath=NULL;
GdkPixbuf *pixbuf;
struct model_pak *data;

data = model_ptr(model, RECALL);
g_assert(data != NULL);

switch (mode)
  {
  case APPEND:
    if (data->grafted)
      return;
    data->grafted = TRUE;
/* acquire a new top-level iterator */
    gtk_tree_store_append(sysenv.tree_store, &root, NULL);
    break;

  case REPLACE:
    if (!data->grafted)
      return;
/* get current selection */
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(sysenv.tree)); 
/* NB: the selection should always be the parent iterator */
    if (selection)
      gtk_tree_selection_get_selected(selection, &treemodel, &root);
    else
      return;

/* test the expand state */
    treepath = gtk_tree_model_get_path(treemodel, &root);
    expand = gtk_tree_view_row_expanded(GTK_TREE_VIEW(sysenv.tree), treepath);

/* delete all children (NB: in REVERSE - to preserve indexing) */
    n = gtk_tree_model_iter_n_children(treemodel, &root);
    for (i=n ; i-- ; )
      {
      if (gtk_tree_model_iter_nth_child(treemodel, &branch, &root, i))
        gtk_tree_store_remove(sysenv.tree_store, &branch);
      else
        {
        printf("Bad child number: %d\n", i);
        break;
        }
      }
    break;
  }

/* setup pixmap */
if (data->id == MORPH)
  pixbuf = gdk_pixbuf_new_from_xpm_data((const gchar **) diamond2_xpm);
else
  {
  switch(data->periodic)
    {
    case 3:
      pixbuf = gdk_pixbuf_new_from_xpm_data((const gchar **) box_xpm);
      break;
    case 2:
      pixbuf = gdk_pixbuf_new_from_xpm_data((const gchar **) surface_xpm);
      break;
    case 1:
      pixbuf = gdk_pixbuf_new_from_xpm_data((const gchar **) polymer_xpm);
      break;
    default:
      pixbuf = gdk_pixbuf_new_from_xpm_data((const gchar **) methane_xpm);
      break;
    }
  }

/* set the parent iterator data */
gtk_tree_store_set(sysenv.tree_store, &root,
                   TREE_PIXMAP, pixbuf, 
                   TREE_NAME, g_strdup(data->basename),
                   TREE_POINTER, data,
                   -1);

/* add the child iterators (options beneath the model name) */
if (data->periodic > 0)
  {
/* Acquire a child iterator */
  gtk_tree_store_append(sysenv.tree_store, &branch, &root);
  gtk_tree_store_set(sysenv.tree_store, &branch,
                     TREE_NAME, "periodicity",
                     TREE_FUNCTION, SGINFO,
                     TREE_POINTER, data,
                     -1);
  }
else
  {
/* symmetry analyzer */
  if (data->id != MORPH)
    {
    gtk_tree_store_append(sysenv.tree_store, &branch, &root);
    gtk_tree_store_set(sysenv.tree_store, &branch,
                       TREE_NAME, "symmetry",
                       TREE_FUNCTION, SYMMETRY,
                       TREE_POINTER, data,
                       -1);
    }
  }

if (data->periodic == 3 || data->id == MORPH)
  {
  gtk_tree_store_append(sysenv.tree_store, &branch, &root);
  gtk_tree_store_set(sysenv.tree_store, &branch,
                     TREE_NAME, "surfaces",
                     TREE_FUNCTION, GENSURF,
                     TREE_POINTER, data,
                     -1);
  }

if (data->id != MORPH)
  {
  gtk_tree_store_append(sysenv.tree_store, &branch, &root);
  gtk_tree_store_set(sysenv.tree_store, &branch,
                     TREE_NAME, "energetics",
                     TREE_FUNCTION, ENERGETICS,
                     TREE_POINTER, data,
                     -1);
  }

if (data->animation)
  {
  gtk_tree_store_append(sysenv.tree_store, &branch, &root);
  gtk_tree_store_set(sysenv.tree_store, &branch,
                     TREE_NAME, "analysis",
                     TREE_FUNCTION, MD_ANALYSIS,
                     TREE_POINTER, data,
                     -1);

  gtk_tree_store_append(sysenv.tree_store, &branch, &root);
  gtk_tree_store_set(sysenv.tree_store, &branch,
                     TREE_NAME, "animation",
                     TREE_FUNCTION, ANIM,
                     TREE_POINTER, data,
                     -1);
  }

/* expand if required */
if (expand)
  gtk_tree_view_expand_row(GTK_TREE_VIEW(sysenv.tree), treepath, FALSE);

/* make the newly created model the current selection */
selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(sysenv.tree)); 
if (selection)
  gtk_tree_selection_select_iter(selection, &root);
}

/********************************/
/* handle tree selection events */
/********************************/
static void tree_selection_changed(GtkTreeSelection *selection, gpointer data)
{
gint function;
GtkTreeIter iter, parent;
GtkTreeModel *model;
struct model_pak *mdata=NULL;

if (gtk_tree_selection_get_selected(selection, &model, &iter))
  {
  gtk_tree_model_get(model, &iter, TREE_POINTER, &mdata, -1);

/* test if selection has a parent */
  if (gtk_tree_model_iter_parent(model, &parent, &iter))
    {
/* get row data */
    gtk_tree_model_get(model, &iter, TREE_FUNCTION, &function, -1);
    gtk_tree_model_get(model, &iter, TREE_POINTER, &mdata, -1);
    g_assert(mdata != NULL);

/* NB: we MUST make the model active before performing any acions */
/* as some dialogs rely on knowing the currently active model */
    pick_model(mdata->number);

/* what action? */
    switch(function)
      {
      case SGINFO:
        space_info(mdata);
        break;

      case SYMMETRY:
        sym_info(NULL, mdata);
        break;

      case GENSURF:
      case MORPH:
        surface_dialog(NULL, mdata);
        break;
    
      case ENERGETICS:
        energetics_dialog(NULL, mdata);
        break;

      case ANIM:
        animate_setup_widget();
        break;

      case MD_ANALYSIS:
        md_dialog();
        break;

      default:
        printf("Not implemented yet.\n");
      }
    }
  else
    {
/* the selection is the parent -> just do a pick model */
    gtk_tree_model_get(model, &iter, TREE_POINTER, &mdata, -1);
    g_assert(mdata != NULL);
    }
/* FIXME - bug here first time around - could be GTK's fault */
/*
printf("pick model: %d, active: %d\n", mdata->number, sysenv.active);
*/
/* enforce selection of the model */
  pick_model(mdata->number);
/* enforce selection of the model name */
  tree_select_active();
  }
}

/*********************************/
/* create the model viewing pane */
/*********************************/
void init_model_tree(GtkWidget *box)
{
GtkWidget *swin;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;
GtkTreeSelection *select;

/* scrolled window for the model pane */
swin = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC,
                               GTK_POLICY_AUTOMATIC);
gtk_container_add(GTK_CONTAINER(box), swin);

/* underlying data storage */
sysenv.tree_store = gtk_tree_store_new(TREE_NCOLS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_INT, G_TYPE_POINTER);
/* actual tree widget */
sysenv.tree = gtk_tree_view_new_with_model(GTK_TREE_MODEL(sysenv.tree_store));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), sysenv.tree);


/* setup the model pixmap rendering colum */
renderer = gtk_cell_renderer_pixbuf_new();
column = gtk_tree_view_column_new_with_attributes("image", renderer, "pixbuf", TREE_PIXMAP, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(sysenv.tree), column);

/* setup the model name rendering colum */
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes("name", renderer, "text", TREE_NAME, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(sysenv.tree), column);

/* setup the selection handler */
select = gtk_tree_view_get_selection(GTK_TREE_VIEW(sysenv.tree));
gtk_tree_selection_set_mode(select, GTK_SELECTION_BROWSE);
g_signal_connect(G_OBJECT(select), "changed",
                 G_CALLBACK(tree_selection_changed),
                 NULL);

gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(sysenv.tree), FALSE);

gtk_widget_show(swin);
}

/*******************************/
/* write gdisrc before exiting */
/*******************************/
void gdis_exit(void)
{
/* popup if any background jobs are running */
if (sysenv.running_tasks)
  {
  printf("There are running background jobs.\n\n");
  printf("TODO : [preserve] [kill] [don't exit]\n");
  }

/* if user has changed element colours, but doesn't want them saved */
/* this will do it anyway - but they can just click reset next time */
if (sysenv.write_gdisrc)
  {
  sysenv.x = sysenv.display_box->allocation.x;
  sysenv.y = sysenv.display_box->allocation.y;
  sysenv.width = sysenv.display_box->allocation.width;
  sysenv.height = sysenv.display_box->allocation.height;

  write_gdisrc();
  }

/* cleanup */
sys_free();
/*
unlink("gdis.ps");
*/

if (sysenv.canvas)
  {
  gtk_widget_pop_visual();
  gtk_widget_pop_colormap();
  gtk_exit(0);
  }

exit(0);
}

/***************************/
/* font setup & processing */
/***************************/
#define DEBUG_FONT_INIT 0
void font_init(gpointer *ptr)
{
GtkStyle *style;

/* get the main window's style */
style = gtk_widget_get_style(window);

/* pango hack */
{
PangoFontDescription *pfd;

pfd = pango_font_description_from_string(sysenv.gfont_name);
sysenv.gfont = gdk_font_from_description(pfd);
sysenv.gfontsize = pango_font_description_get_size(pfd);

pfd = pango_font_description_from_string(sysenv.dfont_name);
sysenv.dfont = gdk_font_from_description(pfd);
}

#if DEBUG_FONT_INIT
printf("gfont = %p\n", sysenv.gfont);
printf("dfont = %p\n", sysenv.dfont);
#endif

/* use specified font if valid, otherwise default */
if (sysenv.gfont)
  {
  gtk_style_set_font(style, sysenv.gfont);
  gtk_widget_set_style(window, style);

/* update main window & one other (if supplied) */
  gtk_widget_queue_draw(window);
  if (ptr)
    gtk_widget_queue_draw(GTK_WIDGET(ptr));
  }
else
  sysenv.gfont = gtk_style_get_font(style);

/* if drawing font not found - use default */
if (!sysenv.dfont)
  sysenv.dfont = gtk_style_get_font(style);

/* get font size (for auto text placement calcs) */
/* average/largest character size??? */
sysenv.dfontsize = gdk_char_width(sysenv.dfont, 'm');
sysenv.gfontsize = gdk_char_width(sysenv.gfont, 'm');

#if DEBUG_FONT_INIT
printf("gfont = %s\n", sysenv.gfont_name);
printf("dfont = %s\n", sysenv.dfont_name);
printf("gfont = %p (%d)\n", sysenv.gfont, sysenv.gfontsize);
printf("dfont = %p (%d)\n", sysenv.dfont, sysenv.dfontsize);
#endif
}

/**************************/
/* multi canvas splitting */
/**************************/
void canvas_split(GtkWidget *w, gpointer *type)
{
struct canvas_pak *canvas;

/*
printf("split: %d\n", GPOINTER_TO_INT(type));
*/

switch(GPOINTER_TO_INT(type))
  {
  case CANVAS_INIT:
    canvas = gl_new_canvas(sysenv.width, sysenv.height);
    gtk_box_pack_start(GTK_BOX(sysenv.display_box), canvas->glarea, TRUE, TRUE, 0);
    gtk_widget_show(canvas->glarea);
  case CANVAS_SINGLE:
    sysenv.canvas_rows=sysenv.canvas_cols=1;
    break;

  case CANVAS_VSPLIT:
/*
    sysenv.canvas_cols++;
*/
    break;
  case CANVAS_HSPLIT:
/*
    sysenv.canvas_rows++;
*/
    break;
  case CANVAS_HVSPLIT:
/*
    sysenv.canvas_rows++;
    sysenv.canvas_cols++;
*/
    break;
  }
}

/******************/
/* MENU structure */
/******************/
static GtkItemFactoryEntry menu_items[] = 
{
  { "/_File",                 NULL, NULL, 0, "<Branch>" },
  { "/File/_New",             NULL, create_new_model, 1, NULL },
  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/File/_Open...",         NULL, file_load_dialog, 1, NULL },
  { "/File/_Close",           NULL, del_tree_selected, 1, NULL },
  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/File/_Save",            NULL, file_save, 1, NULL },
  { "/File/Save _as...",      NULL, file_save_dialog, 1, NULL },
  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/File/_Quit",            NULL, gdis_exit, 1, NULL },

  { "/_Edit",               NULL, NULL, 0, "<Branch>" },
  { "/Edit/_Copy",          NULL, select_copy, 0, NULL },
  { "/Edit/_Paste",         NULL, select_paste, 0, NULL },
  { "/Edit/sep1",           NULL, NULL, 0, "<Separator>" },
  { "/Edit/Colour...",      NULL, select_colour, 0, NULL },
  { "/Edit/Delete",         NULL, select_delete, 0, NULL },
  { "/Edit/Hide",           NULL, select_hide, 0, NULL },
  { "/Edit/Unhide all",     NULL, unhide_atoms, 0, NULL },
  { "/Edit/sep1",           NULL, NULL, 0, "<Separator>" },
  { "/Edit/Select all",     NULL, select_all, 0, NULL },
  { "/Edit/Invert",         NULL, select_invert, 0, NULL },

  { "/_Tools",                      NULL, NULL, 0, "<Branch>" },
  { "/Tools/Model editing...",      NULL, med_interface, 0, NULL },
  { "/Tools/Display properties...", NULL, render_dialog, 0, NULL },
  { "/Tools/sep1",                  NULL, NULL, 0, "<Separator>" },
  { "/Tools/Measurements...",       NULL, geom_info, 0, NULL },
  { "/Tools/Periodic table...",     NULL, gperiodic_win, 0, NULL },
  { "/Tools/sep1",                  NULL, NULL, 0, "<Separator>" },
  { "/Tools/Diffraction...",        NULL, diffract_dialog, 0, NULL },
  { "/Tools/MD initializer...",     NULL, mdi_setup_widget, 0, NULL },
  { "/Tools/Molecular _surfaces...", NULL, surf_dialog, 0, NULL },

/*
  { "/Info",                  NULL, NULL, 0, "<LastBranch>" },
*/
  { "/_Info",                  NULL, NULL, 0, "<Branch>" },
  { "/Info/About...",         NULL, about, 0, NULL },
  { "/Info/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/Info/_Task manager...",  NULL, task_dialog, 0, NULL },
};

/********************************/
/* process key presses directly */
/********************************/
gint cb_key_press(GtkWidget *w, GdkEventKey *event, gpointer dummy)
{
switch(event->keyval)
  {
  case GDK_Escape:
    break;

  case GDK_Delete:
    select_delete();
    break;

  case GDK_F1:
    sysenv.stereo ^= 1;
    if (sysenv.stereo)
      stereo_open_window();
    else
      stereo_close_window();
    break;
  }
return(FALSE);
}

/*********/
/* SETUP */
/*********/
void connect_events()
{
gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
gchar *text;
GList *list;
GtkWidget *gdis_wid;
GtkWidget *hpaned, *vpaned, *vbox, *hbox, *menu_bar, *toolbar, *combo;
GtkWidget *frame, *event_box, *label;
GdkBitmap *mask;
GdkPixmap *gdis_pix=NULL;
GtkStyle *style, *logo_style;
GtkItemFactory *item;

/* enforce true colour (fixes SG problems) */
sysenv.visual = gdk_visual_get_best_with_type(GDK_VISUAL_TRUE_COLOR);
if (!sysenv.visual)
  {
  printf("Error: could not get requested visual.\n");
  exit(1);
  }
sysenv.depth = sysenv.visual->depth;
sysenv.colourmap = gdk_colormap_new(sysenv.visual, TRUE);
if (!sysenv.colourmap)
  {
  printf("Error: could not allocate colourmap.\n");
  exit(1);
  }
gtk_widget_push_colormap(sysenv.colourmap);
gtk_widget_push_visual(sysenv.visual);


/* main window */
window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
gtk_window_set_title(GTK_WINDOW(window),"GTK Display Interface for Structures");
gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);

g_signal_connect(GTK_OBJECT(window), "key_press_event",
                (GtkSignalFunc) cb_key_press, NULL);


/* vbox for the menubar */
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(window), vbox);

/* item factory menu creation */
item = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", NULL);
gtk_item_factory_create_items(item, nmenu_items, menu_items, NULL);
menu_bar = gtk_item_factory_get_widget(item, "<main>");

/* FALSE,FALSE => don't expand to fill (eg on resize) */
gtk_box_pack_start(GTK_BOX(vbox), menu_bar, FALSE, FALSE, 0);

/* hbox for the toolbar */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
toolbar = gtk_toolbar_new();
gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, FALSE, 0);

/* extract some important info for pixmap creation */
gtk_widget_realize(window);
style = gtk_widget_get_style(window);

/* load button */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, folder_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), NULL,
                        "Load new model", "Private",
                        gdis_wid, GTK_SIGNAL_FUNC(file_load_dialog), NULL);

/* save button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, disk_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), NULL,
                        "Save active model", "Private", gdis_wid,
                        GTK_SIGNAL_FUNC(file_save_dialog), NULL);

/* delete button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, cross_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,                   /* label */
                        "Close active model",   /* tooltip */
                        "Private",              /* private info */
                        gdis_wid,               /* icon widget */
                        GTK_SIGNAL_FUNC(del_tree_selected), /* signal */
                        NULL);                  /* signal data */

/* display properties */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, palette_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Display properties",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(render_dialog),
                        NULL);

/* model editing */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, tools_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Model editing",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(med_interface),
                        NULL);

/* geometry button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, geom_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Measurements",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(geom_info),
                        NULL);

/* gperiodic button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, element_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Periodic table",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gperiodic_win),
                        NULL);

/* reset orientation button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, axes_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Reset orientation",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(reset_view),
                        NULL);

/* normal viewing button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, arrow_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Normal viewing",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_mode),
                        GINT_TO_POINTER(FREE));


#if THIS_IS_FIXED
/* make prev model active */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, left_arrow1_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "display previous model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) PREV_MODEL);

gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* make next model active */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, right_arrow1_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "display next model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) NEXT_MODEL);
#endif




#if CANVAS_SPLIT
/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_none_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Single canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_SINGLE));

/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_vert_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_VSPLIT));

/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_horz_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_HSPLIT));

/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_both_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_HVSPLIT));
#endif


/* TODO - can we get rid of the cursor??? */
/* combo box for the selection mode */
list = NULL;
list = g_list_append(list, "atoms");
list = g_list_append(list, "atom labels");
list = g_list_append(list, "elements");
list = g_list_append(list, "elements in molecule");
list = g_list_append(list, "molecules");
list = g_list_append(list, "regions");

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_end(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(set_selection_mode), (gpointer *) combo);

label = gtk_label_new("Selection mode: ");
gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, PANEL_SPACING);


/* MAIN LEFT/RIGHT HBOX PANE */
/* paned window */
hpaned = gtk_hpaned_new();
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)), hpaned);

/* LEFT PANE */
sysenv.mpane = gtk_vbox_new(FALSE, 0);
gtk_paned_pack1(GTK_PANED(hpaned), sysenv.mpane, FALSE, TRUE);
gtk_widget_set_size_request(sysenv.mpane, sysenv.tree_width, -1);


/* MODEL TREE */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(sysenv.mpane),vbox,TRUE,TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

init_model_tree(vbox);

/* TODO - the toggle button changes, depending on whether the */
/* box is visible or not */
/* current atom properties */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(sysenv.mpane), vbox, FALSE, FALSE, 0);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
new_check_button(" atom properties", apb_toggle, NULL, sysenv.apb_on, hbox);
sysenv.apb_frame = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(sysenv.mpane), sysenv.apb_frame, FALSE, FALSE, 0);

atom_properties_box(sysenv.apb_frame);

/* current orientation values */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(sysenv.mpane),vbox,FALSE,FALSE,0);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, FALSE, 0);
new_check_button(" model orientation", rab_toggle, NULL, sysenv.rab_on, hbox);
sysenv.rab_frame = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(sysenv.mpane),sysenv.rab_frame,FALSE,FALSE,0);

rot_adjust_box(sysenv.rab_frame);


/* GDIS logo */
frame = gtk_frame_new(NULL);
gtk_box_pack_end(GTK_BOX(sysenv.mpane), frame, FALSE, TRUE, 0);

/* event box (get an x window for setting black background) */
event_box = gtk_event_box_new();
gtk_container_add(GTK_CONTAINER(frame), event_box);

/* force black background & white foreground */
logo_style = gtk_style_copy(style);
logo_style->bg[GTK_STATE_NORMAL].red   = 0;
logo_style->bg[GTK_STATE_NORMAL].green = 0;
logo_style->bg[GTK_STATE_NORMAL].blue  = 0;
gtk_widget_set_style(GTK_WIDGET(event_box), logo_style);

hbox = gtk_hbox_new (FALSE, 10);
gtk_container_add (GTK_CONTAINER(event_box), hbox);
gtk_container_set_border_width (GTK_CONTAINER(hbox), PANEL_SPACING);


gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                          &style->bg[GTK_STATE_NORMAL], logo_left_81_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_box_pack_start(GTK_BOX(hbox), gdis_wid, FALSE, FALSE, 0);


gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                          &style->bg[GTK_STATE_NORMAL], logo_right_81_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_box_pack_end(GTK_BOX(hbox), gdis_wid, FALSE, FALSE, 0);


/* RIGHT PANE */
vbox = gtk_vbox_new(FALSE, 0);
gtk_paned_pack2(GTK_PANED(hpaned), vbox, TRUE, TRUE);

/* paned window */
vpaned = gtk_vpaned_new();
gtk_container_add(GTK_CONTAINER(vbox), vpaned);

sysenv.display_box = gtk_vbox_new(FALSE, 0);
gtk_paned_add1(GTK_PANED(vpaned), sysenv.display_box);

/* create the main drawing area */
if (gl_init_visual())
  exit(1);

canvas_split(NULL, GINT_TO_POINTER(CANVAS_INIT));


/* text status pane */
sysenv.tpane = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sysenv.tpane),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_paned_add2(GTK_PANED(vpaned), sysenv.tpane);
gtk_widget_set_size_request(sysenv.tpane, -1, sysenv.tray_height);
gtk_widget_show(sysenv.tpane);

text = g_strdup_printf("This is free software, distributed under the terms of the GNU public license (GPL). For more information visit http://www.gnu.org\n");
show_text(WARNING, text);
g_free(text);

text = g_strdup_printf("Welcome to GDIS version %4.2f.%d, Copyright (C) 2003 by Sean Fleming\n",VERSION,PATCH); 
show_text(STANDARD, text);
g_free(text);

/* window signals */
g_signal_connect(GTK_OBJECT(window), "destroy",
                 GTK_SIGNAL_FUNC(gdis_exit), NULL);

/* show all */
gtk_widget_show_all(window);

/* hide requested */
if (!sysenv.apb_on)
  gtk_widget_hide(sysenv.apb_frame);
if (!sysenv.rab_on)
  gtk_widget_hide(sysenv.rab_frame);

font_init(NULL);
}

