/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#ifndef __WIN32
#include <sys/times.h>
#endif

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

gint selected_label = -1;
GtkWidget *label_list, *start_spin, *stop_spin;
GtkWidget *match1, *match2, *match3, *match4, *search;
GtkWidget *meas_tv;
GtkTreeStore *meas_ts=NULL;

enum {MEAS_NAME, MEAS_ATOMS, MEAS_VALUE, MEAS_MODEL, MEAS_POINTER, MEAS_NCOLS};

/*********************************/
/* print the geometry label list */
/*********************************/
void dump_geom_info(void)
{
gint i;
gchar *info;
GSList *list, *list2;
struct model_pak *data;
struct geom_pak *geom;
struct core_pak *core[4];

/* check */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

printf("-----------------------------------------------\n");
for (list=data->geom ; list ; list=g_slist_next(list))
  {
  geom = (struct geom_pak *) list->data;
  info = NULL;

/* get all cores */
  i=0;
  for (list2 = geom->cores ; list2 ; list2=g_slist_next(list2))
    core[i++] = (struct core_pak *) list2->data;

/* measurement type */
  switch(geom->type)
    {
    case BOND:
      g_assert(i == 2);
      info = g_strdup_printf("Bond      %-4s : %-4s               =  %7.3f", 
               core[0]->label, core[1]->label, calc_dist(core[0], core[1]));
      break;

    case DIST:
      g_assert(i == 2);
      info = g_strdup_printf("Distance  %-4s : %-4s               =  %7.3f", 
               core[0]->label, core[1]->label, calc_dist(core[0], core[1]));
      break;

    case ANGLE:
      g_assert(i == 3);
      info = g_strdup_printf("Angle     %-4s : %-4s : %-4s        =  %7.3f",
                             core[0]->label, core[1]->label, core[2]->label,
                                     calc_angle(core[0], core[1], core[2]));
      break;

    case DIHEDRAL:
      g_assert(i == 4);
      info = g_strdup_printf("Torsion   %-4s : %-4s : %-4s : %-4s =  %7.3f",
             core[0]->label, core[1]->label, core[2]->label, core[3]->label,
                         calc_dihedral(core[0], core[1], core[2], core[3]));
      break;
    }

/* output */
  if (info)
    {
    printf("%s\n", info);
    g_free(info);
    }
  } 
printf("-----------------------------------------------\n");
}

/*****************************/
/* safely acquire tree model */
/*****************************/
GtkTreeModel *meas_treemodel(void)
{
if (!meas_tv)
  return(NULL);
if (!GTK_IS_TREE_VIEW(meas_tv))
  return(NULL);
return(gtk_tree_view_get_model(GTK_TREE_VIEW(meas_tv)));
}

/*********************************************/
/* free data pointers assoc with an iterator */
/*********************************************/
void meas_free_iter_data(GtkTreeIter *iter)
{
gchar *text;
GtkTreeModel *treemodel;

treemodel = meas_treemodel();
if (!treemodel)
  return;

gtk_tree_model_get(treemodel, iter, MEAS_NAME, &text, -1);
g_free(text);
gtk_tree_model_get(treemodel, iter, MEAS_ATOMS, &text, -1);
g_free(text);
gtk_tree_model_get(treemodel, iter, MEAS_VALUE, &text, -1);
g_free(text);
}

/************************/
/* graft model iterator */
/************************/
GtkTreeIter *meas_graft_model(struct model_pak *model)
{
GtkTreeModel *treemodel;
struct model_pak *target=NULL;
GtkTreeIter *iter;

/* checks */
g_assert(model != NULL);
if (!meas_ts)
  return(NULL);
treemodel = meas_treemodel();
if (!treemodel)
  return(NULL);

iter = g_malloc(sizeof(GtkTreeIter));

/* get the first parent iterator */
if (gtk_tree_model_get_iter_first(treemodel, iter))
  {
/* search for target model, until we run out of iterators */
  gtk_tree_model_get(treemodel, iter, MEAS_MODEL, &target, -1);

  while (target != model)
    {
    if (gtk_tree_model_iter_next(treemodel, iter))
      gtk_tree_model_get(treemodel, iter, MEAS_MODEL, &target, -1);
    else
      break;
    }
  }

/* append new if model is not on the tree */
if (target != model)
  {
  gtk_tree_store_append(meas_ts, iter, NULL);
  gtk_tree_store_set(meas_ts, iter, MEAS_NAME, model->basename,
                                    MEAS_MODEL, model,
                                    MEAS_POINTER, NULL,
                                    -1);
  }
return(iter);
}

/*******************************/
/* update a single measurement */
/*******************************/
void meas_refresh_single(struct geom_pak *geom, struct model_pak *model)
{
gint i;
gdouble val;
GSList *list;
struct core_pak *core[4];

g_assert(geom != NULL);
g_assert(model != NULL);

/* get measurement core listing */
i=0;
for (list=geom->cores ; list ; list=g_slist_next(list))
  core[i++] = (struct core_pak *) list->data;

/* redo measurements in case they've changed */
switch(geom->type)
  {
  case BOND:
    g_assert(i == 2);
    val = calc_dist(core[0], core[1]);
    g_free(geom->text);
    geom->text = g_strdup_printf("%7.3f", val);
    break;

  case DIST:
    g_assert(i == 2);
    val = calc_dist(core[0], core[1]);
    g_free(geom->text);
    geom->text = g_strdup_printf("%7.3f", val);
    break;

  case ANGLE:
    g_assert(i == 3);
    val = calc_angle(core[0], core[1], core[2]);
    g_free(geom->text);
    geom->text = g_strdup_printf("%7.3f", val);
    break;

  case DIHEDRAL:
    g_assert(i == 4);
    val = calc_dihedral(core[0], core[1], core[2], core[3]);
    g_free(geom->text);
    geom->text = g_strdup_printf("%7.3f", val);
    break;
  }
}

/***************************************/
/* update all measurements for a model */
/***************************************/
void meas_refresh_model(struct model_pak *model)
{
GSList *list;

for (list=model->geom ; list ; list=g_slist_next(list))
  meas_refresh_single(list->data, model);
}

/******************************/
/* graft a single measurement */
/******************************/
void meas_graft_single(struct geom_pak *geom, struct model_pak *model)
{
gint i;
gchar *info[3];
GSList *list;
GtkTreeIter *parent, iter;
struct core_pak *core[4];

/* checks */
g_assert(geom != NULL);
g_assert(model != NULL);

/* update dialog (if it exists) */
if (!meas_ts)
  return;
parent = meas_graft_model(model);
if (!parent)
  return;

/* set child iterator data */
gtk_tree_store_append(meas_ts, &iter, parent);

/* get measurement core listing */
i=0;
for (list=geom->cores ; list ; list=g_slist_next(list))
  core[i++] = (struct core_pak *) list->data;

switch(geom->type)
  {
  case BOND:
    g_assert(i == 2);
    info[0] = g_strdup("Bond");
    info[1] = g_strdup_printf("%4s : %-4s",core[0]->label, core[1]->label);
    info[2] = geom->text;
    break;

  case DIST:
    g_assert(i == 2);
    info[0] = g_strdup("Dist");
    info[1] = g_strdup_printf("%4s : %-4s",core[0]->label, core[1]->label);
    info[2] = geom->text;
    break;

  case ANGLE:
    g_assert(i == 3);
    info[0] = g_strdup("Angle");
    info[1] = g_strdup_printf("%4s : %-4s : %-4s", core[0]->label,
                                  core[1]->label, core[2]->label);
    info[2] = geom->text;
    break;

  case DIHEDRAL:
    g_assert(i == 4);
    info[0] = g_strdup("Torsion");
    info[1] = g_strdup_printf("%4s : %-4s : %-4s : %-4s", core[0]->label,
                           core[1]->label, core[2]->label, core[3]->label);
    info[2] = geom->text;
    break;

  default:
    info[0] = g_strdup("Unknown");
    info[1] = g_strdup("?");
    info[2] = NULL;
  }

/* add the info */
gtk_tree_store_set(meas_ts, &iter, MEAS_NAME, info[0],
                                   MEAS_ATOMS, info[1],
                                   MEAS_VALUE, info[2],
                                   MEAS_MODEL, model,
                                   MEAS_POINTER, geom,
                                   -1);

g_free(parent);
}

/*************************************/
/* remove model and its measurements */
/*************************************/
void meas_prune_all(struct model_pak *model)
{
GtkTreeIter *parent, child;
GtkTreeModel *treemodel;

/* checks */
g_assert(model != NULL);
if (!meas_ts)
  return;
treemodel = meas_treemodel();
if (!treemodel)
  return;

parent = meas_graft_model(model);
if (parent)
  {
/* free all child iterator data (iterators themselves are auto freed after) */
  if (gtk_tree_model_iter_children(treemodel, &child, parent))
    {
    do
      {
      meas_free_iter_data(&child);
      }
    while (gtk_tree_model_iter_next(treemodel, &child));
    }
/* remove parent (auto removes all child iterators) */
  gtk_tree_store_remove(meas_ts, parent); 
  g_free(parent);
  }
}

/***********************************/
/* update measurements for a model */
/***********************************/
void meas_graft_all(struct model_pak *model)
{
gboolean flag;
GtkTreeModel *treemodel;
GtkTreeIter *parent, child, temp;
GSList *list;

/* checks */
g_assert(model != NULL);

/* update underlying measurements */
meas_refresh_model(model);

if (!meas_ts)
  return;
treemodel = meas_treemodel();
if (!treemodel)
  return;

parent = meas_graft_model(model);
if (parent)
  {
/* clear existing measurements (if any) */
  if (gtk_tree_model_iter_children(treemodel, &child, parent))
    {
    flag=FALSE;
    do
      {
      temp = child;
      if (!gtk_tree_model_iter_next(treemodel, &child));
        flag = TRUE;
      gtk_tree_store_remove(meas_ts, &temp); 
      }
    while(!flag);
    }

/* add all measurements */
  for (list=model->geom ; list ; list=g_slist_next(list))
    meas_graft_single(list->data, model);

/* remove parent if there are no measurements to add */
  if (!g_slist_length(model->geom))
    gtk_tree_store_remove(meas_ts, parent); 

/* done */
  g_free(parent);
  }
}

/***************************/
/* geometry label creation */
/***************************/
void new_geom(gint type, gchar *label, GSList *cores, struct model_pak *model)
{
struct geom_pak *geom;

geom = g_malloc(sizeof(struct geom_pak));

/* init */
geom->type = type;
if (label)
  geom->text = g_strdup(label);
else
  geom->text = g_strdup("?");
geom->cores = cores;

model->geom = g_slist_append(model->geom, geom);

meas_refresh_single(geom, model);
meas_graft_single(geom, model);
}

/****************************/
/* free geometry label info */
/****************************/
void meas_free_all(struct model_pak *model)
{
GSList *list;
struct geom_pak *geom;

/* delete all labels */
list = model->geom;
while (list)
  {
  geom = (struct geom_pak *) list->data;
  list = g_slist_next(list);

  model->geom = g_slist_remove(model->geom, geom);
  g_slist_free(geom->cores);
  g_free(geom->text);
  g_free(geom);
  }
model->geom = NULL;
}

/***************************/
/* geometry label deletion */
/***************************/
void meas_prune_selected(void)
{
GtkTreeIter iter;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct geom_pak *geom=NULL;
struct model_pak *model=NULL;

/* checks */
treemodel = meas_treemodel();
if (!treemodel)
  return;

/* get selected iterator */
if (!(selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(meas_tv))))
  return;
if (!gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  return;

gtk_tree_model_get(treemodel, &iter, MEAS_MODEL, &model, -1);
gtk_tree_model_get(treemodel, &iter, MEAS_POINTER, &geom, -1);

g_assert(model != NULL);

/* if measurement selected - delete single, else delete all */
if (geom)
  {
/* free iterator & data */
  meas_free_iter_data(&iter);
  gtk_tree_store_remove(meas_ts, &iter); 

/* free measurement data */
  model->geom = g_slist_remove(model->geom, geom);
  g_slist_free(geom->cores);
  g_free(geom->text);
  g_free(geom);
  }
else
  {
  meas_prune_all(model);
  meas_free_all(model);
  }

redraw_canvas(SINGLE);
}

/*******************************/
/* routines for geometry calcs */
/*******************************/
gdouble calc_dist(struct core_pak *core1, struct core_pak *core2)
{
gdouble r[3];

ARR3SET(r, core1->rx);
ARR3SUB(r, core2->rx);

return(VEC3MAG(r));
}

gdouble calc_angle(struct core_pak *core1, struct core_pak *core2,
                                           struct core_pak *core3)
{
gdouble a[3], b[3];

/* compute vectors (from center atom) */
ARR3SET(a, core1->rx);
ARR3SUB(a, core2->rx);
ARR3SET(b, core3->rx);
ARR3SUB(b, core2->rx);

return(R2D*via(a,b,3)); 
}

gdouble calc_dihedral(struct core_pak *core1, struct core_pak *core2,
                      struct core_pak *core3, struct core_pak *core4)
{
gdouble len;
gdouble a[3], b[3], n[3];

/* compute end atom vectors (from 1-2 axis) */
ARR3SET(a, core1->rx);
ARR3SUB(a, core2->rx);
ARR3SET(b, core4->rx);
ARR3SUB(b, core3->rx);

/* compute normal to the plane in which the dihedral is to be calc'd */
ARR3SET(n, core2->rx);
ARR3SUB(n, core3->rx);
normalize(n, 3);

/* n[0..2] is the normal of the plane */
/* project 1-0 onto the normal (ie dist to plane) */
len = a[0]*n[0] + a[1]*n[1] + a[2]*n[2];
/* subtract vector to plane from 1-0 to get projection on plane */
a[0] -= len*n[0];
a[1] -= len*n[1];
a[2] -= len*n[2];
/* project 2-3 onto the normal */
len = b[0]*n[0] + b[1]*n[1] + b[2]*n[2];
/* subtract vector to plane from 2-3 to get projection on plane */
b[0] -= len*n[0];
b[1] -= len*n[1];
b[2] -= len*n[2];
/* compute angle between projected vectors */
return(180.0*via(a,b,3)/PI); 
}

/*****************/
/* info on bonds */
/*****************/
void gl_info_bond(GtkWidget *w, gint x, gint y, struct model_pak *data)
{
gchar txt[10];
GSList *bond, *cores;
struct core_pak *core[2];

/* seek */
bond = gl_seek_bond(w, x, y, data);
if (!bond)
  return;

core[0] = ((struct bond_pak *) bond->data)->atom1;
core[1] = ((struct bond_pak *) bond->data)->atom2;

g_snprintf(txt,8,"%7.3f", calc_dist(core[0], core[1]));

cores = NULL;
cores = g_slist_prepend(cores, core[0]);
cores = g_slist_prepend(cores, core[1]);
new_geom(BOND, txt, cores, data);

redraw_canvas(SINGLE);
}

/*****************/
/* info on dists */
/*****************/
void info_dist(gint x, gint y, struct model_pak *data)
{
gdouble r[3];
gchar txt[10];
GSList *cores;
static struct core_pak *core[2];

/* attempt to find core */
pixel_coord_map(x, y, r, data);
if (!(core[data->state] = seek_coord2d(r, data)))
  return;

/* are we looking at the 1st or the 2nd call? */
switch (data->state)
  {
  case 0:
/* select */
    select_add_core(core[0], data);
    data->state++;
    break;

  case 1:
/* remove from selection */
    select_del_core(core[0], data);
    select_del_core(core[1], data);

/* create the label */
    g_snprintf(txt,8,"%7.3f", calc_dist(core[0], core[1]));
    cores = NULL;
    cores = g_slist_prepend(cores, core[0]);
    cores = g_slist_prepend(cores, core[1]);
    new_geom(DIST, txt, cores, data);

    data->state--;
    break;

  default:
    g_assert_not_reached();
    data->state=0;
    return;
  }

/* redraw */
redraw_canvas(SINGLE);
}

/******************/
/* info on angles */
/******************/
void info_angle(gint x, gint y, struct model_pak *data)
{
gdouble angle, r[3];
gchar txt[40];
GSList *cores;
static struct core_pak *core[3];

/* attempt to find core */
pixel_coord_map(x, y, r, data);
if (!(core[data->state] = seek_coord2d(r, data)))
  return;

/* are we looking at the 1st or the 2nd call? */
switch (data->state)
  {
  case 0:
  case 1:
    select_add_core(core[data->state], data);
    data->state++;
    break;

  case 2:
/* remove highlighting */
    select_del_core(core[0], data);
    select_del_core(core[1], data);
    select_del_core(core[2], data);

/* create the label */
    angle = calc_angle(core[0], core[1], core[2]);
    g_snprintf(txt,8,"%7.3f",fabs(angle));
    cores = NULL;
    cores = g_slist_prepend(cores, core[0]);
    cores = g_slist_prepend(cores, core[1]);
    cores = g_slist_prepend(cores, core[2]);
    new_geom(ANGLE, txt, cores, data);

    data->state=0;
    break;

  default:
    g_assert_not_reached();
    data->state=0;
    return;
  }

/* update */
redraw_canvas(SINGLE);
}

/****************************/
/* info on torsional angles */
/****************************/
void info_torsion(gint x, gint y, struct model_pak *data)
{
gdouble angle, r[3];
gchar txt[40];
GSList *cores;
static struct core_pak *core[4];

/* attempt to find core */
pixel_coord_map(x, y, r, data);
if (!(core[data->state] = seek_coord2d(r, data)))
  return;

/* what stage are we at? */
switch (data->state)
  {
  case 0:
  case 1:
  case 2:
    select_add_core(core[data->state], data);
    data->state++;
    break;

  case 3:
/* remove highlighting */
    select_del_core(core[0], data);
    select_del_core(core[1], data);
    select_del_core(core[2], data);
    select_del_core(core[3], data);

/* create the label */
    angle = calc_dihedral(core[0], core[1], core[2], core[3]);
    g_snprintf(txt,8,"%7.3f",fabs(angle));

    cores = NULL;
    cores = g_slist_prepend(cores, core[0]);
    cores = g_slist_prepend(cores, core[1]);
    cores = g_slist_prepend(cores, core[2]);
    cores = g_slist_prepend(cores, core[3]);
    new_geom(DIHEDRAL, txt, cores, data);

    data->state=0;
    break;

  default:
    printf("Error: bad state in info_angles()\n");
    data->state=0;
    return;
  }

/* update */
redraw_canvas(SINGLE);
}

/************************/
/* match pairs of atoms */
/************************/
#define DEBUG_PAIR_MATCH 0
gint pair_match(const gchar *label1, const gchar *label2,
                struct core_pak *core1, struct core_pak *core2)
{
gint a, b, i, j, mask;

#if DEBUG_PAIR_MATCH
printf("[%s,%s] : [%s,%s]\n", label1, label2, core1->label, core2->label);
#endif

/* if a or b = 0 => any i or j is accepted */
/* otherwise it must match either i or j */
/* if a and b are non zero, both i & j must match both a & b */
a = elem_type(label1);
b = elem_type(label2);
i = core1->atom_code;
j = core2->atom_code;

/* fill out the mask */
mask = 0;
if (i == a)
  {
/* if input label doesn't match the element symbol length - it means the */
/* user has put in something like H1 - compare this with the atom label */
  if (g_ascii_strcasecmp(label1, elements[i].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core1->label, label1) == 0)
      mask |= 1;
    }
  else
    mask |= 1; 
  }

if (j == a)
  {
  if (g_ascii_strcasecmp(label1, elements[j].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core2->label, label1) == 0)
      mask |= 2;
    }
  else
    mask |= 2; 
  }

if (i == b)
  {
  if (g_ascii_strcasecmp(label2, elements[i].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core1->label, label2) == 0)
      mask |= 4;
    }
  else
    mask |= 4; 
  }

if (j == b)
  {
  if (g_ascii_strcasecmp(label2, elements[j].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core2->label, label2) == 0)
      mask |= 8;
    }
  else
    mask |= 8; 
  }

#if DEBUG_PAIR_MATCH
printf("mask = %d\n", mask);
#endif

/* if both types must match - only two possibilities (a,b) or (b,a) */
/* but we can get further matches (than the required 2) when labels are compared */
if (a && b)
  {
  switch(mask)
    {
/* single valid pair match */
    case 6:
    case 9:
/* valid pair match plus one extra */
    case 7:
    case 11:
    case 13:
    case 14:
/* pair match in all possible combinations */
    case 15:
      break;
/* bad match - exit */
    default:
      return(0);
    }
  }

/* if only one type to match - any match at all will do */
if (a || b)
  if (!mask)
    return(0);

#if DEBUG_PAIR_MATCH
printf("accepted [%d][%d] as a match.\n",i,j);
#endif

return(1);
}

/***************************/
/* geometry search - bonds */
/***************************/
#define DEBUG_BOND_SEARCH 0
gint bond_search(gint type, gdouble r1, gdouble r2, struct model_pak *data)
{
gint a, b, count=0, match;
const gchar *label1, *label2;
gchar txt[80];
gdouble r;
GSList *list, *cores=NULL;
struct core_pak *core[2];
struct bond_pak *bond;

/* setup type requirements */
label1 = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
a = elem_type(label1);

#if DEBUG_BOND_SEARCH
if (a)
  printf("Valid type 1 [%s][%d]\n", label1, a);
#endif

label2 = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));
b = elem_type(label2);

#if DEBUG_BOND_SEARCH
if (b)
  printf("Valid type 2 [%s][%d]\n", label2, b);
#endif

/* we're only interested in the first two bits. */
type &= 3;

#if DEBUG_BOND_SEARCH
printf("type = %d\n", type);
#endif

/* go through bond list */
for (list=data->bonds ; list ; list=g_slist_next(list))
  {
  bond = (struct bond_pak *) list->data;

/* the two atoms */
  core[0] = bond->atom1;
  core[1] = bond->atom2;

/* checks */
  if (bond->type == BOND_HBOND)
    continue;
  if (core[0]->status & (HIDDEN | DELETED))
    continue;
  if (core[1]->status & (HIDDEN | DELETED))
    continue;

/* do we have selection requirements? */
  if (type)
    {
/* check atoms against selection */
    match=0;
    if (g_slist_find(data->selection, core[0]))
      match |= 2;
    if (g_slist_find(data->selection, core[1]))
      match |= 1;

#if DEBUG_BOND_SEARCH
printf("[%s][%s] : ", core[0]->label, core[1]->label);
printf("type, match = %d, %d\n", type, match);
#endif
    switch(type)
      {
/* single matches - input type against the atom that's not in the selection */
      case 1:
        label1 = label2;
      case 2:
        if (match)
          {
          if (!core_match(label1, core[match-1]))
            {
            list = g_slist_next(list);
            continue;
            }
          else
            break;
          }
      case 3:
/* double match required */
        if (type != match)
          {
          list = g_slist_next(list);
          continue;
          }
/* test for the match */
        if (!pair_match(label1, label2, core[0], core[1]))
          {
          list = g_slist_next(list);
          continue;
          }
      }
    }
  else
    {
/* no selection requirements - just attempt to match the labels */
    if (!pair_match(label1, label2, core[0], core[1]))
      {
      list = g_slist_next(list);
      continue;
      }
    }

/* we now have two atoms we can calculate a distance for */
  r = calc_dist(core[0], core[1]);
  if (r >= r1 && r <= r2)
    {
#if DEBUG_BOND_SEARCH
printf("bond %s-%s : %p-%p\n",core[0]->label,core[1]->label,core[0],core[1]);
#endif
    g_snprintf(txt,8,"%7.3f",r);

    cores = NULL;
    cores = g_slist_prepend(cores, core[0]);
    cores = g_slist_prepend(cores, core[1]);
    new_geom(BOND, txt, cores, data);

    count++;
    }
  }
return(count);
}

/*******************************/
/* geometry search - distances */
/*******************************/
#define DEBUG_DIST_SEARCH 0
gint dist_search(gint type, gdouble r1, gdouble r2, struct model_pak *data, gint inter)
{
gint a, b, count=0, flag;
const gchar *label1, *label2;
gchar txt[10];
gdouble r;
GSList *list1, *list2, *list3, *cores;
struct core_pak *core[2];

/* setup type requirements */
label1 = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
a = elem_type(label1);

label2 = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));
b = elem_type(label2);

/* we're only interested in the first two bits. */
type &= 3;

switch(type)
  {
/* any two atoms */
  case 0:
    for (list1=data->cores ; list1 ; list1=g_slist_next(list1))
      {
      core[0] = (struct core_pak *) list1->data;
      if (core[0]->status & (DELETED | HIDDEN))
        continue;

      for (list2=g_slist_next(list1) ; list2 ; list2=g_slist_next(list2))
        {
        core[1] = (struct core_pak *) list2->data;

        if (core[1]->status & (DELETED | HIDDEN))
          continue;

        if (!pair_match(label1, label2, core[0], core[1]))
          continue;

if (inter)
  if (core[0]->mol == core[1]->mol)
    continue;

/*i we now have two atoms we can calculate a distance for */
        r = calc_dist(core[0], core[1]);
        if (r >= r1 && r <= r2)
          {
#if DEBUG_DIST_SEARCH
printf("match %s-%s : %p-%p\n",core[0]->label,core[1]->label,core[0],core[1]);
#endif
/* create geometry label */
          g_snprintf(txt,8,"%7.3f",r);

          cores = NULL;
          cores = g_slist_prepend(cores, core[0]);
          cores = g_slist_prepend(cores, core[1]);
          new_geom(DIST, txt, cores, data);

          count++;
          }
        }
      }
    break;

/* one atom in selection */
/* make it so label2 is always what we're looking to match */
  case 2:
    label2 = label1;
  case 1:
    for (list1=data->selection ; list1 ; list1=g_slist_next(list1))
      {
      core[0] = (struct core_pak *) list1->data;
      if (core[0]->status & (DELETED | HIDDEN))
        continue;

      for (list2=data->cores ; list2 ; list2=g_slist_next(list2))
        {
        core[1] = (struct core_pak *) list2->data;
        if (core[1]->status & (DELETED | HIDDEN))
          continue;

/* core[0] is in selection, only core[1] must match */
        if (!core_match(label2, core[1]))
          continue;

if (inter)
  if (core[0]->mol == core[1]->mol)
    continue;

/* avoid double counting - require atom[0] < atom[1], if both selected */
        flag=0;
        for (list3=data->selection ; list3 ; list3=g_slist_next(list3))
          if (list3->data == core[1])
            if (core[0] >= core[1])
              flag++;

        if (flag)
          continue;

/* we now have two atoms we can calculate a distance for */
        r = calc_dist(core[0], core[1]);
        if (r >= r1 && r <= r2)
          {
#if DEBUG_DIST_SEARCH
printf("match %s-%s : %p-%p\n",core[0]->label,core[1]->label,core[0],core[1]);
#endif
/* create geometry label */
          g_snprintf(txt,8,"%7.3f",r);

          cores = NULL;
          cores = g_slist_prepend(cores, core[0]);
          cores = g_slist_prepend(cores, core[1]);
          new_geom(DIST, txt, cores, data);

          count++;
          }
        }
      }
    break;

/* both atoms in selection */
  default:
    for (list1=data->selection ; list1 ; list1=g_slist_next(list1))
      {
      core[0] = (struct core_pak *) list1->data;
      if (core[0]->status & (DELETED | HIDDEN))
        continue;

      for (list2 = g_slist_next(list1) ; list2 ; list2=g_slist_next(list2))
        {
        core[1] = (struct core_pak *) list2->data;
        if (core[1]->status & (DELETED | HIDDEN))
          continue;

if (inter)
  if (core[0]->mol == core[1]->mol)
    continue;

/* we now have two atoms we can calculate a distance for */
        r = calc_dist(core[0], core[1]);
        if (r >= r1 && r <= r2)
          {
#if DEBUG_DIST_SEARCH
printf("match %s-%s : %p-%p\n",core[0]->label,core[1]->label,core[0],core[1]);
#endif
/* create geometry label */
          g_snprintf(txt,8,"%7.3f",r);

          cores = NULL;
          cores = g_slist_prepend(cores, core[0]);
          cores = g_slist_prepend(cores, core[1]);
          new_geom(DIST, txt, cores, data);

          count++;
          }
        }
      }
    break;
  }
return(count);
}

/****************************/
/* geometry search - angles */
/****************************/
#define DEBUG_ANGLE_SEARCH 0
gint angle_search(gint type, gdouble a1, gdouble a2, struct model_pak *data)
{
gint i, match, count=0;
gint test[3], temp1[3], temp2[3], temp3[3];
gchar txt[10];
gdouble angle;
GSList *list1, *list2, *list3, *list4, *cores=NULL;
struct core_pak *core[3];
struct bond_pak *bond1, *bond2;

/* three levels of matching => three temp arrays for storing the match state */
/* so that at the start of each loop we can restore the state and test again */
/* top level match state */
temp1[0] = elem_type(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry)));
temp1[1] = elem_type(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry)));
temp1[2] = elem_type(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match3)->entry)));

#if DEBUG_ANGLE_SEARCH
printf("user request [%s : %s : %s]\n", 
       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry)),
       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry)),
       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match3)->entry)));
#endif

/* only interested in the first 3 bits */
type &= 7;

/* loop over candidate central atoms */
for (list1=data->cores ; list1 ; list1=g_slist_next(list1))
  {
  core[1] = (struct core_pak *) list1->data;
  if (core[1]->status & (DELETED | HIDDEN))
    continue;

#if DEBUG_ANGLE_SEARCH
printf("[%d : %d : %d]\n", test[0], test[1], test[2]);
#endif

/* compare with requested elements in saved match state */
  ARR3SET(test, temp1);
  for (i=3 ; i-- ; )
    if (test[i])
      if (test[i] == core[1]->atom_code)
        {
/* only one match allowed at this level */
        test[i] = 0;
        break;
        }

/* compare only central atoms with at least 2 bonds */
  if (g_slist_length(core[1]->bonds) > 1)
    {
/* save match state */
    ARR3SET(temp2, test);
/* arm 1 of the three body term */
    for (list2=core[1]->bonds ; list2 ; list2=g_slist_next(list2))
      {
      bond1 = (struct bond_pak *) list2->data;
      if (bond1->atom1 == core[1])
        core[0] = bond1->atom2;
      else
        core[0] = bond1->atom1;

      if (core[0]->status & (DELETED | HIDDEN))
        continue;

/* compare with requested elements in saved match state */
      ARR3SET(test, temp2);
      for (i=3 ; i-- ; )
        if (test[i])
          if (test[i] == core[0]->atom_code)
            {
/* only one match allowed at this level */
            test[i] = 0;
            break;
            }

/* save match state */
      ARR3SET(temp3, test);
/* arm 2 of the three body term */
      for (list3=g_slist_next(list2) ; list3 ; list3=g_slist_next(list3))
        {
        bond2 = (struct bond_pak *) list3->data;
        if (bond2->atom1 == core[1])
          core[2] = bond2->atom2;
        else
          core[2] = bond2->atom1;
        if (core[2]->status & (DELETED | HIDDEN))
          continue;

/* compare with requested elements in saved match state */
        ARR3SET(test, temp3);
        for (i=3 ; i-- ; )
          if (test[i])
            if (test[i] == core[2]->atom_code)
              {
/* only one match allowed at this level */
              test[i] = 0;
              break;
              }

#if DEBUG_ANGLE_SEARCH
printf(" >>> [%s : %s : %s]\n", core[0]->label, core[1]->label, core[2]->label);
printf("   > [%d : %d : %d]\n", test[0], test[1], test[2]);
#endif

/* skip if our 3 cores don't satisfy all user requested elements */
/* ie ignore if there is at least one non-zero element in test[] */
        if (VEC3MAG(test) > FRACTION_TOLERANCE)
          continue;

/* get angle & check if it's within required limits */
        angle = calc_angle(core[0], core[1], core[2]);
        if (angle > a1 && angle < a2)
          {
/* which atoms (if any) are in the selection */
          match=0;
          for (list4=data->selection ; list4 ; list4=g_slist_next(list4))
            {
            if (core[0] == list4->data)
              match++;
            if (core[1] == list4->data)
              match++;
            if (core[2] == list4->data)
              match++;
            }
/* setup list for passing to label creation function */
          cores = NULL;
          cores = g_slist_prepend(cores, core[0]);
          cores = g_slist_prepend(cores, core[1]);
          cores = g_slist_prepend(cores, core[2]);
/* match against the selection requirements */
          switch(type)
            {
            case 0:
/* create geometry label */
              g_snprintf(txt,8,"%8.2f", angle);
              new_geom(ANGLE, txt, cores, data);
              count++;
              break;
            case 1:
            case 2:
            case 4:
/* one atom must be in selection -> any non zero match will do */
              if (match)
                {
                g_snprintf(txt,8,"%8.2f", angle);
                new_geom(ANGLE, txt, cores, data);
                count++;
                }
              break;
            case 3:
            case 5:
            case 6:
/* two atoms must be in selection */
              if (match == 2)
                {
                g_snprintf(txt,8,"%8.2f", angle);
                new_geom(ANGLE, txt, cores, data);
                count++;
                }
              break;
            case 7:
/* all */
              if (match == 3)
                {
                g_snprintf(txt,8,"%8.2f", angle);
                new_geom(ANGLE, txt, cores, data);
                count++;
                }
              break;
            }
          }
        }
      }
    }
  }
return(count);
}

/******************************/
/* geometry search - torsions */
/******************************/
gint dihedral_search(struct model_pak *data, gdouble a1, gdouble a2, gint type)
{
type &= 15;
printf("Not implemented yet!\n");
return(0);
}

/********************************************/
/* geometry search callback - general setup */
/********************************************/
void geom_search(void)
{
gint count=0, match_mask=0;
gdouble x1, x2;
const gchar *tmp;
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

x1 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(start_spin));
x2 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(stop_spin));

/* setup the atom matching requirements */
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
if (g_ascii_strncasecmp(tmp, "Select", 6) == 0)
  match_mask |= 1;
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));
if (g_ascii_strncasecmp(tmp, "Select", 6) == 0)
  match_mask |= 2;
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match3)->entry));
if (g_ascii_strncasecmp(tmp, "Select", 6) == 0)
  match_mask |= 4;

/* get the measurement type (ensure match mask ignores irrelevant bits) */
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(search)->entry));
if (g_ascii_strncasecmp(tmp, "Bonds", 5) == 0)
  count = bond_search(match_mask, x1, x2, data);
if (g_ascii_strncasecmp(tmp, "Distance", 8) == 0)
  count = dist_search(match_mask, x1, x2, data, 0);
if (g_ascii_strncasecmp(tmp, "Intermolecular", 14) == 0)
  count = dist_search(match_mask, x1, x2, data, 1);
if (g_ascii_strncasecmp(tmp, "Angles", 6) == 0)
  count = angle_search(match_mask, x1, x2, data);

redraw_canvas(SINGLE);
}

/********************/
/* clean up on exit */
/********************/
void meas_cleanup(void)
{
/* NB: since the tree store is independant of the model's geom list */
/* it must be completely removed (and then restored) with the dialog */
gtk_tree_store_clear(meas_ts);
meas_tv = NULL;
meas_ts = NULL;
}

/*****************************/
/* manually add measurements */
/*****************************/
void meas_manual_page(GtkWidget *box)
{
GtkWidget *frame, *vbox;

/* Frame - type */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_button_x("Measure distances",
               gtk_switch_mode, GINT_TO_POINTER(DIST_INFO), vbox);
gtksh_button_x("Measure bonds",
               gtk_switch_mode, GINT_TO_POINTER(BOND_INFO), vbox);
gtksh_button_x("Measure angles",
               gtk_switch_mode, GINT_TO_POINTER(ANGLE_INFO), vbox);
gtksh_button_x("Measure dihedrals",
               gtk_switch_mode, GINT_TO_POINTER(DIHEDRAL_INFO), vbox);
}

/***************/
/* search page */
/***************/
void meas_search_page(GtkWidget *box)
{
GtkWidget *frame, *vbox, *hbox, *table, *button, *label;
GtkAdjustment *adj;
GList *match_list=NULL, *search_list=NULL;

/* Frame - type */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* atom matching options */
table = gtk_table_new(3, 4, FALSE);
gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

/* labels for top row */
label = gtk_label_new("Atom 1");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
label = gtk_label_new("Atom 2");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
label = gtk_label_new("Atom 3");
gtk_table_attach_defaults(GTK_TABLE(table),label,2,3,0,1);

/* construct combo options */
match_list = g_list_append(match_list, "Any");
match_list = g_list_append(match_list, "Selected");
search_list = g_list_append(search_list, "Bonds");
search_list = g_list_append(search_list, "Distances");
search_list = g_list_append(search_list, "Intermolecular");
search_list = g_list_append(search_list, "Angles");

/* atom match combo boxes */
match1 = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(match1)->entry), "Any");
gtk_combo_set_popdown_strings(GTK_COMBO(match1), match_list);
gtk_table_attach_defaults(GTK_TABLE(table),match1,0,1,1,2);

match2 = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(match2)->entry), "Any");
gtk_combo_set_popdown_strings(GTK_COMBO(match2), match_list);
gtk_table_attach_defaults(GTK_TABLE(table),match2,1,2,1,2);

match3 = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(match3)->entry), "Any");
gtk_combo_set_popdown_strings(GTK_COMBO(match3), match_list);
gtk_table_attach_defaults(GTK_TABLE(table),match3,2,3,1,2);

/* labels for second row */
label = gtk_label_new("Search type");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
label = gtk_label_new("Search range");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,2,3);

/* search type combo box */
search = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(search)->entry), "Bonds");
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(search)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(search), search_list);
gtk_table_attach_defaults(GTK_TABLE(table),search,0,1,3,4);

/* stop distance spinner */
hbox = gtk_hbox_new(TRUE, 0);

/* start value */
adj = (GtkAdjustment *) gtk_adjustment_new
      (0.1, 0.0, 360.0, 0.1, 1.0, 0);
start_spin = gtk_spin_button_new (adj, 0.01, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON (start_spin), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), start_spin, TRUE, TRUE, 0);

/* stop value */
adj = (GtkAdjustment *) gtk_adjustment_new
      (2.0, 0.0, 360.0, 0.1, 1.0, 0);
stop_spin = gtk_spin_button_new (adj, 0.01, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON (stop_spin), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), stop_spin, TRUE, TRUE, 0);
gtk_table_attach_defaults(GTK_TABLE(table),hbox,1,2,3,4);

/* search button */
button = gtksh_stock_button(GTK_STOCK_FIND, geom_search, NULL, NULL);
gtk_table_attach_defaults(GTK_TABLE(table),button,2,3,3,4);

/* I don't like doing this, but the default size is */
/* much larger than it needs to be, wasting too much space */
gtk_widget_set_size_request(match1, 11*sysenv.gfontsize, -1);
gtk_widget_set_size_request(match2, 11*sysenv.gfontsize, -1);
gtk_widget_set_size_request(match3, 11*sysenv.gfontsize, -1);
gtk_widget_set_size_request(search, 11*sysenv.gfontsize, -1);

gtk_table_set_row_spacings(GTK_TABLE(table), PANEL_SPACING);
gtk_table_set_col_spacings(GTK_TABLE(table), PANEL_SPACING);
}

/**************************/
/* geometric measurements */
/**************************/
void geom_info()
{
gint i, id;
gchar *titles[] = {"  Name  ", " Constituent atoms ", " Value "};
GtkWidget *swin, *notebook;
GtkWidget *frame, *vbox, *hbox, *label;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;
struct model_pak *model;
struct dialog_pak *meas_dialog;

/* request dialog */
/* NB: make sure cleanup routine calls meas_exit() */
if ((id = request_dialog(sysenv.active, GEOMETRY)) < 0)
  return;
meas_dialog = &sysenv.dialog[id];
meas_dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW(meas_dialog->win), "Measurements");
gtk_window_set_default_size(GTK_WINDOW(meas_dialog->win), 250, 450);
g_signal_connect(GTK_OBJECT(meas_dialog->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(meas_dialog->win)->vbox), notebook, FALSE, FALSE, 0);

gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);

/* manual measurement */
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new(" Manual ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
meas_manual_page(vbox);

/* searching */
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new(" Search ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
meas_search_page(vbox);

/* measurements viewing pane */
frame = gtk_frame_new ("Label list");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(meas_dialog->win)->vbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);
swin = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), swin, TRUE, TRUE, 0);


/* NEW -  tree view */
/* underlying data storage */
/* 3 strings - data, 1 pointer - the measurement itself */
/* NB: model assoc. is passed with the selection callback */
if (!meas_ts)
  meas_ts = gtk_tree_store_new(MEAS_NCOLS, 
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_POINTER,
                               G_TYPE_POINTER);

meas_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(meas_ts));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), meas_tv);

/* set up column renderers */
for (i=0 ; i<=MEAS_VALUE ; i++)
  {
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes
              (titles[i], renderer, "text", i, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(meas_tv), column);
  }

/* list modification buttons */
hbox = gtk_hbox_new(TRUE, 10);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

gtksh_button(" Select ", select_model_labels, NULL, hbox, TT);
gtksh_button(" Delete ", meas_prune_selected, NULL, hbox, TT);
gtksh_button(" Dump ", dump_geom_info,  NULL, hbox, TT);

/* terminating button */
gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(meas_dialog->win)->action_area);

/* done */
gtk_widget_show_all(meas_dialog->win);

/* refresh all measurements */
for (i=0 ; i<sysenv.num_models ; i++)
  {
  model = model_ptr(i, RECALL);
  meas_graft_all(model);
  }
gtk_tree_view_expand_all(GTK_TREE_VIEW(meas_tv));
}

