/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <math.h>

#ifndef __WIN32
#include <sys/wait.h>
#endif

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "parse.h"
#include "task.h"
#include "morph.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "spatial.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

#define DEBUG 0
#define DELETE_POV_FILE 0

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];
extern GtkWidget *window;

/* global render parameters */
gdouble acm[9];
gdouble p2a;
struct light_pak current_light;
GtkWidget *scale_spin;

#define SCALE_MAG 10
#define PIX2ANG 0.03

/* this is an OpenGL limitation (ie shininess) */
#define MAX_HL 128

/****************/
/* render setup */
/****************/
void render_setup(GtkWidget *w, gpointer dummy)
{
gint i;
gchar *file, *name, *basename;
gdouble mat[9];
GString *cmd;
struct model_pak *data;
FILE *fp;

/* checks */
if (!sysenv.povray_path)
  {
  show_text(ERROR, "POVRay executable was not found.\n");
  return;
  }
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

cmd = g_string_new(NULL);

/* TODO - progress widget? */
/* TODO - eliminate common code also found in animate.c : animate() */
if (sysenv.render.animate)
  {
/* remove any extension */
  file = strdup_basename(sysenv.render.animate_file);
  if (!strlen(file))
    {
    show_text(ERROR, "Please enter a filename.\n");
    return;
    }

  show_text(STANDARD, "Rendering frames, please wait...\n");

/* preserve the current orientation */
  memcpy(mat, data->rotmat, 9*sizeof(gdouble));

/* acquire a file pointer to the animation file */
  fp = fopen(data->filename, "r");
  if (!fp)
    {
    show_text(ERROR, "Failed to open animation stream.\n");
    return;
    }
 
/* animation loop */
  for (i=0 ; i<data->num_frames ; i++)
    {
/* get the numbered frame */
    if (read_frame(fp, i, data))
      break;
/* get a unique .pov filename */
    name = gun("pov");
    make_pov(sysenv.active, name);
    exec_pov(name);
/* get name without the .pov extension */
    basename = strdup_basename(name);
/* make into sequence (pre pad with 0's to keep list sequential) */
/* FIXME - determine max width & construct format using num_frames */
    g_string_sprintf(cmd,"mv -f %s.tga %s_%03d.tga",basename,file,i);
    system(cmd->str);
/* cleanup */
    g_free(basename);
    g_free(name);
    }

/* done - restore initial frame */
  read_frame(fp, data->cur_frame, data);
  fclose(fp);

/* TODO - checks for required packages? */
/* setup for conversion to anim GIF (or whatever) */
  switch(sysenv.render.animate_type)
    {
    case ANIM_GIF:
      g_string_sprintf(cmd,"%s -delay %d %s_*.tga %s.gif",
                            sysenv.convert_path,
                            (gint) sysenv.render.delay, file, file);
      break;

    case ANIM_MPEG:
/* TODO - need mpeg2encode to make this work */
/* adding a '-quality xx' option doesn't seem to help */
      g_string_sprintf(cmd,"%s -delay %d %s_*.tga %s.mpg",
                       sysenv.convert_path,
                       (gint) sysenv.render.delay, file, file);
      break;

    default:
      printf("Error in render_setup(): this shouldn't happen\n");
      return;
    }
/* execute the conversion command */
  system(cmd->str);

/* delete individual frames */
  g_string_sprintf(cmd,"rm -rf %s_*.tga", file);
  system(cmd->str);

/* clean up */
  g_free(file);
  g_string_free(cmd, TRUE);

  show_text(STANDARD, "Animated movie successfully created.\n");
  }
else
  {
/* single model background job */
  run_pov();
  }
}

/******************************/
/* font cancel button handler */
/******************************/
void font_dialog_close(GtkWidget *w, gpointer fsd)
{
gtk_widget_destroy(GTK_WIDGET(fsd));
}

/**************************/
/* font ok button handler */
/**************************/
void gfont_selected(GtkWidget *w, gpointer fsd)
{
gchar *fontname;

fontname = gtk_font_selection_dialog_get_font_name((GtkFontSelectionDialog *) fsd);
strcpy(sysenv.gfont_name, fontname);
font_init(fsd);
}

/**************************/
/* font ok button handler */
/**************************/
void dfont_selected(GtkWidget *w, gpointer fsd)
{
gchar *fontname;

fontname = gtk_font_selection_dialog_get_font_name((GtkFontSelectionDialog *) fsd);
strcpy(sysenv.dfont_name, fontname);
font_init(fsd);
redraw_canvas(ALL);
}

/***********************/
/* gdis font selection */
/***********************/
void gfont_dialog(void)
{
GtkWidget *fsd;

fsd =  gtk_font_selection_dialog_new("Font selection");

/* setup events for the file selection widget */
g_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fsd)->ok_button),
                    "clicked", (GtkSignalFunc) gfont_selected, fsd);
g_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fsd)->cancel_button),
                   "clicked", (GtkSignalFunc) font_dialog_close, fsd);

gtk_widget_show(fsd);
}

/**********************/
/* drawing font selection */
/**********************/
void dfont_dialog(void)
{
GtkWidget *fsd;

fsd =  gtk_font_selection_dialog_new("Font selection");

/* setup events for the file selection widget */
g_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fsd)->ok_button),
                 "clicked", (GtkSignalFunc) dfont_selected, fsd);
g_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fsd)->cancel_button),
                 "clicked", (GtkSignalFunc) font_dialog_close, fsd);

gtk_widget_show(fsd);
}

/************************************/
/* event handler for display dialog */
/************************************/
void render_refresh(void)
{
redraw_canvas(ALL);
}

/********************/
/* Property toggles */
/********************/
void toggle_axes_type(void)
{
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
if (!model)
 return;

if (model->periodic)
  {
  if (model->axes_type == CARTESIAN)
    model->axes_type = OTHER;
  else
    model->axes_type = CARTESIAN;
  }
calc_coords(REFRESH, model);
redraw_canvas(SINGLE);
}
void cs_toggle(GtkWidget *w, gpointer data)
{
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
if (!model)
 return;

if (model->sof_colourize)
  init_objs(SOF_COLOUR, model);
else
  init_objs(REFRESH_COLOUR, model);
redraw_canvas(SINGLE);
}
void morph_toggle(GtkWidget *w, gpointer data)
{
redraw_canvas(SINGLE);
}

/************************/
/* zoom factor callback */
/************************/
void render_zoom_changed(GtkWidget *w, struct model_pak *model)
{
gdouble zoom;

/* FIXME - for some reason model->zoom hasn't been updated */
/* at this point so we need to get the widget's value */
zoom = SPIN_FVAL(GTK_SPIN_BUTTON(w));

g_assert(zoom > 0.001);

model->scale = model->rmax/zoom;

init_objs(REDO_COORDS, model);

redraw_canvas(SINGLE);
}

/****************/
/* unified hook */
/****************/
gint event_render_modify(GtkWidget *w, gpointer *obj)
{
gint id, refresh=0;
const gchar *entry;
struct model_pak *data;

/* checks */
g_return_val_if_fail(obj != NULL, FALSE);

/* ascertain type of modification required */
id = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(obj), "id"));

data = model_ptr(sysenv.active, RECALL);

sysenv.moving = FALSE;

switch(id)
  {

  case STICK:
  case BALL_STICK:
  case LIQUORICE:
  case CPK:
    sysenv.render.type = id;

/* CURRENT - fine grained rendering modes */
/*
if (data->selection)
  set_selection_rendering(id, data);
else
  {
  sysenv.render.type = id;
  select_all();
  set_selection_rendering(-1, data);
  select_clear();
  }
*/

    refresh++;
    break;

  case ANIM_GIF:
    sysenv.render.animate_type = ANIM_GIF;
    break;
  case ANIM_MPEG:
    sysenv.render.animate_type = ANIM_MPEG;
    break;
  case ANIM_NAME:
    entry = gtk_entry_get_text(GTK_ENTRY(obj));
    g_free(sysenv.render.animate_file);
    sysenv.render.animate_file = g_strdup(entry);
    break;
  case LIGHT_TYPE:
    entry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(obj)->entry));
    if (g_ascii_strncasecmp(entry, "Positional", 10) == 0)
      current_light.type = POSITIONAL;
    else
      current_light.type = DIRECTIONAL;
    break;
  case MORPH_FINISH:
    entry = (gchar *) gtk_entry_get_text(GTK_ENTRY(obj));
    g_free(sysenv.render.morph_finish);
    sysenv.render.morph_finish = g_strdup(entry);
    refresh++;
    break;
  }

if (refresh)
  redraw_canvas(ALL);

return(FALSE);
}

void scale_set_default_values( GtkScale *scale )
{
gtk_range_set_update_policy(GTK_RANGE(scale), GTK_UPDATE_CONTINUOUS);
gtk_scale_set_digits (scale, 1);
gtk_scale_set_value_pos (scale, GTK_POS_TOP);
gtk_scale_set_draw_value (scale, TRUE);
}

/************************************/
/* toggle the render to file option */
/************************************/
void toggle_animate(GtkWidget *a_frame)
{
sysenv.render.animate ^= 1;

if (sysenv.render.animate)
  gtk_widget_set_sensitive(GTK_WIDGET(a_frame), TRUE);
else
  gtk_widget_set_sensitive(GTK_WIDGET(a_frame), FALSE);
}

void update_render_dialog(void)
{
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
g_return_if_fail(data != NULL);

/* update model dependent values */
/*
gtk_spin_button_set_value(GTK_SPIN_BUTTON(scale_spin), SCALE_MAG*data->rmax/data->scale);
*/
}

/***************************/
/* Setup POVray parameters */
/***************************/
void povray_hdr(FILE *fp, struct model_pak *data)
{
gdouble xvec, yvec, amb, cam[3], pos[3], colour[3];
GSList *list;
struct light_pak *light;

fprintf(fp,"#include \"colors.inc\" \n");
fprintf(fp,"#include \"finish.inc\" \n");
fprintf(fp,"#include \"glass.inc\" \n");
fprintf(fp,"#include \"metals.inc\" \n");
fprintf(fp,"#include \"textures.inc\" \n");

/* background colour (except for glass morphologies) */
fprintf(fp,"background { color rgb<%f,%f,%f0> }\n", sysenv.render.bg_colour[0],
                        sysenv.render.bg_colour[1], sysenv.render.bg_colour[2]);  

/* pixel to angstrom conversion, with yet another magic number... */
p2a = 0.565 * (gdouble) sysenv.render.width * data->scale / data->rmax;

/* setup camera position (z is just a value far enough away) */
VEC3SET(cam, data->offset[0]/p2a, data->offset[1]/p2a, 4.0*sysenv.rsize*data->scale);

/* width and height */
xvec = yvec = 2.0*sysenv.rsize/data->scale;

/* orientation */
fprintf(fp,"camera { orthographic location <%f,%f,%f>\n", cam[0], cam[1], cam[2]);
fprintf(fp,"    right <%f,0,0> up <0,%f,0>\n", xvec, yvec);
fprintf(fp,"    look_at <%f,%f,0> }\n", data->offset[0]/p2a, data->offset[1]/p2a);

/* attempt to mimic OpenGL lights */
for (list=sysenv.render.light_list ; list ; list=g_slist_next(list))
  {
  light = (struct light_pak *) list->data;
  ARR3SET(pos, light->x);

/* FIXME - I don't fully understand all the coordinate transforms */
/* necessary to make these positions match up with the on screen axes */
  switch (light->type)
    {
    case POSITIONAL:
      vecmat(acm, pos);
/* NB: y & z switch mimics gl_acm */
      fprintf(fp,"light_source\n  {\n <%f,%f,%f>\n", pos[0], -pos[2], pos[1]);
      break;

    case DIRECTIONAL:
      vecmat(acm, pos);
/* move away far enough so the rays are ~ // */
      VEC3MUL(pos, -100.0*data->rmax);
/* NB: y & z switch mimics gl_acm */
      fprintf(fp,"light_source\n  {\n <%f,%f,%f>\n", pos[0], -pos[2], pos[1]);
      break;

    default:
      continue;
    }

  ARR3SET(colour, light->colour);
  if (sysenv.render.shadowless)
    fprintf(fp,"  color rgb<%f,%f,%f> shadowless }\n", colour[0], colour[1], colour[2]);
  else
    fprintf(fp,"  color rgb<%f,%f,%f> }\n", colour[0], colour[1], colour[2]);
  }

/* morph is too dark with just the above, sky_sphere is *nice* */
/* TODO - leave choice of colour eg white/grey/light blue to the user? */
/* white can be a bit too bright (even with 0 ambience) */
if (data->id == MORPH && !sysenv.render.wire_surface)
  fprintf(fp,"sky_sphere { pigment {gradient y  color_map "
             "{[0, 1 color Gray20 color Gray80]} rotate x*45}}\n");

/* POVRay is a bit darker than OpenGL */
amb = 20.0*sysenv.render.ambience;

/* FIXME - adjustable gamma? */
fprintf(fp,"global_settings { ambient_light rgb<%f, %f, %f> assumed_gamma 2.2}\n",amb,amb,amb);
}

/*****************************/
/* task orientated rendering */
/*****************************/
void exec_pov_task(gpointer *ptr)
{
GString *cmd;

g_return_if_fail(ptr != NULL);
cmd = g_string_new(NULL);

/* build the command line */
g_string_sprintf(cmd, "%s +I%s -Ga -P +W%d +H%d +FT", 
                       sysenv.povray_path, (gchar *) ptr, 
                       (gint) sysenv.render.width, (gint) sysenv.render.height); 
if (sysenv.render.antialias)
  g_string_sprintfa(cmd, " +A +AM2");

task_sync(cmd->str);

g_string_free(cmd, TRUE);
}

/***************************/
/* task orientated viewing */
/***************************/
void exec_img_task(gpointer *ptr)
{
gchar *cmd, *basename;

/* post povray command */
g_return_if_fail(ptr != NULL);

/* remove .pov file */
unlink((gchar *) ptr);

/* execute the viewing command */
basename = strdup_basename((gchar *) ptr);
cmd = g_strdup_printf("%s %s.tga", sysenv.viewer_path, basename);
g_spawn_command_line_async(cmd, NULL);

g_free(basename);
g_free(cmd);
}

/*************************/
/* foreground rendering  */
/*************************/
void exec_pov(gchar *name)
{
GString *cmd;

cmd = g_string_new(NULL);

/* build the command line */
g_string_sprintf(cmd,"%s +I%s -GA -P +W%d +H%d +FT ",
                      sysenv.povray_path, name,
                      (gint) sysenv.render.width , (gint) sysenv.render.height);
if (sysenv.render.antialias)
  g_string_sprintfa(cmd,"+A +AM2 ");

/* after rendering delete input file, */
g_string_sprintfa(cmd,"; rm -rf %s ",name);

/* execute */
system(cmd->str);
printf("\n");
g_string_free(cmd, TRUE);
}

/************************/
/* background rendering */
/************************/
void run_pov()
{
gchar *basename;

/* make an input file */
basename = gun("pov");
make_pov(sysenv.active, basename);
submit_task("POVRay", &exec_pov_task, basename, &exec_img_task, basename, NULL);
}

/****************************/
/* make a POVRAY input file */
/****************************/
/* TODO - eliminate redundancy between this and the opengl code */
#define DEBUG_MKPOV 0
gint make_pov(gint model, gchar *povfile)
{
gint n, m, i, j;
gint r, g, b, flag;
gdouble rad, scale, len;
gdouble x1, y1, z1, x2, y2, z2;
gdouble xmin, ymin, zmin, xmax, ymax, zmax;
gdouble rf, gf, bf;
gdouble vec[3], vec1[3], vec2[3];
gdouble v1[4], v2[4];
gdouble ctrl[8][3], mat4[16], tmp4[16];
GSList *list, *ilist, *plist, *rlist, *list1, *list2, *pipe_list;
struct model_pak *data;
struct core_pak *core1;
struct plane_pak *plane;
struct object_pak *object;
struct ribbon_pak *ribbon;
struct spatial_pak *spatial;
struct pipe_pak *pipe;
struct image_pak *image;
struct vec_pak *p1;
struct vertex_pak *vx1, *vx2;
FILE *fp;

/* open file & init */
fp = fopen(povfile, "w");
if (!fp)
  {
  printf("Error, render(): can't open %s!\n", povfile);
  return(1);
  }

data = model_ptr(model, RECALL);
if (!data)
  return(2);

/* setup axes conversion matrix */
VEC3SET(&acm[0],-1.0, 0.0, 0.0);
VEC3SET(&acm[3], 0.0,-1.0, 0.0);
VEC3SET(&acm[6], 0.0, 0.0,-1.0);

/* strcpy(povfile,models[model].filename); */
/* file_extension -> .pov */

/* setup */
povray_hdr(fp,data);
/* limits - for intelligent axes placement */
xmin = ymin = zmin = 99999999.9;
xmax = ymax = zmax = -99999999.9;
/* pixel to coord scaling factor */
scale =  data->scale * sysenv.subscale * 0.5 / data->rmax;

/* precalc matrix products */
ARR3SET(&mat4[0], &data->latmat[0]);
ARR3SET(&mat4[4], &data->latmat[3]);
ARR3SET(&mat4[8], &data->latmat[6]);
ARR3SET(v1, data->centroid);
vecmat(data->latmat, v1);
mat4[3] = -v1[0];
mat4[7] = -v1[1];
mat4[11] = -v1[2];
VEC4SET(&mat4[12], 0.0, 0.0, 0.0, 1.0);

ARR3SET(&tmp4[0], &data->rotmat[0]);
ARR3SET(&tmp4[4], &data->rotmat[3]);
ARR3SET(&tmp4[8], &data->rotmat[6]);
tmp4[3] = 0.0;
tmp4[7] = 0.0;
tmp4[11] = 0.0;
VEC4SET(&tmp4[12], 0.0, 0.0, 0.0, 1.0);
mat4mat(tmp4, mat4);

/* do atoms */ 
#if DEBUG_MKPOV
printf("Doing atoms...\n");
#endif
/* enumerate periodic images */
plist=NULL;
do
  {
  if (plist)
    {
/* image */
    image = (struct image_pak *) plist->data;
    ARR3SET(vec, image->rx);
    plist = g_slist_next(plist);
    }
  else
    {
/* original */
    VEC3SET(vec, 0.0, 0.0, 0.0);
    plist = data->images;
    }

/* only if same type */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core1 = (struct core_pak *) list->data;
  if (core1->status & (DELETED | HIDDEN))
    continue;

  ARR3SET(vec1, core1->rx);
  ARR3ADD(vec1, vec);
  vecmat(acm, vec1);

/* get current colour */
    r = core1->colour[0];
    g = core1->colour[1];
    b = core1->colour[2];
/* convert to povray rgb format */
    rf = (gdouble) (r) / 65535.0;
    gf = (gdouble) (g) / 65535.0;
    bf = (gdouble) (b) / 65535.0;
    switch(sysenv.render.type)
      {
/* rounded end are now draw in the bond section */
      case LIQUORICE:
      case STICK:
        if (core1->bonds)
          break;
      case BALL_STICK:
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",vec1[0],vec1[1],vec1[2],
                                             sysenv.render.ball_rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);  
        fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                    sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
/*
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
*/

        break;
      case CPK:
        rad = elements[core1->atom_code].vdw;
        rad *= sysenv.render.cpk_scale;
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",vec1[0],vec1[1],vec1[2],rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);  
        fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                    sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
/*
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
*/
        break;
      }
  }
  }
while (plist);

#if DEBUG_MKPOV
printf("Doing bonds...\n");
#endif

/* TODO - ghost/line bonds (ie stage != 0) */
pipe_list = render_get_pipes(0, data);

/* enumerate the supplied pipes (half bonds) */
for (list=pipe_list ; list ; list=g_slist_next(list))
  {
  pipe = (struct pipe_pak *) list->data;

/* original + image iteration */
  ilist = NULL;
  do
    {
/* original */
    ARR3SET(v1, pipe->v1);
    ARR3SET(v2, pipe->v2);
    if (ilist)
      {
      image = (struct image_pak *) ilist->data;
/* image */
      ARR3ADD(v1, image->rx);
      ARR3ADD(v2, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    vecmat(acm, v1);
    vecmat(acm, v2);

    fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
              , v1[0],v1[1],v1[2],v2[0],v2[1],v2[2], sysenv.render.stick_rad);

    fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n"
              , pipe->colour[0], pipe->colour[1], pipe->colour[2]);

    fprintf(fp,"  finish{phong %f phong_size %d }}}\n"
              , sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
    }
  while (ilist);
  }
free_slist(pipe_list);

#if DEBUG_MKPOV
printf("Doing special objects...\n");
#endif

/********/
/* AXES */
/********/
/* FIXME - can't cope with model translation */
if (data->show_axes)
  {
/* origin */
  VEC3SET(vec1, 1.0, 1.0, 0.0);
  VEC3MUL(vec1, data->rmax/data->scale);
/* cope with scaling */
  vec1[2] -= 4.0*sysenv.rsize*data->scale;
/* thickness */
  rad = 0.001 / data->scale;
  for (j=0 ; j<3 ; j++)
    {
/* get end point */
    ARR3SET(vec2, data->axes[j].rx);
    VEC3MUL(vec2, 1.0/data->scale);
    vecmat(acm, vec2);
    ARR3ADD(vec2, vec1);
/* cope with scaling */
    vec2[2] -= 4.0*sysenv.rsize*data->scale;
/* draw */
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2], rad);
    fprintf(fp," texture { pigment {White} } }\n");
    }
  }

/********/
/* CELL */
/********/
if (data->show_cell && data->periodic)
  {
  rad = sysenv.render.frame_thickness * data->scale / 70.0;

/* ends */
  for (j=0 ; j<8 ; j++)
    {
    m = 2*(j/2) + 1;
    n = 4*(j/4);

    ARR3SET(vec, data->cell[m].rx);
    vecmat(acm, vec);
    x1 = vec[0];
    y1 = vec[1];
    z1 = vec[2];

    ARR3SET(vec, data->cell[n].rx);
    vecmat(acm, vec);
    x2 = vec[0];
    y2 = vec[1];
    z2 = vec[2];
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                                 x1,y1,z1,x2,y2,z2,rad);
    fprintf(fp," texture { pigment {White} } }\n");

    ARR3SET(vec, data->cell[n+2].rx);
    vecmat(acm, vec);
    x2 = vec[0];
    y2 = vec[1];
    z2 = vec[2];
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                                 x1,y1,z1,x2,y2,z2,rad);
    fprintf(fp," texture { pigment {White} } }\n");
    }

/* sides */
/* skip for 2D periodic models */
  if (data->periodic == 3)
    {
    m = 0;
    n = 4;
    for (j=0 ; j<4 ; j++)
      {
      ARR3SET(vec, data->cell[m].rx);
      vecmat(acm, vec);
      x1 = vec[0];
      y1 = vec[1];
      z1 = vec[2];

      ARR3SET(vec, data->cell[n].rx);
      vecmat(acm, vec);
      x2 = vec[0];
      y2 = vec[1];
      z2 = vec[2];

      fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                                   x1,y1,z1,x2,y2,z2,rad);
      fprintf(fp," texture { pigment {White} } }\n");
      m++;
      n++;
      }
    }
  }

/***********/
/* ribbons */
/***********/
for (list=data->ribbons ; list ; list=g_slist_next(list))
  {
  object = (struct object_pak *) list->data;
  g_assert(object->type == RIBBON);

  rlist = (GSList *) object->data;
  while (rlist)
    {

    ribbon = (struct ribbon_pak *) rlist->data;

  fprintf(fp, "bicubic_patch {\n  type 1 flatness 0.001\n");
/* NB: POVRay is very slow at rendering ribbons, so halve the quality */
  fprintf(fp, "  u_steps %d  v_steps 2\n", (gint) (sysenv.render.ribbon_quality/2.0));

/* end points */
    ARR3SET(&ctrl[0][0], ribbon->r1);
    ARR3SET(&ctrl[3][0], ribbon->r2);

/* get distance between ribbon points */
    ARR3SET(vec1, ribbon->x1);
    ARR3SUB(vec1, ribbon->x2);
    len = VEC3MAG(vec1);

/* shape control points */
    ARR3SET(&ctrl[1][0], ribbon->r1);
    ARR3SET(&ctrl[2][0], ribbon->r2);

/* segment length based curvature - controls how flat it is at the cyclic group */
    ARR3SET(vec1, ribbon->o1);
    VEC3MUL(vec1, len*sysenv.render.ribbon_curvature);
    ARR3ADD(&ctrl[1][0], vec1);
    ARR3SET(vec2, ribbon->o2);
    VEC3MUL(vec2, len*sysenv.render.ribbon_curvature);
    ARR3ADD(&ctrl[2][0], vec2);

/* compute offsets for ribbon thickness */
    crossprod(vec1, ribbon->n1, ribbon->o1);
    crossprod(vec2, ribbon->n2, ribbon->o2);
    normalize(vec1, 3);
    normalize(vec2, 3);

/* thickness vectors for the two ribbon endpoints */
    VEC3MUL(vec1, 0.5*sysenv.render.ribbon_thickness);
    VEC3MUL(vec2, 0.5*sysenv.render.ribbon_thickness);

/* ensure these are pointing the same way */
    if (via(vec1, vec2, 3) > PI/2.0)
      {
      VEC3MUL(vec2, -1.0);
      }

/* init the bottom edge control points */
    ARR3SET(&ctrl[4][0], &ctrl[0][0]);
    ARR3SET(&ctrl[5][0], &ctrl[1][0]);
    ARR3SET(&ctrl[6][0], &ctrl[2][0]);
    ARR3SET(&ctrl[7][0], &ctrl[3][0]);
/* lift points to make the top edge */
    ARR3ADD(&ctrl[0][0], vec1);
    ARR3ADD(&ctrl[1][0], vec1);
    ARR3ADD(&ctrl[2][0], vec2);
    ARR3ADD(&ctrl[3][0], vec2);
/* lower points to make the bottom edge */
    ARR3SUB(&ctrl[4][0], vec1);
    ARR3SUB(&ctrl[5][0], vec1);
    ARR3SUB(&ctrl[6][0], vec2);
    ARR3SUB(&ctrl[7][0], vec2);

    for (i=0 ; i<8 ; i++)
      {
      vecmat(acm, &ctrl[i][0]);
      VEC3MUL(&ctrl[i][0], data->scale);
      }

    fprintf(fp, "<%f, %f, %f>\n", ctrl[0][0], ctrl[0][1], ctrl[0][2]);
    for (i=1 ; i<4 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);
    for (i=0 ; i<4 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);
    for (i=4 ; i<8 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);
    for (i=4 ; i<8 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);

  fprintf(fp," texture { pigment { color rgbt<%f, %f, %f, %f> } } no_shadow }\n",
                                             sysenv.render.ribbon_colour[0],
                                             sysenv.render.ribbon_colour[1],
                                             sysenv.render.ribbon_colour[2],
                                             1.0-sysenv.render.transmit);
    rlist = g_slist_next(rlist);
    }
  }

/******************/
/* spatial planes */
/******************/
for (list=data->spatial ; list ; list=g_slist_next(list))
  {
  spatial = (struct spatial_pak *) list->data;

  switch(spatial->type)
    {
/* special cases */
    case SPATIAL_VECTOR:
      break;
    case SPATIAL_PLANE:
      break;

/* general cases */
    default:
/* enumerate periodic images */
      plist=NULL;
      do
        {
        if (plist)
          {
/* image */
          image = (struct image_pak *) plist->data;
          ARR3SET(vec2, image->rx);
          plist = g_slist_next(plist);
          }
        else
          {
/* original */
          VEC3SET(vec2, 0.0, 0.0, 0.0);
          plist = data->images;
          }
/* enumerate vertices */
        rlist = (GSList *) spatial->data;
        p1 = NULL;
/* experimental */
/*
        if (g_slist_length(rlist) == 3)
*/
        if (g_slist_length(rlist) == 99)
          {
/* triangle case */
          fprintf(fp, "smooth_triangle { \n");
          while (rlist)
            {
            p1 = (struct vec_pak *) rlist->data;
            ARR3SET(vec1, vec2);
            ARR3ADD(vec1, p1->rx);
            vecmat(acm, vec1);

/* supply normal at the vertex as well */
            ARR3SET(vec, p1->n);
            vecmat(acm, vec);

            fprintf(fp, "< %f, %f, %f>, <%f, %f, %f> ",
                         vec1[0], vec1[1], vec1[2], vec[0], vec[1], vec[2]);

            rlist = g_slist_next(rlist);
  
            if (rlist)
              fprintf(fp, ",\n");
            else
              fprintf(fp, "\n");
            }
          }
        else
          {
/* general case */
          fprintf(fp, "polygon { %d\n", g_slist_length(rlist));
          while (rlist)
            {
            p1 = (struct vec_pak *) rlist->data;
            ARR3SET(vec1, vec2);
            ARR3ADD(vec1, p1->rx);
            vecmat(acm, vec1);
            fprintf(fp, "< %f, %f, %f>", vec1[0], vec1[1], vec1[2]);
  
            rlist = g_slist_next(rlist);
  
            if (rlist)
              fprintf(fp, ",\n");
            else
              fprintf(fp, "\n");
            }
          }

/* FIXME - can we have individual colours at each vertex? */
        g_assert(p1 != NULL);
        fprintf(fp, " texture { \n");
        fprintf(fp, " pigment { color rgbt<%f, %f, %f, %f> } }\n",
                                               p1->colour[0],
                                               p1->colour[1],
                                               p1->colour[2],
                                      1.0-sysenv.render.transmit);

/* NB: POVRay highlight size has reverse sense. */
        fprintf(fp, " finish { phong %f phong_size %d } }\n",
          sysenv.render.shl_strength, (gint) sysenv.render.shl_size);

/*
        fprintf(fp, " finish { specular %d roughness %d } }\n",
          sysenv.render.shl_strength, sysenv.render.shl_size);
*/

        }
      while (plist);
      break;
    }
  }

/**************/
/* MORPHOLOGY */
/**************/
if (data->num_vertices)
  {
/* morphology rendering style */
  if (!sysenv.render.wire_surface)
    {
/* render via plane intersection */
    fprintf(fp,"object\n{\nintersection{\n");

    plist = data->planes;
    while (plist != NULL)
      {
      plane = (struct plane_pak *) plist->data;
      if (plane->present)
        {
/* rotate cartesian normals to match viewing angle */
        ARR3SET(vec, plane->norm);
        vecmat(data->rotmat, vec);
/* apply axes conversion matrix */
        vecmat(acm, vec);

/* NEW - distance to facet */
switch(data->morph_type)
  {
  case EQUIL_UN:
    rad = plane->esurf[0];
    break;
  case GROWTH_UN:
    rad = plane->eatt[0];
    break;
  case EQUIL_RE:
    rad = plane->esurf[1];
    break;
  case GROWTH_RE:
    rad = plane->eatt[1];
    break;
  default:
  case DHKL:
    rad = 1.0/plane->dhkl;
    break;
  }

        fprintf(fp,"plane { <%f, %f, %f>,  %f }\n",vec[0],vec[1],vec[2], rad);
        }
      plist = g_slist_next(plist);
      }
/* crystal composition */
    fprintf(fp,"} interior { ior %f } texture { finish { %s } "
               " pigment { color rgbf <%f,%f,%f,%f> } }\n }\n", 
               sysenv.render.ref_index,
               sysenv.render.morph_finish,
               sysenv.render.morph_colour[0],
               sysenv.render.morph_colour[1],
               sysenv.render.morph_colour[2],
               1.0-sysenv.render.transmit);  
    }
  else
    {
    rad = sysenv.render.frame_thickness * data->rmax / 90.0;
    for (list1=data->vertices ; list1 ; list1=g_slist_next(list1))
      {
      vx1 = (struct vertex_pak *) list1->data;

      for (list2=vx1->adj ; list2 ; list2=g_slist_next(list2))
        {
        vx2 = (struct vertex_pak *) list2->data;

        if (!ridge_visible(vx1, vx2, data))
          continue;

        ARR3SET(vec, vx1->rx);
        vecmat(acm, vec);
        x1 = vec[0];
        y1 = vec[1];
        z1 = vec[2];

        ARR3SET(vec, vx2->rx);
        vecmat(acm, vec);
        x2 = vec[0];
        y2 = vec[1];
        z2 = vec[2];

        flag=0;
        if (fabs(x2-x1) < 0.001)
          flag++;
        if (fabs(y2-y1) < 0.001)
          flag++;
        if (fabs(z2-z1) < 0.001)
          flag++;
        if (flag==3)
          continue;

        fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>,%f\n",
                                  x1,y1,z1,x2,y2,z2,rad);
        fprintf(fp," texture { pigment { color rgb<%f %f %f>} } }\n",
                                          sysenv.render.morph_colour[0],
                                          sysenv.render.morph_colour[1],
                                          sysenv.render.morph_colour[2]);
        }
      }
    }
  }

/* finished making the .pov file */
fclose(fp);
return(0);
}

/*******************************/
/* callbacks for label changes */
/*******************************/
void geom_label_toggle(void)
{
redraw_canvas(SINGLE);
}

void update_geom_line_width(GtkWidget *w, gpointer *ptr)
{
sysenv.render.geom_line_width = GTK_ADJUSTMENT(w)->value;
redraw_canvas(SINGLE);
}

/***********************/
/* main rendering page */
/***********************/
void render_main_page(GtkWidget *box, struct model_pak *data)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;
GtkWidget *button, *spin;
GSList *group=NULL;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* left pane */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add (GTK_CONTAINER(frame),vbox);

/* do the first radio button */
button = gtk_radio_button_new_with_label (NULL, "Stick");
g_signal_connect(GTK_OBJECT(button), "pressed",
                 GTK_SIGNAL_FUNC (event_render_modify), (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
g_object_set_data(G_OBJECT(button), "id", (gpointer) STICK);
gtk_widget_show(button);
if (sysenv.render.type == STICK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the rest of the buttons */
button = gtk_radio_button_new_with_label(group, "Ball & stick");
g_signal_connect(GTK_OBJECT(button), "pressed",
                 GTK_SIGNAL_FUNC(event_render_modify), (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
g_object_set_data(G_OBJECT(button), "id", (gpointer) BALL_STICK);
gtk_widget_show(button);
if (sysenv.render.type == BALL_STICK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* next button */
button = gtk_radio_button_new_with_label(
                      gtk_radio_button_group(GTK_RADIO_BUTTON(button)),
                      "Cylinder");
g_signal_connect(GTK_OBJECT(button), "pressed",
                 GTK_SIGNAL_FUNC(event_render_modify), (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
g_object_set_data(G_OBJECT(button), "id", (gpointer) LIQUORICE);
gtk_widget_show(button);
if (sysenv.render.type == LIQUORICE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* last button */
button = gtk_radio_button_new_with_label(
                      gtk_radio_button_group(GTK_RADIO_BUTTON(button)),
                      "CPK");
g_signal_connect(GTK_OBJECT(button), "pressed",
                 GTK_SIGNAL_FUNC (event_render_modify), (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
g_object_set_data(G_OBJECT(button), "id", (gpointer) CPK);
gtk_widget_show(button);
if (sysenv.render.type == CPK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* extra toggles */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

gtksh_direct_check("Antialias", &sysenv.render.antialias,
                   render_refresh, NULL, vbox);
gtksh_direct_check("Wire frame model", &sysenv.render.wire_model,
                   render_refresh, NULL, vbox);
gtksh_direct_check("Wire frame surfaces", &sysenv.render.wire_surface,
                   render_refresh, NULL, vbox);
gtksh_direct_check("Hidden lines", &sysenv.render.wire_show_hidden,
                   render_refresh, NULL, vbox);

/* zoom frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

spin = gtksh_auto_spin("Zoom factor", &data->zoom, 0.1, 500.0, 1.0,
                render_zoom_changed, data, vbox);
gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 2);

/* surface transmission frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
gtksh_direct_spin("Surface opacity",
                  &sysenv.render.transmit, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

/* radii frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);

vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_spin("Ball radius",
                  &sysenv.render.ball_rad, 0.1, 0.5, 0.02,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Cylinder radius",
                  &sysenv.render.stick_rad, 0.02, 0.5, 0.01,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Stick thickness",
                  &sysenv.render.stick_thickness, 0.1, 5.0, 0.1,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Frame radius",
                  &sysenv.render.frame_thickness, 0.1, 5.0, 0.1,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("CPK scaling",
                  &sysenv.render.cpk_scale, 0.1, 3.0, 0.1,
                  render_refresh, NULL, vbox);

/* highlighting frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 5);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_spin("Atom highlight power", &sysenv.render.ahl_strength, 0.0, 1.0, 0.1,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Atom highlight focus", &sysenv.render.ahl_size, 0.0, MAX_HL, 5.0,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Surface highlight power", &sysenv.render.shl_strength, 0.0, 1.0, 0.1,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Surface highlight focus", &sysenv.render.shl_size, 0.0, MAX_HL, 5.0,
                   render_refresh, NULL, vbox);

/* ribbon control frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_spin("Ribbon curvature control",
                  &sysenv.render.ribbon_curvature, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Ribbon thickness",
                  &sysenv.render.ribbon_thickness, 0.4, 6.0, 0.2,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Ribbon quality",
                  &sysenv.render.ribbon_quality, 1.0, 30.0, 1.0,
                  render_refresh, NULL, vbox);
}

/************************************/
/* colouring method change callback */
/************************************/
void set_colour_scheme(GtkWidget *w, gpointer *dummy)
{
const gchar *tmp;
struct model_pak *model;

tmp = gtk_entry_get_text(GTK_ENTRY(w));

model = model_ptr(sysenv.active, RECALL);
if (!model)
  return;

/* default to atom selection */
if (g_ascii_strncasecmp(tmp, "element", 7) == 0)
  model_colour_scheme(ELEM, model);

if (g_ascii_strncasecmp(tmp, "molecule", 8) == 0)
  model_colour_scheme(MOL, model);

if (g_ascii_strncasecmp(tmp, "region", 6) == 0)
  model_colour_scheme(REGION, model);

redraw_canvas(SINGLE);
}

/**************************/
/* colour control options */
/**************************/
void render_colours_section(GtkWidget *box)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame, *combo;
GtkWidget *label, *button;
GtkStyle *style;
GList *list=NULL;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);


/* TODO - simplify by having a combo of names, and a single colour editing box */
/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* background colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Edit background colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.bg_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.bg_colour[0]*65535.0;
style->bg[0].green = sysenv.render.bg_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.bg_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);

/* crystal colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Edit crystal colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.morph_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.morph_colour[0]*65535.0;
style->bg[0].green = sysenv.render.morph_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.morph_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);

/* re-entrant colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Edit re-entrant colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.rsurf_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.rsurf_colour[0]*65535.0;
style->bg[0].green = sysenv.render.rsurf_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.rsurf_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);

/* ribbon colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Edit ribbon colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.ribbon_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.ribbon_colour[0]*65535.0;
style->bg[0].green = sysenv.render.ribbon_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.ribbon_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);


/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* NEW - colouring scheme */
/* TODO - radio button */
list = NULL;
list = g_list_append(list, "element");
list = g_list_append(list, "molecule");
list = g_list_append(list, "region");

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

label = gtk_label_new ("Colouring scheme ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_end(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(set_colour_scheme), NULL);
}

/**********************/
/* light list globals */
/**********************/
enum 
{
RENDER_LIGHT_COLOUR,
RENDER_LIGHT_VECTOR,
RENDER_LIGHT_TYPE,
RENDER_LIGHT_ATTRIBUTE,
RENDER_LIGHT_NCOLS
};

GtkWidget *render_light_tv;
GtkListStore *render_light_ls;


/****************************************************/
/* convert 3 doubles [0:1] into a hex colour string */
/****************************************************/
gchar *get_hex_colour(gdouble *colour)
{
guint r, g, b;
gdouble rgb[3];

ARR3SET(rgb, colour);
VEC3MUL(rgb, 255.0);

r = rgb[0];
g = rgb[1];
b = rgb[2];

return(g_strdup_printf("#%2X%2X%2X", r, g, b));
}

/**********************************/
/* get the currently selected row */
/**********************************/
gint render_light_selected(void)
{
gint row=0;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
GtkTreeIter iter;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(render_light_tv));
selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(render_light_tv));

if (gtk_tree_model_get_iter_first(treemodel, &iter))
  {
  do
    {
    if (gtk_tree_selection_iter_is_selected(selection, &iter))
      return(row);
    row++;
    }
  while (gtk_tree_model_iter_next(treemodel, &iter));
  }
return(-1);
}

/***********************************/
/* construct the light source list */
/***********************************/
void update_light_list(void)
{
gint row, num_rows=0;
gchar *text;
GSList *list;
GtkTreeIter iter;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct light_pak *light;

/* checks */
g_assert(render_light_ls != NULL);

/* store */
row = render_light_selected();

/* re-populate */
gtk_list_store_clear(render_light_ls);
for (list=sysenv.render.light_list ; list ; list=g_slist_next(list))
  {
  light = (struct light_pak *) list->data;
  gtk_list_store_append(render_light_ls, &iter);

/* position or direction vector */
  text = g_strdup_printf(" (%5.1f, %5.1f, %5.1f) ", light->x[0], light->x[1], light->x[2]);
  gtk_list_store_set(render_light_ls, &iter, RENDER_LIGHT_VECTOR, text, -1);

/* type */
  if (light->type == DIRECTIONAL)
    text = g_strdup(" Directional");
  else
    text = g_strdup(" Positional");
  gtk_list_store_set(render_light_ls, &iter, RENDER_LIGHT_TYPE, text, -1);

/* FIXME - easier way to specify the RGB colour? */
  text = get_hex_colour(light->colour);

  gtk_list_store_set(render_light_ls, &iter, RENDER_LIGHT_ATTRIBUTE, text, -1);
  num_rows++;
  }

/* restore selected row, or the previous (if possible) if deleted  */
if (row >= num_rows && row)
  row--;
if (row >= 0)
  {
  treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(render_light_tv));
  if (gtk_tree_model_iter_nth_child(treemodel, &iter, NULL, row))
    {
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(render_light_tv)); 
    if (selection)
      gtk_tree_selection_select_iter(selection, &iter);
    }
  }
}

/***********************************************************/
/* modify light position to match current rotational frame */
/***********************************************************/
void mod_light_pos(gdouble *x, struct model_pak *model)
{
gdouble mat[9];

g_assert(model != NULL);

/* apply current rotational reference */
vecmat(model->rotmat, x);
/* apply the inverse of the initial rotmat (ie get "pure" rotation) */
/* NB: initial rotmat is NOT the identity */
init_rotmat(mat);
invmat(mat);
vecmat(mat, x);
}

/*******************************************/
/* light list manipulation, add new source */
/*******************************************/
void add_current_light(GtkWidget *w, gpointer dummy)
{
struct light_pak *light;
struct model_pak *model;

/* duplicate data for the list */
light = g_malloc(sizeof(struct light_pak));
memcpy(light, &current_light, sizeof(struct light_pak));

/* NEW */
model = model_ptr(sysenv.active, RECALL);
if (model)
  mod_light_pos(light->x, model);

/* append & update */
sysenv.render.light_list = g_slist_append(sysenv.render.light_list, light);
update_light_list();

redraw_canvas(SINGLE);
}

/******************************************/
/* light list manipulation, modify source */
/******************************************/
void mod_selected_light(GtkWidget *w, gpointer dummy)
{
gint row;
struct light_pak *light;
struct model_pak *model;

row = render_light_selected();
if (row < 0)
  return;

/* get the light's data from the list */
light = (struct light_pak *) g_slist_nth_data(sysenv.render.light_list, row);

/* overwrite with the current data */
memcpy(light, &current_light, sizeof(struct light_pak));

/* NEW */
model = model_ptr(sysenv.active, RECALL);
if (model)
  mod_light_pos(light->x, model);

update_light_list();

redraw_canvas(SINGLE);
}

/******************************************/
/* light list manipulation, delete source */
/******************************************/
void del_selected_light(GtkWidget *w, gpointer dummy)
{
gint row;
struct light_pak *light;

row = render_light_selected();
if (row < 0)
  return;

light = (struct light_pak *) g_slist_nth_data(sysenv.render.light_list, row);

sysenv.render.light_list = g_slist_remove(sysenv.render.light_list, light);

update_light_list();

redraw_canvas(SINGLE);
}

/******************/
/* render cleanup */
/******************/
void render_cleanup(void)
{
gtk_list_store_clear(render_light_ls);
render_light_ls = NULL;
render_light_tv = NULL;
}

/*********************************************/
/* callback to fill dialog with light values */
/*********************************************/
void render_light_activate(GtkTreeView *treeview, GtkTreePath *treepath)
{
gint n;
struct light_pak *light;

/* get selected light */
n = render_light_selected();
if (n < 0)
  return;

/* get the light's data from the list */
light = g_slist_nth_data(sysenv.render.light_list, n);

/* transfer to current displayed values */
if (light)
  {
/* overwrite */
  memcpy(&current_light, light, sizeof(struct light_pak));
/* update */
  gtksh_relation_update(NULL);
  }
}

/****************************/
/* lighting control options */
/****************************/
void render_lighting_section(GtkWidget *box)
{
gint i;
gchar *titles[] = {" Colour ", "  Absolute vector  ", " Type "};
GList *list=NULL;
GtkWidget *swin, *vbox1, *vbox2, *vbox, *hbox, *frame;
GtkWidget *label, *combo, *button;
GtkStyle *style1;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;

/* NB: for the coloured boxes we need to copy a style from a realized widget! */

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* box 1 - colour (and edit button) */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Current light source colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* light colour */
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), current_light.colour);
style1 = gtk_style_copy(gtk_widget_get_style(window));
style1->bg[0].red   = current_light.colour[0] * 65535;
style1->bg[0].green = current_light.colour[1] * 65535; 
style1->bg[0].blue  = current_light.colour[2] * 65535; 
gtk_widget_set_style(GTK_WIDGET(button), style1);


/* box 2 - light components */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);

gtksh_direct_spin("Ambient component", &current_light.ambient,
                  0.0, 1.0, 0.1, NULL, NULL, vbox);

gtksh_direct_spin("Diffuse component", &current_light.diffuse,
                  0.0, 1.0, 0.1, NULL, NULL, vbox);

gtksh_direct_spin("Specular component", &current_light.specular,
                  0.0, 1.0, 0.1, NULL, NULL, vbox);


/* left box - light type */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* type label */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
label = gtk_label_new("  Type  ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
/* TODO - pulldown dir/pos */
/* NEW - combo box for the selection mode */
list = g_list_append(list, "Directional");
list = g_list_append(list, "Positional");
combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(event_render_modify), (gpointer) combo);
g_object_set_data(G_OBJECT(combo), "id", (gpointer) LIGHT_TYPE);


/* position/direction vector input */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

/* x component */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
label = gtk_label_new("X");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

gtksh_direct_spin(NULL, &current_light.x[0],
                  -1000.0, 1000.0, 0.1, NULL, NULL, vbox);

/* y component */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
label = gtk_label_new("Y");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

gtksh_direct_spin(NULL, &current_light.x[1],
                  -1000.0, 1000.0, 0.1, NULL, NULL, vbox);

/* z component */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
label = gtk_label_new("Z");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

gtksh_direct_spin(NULL, &current_light.x[2],
                  -1000.0, 1000.0, 0.1, NULL, NULL, vbox);


/* next frame - actions */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
gtksh_button(" Add ", add_current_light, NULL, hbox, TT);
gtksh_button(" Modify ", mod_selected_light, NULL, hbox, TT);
gtksh_button(" Delete ", del_selected_light, NULL, hbox, TT);

/* right box 1 - light source listing */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, TRUE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* scrolled win for the list */
swin = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), swin, TRUE, TRUE, 0);
gtk_widget_set_size_request(swin, 24*sysenv.gfontsize, -1);

/* NEW - light list in a list store */
render_light_ls = gtk_list_store_new(RENDER_LIGHT_NCOLS, 
                                     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
render_light_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(render_light_ls));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), render_light_tv);
for (i=0 ; i<=RENDER_LIGHT_TYPE ; i++)
  {
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(titles[i], renderer, "text", i, NULL);

if (!i)
  gtk_tree_view_column_add_attribute(column, renderer, "background", RENDER_LIGHT_ATTRIBUTE);

  gtk_tree_view_append_column(GTK_TREE_VIEW(render_light_tv), column);
  }

g_signal_connect(render_light_tv, "row_activated",
                 G_CALLBACK(render_light_activate), NULL);
}

/***************************/
/* OpenGL specific options */
/***************************/
void render_opengl_section(GtkWidget *box, struct model_pak *model)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* perspective projection */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_check("Perspective projection", &sysenv.render.perspective,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Vanishing point",
                  &sysenv.render.vp_dist, 1.0, 100.0, 1.0,
                  render_refresh, NULL, vbox);

/* depth queuing */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_check("Depth queueing", &sysenv.render.fog,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Depth queueing start",
                  &sysenv.render.fog_start, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Depth queueing strength",
                  &sysenv.render.fog_density, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

/* new frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* sphere quality */
gtksh_direct_spin("Sphere quality",
                  &sysenv.render.sphere_quality, 0.0, 4.0, 1.0,
                  render_refresh, NULL, vbox);

/* cylinder quality */
gtksh_direct_spin("Cylinder quality",
                  &sysenv.render.cylinder_quality, 4.0, 20.0, 1.0,
                  render_refresh, NULL, vbox);

/* quality */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_check("Fast rotation", &sysenv.render.fast_rotation,
                   NULL, NULL, vbox);
gtksh_direct_check("Selection halos", &sysenv.render.halos,
                   render_refresh, NULL, vbox);

/* new frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

gtksh_auto_check("Label geometric measurements", geom_label_toggle, NULL,
                                          &model->show_geom_labels, hbox);

gtksh_direct_spin("Line width",
                  &sysenv.render.geom_line_width, 0.5, 9.0, 0.5,
                  render_refresh, NULL, vbox);
}

/***************************/
/* POVRay specific options */
/***************************/
void render_povray_section(GtkWidget *box)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame, *a_frame;
GtkWidget *label, *button, *file_entry, *entry;
GSList *group=NULL;
struct model_pak *data;

/* TODO - allow render dialog even when no models, but dont draw */
/* or make insensitive model specific buttons */
data = model_ptr(sysenv.active, RECALL);

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* new frame */
frame = gtk_frame_new("Image size");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_spin("width",
                  &sysenv.render.width, 100, 2000, 100,
                  NULL, NULL, vbox);

gtksh_direct_spin("height",
                  &sysenv.render.height, 100, 2000, 100,
                  NULL, NULL, vbox);

/* new frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

gtksh_direct_check("Shadowless", &sysenv.render.shadowless, NULL, NULL, vbox);

/* animted movie options */
if (data)
if (data->animation)
  {
  frame = gtk_frame_new(NULL);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, TRUE, TRUE, 0);

/* check block - save to file only */
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  a_frame = gtk_vbox_new(TRUE, 0);
  gtk_box_pack_end(GTK_BOX(vbox), a_frame, TRUE, TRUE, 0);

  new_check_button("Rendered animation", toggle_animate, a_frame,
                                    sysenv.render.animate, vbox);


  vbox = gtk_vbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(a_frame), vbox, TRUE, TRUE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* start off with a button */
  button = gtk_radio_button_new_with_label (NULL, "Animated GIF");
  g_signal_connect(GTK_OBJECT(button), "clicked",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) button);
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, TRUE, 0);
  g_object_set_data(G_OBJECT(button), "id", (gpointer) ANIM_GIF);

/* make a radio group */
  group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the rest of the buttons */
  button = gtk_radio_button_new_with_label(group, "MPEG");
  g_signal_connect(GTK_OBJECT(button), "clicked",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) button);
  gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, TRUE, 0);
  g_object_set_data(G_OBJECT(button), "id", (gpointer) ANIM_MPEG);

/* delay */
gtksh_direct_spin("Delay (ms)",
                  &sysenv.render.delay, 0, 100, 5,
                  NULL, NULL, vbox);

/* file entry */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
  label = gtk_label_new("Filename ");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
  file_entry = gtk_entry_new_with_max_length(FILELEN-1);
  gtk_box_pack_end(GTK_BOX (hbox), file_entry, FALSE, TRUE, 0);
  gtk_entry_set_text(GTK_ENTRY(file_entry), sysenv.render.animate_file);

/* update hook */
  g_signal_connect(GTK_OBJECT(file_entry), "changed",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) file_entry);
  g_object_set_data(G_OBJECT(file_entry), "id", (gpointer) ANIM_NAME);

/* set initial state */
  if (!sysenv.render.animate)
    gtk_widget_set_sensitive(a_frame, FALSE);
  }

/* new frame */
frame = gtk_frame_new("Glass morphology");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 5);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* refractive index */
gtksh_direct_spin("Refractive index",
                  &sysenv.render.ref_index, 1.0, 3.0, 0.1,
                  NULL, NULL, vbox);

/* texture - glassy, metallic, etc. */
hbox = gtk_hbox_new (FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
label = gtk_label_new("Surface finish ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
entry = gtk_entry_new_with_max_length(LINELEN-1);
gtk_box_pack_end(GTK_BOX(hbox), entry, FALSE, TRUE, 0);

/* this causes GTK to complain (open , change ribbon colour, close, open */
/* but I cannot figure out why */
/*
gtk_entry_set_text(GTK_ENTRY(entry), sysenv.render.morph_finish);
*/

/* update hook */
g_signal_connect(GTK_OBJECT(entry), "activate",
                 GTK_SIGNAL_FUNC(event_render_modify), (gpointer) entry);
g_object_set_data(G_OBJECT(entry), "id", (gpointer) MORPH_FINISH);

/* run POVRay job in the background */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_end(GTK_BOX(box), hbox, TRUE, FALSE, PANEL_SPACING);
gtksh_button("    Render    ", render_setup, NULL, hbox, TF);
}

/*****************/
/* misc. options */
/*****************/
void render_misc_section(GtkWidget *box)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;
struct model_pak *model;

/* TODO - allow render dialog even when no models, but dont draw */
/* or make insensitive model specific buttons */
model = model_ptr(sysenv.active, RECALL);

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* left pane */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* relation check buttons */
gtksh_auto_check("show surface energy", render_refresh, NULL, &model->show_energy, vbox);
gtksh_auto_check("show charges", render_refresh, NULL, &model->show_charges, vbox);
gtksh_auto_check("show atom labels", render_refresh, NULL, &model->show_atom_labels, vbox);
gtksh_auto_check("show cores", render_refresh, NULL, &model->show_cores, vbox);
gtksh_auto_check("show shells", render_refresh, NULL, &model->show_shells, vbox);
gtksh_auto_check("show axes", render_refresh, NULL, &model->show_axes, vbox);
gtksh_auto_check("show cell", render_refresh, NULL, &model->show_cell, vbox);
gtksh_auto_check("show cell images", render_refresh, NULL, &model->show_cell_images, vbox);
gtksh_auto_check("show cell lengths", render_refresh, NULL, &model->show_cell_lengths, vbox);

/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* change font options */
gtksh_button_x("Change text font", gfont_dialog, NULL, vbox);
gtksh_button_x("Change graphics font", dfont_dialog, NULL, vbox);
gtksh_button_x("Toggle axes type", toggle_axes_type, NULL, vbox);

/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_auto_check("Colour using SOF ", cs_toggle, NULL, &model->sof_colourize, vbox);
gtksh_auto_check("Show all surfaces ", morph_toggle, NULL, &model->hpr, vbox);
gtksh_auto_check("Labelled morphology ", morph_toggle, NULL, &model->morph_label, vbox);
}

/***********************/
/* render setup widget */
/***********************/
void render_dialog()
{
gint id;
GtkWidget *frame;
GtkWidget *label, *notebook, *page;
struct dialog_pak *rsd;
struct light_pak *ldata;
struct model_pak *model;

/* NB: some check boxes depend on having a valid model */
model = model_ptr(sysenv.active, RECALL);
if (!model)
  return;

/* request dialog slot */
if ((id = request_dialog(sysenv.active, POVRAY)) < 0)
  return;
rsd = &sysenv.dialog[id];
rsd->win = gtk_dialog_new();
g_signal_connect(GTK_OBJECT(rsd->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
gtk_window_set_title(GTK_WINDOW(rsd->win), "Display properties");

/* init current light */
ldata = (struct light_pak *) ((GSList *) sysenv.render.light_list)->data;

if (ldata)
  memcpy(&current_light, ldata, sizeof(struct light_pak));
else
  {
  show_text(ERROR, "Empty light list.\n");
  return;
  }

/* notebook frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(rsd->win)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(frame), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);

/* main page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Main");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_main_page(page, model);

/* colour page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Colours");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_colours_section(page);

/* lighting page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Lights");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_lighting_section(page);

/* OpenGL page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("OpenGL");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_opengl_section(page, model);

/* POVRay page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("POVRay");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_povray_section(page);

/* misc page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Misc");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_misc_section(page);

/* terminating button */
gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(rsd->win)->action_area);

/* display the dialog */
gtk_widget_show_all(rsd->win);

update_light_list();
}
