/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <unistd.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "parse.h"
#include "task.h"
#include "morph.h"
#include "sginfo.h"
#include "matrix.h"
#include "space.h"
#include "surface.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"

/* main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

enum
{
SURF_TITLE,
SURF_REGIONS,
SURF_ESURF_UNRE,
SURF_EATT_UNRE,
SURF_ESURF_RE,
SURF_EATT_RE,
SURF_DIPOLE,
SURF_GNORM,
SURF_MODEL,
SURF_PLANE,
SURF_SHIFT,
SURF_NCOLS
};

/* globals */
GtkTreeStore *surf_tree_store;
GtkWidget *surf_tree_view;
GtkWidget *surf_morph_energy[4];
gint all_planes = FALSE;
gdouble rank_value = 10.0;
struct model_pak surfdata;
gchar *titles[] = {"hkl/shift", "Dhkl/depth",
                   "Esurf (u)", "Eatt (u)", "Esurf (r)", "Eatt (r)",
                   "dipole  ", "gnorm  "};

/**************************************************/
/* free data pointers assoc with a child iterator */
/**************************************************/
void surf_free_child(GtkTreeIter *iter)
{
gchar *text;
GtkTreeModel *treemodel;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;

gtk_tree_model_get(treemodel, iter, SURF_TITLE, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_REGIONS, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_ESURF_UNRE, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_EATT_UNRE, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_ESURF_RE, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_EATT_RE, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_DIPOLE, &text, -1);
if (text)
  g_free(text);
gtk_tree_model_get(treemodel, iter, SURF_GNORM, &text, -1);
if (text)
  g_free(text);
}

/***************************************************/
/* free data pointers assoc with a parent iterator */
/***************************************************/
void surf_free_parent(GtkTreeIter *parent)
{
GtkTreeIter child;
GtkTreeModel *treemodel;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;

surf_free_child(parent);
if (gtk_tree_model_iter_children(treemodel, &child, parent))
  {
  do
    {
    surf_free_child(&child);
    }
  while (gtk_tree_model_iter_next(treemodel, &child));
  }
}

/*********************************************/
/* free all data pointers in the entire list */
/*********************************************/
void surf_free_all(void)
{
GtkTreeIter parent;
GtkTreeModel *treemodel;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;

if (gtk_tree_model_get_iter_first(treemodel, &parent))
  {
  do
    {
    surf_free_parent(&parent);
    }
  while (gtk_tree_model_iter_next(treemodel, &parent));
  }
}

/*
#include "sysabs.h"
*/

/********************************************/
/* allocate for new plane & init for safety */
/********************************************/
/* NB: assumes the plane doesn't exist (use fn_seek_plane() to check) */
#define DEBUG_CREATE_PLANE 0
struct plane_pak *fn_make_plane(gdouble *m, struct model_pak *model)
{
gint h, k, l, n;
gdouble vec[3];
struct plane_pak *plane=NULL;

/* checks */
g_return_val_if_fail(model != NULL, NULL);
/*
g_return_val_if_fail(model->sginfo.raw != NULL, NULL);
*/

/* alloc */
plane = g_malloc(sizeof(struct plane_pak));
if (!plane)
  return(NULL);

#if DEBUG_CREATE_PLANE
printf("creating: %d %d %d -> ", (gint) m[0], (gint) m[1], (gint) m[2]);
#endif

/* save orig */
h = m[0];
k = m[1];
l = m[2];
n = 2;

/* NB: check for my new absence testing function */
/*
printf("%d %d %d : [%d ?= %d]\n", h, k, l,
IsSysAbsent_hkl((T_SgInfo *) model->sginfo.raw, h, k , l, NULL),
surf_sysabs(model, h, k, l));
*/

/* check for reflection planes */
/*
while (IsSysAbsent_hkl((T_SgInfo *) model->sginfo.raw, h, k , l, NULL) && n<100)
*/

/* my absence testing function */
while (surf_sysabs(model, h, k, l) && n<100)
  {
  h = n*m[0];
  k = n*m[1];
  l = n*m[2];
  n++;
  }

if (n == 100)
  {
  printf("WARNING: screwed up systematic absence check.\n");
  h /= 99;
  k /= 99;
  l /= 99;
  n = 2;
  }

#if DEBUG_CREATE_PLANE
printf("%d %d %d  (x %d)\n", h, k, l, n-1);
#endif

/* init */
VEC3SET(plane->m, (gdouble) h, (gdouble) k, (gdouble) l);
VEC3SET(plane->norm, (gdouble) h, (gdouble) k, (gdouble) l);
VEC3SET(plane->index, h, k, l);
plane->shifts = NULL;
plane->vertices = NULL;

/* calc Dhkl */
VEC3SET(vec, h, k, l);
vecmat(model->rlatmat, vec);
plane->dhkl = 1.0/VEC3MAG(vec);
plane->esurf[0] = 0.0;
plane->esurf[1] = 0.0;
plane->eatt[0] = 0.0;
plane->eatt[1] = 0.0;
plane->f[0] = 0.0;
plane->f[1] = 0.0;
/* calc? */
plane->multiplicity = 1;
plane->present = TRUE;
plane->visible = FALSE;
plane->primary = TRUE;
VEC3SET(plane->rx, 0.0, 0.0, 0.0);

return(plane);
}

/************************************************************/
/* determine if a particular plane has already been created */
/************************************************************/
struct plane_pak *fn_seek_plane(gdouble *hkl, struct model_pak *model)
{
GSList *list;
struct plane_pak *comp, *plane;

/* get corrected (sys absent!) miller index */
comp = fn_make_plane(hkl, model);
g_return_val_if_fail(comp != NULL, NULL);

for (list=model->planes ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;
  if (facet_equiv(model, comp->index, plane->index))
    return(plane);
  }

g_free(comp);
return(NULL);
}

/***************************/
/* attempt to match shifts */
/***************************/
gint compare_shifts(gconstpointer *s1, gconstpointer *s2)
{
struct shift_pak *s1data;
struct shift_pak *s2data;
s1data = (struct shift_pak *) s1;
s2data = (struct shift_pak *) s2;

/* seek mismatches & return failure */
if (s1data->shift != s2data->shift)
  return(1);
if (s1data->region[0] != s2data->region[0])
  return(1);
if (s1data->region[1] != s2data->region[1])
  return(1);

/* otherwise, return succesful match */
return(0);
}

/********************************************/
/* allocate for new shift & init for safety */
/********************************************/
struct shift_pak *create_shift(gdouble shift)
{
struct shift_pak *sdata;

/* alloc */
sdata = g_malloc(sizeof(struct shift_pak));
if (!sdata)
  return(NULL);

sdata->shift = shift;
sdata->dipole_computed = FALSE;
sdata->dipole = 99.9;
sdata->region[0] = 1;
sdata->region[1] = 1;
sdata->esurf[0] = 0.0;
sdata->esurf[1] = 0.0;
sdata->eatt[0] = 0.0;
sdata->eatt[1] = 0.0;
sdata->gnorm = -1.0;
sdata->procfile = NULL;

return(sdata);
}

/*********************************************************/
/* compute net charge and dipole (if model is a surface) */
/*********************************************************/
/* TODO - a bit more general now - put somewhere else */
/* TODO - number argument - the order of the multipole expansion */
#define DEBUG_CALC_EMP 0
void calc_emp(struct model_pak *data)
{
gdouble zdip, qsum, x[3];
GSList *list;
struct elem_pak elem;
struct core_pak *core;

/* TODO - speed up by getting to a local var the elem data for unique atoms */
g_assert(data != NULL);

/*
printf("calc_emp(%p)\n", data);
dump_cores(data->cores);
*/

/* cores & shell dipole calc */
zdip=qsum=0.0;
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
/* only for region 1 */
  if (core->region != REGION1A)
    continue;
  if (core->status & DELETED)
    continue;

/*
#if DEBUG_CALC_EMP
printf("core: %s, q: %f [%d]\n", core->label, core->charge, core->lookup_charge);
if (core->shell)
  {
printf("shell: %s, q: %f [%d]\n", (core->shell)->label, (core->shell)->charge, (core->shell)->lookup_charge);
  }
#endif
*/

/* core contribution */
/*
  if (core->lookup_charge)
    get_elem_data(core->atom_code, &elem, data);
  else
*/
    elem.charge = core->charge;

/* get cartesian z */
  ARR3SET(x, core->x);
  vecmat(data->latmat, x);

/*
#if DEBUG_CALC_EMP
printf("c: [z,q] = [%f, %f]\n", x[2], elem.charge);
#endif
*/

  zdip += elem.charge * x[2];
  qsum += elem.charge;

/* NB: add shell constribution at the core's z location */
  if (core->shell)
    {
/*
    if ((core->shell)->lookup_charge)
      get_elem_data(core->atom_code, &elem, data);
    else
*/
      elem.shell_charge = (core->shell)->charge;

/* get cartesian z */
    ARR3SET(x, (core->shell)->x);
    vecmat(data->latmat, x);

/*
#if DEBUG_CALC_EMP
printf("s: [z,q] = [%f, %f]\n", x[2], elem.shell_charge);
#endif
*/

    zdip += elem.shell_charge * x[2];
    qsum += elem.shell_charge;
    }
  }

#if DEBUG_CALC_EMP
printf("surface: %f %f %f  (%f)\n", 
data->surface.miller[0], data->surface.miller[1], data->surface.miller[2], 
data->surface.shift);
printf("net dipole: %e\n", zdip);
printf("sum charge: %e\n", qsum);
if (qsum*qsum > FRACTION_TOLERANCE)
  printf("Warning: your model has a net charge = %f\n", qsum);
#endif

/* only surfaces get the z dipole */
if (data->periodic == 2)
  data->gulp.sdipole = zdip;

data->gulp.qsum = qsum;
}

/*************************************************/
/* set values for a shift in the treeview widget */
/*************************************************/
void fn_surf_set_shift(GtkTreeIter *iter, struct shift_pak *shift)
{
gint i;
gchar *text;

for (i=SURF_TITLE ; i<=SURF_GNORM ; i++)
  {
  switch(i)
    {
    case SURF_REGIONS:
      text = g_strdup_printf("%2d:%2d", shift->region[0], shift->region[1]);
      break;

    case SURF_EATT_UNRE:
      text = g_strdup_printf("%9.4f", shift->eatt[0]);
      break;

    case SURF_ESURF_UNRE:
      text = g_strdup_printf("%9.4f", shift->esurf[0]);
      break;

    case SURF_EATT_RE:
      text = g_strdup_printf("%9.4f", shift->eatt[1]);
      break;

    case SURF_ESURF_RE:
      text = g_strdup_printf("%9.4f", shift->esurf[1]);
      break;

    case SURF_DIPOLE:
      if (shift->dipole_computed)
        text = g_strdup_printf("%9.4f", shift->dipole);
      else
        text = g_strdup("?");
      break;

    case SURF_GNORM:
      if (shift->gnorm < 0.0)
        text = g_strdup_printf("(unknown)");
      else
        text = g_strdup_printf("%9.4f", shift->gnorm);
      break;

    default:
      continue;
    }
/*
printf("[%d] : %s\n", i, text);
*/
  gtk_tree_store_set(surf_tree_store, iter, i, text, -1);
  }
}

/***********************************************/
/* search and update a shift's treeview values */
/***********************************************/
void fn_surf_refresh_shift(struct shift_pak *shift)
{
}

/***********************************************/
/* search and update a plane's treeview values */
/***********************************************/
void fn_surf_refresh_plane(struct plane_pak *plane)
{
}

/******************************************************/
/* update all shift/plane data in the treeview widget */
/******************************************************/
#define DEBUG_SURF_REFRESH_ALL 0
void fn_surf_refresh_all(void)
{
gint n;
GtkTreeModel *treemodel;
GtkTreeIter parent, child;
struct shift_pak *shift;
struct model_pak *model;

/* checks */
if (!dialog_active(GENSURF))
  return;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;

/* are there branches on the tree? */
if (gtk_tree_model_get_iter_first(treemodel, &parent))
  {
/* loop over all parents */
  do
    {
/* loop over all children */
    n=0;
    while (gtk_tree_model_iter_nth_child(treemodel, &child, &parent, n))
      {
/* check if plane is already in the store */
      gtk_tree_model_get(treemodel, &child, SURF_MODEL, &model, -1);
      gtk_tree_model_get(treemodel, &child, SURF_SHIFT, &shift, -1);
      g_assert(model != NULL);
      g_assert(shift != NULL);

#if DEBUG_SURF_REFRESH_ALL
printf("updating shift: %p\n", shift);
#endif

      fn_surf_set_shift(&child, shift);
      n++;
      }
    }
  while (gtk_tree_model_iter_next(treemodel, &parent)); 
  }
}

/****************************************************/
/* acquire a parent iterator for the supplied plane */
/****************************************************/
#define DEBUG_ACQUIRE_PLANE_ITER 0
gint fn_acquire_plane_iterator(GtkTreeIter *iter, struct plane_pak *plane)
{
GtkTreeModel *treemodel;
struct plane_pak *comp;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return(1);

/* are there branches on the tree? */
if (gtk_tree_model_get_iter_first(treemodel, iter))
  {
#if DEBUG_ACQUIRE_PLANE_ITER
printf("searching for: %p\n", plane);
#endif

/* search all parent iterators */
  do
    {
/* check if plane is already in the store */
    gtk_tree_model_get(treemodel, iter, SURF_PLANE, &comp, -1);

#if DEBUG_ACQUIRE_PLANE_ITER
printf("comp: %p\n", comp);
#endif

    if (comp == plane)
      return(0);
    }
  while (gtk_tree_model_iter_next(treemodel, iter)); 
  }

/* no match found - acquire new parent iterator */
gtk_tree_store_append(surf_tree_store, iter, NULL);
return(0);
}

/*********************************/
/* add a shift to the tree store */
/*********************************/
void fn_graft_shift(GtkTreeIter *parent, struct shift_pak *shift, 
                                         struct plane_pak *plane,
                                         struct model_pak *model)
{
gchar *text;
GtkTreeIter child;

g_assert(shift != NULL);

/* set child iterator data */
gtk_tree_store_append(surf_tree_store, &child, parent);
gtk_tree_store_set(surf_tree_store, &child,
                   SURF_MODEL, model, SURF_PLANE, plane, SURF_SHIFT, shift, -1);

/* set shift title */
text = g_strdup_printf("%6.4f", shift->shift);
gtk_tree_store_set(surf_tree_store, &child, SURF_TITLE, text, -1);

/* set shift dipole */
if (shift->dipole_computed)
  text = g_strdup_printf("%9.4f", shift->dipole);
else
  text = g_strdup_printf(" ? ");

gtk_tree_store_set(surf_tree_store, &child, SURF_DIPOLE, text, -1);

/* set shift values */
fn_surf_set_shift(&child, shift);
}

/******************************************************/
/* remove all shifts listed in the tree under a plane */
/******************************************************/
void fn_prune_shifts(struct plane_pak *plane)
{
gint n;
GtkTreeIter child, parent;
GtkTreeModel *treemodel;

g_assert(plane != NULL);

/* acquire parent level iterator */
if (fn_acquire_plane_iterator(&parent, plane))
  return;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;

n = gtk_tree_model_iter_n_children(treemodel, &parent) - 1;
while (n >= 0)
  {
  if (gtk_tree_model_iter_nth_child(treemodel, &child, &parent, n))
    {
    surf_free_child(&child);
    gtk_tree_store_remove(surf_tree_store, &child);
    }
  n--;
  }
}

/******************************************/
/* add a list of shifts to the tree store */
/******************************************/
void fn_graft_shift_list(struct plane_pak *plane, struct model_pak *model)
{
GSList *list;
GtkTreeIter parent;
struct shift_pak *shift;

g_assert(plane != NULL);

/* acquire parent level iterator */
if (fn_acquire_plane_iterator(&parent, plane))
  {
  printf("Error setting up iterator.\n");
  return;
  }

/* loop over all shifts */
for (list=plane->shifts ; list ; list=g_slist_next(list))
  {
  shift = (struct shift_pak *) list->data;

  fn_graft_shift(&parent, shift, plane, model);
  }
}

/*********************************/
/* add a plane to the tree store */
/*********************************/
void fn_graft_plane(struct plane_pak *plane, struct model_pak *model)
{
gchar *text;
GtkTreeIter root;
GtkTreeModel *treemodel;
GtkTreePath *treepath;

g_assert(model != NULL);
g_assert(plane != NULL);

/* acquire parent level iterator */
if (fn_acquire_plane_iterator(&root, plane))
  {
  printf("Error setting up iterator.\n");
  return;
  }

/* set the parent iterator data */
text = g_strdup_printf("(%d%d%d)", plane->index[0], plane->index[1], plane->index[2]);
gtk_tree_store_set(surf_tree_store, &root, SURF_TITLE, text, -1);
text = g_strdup_printf("%f", plane->dhkl);
gtk_tree_store_set(surf_tree_store, &root, SURF_REGIONS, text, -1);
gtk_tree_store_set(surf_tree_store, &root,
  SURF_MODEL, model, SURF_PLANE, plane, SURF_SHIFT, NULL, -1);

/*
printf("Adding model: %p, plane: %p\n", model, plane);
*/

fn_graft_shift_list(plane, model);

/* get treepath of root */
treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (treemodel)
  {
  treepath = gtk_tree_model_get_path(treemodel, &root);
  gtk_tree_view_expand_row(GTK_TREE_VIEW(surf_tree_view), treepath, TRUE);
  }
}

/******************************************/
/* add a list of planes to the tree store */
/******************************************/
void fn_graft_plane_list(GSList *plist, struct model_pak *model)
{
GSList *list;
struct plane_pak *plane;

/* go through the plane list */
for (list=plist ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;
  if (plane->primary)
    fn_graft_plane(plane, model);
  }
}

/*********************/
/* add shift to list */
/*********************/
gint fn_add_plane_with_shift(void)
{
GtkTreeIter iter;
struct model_pak *model, surf;
struct plane_pak *plane;
struct shift_pak *shift;

/* checks */
model = model_ptr(surfdata.surface.model, RECALL);
if (!model)
  return(FALSE);
if (model->id == MORPH)
  return(FALSE);

/*
P3VEC("Adding: ", surfdata.surface.miller);
*/

/* create new shift with current region values */
shift = create_shift(surfdata.surface.shift);
shift->region[0] = surfdata.surface.region[0];
shift->region[1] = surfdata.surface.region[1];

/* init for dipole calculation */
template_model(&surf);
surf.surface.shift = shift->shift;
surf.surface.region[0] = 1;
surf.surface.region[1] = 0;

/* does the plane already exist? */
plane = fn_seek_plane(surfdata.surface.miller, model);
if (plane)
  {
ARR3SET(surf.surface.miller, plane->index);
generate_surface(model, &surf);
shift->dipole = surf.gulp.sdipole;
shift->dipole_computed = TRUE;

  plane->shifts = g_slist_append(plane->shifts, shift);

/* add to shift to treestore */
  if (!fn_acquire_plane_iterator(&iter, plane))
    fn_graft_shift(&iter, shift, plane, model);
  else
    show_text(ERROR, "fn_add_plane_with_shift() : can't acquire plane iterator.\n");
  }
else
  {
/* create new plane with current hkl */
  plane = fn_make_plane(surfdata.surface.miller, model);
  if (plane)
    { 
ARR3SET(surf.surface.miller, plane->index);
generate_surface(model, &surf);
shift->dipole = surf.gulp.sdipole;
shift->dipole_computed = TRUE;

    plane->shifts = g_slist_append(plane->shifts, shift);

/* append new plane to model */
    model->planes = g_slist_append(model->planes, plane);

/* add to whole plane to treestore */
    fn_graft_plane(plane, model);
    }
  }

free_model(&surf);

return(TRUE);
}

/**********************************************/
/* search for valid shifts - output to a file */
/**********************************************/
#define DEBUG_ADD_VALID_SHIFTS 0
gint fn_add_valid_shifts(void)
{
struct model_pak *model;
struct surface_pak *surfdat = &surfdata.surface;
struct plane_pak *plane;

/* checks */
model = model_ptr(surfdat->model, RECALL);
g_return_val_if_fail(model != NULL, 1);
if (model->periodic != 3)
  {
  show_text(ERROR, "Base model is not 3D periodic.\n");
  return(1);
  }
if (!surfdata.surface.miller[0] && !surfdata.surface.miller[1] && !surfdata.surface.miller[2])
  {
  show_text(ERROR, "Don't be silly.\n");
  return(2);
  }

#if DEBUG_ADD_VALID_SHIFTS
printf("Generating surface for model: %d\n", surfdat->model);
#endif

/* does the plane already exist? */
plane = fn_seek_plane(surfdata.surface.miller, model);
if (plane)
  fn_prune_shifts(plane);
else
  {
  plane = fn_make_plane(surfdata.surface.miller, model);
/* add to list */
  if (plane)
    model->planes = g_slist_append(model->planes, plane);
  else
    return(3);
  }

/* find valid shifts for the plane */
/* NB: this REPLACES existing shifts */
calc_valid_shifts(model, plane);

/* update treestore */
fn_graft_plane(plane, model);

return(FALSE);
}
/********************************************************/
/* scan a plane's shift list for the best energy values */
/********************************************************/
void update_plane_energy(struct plane_pak *pdata, struct model_pak *data)
{
gint i;
GSList *list;
struct plane_pak *plane;
struct shift_pak *sdata;

/* checks */
g_assert(data != NULL);
g_assert(pdata != NULL);

/* set best values to those of the 1st shift */
list = pdata->shifts;
if (list)
  {
  sdata = (struct shift_pak *) list->data;

/* init the planes best values */
  for (i=0 ; i<2 ; i++)
    {
    pdata->esurf[i] = sdata->esurf[i];
    pdata->eatt[i] = sdata->eatt[i];
    }
  }
else
  return;

/* search the shift list for better values */
while (list != NULL)
  {
  sdata = (struct shift_pak *) list->data;
  for (i=0 ; i<2 ; i++)
    {
/* surface energy */
    if (sdata->esurf[i] < pdata->esurf[i])
      pdata->esurf[i] = sdata->esurf[i];
/* attachment energy */
    if (sdata->eatt[i] > pdata->eatt[i])
      pdata->eatt[i] = sdata->eatt[i];
    }
  list = g_slist_next(list);
  }

/* update symmetry equivalent planes also */
for (list=data->planes ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;
  if (plane == pdata)
    continue;
  if (facet_equiv(data, pdata->index, plane->index))
    {
    for (i=0 ; i<2 ; i++)
      {
      plane->esurf[i] = pdata->esurf[i];
      plane->eatt[i] = pdata->eatt[i];
      }
    }
  }
}

/*************************/
/* construct the surface */
/*************************/
#define MAKE_SURFACE_DEBUG 0
void make_surface(gint source, struct plane_pak *pdata, struct shift_pak *sdata)
{
gint bmode, new, r1size;
gdouble sbe;
GSList *list;
struct model_pak *data, *surf;
struct core_pak *core;

/* checks */
data = model_ptr(source, RECALL);
g_return_if_fail(data != NULL);
if (data->periodic != 3)
  {
  show_text(ERROR, "base model is not 3D periodic.\n");
  return;
  }
if (!pdata->index[0] && !pdata->index[1] && !pdata->index[2])
  {
  show_text(ERROR, "Don't be silly.\n");
  return;
  }

#if MAKE_SURFACE_DEBUG
printf("Generating surface for model: %d\n", source);
#endif
/* save old mode */
bmode = data->build_molecules;
/* ignore mols when building surface? */
if (data->surface.ignore_bonding)
  {
  data->build_molecules = FALSE;
  calc_bonds(data);
  calc_mols(data);
  }

/* allocate & init for surface data */
new = sysenv.num_models;
surf = model_ptr(new, ASSIGN);
g_return_if_fail(surf != NULL);
/* NEW - label it as MARVIN, so it's build mode follows the */
/* source model, rather than the GULP setup data - see prep_model() */
surf->id = MARVIN;

/* transfer appropriate GULP setup data to new model */
copy_gulp_data(data, surf);
copy_elem_data(data, surf);

surf->gulp.run = E_SINGLE;
surf->gulp.method = CONV;

/* transfer surface data to new model */
/* NB: memcpy would produce undesired effects if pointers were */
/* later added to the surface struct */
ARR3SET(surf->surface.miller, pdata->index);
surf->surface.shift = sdata->shift;
surf->surface.region[0] = sdata->region[0];
surf->surface.region[1] = sdata->region[1];
/* setup for region display */
if (surf->surface.region[0])
  surf->region_empty[REGION1A] = FALSE;
if (surf->surface.region[1])
  surf->region_empty[REGION2A] = FALSE;

surf->build_molecules = bmode;

#if MAKE_SURFACE_DEBUG
printf("Miller (%d,%d,%d)\n",surf->surface.miller[0]
                            ,surf->surface.miller[1]
                            ,surf->surface.miller[2]);
printf("Shift %f\n",surf->surface.shift);
printf("Region sizes (%d,%d)\n",surf->surface.region[0]
                               ,surf->surface.region[1]);
#endif

/* TODO - valid shift/region size loop??? */
generate_surface(data, surf);
prep_model(surf);

/* restore connectivity in source model */
if (data->surface.ignore_bonding)
  {
  data->build_molecules = bmode;
  calc_bonds(data);
  calc_mols(data);
  }

/* calculate the bulk energy needed for GULP surface calcs */
sbe = data->gulp.energy;
if (fabs(sbe) < FRACTION_TOLERANCE)
  {
  show_text(WARNING, "Suspicious total energy. Has it been properly calculated?\n");
  }

/* calc the number of region 1 atoms */
r1size=0;
for (list=surf->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->region == REGION1A)
    r1size++;
  }

sbe /= data->num_atoms;    /* E per atom in unit cell */
sbe *= r1size;             /* E scaled to region 1 size */
surf->gulp.sbulkenergy = sbe;

/* employ src file? */
new_tree_item(surf->number, APPEND);
/*
tree_select(surf->number);
*/
redraw_canvas(ALL);
}

/*************/
/* rank cuts */
/*************/
gint zsort(gdouble *z1, gdouble *z2)
{
if (*z1 > *z2)
  return(1);

if (*z2 > *z1)
  return(-1);

return(0);
}

/****************/
/* general sort */
/****************/
void sort(gdouble *array, gint *order, gint size)
{
gint i, swap;
gdouble tmp;

for (i=0 ; i<size ; i++)
  *(order+i) = i;

swap=1;
while (swap)
  {
  swap=0;
  for (i=1 ; i<size ; i++)
    {
    if (array[i-1] < array[i])
      {
/* swap elements in array */
      tmp = array[i-1];
      array[i-1] = array[i];
      array[i] = tmp;
/* register the change in order */
      tmp = order[i-1];
      order[i-1] = order[i];
      order[i] = tmp;
/* elements were swapped */
      swap++;
      }
    }
  }
}

/***************************************/
/* smallest common multiple for a pair */
/***************************************/
gint scf(gint a, gint b)
{
if (!b)
  return(abs(a));
return(scf(b, a%b));
}

/***************************************/
/* smallest common multiple for vector */
/***************************************/
gint simplify(gint *vec, gint size)
{
gint i, n;

n = scf(*vec, *(vec+1));  
for (i=2 ; i<size ; i++)
  n = scf(n, *(vec+i));  
return(n);
}

/*************************************/
/* construct a particular morphology */
/*************************************/
void make_morph(void)
{
gint status;
gchar *filename;
GSList *plist;
struct model_pak *data;
struct plane_pak *pdata;

/* get the current model */
data = model_ptr(surfdata.surface.model, RECALL);
if (!data)
  return;

/* create a suitable name */
filename = g_strdup_printf("%s.gmf", data->basename);

/* go through all planes */
plist = data->planes;
while (plist != NULL)
  {
  pdata = (struct plane_pak *) plist->data;
/* get the best energy from the shift list */
  update_plane_energy(pdata, data);
  plist = g_slist_next(plist);
  }

/* store the planes */
write_gmf(filename, data);

/* read morphology file in & compute hull */
data = model_ptr(sysenv.num_models, ASSIGN);
/* default morphology computation type */
data->morph_type = DHKL;

/* only successful if properly loaded */
status = 2;
if (data)
  {
  status--;
  if (!read_gmf(filename, data))
    status--;
  }

if (!status)
  {
  new_tree_item(data->number, APPEND);
/*
  tree_select(data->number);
*/
  }

g_free(filename);
}
/* TODO - rewrite to generate specified number of faces (rank_spin) */
/* based on Dhkl ranking  - maybe! - calc valid shifts for each */

gint dhkl_compare(gpointer ptr1, gpointer ptr2)
{
struct plane_pak *plane1=ptr1, *plane2=ptr2;

if (plane1->dhkl > plane2->dhkl)
  return(-1);
if (plane1->dhkl < plane2->dhkl)
  return(1);

return(0);
}

/* TODO - use this to remove all the while(list!=NULL) loops */
/*
void g_slist_foreach(GSList *list, GFunc func, gpointer user_data);

Calls a function for each element of a GSList.
               list : a GSList.
               func : the function to call with each element's data.
        user_data : user data to pass to the function. 

GSList* g_slist_insert_sorted(GSList *list, gpointer data, GCompareFunc func);

Inserts a new element into the list, using the given comparison function to determine its position.

               list : a GSList.
               data : the data for the new element.
               func : the function to compare elements in the list.
                      It should return a number > 0 if the first parameter comes
                      after the second parameter in the sort order.
            Returns : the new start of the GSList. 
*/

/******************************************************/
/* adds ranked faces (from data) to the supplied list */
/******************************************************/
/* if num -> get num ranked faces, else get until Dhkl < min */
#define DEBUG_GET_RANKED_FACES 0
GSList *get_ranked_faces(gint num, gdouble min, struct model_pak *data)
{
gint n, h, k, l, limit;
gint f1[3], f2[3];
gdouble m[3];
GSList *list=NULL, *plist=NULL, *list1=NULL, *list2=NULL;
struct plane_pak *pdata;

/* loop over a large range */
/* FIXME - should loop until num/max is satisfied */
limit = 4;
for (h=limit ; h>=-limit ; h--)
  {
  for (k=limit ; k>=-limit ; k--)
    {
    for (l=limit ; l>=-limit ; l--)
      {
/* skip (000) */
      if (!h && !k && !l)
        continue;
      VEC3SET(f1, h, k, l);
/* skip things with a common multiple > 1, since */
/* fn_make_plane() takes care of systematic absences */
/* NB: this is ok for morph pred. but for XRD we may want such planes */
if (num)
  {
      n = simplify(f1, 3);
      if (n != 1)
        continue;
  }

/* add the plane */
      ARR3SET(m, f1);
      pdata = fn_make_plane(m, data);
      if (pdata)
        plist = g_slist_prepend(plist, pdata);
      }
    }
  }
plist = g_slist_reverse(plist);

/* sort via Dhkl */
plist = g_slist_sort(plist, (gpointer) dhkl_compare);

/*
VEC3SET(f1, 1, 0, -4);
printf("f1 : [%d %d %d]\n", f1[0], f1[1], f1[2]);
n = simplify(f1, 3);
printf("n = %d\n", n);
*/

/* mark the planes that are symmetry related */
list1 = plist;
while (list1 != NULL)
  {
/* get a reference (primary) plane */
  pdata = (struct plane_pak *) list1->data;
  if (pdata->primary)
    {
    ARR3SET(f1, pdata->index);
    }
  else
    {
    list1 = g_slist_next(list1);
    continue;
    }
/* scan for symmetry related planes */
  list2 = g_slist_next(list1);
  while (list2 != NULL)
    {
    pdata = (struct plane_pak *) list2->data;
    ARR3SET(f2, pdata->index);
/* get the hkl's and test */
#if DEBUG_GET_RANKED_FACES
printf("testing: %d %d %d  &  %d %d %d\n", f1[0], f1[1], f1[2], f2[0], f2[1], f2[2]);
#endif
    if (facet_equiv(data,f1,f2))
      pdata->primary = FALSE;

    list2 = g_slist_next(list2);
    }
  list1 = g_slist_next(list1);
  }

/* write the planes list to the model */
/* write only up to the requested number */
list1 = plist;
n=0;
while (list1 != NULL)
  {
/* get a reference (primary) plane */
  pdata = (struct plane_pak *) list1->data;
  if (pdata->primary)
    {
/* NEW - only add valid shifts */
    calc_valid_shifts(data, pdata);

/* default single shift */
/*
    sdata = create_shift(0.0);
    pdata->shifts = g_slist_append(pdata->shifts, sdata);
*/

/* add plane (don't prepend - there may be others already there) */
    list = g_slist_prepend(list, pdata);
    n++;
    if (num)
      {
      if (n == num)
        break;
      }
    else
      {
      if (pdata->dhkl < min)
        break;
      }
    }
  list1 = g_slist_next(list1);
  }
list = g_slist_reverse(list);
return(list);
}

/**********************/
/* GULP debugging aid */
/**********************/
void gulp_dump(struct model_pak *data)
{
printf("  num atoms = %d\n", data->num_atoms);
printf("        sbe = %f\n", data->gulp.sbulkenergy);
printf("     energy = %f\n", data->gulp.energy);
printf("E surf (un) = %f\n", data->gulp.esurf[0]);
printf("E surf (re) = %f\n", data->gulp.esurf[1]);
printf(" E att (un) = %f\n", data->gulp.eatt[0]);
printf(" E att (re) = %f\n", data->gulp.eatt[1]);
}

/**************************************************/
/* eliminate repeated z values in a (sorted) list */
/**************************************************/
GSList *trim_zlist(GSList *zlist)
{
gdouble dz, *z1, *z2;
GSList *list, *rlist;

/* get 1st item */
list = zlist;
if (list)
  {
  z1 = (gdouble *) list->data;
  list = g_slist_next(list);
  }
else
  return(NULL);

/* compare and eliminate repeats */
rlist = zlist;
while (list)
  {
  z2 = (gdouble *) list->data;
  list = g_slist_next(list);

  dz = fabs(*z1 - *z2);
  if (dz < FRACTION_TOLERANCE)
    {
    rlist = g_slist_remove(rlist, z2);
    g_free(z2);
    }
  else
    z1 = z2;
  }
return(rlist);
}

/*********************************************/
/* append a list of valid shifts for a plane */
/*********************************************/
#define DEBUG_CALC_SHIFTS 0
gint calc_valid_shifts(struct model_pak *data, struct plane_pak *pdata)
{
gint build, num_cuts, num_shifts, dummy[3];
gdouble gcd, zlim, z0, *z1, *z2, vec[3];
GSList *slist, *list, *zlist;
struct model_pak *surf;
struct shift_pak *sdata;
struct core_pak *core;
struct mol_pak *mol;

/* TODO - don't use model_ptr, rather just allocate & template */
/* OR get free_model to remove it from the model list */
/* allocate & init for surface */
/*
surf = model_ptr(sysenv.num_models, ASSIGN);
*/

surf = g_malloc(sizeof(struct model_pak));
template_model(surf);

surf->mode = FREE;
surf->id = GULP;
surf->periodic = 2;

/* actual surface we want constructed */
ARR3SET(surf->surface.miller, pdata->index);
surf->surface.shift = 0.0;
surf->surface.region[0] = 1.0;
surf->surface.region[1] = 0.0;

gcd = GCD(pdata->index[0], GCD(pdata->index[1], pdata->index[2]));

#if DEBUG_CALC_SHIFTS
printf("*** calc shifts ***\n");
printf("miller: %f %f %f\n",surf->surface.miller[0]
                            ,surf->surface.miller[1]
                            ,surf->surface.miller[2]);
printf("region: %d %d\n", (gint) surf->surface.region[0],
                          (gint) surf->surface.region[1]);
printf("   gcd: %f\n", gcd);
printf(" Shift: %f\n", surf->surface.shift);
#endif

generate_surface(data, surf);
if (surf->surface.dspacing == 0.0)
  {
  printf("calc_valid_shifts() error, invalid dspacing.\n");
  goto calc_valid_shifts_cleanup;
  }

#if DEBUG_CALC_SHIFTS
printf("  Dhkl: %f\n", surf->surface.dspacing);
#endif

/* transfer appropriate setup data to new model */
copy_gulp_data(data, surf);
copy_elem_data(data, surf);
surf->gulp.run = E_SINGLE;
surf->gulp.method = CONV;

/* build lattice matrix & connectivity */
prep_model(surf);
/* enforce whole molecules */
if (!sysenv.unfragment)
  unfragment(surf);

/* only sample one dhkl's worth of cuts */
zlim = surf->surface.dspacing + EPSILON;

/* get all possible shifts from z coordinates */
zlist = NULL;
if (data->surface.ignore_bonding)
  {
#if DEBUG_CALC_SHIFTS
printf("Ignoring bonding...\n");
#endif
/* core z coord */
  for (list=surf->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;
    ARR3SET(vec, core->x);
    vecmat(surf->latmat, vec);

/* grab one slice [0, Dhkl] */
    if (-vec[2] > zlim)
      continue;
    if (vec[2] > 0.0)
      continue;

/* NB: cartesian z direction is opposite to shift value sign */
    z1 = g_malloc(sizeof(gdouble));
    *z1 = -vec[2];

    zlist = g_slist_prepend(zlist, z1);
    }
  }
else
  {
#if DEBUG_CALC_SHIFTS
printf("Using molecule centroids... [%d]\n", g_slist_length(surf->moles));
#endif
/* molecule centroid */
  for (list=surf->moles ; list ; list=g_slist_next(list))
    {
    mol = (struct mol_pak *) list->data;
    ARR3SET(vec, mol->centroid);
    vecmat(surf->latmat, vec);

/* grab one slice [0, Dhkl] */
    if (-vec[2] > zlim)
      continue;
    if (vec[2] > 0.0)
      continue;

/* NB: cartesian z direction is opposite to shift value sign */
    z1 = g_malloc(sizeof(gdouble));
    *z1 = -vec[2];
    zlist = g_slist_prepend(zlist, z1);
    }
  }

zlist = g_slist_sort(zlist, (gpointer) zsort);
num_cuts = g_slist_length(zlist);

#if DEBUG_CALC_SHIFTS
printf("Found %d cuts.\n", num_cuts);
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
  printf(" %f", *z1);
  }
printf("\n");
#endif

zlist = trim_zlist(zlist);
num_cuts = g_slist_length(zlist);

#if DEBUG_CALC_SHIFTS
printf("Found %d unique cuts.\n", num_cuts);
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
  printf("%f ", *z1);
  }
printf("\n");
#endif

/* failsafe */
if (!zlist)
  goto calc_valid_shifts_cleanup;

/* use midpoints to avoid cutoff problems */
/* the last cut will be unchanged, but could be removed anyway */
/* (due to the cyclic nature of slice cutting) so should be fine */
list = zlist;
z1 = (gdouble *) list->data;
/* NB: store first cut to combine with the last cut (ie wrap around) */
z0 = *z1;
list = g_slist_next(list);
while (list)
  {
  z2 = (gdouble *) list->data;

  *z1 = 0.5*(*z1 + *z2);

  z1 = z2;

  list = g_slist_next(list);
  }

/* special case for last cut - wrap back (case 1) */
*z1 = 0.5*(z0 + *z1 - surf->surface.dspacing);
/* case 2, wrap forward */
if (*z1 < -FRACTION_TOLERANCE)
  *z1 += surf->surface.dspacing;
else
  *z1 = fabs(*z1);  /* force -0.0000 -> +0.0000 */

#if DEBUG_CALC_SHIFTS
printf("Midpoint equivalent cuts:\n");
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
  printf(" %f", *z1);
  }
printf("\n");
#endif

/* convert from physical z coords to a shift value */
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
/* NEW - shifts must be calculated wrt hkl WITH NO GCD */
/* ie divide by the "full" dspacing for the surface */
  *z1 *= 1.0 / (gcd * surf->surface.dspacing);
  }

/* clamp */
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
/* NB: z1 is a pointer to ONE gdouble, so only clamp in 1D */
  fractional_clamp(z1, dummy, 1);
  }

/* re-rank */
zlist = g_slist_sort(zlist, (gpointer) zsort);

#if DEBUG_CALC_SHIFTS
printf("Calculated shifts: %d\n", num_cuts);
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
  printf(" %f", *z1);
  }
printf("\n");
#endif

zlist = trim_zlist(zlist);
num_shifts = g_slist_length(zlist);

#if DEBUG_CALC_SHIFTS
printf("Unique shifts: %d\n", num_shifts);
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;
  printf(" %f", *z1);
  }
printf("\n");
#endif

/* turn bonding off */
build = data->build_molecules;
if (data->surface.ignore_bonding && build)
  {
  data->build_molecules = FALSE;
  calc_bonds(data);
  calc_mols(data);
  }

/* generate & check each one */
/* NB: this'll overwrite our 1st model, saving having to delete it */
slist = NULL;
for (list=zlist ; list ; list=g_slist_next(list))
  {
  z1 = (gdouble *) list->data;

/* destroy surfs core/shell lists */
  free_core_list(surf);

/* make the surface (also computes dipole) */
  surf->surface.shift = *z1;
  generate_surface(data, surf);
  prep_model(surf);

#if DEBUG_CALC_SHIFTS
printf("Shift = %f, dipole = %f\n", *z1, surf->gulp.sdipole);
#endif

/* create new model if valid cut (unless user wants all cuts) */
  if (fabs(surf->gulp.sdipole) < data->gulp.sdipole_tolerance
      || data->surface.include_polar)
    {
    sdata = create_shift(*z1);
    sdata->dipole = surf->gulp.sdipole;
    sdata->dipole_computed = TRUE;
    slist = g_slist_prepend(slist, sdata);
    }
  }

/* new if shift list is non-empty, overwrite only if we found something */
if (slist)
  {
  slist = g_slist_reverse(slist);
  if (pdata->shifts)
    g_slist_free(pdata->shifts);
  pdata->shifts = slist;
  }

/* restore source model's connectivity */
if (data->surface.ignore_bonding && build)
  {
  data->build_molecules = build;
  calc_bonds(data);
  calc_mols(data);
  }

/* this has data only if the above was successful */
free_slist(zlist);

calc_valid_shifts_cleanup:

/* cleanup */
free_model(surf);
/*
model_ptr(surf->number, RELEASE);
sysenv.active = data->number;
*/

return(0);
}

/***********************************/
/* computes the dipole for a plane */
/***********************************/
#define DEBUG_TEST_VALID_SHIFTS 0
void test_valid_shifts(struct plane_pak *plane, struct model_pak *data)
{
GSList *list;
struct shift_pak *shift;
struct model_pak surf;

#if DEBUG_TEST_VALID_SHIFTS
printf("Testing for valid shifts [%d %d %d]\n",
        plane->index[0], plane->index[1], plane->index[2]); 
#endif

/* init the model to be used for generating shifts */
template_model(&surf);
copy_gulp_data(data, &surf);
copy_elem_data(data, &surf);

/* test the shifts */
for (list=plane->shifts ; list ; list=g_slist_next(list))
  {
  shift = (struct shift_pak *) list->data;

/* init the surface we want */
  ARR3SET(surf.surface.miller, plane->index);
  surf.surface.region[0] = 1;
  surf.surface.region[1] = 0;
  surf.surface.shift = shift->shift;

/* destroy old cores & then create the surface */
  free_core_list(&surf);
  generate_surface(data, &surf);
  prep_model(&surf);

#if DEBUG_TEST_VALID_SHIFTS
printf("[%f:%f]\n", shift->shift, surf.gulp.sdipole);
#endif

  shift->dipole = surf.gulp.sdipole;
  shift->dipole_computed = TRUE;
  }
}

/******************************/
/* do the energy calculations */
/******************************/
#define DEBUG_EXEC_ECALC_TASK 0
void exec_ecalc_task(struct model_pak *tmpdata)
{
gint r1size;
gdouble sbe;
GSList *list;
struct model_pak *data, surf;
struct core_pak *core;

/* get & check source model */
data = model_ptr(tmpdata->surface.model, RECALL);
g_return_if_fail(data != NULL);
g_return_if_fail(data->periodic == 3);

/* surface bulk energy */
sbe = data->gulp.energy;
sbe /= (gdouble) data->num_atoms;

/* init model for surface data */
template_model(&surf);
surf.id = GULP;
surf.periodic = 2;
surf.fractional = TRUE;
/* transfer surface data to new model */
ARR3SET(surf.surface.miller, tmpdata->surface.miller);
surf.surface.shift = tmpdata->surface.shift;
surf.surface.region[0] = tmpdata->surface.region[0];
surf.surface.region[1] = tmpdata->surface.region[1];

/* transfer appropriate setup data to new model */
copy_gulp_data(tmpdata, &surf);
copy_elem_data(tmpdata, &surf);

/* NB: surface calcs are always conv */
surf.gulp.method = CONV;

#if DEBUG_EXEC_ECALC_TASK
printf("*** generating surface ***\n");
printf("Miller (%d,%d,%d)\n",surf.surface.miller[0]
                            ,surf.surface.miller[1]
                            ,surf.surface.miller[2]);
printf("Shift %f\n",surf.surface.shift);
printf("Region sizes (%d,%d)\n",surf.surface.region[0], surf.surface.region[1]);
#endif

/* create & init */
generate_surface(data, &surf);
prep_model(&surf);

/* NEW */
g_free(surf.gulp.dump_file);
surf.gulp.dump_file = g_strdup(tmpdata->gulp.dump_file);


/* compute surface bulk energy */
r1size=0;
for (list=surf.cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->region == REGION1A)
    r1size++;
  }
if (!r1size)
  printf("Warning: empty region 1.\n");

surf.gulp.sbulkenergy = sbe * (gdouble) r1size;

/* use gulp to calc */
if (write_gulp(tmpdata->gulp.temp_file, &surf))
  printf("write failed.\n");

if (tmpdata->gulp.no_exec)
  {
#if DEBUG_EXEC_ECALC_TASK
printf("Skipping GULP execution on user request.\n");
#endif
  return;
  }

if (exec_gulp(tmpdata->gulp.temp_file, tmpdata->gulp.out_file))
  printf("exec failed.\n");

#if DEBUG_EXEC_ECALC_TASK
if (read_gulp_output(tmpdata->gulp.out_file, &surf))
  printf("read failed.\n");
gulp_dump(&surf);
#endif
}

/*********************************************/
/* process result from an energy calculation */
/*********************************************/
#define DEBUG_PROC_ECALC_TASK 0
void proc_ecalc_task(struct model_pak *tmpdata)
{
struct shift_pak *sdata;
struct model_pak surf, *data;

g_return_if_fail(tmpdata != NULL);

/* don't attempt to process if gulp execution was turned off */
if (tmpdata->gulp.no_exec)
  return;

/* init */
template_model(&surf);
surf.grafted = TRUE;

/* read gulp file */
if (read_gulp_output(tmpdata->gulp.out_file, &surf))
  printf("proc_ecalc_task() error: read failed.\n");

#if DEBUG_PROC_ECALC_TASK
gulp_dump(&surf);
#else
/* clean up */
unlink(tmpdata->gulp.temp_file);
unlink(tmpdata->gulp.out_file);
#endif

/* feed energy into shift */
sdata = (struct shift_pak *) (tmpdata->planes)->data;
if (sdata)
  {
  sdata->esurf[0] = surf.gulp.esurf[0];
  sdata->esurf[1] = surf.gulp.esurf[1];
  sdata->eatt[0] = surf.gulp.eatt[0];
  sdata->eatt[1] = surf.gulp.eatt[1];
  sdata->gnorm = surf.gulp.gnorm;
  }

/* update surface dialog */
data = model_ptr(tmpdata->surface.model, RECALL);

/* TODO - a more fine grained update */
fn_surf_refresh_all();

/* free tmpdata */
/* FIXME (hack) - this was a borrowed pointer, so don't free it */
tmpdata->planes = NULL;
free_model(tmpdata);
}

/***********************************************/
/* submit a background energy calculation task */
/***********************************************/
void new_ecalc_task(gint source, struct plane_pak *pdata, struct shift_pak *sdata)
{
struct model_pak *tmpdata, *data;

/* duplicate the data passed, since it may be changed */
/* or destroyed by the user while the task is still queued */
tmpdata = g_malloc(sizeof(struct model_pak));
template_model(tmpdata);

/* duplicate GULP setup of source model at this point */
data = model_ptr(source, RECALL);
copy_gulp_data(data, tmpdata);
copy_elem_data(data, tmpdata);

tmpdata->surface.model = source;
tmpdata->surface.shift = sdata->shift;
tmpdata->surface.region[0] = sdata->region[0];
tmpdata->surface.region[1] = sdata->region[1];
ARR3SET(tmpdata->surface.miller, pdata->index);

/* set the filenames to use */
g_free(tmpdata->gulp.temp_file);
tmpdata->gulp.temp_file = g_strdup_printf("%s_%d%d%d_%2.4f_%d_%d.gin",
    data->basename, pdata->index[0], pdata->index[1], pdata->index[2],
                    sdata->shift, sdata->region[0], sdata->region[1]);

g_free(tmpdata->gulp.dump_file);
tmpdata->gulp.dump_file = g_strdup_printf("%s_%d%d%d_%2.4f_%d_%d.res",
   data->basename, pdata->index[0], pdata->index[1], pdata->index[2],
                   sdata->shift, sdata->region[0], sdata->region[1]);

g_free(tmpdata->gulp.out_file);
tmpdata->gulp.out_file = g_strdup_printf("%s_%d%d%d_%2.4f_%d_%d.got",
   data->basename, pdata->index[0], pdata->index[1], pdata->index[2],
                   sdata->shift, sdata->region[0], sdata->region[1]);


/* FIXME - cheat, by tacking the shift to update on the *planes* slist */
tmpdata->planes = g_slist_append(tmpdata->planes, sdata);

submit_task("Energy", &exec_ecalc_task, tmpdata, &proc_ecalc_task, tmpdata, NULL);
}

/***********************************/
/* single region convergence cycle */
/***********************************/
#define DEBUG_SURF_CONV 1
gint surf_conv(struct model_pak *surf, gint type)
{
gint n, flag, relax, region, r1size;
gdouble sbe, de, old_energy;
gchar *inp, *out;
GSList *list;
struct model_pak *data;
struct core_pak *core;

switch(type)
  {
  case REGION1A:
    region = 0;
    break;
  case REGION2A:
    region = 1;
    break;
  default:
    printf("surf_conv() error: bad region type.\n");
    return(1);
  }

data = model_ptr(surf->surface.model, RECALL);
g_return_val_if_fail(data != NULL, 2);

/* which energy to converge */
if (surf->gulp.run == E_OPTIMIZE)
  relax = 1;
else
  relax = 0;

/* surface bulk energy */
sbe = data->gulp.energy;
sbe /= (gdouble) data->num_atoms;

/* converge region size */
n=flag=0;
old_energy = 0.0;
de = 99999999.9;
for (;;)
  {
/* make & init */
  free_core_list(surf);

  generate_surface(data, surf);
  prep_model(surf);

/* TODO - this is possibly done enough to make it a call */
/* compute surface bulk energy */
  r1size=0;
  for (list=surf->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;
    if (core->region == REGION1A)
      r1size++;
    }
  if (!r1size)
    printf("Warning: empty region 1.\n");

  surf->gulp.sbulkenergy = sbe * (gdouble) r1size;

/* create appropriate name */
  inp = g_strdup_printf("%s_%d%d%d_%2.4f_%d_%d.gin",
                        data->basename, 
                       (gint) surf->surface.miller[0],
                       (gint) surf->surface.miller[1],
                       (gint) surf->surface.miller[2],
                        surf->surface.shift,
                       (gint) surf->surface.region[0],
                       (gint) surf->surface.region[1]);

  out = g_strdup_printf("%s_%d%d%d_%2.4f_%d_%d.got",
                        data->basename, 
                       (gint) surf->surface.miller[0],
                       (gint) surf->surface.miller[1],
                       (gint) surf->surface.miller[2],
                        surf->surface.shift,
                       (gint) surf->surface.region[0],
                       (gint) surf->surface.region[1]);

/* use gulp to calc */
  if (write_gulp(inp, surf))
    return(2);

  if (exec_gulp(inp, out))
    return(3);

/* read_gulp_out() can call show_text() which modifies */
/* the message widget, hence the thread lock is required */
/* FIXME - currently, this function surf_conv() is always */
/* called in a thread but what if it isn't? ie will the */
/* threads enter call deadlock if we're in the main GTK loop? */
  gdk_threads_enter();
  if (read_gulp_output(out, surf))
    return(4);
  gdk_threads_leave();

/* get difference */
  if (surf->surface.converge_eatt)
    {
    de = surf->gulp.eatt[relax] - old_energy;
    old_energy = surf->gulp.eatt[relax];
    }
  else
    {
    de = surf->gulp.esurf[relax] - old_energy;
    old_energy = surf->gulp.esurf[relax];
    }

/* bad convergence checks */
  if (n && de > 0.1)
    {
    if (fabs(surf->gulp.sdipole) > 0.1)
      {
      printf("ERROR: polar surface, with diverging surface energy.\n");
      return(5);
      }
    }

#if DEBUG_SURF_CONV
if (surf->surface.converge_eatt)
  {
  printf("[%d:%d]  Eatt = %f (%f)\n", (gint) surf->surface.region[0],
                                      (gint) surf->surface.region[1],
                                      surf->gulp.eatt[relax], de);
  }
else
  {
  printf("[%d:%d] Esurf = %f (%f)\n", (gint) surf->surface.region[0],
                                      (gint) surf->surface.region[1],
                                      surf->gulp.esurf[relax], de);
  }
#endif

/* remove old files for next cycle */
  unlink(inp);
  unlink(out);
  g_free(inp);
  g_free(out);

/* convergence check */
  if (surf->surface.converge_eatt)
    {
    if (fabs(de) < MAX_DEEATT)
      break;
    }
  else
    {
    if (fabs(de) < MAX_DESURF)
      break;
    }

/* next size */
  surf->surface.region[region]++;
  n++;
  }

#if DEBUG_SURF_CONV
printf("Region %d converged.\n", region+1);
#endif

/* we can go back one size, due to the way convergence is checked */
surf->surface.region[region]--;

return(0);
}

/*****************************************/
/* region convergence calculation (task) */
/*****************************************/
#define DEBUG_EXEC_REGCON_TASK 1
void exec_regcon_task(struct model_pak *tmpdata, struct task_pak *task)
{
gint m, r;
gchar *procfile;
struct model_pak surf;
struct model_pak *data;
struct plane_pak *pdata;
struct shift_pak *sdata;

/* checks */
sdata = (struct shift_pak *) (tmpdata->planes)->data;
procfile = sdata->procfile;

data = model_ptr(tmpdata->surface.model, RECALL);
g_return_if_fail(data != NULL);
if (data->periodic != 3)
  return;
if (!tmpdata->surface.miller[0] && !tmpdata->surface.miller[1] && !tmpdata->surface.miller[2])
  return;

#if DEBUG_EXEC_REGCON_TASK
printf("----------------------------------------------\n");
printf("Converging regions for model: %d, ", tmpdata->surface.model);
if (data->surface.converge_eatt)
  printf("Eatt based.\n");
else
  printf("Esurf based.\n");
#endif

/* complete molecules */
if (data->surface.ignore_bonding)
  {
  data->build_molecules = FALSE;
  calc_bonds(data);
  calc_mols(data);
  }

/* init surface structure */
template_model(&surf);
surf.id = GULP;
surf.periodic = 2;
surf.fractional = TRUE;
ARR3SET(surf.surface.miller, tmpdata->surface.miller);
surf.surface.shift = tmpdata->surface.shift;
surf.surface.model = data->number;

/* transfer appropriate setup data to new model */
copy_gulp_data(tmpdata, &surf);
copy_elem_data(tmpdata, &surf);

/* NB: surface calcs are always conv */
surf.gulp.method = CONV;

/* estimate starting region sizes from the dspacing for the plane */
pdata = fn_make_plane(surf.surface.miller, data);
g_return_if_fail(pdata != NULL);

/* mult dhkl by the gcd */
m = GCD(surf.surface.miller[0], GCD(surf.surface.miller[1], surf.surface.miller[2]));

if (pdata->dhkl < MIN_THICKNESS)
  {
/* FIXME - what to do if dhkl is very small or even zero */
  r = 1 + (gint) (MIN_THICKNESS / (m*pdata->dhkl));

  if (r > surf.surface.region[0])
    surf.surface.region[0] = r;
  if (r > surf.surface.region[1])
    surf.surface.region[1] = r;
  }

#if DEBUG_EXEC_REGCON_TASK
printf("----------------------------------------------\n");
printf("              Miller: %d %d %d \n", (gint) surf.surface.miller[0]
                                          , (gint) surf.surface.miller[1]
                                          , (gint) surf.surface.miller[2]);
printf("                Dhkl: %f\n", pdata->dhkl);
printf("               Shift: %f\n", surf.surface.shift);
printf("Initial region sizes: %d , %d\n", (gint) surf.surface.region[0],
                                          (gint) surf.surface.region[1]);
printf("----------------------------------------------\n");
#endif

/* for region size consistancy between un and relaxed */
/* convergence calcs, always do an unrelaxed calc 1st */
surf.gulp.run = E_SINGLE;
/* Eatt or Esurf bases? */
surf.surface.converge_eatt = data->surface.converge_eatt;

#if DEBUG_EXEC_REGCON_TASK
printf("[1] unrelaxed convergence cycle...\n");
#endif

/* converge region 2 size */
if (data->surface.converge_r2)
  {
  if (surf_conv(&surf, REGION2A))
    {
    task->message = g_strdup("Failed unrelaxed convergence of region 2.");
    return;
    }
  }
#if DEBUG_EXEC_REGCON_TASK
else
  printf("Skipping region 2 convergence...\n");
#endif
  
/* converge region 1 size */
if (data->surface.converge_r1)
  {
  if (surf_conv(&surf, REGION1A))
    {
    task->message = g_strdup("Failed unrelaxed convergence of region 1.");
    return;
    }
  }
#if DEBUG_EXEC_REGCON_TASK
else
  printf("Skipping region 1 convergence...\n");
#endif

/* NEW - now, if relaxed is desired - repeat with unrelaxed as starting pt */
if (tmpdata->gulp.run == E_OPTIMIZE)
  {
  surf.gulp.run = E_OPTIMIZE;

#if DEBUG_EXEC_REGCON_TASK
printf("[2] relaxed convergence cycle...\n");
#endif

/* converge region 2 size */
  if (data->surface.converge_r2)
    {
    if (surf_conv(&surf, REGION2A))
      {
      task->message = g_strdup("Failed relaxed convergence of region 2.");
      return;
      }
    }
#if DEBUG_EXEC_REGCON_TASK
else
  printf("Skipping region 2 convergence...\n");
#endif
/* converge region 1 size */
  if (data->surface.converge_r1)
    {
    if (surf_conv(&surf, REGION1A))
      {
      task->message = g_strdup("Failed relaxed convergence of region 1.");
      return;
      }
    }
#if DEBUG_EXEC_REGCON_TASK
else
  printf("Skipping region 1 convergence...\n");
#endif
  }

/* create a shift to put the converged region data in */
sdata = create_shift(tmpdata->surface.shift);
sdata->region[0] = surf.surface.region[0];
sdata->region[1] = surf.surface.region[1];
sdata->esurf[0] = surf.gulp.esurf[0];
sdata->eatt[0] = surf.gulp.eatt[0];
sdata->esurf[1] = surf.gulp.esurf[1];
sdata->eatt[1] = surf.gulp.eatt[1];
sdata->gnorm = surf.gulp.gnorm;

pdata->shifts = g_slist_append(pdata->shifts, sdata);
surf.planes = g_slist_append(surf.planes, pdata);

/* replace surface data with the source model's */
ARR3SET(&surf.pbc[0], &data->pbc[0]);
ARR3SET(&surf.pbc[3], &data->pbc[3]);
g_free(surf.sginfo.spacename);
surf.sginfo.spacename = g_strdup(data->sginfo.spacename);

/* save converged plane & shift data to the process communication file. */
write_gmf(procfile, &surf);

/* done */
g_free(pdata);
g_free(sdata);
}

/**************************************/
/* region convergence task processing */
/**************************************/
#define DEBUG_PROC_REGON_TASK 0
void proc_regcon_task(struct model_pak *tmpdata)
{
GSList *plist, *slist;
struct model_pak surf;
struct plane_pak *pdata;
struct shift_pak *sdata, *shift;

g_return_if_fail(tmpdata != NULL);

/* the data in tmpdata->planes belongs to the main model, */
/* and we don't want it free'd when tmpdata is destroyed */
shift = (struct shift_pak *) (tmpdata->planes)->data;
tmpdata->planes = NULL;

g_return_if_fail(shift != NULL);

#if DEBUG_PROC_REGON_TASK
printf("Processing regcon file: %s\n", shift->procfile);
printf("old data: %f  %f  %f\n", shift->shift, shift->esurf[0], shift->eatt[0]);
#endif

/* init the surf for adding planes from a file */
template_model(&surf);
genpos(&surf);
if (load_planes(shift->procfile, &surf))
  printf("proc_regcon_task() error: bad planes file (%s)\n", shift->procfile);
else
  unlink(shift->procfile);

/* update shift info for the supplied plane (if any/if desired) */
plist = surf.planes;
if (plist)
  {
  pdata = (struct plane_pak *) plist->data;

#if DEBUG_PROC_REGON_TASK
printf("Reading plane: %d %d %d\n", pdata->index[0], pdata->index[1], pdata->index[2]);
#endif

  slist = pdata->shifts;
  if (slist)
    {
    sdata = (struct shift_pak *) slist->data;

    if (fabs(shift->shift - sdata->shift) > FRACTION_TOLERANCE)
      printf("Error: shift mismatch in procfile (job interference?)\n");
    else
      {
      shift->region[0] = sdata->region[0];
      shift->region[1] = sdata->region[1];
      shift->esurf[0] = sdata->esurf[0];
      shift->esurf[1] = sdata->esurf[1];
      shift->eatt[0] = sdata->eatt[0];
      shift->eatt[1] = sdata->eatt[1];
      shift->gnorm = sdata->gnorm;
      }
    }
  }

#if DEBUG_PROC_REGON_TASK
printf("new data: (%d,%d) %f  %f  %f\n", shift->region[0], shift->region[1], shift->shift, shift->esurf[0], shift->eatt[0]);
#endif

/* update current spinner values */
/*
gtk_spin_button_set_value(GTK_SPIN_BUTTON(r1_spin), r1);
gtk_spin_button_set_value(GTK_SPIN_BUTTON(r2_spin), r2);
*/

/* TODO - a more fine grained update */
fn_surf_refresh_all();

/* TODO - free model (surf)'s data plist & slist */
g_free(shift->procfile);

/* free tmpdata & everything in it */
free_model(tmpdata);
}

/*********************************/
/* region convergence task setup */
/*********************************/
void new_regcon_task(gint source, struct plane_pak *pdata, struct shift_pak *sdata)
{
struct model_pak *tmpdata, *data;

/* duplicate the data passed, since it may be changed */
/* or destroyed by the user while the task is still queued */
tmpdata = g_malloc(sizeof(struct model_pak));
template_model(tmpdata);

/* duplicate GULP setup of source model at this point */
data = model_ptr(source, RECALL);
copy_gulp_data(data, tmpdata);
copy_elem_data(data, tmpdata);

tmpdata->surface.model = source;
tmpdata->surface.shift = sdata->shift;
tmpdata->surface.region[0] = sdata->region[0];
tmpdata->surface.region[1] = sdata->region[1];
ARR3SET(tmpdata->surface.miller, pdata->index);

/* process communication file */
sdata->procfile = gun("gmf");

/* FIXME - cheat, by tacking the shift to update on the *planes* slist */
tmpdata->planes = g_slist_append(tmpdata->planes, sdata);

submit_task("Regcon", &exec_regcon_task, tmpdata, &proc_regcon_task, tmpdata, NULL);
}

/*************************/
/* surface creation task */
/*************************/
void cb_surf_create(GtkWidget *w, struct model_pak *model)
{
struct plane_pak *plane;
struct shift_pak *shift;

/* setup plane */
plane = fn_make_plane(surfdata.surface.miller, model);
g_return_if_fail(plane != NULL);

/* setup shift */
shift = create_shift(surfdata.surface.shift);
shift->region[0] = surfdata.surface.region[0];
shift->region[1] = surfdata.surface.region[1];

/* create */
make_surface(model->number, plane, shift);  
/* TODO - free plane & shift ??? */
}

/**************************************/
/* process GtkTreeStore surface tasks */
/**************************************/
#define DEBUG_SURF_TASK 0
void cb_surf_task(GtkWidget *w, gint type)
{
gint all_shifts=FALSE;
GtkTreeIter parent, child;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct model_pak *model=NULL;
struct plane_pak *plane=NULL;
struct shift_pak *shift=NULL;

/* checks */
treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;

/* setup starting iterator */
if (all_planes)
  {
/* get first iterator on the tree */
  if (!gtk_tree_model_get_iter_first(treemodel, &parent))
    return;
  all_shifts = TRUE;
  child = parent;
  }
else
  {
/* get selection */
  if (!(selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(surf_tree_view))))
    return;
/* get selected iterator */
  if (!gtk_tree_selection_get_selected(selection, &treemodel, &child))
    return;
  }

/* FIXME - this while loop sometimes goes completely haywire, especially */
/* when calculating valid shifts & operating on all planes */

/* child should now be valid, and point to either a plane or shift */
/* depending on the loop type required (eg all shifts or just one) */
gtk_tree_model_get(treemodel, &child, SURF_MODEL, &model, -1);
do
  {
  gtk_tree_model_get(treemodel, &child, SURF_PLANE, &plane, -1);
  gtk_tree_model_get(treemodel, &child, SURF_SHIFT, &shift, -1);

/* does the current iterator point to a shift? */
  if (shift)
    {
/* yes, get the parent plane */
    gtk_tree_model_iter_parent(treemodel, &parent, &child);
    }
  else
    {
/* no, the child is a plane - do all shifts in the plane */
    parent = child;
    all_shifts = TRUE;

/* attempt to get the first shift of the plane */
    if (gtk_tree_model_iter_children(treemodel, &child, &parent))
      gtk_tree_model_get(treemodel, &child, SURF_SHIFT, &shift, -1);
    else
      shift = NULL;
    }

g_assert(plane != NULL);

#if DEBUG_SURF_TASK
printf("plane: %p [%d], shift: %p [%d]\n", plane, all_planes, shift, all_shifts);
#endif

/* type of operation required */
  switch (type)
    {
    case CALC_SHIFTS:
      fn_prune_shifts(plane);
      calc_valid_shifts(model, plane);
      fn_graft_plane(plane, model);
/* force skip to next plane */
      child = parent;
      all_shifts=FALSE;
      break;

    case CALC_ENERGY:
      if (shift)
        new_ecalc_task(model->number, plane, shift);
      break;

    case CONV_REGIONS:
      if (shift)
        new_regcon_task(model->number, plane, shift);  
      break;

    case MAKE_FACES:
      if (shift)
        make_surface(model->number, plane, shift);  
      break;
    }

/* get next shift iterator */
  if (gtk_tree_model_iter_next(treemodel, &child))
    {
/* got next iterator, proceed if we must do all planes */
    if (all_planes)
      all_shifts = TRUE;
    }
  else
    {
/* no iterator - check the loop type to see if we should continue */
    if (all_planes)
      {
/* get next parent (terminate if it doesn't exist) */
      if (gtk_tree_model_iter_next(treemodel, &parent))
        child = parent;
      else
        all_shifts = FALSE;
      }
    else
      all_shifts = FALSE;
    }
  }
while (all_shifts);

/* a more fine grained update */
fn_surf_refresh_all();
}

/****************************************/
/* save the current surface calculation */
/****************************************/
void export_planes(gchar *name)
{
struct model_pak *data;
GString *filename;

data = model_ptr(surfdata.surface.model, RECALL);
if (!data)
  return;

/* process filename to force .gmf extension */
filename = g_string_new(strdup_basename(name));
g_string_sprintfa(filename, ".gmf");

write_gmf(filename->str, data);

close_dialog_type(FILE_SELECT);

g_string_free(filename, TRUE);
}

/******************************************/
/* apply a new morphology type to a model */
/******************************************/
void change_morph_type(struct model_pak *data, gint type)
{
gdouble mat[9];

g_return_if_fail(data != NULL);

/* check it's a valid type? */
data->morph_type = type;

/* free vertex/adj data */
free_vertices(data);

memcpy(mat, data->rotmat, 9*sizeof(gdouble));

exec_cdd(data);
prep_model(data);

memcpy(data->rotmat, mat, 9*sizeof(gdouble));
calc_coords(REFRESH, data);

redraw_canvas(SINGLE);
}

/*******************************************/
/* delete a plane and symmetry equivalents */
/*******************************************/
/* returns the number of planes deleted */
gint delete_plane(struct plane_pak *plane, struct model_pak *data)
{
gint n=1;
GSList *list;
struct plane_pak *pcomp;

g_assert(plane != NULL);
g_assert(data != NULL);

/* remove equivalents */
list = data->planes;
while (list)
  {
  pcomp = (struct plane_pak *) list->data;
  list = g_slist_next(list);

/* NB: don't delete the reference plane until the end */
  if (pcomp == plane)
    continue;

  if (facet_equiv(data, plane->index, pcomp->index))
    {
    data->planes = g_slist_remove(data->planes, pcomp);
    g_free(pcomp);
    n++;
    }
  }
/* remove main plane */
data->planes = g_slist_remove(data->planes, plane);
g_free(plane);

data->num_planes = g_slist_length(data->planes);
if (!data->num_planes)
  data->planes = NULL;

return(n);
}

/***************************/
/* shift deletion callback */
/***************************/
void cb_surf_delete_selected(GtkWidget *w, gpointer dummy)
{
GtkTreeIter iter, iter2;
GtkTreeModel *treemodel;
GtkTreePath *treepath;
GtkTreeSelection *selection;
struct model_pak *model=NULL;
struct plane_pak *plane=NULL;
struct shift_pak *shift=NULL;

selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(surf_tree_view)); 
if (selection)
  {
  if (gtk_tree_selection_get_selected(selection, &treemodel, &iter))
    {
/* get row data */
    gtk_tree_model_get(treemodel, &iter, SURF_MODEL, &model, -1);
    gtk_tree_model_get(treemodel, &iter, SURF_PLANE, &plane, -1);
    gtk_tree_model_get(treemodel, &iter, SURF_SHIFT, &shift, -1);
    g_assert(model != NULL);
    g_assert(plane != NULL);
/* attempt to select the previous iterator at the current level */
    treepath = gtk_tree_model_get_path(treemodel, &iter);
    if (gtk_tree_path_prev(treepath))
      {
      if (gtk_tree_model_get_iter(treemodel, &iter2, treepath))
        gtk_tree_selection_select_iter(selection, &iter2);
      }
    else
      {
/* else, attempt to select next iterator */
      iter2 = iter;
      if (gtk_tree_model_iter_next(treemodel, &iter2))
        gtk_tree_selection_select_iter(selection, &iter2);
      }
/* remove the data */
    if (shift)
      {
      plane->shifts = g_slist_remove(plane->shifts, shift);
      surf_free_child(&iter);
      }
    else
      {
      delete_plane(plane, model);
      surf_free_parent(&iter);
      }
/* remove from the tree view widget */
    gtk_tree_store_remove(surf_tree_store, &iter);
    }
  }
}

/***************************/
/* shift deletion callback */
/***************************/
void cb_prune_surf_list(GtkWidget *w, gint type)
{
gint m, n, mode;
GtkTreeIter iter, parent;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct model_pak *model=NULL;
struct plane_pak *plane=NULL;
struct shift_pak *shift=NULL;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;
selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(surf_tree_view));
if (!selection)
  return;

/* get selected iterator */
if (!gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  {
/* nothing selected - goto root */
  if (!gtk_tree_model_get_iter_first(treemodel, &parent))
    return;
  gtk_tree_model_get(treemodel, &parent, SURF_MODEL, &model, -1);
  mode = ALL_PLANES;
  }
else
  {
  if (gtk_tree_model_iter_parent(treemodel, &parent, &iter))
    {
/* iter is child (ie shift) */
    gtk_tree_model_get(treemodel, &parent, SURF_MODEL, &model, -1);
    mode = ALL_SHIFTS;
    }
  else
    {
/* iter is a plane -> loop over all -> goto root */
    if (!gtk_tree_model_get_iter_first(treemodel, &parent))
      return;
    gtk_tree_model_get(treemodel, &parent, SURF_MODEL, &model, -1);
    mode = ALL_PLANES;
    }
  }

/* action type */
switch (mode)
  {
  case ALL_PLANES:
    if (type == INVALID_SHIFTS)
      {
/* loop backwards over planes */
      m = gtk_tree_model_iter_n_children(treemodel, NULL) - 1;
      while (m >= 0)
        {
        if (!gtk_tree_model_iter_nth_child(treemodel, &parent, NULL, m))
          break;
        m--;
        gtk_tree_model_get(treemodel, &parent, SURF_PLANE, &plane, -1);

/* loop backwards over shifts in current plane */
        n = gtk_tree_model_iter_n_children(treemodel, &parent) - 1;
        while (n >= 0)
          {
          if (!gtk_tree_model_iter_nth_child(treemodel, &iter, &parent, n))
            break;
          n--;
          gtk_tree_model_get(treemodel, &iter, SURF_SHIFT, &shift, -1);

/* test dipole */
          if (fabs(shift->dipole) >= model->gulp.sdipole_tolerance)
            {
            plane->shifts = g_slist_remove(plane->shifts, shift);
            surf_free_child(&iter);
            gtk_tree_store_remove(surf_tree_store, &iter);
            }
          }

/* TODO - check box for wether user wants empty planes automatically deleted */
/* eg a call that scans the entire list to check for empty planes all at once? */
/*
        if (!g_slist_length(plane->shifts))
          printf("Empty plane.\n");
*/

        }
      }
    else
      {
/* free iterator colum data */
      surf_free_all();
/* clear the tree view widget */ 
      gtk_tree_store_clear(surf_tree_store);
/* free the underlying data */
      free_plane_list(model);
      }
    break;

  case ALL_SHIFTS:
/* loop backwards to delete */
    n = gtk_tree_model_iter_n_children(treemodel, &parent) - 1;
    while (n >= 0)
      {
      if  (!gtk_tree_model_iter_nth_child(treemodel, &iter, &parent, n))
        break;
      n--;

      gtk_tree_model_get(treemodel, &iter, SURF_PLANE, &plane, -1);
      gtk_tree_model_get(treemodel, &iter, SURF_SHIFT, &shift, -1);

      if (type == INVALID_SHIFTS)
        if (fabs(shift->dipole) < model->gulp.sdipole_tolerance)
           continue; 

      plane->shifts = g_slist_remove(plane->shifts, shift);
      surf_free_child(&iter);
      gtk_tree_store_remove(surf_tree_store, &iter);
      }
    break;
  }
}

/*****************************************************/
/* callbacks to change the morphology type displayed */
/*****************************************************/
void bfdh_morph(struct model_pak *data)
{
change_morph_type(data, DHKL);
}
void equn_morph(struct model_pak *data)
{
change_morph_type(data, EQUIL_UN);
}
void grun_morph(struct model_pak *data)
{
change_morph_type(data, GROWTH_UN);
}
void eqre_morph(struct model_pak *data)
{
change_morph_type(data, EQUIL_RE);
}
void grre_morph(struct model_pak *data)
{
change_morph_type(data, GROWTH_RE);
}

/***********************************/
/* shift energy value modification */
/***********************************/
void shift_commit(GtkWidget *w, gpointer dummy)
{
GtkTreeIter iter;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct model_pak *model;
struct plane_pak *plane;
struct shift_pak *shift;

/* get treemodel */
treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(surf_tree_view));
if (!treemodel)
  return;
/* get selection */
if (!(selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(surf_tree_view))))
  return;
/* get selected iterator */
if (!gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  return;

gtk_tree_model_get(treemodel, &iter, SURF_MODEL, &model, -1);
gtk_tree_model_get(treemodel, &iter, SURF_PLANE, &plane, -1);
gtk_tree_model_get(treemodel, &iter, SURF_SHIFT, &shift, -1);

if (!model || !plane || !shift)
  return;

shift->esurf[0] = str_to_float(gtk_entry_get_text(GTK_ENTRY(surf_morph_energy[0])));
shift->eatt[0] = str_to_float(gtk_entry_get_text(GTK_ENTRY(surf_morph_energy[1])));
shift->esurf[1] = str_to_float(gtk_entry_get_text(GTK_ENTRY(surf_morph_energy[2])));
shift->eatt[1] = str_to_float(gtk_entry_get_text(GTK_ENTRY(surf_morph_energy[3])));

fn_surf_set_shift(&iter, shift);

if (model->id == MORPH)
  {
  update_plane_energy(plane, model);
  change_morph_type(model, model->morph_type);
  }
}

/************************************************/
/* added ranked faces based on dhkl to the list */
/************************************************/
#define DEBUG_ADD_RANKED_FACES 0
void add_ranked_faces(GtkWidget *w, gpointer dummy)
{
struct model_pak *data;
GSList *list;

/* checks */
data = model_ptr(surfdata.surface.model, RECALL);
if (!data)
  return;
if (data->periodic != 3)
  return;

#if DEBUG_ADD_RANKED_FACES
printf("   Maximum faces: %f\n", rank_value);
#endif

list = get_ranked_faces((gint) rank_value, 0.0, data);

data->planes = g_slist_concat(data->planes, list);

fn_graft_plane_list(list, data);
}

/*****************************************/
/* import a previous surface calculation */
/*****************************************/
void import_planes(gchar *filename)
{
GSList *plist;
struct model_pak *data;
struct plane_pak *pdata;

data = model_ptr(surfdata.surface.model, RECALL);
if (!data)
  return;
if (data->periodic != 3)
  {
  close_dialog_type(FILE_SELECT);
  show_text(ERROR, "Source structure is not 3D periodic.");
  return;
  }

/* get the planes */
if (load_planes(filename, data))
  {
  show_text(ERROR, "Bad planes file.");
  return;
  }

plist = data->planes;
while (plist != NULL)
  {
  pdata = (struct plane_pak *) plist->data;
 
/* if shift list is empty (primary plane only!) */
/* then add the result of a calc valid shifts */
/* otherwise verify the supplied shift */
  if (pdata->primary)
    {
    if (pdata->shifts)
      test_valid_shifts(pdata, data);
    else
      calc_valid_shifts(data, pdata);
    }

  plist = g_slist_next(plist);
  }

/* clean up */
close_dialog_type(FILE_SELECT);

fn_graft_plane_list(data->planes, data);
}

/***********************************************/
/* handle import/export file selection request */
/***********************************************/
void import_dialog(void)
{
file_dialog("Import planes (morphology file)", NULL, (gpointer) import_planes, MORPH);
}
void export_dialog(void)
{
file_dialog("Export planes (morphology file)", NULL, (gpointer) export_planes, MORPH);
}

/**********************************/
/* tree selection change callback */
/**********************************/
static void cb_surf_tree_selection_changed(GtkTreeSelection *selection, gpointer data)
{
gint i, j;
gchar *text;
GtkTreeIter iter;
GtkTreeModel *treemodel;
struct model_pak *model=NULL;
struct plane_pak *plane=NULL;
struct shift_pak *shift=NULL;

if (gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  {
/* get row data */
  gtk_tree_model_get(treemodel, &iter, SURF_MODEL, &model, -1);

  g_assert(model != NULL);
  gtk_tree_model_get(treemodel, &iter, SURF_PLANE, &plane, -1);
  gtk_tree_model_get(treemodel, &iter, SURF_SHIFT, &shift, -1);

/* disallow parent selection? */
  if (model->id == MORPH)
    {
    j=0;
    for (i=SURF_ESURF_UNRE ; i<=SURF_EATT_RE ; i++)
      {
      gtk_tree_model_get(treemodel, &iter, i, &text, -1);
      if (text)
        gtk_entry_set_text(GTK_ENTRY(surf_morph_energy[j]), text);
      j++;
      }
    }
  else
    {
    if (plane)
      {
      ARR3SET(surfdata.surface.miller, plane->index);
      }
    if (shift)
      {
      surfdata.surface.shift = shift->shift;
      surfdata.surface.region[0] = shift->region[0];
      surfdata.surface.region[2] = shift->region[1];
      }
/* update widgets */
    gtksh_relation_update(model);
    }
  }
}

/************************************/
/* surface creation/cut elimination */
/************************************/
void surface_dialog(GtkWidget *w, struct model_pak *data)
{
gint i, j, id;
gchar *title;
GtkWidget *hbox1, *hbox, *vbox1, *vbox2, *vbox3, *vbox, *frame, *scr_win;
GtkWidget *label, *button, *spin;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;
GtkTreeSelection *select;
struct dialog_pak *surfwin;

/* checks */
g_return_if_fail(data != NULL);
if (data->periodic != 3 && data->id != MORPH)
  return;
if (fabs(data->gulp.energy) < FRACTION_TOLERANCE && data->id != MORPH)
  {
  show_text(WARNING, "Suspicious total energy. Has it been properly calculated?\n");
  }

/* force single pt as default */
data->gulp.run = E_SINGLE;

/* init the current surface */
all_planes = FALSE;
template_model(&surfdata);
surfdata.surface.model = data->number;
surfdata.surface.shift = data->surface.shift;
surfdata.surface.region[0] = data->surface.region[0];
surfdata.surface.region[1] = data->surface.region[1];
ARR3SET(surfdata.surface.miller, data->surface.miller);

/* new dialog */
if ((id = request_dialog(sysenv.active, GENSURF)) < 0)
  return;
surfwin = &sysenv.dialog[id];
surfwin->win = gtk_dialog_new();

if (data->id == MORPH)
  title = g_strdup_printf("%s morphology", data->basename);
else
  {
  title = g_strdup_printf("%s surfaces", data->basename);

/* enforce whole molecules */
  unfragment(data);
  init_objs(REFRESH_COLOUR, data);
  init_objs(CENT_COORDS, data);
  }

gtk_window_set_title(GTK_WINDOW (surfwin->win), title);
g_signal_connect(GTK_OBJECT(surfwin->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
gtk_window_set_default_size(GTK_WINDOW(surfwin->win), 800, 600);


/* main vbox */
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(GTK_DIALOG(surfwin->win)->vbox), vbox);

/* hbox for split pane */
hbox1 = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(vbox), hbox1, TRUE, TRUE, 10);

if (data->id == MORPH)
  {
/* left vbox */
  vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
  gtk_box_pack_start(GTK_BOX(hbox1), vbox1, FALSE, FALSE, PANEL_SPACING);

  frame = gtk_frame_new("Morphology type");
  gtk_box_pack_start(GTK_BOX(vbox1),frame,FALSE,FALSE,0);
 
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);
 
/* display type */
/* TODO - for loop ??? */
  new_radio_group(0, vbox, TT);
  button = add_radio_button("BFDH", (gpointer) bfdh_morph, data);
  if (data->morph_type == DHKL)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
 
  button = add_radio_button("Equilibrium unrelaxed", (gpointer) equn_morph, data);
  if (data->morph_type == EQUIL_UN)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
 
  button = add_radio_button("Growth unrelaxed", (gpointer) grun_morph, data);
  if (data->morph_type == GROWTH_UN)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
 
  button = add_radio_button("Equilibrium relaxed", (gpointer) eqre_morph, data);
  if (data->morph_type == EQUIL_RE)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
 
  button = add_radio_button("Growth relaxed", (gpointer) grre_morph, data);
  if (data->morph_type == GROWTH_RE)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
 
 
/* editable shift values */
  frame = gtk_frame_new("Shift values");
  gtk_box_pack_start(GTK_BOX(vbox1),frame,FALSE,FALSE,0);

  vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
 
  j=0;
  for (i=SURF_ESURF_UNRE ; i<=SURF_EATT_RE ; i++)
    {
    hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    switch (i)
      {
      case SURF_ESURF_UNRE:
        label = gtk_label_new("Esurf (unrelaxed)");
        break;
      case SURF_EATT_UNRE:
        label = gtk_label_new("Eatt (unrelaxed)");
        break;
      case SURF_ESURF_RE:
        label = gtk_label_new("Esurf (relaxed)");
        break;
      case SURF_EATT_RE:
      default:
        label = gtk_label_new("Eatt (relaxed)");
        break;
      }
/* row */
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    surf_morph_energy[j] = gtk_entry_new();
    gtk_box_pack_end(GTK_BOX(hbox), surf_morph_energy[j], FALSE, FALSE, 0);
    gtk_widget_set_size_request(surf_morph_energy[j], 9*sysenv.gfontsize, -1);
    j++;
    }
  }
else
  {
/* left vbox */
vbox = gtk_vbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(hbox1), vbox, FALSE, FALSE, 10);

frame = gtk_frame_new("Surface");
gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,0);
vbox3 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox3);
gtk_container_set_border_width(GTK_CONTAINER(vbox3), PANEL_SPACING);

/* two vboxes; one for labels, the other for the spinners */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox3), hbox, TRUE, FALSE, 0);
vbox1 = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, FALSE, 0);
vbox2 = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, FALSE, 0);

/* labels */
label = gtk_label_new("Miller");
gtk_box_pack_start(GTK_BOX(vbox1), label, TRUE, FALSE, 0);
label = gtk_label_new("Shift");
gtk_box_pack_start(GTK_BOX(vbox1), label, TRUE, FALSE, 0);
label = gtk_label_new("Depths");
gtk_box_pack_start(GTK_BOX(vbox1), label, TRUE, FALSE, 0);

/* miller */
hbox = gtk_hbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox2), hbox, TRUE, FALSE, 0);

gtksh_direct_spin(NULL, 
                 &surfdata.surface.miller[0], -99, 99, 1,
                  NULL, NULL, hbox);

gtksh_direct_spin(NULL, 
                 &surfdata.surface.miller[1], -99, 99, 1,
                  NULL, NULL, hbox);

gtksh_direct_spin(NULL, 
                 &surfdata.surface.miller[2], -99, 99, 1,
                  NULL, NULL, hbox);

/* shift */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox2), hbox, TRUE, FALSE, 0);
spin = gtksh_direct_spin(NULL, 
                        &surfdata.surface.shift, 0.0, 1.0, 0.05,
                         NULL, NULL, hbox);
gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 4);
/* make the spin box big enough to show all 4 dp (GTK screws it up) */
gtk_widget_set_size_request(spin, 6*sysenv.gfontsize, -1);

/* regions */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(vbox2), hbox, TRUE, FALSE, 0);

gtksh_direct_spin(NULL, 
                 &surfdata.surface.region[0], 0, 99, 1,
                  NULL, NULL, hbox);

gtksh_direct_spin(NULL, 
                 &surfdata.surface.region[1], 0, 99, 1,
                  NULL, NULL, hbox);

/* create this surface */
gtksh_button(" Create ", cb_surf_create, (gpointer) data, hbox, TF);

/* action labels */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, PANEL_SPACING);

vbox1 = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox1);
gtk_container_set_border_width(GTK_CONTAINER(vbox1), PANEL_SPACING);

gtksh_button_label("Add the current surface", " > ",
                   fn_add_plane_with_shift, NULL,
                   vbox1);
gtksh_button_label("Add all valid shifts ", " > ",
                   fn_add_valid_shifts, NULL,
                   vbox1);

/* frame for adding a specified number of Dhkl ranked faces */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(hbox)), PANEL_SPACING);
gtksh_direct_spin("Add ", &rank_value, 1, 50, 1, NULL, NULL, hbox);
label = gtk_label_new(" Dhkl ranked faces ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
gtksh_button(" > ", add_ranked_faces, NULL, hbox, LB);

/* frame for file ops */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, PANEL_SPACING);
vbox1 = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox1);
gtk_container_set_border_width(GTK_CONTAINER(vbox1), PANEL_SPACING);
gtksh_button_label("Import planes from file", " > ",
                   import_dialog, NULL,
                   vbox1);
gtksh_button_label("Export planes to file", " < ",
                   export_dialog, NULL,
                   vbox1);
gtksh_button_label("Create morphology from planes", " < ",
                   make_morph, NULL,
                   vbox1);

/* frame for convergence check buttons */
  frame = gtk_frame_new("Convergence");
  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,PANEL_SPACING);
  vbox1 = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame), vbox1);
  gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox1)), PANEL_SPACING);
  gtksh_direct_check("attachment energy based", &data->surface.converge_eatt, NULL, NULL, vbox1);
  gtksh_direct_check("converge region 1", &data->surface.converge_r1, NULL, NULL, vbox1);
  gtksh_direct_check("converge region 2", &data->surface.converge_r2, NULL, NULL, vbox1);

/* frame for option check buttons */
  frame = gtk_frame_new("Options");
  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,PANEL_SPACING);
  vbox1 = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame), vbox1);
  gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox1)), PANEL_SPACING);
  gtksh_direct_check("operate on entire list", &all_planes, NULL, NULL, vbox1);
  gtksh_direct_check("allow polar surfaces", &data->surface.include_polar, NULL, NULL, vbox1);
  gtksh_direct_check("allow bond cleaving", &data->surface.ignore_bonding, NULL, NULL, vbox1);
  gtksh_direct_check("preserve depth periodicity", &data->surface.true_cell, NULL, NULL, vbox1);
  }

/* right vbox */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 0);

/* valid cut listing */
frame = gtk_frame_new("Current surface list");
gtk_box_pack_start(GTK_BOX(vbox1),frame,TRUE,TRUE,0);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* scrolled model pane */
scr_win = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win),
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), scr_win, TRUE, TRUE, 0);

/* planes storage */
surf_tree_store = gtk_tree_store_new(SURF_NCOLS, G_TYPE_STRING,
                                     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                                     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                                     G_TYPE_POINTER, G_TYPE_POINTER, G_TYPE_POINTER);

/* planes viewing widget */
surf_tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(surf_tree_store));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scr_win), surf_tree_view);

/* set up column renderers */
for (i=0 ; i<=SURF_GNORM ; i++)
  {
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(titles[i], renderer, "text", i, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(surf_tree_view), column);
  }

/* setup the selection handler */
select = gtk_tree_view_get_selection(GTK_TREE_VIEW(surf_tree_view));
gtk_tree_selection_set_mode(select, GTK_SELECTION_BROWSE);
g_signal_connect(G_OBJECT(select), "changed",
                 G_CALLBACK(cb_surf_tree_selection_changed),
                 NULL);

fn_graft_plane_list(data->planes, data);


if (data->id == MORPH)
  {
/* shift operation button row */
  hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtksh_button("Commit ", shift_commit, NULL, hbox, TT);
  gtksh_button("Delete ", cb_surf_delete_selected, NULL, hbox, TT);
  }
else
  {
/* shift operation button row */
  hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtksh_button("Create surface", cb_surf_task, (gpointer) MAKE_FACES, hbox, TT);
  gtksh_button("Converge regions", cb_surf_task, (gpointer) CONV_REGIONS, hbox, TT);
  gtksh_button("Calculate energy", cb_surf_task, (gpointer) CALC_ENERGY, hbox, TT);
  gtksh_button("Calculation setup", energetics_dialog, data, hbox, TT);
/* shift operation button row */
  hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtksh_button("Find valid shifts", cb_surf_task, (gpointer) CALC_SHIFTS, hbox, TT);
  gtksh_button("Remove selected", cb_surf_delete_selected, NULL, hbox, TT);
  gtksh_button("Remove invalid", cb_prune_surf_list, GINT_TO_POINTER(INVALID_SHIFTS), hbox, TT);
  gtksh_button("Remove all", cb_prune_surf_list, GINT_TO_POINTER(ALL_SHIFTS), hbox, TT);
  }

/* terminating button */
gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(surfwin->win)->action_area);

/* done */
gtk_widget_show_all(surfwin->win);
}

