/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "spatial.h"
#include "select.h"
#include "interface.h"
#include "gtkshorts.h"
#include "opengl.h"

/* main structure */
extern struct sysenv_pak sysenv;

/* handle all the vector/plane creation, intersections etc. */
/* add all the vector normalization/crossprod etc. stuff as well */

/***********************************************************************/
/* create a spatial vector or plane, given a position and offset point */
/***********************************************************************/
/* direction sense: a->b */
void fn_make_spatial(gint type, gdouble *a, gdouble *b, struct model_pak *model)
{
struct spatial_pak *spatial;
struct vec_pak *vec;

switch (type)
  {
  case SPATIAL_VECTOR:
    spatial = g_malloc(sizeof(struct spatial_pak));
    model->spatial = g_slist_prepend(model->spatial, spatial);
    spatial->type = SPATIAL_VECTOR;
    spatial->data = NULL;

    vec = g_malloc(sizeof(struct vec_pak));
    ARR3SET(vec->x, a);
    ARR3ADD(vec->x, b);
    spatial->data = g_slist_prepend(spatial->data, vec);

    vec = g_malloc(sizeof(struct vec_pak));
    ARR3SET(vec->x, a);
    spatial->data = g_slist_prepend(spatial->data, vec);
    break;

  default:
    printf("Not implemented yet.\n");
  }
}

/***********************************************/
/* compute best fit plane for a list of atoms */
/**********************************************/
#define DEBUG_COMPUTE_PLANE 0
void compute_plane(gdouble *res, GList *cores)
{
gdouble tmp[3], vec1[3], vec2[3];
struct core_pak *core1, *core2, *core3;

/* FIXME - this only uses the 1st 3 points, in future - an average for n pts */
/* http://www.geocities.com/CollegePark/Campus/1278/JBook/LSQ_Plane.html */
/* shouldn't matter when this routine is properly implemented, as we'll go */
/* through the atom_list in the usual fashoin & terminate when it is NULL, */
/* not when it's data ptr is NULL */ 
g_assert(cores != NULL);
g_assert(g_list_length(cores) > 2);

core1 = g_list_nth_data(cores, 0);
core2 = g_list_nth_data(cores, 1);
core3 = g_list_nth_data(cores, 2);

#if DEBUG_COMPUTE_PLANE
printf("computing normal from atoms: %p, %p, %p\n", core1, core2, core3);
#endif

ARR3SET(tmp, core1->x);
ARR3SET(vec1, core2->x);
ARR3SET(vec2, core3->x);

/* make vectors 0->1 & 0->2 */
ARR3SUB(vec1, tmp);
ARR3SUB(vec2, tmp);

/* get normal */
crossprod(res, vec1, vec2);
}

/*********************************************/
/* remove all spatials of the specified type */
/*********************************************/
void spatial_delete_type(gint type, struct model_pak *data)
{
GSList *list;
struct spatial_pak *spatial;

/* cycle through all vector type spatials */
list = data->spatial;
while (list)
  {
  spatial = (struct spatial_pak *) list->data;
  list = g_slist_next(list);

/* delete the spatial if of supplied type */
  if (spatial->type == type)
    {
    free_slist(spatial->data);
    data->spatial = g_slist_remove(data->spatial, spatial);
    g_free(spatial);
    }
  }
}

/*************************/
/* delete number spatial */
/*************************/
void delete_spatial(gint i, struct model_pak *data)
{
gpointer *ptr;
GSList *list;

ptr = g_slist_nth_data(data->spatial, i);
if (ptr)
  {
/* remove the spatial object's data */
  list = ((struct spatial_pak *) ptr)->data;
  g_slist_free(list);
/* remove the spatial object itself */
  data->spatial = g_slist_remove(data->spatial, ptr);
  }
else
  printf("bad spatial.\n");
}

/*************************************************/
/* delete vectors with supplied tail coordinates */
/*************************************************/
void delete_vector_at(gdouble *r, struct model_pak *data)
{
gdouble x[3];
GSList *list1, *list2;
struct spatial_pak *spatial;
struct vec_pak *vec;

/* cycle through all vector type spatials */
list1 = data->spatial;
while (list1)
  {
  spatial = (struct spatial_pak *) list1->data;
  list1 = g_slist_next(list1);
  if (spatial->type != SPATIAL_VECTOR)
    continue;
  list2 = spatial->data;

/* compare vector tail position to input */
  vec = (struct vec_pak *) list2->data;
  ARR3SET(x, vec->rx);
  ARR3SUB(x, r);

/* 2D compare */
  x[2] = 0.0;
  if (VEC3MAGSQ(x) < POSITION_TOLERANCE)
    {
/* delete the spatial */
    free_slist(list2);
    data->spatial = g_slist_remove(data->spatial, spatial);
    g_free(spatial);
    }
  }
}

/**************************************/
/* spatial point definition primitive */
/**************************************/
void new_spatial_point(gint type, struct core_pak *core, struct model_pak *data)
{
gint reset=0;
gdouble plane[4], vec[3], ortho1[3], ortho2[3];
GList *list;
static GList *cores=NULL;
struct core_pak *core1, *core2;
struct vec_pak *vec1, *vec2;
struct spatial_pak *spatial;

/* add the new atom */
cores = g_list_append(cores, core);
select_add_core(core, data);
data->state++;

switch(type)
  {
  case DEFINE_VECTOR:
    if (data->state==2)
      {
/* new vector */
      spatial = g_malloc(sizeof(struct spatial_pak));
      spatial->type = SPATIAL_VECTOR;
/* two end points */
      vec1 = g_malloc(sizeof(struct vec_pak));
      vec2 = g_malloc(sizeof(struct vec_pak));
      core1 = (struct core_pak *) g_list_nth_data(cores, 0);
/* get the (lattice space) separation vector */
      ARR3SET(vec1->x, core1->x);
      ARR3SET(vec2->x, core->x);
      ARR3SUB(vec2->x, core1->x);

/* convert to real space & scale */
      vecmat(data->latmat, vec2->x);

/* useful debugging tool */
/*
P3VEC("true vec: ", vec2->x);
*/

      normalize(vec2->x, 3);
      VEC3MUL(vec2->x, 0.7);
/* convert back to lattice space */
      vecmat(data->ilatmat, vec2->x);
      ARR3ADD(vec2->x, vec1->x);
/* create & init the spatial object */
      spatial->data = NULL;
      spatial->data = g_slist_append(spatial->data, vec1);
      spatial->data = g_slist_append(spatial->data, vec2);
      data->spatial = g_slist_append(data->spatial, spatial);
/* drawing update */
      init_objs(REDO_COORDS, data);
      redraw_canvas(SINGLE);
      reset++;
      }
    break;

  case DEFINE_PLANE:
    if (data->state == 3)
      {
/* new plane */
      spatial = g_malloc(sizeof(struct spatial_pak));
      spatial->type = SPATIAL_PLANE;

/* compute bounding points */
      compute_plane(plane, cores);

      core1 = (struct core_pak *) g_list_nth_data(cores, 0);
      core2 = (struct core_pak *) g_list_nth_data(cores, 1);
       
/* vector from first to second point (NB: normalized in cartesian space) */
      ARR3SET(ortho1, core2->x);
      ARR3SUB(ortho1, core1->x);
      vecmat(data->latmat, ortho1);
      normalize(ortho1, 3);
      vecmat(data->ilatmat, ortho1);

/* get orthogonal vector */
      crossprod(ortho2, plane, ortho1);
      vecmat(data->latmat, ortho2);
      normalize(ortho2, 3);
      vecmat(data->ilatmat, ortho2);

/* create & init the spatial object */
      spatial->data = NULL;
      data->spatial = g_slist_append(data->spatial, spatial);

/* define 4 points for the plane (in general, will be a polygon) */
      ARR3SET(vec, core1->x);

      vec1 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(vec1->x, vec);    
      ARR3ADD(vec1->x, ortho1);
      ARR3ADD(vec1->x, ortho2);
      spatial->data = g_slist_append(spatial->data, vec1);

      vec1 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(vec1->x, vec);    
      ARR3ADD(vec1->x, ortho1);
      ARR3SUB(vec1->x, ortho2);
      spatial->data = g_slist_append(spatial->data, vec1);

      vec1 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(vec1->x, vec);    
      ARR3SUB(vec1->x, ortho1);
      ARR3SUB(vec1->x, ortho2);
      spatial->data = g_slist_append(spatial->data, vec1);

      vec1 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(vec1->x, vec);    
      ARR3SUB(vec1->x, ortho1);
      ARR3ADD(vec1->x, ortho2);
      spatial->data = g_slist_append(spatial->data, vec1);

/* drawing update */
      init_objs(REDO_COORDS, data);
      redraw_canvas(SINGLE);
      reset++;
      }
    break;
  }

/* completed an object */
if (reset)
  {
  for (list=cores ; list ; list=g_list_next(list))
    {
    core1 = (struct core_pak *) list->data;
    select_del_core(core1, data);
    }
  g_list_free(cores);
  cores=NULL;
  data->state = 0;
  }
}

