/*
 *      plugindata.h - this file is part of Geany, a fast and lightweight IDE
 *
 *      Copyright 2007-2008 Enrico Tröger <enrico(dot)troeger(at)uvena(dot)de>
 *      Copyright 2007-2008 Nick Treleaven <nick(dot)treleaven(at)btinternet(dot)com>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *      MA 02110-1301, USA.
 *
 * $Id: plugindata.h 2496 2008-04-17 17:48:12Z eht16 $
 */

/**
 * @file plugindata.h
 * This file defines the plugin API, the interface between Geany and its plugins.
 * For detailed documentation of the plugin system please read the plugin
 * API documentation.
 **/


#ifndef PLUGIN_H
#define PLUGIN_H

/* The API version should be incremented whenever any plugin data types below are
 * modified or appended to. */
static const gint api_version = 53;

/* The ABI version should be incremented whenever existing fields in the plugin
 * data types below have to be changed or reordered. It should stay the same if fields
 * are only appended, as this doesn't affect existing fields. */
static const gint abi_version = 24;

/** Check the plugin can be loaded by Geany.
 * This performs runtime checks that try to ensure:
 * - Geany ABI data types are compatible with this plugin.
 * - Geany sources provide the required API for this plugin. */
#define PLUGIN_VERSION_CHECK(api_required) \
	gint version_check(gint abi_ver) \
	{ \
		if (abi_ver != abi_version) \
			return -1; \
		if (api_version < (api_required)) \
			return (api_required); \
		else return 0; \
	}


/** Plugin info structure to hold basic information about a plugin.
 *  Should only be set with PLUGIN_INFO. */
typedef struct PluginInfo
{
	/** The name of the plugin. */
	gchar	*name;
	/** The description of the plugin. */
	gchar	*description;
	/** The version of the plugin. */
	gchar	*version;
	/** The author of the plugin. */
	gchar	*author;
	/** Reserved for later use. */
	gpointer reserved2;
}
PluginInfo;

#include <string.h>

/** Set the plugin name and some other basic information about a plugin. */
#define PLUGIN_INFO(p_name, p_description, p_version, p_author) \
	PluginInfo *info(void) \
	{ \
		static PluginInfo p_info; \
		\
		memset(&p_info, 0, sizeof(PluginInfo)); \
		p_info.name = (p_name); \
		p_info.description = (p_description); \
		p_info.version = (p_version); \
		p_info.author = (p_author); \
		return &p_info; \
	}


/** Declare and initialise a keybinding group.
 * @code KeyBindingGroup plugin_key_group[1]; @endcode
 * You must then set the @c plugin_key_group::keys[] entries for the group in init().
 * The @c plugin_key_group::label field is set by Geany after @c init()
 * is called, to the name of the plugin.
 * @param group_name A unique group name (without quotes) to be used in the
 * configuration file, such as @c html_chars.
 * @param key_count	The number of keybindings the group will hold. */
#define PLUGIN_KEY_GROUP(group_name, key_count) \
	static KeyBinding plugin_keys[key_count]; \
	\
	/* We have to declare plugin_key_group as a single element array.
	 * Declaring as a pointer to a struct doesn't work with g_module_symbol(). */ \
	KeyBindingGroup plugin_key_group[1] = \
	{ \
		{G_STRINGIFY(group_name), NULL, key_count, plugin_keys} \
	};


/** callback array entry */
typedef struct GeanyCallback
{
	/** The name of signal, must be an existing signal. For a list of available signals,
	 *  please see the @link signals Signal documentation @endlink. */
	gchar		*signal_name;
	/** A callback function which is called when the signal is emitted. */
	GCallback	callback;
	/** Set to TRUE to connect your handler with g_signal_connect_after(). */
	gboolean	after;
	/** The user data passed to the signal handler. */
	gpointer	user_data;
}
GeanyCallback;



/** Flags to be set by plugins in PluginFields struct. */
typedef enum
{
	/** Whether a plugin's menu item should be disabled when there are no open documents */
	PLUGIN_IS_DOCUMENT_SENSITIVE	= 1 << 0
}
PluginFlags;

/** Fields set and owned by the plugin. */
/* Note: Remember to increment api_version (and abi_version if necessary) when making changes. */
typedef struct PluginFields
{
	/** Bitmask of PluginFlags. */
	PluginFlags	flags;
	/** Pointer to a plugin's menu item which will be automatically enabled/disabled when there
	 *  are no open documents and PLUGIN_IS_DOCUMENT_SENSITIVE is set.
	 *  This is required if using PLUGIN_IS_DOCUMENT_SENSITIVE, ignored otherwise */
	GtkWidget	*menu_item;
}
PluginFields;


/** This contains pointers to global variables and functions owned by Geany for plugins to use.
 *  Variables and functions will be appended when needed by plugin authors. */
typedef struct GeanyData
{
	GeanyApp	*app;					/**< Geany application data fields */
	GtkWidget	*tools_menu;			/**< Most plugins should add menu items to the Tools menu only */
	GArray		*doc_array;				/**< Dynamic array of document pointers */
	struct filetype		**filetypes;
	struct GeanyPrefs	*prefs;
	struct EditorPrefs	*editor_prefs;	/**< Editor settings */
	struct BuildInfo	*build_info;	/**< Current build information */

	struct DocumentFuncs		*documents;		/**< See document.h */
	struct ScintillaFuncs		*sci;			/**< See sciwrappers.h */
	struct TemplateFuncs		*templates;		/**< See templates.h */
	struct UtilsFuncs			*utils;			/**< See utils.h */
	struct UIUtilsFuncs			*ui;			/**< See ui_utils.h */
	struct SupportFuncs			*support;		/**< See support.h */
	struct DialogFuncs			*dialogs;		/**< See dialogs.h */
	struct MsgWinFuncs			*msgwindow;		/**< See msgwindow.h */
	struct EncodingFuncs		*encoding;		/**< See encodings.h */
	struct KeybindingFuncs		*keybindings;	/**< See keybindings.h */
	struct TagManagerFuncs		*tm;			/**< See tagmanager/include */
	struct SearchFuncs			*search;		/**< See search.h */
	struct HighlightingFuncs	*highlighting;	/**< See highlighting.h */
	struct FiletypeFuncs		*filetype;		/**< See filetypes.h */
	struct NavQueueFuncs        *navqueue;      /**< See navqueue.h */
}
GeanyData;


/* For more info about these functions, see the main source code.
 * E.g. for GeanyData::documents->new_file(), see document_new_file() in document.c. */


/* See document.h */
typedef struct DocumentFuncs
{
	gint	(*new_file) (const gchar *filename, struct filetype *ft, const gchar *text);
	gint	(*get_cur_idx) (void);
	gint	(*get_n_idx) (guint i);
	gint	(*find_by_filename) (const gchar *filename, gboolean is_tm_filename);
	struct document* (*get_current) (void);
	gboolean (*save_file) (gint idx, gboolean force);
	gint	(*open_file) (const gchar *locale_filename, gboolean readonly,
			struct filetype *ft, const gchar *forced_enc);
	void	(*open_files) (const GSList *filenames, gboolean readonly, struct filetype *ft,
			const gchar *forced_enc);
	gboolean (*remove) (guint page_num);
	gboolean (*reload_file) (gint idx, const gchar *forced_enc);
	void	(*set_encoding) (gint idx, const gchar *new_encoding);
	void	(*set_text_changed) (gint idx);
	void	(*set_filetype) (gint idx, filetype *type);
}
DocumentFuncs;


struct _ScintillaObject;

/* See sciwrappers.h */
typedef struct ScintillaFuncs
{
	long int (*send_message) (struct _ScintillaObject* sci, unsigned int iMessage,
			long unsigned int wParam, long int lParam);
	void	(*send_command) (struct _ScintillaObject* sci, gint cmd);

	void	(*start_undo_action) (struct _ScintillaObject* sci);
	void	(*end_undo_action) (struct _ScintillaObject* sci);
	void	(*set_text) (struct _ScintillaObject *sci, const gchar *text);
	void	(*insert_text) (struct _ScintillaObject *sci, gint pos, const gchar *text);
	void	(*get_text) (struct _ScintillaObject *sci, gint len, gchar* text);
	gint	(*get_length) (struct _ScintillaObject *sci);
	gint	(*get_current_position) (struct _ScintillaObject *sci);
	void	(*set_current_position) (struct _ScintillaObject* sci, gint position, gboolean scroll_to_caret);
	gint	(*get_col_from_position) (struct _ScintillaObject* sci, gint position);
	gint	(*get_line_from_position) (struct _ScintillaObject* sci, gint position);
	gint	(*get_position_from_line) (struct _ScintillaObject* sci, gint line);
	void	(*replace_sel) (struct _ScintillaObject* sci, const gchar* text);
	void	(*get_selected_text) (struct _ScintillaObject* sci, gchar* text);
	gint	(*get_selected_text_length) (struct _ScintillaObject* sci);
	gint	(*get_selection_start) (struct _ScintillaObject* sci);
	gint	(*get_selection_end) (struct _ScintillaObject* sci);
	gint	(*get_selection_mode) (struct _ScintillaObject* sci);
	void	(*set_selection_mode) (struct _ScintillaObject* sci, gint mode);
	void	(*set_selection_start) (struct _ScintillaObject* sci, gint position);
	void	(*set_selection_end) (struct _ScintillaObject* sci, gint position);
	void	(*get_text_range) (struct _ScintillaObject* sci, gint start, gint end, gchar*text);
	gchar*	(*get_line) (struct _ScintillaObject* sci, gint line_num);
	gint	(*get_line_length) (struct _ScintillaObject* sci, gint line);
	gint	(*get_line_count) (struct _ScintillaObject* sci);
	gboolean (*get_line_is_visible) (struct _ScintillaObject* sci, gint line);
	void	(*ensure_line_is_visible) (struct _ScintillaObject* sci, gint line);
	void	(*scroll_caret) (struct _ScintillaObject* sci);
	gint	(*find_bracematch) (struct _ScintillaObject* sci, gint pos);
	gint	(*get_style_at) (struct _ScintillaObject *sci, gint position);
	gchar	(*get_char_at) (struct _ScintillaObject *sci, gint pos);
	gint	(*get_current_line) (struct _ScintillaObject *sci);
}
ScintillaFuncs;


/* See templates.h */
typedef struct TemplateFuncs
{
	gchar*		(*get_template_fileheader) (gint filetype_idx, const gchar *fname);
}
TemplateFuncs;


/* See utils.h */
typedef struct UtilsFuncs
{
	gboolean	(*str_equal) (const gchar *a, const gchar *b);
	gboolean	(*string_replace_all) (GString *haystack, const gchar *needle,
				 const gchar *replacement);
	GSList*		(*get_file_list) (const gchar *path, guint *length, GError **error);
	gint		(*write_file) (const gchar *filename, const gchar *text);
	gchar*		(*get_locale_from_utf8) (const gchar *utf8_text);
	gchar*		(*get_utf8_from_locale) (const gchar *locale_text);
	gchar*		(*remove_ext_from_filename) (const gchar *filename);
	gint		(*mkdir) (const gchar *path, gboolean create_parent_dirs);
	gboolean	(*get_setting_boolean) (GKeyFile *config, const gchar *section, const gchar *key,
				 const gboolean default_value);
	gint		(*get_setting_integer) (GKeyFile *config, const gchar *section, const gchar *key,
				 const gint default_value);
	gchar*		(*get_setting_string) (GKeyFile *config, const gchar *section, const gchar *key,
				 const gchar *default_value);
	gboolean	(*spawn_sync) (const gchar *dir, gchar **argv, gchar **env, GSpawnFlags flags,
				 GSpawnChildSetupFunc child_setup, gpointer user_data, gchar **std_out,
				 gchar **std_err, gint *exit_status, GError **error);
	gboolean	(*spawn_async) (const gchar *dir, gchar **argv, gchar **env, GSpawnFlags flags,
				 GSpawnChildSetupFunc child_setup, gpointer user_data, GPid *child_pid,
				 GError **error);
}
UtilsFuncs;


/* See ui_utils.h */
typedef struct UIUtilsFuncs
{
	GtkWidget*	(*dialog_vbox_new) (GtkDialog *dialog);
	GtkWidget*	(*frame_new_with_alignment) (const gchar *label_text, GtkWidget **alignment);

	/* set_statusbar() also appends to the message window status tab if log is TRUE. */
	void		(*set_statusbar) (gboolean log, const gchar *format, ...) G_GNUC_PRINTF (2, 3);

	void		(*table_add_row) (GtkTable *table, gint row, ...) G_GNUC_NULL_TERMINATED;
	GtkWidget*	(*path_box_new) (const gchar *title, GtkFileChooserAction action, GtkEntry *entry);
	GtkWidget*	(*button_new_with_image) (const gchar *stock_id, const gchar *text);
}
UIUtilsFuncs;


/* See dialogs.h */
typedef struct DialogFuncs
{
	gboolean	(*show_question) (const gchar *text, ...);
	void		(*show_msgbox) (gint type, const gchar *text, ...);
	gboolean	(*show_save_as) (void);
}
DialogFuncs;


/* See support.h */
typedef struct SupportFuncs
{
	GtkWidget*	(*lookup_widget) (GtkWidget *widget, const gchar *widget_name);
}
SupportFuncs;


/* See msgwindow.h */
typedef struct MsgWinFuncs
{
	/* status_add() does not set the status bar - use ui->set_statusbar() instead. */
	void		(*status_add) (const gchar *format, ...);
	void		(*compiler_add) (gint msg_color, const gchar *format, ...) G_GNUC_PRINTF (2, 3);
}
MsgWinFuncs;


/* See encodings.h */
typedef struct EncodingFuncs
{
	gchar*			(*convert_to_utf8) (const gchar *buffer, gsize size, gchar **used_encoding);
	gchar* 			(*convert_to_utf8_from_charset) (const gchar *buffer, gsize size,
													 const gchar *charset, gboolean fast);
	const gchar*	(*get_charset_from_index) (gint idx);
}
EncodingFuncs;


struct KeyBindingGroup;
typedef void (*_KeyCallback) (guint key_id);

/* See keybindings.h */
typedef struct KeybindingFuncs
{
	void		(*send_command) (guint group_id, guint key_id);
	void		(*set_item) (struct KeyBindingGroup *group, gsize key_id,
					_KeyCallback callback, guint key, GdkModifierType mod,
					gchar *name, gchar *label, GtkWidget *menu_item);
}
KeybindingFuncs;


/* See highlighting.h */
typedef struct HighlightingFuncs
{
	const struct HighlightingStyle* (*get_style) (gint ft_id, gint style_id);
}
HighlightingFuncs;


/* See filetypes.h */
typedef struct FiletypeFuncs
{
	filetype*	(*detect_from_filename) (const gchar *utf8_filename);
	filetype*	(*get_from_uid) (gint uid);
}
FiletypeFuncs;


/* See search.h */
typedef struct SearchFuncs
{
	void		(*show_find_in_files_dialog) (const gchar *dir);
}
SearchFuncs;


/* See tagmanager/include */
typedef struct TagManagerFuncs
{
	gchar*			(*get_real_path) (const gchar *file_name);
	TMWorkObject*	(*source_file_new) (const char *file_name, gboolean update, const char *name);
	gboolean		(*workspace_add_object) (TMWorkObject *work_object);
	gboolean		(*source_file_update) (TMWorkObject *source_file, gboolean force,
					 gboolean recurse, gboolean update_parent);
	void			(*work_object_free) (gpointer work_object);
	gboolean		(*workspace_remove_object) (TMWorkObject *w, gboolean do_free, gboolean update);
}
TagManagerFuncs;

typedef struct NavQueueFuncs
{
	gboolean		(*goto_line) (gint old_idx, gint new_idx, gint line);
}NavQueueFuncs;


/* Deprecated aliases */
#ifndef GEANY_DISABLE_DEPRECATED

typedef GeanyData PluginData;	/* for compatibility with API < 7 */

#define VERSION_CHECK(api_required) \
	PLUGIN_VERSION_CHECK(api_required)

#endif

#endif
