#line 61 "../noweb/o_grips.nw"
/* gEDA - GPL Electronic Design Automation
 * gschem - gEDA Schematic Capture
 * Copyright (C) 1998-2000 Ales V. Hvezda
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */


#line 11 "../noweb/o_grips.nw"
/* DO NOT read or edit this file ! Use ../noweb/o_grips.nw instead */

#line 84 "../noweb/o_grips.nw"
#include <config.h>

#include <stdio.h>
#include <math.h>

#include <libgeda/libgeda.h>

#include "../include/globals.h"
#include "../include/prototype.h"

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>
#endif

#line 102 "../noweb/o_grips.nw"
/* Kazu on July 16, 1999 - Added these macros to simplify the code */
#define GET_BOX_WIDTH(w)                        \
        abs((w)->last_x - (w)->start_x)
#define GET_BOX_HEIGHT(w)                       \
	        abs((w)->last_y - (w)->start_y)
#define GET_BOX_LEFT(w)                         \
	        min((w)->start_x, (w)->last_x);
#define GET_BOX_TOP(w)                          \
	        min((w)->start_y, (w)->last_y);

#define GET_PICTURE_WIDTH(w)			\
	abs((w)->last_x - (w)->start_x)
#define GET_PICTURE_HEIGHT(w)			\
	(w)->pixbuf_wh_ratio == 0 ? 0 : abs((w)->last_x - (w)->start_x)/(w)->pixbuf_wh_ratio
#define GET_PICTURE_LEFT(w)				\
	min((w)->start_x, (w)->last_x);
#define GET_PICTURE_TOP(w)				\
	(w)->start_y < (w)->last_y ? (w)->start_y  : \
                                     (w)->start_y-abs((w)->last_x - (w)->start_x)/(w)->pixbuf_wh_ratio;

#line 137 "../noweb/o_grips.nw"
static int whichone_changing = -1;
static OBJECT *object_changing;


#line 156 "../noweb/o_grips.nw"
OBJECT *o_grips_search(TOPLEVEL * w_current, int x, int y, int *whichone)
{
  OBJECT *object = NULL;
  OBJECT *found = NULL;
  SELECTION *s_current;
  int size, x2size;

  if (!whichone) {
    return (NULL);
  }

  /* get the size of the grip according to zoom level */
  size = o_grips_size(w_current);
  /* size is half the width of a grip, x2size is full width */
  x2size = size * 2;

  /* skip over head */
  s_current = w_current->page_current->selection2_head->next;
  while (s_current) {
    if (s_current->selected_object) {
      object = s_current->selected_object;
      switch (object->type) {

      case (OBJ_ARC):
	/* check the grips of the arc object */
	found = o_grips_search_arc(w_current, object,
				   x, y, size, whichone);
	if (found != NULL)
	  return found;
	break;

      case (OBJ_BOX):
	/* check the grips of the box object */
	found = o_grips_search_box(w_current, object,
				   x, y, size, whichone);
	if (found != NULL)
	  return found;
	break;

      case (OBJ_PICTURE):
	/* check the grips of the picture object */
#ifndef HAS_GTK12
	found = o_grips_search_picture(w_current, object,
				       x, y, size, whichone);
	if (found != NULL)
	  return found;
#endif
	break;

      case (OBJ_CIRCLE):
	/* check the grips of the circle object */
	found = o_grips_search_circle(w_current, object,
				      x, y, size, whichone);
	if (found != NULL)
	  return found;
	break;

      case (OBJ_LINE):
      case (OBJ_PIN):
      case (OBJ_NET):
      case (OBJ_BUS):
	/* check the grips of the line object */
	/* the function is the same for line, pin, net, bus */
	found = o_grips_search_line(w_current, object,
				    x, y, size, whichone);
	if (found != NULL)
	  return found;
	break;

#if 0
	/* This code is wrong.  Continue searching even if the object */
	/* does not have grips */
      default:
	/* object type is unknown : error condition */
	return NULL;
#endif
      }
    }
    s_current = s_current->next;
  }

  return (NULL);
}

#line 787 "../noweb/o_grips.nw"
OBJECT *o_grips_search_arc(TOPLEVEL * w_current, OBJECT * o_current,
			   int x, int y, int size, int *whichone)
{
  int centerx, centery, radius, start_angle, end_angle;
  int left, top, right, bottom;
  int x2size;
  double tmp;

  centerx = o_current->arc->screen_x;
  centery = o_current->arc->screen_y;
  radius = o_current->arc->screen_width / 2;
  start_angle = o_current->arc->start_angle;
  end_angle = o_current->arc->end_angle;

  /* width/height of the grip */
  x2size = 2 * size;

  /* check the grip on the center of the arc */
  left = centerx - size;
  top = centery - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = ARC_RADIUS;
    return (o_current);
  }

  /* check the grip at the start angle of the arc */
  tmp = ((double) start_angle) * M_PI / 180;
  left = centerx + radius * cos(tmp) - size;
  top = centery - radius * sin(tmp) - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = ARC_START_ANGLE;
    return (o_current);
  }

  /* check the grip at the end angle of the arc */
  tmp = ((double) start_angle + end_angle) * M_PI / 180;
  left = centerx + radius * cos(tmp) - size;
  top = centery - radius * sin(tmp) - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = ARC_END_ANGLE;
    return (o_current);
  }

  return NULL;
}

#line 979 "../noweb/o_grips.nw"
OBJECT *o_grips_search_box(TOPLEVEL * w_current, OBJECT * o_current,
			   int x, int y, int size, int *whichone)
{
  int left, right, top, bottom;
  int x2size;

  /* width/height of the grip */
  x2size = 2 * size;

  /* inside upper left grip ? */
  left = o_current->box->screen_upper_x - size;
  top = o_current->box->screen_upper_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = BOX_UPPER_LEFT;
    return (o_current);
  }

  /* inside lower right grip ? */
  left = o_current->box->screen_lower_x - size;
  top = o_current->box->screen_lower_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = BOX_LOWER_RIGHT;
    return (o_current);
  }

  /* inside upper right grip ? */
  left = o_current->box->screen_lower_x - size;
  top = o_current->box->screen_upper_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = BOX_UPPER_RIGHT;
    return (o_current);
  }

  /* inside lower left grip ? */
  left = o_current->box->screen_upper_x - size;
  top = o_current->box->screen_lower_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = BOX_LOWER_LEFT;
    return (o_current);
  }

  return NULL;
}

#line 20 "../noweb/o_grips.nw"
#ifndef HAS_GTK12
#line 1273 "../noweb/o_grips.nw"
OBJECT *o_grips_search_picture(TOPLEVEL * w_current, OBJECT * o_current,
			       int x, int y, int size, int *whichone)
{
  int left, right, top, bottom;
  int x2size;

  /* width/height of the grip */
  x2size = 2 * size;

  /* inside upper left grip ? */
  left = o_current->picture->screen_upper_x - size;
  top = o_current->picture->screen_upper_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = PICTURE_UPPER_LEFT;
    return (o_current);
  }

  /* inside lower right grip ? */
  left = o_current->picture->screen_lower_x - size;
  top = o_current->picture->screen_lower_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = PICTURE_LOWER_RIGHT;
    return (o_current);
  }

  /* inside upper right grip ? */
  left = o_current->picture->screen_lower_x - size;
  top = o_current->picture->screen_upper_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = PICTURE_UPPER_RIGHT;
    return (o_current);
  }

  /* inside lower left grip ? */
  left = o_current->picture->screen_upper_x - size;
  top = o_current->picture->screen_lower_y - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = PICTURE_LOWER_LEFT;
    return (o_current);
  }

  return NULL;
}

#line 22 "../noweb/o_grips.nw"
#endif
#line 1443 "../noweb/o_grips.nw"
OBJECT *o_grips_search_circle(TOPLEVEL * w_current, OBJECT * o_current,
			      int x, int y, int size, int *whichone)
{
  int left, top, right, bottom;
  int x1, y1;
  int x2size;

  /* width/height of the grip */
  x2size = 2 * size;

  /* check the grip for radius */
  x1 = o_current->circle->screen_x + o_current->circle->screen_radius;
  y1 = o_current->circle->screen_y + o_current->circle->screen_radius;
  left = x1 - size;
  top = y1 - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    /* printf("found something 0!\n"); */
    *whichone = CIRCLE_RADIUS;
    return (o_current);
  }

  return NULL;
}

#line 1599 "../noweb/o_grips.nw"
OBJECT *o_grips_search_line(TOPLEVEL * w_current, OBJECT * o_current,
			    int x, int y, int size, int *whichone)
{
  int left, top, right, bottom;
  int x2size;

  /* width/height of the grip */
  x2size = 2 * size;

  /* check the grip on the end of line 1 */
  left = o_current->line->screen_x[LINE_END1] - size;
  top = o_current->line->screen_y[LINE_END1] - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = LINE_END1;
    return (o_current);
  }

  /* check the grip on the end of line 2 */
  left = o_current->line->screen_x[LINE_END2] - size;
  top = o_current->line->screen_y[LINE_END2] - size;
  right = left + x2size;
  bottom = top + x2size;
  if (inside_region(left, top, right, bottom, x, y)) {
    *whichone = LINE_END2;
    return (o_current);
  }

  return NULL;
}


#line 251 "../noweb/o_grips.nw"
int o_grips_start(TOPLEVEL * w_current, int x, int y)
{
  OBJECT *object;
  int whichone;

  if (w_current->draw_grips == FALSE) {
    return (FALSE);
  }

  /* search if there is a grip on a selected object at (x,y) */
  object = o_grips_search(w_current, x, y, &whichone);
  if (object) {
    /* there is one */
    /* depending on its type, start the modification process */
    switch (object->type) {

    case (OBJ_ARC):
      /* start the modification of a grip on an arc */
      o_grips_start_arc(w_current, object, x, y, whichone);

      whichone_changing = whichone;
      object_changing = object;
      return (TRUE);
      break;

    case (OBJ_BOX):
      /* start the modification of a grip on a box */
      o_grips_start_box(w_current, object, x, y, whichone);

      whichone_changing = whichone;
      object_changing = object;
      return (TRUE);
      break;

    case (OBJ_PICTURE):
      /* start the modification of a grip on a picture */
#ifndef HAS_GTK12
      o_grips_start_picture(w_current, object, x, y, whichone);

      whichone_changing = whichone;
      object_changing = object;
      return (TRUE);
#endif
      break;

    case (OBJ_CIRCLE):
      /* start the modification of a grip on a circle */
      o_grips_start_circle(w_current, object, x, y, whichone);

      whichone_changing = whichone;
      object_changing = object;
      return (TRUE);
      break;

    case (OBJ_LINE):
      /* start the modification of a grip on a line */
      o_grips_start_line(w_current, object, x, y, whichone);

      whichone_changing = whichone;
      object_changing = object;
      return (TRUE);
      break;

    case (OBJ_NET):
      w_current->last_drawb_mode = -1;
      w_current->last_x = object->line->screen_x[whichone];
      w_current->last_y = object->line->screen_y[whichone];
      w_current->start_x = object->line->screen_x[!whichone];
      w_current->start_y = object->line->screen_y[!whichone];

      o_net_erase(w_current, object);
      gdk_gc_set_foreground(w_current->xor_gc,
			    x_get_darkcolor(w_current->select_color));
      gdk_draw_line(w_current->window, w_current->xor_gc,
		    w_current->start_x, w_current->start_y,
		    w_current->last_x, w_current->last_y);
      o_line_erase_grips(w_current, object);

      whichone_changing = whichone;
      object_changing = object;
      gdk_gc_set_foreground(w_current->gc,
			    x_get_color(w_current->background_color));
      return (TRUE);

      break;

    case (OBJ_PIN):

      w_current->last_drawb_mode = -1;
      w_current->last_x = object->line->screen_x[whichone];
      w_current->last_y = object->line->screen_y[whichone];
      w_current->start_x = object->line->screen_x[!whichone];
      w_current->start_y = object->line->screen_y[!whichone];

      o_pin_erase(w_current, object);
      gdk_gc_set_foreground(w_current->xor_gc,
			    x_get_darkcolor(w_current->select_color));
      gdk_draw_line(w_current->window, w_current->xor_gc,
		    w_current->start_x, w_current->start_y,
		    w_current->last_x, w_current->last_y);
      o_line_erase_grips(w_current, object);

      whichone_changing = whichone;
      object_changing = object;
      return (TRUE);

      break;

    case (OBJ_BUS):
      w_current->last_drawb_mode = -1;
      w_current->last_x = object->line->screen_x[whichone];
      w_current->last_y = object->line->screen_y[whichone];
      w_current->start_x = object->line->screen_x[!whichone];
      w_current->start_y = object->line->screen_y[!whichone];

      o_bus_erase(w_current, object);
      gdk_gc_set_foreground(w_current->xor_gc,
			    x_get_darkcolor(w_current->select_color));
      gdk_draw_line(w_current->window, w_current->xor_gc,
		    w_current->start_x, w_current->start_y,
		    w_current->last_x, w_current->last_y);
      o_line_erase_grips(w_current, object);

      whichone_changing = whichone;
      object_changing = object;
      gdk_gc_set_foreground(w_current->gc,
			    x_get_color(w_current->background_color));
      return (TRUE);

      break;

    default:
      /* object type unknown : error condition */
      return (FALSE);

    }
  }

  return (FALSE);
}

#line 858 "../noweb/o_grips.nw"
void
o_grips_start_arc(TOPLEVEL * w_current, OBJECT * o_current,
		  int x, int y, int whichone)
{
  w_current->last_drawb_mode = -1;

  /* erase the arc before */
  o_arc_erase(w_current, o_current);

  /* describe the arc with TOPLEVEL variables */
  /* center */
  w_current->start_x = o_current->arc->screen_x;
  w_current->start_y = o_current->arc->screen_y;
  /* radius */
  w_current->distance = o_current->arc->screen_width / 2;
  /* angles */
  w_current->loc_x = o_current->arc->start_angle;
  w_current->loc_y = o_current->arc->end_angle;

  /* draw the first temporary arc */
  o_arc_rubberarc_xor(w_current);

}

#line 1048 "../noweb/o_grips.nw"
void
o_grips_start_box(TOPLEVEL * w_current, OBJECT * o_current,
		  int x, int y, int whichone)
{
  w_current->last_drawb_mode = -1;

  /* erase the box before */
  o_box_erase(w_current, o_current);

  /* (last_x,last_y)    is the selected corner */
  /* (start_x, start_y) is the opposite corner */
  switch (whichone) {
  case BOX_UPPER_LEFT:
    w_current->last_x = o_current->box->screen_upper_x;
    w_current->last_y = o_current->box->screen_upper_y;
    w_current->start_x = o_current->box->screen_lower_x;
    w_current->start_y = o_current->box->screen_lower_y;
    break;
  case BOX_LOWER_RIGHT:
    w_current->last_x = o_current->box->screen_lower_x;
    w_current->last_y = o_current->box->screen_lower_y;
    w_current->start_x = o_current->box->screen_upper_x;
    w_current->start_y = o_current->box->screen_upper_y;
    break;
  case BOX_UPPER_RIGHT:
    w_current->last_x = o_current->box->screen_lower_x;
    w_current->last_y = o_current->box->screen_upper_y;
    w_current->start_x = o_current->box->screen_upper_x;
    w_current->start_y = o_current->box->screen_lower_y;
    break;
  case BOX_LOWER_LEFT:
    w_current->last_x = o_current->box->screen_upper_x;
    w_current->last_y = o_current->box->screen_lower_y;
    w_current->start_x = o_current->box->screen_lower_x;
    w_current->start_y = o_current->box->screen_upper_y;
    break;
  default:
    return;			/* error */
  }

  /* draw the first temporary box */
  o_box_rubberbox_xor(w_current);

}

#line 29 "../noweb/o_grips.nw"
#ifndef HAS_GTK12
#line 1207 "../noweb/o_grips.nw"
void
o_grips_start_picture(TOPLEVEL * w_current, OBJECT * o_current,
		      int x, int y, int whichone)
{
  w_current->last_drawb_mode = -1;

  /* erase the picture before */
  o_picture_erase(w_current, o_current);
  w_current->current_pixbuf = o_current->picture->original_picture;
  w_current->pixbuf_filename = o_current->picture->filename;
  w_current->pixbuf_wh_ratio = o_current->picture->ratio;

  /* (last_x,last_y)    is the selected corner */
  /* (start_x, start_y) is the opposite corner */
  switch (whichone) {
  case PICTURE_UPPER_LEFT:
    w_current->last_x = o_current->picture->screen_upper_x;
    w_current->last_y = o_current->picture->screen_upper_y;
    w_current->start_x = o_current->picture->screen_lower_x;
    w_current->start_y = o_current->picture->screen_lower_y;
    break;
  case PICTURE_LOWER_RIGHT:
    w_current->last_x = o_current->picture->screen_lower_x;
    w_current->last_y = o_current->picture->screen_lower_y;
    w_current->start_x = o_current->picture->screen_upper_x;
    w_current->start_y = o_current->picture->screen_upper_y;
    break;
  case PICTURE_UPPER_RIGHT:
    w_current->last_x = o_current->picture->screen_lower_x;
    w_current->last_y = o_current->picture->screen_upper_y;
    w_current->start_x = o_current->picture->screen_upper_x;
    w_current->start_y = o_current->picture->screen_lower_y;
    break;
  case PICTURE_LOWER_LEFT:
    w_current->last_x = o_current->picture->screen_upper_x;
    w_current->last_y = o_current->picture->screen_lower_y;
    w_current->start_x = o_current->picture->screen_lower_x;
    w_current->start_y = o_current->picture->screen_upper_y;
    break;
  default:
    return;			/* error */
  }

  /* draw the first temporary picture */
  o_picture_rubberbox_xor(w_current);

}

#line 31 "../noweb/o_grips.nw"
#endif
#line 1486 "../noweb/o_grips.nw"
void
o_grips_start_circle(TOPLEVEL * w_current, OBJECT * o_current,
		     int x, int y, int whichone)
{
  w_current->last_drawb_mode = -1;

  /* erase the circle before */
  o_circle_erase(w_current, o_current);

  /* describe the circle with TOPLEVEL variables */
  /* (start_x, start_y) is the center of the circle */
  w_current->start_x = o_current->circle->screen_x;
  w_current->start_y = o_current->circle->screen_y;
  /* (last_x,last_y)    is the point on circle on the right of center */
  w_current->last_x =
      o_current->circle->screen_x + o_current->circle->screen_radius;
  w_current->last_y = o_current->circle->screen_y;
  /* distance           is the radius of the circle */
  w_current->distance = o_current->circle->screen_radius;

  /* draw the first temporary circle */
  o_circle_rubbercircle_xor(w_current);

}

#line 1647 "../noweb/o_grips.nw"
void
o_grips_start_line(TOPLEVEL * w_current, OBJECT * o_current,
		   int x, int y, int whichone)
{
  w_current->last_drawb_mode = -1;

  /* erase the line before */
  o_line_erase(w_current, o_current);

  /* describe the line with TOPLEVEL variables */
  w_current->last_x = o_current->line->screen_x[whichone];
  w_current->last_y = o_current->line->screen_y[whichone];
  w_current->start_x = o_current->line->screen_x[!whichone];
  w_current->start_y = o_current->line->screen_y[!whichone];

  /* draw the first temporary line */
  o_line_rubberline_xor(w_current);

}


#line 408 "../noweb/o_grips.nw"
void o_grips_motion(TOPLEVEL * w_current, int x, int y)
{

  if (w_current->inside_action == 0) {
    o_redraw(w_current, w_current->page_current->object_head);
    return;
  }

  /* no object changing */
  if (object_changing == NULL) {
    /* stop grip process */
    o_redraw(w_current, w_current->page_current->object_head);
    return;
  }

  switch (object_changing->type) {

  case (OBJ_ARC):
    /* erase, update and draw an arc */
    o_grips_motion_arc(w_current, x, y, whichone_changing);
    break;

  case (OBJ_BOX):
    /* erase, update and draw a box */
    o_grips_motion_box(w_current, x, y, whichone_changing);
    break;

  case (OBJ_PICTURE):
    /* erase, update and draw a box */
#ifndef HAS_GTK12
    o_grips_motion_picture(w_current, x, y, whichone_changing);
#endif
    break;

  case (OBJ_CIRCLE):
    /* erase, update and draw a circle */
    o_grips_motion_circle(w_current, x, y, whichone_changing);
    break;

  case (OBJ_LINE):
  case (OBJ_NET):
  case (OBJ_PIN):
  case (OBJ_BUS):
    /* erase, update and draw a line */
    /* same for net, pin and bus as they share the same internal rep. */
    o_grips_motion_line(w_current, x, y, whichone_changing);
    break;

  default:
    return;			/* error condition */
  }

}

#line 897 "../noweb/o_grips.nw"
void o_grips_motion_arc(TOPLEVEL * w_current, int x, int y, int whichone)
{
  o_arc_rubberarc(w_current, x, y, whichone);

}

#line 1106 "../noweb/o_grips.nw"
void o_grips_motion_box(TOPLEVEL * w_current, int x, int y, int whichone)
{
  /* erase, update and draw the temporary box */
  o_box_rubberbox(w_current, x, y);

}

#line 38 "../noweb/o_grips.nw"
#ifndef HAS_GTK12
#line 1339 "../noweb/o_grips.nw"
void
o_grips_motion_picture(TOPLEVEL * w_current, int x, int y, int whichone)
{
  /* erase, update and draw the temporary picture */
  o_picture_rubberbox(w_current, x, y);

}

#line 40 "../noweb/o_grips.nw"
#endif
#line 1523 "../noweb/o_grips.nw"
void
o_grips_motion_circle(TOPLEVEL * w_current, int x, int y, int whichone)
{
  /* erase, update and draw the temporary circle */
  o_circle_rubbercircle(w_current, x, y);

}

#line 1678 "../noweb/o_grips.nw"
void o_grips_motion_line(TOPLEVEL * w_current, int x, int y, int whichone)
{
  /* erase, update and draw the temporary line */
  o_line_rubberline(w_current, x, y);

}


#line 478 "../noweb/o_grips.nw"
void o_grips_end(TOPLEVEL * w_current)
{
  OBJECT *object = NULL;
  int x, y;
  GList *other_objects = NULL;
  GList *connected_objects = NULL;
  int size;

  object = object_changing;

  if (!object) {
    /* actually this is an error condition hack */
    w_current->inside_action = 0;
    i_set_state(w_current, SELECT);
    return;
  }

  switch (object->type) {

  case (OBJ_ARC):
    /* modify an arc object */
    o_grips_end_arc(w_current, object, whichone_changing);
    break;

  case (OBJ_BOX):
    /* modify a box object */
    o_grips_end_box(w_current, object, whichone_changing);
    break;

  case (OBJ_PICTURE):
    /* modify a picture object */
#ifndef HAS_GTK12
    o_grips_end_picture(w_current, object, whichone_changing);
#endif
    break;

  case (OBJ_CIRCLE):
    /* modify a circle object */
    o_grips_end_circle(w_current, object, whichone_changing);
    break;

  case (OBJ_LINE):
    /* modify a line object */
    o_grips_end_line(w_current, object, whichone_changing);
    break;

  case (OBJ_NET):
    /* don't allow zero length nets / lines / pins
     * this ends the net drawing behavior 
     * we want this? hack */
    if ((w_current->start_x == w_current->last_x) &&
	(w_current->start_y == w_current->last_y)) {
      w_current->start_x = (-1);
      w_current->start_y = (-1);
      w_current->last_x = (-1);
      w_current->last_y = (-1);
      w_current->inside_action = 0;
      i_set_state(w_current, SELECT);
      o_net_eraserubber(w_current);
      o_redraw_single(w_current, object);
      i_update_toolbar(w_current);
      return;
    }


    SCREENtoWORLD(w_current, w_current->last_x, w_current->last_y, &x, &y);

    x = snap_grid(w_current, x);
    y = snap_grid(w_current, y);

    o_cue_undraw(w_current, object);
    o_net_erase(w_current, object);
    /* erase xor line */
    gdk_gc_set_foreground(w_current->xor_gc,
			  x_get_darkcolor(w_current->select_color));
    gdk_draw_line(w_current->window, w_current->xor_gc,
		  w_current->start_x, w_current->start_y,
		  w_current->last_x, w_current->last_y);
    o_line_erase_grips(w_current, object);

    other_objects = s_conn_return_others(other_objects, object);
    s_conn_remove(w_current, object);

    o_net_modify(w_current, object, x, y, whichone_changing);

    s_conn_update_object(w_current, object);
    o_net_recalc(w_current, object);

    /* get the other connected objects and redraw them */
    connected_objects = s_conn_return_others(connected_objects, object);

    /* add bus rippers if necessary */
    if (o_net_add_busrippers(w_current, object, connected_objects)) {

      o_net_erase(w_current, object);
      /*o_line_erase_grips(w_current, object); */

      if (w_current->net_style == THICK) {
	size = SCREENabs(w_current, 10);

	if (size < 0)
	  size = 0;

	gdk_gc_set_line_attributes(w_current->gc, size,
				   GDK_LINE_SOLID,
				   GDK_CAP_BUTT, GDK_CAP_NOT_LAST);
      }

      gdk_gc_set_foreground(w_current->gc,
			    x_get_color(w_current->background_color));
      gdk_draw_line(w_current->window, w_current->gc,
		    w_current->start_x, w_current->start_y,
		    w_current->last_x, w_current->last_y);

      o_cue_undraw(w_current, object);
      o_net_draw(w_current, object);
      o_cue_draw_single(w_current, object);

      if (w_current->net_style == THICK) {
	gdk_gc_set_line_attributes(w_current->gc, 0,
				   GDK_LINE_SOLID,
				   GDK_CAP_NOT_LAST, GDK_JOIN_MITER);
      }
    }

    /* draw the object objects */
    o_cue_undraw_list(w_current, other_objects);
    o_cue_draw_list(w_current, other_objects);

    o_redraw_single(w_current, object);

    if (connected_objects) {
      g_list_free(connected_objects);
      connected_objects = NULL;
    }

    /* get the other connected objects and redraw them */
    connected_objects = s_conn_return_others(connected_objects, object);

    o_cue_undraw_list(w_current, connected_objects);
    o_cue_draw_list(w_current, connected_objects);
    /* finally draw this objects cues */
    o_cue_draw_single(w_current, object);
    break;

  case (OBJ_PIN):
    /* don't allow zero length nets / lines / pins
     * this ends the net drawing behavior 
     * we want this? hack */
    if ((w_current->start_x == w_current->last_x) &&
	(w_current->start_y == w_current->last_y)) {
      w_current->start_x = (-1);
      w_current->start_y = (-1);
      w_current->last_x = (-1);
      w_current->last_y = (-1);
      o_redraw_single(w_current, object);
      w_current->inside_action = 0;
      i_set_state(w_current, SELECT);
      i_update_toolbar(w_current);
      return;
    }


    SCREENtoWORLD(w_current, w_current->last_x, w_current->last_y, &x, &y);

    x = snap_grid(w_current, x);
    y = snap_grid(w_current, y);

    o_cue_undraw(w_current, object);
    o_pin_erase(w_current, object);
    /* erase xor line */
    gdk_gc_set_foreground(w_current->xor_gc,
			  x_get_darkcolor(w_current->select_color));
    gdk_draw_line(w_current->window, w_current->xor_gc,
		  w_current->start_x, w_current->start_y,
		  w_current->last_x, w_current->last_y);
    o_line_erase_grips(w_current, object);

    other_objects = s_conn_return_others(other_objects, object);
    s_conn_remove(w_current, object);

    o_pin_modify(w_current, object, x, y, whichone_changing);
    s_conn_update_object(w_current, object);
    o_redraw_single(w_current, object);

    /* draw the object objects */
    o_cue_undraw_list(w_current, other_objects);
    o_cue_draw_list(w_current, other_objects);

    /* get the other connected objects and redraw them */
    connected_objects = s_conn_return_others(connected_objects, object);
    o_cue_undraw_list(w_current, connected_objects);
    o_cue_draw_list(w_current, connected_objects);

    /* finally draw this objects cues */
    o_cue_draw_single(w_current, object);
    break;

  case (OBJ_BUS):
    /* don't allow zero length nets / lines / pins
     * this ends the net drawing behavior 
     * we want this? hack */
    if ((w_current->start_x == w_current->last_x) &&
	(w_current->start_y == w_current->last_y)) {
      w_current->start_x = (-1);
      w_current->start_y = (-1);
      w_current->last_x = (-1);
      w_current->last_y = (-1);
      o_net_eraserubber(w_current);
      o_redraw_single(w_current, object);
      w_current->inside_action = 0;
      i_set_state(w_current, SELECT);
      i_update_toolbar(w_current);
      return;
    }

    SCREENtoWORLD(w_current, w_current->last_x, w_current->last_y, &x, &y);

    x = snap_grid(w_current, x);
    y = snap_grid(w_current, y);

    o_cue_undraw(w_current, object);
    o_bus_erase(w_current, object);
    /* erase xor line */
    gdk_gc_set_foreground(w_current->xor_gc,
			  x_get_darkcolor(w_current->select_color));
    gdk_draw_line(w_current->window, w_current->xor_gc,
		  w_current->start_x, w_current->start_y,
		  w_current->last_x, w_current->last_y);
    o_line_erase_grips(w_current, object);

    other_objects = s_conn_return_others(other_objects, object);
    s_conn_remove(w_current, object);

    o_bus_modify(w_current, object, x, y, whichone_changing);
    s_conn_update_object(w_current, object);
    o_redraw_single(w_current, object);

    /* draw the object objects */
    o_cue_undraw_list(w_current, other_objects);
    o_cue_draw_list(w_current, other_objects);

    /* get the other connected objects and redraw them */
    connected_objects = s_conn_return_others(connected_objects, object);
    o_cue_undraw_list(w_current, connected_objects);
    o_cue_draw_list(w_current, connected_objects);

    /* finally draw this objects cues */
    o_cue_draw_single(w_current, object);
    break;

  default:
    return;
  }


  w_current->page_current->CHANGED = 1;

  g_list_free(other_objects);
  other_objects = NULL;
  g_list_free(connected_objects);
  connected_objects = NULL;

  /* reset global variables */
  whichone_changing = -1;
  object_changing = NULL;

  o_undo_savestate(w_current, UNDO_ALL);
}

#line 918 "../noweb/o_grips.nw"
void
o_grips_end_arc(TOPLEVEL * w_current, OBJECT * o_current, int whichone)
{
  int arg1, arg2;

  /* erase the temporary arc */
  o_arc_rubberarc_xor(w_current);

  /* determination of the parameters to give to o_arc_modify() */
  switch (whichone) {
  case ARC_RADIUS:
    /* convert the radius in world coords */
    arg1 = WORLDabs(w_current, w_current->distance);
    /* second parameter is not used */
    arg2 = -1;
    break;

  case ARC_START_ANGLE:
    /* get the start angle from w_current */
    arg1 = w_current->loc_x;
    /* second parameter is not used */
    arg2 = -1;
    break;

  case ARC_END_ANGLE:
    /* get the end angle from w_current */
    arg1 = w_current->loc_y;
    /* second parameter is not used */
    arg2 = -1;
    break;

  default:
    return;
  }

  /* modify the arc with the parameters determined above */
  o_arc_modify(w_current, o_current, arg1, arg2, whichone);

  /* display the new arc */
  o_redraw_single(w_current, o_current);

}

#line 1123 "../noweb/o_grips.nw"
void
o_grips_end_box(TOPLEVEL * w_current, OBJECT * o_current, int whichone)
{
  int box_width, box_height;
  int x, y;


#line 1144 "../noweb/o_grips.nw"
  box_width = GET_BOX_WIDTH(w_current);
  box_height = GET_BOX_HEIGHT(w_current);

#line 1130 "../noweb/o_grips.nw"


#line 1151 "../noweb/o_grips.nw"
/* don't allow zero width/height boxes
 * this ends the box drawing behavior 
 * we want this? hack */
  if ((box_width == 0) && (box_height == 0)) {
    w_current->start_x = (-1);
    w_current->start_y = (-1);
    w_current->last_x = (-1);
    w_current->last_y = (-1);

    w_current->inside_action = 0;
    i_set_state(w_current, SELECT);

    o_redraw_single(w_current, o_current);
    i_update_toolbar(w_current);

    return;
  }

#line 1133 "../noweb/o_grips.nw"

#line 1174 "../noweb/o_grips.nw"
  SCREENtoWORLD(w_current, w_current->last_x, w_current->last_y, &x, &y);
  x = snap_grid(w_current, x);
  y = snap_grid(w_current, y);

  o_box_modify(w_current, o_current, x, y, whichone);


#line 1135 "../noweb/o_grips.nw"

#line 1186 "../noweb/o_grips.nw"
/* erase the temporary box */
  o_box_rubberbox_xor(w_current);

/* draw the modified box */
  o_redraw_single(w_current, o_current);


#line 1137 "../noweb/o_grips.nw"
}

#line 47 "../noweb/o_grips.nw"
#ifndef HAS_GTK12
#line 1355 "../noweb/o_grips.nw"
void
o_grips_end_picture(TOPLEVEL * w_current, OBJECT * o_current, int whichone)
{
  int picture_width, picture_height;
  int x, y;


#line 1376 "../noweb/o_grips.nw"
  picture_width = GET_PICTURE_WIDTH(w_current);
  picture_height = GET_PICTURE_HEIGHT(w_current);

#line 1362 "../noweb/o_grips.nw"


#line 1383 "../noweb/o_grips.nw"
/* don't allow zero width/height picturees
 * this ends the picture drawing behavior 
 * we want this? hack */
  if ((picture_width == 0) && (picture_height == 0)) {
    w_current->start_x = (-1);
    w_current->start_y = (-1);
    w_current->last_x = (-1);
    w_current->last_y = (-1);

    w_current->inside_action = 0;
    i_set_state(w_current, SELECT);

    o_redraw_single(w_current, o_current);
    i_update_toolbar(w_current);

    return;
  }

#line 1365 "../noweb/o_grips.nw"

#line 1406 "../noweb/o_grips.nw"
  SCREENtoWORLD(w_current, w_current->last_x, w_current->last_y, &x, &y);
  x = snap_grid(w_current, x);
  y = snap_grid(w_current, y);

  o_picture_modify(w_current, o_current, x, y, whichone);


#line 1367 "../noweb/o_grips.nw"

#line 1418 "../noweb/o_grips.nw"
/* erase the temporary picture */
  o_picture_rubberbox_xor(w_current);

/* draw the modified picture */
  o_redraw_single(w_current, o_current);

  w_current->current_pixbuf = NULL;
  w_current->pixbuf_filename = NULL;
  w_current->pixbuf_wh_ratio = 0;


#line 1369 "../noweb/o_grips.nw"
}

#line 49 "../noweb/o_grips.nw"
#endif
#line 1546 "../noweb/o_grips.nw"
void
o_grips_end_circle(TOPLEVEL * w_current, OBJECT * o_current, int whichone)
{
  int radius;

  /* erase the temporary circle */
  o_circle_rubbercircle_xor(w_current);

  /* don't allow zero radius circles
   * this ends the circle drawing behavior 
   * we want this? hack */
  if ((w_current->start_x == w_current->last_x) &&
      (w_current->start_y == w_current->last_y)) {
    w_current->start_x = (-1);
    w_current->start_y = (-1);
    w_current->last_x = (-1);
    w_current->last_y = (-1);

    /* return to select mode */
    w_current->inside_action = 0;
    i_set_state(w_current, SELECT);
    i_update_toolbar(w_current);

    o_redraw_single(w_current, o_current);
    return;
  }

  /* convert the radius in world unit */
  radius = WORLDabs(w_current, w_current->distance);

  /* modify the radius of the circle */
  o_circle_modify(w_current, o_current, radius, -1, CIRCLE_RADIUS);

  /* display the new circle */
  o_redraw_single(w_current, o_current);

}

#line 1700 "../noweb/o_grips.nw"
void
o_grips_end_line(TOPLEVEL * w_current, OBJECT * o_current, int whichone)
{
  int x, y;

  /* erase the temporary line */
  o_line_rubberline_xor(w_current);

  /* don't allow zero length nets / lines / pins
   * this ends the net drawing behavior 
   * we want this? hack */
  if ((w_current->start_x == w_current->last_x) &&
      (w_current->start_y == w_current->last_y)) {
    w_current->start_x = (-1);
    w_current->start_y = (-1);
    w_current->last_x = (-1);
    w_current->last_y = (-1);

    /* return to select mode */
    w_current->inside_action = 0;
    i_set_state(w_current, SELECT);
    i_update_toolbar(w_current);

    o_redraw_single(w_current, o_current);
    return;
  }

  /* convert the line end coords in world unit */
  SCREENtoWORLD(w_current, w_current->last_x, w_current->last_y, &x, &y);
  x = snap_grid(w_current, x);
  y = snap_grid(w_current, y);

  /* modify the right line end according to whichone */
  o_line_modify(w_current, o_current, x, y, whichone);

  /* display the new line */
  o_redraw_single(w_current, o_current);

}


#line 1754 "../noweb/o_grips.nw"
int o_grips_size(TOPLEVEL * w_current)
{
  int factor, size;

  factor = (int) w_current->page_current->to_world_x_constant;
  if (factor > SMALL_ZOOMFACTOR1) {
    /* big zoom factor : small size converted to screen unit */
    size = SCREENabs(w_current, GRIP_SIZE1);
  } else if (factor > SMALL_ZOOMFACTOR2) {
    /* medium zoom factor : medium size converted to screen unit */
    size = SCREENabs(w_current, GRIP_SIZE2);
  } else {
    /* small zoom factor : big size converted to screen unit */
    size = SCREENabs(w_current, GRIP_SIZE3);
  }

  return size;
}


#line 1788 "../noweb/o_grips.nw"
void o_grips_draw(TOPLEVEL * w_current, int x, int y)
{
  GdkColor *color;
  int size, x2size;


#line 1808 "../noweb/o_grips.nw"
/* size is half the width of grip */
  size = o_grips_size(w_current);
/* x2size is full width */
  x2size = 2 * size;


#line 1796 "../noweb/o_grips.nw"

#line 1819 "../noweb/o_grips.nw"
  if (w_current->override_color != -1) {
    /* override : use the override_color instead */
    color = x_get_color(w_current->override_color);
  } else {
    /* use the normal selection color */
    color = x_get_color(w_current->select_color);
  }
/* set the color for the grip */
  gdk_gc_set_foreground(w_current->gc, color);

/* set the line options for grip : solid, 1 pix wide */
  gdk_gc_set_line_attributes(w_current->gc, 0, GDK_LINE_SOLID,
			     GDK_CAP_BUTT, GDK_JOIN_MITER);


#line 1798 "../noweb/o_grips.nw"

#line 1838 "../noweb/o_grips.nw"
/* draw the grip in window */
  gdk_draw_rectangle(w_current->window, w_current->gc, FALSE,
		     x - size, y - size, x2size, x2size);
/* draw the grip in backingstore */
  gdk_draw_rectangle(w_current->backingstore, w_current->gc, FALSE,
		     x - size, y - size, x2size, x2size);


#line 1800 "../noweb/o_grips.nw"
}

#line 1860 "../noweb/o_grips.nw"
void o_grips_erase(TOPLEVEL * w_current, int x, int y)
{
  /* set overriding color */
  w_current->override_color = w_current->background_color;
  /* draw a grip with backgound color : erase grip */
  o_grips_draw(w_current, x, y);
  /* return to default */
  w_current->override_color = -1;
}
