"""SyntaxCompleterPlugin enabled word and symbol completion."""
# Copyright (C) 2007-2011 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the GNU General Public License version 2
# (see the file COPYING).

__metaclass__ = type

__all__ = [
    'SyntaxCompleterPlugin',
    ]

from gettext import gettext as _

from gi.repository import GObject

from gdpbase import (
    GDPPluginMixin,
    Gedit,
    )
from gdp.syntaxcompleter import SyntaxController


class SyntaxCompleterPlugin(GDPPluginMixin,
                            GObject.Object, Gedit.WindowActivatable):
    """Automatically complete words from the list of words in the document."""
    __gtype_name__ = "GDPSyntaxCompleterPlugin"
    window = GObject.property(type=GObject.Object)

    CONTROLLER_CLASS = SyntaxController
    ACTION_GROUP_NAME = 'GDPSyntaxActions'
    MENU_PATH = '/MenuBar/ToolsMenu/ToolsOps_2/CompleteWord'
    MENU_XML = """
        <ui>
          <menubar name="MenuBar">
            <menu name='ToolsMenu' action='Tools'>
              <placeholder name="ToolsOps_2">
                <separator />
                <menuitem action="CompleteWord"/>
                <separator />
              </placeholder>
            </menu>
          </menubar>
        </ui>
        """

    def actions(self, syntaxer):
        """See `GDPPluginMixin`"""
        return  [
            ('CompleteWord', None, _("Complete _word"),
                '<Alt>slash',
                _("Complete the word at the cursor."),
                syntaxer.show_syntax_view),
            ]

    def __init__(self):
        """Initialize the plugin the whole Gedit application."""
        GObject.Object.__init__(self)
        self.controller = None

    def do_activate(self):
        self.activate()
        self.connect_signal(
            self.window, 'tab-added', self.on_tab_added_or_changed)
        self.connect_signal(
            self.window, 'active-tab-changed', self.on_tab_added_or_changed)

    def do_deactivate(self):
        self.deactivate()

    def on_tab_added_or_changed(self, window, tab):
        """Callback method for tab-added or active-tab-changed window signal.

        Correct the language and update the menu.
        """
        self.controller.correct_language(tab.get_document())
        view = tab.get_view()
        if view == window.get_active_view():
            self.controller.set_view(view)
            manager = self.window.get_ui_manager()
            manager.get_action(self.MENU_PATH).props.sensitive = True
