/*
** 1998-11-26 -	A neat little module to manage directory histories in the panes. Initially, this
**		will just remember the actual paths, but future versions might include storing
**		info about selected files, sizes, and so on...
** 1998-12-06 -	Rewritten after a suggestion from J. Hanson (<johan@tiq.com>), to simply hash on
**		the inode numbers rather than the file names. Should reduce time and memory complex-
**		ities by a whole lot.
** 1998-12-15 -	Now stores the vertical position in a relative way, rather than using absolute pixel
**		coordinates. Not good, but better.
** 1998-12-23 -	Eh. Inode numbers are _not_ unique across devices (which might be why there's
**		a st_dev field in the stat structure). This of course makes them a bad choice
**		for unique file identifiers - when buffering a directory containing inodes from
**		various devices (such as / typically does), things went really haywire. To fix
**		this, we now store the device identifier too. The pair (device,inode) really should
**		be unique.
** 1999-09-05 -	Cut away the fancy browser-esque history system implemented for 0.11.8, since it wasn't
**		complete, and I didn't like it much. Going back to a simple combo as before.
** 1999-11-13 -	Prettied up handling of vertical position remembering somewhat, exported it separately.
** 2002-08-09 -	DHEntries can't be keyed on inode, since that can be fooled by deleting a directory,
**		and then creating a new (differently named) one. The filesystem can re-use the inode.
** 2008-01-26 -	Adapted to keep a separate UTF-8 copy of the path names, for GTK+ 2. Ugly, needs to
**		be way better. Later? Heh.
*/

#include "gentoo.h"

#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

#include "cmdseq.h"
#include "convutil.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "strutil.h"
#include "xmlutil.h"

#include "dirhistory.h"

/* ----------------------------------------------------------------------------------------- */

#define	HISTORY_SIZE	(16)	/* Maximum number of items in history. Should this be dynamic? */

/* ----------------------------------------------------------------------------------------- */

typedef struct {			/* A history key. A (device,inode) pair. */
	dev_t	dev;
	ino_t	inode;
	guint	hash;
} HKey;

/* This is used to store the selected files in a pane, and nothing else. */
struct DHSel {
	GHashTable	*hash;		/* It's a hash of HKeys, as above. For fast 'apply' operations. */
};

/* Info about a single remembered directory. The 'hist' field of dirpanes stores a list of these. */
typedef struct {
	gchar	dpath[2*PATH_MAX];	/* The path to this directory, in UTF-8. Must be first! FIXME: Kind of a gross assumption. */
	gchar	path[PATH_MAX];		/* The path to this directory. */
	gchar	realpath[PATH_MAX];	/* Canonical path, used as key since <dev,inode> isn't enough. */
	DHSel	*sel;			/* The selection last time we were here. */
	gfloat	vpos;			/* Vertical position. */
	gint	focus_row;		/* Row that had the keyboard-controlled focus, or -1. */
} DHEntry;

struct DirHistory {
	GList	*history;		/* Visited directories. Recent are close to head. */
};

/* Here's a glib GMemChunk we use to allocate HKeys, in the hope of getting better efficiency
** than g_malloc() would give us. Note that there is just one such GMemChunk, from which we
** allocate HKeys for _all_ directories (DHEntries).
*/
static GMemChunk	*the_chunk = NULL;

/* ----------------------------------------------------------------------------------------- */

static DHSel *	dirsel_set(DHSel *sel, const DirPane *dp);

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-08 -	Create a new dirpane history structure. */
DirHistory * dph_dirhistory_new(void)
{
	DirHistory	*dh;

	dh = g_malloc(sizeof *dh);
	dh->history = NULL;

	return dh;
}

/* 2004-02-25 -	Get the entry with the given <index>, where 0 is the index of the most recent one. */
const gchar * dph_dirhistory_get_entry(const DirHistory *dh, guint index)
{
	if(dh != NULL)
	{
		const DHEntry   *dhe = (const DHEntry *) g_list_nth_data(dh->history, index);
 		if(dhe != NULL)
			return dhe->path;
	}
	return NULL;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-12-23 -	Set a history key according to information in <st>. */
static void hkey_set(HKey *hk, const struct stat *st)
{
	if(hk != NULL && st != NULL)
	{
		hk->dev   = st->st_dev;
		hk->inode = st->st_ino;
	}
}

/* 1998-12-23 -	Create a new history key, initialized with information from <st>. */
static HKey * hkey_new(struct stat *st)
{
	HKey	*hk = NULL;

	if(the_chunk == NULL)
		the_chunk = g_mem_chunk_new("HKey", sizeof *hk, 1024, G_ALLOC_AND_FREE);
	if(the_chunk != NULL)
	{
		if((hk = g_mem_chunk_alloc(the_chunk)) != NULL)
			hkey_set(hk, st);
	}
	return hk;
}

/* 1998-12-23 -	Compare two history keys for equality. */
static gint hkey_equal(gconstpointer a, gconstpointer b)
{
	const HKey	*ha = a, *hb = b;

	return (ha->dev == hb->dev) && (ha->inode == hb->inode);
}

/* 1998-12-23 -	Compute hash value from a history key. Nothing fancy. */
static guint hkey_hash(gconstpointer a)
{
	const HKey	*ha = a;

	return (guint) ha->dev ^ (guint) ha->inode;
}

/* 1998-12-23 -	A g_hash_table_foreach() callback that frees a hash key. Note that both <key>
**		and <value> point at the same HKey structure here.
*/
static void hkey_free(gpointer key, gpointer value, gpointer user)
{
	g_mem_chunk_free(the_chunk, key);
}

/* 1999-06-06 -	A routine just like the one above, but for g_hash_table_foreach_remove(). */
static gboolean hkey_free2(gpointer key, gpointer value, gpointer user)
{
	g_mem_chunk_free(the_chunk, key);

	return TRUE;		/* Remove me, please. */
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-05 -	Create a new empty DHEntry structure. */
static DHEntry * dhentry_new(void)
{
	DHEntry	*data;

	data = g_malloc(sizeof *data);
	data->dpath[0] = '\0';
	data->path[0] = '\0';
	data->realpath[0] = '\0';
	data->sel = NULL;
	data->vpos = 0.0f;
	data->focus_row = -1;

	return data;
}

static void dhentry_path_set(DHEntry *dhe, const gchar *path)
{
	Conv	conv;
	gchar	*dpl;

	g_snprintf(dhe->path, sizeof dhe->path, "%s", path);
	conv_init(&conv);
	dpl = conv_add_reverse(&conv, path);
	g_snprintf(dhe->dpath, sizeof dhe->dpath, "%s", dpl);
	conv_end(&conv);
}

static void dhentry_realpath_set(DHEntry *dhe, const gchar *realpath)
{
	g_snprintf(dhe->realpath, sizeof dhe->realpath, "%s", realpath);
}

/* 1999-06-09 -	Update <dhe> to mimic state of <dp>. */
static void dhentry_update(DHEntry *dhe, const DirPane *dp)
{
	if(dp->main->cfg.dp_history.select)
		dhe->sel = dirsel_set(dhe->sel, dp);
	else
	{
		dph_dirsel_destroy(dhe->sel);
		dhe->sel = NULL;
	}
	dhe->vpos = dph_vpos_get(dp);
	if(dp->focus_row != -1)
		dhe->focus_row = dp->focus_row;
	else
		dhe->focus_row = -1;
}

/* 1999-06-09 -	Apply state conserved in <dhe> to the rows of <dp>. */
static void dhentry_apply(const DHEntry *dhe, DirPane *dp)
{
	dph_dirsel_apply(dp, dhe->sel);
	dph_vpos_set(dp, dhe->vpos);
	if(dhe->focus_row != -1)	/* Was focus active? */
		dp_focus(dp, dhe->focus_row);
}

/* 1999-06-06 -	Destroy a directory history entry. */
static void dhentry_destroy(DHEntry *dh)
{
	if(dh->sel != NULL)
		g_hash_table_foreach_remove(dh->sel->hash, hkey_free2, NULL);
	g_free(dh->sel);
	g_free(dh);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-05 -	Create a new empty selection. */
static DHSel * dirsel_new(void)
{
	DHSel	*sel;

	sel = g_malloc(sizeof *sel);
	sel->hash = g_hash_table_new(hkey_hash, hkey_equal);

	return sel;
}

static gint dummy_func(gpointer a, gpointer b, gpointer c)
{
	return TRUE;
}

static void dirsel_clear(DHSel *sel)
{
	if((sel == NULL) || (sel->hash == NULL))
		return;

	g_hash_table_foreach_remove(sel->hash, dummy_func, NULL);
}

/* 1999-03-05 -	Given an <sel> structure, replace selection with that of <dp>. Returns the
**		new selection (or NULL if <dp> had no selected items). If <sel> is NULL on
**		entry, a new selection will be created and returned.
*/
static DHSel * dirsel_set(DHSel *sel, const DirPane *dp)
{
	GSList	*slist;

	if((slist = dp_get_selection_full(dp)) != NULL)
	{
		HKey		*key;
		const GSList	*iter;

		if(sel == NULL)
			sel = dirsel_new();
		else
			dirsel_clear(sel);
		
		for(iter = slist; iter != NULL; iter = g_slist_next(iter))
		{
			key = hkey_new(&DP_SEL_LSTAT(iter));
			g_hash_table_insert(sel->hash, key, key);	/* *Value* is returned on lookup!! */
		}
		dp_free_selection(slist);
		return sel;
	}
	return NULL;
}

/* 1999-03-05 -	This returns an opaque representation of all selected rows of <dp>. The
**		selection is not related to the order in which these rows are displayed
**		in the pane, so it's handy to use before e.g. resorting the pane.
**		Note that NULL is a valid representation if there is no selection.
*/
DHSel * dph_dirsel_new(DirPane *dp)
{
	return dirsel_set(NULL, dp);
}

/* 1999-03-05 -	Apply given given <sel> selection to <dp>, making those rows selected
**		again. <dp> need not be the same as when the selection was created,
**		and it need not have the same contents. This is not terribly efficient,
**		but I think it'll be OK.
*/
void dph_dirsel_apply(DirPane *dp, const DHSel *sel)
{
	HKey	key;
	guint	i;

	if((sel == NULL) || (sel->hash == NULL))
		return;

	for(i = 0; i < dp->dir.num_rows; i++)
	{
		hkey_set(&key, &DP_ROW_LSTAT(&dp->dir.row[i]));
		if(g_hash_table_lookup(sel->hash, &key))
			dp_select(dp, i);
	}
}

/* 1999-03-05 -	Destroy a selection, this is handy when you're done with it (like after
**		having applied it).
*/
void dph_dirsel_destroy(DHSel *sel)
{
	if(sel == NULL)
		return;

	if(sel->hash != NULL)
	{
		g_hash_table_foreach(sel->hash, hkey_free, NULL);
		g_hash_table_destroy(sel->hash);
	}
	g_free(sel);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-11-13 -	Return a floating-point number that somehow encodes the vertical position of
**		the given pane. The only use for that number is to pass it to dph_vpos_set().
*/
gfloat dph_vpos_get(const DirPane *dp)
{
	GtkAdjustment	*adj;

	if((adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(dp->scwin))) != NULL)
		return adj->value;
	return -1.0f;
}

/* 1999-11-13 -	Set the given pane's vertical position to resemble what is was when <vpos>
**		was returned by dph_vpos_get().
*/
void dph_vpos_set(DirPane *dp, gfloat vpos)
{
	GtkAdjustment	*adj;

	if((vpos >= 0.0f) && (adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(dp->scwin))) != NULL)
		gtk_adjustment_set_value(adj, vpos);
}

/* ----------------------------------------------------------------------------------------- */

/* 2002-06-13 -	A new way to look at history handling. For the, like, third time or so. Anyway,
**		this should just save the state of <dp>. This means the pane should already be
**		displaying a directory.
*/
void dph_state_save(DirPane *dp)
{
	gchar	rp[PATH_MAX];

	if(dp->dir.num_rows == 0)		/* No directory shown? Then give up. */
		return;
	if(realpath(dp->dir.path, rp) == NULL)
		return;
	if((dp->hist->history == NULL) || strcmp(rp, ((DHEntry *) dp->hist->history->data)->realpath))
	{
		dp->hist->history = g_list_prepend(dp->hist->history, dhentry_new());
		dhentry_path_set(dp->hist->history->data, dp->dir.path);
		dhentry_realpath_set(dp->hist->history->data, rp);
	}
	dhentry_update(dp->hist->history->data, dp);
}

/* 2002-06-13 -	Other half of the third-generation history interface. This restores any saved
**		state for the directory currently shown by <dp>.
*/
void dph_state_restore(DirPane *dp)
{
	GList	*iter;
	gchar	rp[PATH_MAX];
	DHEntry	*entry;

	/* Search for key matching current dir in history. */
	if(realpath(dp->dir.path, rp) == NULL)
		return;
	for(iter = dp->hist->history, entry = NULL; iter != NULL; iter = g_list_next(iter))
	{
		entry = iter->data;
		if(strcmp(rp, entry->realpath) == 0)
			break;
	}
	if(iter != NULL)
	{
		dhentry_apply(entry, dp);
		if(g_list_previous(iter) != NULL)
		{
			dp->hist->history = g_list_remove_link(dp->hist->history, iter);
			g_list_free_1(iter);
			dp->hist->history = g_list_prepend(dp->hist->history, entry);
		}
	}
	else
	{
		dp->hist->history = g_list_prepend(dp->hist->history, dhentry_new());
		dhentry_path_set(dp->hist->history->data, dp->dir.path);
		dhentry_realpath_set(dp->hist->history->data, rp);
		dhentry_apply(dp->hist->history->data, dp);
		/* Prune history list, if it has grown too long. */
		if(g_list_length(dp->hist->history) > HISTORY_SIZE)
		{
			GList	*tail, *next;

			tail = g_list_nth(dp->hist->history, HISTORY_SIZE);
			for(; tail != NULL; tail = next)
			{
				next = g_list_next(tail);
				dp->hist->history = g_list_remove_link(dp->hist->history, tail);
				dhentry_destroy(tail->data);
				g_list_free_1(tail);
			}
		}
	}
	/* Update path history widgets. Relies on <dpath> member being first in DHEntry. */
	dp_history_set(dp, dp->hist->history);
}

/* ----------------------------------------------------------------------------------------- */

/* 2002-02-15 -	Return first path from history for <dp>. */
const gchar * dph_history_get_first(const DirPane *dp)
{
	if(dp->hist->history != NULL)
		return dp->hist->history->data;
	return NULL;
}

/* ----------------------------------------------------------------------------------------- */

static const gchar * history_filename(MainInfo *min)
{
	static gchar	buffer[PATH_MAX];

	if(fut_path_component("~/.gentoo-history", buffer))
		return buffer;
	return NULL;
}

/* 2002-02-15 -	Save contents of history lists for <num> panes, arrayed at <dp>. Weird inter-
**		face, good code. In a blatant case of code reuse, we use our trusted XML module.
**		We do not, however, reuse the main config file, but keep the histories separate.
*/
void dph_history_save(MainInfo *min, const DirPane *dp, gsize num)
{
	FILE	*out;
	gsize	i;
	GList	*iter;

	if((out = xml_put_open(history_filename(min))) == NULL)
		return;

	xml_put_node_open(out, "DirHistory");
	xml_put_node_open(out, "Panes");
	for(i = 0; i < num; i++)
	{
		xml_put_node_open(out, "Pane");
		xml_put_uinteger(out, "index", i);
		xml_put_node_open(out, "History");
		for(iter = dp[i].hist->history; iter != NULL; iter = g_list_next(iter))
		{
			if(*(gchar *) iter->data == '\0')
				continue;
			xml_put_text(out, "dir", iter->data);
		}
		xml_put_node_close(out, "History");
		xml_put_node_close(out, "Pane");
	}
	xml_put_node_close(out, "Panes");
	xml_put_node_close(out, "DirHistory");

	xml_put_close(out);
}

/* ----------------------------------------------------------------------------------------- */

static void load_dir(const XmlNode *node, gpointer user)
{
	DirPane		*dp = user;
	const gchar	*text;
	DHEntry		*ent;

	if(!xml_get_text(node, "dir", &text))
		return;
	if(text && *text == '\0')
		return;
	ent = dhentry_new();
	stu_strncpy(ent->path, text, sizeof ent->path);
	if(realpath(ent->path, ent->realpath) != NULL)
	{
		Conv		conv;
		const gchar	*display;

		display = conv_begin_reverse(&conv, ent->path);
		g_strlcpy(ent->dpath, display, sizeof ent->dpath);
		dp->hist->history = g_list_append(dp->hist->history, ent);
		conv_end(&conv);
	}
	else
		dhentry_destroy(ent);
}

static void load_pane(const XmlNode *node, gpointer user)
{
	MainInfo	*min = user;
	guint		index;

	if(!xml_get_uinteger(node, "index", &index))
		return;
	if((node = xml_tree_search(node, "History")) == NULL)
		return;
	xml_node_visit_children(node, load_dir, min->gui->pane + index);
}

void dph_history_load(MainInfo *min, DirPane *dp, gsize num)
{
	XmlNode		*tree;
	const XmlNode	*node;

	if((tree = xml_tree_load(history_filename(min))) == NULL)
		return;

	if((node = xml_tree_search(tree, "DirHistory")) == NULL)
		return;
	if((node = xml_tree_search(node, "Panes")) == NULL)
		return;

	xml_node_visit_children(node, load_pane, min);
	xml_tree_destroy(tree);
}
