/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "domain.h"

#include "config.h"
#include "advection.h"
#include "source.h"
#ifdef HAVE_MPI
#  include "mpi_boundary.h"
#  include "init.h"
#endif /* HAVE_MPI */

/* GfsDomain: Object */

static void box_pressure_bc (GfsBox * box, gpointer * datum)
{
  FttTraverseFlags * flags = datum[0];
  gint * max_depth = datum[1];
  GfsVariable * p = datum[2];
  FttDirection d;

  for (d = 0; d < FTT_NEIGHBORS; d++) 
    if (GFS_IS_BOUNDARY (box->neighbor[d])) {
      GfsBoundary * boundary = GFS_BOUNDARY (box->neighbor[d]);

      if (GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->pressure) {
	GfsBoundaryClass * klass = 
	  GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass);

	boundary->v = p;
	ftt_face_traverse_boundary (boundary->root, 
				    boundary->d,
				    FTT_PRE_ORDER, *flags, *max_depth,
				    (FttFaceTraverseFunc) klass->pressure, 
				    boundary);
	gfs_boundary_send (boundary);
      }
    }
}

static void box_center_bc (GfsBox * box, GfsVariable * v)
{
  FttDirection d;

  for (d = 0; d < FTT_NEIGHBORS; d++) 
    if (GFS_IS_BOUNDARY (box->neighbor[d])) {
      GfsBoundary * boundary = GFS_BOUNDARY (box->neighbor[d]);
      
      if (GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->center) {
	GfsBoundaryClass * klass = 
	  GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass);
	
	boundary->v = v;
	ftt_face_traverse_boundary (boundary->root, 
				    boundary->d,
				    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
				    (FttFaceTraverseFunc) klass->center,
				    boundary);
        gfs_boundary_send (boundary);
      }
    }
}

static void direction_face_bc (GtsObject * neighbor,
			       GfsVariable * v)
{
  if (GFS_IS_BOUNDARY (neighbor)) {
    GfsBoundary * boundary = GFS_BOUNDARY (neighbor);

    if (GFS_BOUNDARY_CLASS (neighbor->klass)->face) {
      GfsBoundaryClass * klass = GFS_BOUNDARY_CLASS (neighbor->klass);
    
      boundary->v = v;
      boundary->type = GFS_BOUNDARY_CENTER_VARIABLE;
      ftt_face_traverse_boundary (boundary->root, 
				  boundary->d,
				  FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
				  (FttFaceTraverseFunc) klass->face,
				  boundary);
      boundary->type = GFS_BOUNDARY_FACE_VARIABLE;
      gfs_boundary_send (boundary);
    }
  }
}

static void box_face_bc (GfsBox * box, gpointer * datum)
{
  GfsVariable * v = datum[2];
  FttComponent * c = datum[3];

  if (*c == FTT_XYZ) {
    FttDirection d;
    
    for (d = 0; d < FTT_NEIGHBORS; d++)
      direction_face_bc (box->neighbor[d], v);
  }
  else {
    direction_face_bc (box->neighbor[2*(*c)], v);
    direction_face_bc (box->neighbor[2*(*c) + 1], v);
  }
}

static void box_receive_bc (GfsBox * box, gpointer * datum)
{
  FttTraverseFlags * flags = datum[0];
  gint * max_depth = datum[1];
  FttComponent * c = datum[3];

  if (*c == FTT_XYZ) {
    FttDirection d;
    
    for (d = 0; d < FTT_NEIGHBORS; d++) {
      FttDirection od = FTT_OPPOSITE_DIRECTION (d);

      if (GFS_IS_BOUNDARY (box->neighbor[od]))
	gfs_boundary_receive (GFS_BOUNDARY (box->neighbor[od]),
			     *flags, *max_depth);
    }
  }
  else {
    if (GFS_IS_BOUNDARY (box->neighbor[2*(*c) + 1]))
      gfs_boundary_receive (GFS_BOUNDARY (box->neighbor[2*(*c) + 1]),
			   *flags, *max_depth);
    if (GFS_IS_BOUNDARY (box->neighbor[2*(*c)]))
      gfs_boundary_receive (GFS_BOUNDARY (box->neighbor[2*(*c)]),
			   *flags, *max_depth);
  }
}

static void box_match (GfsBox * box)
{
  FttDirection d;

  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (GFS_IS_BOUNDARY (box->neighbor[d])) {
      GfsBoundary * boundary = GFS_BOUNDARY (box->neighbor[d]);

      g_assert (GFS_BOUNDARY_CLASS (box->neighbor[d]->klass)->match);
      boundary->type = GFS_BOUNDARY_MATCH_VARIABLE;
      (* GFS_BOUNDARY_CLASS (box->neighbor[d]->klass)->match) (boundary);
      gfs_boundary_send (boundary);
    }
}

static void box_synchronize (GfsBox * box, FttComponent * c)
{
  if (*c == FTT_XYZ) {
    FttDirection d;
    
    for (d = 0; d < FTT_NEIGHBORS; d++)
      if (GFS_IS_BOUNDARY (box->neighbor[d]))
	gfs_boundary_synchronize (GFS_BOUNDARY (box->neighbor[d]));
  }
  else {
    if (GFS_IS_BOUNDARY (box->neighbor[2*(*c)]))
      gfs_boundary_synchronize (GFS_BOUNDARY (box->neighbor[2*(*c)]));
    if (GFS_IS_BOUNDARY (box->neighbor[2*(*c) + 1]))
      gfs_boundary_synchronize (GFS_BOUNDARY (box->neighbor[2*(*c) + 1]));
  }
}

static void domain_pressure_bc (GfsDomain * domain,
				FttTraverseFlags flags,
				gint max_depth,
				GfsVariable * p)
{
  FttComponent c = FTT_XYZ;
  gpointer datum[4];

  datum[0] = &flags;
  datum[1] = &max_depth;
  datum[2] = p;
  datum[3] = &c;

  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_pressure_bc, datum);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_receive_bc, datum);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_synchronize, &c);
}

static void domain_center_bc (GfsDomain * domain,
			      GfsVariable * v)
{
  FttComponent c = FTT_XYZ;
  FttTraverseFlags flags = FTT_TRAVERSE_LEAFS;
  gint max_depth = -1;
  gpointer datum[4];

  datum[0] = &flags;
  datum[1] = &max_depth;
  datum[2] = v;
  datum[3] = &c;

  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_center_bc, v);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_receive_bc, datum);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_synchronize, &c);
}

static void domain_face_bc (GfsDomain * domain,
			    FttComponent c,
			    GfsVariable * v)
{
  FttTraverseFlags flags = FTT_TRAVERSE_LEAFS;
  gint max_depth = -1;
  gpointer datum[4];

  datum[0] = &flags;
  datum[1] = &max_depth;
  datum[2] = v;
  datum[3] = &c;

  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_face_bc, datum);
  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_receive_bc, datum);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_synchronize, &c);
}

static void domain_match (GfsDomain * domain)
{
  FttComponent c = FTT_XYZ;
  FttTraverseFlags flags = FTT_TRAVERSE_LEAFS;
  gint max_depth = -1;
  gpointer datum[4];

  datum[0] = &flags;
  datum[1] = &max_depth;
  datum[2] = NULL;
  datum[3] = &c;

  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_match, NULL);
  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_receive_bc, datum);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_synchronize, &c);
}

static void domain_write (GtsObject * o, FILE * fp)
{
  GfsDomain * domain = GFS_DOMAIN (o);

  if (GTS_OBJECT_CLASS (gfs_domain_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_domain_class ())->parent_class->write) (o, fp);

  fputs (" { ", fp);
  if (domain->rootlevel != 0)
    fprintf (fp, "rootlevel = %u ", domain->rootlevel);
  if (domain->refpos.x != 0.)
    fprintf (fp, "x = %g ", domain->refpos.x);
  if (domain->refpos.y != 0.)
    fprintf (fp, "y = %g ", domain->refpos.y);
  if (domain->refpos.z != 0.)
    fprintf (fp, "z = %g ", domain->refpos.z);
  if (domain->lambda.x != 1.)
    fprintf (fp, "lx = %g ", domain->lambda.x);
  if (domain->lambda.y != 1.)
    fprintf (fp, "ly = %g ", domain->lambda.y);
  if (domain->lambda.z != 1.)
    fprintf (fp, "lz = %g ", domain->lambda.z);
  if (domain->max_depth_write > -2) {
    GfsVariable * v = domain->variables_io;

    if (v != NULL) {
      fprintf (fp, "variables = %s", v->name);
      v = v->next;
      while (v) {
	fprintf (fp, ",%s", v->name);
	v = v->next;
      }
      fputc (' ', fp);
    }
  }
  fputc ('}', fp);
}

static void domain_read (GtsObject ** o, GtsFile * fp)
{
  GfsDomain * domain = GFS_DOMAIN (*o);
  GtsFileVariable var[] = {
    {GTS_UINT,   "rootlevel", TRUE},
    {GTS_DOUBLE, "x",         TRUE},
    {GTS_DOUBLE, "y",         TRUE},
    {GTS_DOUBLE, "z",         TRUE},
    {GTS_DOUBLE, "lx",        TRUE},
    {GTS_DOUBLE, "ly",        TRUE},
    {GTS_DOUBLE, "lz",        TRUE},
    {GTS_STRING, "variables", TRUE},
    {GTS_NONE}
  };
  gchar * variables = NULL;

  if (GTS_OBJECT_CLASS (gfs_domain_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_domain_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  var[0].data = &domain->rootlevel;
  var[1].data = &domain->refpos.x;
  var[2].data = &domain->refpos.y;
  var[3].data = &domain->refpos.z;
  var[4].data = &domain->lambda.x;
  var[5].data = &domain->lambda.y;
  var[6].data = &domain->lambda.z;
  var[7].data = &variables;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR) {
    g_free (variables);
    return;
  }

  if (var[4].set && domain->lambda.x <= 0.) {
    gts_file_error (fp, "lx must be strictly positive");
    return;
  }
  if (var[5].set && domain->lambda.y <= 0.) {
    gts_file_error (fp, "ly must be strictly positive");
    return;
  }
  if (var[6].set && domain->lambda.z <= 0.) {
    gts_file_error (fp, "lz must be strictly positive");
    return;
  }

  if (variables != NULL) {
    gchar * variables1, * s;
    gboolean empty = TRUE;

    variables1 = g_strdup (variables);
    s = strtok (variables1, ",");
    while (s) {
      gfs_domain_add_variable (domain, s);
      empty = FALSE;
      s = strtok (NULL, ",");
    }
    g_free (variables1);

    if (!empty) {
      gchar * error;

      if (domain->variables_io != domain->variables)
	gfs_variable_list_destroy (domain->variables_io);
      domain->variables_io = gfs_variables_from_list (domain->variables, 
						      variables, &error);
      g_assert (domain->variables_io);
    }
    g_free (variables);
  }
}

static void domain_destroy (GtsObject * o)
{
  GfsDomain * domain = GFS_DOMAIN (o);

  g_timer_destroy (domain->timer);
  gfs_variable_list_destroy (domain->variables);
  if (domain->variables_io != domain->variables)
    gfs_variable_list_destroy (domain->variables_io);

  (* GTS_OBJECT_CLASS (gfs_domain_class ())->parent_class->destroy) (o);
}

static void domain_class_init (GfsDomainClass * klass)
{
  klass->pressure_bc = domain_pressure_bc;
  klass->center_bc   = domain_center_bc;
  klass->face_bc     = domain_face_bc;
  klass->match       = domain_match;

  GTS_OBJECT_CLASS (klass)->read = domain_read;
  GTS_OBJECT_CLASS (klass)->write = domain_write;
  GTS_OBJECT_CLASS (klass)->destroy = domain_destroy;
}

static void domain_init (GfsDomain * domain)
{
#ifdef HAVE_MPI
  int size;

  MPI_Comm_size (MPI_COMM_WORLD, &size);
  if (size > 1)
    MPI_Comm_rank (MPI_COMM_WORLD, &domain->pid);
  else
    domain->pid = -1;
#else /* not HAVE_MPI */
  domain->pid = -1;
#endif /* not HAVE_MPI */

  domain->timer = g_timer_new ();
  gts_range_init (&domain->mac_projection);
  gts_range_init (&domain->approximate_projection);
  gts_range_init (&domain->centered_velocity_advection);
  gts_range_init (&domain->tracer_advection);
  gts_range_init (&domain->predicted_face_velocities);
  gts_range_init (&domain->adapt);
  gts_range_init (&domain->timestep);
  gts_range_init (&domain->size);

  gts_range_init (&domain->pressure_bc);
  gts_range_init (&domain->center_bc);
  gts_range_init (&domain->face_bc);
  gts_range_init (&domain->match);
  domain->profile_bc = FALSE;

  gts_range_init (&domain->mpi_messages);
  gts_range_init (&domain->mpi_wait);

  domain->rootlevel = 0;
  domain->refpos.x = domain->refpos.y = domain->refpos.z = 0.;
  domain->lambda.x = domain->lambda.y = domain->lambda.z = 1.;
  domain->variables = gfs_variable_list_copy (gfs_centered_variables, 
					      GTS_OBJECT (domain));
  domain->variables_size = sizeof (GfsStateVector);
  domain->variables_io = domain->variables;
  domain->max_depth_write = -1;
}

GfsDomainClass * gfs_domain_class (void)
{
  static GfsDomainClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_domain_info = {
      "GfsDomain",
      sizeof (GfsDomain),
      sizeof (GfsDomainClass),
      (GtsObjectClassInitFunc) domain_class_init,
      (GtsObjectInitFunc) domain_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gts_wgraph_class ()),
				  &gfs_domain_info);
  }

  return klass;
}

/**
 * gfs_domain_pressure_bc:
 * @domain: a #GfsDomain.
 * @flags: the traversal flags.
 * @max_depth: the maximum depth of the traversal.
 * @p: the #GfsVariable to use as pressure.
 *
 * Calls the @pressure_bc() method of @domain.
 */
void gfs_domain_pressure_bc (GfsDomain * domain,
			     FttTraverseFlags flags,
			     gint max_depth,
			     GfsVariable * p)
{
  gdouble start = 0., end;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (p != NULL);

  if (domain->profile_bc)
    start = g_timer_elapsed (domain->timer, NULL);

  g_assert (GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->pressure_bc);
  (* GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->pressure_bc) 
    (domain, flags, max_depth, p);

  if (domain->profile_bc) {
    end = g_timer_elapsed (domain->timer, NULL);
    gts_range_add_value (&domain->pressure_bc, end - start);
    gts_range_update (&domain->pressure_bc);
  }
}

/**
 * gfs_domain_center_bc:
 * @domain: a #GfsDomain.
 * @v: a #GfsVariable.
 *
 * Calls the @center_bc() method of @domain.
 */
void gfs_domain_center_bc (GfsDomain * domain,
			   GfsVariable * v)
{
  gdouble start = 0., end;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (v != NULL);

  if (domain->profile_bc)
    start = g_timer_elapsed (domain->timer, NULL);

  g_assert (GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->center_bc);
  (* GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->center_bc) (domain, v);

  if (domain->profile_bc) {
    end = g_timer_elapsed (domain->timer, NULL);
    gts_range_add_value (&domain->center_bc, end - start);
    gts_range_update (&domain->center_bc);
  }
}

/**
 * gfs_domain_face_bc:
 * @domain: a #GfsDomain.
 * @c: a component.
 * @v: a #GfsVariable.
 *
 * Calls the @face_bc() method of @domain.
 */
void gfs_domain_face_bc (GfsDomain * domain,
			 FttComponent c,
			 GfsVariable * v)
{
  gdouble start = 0., end;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (c == FTT_XYZ || (c >= 0 && c < FTT_DIMENSION));
  g_return_if_fail (v != NULL);

  if (domain->profile_bc)
    start = g_timer_elapsed (domain->timer, NULL);

  g_assert (GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->face_bc);
  (* GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->face_bc) (domain, c, v);

  if (domain->profile_bc) {
    end = g_timer_elapsed (domain->timer, NULL);
    gts_range_add_value (&domain->face_bc, end - start);
    gts_range_update (&domain->face_bc);
  }
}

/**
 * gfs_domain_match:
 * @domain: a #GfsDomain.
 *
 * Calls the @match() method of @domain.
 */
void gfs_domain_match (GfsDomain * domain)
{
  gdouble start = 0., end;

  g_return_if_fail (domain != NULL);

  if (domain->profile_bc)
    start = g_timer_elapsed (domain->timer, NULL);

  g_assert (GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->match);
  /* two calls are necessay for certain matching cycles (for corners
     between domains. see match.fig) */
  (* GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->match) (domain);
  (* GFS_DOMAIN_CLASS (GTS_OBJECT (domain)->klass)->match) (domain);

  if (domain->profile_bc) {
    end = g_timer_elapsed (domain->timer, NULL);
    gts_range_add_value (&domain->match, end - start);
    gts_range_update (&domain->match);
  }
}

static void box_traverse (GfsBox * box, gpointer * datum)
{
  FttTraverseType * order = datum[0];
  FttTraverseFlags * flags = datum[1];
  gint * max_depth = datum[2];
  FttCellTraverseFunc func = (FttCellTraverseFunc) datum[3];
  gpointer data = datum[4];

  ftt_cell_traverse (box->root, *order, *flags, *max_depth, func, data);
}

/**
 * gfs_domain_cell_traverse:
 * @domain: a #GfsDomain.
 * @order: the order in which the cells are visited - %FTT_PRE_ORDER,
 * %FTT_POST_ORDER. 
 * @flags: which types of children are to be visited.
 * @max_depth: the maximum depth of the traversal. Cells below this
 * depth will not be traversed. If @max_depth is -1 all cells in the
 * tree are visited.
 * @func: the function to call for each visited #FttCell.
 * @data: user data to pass to @func.
 *
 * Traverses the cell trees of @domain. Calls the given function for
 * each cell visited.  
 */
void gfs_domain_cell_traverse (GfsDomain * domain,
			       FttTraverseType order,
			       FttTraverseFlags flags,
			       gint max_depth,
			       FttCellTraverseFunc func,
			       gpointer data)
{
  gpointer datum[5];

  datum[0] = &order;
  datum[1] = &flags;
  datum[2] = &max_depth;
  datum[3] = func;
  datum[4] = data;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (func != NULL);

  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_traverse, datum);
}

static void box_traverse_box (GfsBox * box, gpointer * datum)
{
  FttTraverseType * order = datum[0];
  FttTraverseFlags * flags = datum[1];
  gint * max_depth = datum[2];
  FttCellTraverseFunc func = (FttCellTraverseFunc) datum[3];
  gpointer data = datum[4];
  GtsBBox * bb = datum[5];

  ftt_cell_traverse_box (box->root, bb, 
			 *order, *flags, *max_depth, func, data);
}

/**
 * gfs_domain_cell_traverse_box:
 * @domain: a #GfsDomain.
 * @box: a #GtsBBox.
 * @order: the order in which the cells are visited - %FTT_PRE_ORDER,
 * %FTT_POST_ORDER. 
 * @flags: which types of children are to be visited.
 * @max_depth: the maximum depth of the traversal. Cells below this
 * depth will not be traversed. If @max_depth is -1 all cells in the
 * tree are visited.
 * @func: the function to call for each visited #FttCell.
 * @data: user data to pass to @func.
 *
 * Traverses the cell trees of @domain. Calls the given function for
 * each cell visited. Only the cells overlapping with @box are visited.
 */
void gfs_domain_cell_traverse_box (GfsDomain * domain,
				  GtsBBox * box,
				  FttTraverseType order,
				  FttTraverseFlags flags,
				  gint max_depth,
				  FttCellTraverseFunc func,
				  gpointer data)
{
  gpointer datum[6];

  datum[0] = &order;
  datum[1] = &flags;
  datum[2] = &max_depth;
  datum[3] = func;
  datum[4] = data;
  datum[5] = box;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (box != NULL);
  g_return_if_fail (func != NULL);

  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) box_traverse_box, datum);
}

static void traverse_mixed (GfsBox * box, gpointer * datum)
{
  FttCellTraverseFunc func = (FttCellTraverseFunc) datum[0];
  gpointer data = datum[1];

  gfs_cell_traverse_mixed (box->root, func, data);
}

/**
 * gfs_domain_traverse_mixed:
 * @domain: a #GfsDomain.
 * @func: the function to call for each visited #FttCell.
 * @data: user data to pass to @func.
 *
 * Calls @func for each leaf mixed cell of @domain.
 */
void gfs_domain_traverse_mixed (GfsDomain * domain,
			       FttCellTraverseFunc func,
			       gpointer data)
{
  gpointer datum[2];

  datum[0] = func;
  datum[1] = data;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (func != NULL);

  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) traverse_mixed, datum);
}

static void box_depth (GfsBox * box, guint * depth)
{
  guint d = ftt_cell_depth (box->root);

  if (d > *depth)
    *depth = d;
}

/**
 * gfs_domain_depth:
 * @domain: a #GfsDomain.
 *
 * Returns: the maximum depth of the cell trees of @domain. This
 * function is global i.e. it returns the maximum depth over all the
 * processes (for parallel execution).
 */
guint gfs_domain_depth (GfsDomain * domain)
{
  guint depth = 0;

  g_return_val_if_fail (domain != NULL, 0);

  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) box_depth, &depth);
#ifdef HAVE_MPI
  if (domain->pid >= 0) {
    guint global_depth;

    MPI_Allreduce (&depth, &global_depth, 1, MPI_UNSIGNED, MPI_MAX, 
		   MPI_COMM_WORLD);
    depth = global_depth;
  }
#endif /* HAVE_MPI */
  return depth;
}

#include "ftt_internal.c"

/**
 * gfs_domain_face_traverse:
 * @domain: a #GfsDomain.
 * @c: only the faces orthogonal to this component will be traversed - one of
 * %FTT_X, %FTT_Y, (%FTT_Z), %FTT_XYZ.
 * @order: the order in which the cells are visited - %FTT_PRE_ORDER,
 * %FTT_POST_ORDER. 
 * @flags: which types of children and faces are to be visited.
 * @max_depth: the maximum depth of the traversal. Cells below this
 * depth will not be traversed. If @max_depth is -1 all cells in the
 * tree are visited.
 * @func: the function to call for each visited #FttCellFace.
 * @data: user data to pass to @func.
 *
 * Traverses a @domain. Calls the given function for each face
 * of the cells of the domain.
 *
 * If %FTT_TRAVERSE_BOUNDARY_FACES is not set in @flags, only
 * "double-sided" faces are traversed i.e. the @neighbor field of the
 * face is never %NULL.  
 */
void gfs_domain_face_traverse (GfsDomain * domain,
			       FttComponent c,
			       FttTraverseType order,
			       FttTraverseFlags flags,
			       gint max_depth,
			       FttFaceTraverseFunc func,
			       gpointer data)
{
  FttDirection d;
  gpointer datum[6];
  gboolean check = FALSE;
  gboolean boundary_faces;
  
  g_return_if_fail (domain != NULL);
  g_return_if_fail (c >= FTT_X && c <= FTT_XYZ);
  g_return_if_fail (func != NULL);

  boundary_faces = ((flags & FTT_TRAVERSE_BOUNDARY_FACES) != 0);
  datum[1] = &max_depth;
  datum[2] = func;
  datum[3] = data;
  datum[4] = &check;
  datum[5] = &boundary_faces;
  if (c == FTT_XYZ) {
    if (boundary_faces) {
      check = TRUE;
      gfs_domain_cell_traverse (domain, order, flags, max_depth, 
	  (FttCellTraverseFunc) traverse_all_faces, 
				datum);
    }
    else {
      gfs_domain_cell_traverse (domain, order, flags, max_depth, 
	  (FttCellTraverseFunc) traverse_all_direct_faces, 
				datum);
      datum[0] = &d;
      check = TRUE;
      for (d = 1; d < FTT_NEIGHBORS; d += 2)
	gfs_domain_cell_traverse_boundary (domain, 
					   d, order, flags, max_depth, 
		     (FttCellTraverseFunc) traverse_face, datum);
    }
  }
  else {
    if (boundary_faces) {
      check = TRUE;
      datum[0] = &c;
      gfs_domain_cell_traverse (domain, order, flags, max_depth, 
				(FttCellTraverseFunc) traverse_face_component,
				datum);
    }
    else {
      d = 2*c;
      datum[0] = &d;
      gfs_domain_cell_traverse (domain, order, flags, max_depth, 
				(FttCellTraverseFunc) traverse_face_direction, 
				datum);
      d = 2*c + 1;
      check = TRUE;
      gfs_domain_cell_traverse_boundary (domain, d, order, flags, max_depth, 
		   (FttCellTraverseFunc) traverse_face, datum);
    }
  }
  gfs_domain_cell_traverse (domain, order, flags, max_depth, 
			    (FttCellTraverseFunc) reset_flag, NULL);
}

static void cell_traverse_boundary (GfsBox * box, gpointer * datum)
{
  FttDirection * d = datum[0];

  if (!GFS_IS_BOX (box->neighbor[*d])) {
    FttTraverseType * order = datum[1];
    FttTraverseFlags * flags = datum[2];
    gint * max_depth = datum[3];
    FttCellTraverseFunc func = (FttCellTraverseFunc) datum[4];
    gpointer data = datum[5];

    ftt_cell_traverse_boundary (box->root, 
				*d, *order, *flags, *max_depth, func, data);
  }
}

/**
 * gfs_domain_cell_traverse_boundary:
 * @domain: a #GfsDomain.
 * @d: the direction of the boundary to traverse.
 * @order: the order in which the cells are visited - %FTT_PRE_ORDER,
 * %FTT_POST_ORDER. 
 * @flags: which types of children are to be visited.
 * @max_depth: the maximum depth of the traversal. Cells below this
 * depth will not be traversed. If @max_depth is -1 all cells in the
 * tree are visited.
 * @func: the function to call for each visited #FttCell.
 * @data: user data to pass to @func.
 *
 * Traverses the boundary of a domain in direction @d. Calls the given
 * function for each cell visited.  
 */
void gfs_domain_cell_traverse_boundary (GfsDomain * domain,
					FttDirection d,
					FttTraverseType order,
					FttTraverseFlags flags,
					gint max_depth,
					FttCellTraverseFunc func,
					gpointer data)
{
  gpointer datum[6];
  
  datum[0] = &d;
  datum[1] = &order;
  datum[2] = &flags;
  datum[3] = &max_depth;
  datum[4] = func;
  datum[5] = data;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (d < FTT_NEIGHBORS);
  g_return_if_fail (func != NULL);

  gts_container_foreach (GTS_CONTAINER (domain), 
			 (GtsFunc) cell_traverse_boundary, datum);
}

static void add_stats (const FttCell * cell, gpointer * data)
{
  GtsRange * s = data[0];
  GfsVariable * v = data[1];

  gts_range_add_value (s, GFS_VARIABLE (cell, v->i));
}

#ifdef HAVE_MPI
static void range_reduce (void * i, void * o, 
			  int * len,
			  MPI_Datatype * type)
{
  gdouble * in = (gdouble *) i;
  gdouble * inout = (gdouble *) o;
  g_assert (*len == 5);
  
  if (in[0] < inout[0]) /* min */
    inout[0] = in[0];
  if (in[1] > inout[1]) /* max */
    inout[1] = in[1];
  inout[2] += in[2];    /* sum */
  inout[3] += in[3];    /* sum2 */
  inout[4] += in[4];    /* n */
}

static void domain_range_reduce (GfsDomain * domain, GtsRange * s)
{
  if (domain->pid >= 0) {
    double in[5];
    double out[5] = { G_MAXDOUBLE, - G_MAXDOUBLE, 0., 0., 0. };
    MPI_Op op;
    
    MPI_Op_create (range_reduce, TRUE, &op);
    in[0] = s->min; in[1] = s->max; in[2] = s->sum; in[3] = s->sum2;
    in[4] = s->n;
    MPI_Allreduce (in, out, 5, MPI_DOUBLE, op, MPI_COMM_WORLD);
    MPI_Op_free (&op);
    s->min = out[0]; s->max = out[1]; s->sum = out[2]; s->sum2 = out[3];
    s->n = out[4];
  }
}
#endif /* HAVE_MPI */

/**
 * gfs_domain_stats_variable:
 * @domain: the domain to obtain statistics from.
 * @v: a #GfsVariable.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Traverses the domain defined by @domain using gfs_domain_cell_traverse()
 * and gathers statistics about variable @v.
 *
 * Returns: a #GtsRange containing the statistics about @v.
 */
GtsRange gfs_domain_stats_variable (GfsDomain * domain,
				    GfsVariable * v,
				    FttTraverseFlags flags,
				    gint max_depth)
{
  GtsRange s;
  gpointer data[2];

  g_return_val_if_fail (domain != NULL, s);
  g_return_val_if_fail (v != NULL, s);

  gts_range_init (&s);
  data[0] = &s;
  data[1] = v;
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth, 
			   (FttCellTraverseFunc) add_stats, data);
#ifdef HAVE_MPI
  domain_range_reduce (domain, &s);
#endif /* HAVE_MPI */
  gts_range_update (&s);

  return s;
}

static void add_stats_solid (FttCell * cell, GtsRange * s)
{
  gts_range_add_value (s, GFS_STATE (cell)->solid->a);
}

/**
 * gfs_domain_stats_solid:
 * @domain: the domain to obtain statistics from.
 *
 * Traverses the domain defined by @domain using gfs_domain_traverse_mixed()
 * and gathers statistics about the solid volume fraction in mixed cells.
 *
 * Returns: statistics about the solid volume fraction @a in mixed cells.
 */
GtsRange gfs_domain_stats_solid (GfsDomain * domain)
{
  GtsRange s;

  g_return_val_if_fail (domain != NULL, s);

  gts_range_init (&s);
  gfs_domain_traverse_mixed (domain,
			    (FttCellTraverseFunc) add_stats_solid, &s);
#ifdef HAVE_MPI
  domain_range_reduce (domain, &s);
#endif /* HAVE_MPI */
  gts_range_update (&s);

  return s;
}

static void add_stats_merged (GSList * m, gpointer * data)
{
  GtsRange * solid =  data[0];
  GtsRange * number = data[1];
  gdouble a = 0.;
  guint n = 0;

  while (m) {
    FttCell * c = m->data;

    a += GFS_IS_MIXED (c) ? GFS_STATE (c)->solid->a : 1.;
    n++;
    m = m->next;
  }
  if (n > 1 || a < 1.)
    gts_range_add_value (solid, a);
  if (n > 1)
    gts_range_add_value (number, n);
}

/**
 * gfs_domain_stats_merged:
 * @domain: the domain to obtain statistics from.
 * @solid: #GtsRange in which to return stats for the total solid
 * volume fraction of merged cells. 
 * @number: #GtsRange in which to return stats for the number of cells
 * used per merged cell.
 *
 * Traverses the domain defined by @domain using
 * gfs_domain_traverse_merged() and gathers statistics about the total
 * solid volume fraction of merged cells and the number of cells used
 * per merged cell.
 */
void gfs_domain_stats_merged (GfsDomain * domain,
			     GtsRange * solid,
			     GtsRange * number)
{
  gpointer data[2];

  g_return_if_fail (domain != NULL);
  g_return_if_fail (solid != NULL);
  g_return_if_fail (number != NULL);

  gts_range_init (solid);
  gts_range_init (number);
  data[0] = solid;
  data[1] = number;
  gfs_domain_traverse_merged (domain,
			     (GfsMergedTraverseFunc) add_stats_merged, data);
#ifdef HAVE_MPI
  domain_range_reduce (domain, solid);
  domain_range_reduce (domain, number);
#endif /* HAVE_MPI */
  gts_range_update (solid);
  gts_range_update (number);
}

static void cell_count (FttCell * cell, guint * count)
{
  (*count)++;
}

#ifdef HAVE_MPI
static void boundary_size (GfsBox * box, guint * count)
{
  FttDirection d;

  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (GFS_IS_BOUNDARY_MPI (box->neighbor[d]))
      ftt_cell_traverse (GFS_BOUNDARY (box->neighbor[d])->root,
			 FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			 (FttCellTraverseFunc) cell_count, count);
}
#endif /* HAVE_MPI */

/**
 * gfs_domain_stats_balance:
 * @domain: the domain to obtain statistics from.
 * @size: #GtsRange in which to return stats for the total size of the domain.
 * @boundary: #GtsRange in which to return stats for the size of the parallel 
 * boundaries of the domain.
 * @mpiwait:  #GtsRange in which to return stats for the average time spend
 * waiting for MPI calls in each PE.
 *
 * Gathers statistics about the sizes of the domains, their parallel
 * boundaries and the execution time on each PE.  
 */
void gfs_domain_stats_balance (GfsDomain * domain,
			       GtsRange * size,
			       GtsRange * boundary,
			       GtsRange * mpiwait)
{
  guint count;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (size != NULL);
  g_return_if_fail (boundary != NULL);
  g_return_if_fail (mpiwait != NULL);

  gts_range_init (size);
  gts_range_init (boundary);
  gts_range_init (mpiwait);
  count = 0;
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			   (FttCellTraverseFunc) cell_count, &count);
  gts_range_add_value (size, count);
  if (domain->timestep.n > 0)
    gts_range_add_value (mpiwait, domain->mpi_wait.sum/domain->timestep.n);
#ifdef HAVE_MPI
  count = 0;
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) boundary_size, &count);
  gts_range_add_value (boundary, count);
  domain_range_reduce (domain, size);
  domain_range_reduce (domain, boundary);
  domain_range_reduce (domain, mpiwait);
#endif /* HAVE_MPI */
  gts_range_update (size);
  gts_range_update (boundary);
  gts_range_update (mpiwait);
}

static void add_norm (const FttCell * cell, gpointer * data)
{
  GfsNorm * n = data[0];
  GfsVariable * v = data[1];

  gfs_norm_add (n, GFS_VARIABLE (cell, v->i),
		ftt_cell_volume (cell)*(GFS_IS_MIXED (cell) ?
					GFS_STATE (cell)->solid->a : 1.));
}

#ifdef HAVE_MPI
static void norm_reduce (void * i, void * o, 
			 int * len,
			 MPI_Datatype * type)
{
  gdouble * in = (gdouble *) i;
  gdouble * inout = (gdouble *) o;
  g_assert (*len == 5);
  
  inout[0] += in[0];    /* bias */
  inout[1] += in[1];    /* first */
  inout[2] += in[2];    /* second */
  if (in[3] > inout[3]) /* infty */
    inout[3] = in[3];    
  inout[4] += in[4];    /* w */
}

static void domain_norm_reduce (GfsDomain * domain, GfsNorm * n)
{
  if (domain->pid >= 0) {
    double in[5];
    double out[5] = { 0., 0., 0., - G_MAXDOUBLE, 0. };
    MPI_Op op;

    MPI_Op_create (norm_reduce, TRUE, &op);
    in[0] = n->bias; in[1] = n->first; in[2] = n->second; in[3] = n->infty;
    in[4] = n->w;
    MPI_Allreduce (in, out, 5, MPI_DOUBLE, op, MPI_COMM_WORLD);
    MPI_Op_free (&op);
    n->bias = out[0]; n->first = out[1]; n->second = out[2]; n->infty = out[3];
    n->w = out[4];
  }
}
#endif /* HAVE_MPI */

/**
 * gfs_domain_norm_variable:
 * @domain: the domain to obtain norm from.
 * @v: a #GfsVariable.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Traverses the domain defined by @domain using gfs_domain_cell_traverse()
 * and gathers norm statistics about variable @v.
 *
 * Returns: a #GfsNorm containing the norm statistics about @v.
 */
GfsNorm gfs_domain_norm_variable (GfsDomain * domain,
				  GfsVariable * v,
				  FttTraverseFlags flags,
				  gint max_depth)
{
  GfsNorm n;
  gpointer data[2];

  g_return_val_if_fail (domain != NULL, n);
  g_return_val_if_fail (v != NULL, n);
  
  gfs_norm_init (&n);
  data[0] = &n;
  data[1] = v;
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth, 
			   (FttCellTraverseFunc) add_norm, data);
#ifdef HAVE_MPI
  domain_norm_reduce (domain, &n);
#endif /* HAVE_MPI */
  gfs_norm_update (&n);

  return n;
}

static void add_norm_residual (const FttCell * cell, GfsNorm * n)
{
  gdouble size = ftt_cell_size (cell);

  gfs_norm_add (n, GFS_STATE (cell)->res/(size*size), 1.);
}

/**
 * gfs_domain_norm_residual:
 * @domain: the domain to obtain the norm from.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Traverses the domain defined by @domain using gfs_domain_cell_traverse()
 * and gathers norm statistics about the volume weighted relative residual
 * (i.e. the sum of the residual over the volume defined by each cell
 * divided by the total volume of the cell).
 *
 * Returns: a #GfsNorm containing the norm statistics about the volume
 * weighted relative residual.  
 */
GfsNorm gfs_domain_norm_residual (GfsDomain * domain,
				FttTraverseFlags flags,
				gint max_depth)
{
  GfsNorm n;

  g_return_val_if_fail (domain != NULL, n);
  
  gfs_norm_init (&n);
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth, 
			   (FttCellTraverseFunc) add_norm_residual, &n);
#ifdef HAVE_MPI
  domain_norm_reduce (domain, &n);
#endif /* HAVE_MPI */
  gfs_norm_update (&n);

  return n;
}

static void add_norm_velocity (const FttCell * cell, GfsNorm * n)
{
  FttComponent c;
  gdouble unorm = 0.;
  
  for (c = 0; c < FTT_DIMENSION; c++) {
    gdouble uc = GFS_VARIABLE (cell, GFS_VELOCITY_INDEX (c));

    unorm += uc*uc;
  }
  gfs_norm_add (n, sqrt (unorm), 
		ftt_cell_volume (cell)*(GFS_IS_MIXED (cell) ? 
					GFS_STATE (cell)->solid->a : 1.));
}

/**
 * gfs_domain_norm_velocity:
 * @domain: the domain to obtain the norm from.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Traverses the domain defined by @domain using gfs_domain_cell_traverse()
 * and gathers norm statistics about velocity.
 *
 * Returns: a #GfsNorm containing the norm statistics about the velocity.
 */
GfsNorm gfs_domain_norm_velocity (GfsDomain * domain,
				  FttTraverseFlags flags,
				  gint max_depth)
{
  GfsNorm n;

  g_return_val_if_fail (domain != NULL, n);
  
  gfs_norm_init (&n);
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth, 
			   (FttCellTraverseFunc) add_norm_velocity, &n);
#ifdef HAVE_MPI
  domain_norm_reduce (domain, &n);
#endif /* HAVE_MPI */
  gfs_norm_update (&n);

  return n;
}

static void set_ref_pos (GfsBox * box, FttVector * pos)
{
  if (box->id == 1)
    gfs_box_set_pos (box, pos);
}

#ifdef HAVE_MPI
static void removed_list (GfsBox * box, gpointer * data)
{
  GfsDomain * domain = data[0];
  GSList ** removed = data[1];

  if (box->pid != domain->pid)
    *removed = g_slist_prepend (*removed, box);
}

static void mpi_links (GfsBox * box, GfsDomain * domain)
{
  FttDirection d;
  GtsObject * neighbor[FTT_NEIGHBORS];
  gint pid = box->pid;
  gint id = box->id;

  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (GFS_IS_BOX (box->neighbor[d]) && 
	GFS_BOX (box->neighbor[d])->pid == domain->pid)
      neighbor[d] = box->neighbor[d];
    else
      neighbor[d] = NULL;
  gts_object_destroy (GTS_OBJECT (box));

  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (neighbor[d])
#ifndef DUMMY_MPI
      gfs_boundary_mpi_new (gfs_boundary_mpi_class (),
			   GFS_BOX (neighbor[d]), 
			   FTT_OPPOSITE_DIRECTION (d), 
			   pid, id);
#else  /* DUMMY_MPI */
      gfs_boundary_new (GFS_BOUNDARY_CLASS (gfs_boundary_outflow_class ()),
		       GFS_BOX (neighbor[d]), 
		       FTT_OPPOSITE_DIRECTION (d));
#endif /* DUMMY_MPI */
}
#endif /* HAVE_MPI */

/**
 * gfs_domain_read:
 * @domain: a #GfsDomain.
 * @fp: a #GtsFile.
 *
 * Adds to @domain the graph nodes (#GfsBox) and edges and the
 * corresponding boundaries (#GfsBoundaryMpi if necessary) defined in
 * @fp.
 *
 * Returns: 0 on success or the line number at which an error occured,
 * in which case the corresponding @fp fields (@pos and @error) are
 * set.  
 */
guint gfs_domain_read (GfsDomain * domain,
		      GtsFile * fp)
{
  g_return_val_if_fail (domain != NULL, 1);
  g_return_val_if_fail (fp != NULL, 1);
  g_return_val_if_fail 
    (gts_object_class_is_from_class (GTS_GRAPH (domain)->node_class, 
				     gfs_box_class ()), 1);
  g_return_val_if_fail 
    (gts_object_class_is_from_class (GTS_GRAPH (domain)->edge_class, 
				     gfs_gedge_class ()), 1);
						 
  if (gts_graph_read (GTS_GRAPH (domain), fp))
    return fp->line;

  gts_graph_foreach_edge (GTS_GRAPH (domain),
			  (GtsFunc) gfs_gedge_link_boxes, NULL);
  gts_container_foreach (GTS_CONTAINER (domain),
			 (GtsFunc) set_ref_pos, &domain->refpos);

#ifdef HAVE_MPI
  if (domain->pid >= 0) {
    GSList * removed = NULL;
    gpointer data[2];
    
    data[0] = domain;
    data[1] = &removed;
    gts_container_foreach (GTS_CONTAINER (domain), 
			   (GtsFunc) removed_list, data);
    g_slist_foreach (removed, (GFunc) mpi_links, domain);
    g_slist_free (removed);
  }
#endif /* HAVE_MPI */

  return 0;
}

static void box_split (GfsBox * box, gpointer * data)
{
  GSList ** boxlist = data[0];
  guint * bid = data[1];
  gboolean * one_box_per_pe = data[2];
  gint * pid = data[3];
  guint refid = FTT_DIMENSION == 2 ? 2 : 6;
  FttCellChildren child;
  FttDirection d;
  guint i;
  GfsDomain * domain = gfs_box_domain (box);

  *boxlist = g_slist_prepend (*boxlist, box);

  if (FTT_CELL_IS_LEAF (box->root))
    ftt_cell_refine_single (box->root, (FttCellInitFunc) gfs_cell_init, 
			    domain);

  ftt_cell_children (box->root, &child);
  for (i = 0; i < FTT_CELLS; i++)
    if (child.c[i]) {
      GfsBox * newbox = GFS_BOX (gts_object_new (GTS_OBJECT (box)->klass));

      ftt_cell_destroy (newbox->root, NULL);
      newbox->root = NULL;
      if (*one_box_per_pe)
	newbox->pid = (*pid)++;
      else
	newbox->pid = box->pid;
      if (box->id == 1 && i == refid)
	newbox->id = 1;
      else
	newbox->id = (*bid)++;
      *((guint *) &GFS_STATE (child.c[i])->div) = GPOINTER_TO_UINT (newbox);

      if (FTT_CELL_IS_LEAF (child.c[i]))
	ftt_cell_refine_single (child.c[i], (FttCellInitFunc) gfs_cell_init, 
				domain);
    }

  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (GFS_IS_BOUNDARY (box->neighbor[d])) {
      GfsBoundary * boundary = GFS_BOUNDARY (box->neighbor[d]);
      FttDirection od = FTT_OPPOSITE_DIRECTION (d);

      ftt_cell_children (boundary->root, &child);
      for (i = 0; i < FTT_CELLS; i++)
	if (child.c[i] && FTT_CELL_IS_LEAF (child.c[i]))
	  ftt_cell_refine_single (child.c[i], (FttCellInitFunc) gfs_cell_init,
				  domain);
      ftt_cell_destroy_root (boundary->root, &child, gfs_cell_cleanup);
      boundary->root = NULL;

      ftt_cell_children_direction (box->root, d, &child);
      for (i = 0; i < FTT_CELLS/2; i++)
	if (child.c[i]) {
	  FttCell * neighbor = ftt_cell_neighbor (child.c[i], d);
	  GfsBoundary * newboundary = 
	    GFS_BOUNDARY (gts_object_clone (GTS_OBJECT (boundary)));
	  GfsBox * newbox;

	  g_assert (neighbor);
	  newboundary->root = neighbor;
	  newboundary->d = od;
	  newbox = 
	    GUINT_TO_POINTER (*((guint *) &GFS_STATE (child.c[i])->div));
	  g_assert (newbox);
	  newbox->neighbor[d] = GTS_OBJECT (newboundary);
	  newboundary->box = newbox;
	}
      gts_object_destroy (GTS_OBJECT (boundary));
    }
}

static void box_link (GfsBox * box, GfsDomain * domain)
{
  FttCellChildren child;
  guint i;

  ftt_cell_children (box->root, &child);
  for (i = 0; i < FTT_CELLS; i++)
    if (child.c[i]) {
       GfsBox * newbox = 
	 GUINT_TO_POINTER (*((guint *) &GFS_STATE (child.c[i])->div));
       FttDirection d;
       
       g_assert (newbox);
       gts_container_add (GTS_CONTAINER (domain), GTS_CONTAINEE (newbox));
       for (d = 0; d < FTT_NEIGHBORS; d++)
	 if (newbox->neighbor[d] == NULL) {
	   FttCell * neighbor = ftt_cell_neighbor (child.c[i], d);
	
	   if (neighbor) {
	     GfsBox * newbox1 = 
	       GUINT_TO_POINTER (*((guint *) &GFS_STATE (neighbor)->div));
	     FttDirection od = FTT_OPPOSITE_DIRECTION (d);
	     GfsGEdge * edge;

	     g_assert (newbox1);
	     newbox->neighbor[d] = GTS_OBJECT (newbox1);
	     g_assert (newbox1->neighbor[od] == NULL);
	     newbox1->neighbor[od] = GTS_OBJECT (newbox);
	     edge = GFS_GEDGE (gts_gedge_new (GTS_GRAPH (domain)->edge_class,
					      GTS_GNODE (newbox), 
					      GTS_GNODE (newbox1)));
	     edge->d = d;
	   }
	 }
    }
}

static void box_destroy (GfsBox * box)
{
  GfsBox * newbox[FTT_CELLS];
  FttCellChildren child;
  guint i;

  ftt_cell_children (box->root, &child);
  for (i = 0; i < FTT_CELLS; i++)
    if (child.c[i])
      newbox[i] = GUINT_TO_POINTER (*((guint *) &GFS_STATE (child.c[i])->div));
    else
      newbox[i] = NULL;

  ftt_cell_destroy_root (box->root, &child, gfs_cell_cleanup);
  box->root = NULL;
  for (i = 0; i < FTT_CELLS; i++)
    if (child.c[i])
      newbox[i]->root = child.c[i];

  gts_object_destroy (GTS_OBJECT (box));
}

static void get_ref_pos (GfsBox * box, FttVector * pos)
{
  if (box->id == 1)
    ftt_cell_pos (box->root, pos);
}

/**
 * gfs_domain_split:
 * @domain: a #GfsDomain.
 * @one_box_per_pe: if %TRUE each new box created is assigned to a
 * different process, otherwise the newly created box inherits the pid
 * of its parent.
 *
 * Splits each box of @domain into its (4 in 2D, 8 in 3D)
 * children. The corresponding newly created boxes are added to the
 * graph and the parent boxes are destroyed.
 */
void gfs_domain_split (GfsDomain * domain, gboolean one_box_per_pe)
{
  GSList * list = NULL;
  guint bid = 2;
  gint pid = 0;
  gpointer data[4];

  g_return_if_fail (domain != NULL);

  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_ALL, 1,
  			   (FttCellTraverseFunc) gfs_cell_reset, gfs_div);
  data[0] = &list;
  data[1] = &bid;
  data[2] = &one_box_per_pe;
  data[3] = &pid;
  gts_container_foreach (GTS_CONTAINER (domain), (GtsFunc) box_split, data);
  g_slist_foreach (list, (GFunc) box_link, domain);
  g_slist_foreach (list, (GFunc) box_destroy, NULL);
  g_slist_free (list);

  gfs_domain_match (domain);
  domain->rootlevel++;
  gts_container_foreach (GTS_CONTAINER (domain), (GtsFunc) get_ref_pos, 
			 &domain->refpos);
}

static void box_locate (GfsBox * box, gpointer * data)
{
  FttVector * target = data[0];
  gint * max_depth = data[1];
  FttCell ** cell = data[2];

  if (*cell == NULL)
    *cell = ftt_cell_locate (box->root, *target, *max_depth);
}

/**
 * gfs_domain_locate:
 * @domain: a #GfsDomain.
 * @target: position of the point to look for.
 * @max_depth: maximum depth to consider (-1 means no restriction).
 *
 * Locates the cell of @domain containing @target. This is done
 * efficiently in log(n) operations by using the topology of the cell
 * trees.
 *
 * Returns: a #FttCell of @domain containing (boundary included) the
 * point defined by @target or %NULL if @target is not contained in
 * any cell of @domain.  
 */
FttCell * gfs_domain_locate (GfsDomain * domain,
			    FttVector target,
			    gint max_depth)
{
  FttCell * cell = NULL;
  gpointer data[3];

  g_return_val_if_fail (domain != NULL, NULL);

  data[0] = &target;
  data[1] = &max_depth;
  data[2] = &cell;
  gts_container_foreach (GTS_CONTAINER (domain), (GtsFunc) box_locate, data);

  return cell;
}

/**
 * gfs_domain_advect_point:
 * @domain: a #GfsDomain.
 * @p: a #GtsPoint.
 * @dt: the time step.
 *
 * Updates the coordinates of point @p at time t + @dt using the
 * velocity field defined by @domain.
 *
 * If @p is not contained within @domain, the coordinates are unchanged.
 */
void gfs_domain_advect_point (GfsDomain * domain, 
			     GtsPoint * p,
			     gdouble dt)
{
  FttCell * cell;
  FttVector p0, p1;
  FttComponent c;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (p != NULL);

  p0.x = p1.x = p->x; 
  p0.y = p1.y = p->y;
  p0.z = p1.z = p->z;
  cell = gfs_domain_locate (domain, p0, -1);
  if (cell == NULL)
    return;
  for (c = 0; c < FTT_DIMENSION; c++)
    (&p1.x)[c] += dt*gfs_interpolate (cell, p0, GFS_VELOCITY_INDEX (c))/2.;
  cell = gfs_domain_locate (domain, p1, -1);
  if (cell == NULL)
    return;
  for (c = 0; c < FTT_DIMENSION; c++)
    (&p->x)[c] += dt*gfs_interpolate (cell, p1, GFS_VELOCITY_INDEX (c));
}

static void count (FttCell * cell, guint * n)
{
  (*n)++;
}

/**
 * gfs_domain_size:
 * @domain: a #GfsDomain.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Returns: the number of cells of @domain traversed using @flags and
 * @max_depth.
 */
guint gfs_domain_size (GfsDomain * domain,
		       FttTraverseFlags flags,
		       gint max_depth)
{
  guint n = 0;

  g_return_val_if_fail (domain != NULL, 0);
  
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth, 
			   (FttCellTraverseFunc) count, &n);
#ifdef HAVE_MPI
  if (domain->pid >= 0) {
    guint sn;

    MPI_Allreduce (&n, &sn, 1, MPI_UNSIGNED, MPI_SUM, MPI_COMM_WORLD);
    n = sn;
  }
#endif /* HAVE_MPI */
  return n;
}

static void minimum_cfl (FttCell * cell, gpointer * data)
{
  gdouble * cfl = data[0];
  GfsVariable * v = data[1];
  gdouble size = ftt_cell_size (cell);
  FttComponent c;

  for (c = 0; c < FTT_DIMENSION; c++, v = v->next) {
    if (GFS_VARIABLE (cell, v->i) != 0.) {
      gdouble cflu = size/fabs (GFS_VARIABLE (cell, v->i));

      if (cflu*cflu < *cfl)
	*cfl = cflu*cflu;
    }
    if (v->sources) {
      gdouble g = gfs_variable_source (v, cell);

      if (g != 0.) {
	gdouble cflg = 2.*size/fabs (g);

	if (cflg < *cfl)
	  *cfl = cflg;
      }
    }
  }
}

/**
 * gfs_domain_cfl:
 * @domain: a #GfsDomain.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Returns: the minimum over the cells of @domain (traversed using
 * @flags and @max_depth) of the time scale defined by the size of the
 * cell and the norm of either the local velocity or the local
 * acceleration.
 */
gdouble gfs_domain_cfl (GfsDomain * domain,
			FttTraverseFlags flags,
			gint max_depth)
{
  gdouble cfl = 1.;
  gpointer data[2];

  g_return_val_if_fail (domain != NULL, 0.);

  data[0] = &cfl;
  data[1] = gfs_variable_from_name (domain->variables, "U");
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth, 
			    (FttCellTraverseFunc) minimum_cfl, data);
#ifdef HAVE_MPI
  if (domain->pid >= 0) {
    gdouble gcfl;

    MPI_Allreduce (&cfl, &gcfl, 1, MPI_DOUBLE, MPI_MIN, MPI_COMM_WORLD);
    cfl = gcfl;
  }
#endif /* HAVE_MPI */
  return sqrt (cfl);
}

/**
 * gfs_cell_init:
 * @cell: a #FttCell.
 * @domain: a #GfsDomain containing @cell.
 *
 * Allocates the memory for fluid state data associated to @cell.
 */
void gfs_cell_init (FttCell * cell, GfsDomain * domain)
{
  g_return_if_fail (cell != NULL);
  g_return_if_fail (cell->data == NULL);
  g_return_if_fail (domain != NULL);

  cell->data = g_malloc0 (domain->variables_size);
}

/**
 * gfs_cell_copy:
 * @from: a #FttCell to copy attributes from.
 * @to: a #FttCell to copy attributes to.
 * @domain: the #GfsDomain containing @from.
 *
 * Copies the attributes of the fluid cell @from to the fluid cell @to.
 */
void gfs_cell_copy (const FttCell * from, 
		    FttCell * to,
		    GfsDomain * domain)
{
  GfsSolidVector * solid;
  GfsStateVector * froms, * tos;

  g_return_if_fail (from != NULL);
  g_return_if_fail (to != NULL);
  g_return_if_fail (from != to);  
  g_return_if_fail (domain != NULL);

  froms = GFS_STATE (from);
  tos = GFS_STATE (to);
  if (froms != NULL) {
    if (tos == NULL) {
      gfs_cell_init (to, domain);
      tos = GFS_STATE (to);
    }
    solid = tos->solid;
    memcpy (to->data, from->data, domain->variables_size);
    if (froms->solid == NULL) {
      if (solid)
	g_free (solid);
    }
    else {
      tos->solid = solid;
      *solid = *(froms->solid);
    }
  }
  else if (tos != NULL)
    gfs_cell_cleanup (to);
}

/**
 * gfs_cell_read:
 * @cell: a #FttCell.
 * @fp: a #GtsFile.
 * @domain: the #GfsDomain containing @cell.
 *
 * Reads from @fp the fluid data associated with @cell and described
 * by @domain->variables_io. This function is generally used in
 * association with ftt_cell_read().  
 */
void gfs_cell_read (FttCell * cell, GtsFile * fp, GfsDomain * domain)
{
  gdouble s0;
  GfsStateVector * s;
  GfsVariable * v;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (fp != NULL);
  g_return_if_fail (domain != NULL);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (solid->s[0])");
    return;
  }
  s0 = atof (fp->token->str);
  gts_file_next_token (fp);

  gfs_cell_init (cell, domain);
  s = cell->data;
  if (s0 >= 0.) {
    guint i;

    s->solid = g_malloc0 (sizeof (GfsSolidVector));
    s->solid->s[0] = s0;

    for (i = 1; i < FTT_NEIGHBORS; i++) {
      if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
	gts_file_error (fp, "expecting a number (solid->s[%d])", i);
	return;
      }
      s->solid->s[i] = atof (fp->token->str);
      gts_file_next_token (fp);
    }
    if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
      gts_file_error (fp, "expecting a number (solid->a)");
      return;
    }
    s->solid->a = atof (fp->token->str);
    gts_file_next_token (fp);
  }

  v = domain->variables_io;
  while (v) {
    if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
      gts_file_error (fp, "expecting a number (%s)", v->name);
      return;
    }
    GFS_VARIABLE (cell, v->i) = atof (fp->token->str);
    gts_file_next_token (fp);
    v = v->next;
  }
}

/**
 * gfs_domain_add_variable:
 * @domain: a #GfsDomain.
 * @name: the name of the variable to add.
 *
 * Adds a new variable @name to @domain.
 *
 * Returns: the new variable or %NULL if a variable with the same name
 * already exists.  
 */
GfsVariable * gfs_domain_add_variable (GfsDomain * domain, const gchar * name)
{
  GfsVariable * v, * last;

  g_return_val_if_fail (domain != NULL, NULL);
  g_return_val_if_fail (name != NULL, NULL);

  if (gfs_variable_from_name (domain->variables, name))
    return NULL;

  g_return_val_if_fail (gts_container_size (GTS_CONTAINER (domain)) == 0, 
			NULL);

  v = last = domain->variables;
  while (v) {
    last = v;
    v = v->next;
  }
  g_assert (last);

  last->next = v = 
    gfs_variable_new (GFS_VARIABLE_CLASS (GTS_OBJECT (last)->klass),
		      GTS_OBJECT (domain), name, last->i + 1);
  domain->variables_size += sizeof (gdouble);

  return v;
}
