/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <stdlib.h>
#include "timestep.h"
#include "source.h"

/**
 * gfs_projection_params_write:
 * @par: the projection parameters.
 * @fp: a file pointer.
 *
 * Writes in @fp a text representation of the projection parameters
 * @par.  
 */
void gfs_projection_params_write (GfsProjectionParams * par, FILE * fp)
{
  g_return_if_fail (par != NULL);
  g_return_if_fail (fp != NULL);

  fprintf (fp,
           "{\n"
	   "  tolerance = %g\n"
	   "  nrelax    = %u\n"
	   "  minlevel  = %u\n"
	   "  nitermax  = %u\n"
	   "}",
	   par->tolerance,
	   par->nrelax,
	   par->minlevel,
	   par->nitermax);
}

void gfs_projection_params_init (GfsProjectionParams * par)
{
  g_return_if_fail (par != NULL);

  par->tolerance = 1e-3;
  par->nrelax    = 4;
  par->minlevel  = 0;
  par->nitermax  = 100;
}

void gfs_projection_params_read (GfsProjectionParams * par, GtsFile * fp)
{
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "tolerance", TRUE},
    {GTS_UINT,   "nrelax",    TRUE},
    {GTS_UINT,   "minlevel",  TRUE},
    {GTS_UINT,   "nitermax",  TRUE},
    {GTS_NONE}
  };

  g_return_if_fail (par != NULL);
  g_return_if_fail (fp != NULL);

  var[0].data = &par->tolerance;
  var[1].data = &par->nrelax;
  var[2].data = &par->minlevel;
  var[3].data = &par->nitermax;

  gfs_projection_params_init (par);
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  if (par->tolerance <= 0.) {
    gts_file_variable_error (fp, var, "tolerance",
			     "tolerance `%g' must be strictly positive",
			     par->tolerance);
    return;
  }
  if (par->nrelax == 0)
    gts_file_variable_error (fp, var, "nrelax", "nrelax must be non zero");
}

/**
 * gfs_correct_normal_velocities:
 * @face: a #FttCellFace.
 * @par: the advection parameters.
 *
 * Corrects the velocity normal to @face with the gradient of the
 * pressure in the corresponding direction.
 *
 * Also adds half this gradient to the corresponding %GFS_G variable of
 * the cells on either sides of the @face. If this function is called
 * for each face of a domain, the %GFS_G variables in each cell will
 * then contain the (interpolated) value of the pressure gradient.  
 *
 * This functions assumes that the face coefficients for the Poisson
 * problem have been initialized using gfs_poisson_coefficients().  
 */
void gfs_correct_normal_velocities (const FttCellFace * face,
				    const GfsAdvectionParams * par)
{
  GfsGradient g;
  gdouble dp;
  FttFaceType type;
  FttComponent c;
  GfsStateVector * s;

  g_return_if_fail (face != NULL);
  g_return_if_fail (par != NULL);

  s = GFS_STATE (face->cell);
  type = ftt_face_type (face);
  c = face->d/2;
  if (type == FTT_BOUNDARY) {
    GfsVariable * v = par->v;

    /* find corresponding momentum variable */
    while (c) { v = v->next; c--; }
    if (v->sources)
      /* pressure gradient compensates local momentum source term */
      s->g[face->d/2] += par->dt*gfs_variable_source (v, face->cell)/2.;
    return;
  }

  gfs_face_weighted_gradient (face, &g, GFS_P, -1);
  dp = (g.b - g.a*s->p)/ftt_cell_size (face->cell);
  if (!FTT_FACE_DIRECT (face))
    dp = - dp;

  if (s->solid && s->solid->s[face->d] > 0.)
    dp /= s->solid->s[face->d];

  GFS_FACE_NORMAL_VELOCITY_LEFT (face) -= dp;
  s->g[c] += dp/2.;

  switch (type) {
  case FTT_FINE_FINE:
    GFS_FACE_NORMAL_VELOCITY_RIGHT (face) -= dp;
    GFS_STATE (face->neighbor)->g[c] += dp/2.;
    break;
  case FTT_FINE_COARSE: {
    /* check for mixed cell refinement violation (topology.fig) */
    g_assert (!GFS_IS_MIXED (face->cell) || 
	      GFS_STATE (face->cell)->solid->s[face->d] == 1.);
    GFS_FACE_NORMAL_VELOCITY_RIGHT (face) -= dp/(FTT_CELLS/2);
    GFS_STATE (face->neighbor)->g[c] += dp/FTT_CELLS;
    break;
  }
  default:
    g_assert_not_reached ();
  }  
}

/**
 * gfs_correct_centered_velocity:
 * @face: a #FttCellFace.
 * @par: the advection parameters.
 *
 * (Half) Corrects the centered velocity in each cell on either side
 * of @face with the pressure gradient. For the correction to be
 * complete, this function has to be called for each face of the
 * domain.
 */
void gfs_correct_centered_velocity (const FttCellFace * face,
				    const GfsAdvectionParams * par)
{
  GfsGradient g;
  gdouble dp;
  guint v;
  FttFaceType type;

  g_return_if_fail (face != NULL);
  g_return_if_fail (par != NULL);

  type = ftt_face_type (face);
  v = GFS_VELOCITY_INDEX (face->d/2);
  if (type == FTT_BOUNDARY) {
    GfsVariable * var = par->v;
    FttComponent c = face->d/2;

    /* find corresponding momentum variable */
    while (c) { var = var->next; c--; }
    if (var->sources)
      /* pressure gradient compensates local momentum source term */
      GFS_VARIABLE (face->cell, v) -= 
	par->dt*gfs_variable_source (var, face->cell)/2.;
    return;
  }

  gfs_face_weighted_gradient (face, &g, GFS_P, -1);
  dp = (g.b - g.a*GFS_VARIABLE (face->cell, GFS_P))/ftt_cell_size (face->cell);
  if (!FTT_FACE_DIRECT (face))
    dp = - dp;

  if (GFS_STATE (face->cell)->solid && 
      GFS_STATE (face->cell)->solid->s[face->d] > 0.)
    dp /= GFS_STATE (face->cell)->solid->s[face->d];

  GFS_VARIABLE (face->cell, v) -= dp/2.;

  switch (type) {
  case FTT_FINE_FINE:
    GFS_VARIABLE (face->neighbor, v) -= dp/2.;
    break;
  case FTT_FINE_COARSE:
    /* check for mixed cell refinement violation (topology.fig) */ 
    g_assert (!GFS_IS_MIXED (face->cell) || 
	      GFS_STATE (face->cell)->solid->s[face->d] == 1.);
    GFS_VARIABLE (face->neighbor, v) -= dp/FTT_CELLS;
    break;
  default:
    g_assert_not_reached ();
  }
}

static void reset_gradient (FttCell * cell)
{
  FttComponent c;

  for (c = 0; c < FTT_DIMENSION; c++)
    GFS_STATE (cell)->g[c] = 0.;
}

/**
 * gfs_mac_projection:
 * @domain: a #GfsDomain.
 * @par: the projection control parameters.
 * @apar: the advection parameters.
 *
 * Corrects the face-centered velocity field (MAC field) on the leaf
 * level of @domain using an exact (MAC) projection. The resulting
 * face-centered velocity field is (almost) exactly divergence
 * free. The (potential) pressure field is also obtained as a
 * by-product as well as its gradient at the center of the leaf cells
 * of the domain (the gradient is stored in the %GFS_G variables and is
 * obtained by simple averaging from the face values to the center).
 *
 * The @residual field of the @par projection parameters is set to the
 * norm of the residual after the projection. The @niter field of the
 * @par projection parameters is set to the number of iterations
 * performed to solve the Poisson equation. The other projection
 * parameters are not modified.
 */
void gfs_mac_projection (GfsDomain * domain,
			 GfsProjectionParams * par,
			 GfsAdvectionParams * apar)
{
  guint minlevel, maxlevel;
  FttComponent c;
  GfsVariable * g;
  gdouble dt;
  gdouble start, end;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (par != NULL);
  g_return_if_fail (apar != NULL);

  start = g_timer_elapsed (domain->timer, NULL);
  
  apar->v = gfs_variable_from_name (domain->variables, "U");
  dt = apar->dt;
  apar->dt /= 2.;

  /* Initialize face coefficients */
  gfs_poisson_coefficients (domain, apar->c, apar->rho);

  /* compute MAC divergence */
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) gfs_normal_divergence, 
			    NULL);

#if 0
  {
    FILE * fp = fopen ("/tmp/mac", "wt");
    GfsNorm norm;

    gfs_write_mac_velocity (domain, 0.9, FTT_TRAVERSE_LEAFS, -1, NULL, fp);
    fclose (fp);
    norm = gfs_domain_norm_variable (domain, gfs_div, FTT_TRAVERSE_LEAFS, -1);
    fprintf (stderr, "mac div before: %g %g %g\n",
	     norm.first, norm.second, norm.infty);
  }
#endif

  /* solve for pressure */
  minlevel = domain->rootlevel;
  if (par->minlevel > minlevel)
    minlevel = par->minlevel;
  maxlevel = gfs_domain_depth (domain);
  gfs_residual (domain, FTT_TRAVERSE_LEAFS, -1, gfs_p, gfs_div, gfs_res);
  par->residual_before = par->residual = 
    gfs_domain_norm_residual (domain, FTT_TRAVERSE_LEAFS, -1);
  par->niter = 0;
  while (par->residual.infty > par->tolerance && 
	 par->niter < par->nitermax) {
    gfs_poisson (domain, minlevel, maxlevel, par->nrelax, gfs_p, gfs_div);
    par->residual = gfs_domain_norm_residual (domain, FTT_TRAVERSE_LEAFS, -1);
    par->niter++;
  }
  
  /* correct face velocities */
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) reset_gradient, NULL);
  gfs_domain_face_traverse (domain, FTT_XYZ, 
			    FTT_PRE_ORDER, 
			    FTT_TRAVERSE_LEAFS | FTT_TRAVERSE_BOUNDARY_FACES, 
			    -1,
      (FttFaceTraverseFunc) gfs_correct_normal_velocities, 
			    apar);
  for (c = 0, g = gfs_gx; c < FTT_DIMENSION; c++, g = g->next)
    gfs_domain_center_bc (domain, g);

#if 0
  {
    GfsNorm norm;

    gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) gfs_normal_divergence, 
			      NULL);
    norm = gfs_domain_norm_variable (domain, gfs_div, FTT_TRAVERSE_LEAFS, -1);
    fprintf (stderr, "mac div after: %g %g %g\n",
	     norm.first, norm.second, norm.infty);
  }
#endif

  apar->dt = dt;

  end = g_timer_elapsed (domain->timer, NULL);
  gts_range_add_value (&domain->mac_projection, end - start);
  gts_range_update (&domain->mac_projection);
}

/**
 * gfs_approximate_projection:
 * @domain: a #GfsDomain.
 * @par: the projection control parameters.
 * @apar: the advection parameters.
 *
 * Corrects the centered velocity field on the leaf level of @domain
 * using an approximate projection. The resulting centered velocity
 * field is approximately divergence free. The (potential) pressure
 * field is also obtained as a by-product.
 *
 * The @residual field of the @par projection parameters is set to the
 * norm of the residual (on the MAC grid) after the projection. The
 * @niter field of the @par projection parameters is set to the number
 * of iterations performed to solve the Poisson equation. The other
 * projection parameters are not modified.
 *
 * The Poisson equation for the pressure is first solved on a MAC grid
 * where the MAC velocities are obtained from the centered velocities
 * by simple averaging. The resulting pressure gradients (defined on
 * the faces) are then averaged down on the center of the cells to
 * correct the centered velocity.  
 */
void gfs_approximate_projection (GfsDomain * domain,
				 GfsProjectionParams * par,
				 GfsAdvectionParams * apar)
{
  guint minlevel, maxlevel;
  FttComponent c;
  gdouble start, end;
  GfsVariable * v;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (par != NULL);
  g_return_if_fail (apar != NULL);

  start = g_timer_elapsed (domain->timer, NULL);

  /* Initialize face coefficients */
  gfs_poisson_coefficients (domain, apar->c, apar->rho);

  /* compute MAC velocities from centered velocities */
  gfs_domain_face_traverse (domain, FTT_XYZ,
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
      (FttFaceTraverseFunc) gfs_face_reset_normal_velocity,
			    NULL);
  gfs_domain_face_traverse (domain, FTT_XYZ,
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
      (FttFaceTraverseFunc) gfs_face_interpolated_normal_velocity, NULL);

#if 0
  {
    FILE * fp = fopen ("/tmp/macapprox", "wt");
    GfsNorm norm;

    gfs_write_mac_velocity (domain, 0.9, FTT_TRAVERSE_LEAFS, -1, NULL, fp);
    fclose (fp);
    norm = gfs_domain_norm_variable (domain, gfs_div, FTT_TRAVERSE_LEAFS, -1);
    fprintf (stderr, "mac div before: %g %g %g\n",
	     norm.first, norm.second, norm.infty);
  }
#endif

  /* compute MAC divergence */
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) gfs_normal_divergence, 
			    NULL);
  
  /* solve for pressure */
  minlevel = domain->rootlevel;
  if (par->minlevel > minlevel)
    minlevel = par->minlevel;
  maxlevel = gfs_domain_depth (domain);
  gfs_residual (domain, FTT_TRAVERSE_LEAFS, -1, gfs_p, gfs_div, gfs_res);
  par->residual_before = par->residual = 
    gfs_domain_norm_residual (domain, FTT_TRAVERSE_LEAFS, -1);
  par->niter = 0;
  while (par->residual.infty > par->tolerance && 
	 par->niter < par->nitermax) {
#if 0
    fprintf (stderr, "%d bias: %g first: %g second: %g infty: %g\n",
	     par->niter, 
	     par->residual.bias, 
	     par->residual.first, 
	     par->residual.second, 
	     par->residual.infty);
#endif
    gfs_poisson (domain, minlevel, maxlevel, par->nrelax, gfs_p, gfs_div);
    par->residual = gfs_domain_norm_residual (domain, FTT_TRAVERSE_LEAFS, -1);
    par->niter++;
  }

  /* correct centered velocities */
  apar->v = gfs_variable_from_name (domain->variables, "U");
  gfs_domain_face_traverse (domain, FTT_XYZ,
			    FTT_PRE_ORDER,
			    FTT_TRAVERSE_LEAFS | FTT_TRAVERSE_BOUNDARY_FACES,
			    -1,
      (FttFaceTraverseFunc) gfs_correct_centered_velocity,
			    apar);
  v = gfs_variable_from_name (domain->variables, "U");
  for (c = 0; c < FTT_DIMENSION; c++, v = v->next)
    gfs_domain_center_bc (domain, v);

  end = g_timer_elapsed (domain->timer, NULL);
  gts_range_add_value (&domain->approximate_projection, end - start);
  gts_range_update (&domain->approximate_projection);
}

/**
 * gfs_centered_velocity_advection:
 * @domain: a #GfsDomain.
 * @par: the advection parameters.
 *
 * Advects the (centered) velocity field using the current
 * face-centered (MAC) velocity field and @par->flux to compute the
 * velocity flux through the faces of each cell.
 *
 * For each component of the velocity, before calling the @par->flux
 * function the face values are first defined (at time t + dt/2) and
 * can then be used within the @par->flux function.
 *
 * "Small" cut cells are treated using a cell-merging approach to
 * avoid any restrictive CFL stability condition.  
 */
void gfs_centered_velocity_advection (GfsDomain * domain,
				      GfsAdvectionParams * par)
{
  FttComponent c;
  static GfsVariable * advection_term[FTT_DIMENSION];
  gdouble start, end;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (par != NULL);

  start = g_timer_elapsed (domain->timer, NULL);

  advection_term[0] = gfs_div;
  advection_term[1] = gfs_res;
#ifndef FTT_2D
  advection_term[2] = gfs_dp;
#endif /* FTT_3D */
  par->use_centered_velocity = FALSE;
  par->v = gfs_variable_from_name (domain->variables, "U");
  for (c = 0; c < FTT_DIMENSION; c++, par->v = par->v->next) {
    par->fv = advection_term[c];
    gfs_domain_cell_traverse (domain, 
			     FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			     (FttCellTraverseFunc) gfs_cell_reset, par->fv);
    gfs_domain_cell_traverse (domain, 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
        (FttCellTraverseFunc) gfs_cell_advected_face_values, par);
    gfs_domain_face_bc (domain, FTT_XYZ, par->v);
    gfs_domain_face_traverse (domain, FTT_XYZ,
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttFaceTraverseFunc) par->flux, par);
  }
  par->v = gfs_variable_from_name (domain->variables, "U");
  for (c = 0; c < FTT_DIMENSION; c++, par->v = par->v->next) {
    par->fv = advection_term[c];
    gfs_domain_traverse_merged (domain,
				(GfsMergedTraverseFunc) gfs_advection_update, 
				par);
  }
  par->v = gfs_variable_from_name (domain->variables, "U");
  for (c = 0; c < FTT_DIMENSION; c++, par->v = par->v->next)
    gfs_domain_center_bc (domain, par->v);

  end = g_timer_elapsed (domain->timer, NULL);
  gts_range_add_value (&domain->centered_velocity_advection, end - start);
  gts_range_update (&domain->centered_velocity_advection);
}

static void save_previous (FttCell * cell, gpointer * data)
{
  GfsVariable * v = data[0];
  GfsVariable * vh = data[1];

  GFS_VARIABLE (cell, vh->i) = GFS_VARIABLE (cell, v->i);
}

static void average_previous (FttCell * cell, gpointer * data)
{
  GfsVariable * v = data[0];
  GfsVariable * vh = data[1];

  GFS_VARIABLE (cell, vh->i) = (GFS_VARIABLE (cell, vh->i) +
				GFS_VARIABLE (cell, v->i))/2.;
}

/**
 * gfs_tracer_advection:
 * @domain: a #GfsDomain.
 * @par: the advection parameters.
 * @half: a #GfsVariable or %NULL.
 *
 * Advects the @v field of @par using the current face-centered (MAC)
 * velocity field.
 *
 * If @half is not %NULL, the half-timestep value of @par->v is
 * stored in the corresponding variable.  
 */
void gfs_tracer_advection (GfsDomain * domain,
			   GfsAdvectionParams * par,
			   GfsVariable * half)
{
  gdouble start, end;
  gpointer data[2];

  g_return_if_fail (domain != NULL);
  g_return_if_fail (par != NULL);

  start = g_timer_elapsed (domain->timer, NULL);

  if (half) {
    data[0] = par->v;
    data[1] = half;
    gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) save_previous, data);
  }
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) gfs_cell_reset, par->fv);
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
      (FttCellTraverseFunc) gfs_cell_advected_face_values,
			    par);
  gfs_domain_face_bc (domain, FTT_XYZ, par->v);
  gfs_domain_face_traverse (domain, FTT_XYZ,
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttFaceTraverseFunc) par->flux, par);
  gfs_domain_traverse_merged (domain, 
			      (GfsMergedTraverseFunc) gfs_advection_update, 
			      par);
  gfs_domain_center_bc (domain, par->v);
  if (half) {
    gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) average_previous, data);
    gfs_domain_center_bc (domain, half);
  }

  end = g_timer_elapsed (domain->timer, NULL);
  gts_range_add_value (&domain->tracer_advection, end - start);
  gts_range_update (&domain->tracer_advection);
}

/**
 * gfs_predicted_face_velocities:
 * @domain: a #GfsDomain.
 * @par: the advection parameters.
 *
 * Fills the face (MAC) normal velocities of each leaf cell of @domain
 * with the predicted values at time t + dt/2 using a godunov type
 * advection scheme.  
 */
void gfs_predicted_face_velocities (GfsDomain * domain,
				    GfsAdvectionParams * par)
{
  FttComponent c;
  gdouble start, end;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (par != NULL);

  start = g_timer_elapsed (domain->timer, NULL);

  gfs_domain_face_traverse (domain, FTT_XYZ,
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
      (FttFaceTraverseFunc) gfs_face_reset_normal_velocity, 
			    NULL);
  par->use_centered_velocity = TRUE;
  par->v = gfs_variable_from_name (domain->variables, "U");
  for (c = 0; c < FTT_DIMENSION; c++, par->v = par->v->next) {
    gfs_domain_cell_traverse (domain, 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
        (FttCellTraverseFunc) gfs_cell_advected_face_values, par);
    gfs_domain_face_bc (domain, c, par->v);
    gfs_domain_face_traverse (domain, c,
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
        (FttFaceTraverseFunc) gfs_face_advected_normal_velocity, 
			      NULL);
  }

  end = g_timer_elapsed (domain->timer, NULL);
  gts_range_add_value (&domain->predicted_face_velocities, end - start);
  gts_range_update (&domain->predicted_face_velocities);
}
