#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "config.h"
#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif /* HAVE_GETOPT_H */
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "fluid.h"
#include "output.h"
#include "solid.h"
#include "graphic.h"
#include "init.h"

static void points_number (FttCell * cell, guint * npoints)
{
  GFS_STATE (cell)->div = *npoints;
  (*npoints)++;
}

static void write_points (FttCell * cell, FILE * fp)
{
  FttVector p;

  ftt_cell_pos (cell, &p);
  fprintf (fp, "%g %g %g\n", p.x, p.y, p.z);
}

static void gfs_draw_vtk (FttCell * root, FILE * fp)
{
  guint npoints = 0;

  g_return_if_fail (root != NULL);
  g_return_if_fail (fp != NULL);

  ftt_cell_traverse (root, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
		     (FttCellTraverseFunc) points_number, &npoints);
  fprintf (fp, 
	   "# vtk DataFile Version 2.0\n"
	   "Generated by gfs2other\n"
	   "ASCII\n"
	   "DATASET UNSTRUCTURED_GRID\n"
	   "POINTS %d float\n",
	   npoints);
  ftt_cell_traverse (root, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
		     (FttCellTraverseFunc) write_points, fp);
}

static void draw_vector (FttCell * cell, gpointer * data) 
{
  gdouble * scale = data[0];
  GtsBBox * bbox = data[1];
  FILE * fp = stdout;
  FttVector pos;
  
  ftt_cell_pos (cell, &pos);
  if (bbox == NULL || (pos.x >= bbox->x1 && pos.x <= bbox->x2 &&
		       pos.y >= bbox->y1 && pos.y <= bbox->y2 &&
		       pos.z >= bbox->z1 && pos.z <= bbox->z2)) {
    FttVector f;
  
    f.x = GFS_STATE (cell)->u*(*scale);
    f.y = GFS_STATE (cell)->v*(*scale);
#ifdef FTT_2D
    f.z = 0.;
#else
    f.z = GFS_STATE (cell)->w*(*scale);
#endif
  
    fprintf (fp, "VECT 1 3 0 3 0 %g %g %g %g %g %g %g %g %g\n",
	     pos.x + f.x - (f.x - f.y/2.)/5.,
	     pos.y + f.y - (f.x/2. + f.y)/5.,
	     pos.z + f.z,
	     pos.x + f.x,
	     pos.y + f.y,
	     pos.z + f.z,
	     pos.x + f.x - (f.x + f.y/2.)/5.,
	     pos.y + f.y + (f.x/2. - f.y)/5.,
	     pos.z + f.z);
    fprintf (fp, "VECT 1 2 0 2 0 %g %g %g %g %g %g\n",
	     pos.x, pos.y, pos.z,
	     pos.x + f.x,
	     pos.y + f.y,
	     pos.z + f.z);
  }
}

int main (int argc, char * argv[])
{
  GfsBox * box;
  GfsDomain * domain;
  int c = 0;
  GfsVariable * var = NULL;
  GtsFile * fp;
  
  gboolean verbose = FALSE;
  gboolean mesh = FALSE;
  gboolean squares = FALSE;
  FttTraverseFlags flags = FTT_TRAVERSE_LEAFS;
  gint level = -1;
  gdouble vector = 0., size;

  gdouble min = G_MAXDOUBLE;
  gdouble max = G_MAXDOUBLE;

  GtsBBox bbox;

  gboolean vtk = FALSE;

  bbox.x1 = bbox.y1 = -0.5;
  bbox.x2 = bbox.y2 = 0.5;
  bbox.z1 = bbox.z2 = 0.;

  gfs_init (&argc, &argv);

  /* parse options using getopt */
  while (c != EOF) {
#ifdef HAVE_GETOPT_LONG
    static struct option long_options[] = {
      {"vtk", no_argument, NULL, 'k'},
      {"cx", required_argument, NULL, 'x'},
      {"cy", required_argument, NULL, 'y'},
      {"cz", required_argument, NULL, 'z'},
      {"min", required_argument, NULL, 'n'},
      {"max", required_argument, NULL, 'M'},
      {"level", required_argument, NULL, 'l'},
      {"vector", required_argument, NULL, 's'},
      {"gts", required_argument, NULL, 'g'},
      {"squares", required_argument, NULL, 'S'},
      {"mesh", no_argument, NULL, 'm'},
      {"help", no_argument, NULL, 'h'},
      {"verbose", no_argument, NULL, 'v'},
    };
    int option_index = 0;
    switch ((c = getopt_long (argc, argv, "hvg:ms:S:l:n:M:x:y:z:k",
			      long_options, &option_index))) {
#else /* not HAVE_GETOPT_LONG */
    switch ((c = getopt (argc, argv, "hvg:ms:S:l:n:M:x:y:z:k"))) {
#endif /* not HAVE_GETOPT_LONG */
    case 'k': /* vtk */
      vtk = TRUE;
      break;
    case 'x': /* cx */
      bbox.x1 = bbox.x2 = atof (optarg);
      bbox.y1 = bbox.z1 = -0.5;
      bbox.y2 = bbox.z2 = 0.5;
      break;
    case 'y': /* cy */
      bbox.y1 = bbox.y2 = atof (optarg);
      bbox.x1 = bbox.z1 = -0.5;
      bbox.x2 = bbox.z2 = 0.5;
      break;
    case 'z': /* cz */
      bbox.z1 = bbox.z2 = atof (optarg);
      bbox.x1 = bbox.y1 = -0.5;
      bbox.x2 = bbox.y2 = 0.5;
      break;
    case 'n': /* min */
      min = atof (optarg);
      break;
    case 'M': /* max */
      max = atof (optarg);
      break;
    case 'l': /* level */
      level = atoi (optarg);
      flags = FTT_TRAVERSE_LEVEL;
      break;
    case 'm': /* mesh */
      mesh = TRUE;
      break;
    case 'S': /* squares */
      squares = TRUE;
      /* fall through */
    case 'g': /* gts */
      if ((var = gfs_variable_from_name (gfs_derived_first, optarg)) == NULL) {
	fprintf (stderr, 
		 "gfs2other: unknown variable `%s'\n"
		 "Try `gfs2other --help' for more information.\n",
		 optarg);
	return 1; /* failure */
      }
      break;
    case 's': /* vector */
      vector = atof (optarg);
      break;
    case 'v': /* verbose */
      verbose = TRUE;
      break;
    case 'h': /* help */
      fprintf (stderr,
     "Usage: gfs2other [OPTION] < GFS_FILE\n"
     "Converts a GFS file to other (graphical) formats. The default is to output\n"
     "an OOGL (Geomview) representation of the boundaries of the refined domains.\n"
     "\n"
     "  -l L    --level=L     traverse only cells at level L\n"
     "  -g VAR  --gts=VAR     output a GTS file representation of variable VAR\n"
     "  -S VAR  --squares=VAR output an OOGL representation of variable VAR\n"
     "  -s S    --vector=S    output an OOGL representation of the velocity vector field\n"
     "                        S is the scale of the vectors\n"
     "  -n VAL  --min=VAL     sets the minimum value of the scalar field to VAL\n"
     "  -M VAL  --max=VAL     sets the maximum value of the scalar field to VAL\n"
     "  -m      --mesh        output an OOGL representation of the mesh\n"
     "  -x V    --cx=V        cut mesh along x=V\n"
     "  -y V    --cy=V        cut mesh along y=V\n"
     "  -z V    --cz=V        cut mesh along z=V (default is --cz=0)\n"
     "  -k      --vtk         outputs VTK representation of data\n"
     "  -v      --verbose     display statistics and other info\n"
     "  -h      --help        display this help and exit\n"
     "\n"
     "Reports bugs to %s\n",
	       FTT_MAINTAINER);
      return 0; /* success */
      break;
    case '?': /* wrong options */
      fprintf (stderr, "Try `gfs2other --help' for more information.\n");
      return 1; /* failure */
    }
  }

  fp = gts_file_new (stdin);
  domain = GFS_DOMAIN (gts_graph_new (GTS_GRAPH_CLASS (gfs_domain_class ()), 
				     GTS_GNODE_CLASS (gfs_box_class ()),
				     gts_gedge_class ()));
  box = gfs_box_new (gfs_box_class ());
  box->root = ftt_cell_new ((FttCellInitFunc) gfs_cell_init, domain);
  gts_container_add (GTS_CONTAINER (domain), GTS_CONTAINEE (box));
  box->root = ftt_cell_read (fp, (FttCellReadFunc) gfs_cell_read, domain);
  if (fp->type == GTS_ERROR) {
    fprintf (stderr, 
	     "gfs2other: file on standard input is not a valid GFS file\n"
	     "stdin:%d:%d: %s\n",
	     fp->line, fp->pos, fp->error);
    return 1;
  }
  gts_file_destroy (fp);

  size = ftt_level_size (ftt_cell_depth (box->root));
  bbox.x2 += size;
  bbox.y2 += size;
  bbox.z2 += size;

  if (var != NULL) {
    GtsRange stats;

    if (var->derived) {
      gfs_variable_set_parent (var, GTS_OBJECT (domain));
      ftt_cell_traverse (box->root, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			 (FttCellTraverseFunc) var->derived, var);
    }
    ftt_cell_traverse (box->root, FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
		       (FttCellTraverseFunc) gfs_get_from_below_intensive,
		       var);
    stats = gfs_stats_variable (box->root, var, flags, level);
    if (verbose)
      fprintf (stderr, 
	       "min: %g avg: %g| %g max: %g n: %7d\n",
	       stats.min, stats.mean, stats.stddev, stats.max, stats.n);
    if (min == G_MAXDOUBLE)
      min = stats.min;
    if (max == G_MAXDOUBLE)
      max = stats.max;

    if (squares)
      gfs_write_squares (domain, var, min, max,
			 flags, level, &bbox, stdout);
    else
      gfs_write_gts (domain, var, flags, level, &bbox, stdout);
  }
  else if (mesh) {
    printf ("LIST {\n");
    ftt_cell_traverse (box->root, FTT_PRE_ORDER, flags, level,
		       (FttCellTraverseFunc) ftt_cell_draw, stdout);
    printf ("}\n");
  }
  else if (vector > 0.) {
    GfsNorm norm;
    gdouble scale;
    FttComponent c;
    gpointer data[2];
    GfsVariable * var;

    data[0] = &scale;
    data[1] = &bbox;
    var = gfs_variable_from_name (domain->variables, "U");
    for (c = 0; c < FTT_DIMENSION; c++, var = var->next)
      ftt_cell_traverse (box->root, FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			 (FttCellTraverseFunc) gfs_get_from_below_intensive,
			 var);
    norm = gfs_domain_norm_velocity (domain, flags, level);
    scale = norm.infty > 0. ? 
      ftt_level_size (level < 0 ? ftt_cell_depth (box->root) : level)*
      vector/norm.infty : 1.;
    puts ("LIST{");
    ftt_cell_traverse (box->root, FTT_PRE_ORDER, flags, level,
		       (FttCellTraverseFunc) draw_vector, data);
    putchar ('}');
  }
  else if (vtk)
    gfs_draw_vtk (box->root, stdout);
#if 0
  else
    gfs_draw_boundaries (box->root, stdout);
#endif

  return 0;
}
