#include "printer.h"
#include "termorder.h"
#include "polyhedralcone.h"
#include "polyhedralfan.h"
#include "polynomialring.h"
#include "log.h"

//--------------------------------------------------
// Printer
//--------------------------------------------------
void Printer::printTermOrder(TermOrder const &t)
{
  t.print(*this);
}

string Printer::variableIndexToString(PolynomialRing const &r, int i)
{
  return r.getVariableName(i);
  //  fprintf(Stderr,"vits%i\n",i);
    /*  static char s[4];
  if(i>=0 && i<26)sprintf(s,"%c",i+'a');
  else if(i>=26 && i<52)sprintf(s,"%c",i+'A'-26);
  else assert(0);
  return string(s);
    */
}


void Printer::printVariable(PolynomialRing const &r, int i)
{
  printString(variableIndexToString(r,i));
}

void Printer::printPolyhedralCone(class PolyhedralCone const &c)
{
  PolyhedralCone C=c;
  C.print(this);
}


void Printer::printPolyhedralFan(class PolyhedralFan const &c)
{
  c.print(this);
}


void Printer::printField(class Field const &f)
{
  printString(f.toString().c_str());
}

void Printer::printPolynomialRing(class PolynomialRing const &r)
{
  printField(r.getField());
  printString("[");
  for(int i=0;i<r.getNumberOfVariables();i++)
    {
      if(i!=0)printString(",");
      printString(variableIndexToString(r,i));
    }
  printString("]");
}

//--------------------------------------------------
// LatexPrinter
//--------------------------------------------------

void LatexPrinter::pushMathMode()
{
  if(mathModeLevel==0)fprintf(f," $ ");
  mathModeLevel++;
}


void LatexPrinter::popMathMode()
{
  mathModeLevel--;
  if(mathModeLevel==0)fprintf(f," $ ");
}


void LatexPrinter::printMonomial(const Monomial &m, bool alwaysWriteSign, bool writeIfOne)
{
  const IntegerVector &exponent=m.exponent;
  const int sign=1;
  pushMathMode();

  bool variablePrinted=false;
  for(int i=0;i<exponent.size();i++)if(exponent[i]*sign>0)
    {
      fprintf(f,"%s",variableIndexToString(m.getRing(),i).c_str());
      if(int(exponent[i]*sign)!=1)
	{
	  fprintf(f,"^{%i}",int(exponent[i]*sign));
	}
      variablePrinted=true;
    }
  if(!variablePrinted && writeIfOne)fprintf(f,"1");

  popMathMode();
}


void LatexPrinter::printInteger(int i, int minimalFieldWidth)
{
  fprintf(f,"%i",i);
}


void LatexPrinter::printFieldElement(const FieldElement &e, bool writeIfOne, bool alwaysWriteSign)
{
  printString(e.toString(writeIfOne,alwaysWriteSign,true));
}

void LatexPrinter::printTerm(const Term &t)
{
  
}


void LatexPrinter::printPolynomial(const Polynomial &p)
{
  pushMathMode();

  bool first=true;
  // If the polynomial has a marked term it is written first
  IntegerVector e=p.getMarked().m.exponent;
  for(TermMap::const_iterator i=p.terms.begin();i!=p.terms.end();i++)
    if(e==i->first.exponent)
      {
	printFieldElement(i->second,i->first.exponent.isZero(),!first);
	printMonomial(i->first,false,false);
	first=false;
      }
  for(TermMap::const_iterator i=p.terms.begin();i!=p.terms.end();i++)
    if(e!=i->first.exponent)
      {
	printFieldElement(i->second,i->first.exponent.isZero(),!first);
	printMonomial(i->first,false,false);
	first=false;
      }
  popMathMode();
}

void LatexPrinter::printPolynomialSet(const PolynomialSet &s, bool newLine)
{
  pushMathMode();
  printString("\\{");
  if(newLine)
    {
      popMathMode();
      printNewLine();
      pushMathMode();
    }
  for(PolynomialSet::const_iterator i=s.begin();i!=s.end();i++)
    {
      if(i!=s.begin())
        {
          printString(",\n");
          if(newLine)
            {
              popMathMode();
              printNewLine();
              pushMathMode();
            }
        }
      printPolynomial(*i);
    }
  printString("\\}\n");
  popMathMode();
}


void LatexPrinter::printPolynomialSetList(const PolynomialSetList &s)
{
  pushMathMode();
  printString("\\{");
  for(PolynomialSetList::const_iterator i=s.begin();i!=s.end();i++)
    {
      if(i!=s.begin())printString(",\n");
      printPolynomialSet(*i);
    }
  printString("\\}\n");
  popMathMode();
}


void LatexPrinter::printVectorList(const IntegerVectorList &s, bool indexed)
{
  int index=0;
  pushMathMode();
  printString("\\{");
  for(IntegerVectorList::const_iterator i=s.begin();i!=s.end();i++)
    {
      if(i!=s.begin())printString(",\n");
      if(indexed)
	{
	  printInteger(index++);
	  printString(": ");
	}
      printVector(*i);
    }
  printString("\\}\n");
  popMathMode();
}



void LatexPrinter::printVector(const IntegerVector &v, bool curly, int minimalFieldWidth)
{
  int s=v.size();
  int i=0;

  fprintf(f,"$(");
  if(i<s)fprintf(f,"%i",(v[i]));
  for(i++;i<s;i++)fprintf(f,",%i",(v[i]));
  fprintf(f,")$\n");
}


void LatexPrinter::printString(const string &s)
{
  fprintf(f,"%s",s.c_str());
}


void LatexPrinter::printNewLine()
{
  fprintf(f,"\n\n");
}


void LatexPrinter::printLatexStart()
{
  fprintf(f,"\\documentclass[12pt,a4paper,dvips]{article}"
          "\\usepackage[english]{babel}"
          "\\usepackage{t1enc}"
          "\\usepackage{a4}"
          "\\usepackage{epsfig}"
          "\\usepackage{amsfonts}"
          "\\usepackage{latexsym}"
          "\\begin{document}\n");
}


void LatexPrinter::printLatexEnd()
{
  fprintf(f,"\\end{document}");
}

//--------------------------------------------------
// AsciiPrinter
//--------------------------------------------------

void AsciiPrinter::printInteger(int i, int minimalFieldWidth)
{
  static char s[16];
  sprintf(s,"%%%ii",minimalFieldWidth);
  fprintf(f,s,i);
}


void AsciiPrinter::printMonomial(const Monomial &m, bool alwaysWriteSign, bool writeIfOne)
{
  const IntegerVector &exponent=m.exponent;
  const int sign=1;

  //  fprintf(Stderr,"[%i]",m.exponent.size());

  bool variablePrinted=false;
  for(int i=0;i<exponent.size();i++)if(exponent[i]*sign>0)
    {
      if(variablePrinted)fprintf(f,"*");
      fprintf(f,"%s",variableIndexToString(m.getRing(),i).c_str());
      //      fprintf(f,"%c",i+'a');
      if(int(exponent[i]*sign)!=1)
	{
	  fprintf(f,"^%i",int(exponent[i]*sign));
	}
      variablePrinted=true;
    }
  if(!variablePrinted && writeIfOne)fprintf(f,"1");
  //  fprintf(stderr,"(%i)",exponent.size());
}


void AsciiPrinter::printFieldElement(const FieldElement &e, bool writeIfOne, bool alwaysWriteSign)
{
  printString(e.toString(writeIfOne,alwaysWriteSign));
}

void AsciiPrinter::printTerm(const Term &t)
{
  
}


void AsciiPrinter::printPolynomial(const Polynomial &p)
{
  bool first=true;

  if(p.terms.empty())
    {
      printString("0");
      return;
    }
  // If the polynomial has a marked term it is written first
  //   printString("_");
  IntegerVector e=p.getMarked().m.exponent;
  for(TermMap::const_iterator i=p.terms.begin();i!=p.terms.end();i++)
    if(e==i->first.exponent)
      {
	printFieldElement(i->second,i->first.exponent.isZero(),!first);
	if((!i->first.exponent.isZero())&&(!i->second.isOne())&&(!(-(i->second)).isOne()))printString("*");
	printMonomial(i->first,false,false);
	first=false;
      }
  //    printString("_");
  for(TermMap::const_iterator i=p.terms.begin();i!=p.terms.end();i++)
    if(e!=i->first.exponent)
      {
	printFieldElement(i->second,i->first.exponent.isZero(),!first);
	if((!i->first.exponent.isZero())&&(!i->second.isOne())&&(!(-(i->second)).isOne()))printString("*");
	printMonomial(i->first,false,false);
	first=false;
      }
}


void AsciiPrinter::printPolynomialSet(const PolynomialSet &s, bool newLine)
{
  printString("{");
  printNewLine();
  for(PolynomialSet::const_iterator i=s.begin();i!=s.end();i++)
    {
      if(i!=s.begin())printString(",\n");
      printPolynomial(*i);
    }
  printString("}\n");
}


void AsciiPrinter::printPolynomialSetList(const PolynomialSetList &s)
{
  printString("{");
  printNewLine();
  for(PolynomialSetList::const_iterator i=s.begin();i!=s.end();i++)
    {
      if(i!=s.begin())printString(",\n");
      printPolynomialSet(*i);
    }
  printString("}\n");
}


void AsciiPrinter::printVectorList(const IntegerVectorList &s, bool indexed)
{
  int index=0;
  printChar(vectorListLeftBrackets());
  printNewLine();
  for(IntegerVectorList::const_iterator i=s.begin();i!=s.end();i++)
    {
      if(i!=s.begin())printString(",\n");
      if(indexed)
	{
	  printInteger(index++);
	  printString(": ");
	}
      printVector(*i);
      //      printNewLine();
    }
  printChar(vectorListRightBrackets());
  printNewLine();
}


void AsciiPrinter::printVector(const IntegerVector &v, bool curly, int minimalFieldWidth)
{
  fprintf(f,"%c",curly?'{':vectorLeftBrackets());
  for(int i=0;i<v.size();i++)
    {
      if(i!=0)fprintf(f,",");
      printInteger(v[i],minimalFieldWidth);
    }
  fprintf(f,"%c",curly?'}':vectorRightBrackets());
}


void AsciiPrinter::printString(const string &s)
{
  fprintf(f,"%s",s.c_str());
}


void AsciiPrinter::printNewLine()
{
  fprintf(f,"\n");
}

/*
//--------------------------------------------------
// XmlPrinter
//--------------------------------------------------
void XmlPrinter::printMonomial(const Monomial &m)
{
}


void XmlPrinter::printVector(const IntegerVector &v)
{
}


void XmlPrinter::printString(const string &s)
{
}
*/



AsciiPrinter debug(Stderr);
