#!/usr/bin/perl
#
# ggcov - A GTK frontend for exploring gcov coverage data
# Copyright (c) 2006 Greg Banks <gnb@users.sourceforge.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# $Id: gencallbacks.pl,v 1.3 2010/05/09 05:37:15 gnb Exp $
#
#
#
# Generate a C data structure which maps names to function pointers
# for all the GLADE_CALLBACKS in the given C source files.
#
# Usage: ./gencallbacks.pl *.[cC] > callbacks.c
#

my $debug = 0;
my @functions;

sub balanced_parentheses($)
{
    my $str = shift;

    my $lefts = $str;
    $lefts =~ s/[^(]//g;

    my $rights = $str;
    $rights =~ s/[^)]//g;

    print STDERR "balanced_parentheses: lefts=\"$lefts\" rights=\"$rights\"\n" if ($debug > 1);

    return (length($lefts) == length($rights));
}

sub scan_file($)
{
    my $filename = shift;

    open FH, "<$filename"
	or die "Can't open $filename for reading";

    my $state = 0;
    my $return_type;
    my $function;
    my $arguments;

    while (<FH>)
    {
	chomp;
	s/\s+$//;

	print STDERR "[$state] $_\n" if ($debug > 1);

	if ($state == 1)
	{
	    ($function, $arguments) = m/^([a-z_][a-z0-9_]*)(\(.*)$/;
	    if (defined($function))
	    {
		print STDERR "Found: \"$function\"\n" if ($debug);
	    }
	    # collapse whitespace in args
	    $arguments =~ s/\s+/ /g;

	    # check for end of arguments
	    if (balanced_parentheses($arguments))
	    {
		$state = 0; # finished
		push(@functions, {
			return_type => $return_type,
			name => $function,
			arguments => $arguments,
			filename => $filename});
	    }
	    else
	    {
		$state = 2; # keep scanning lines for args
	    }
	}
	elsif ($state == 2)
	{
	    # collapse whitespace
	    s/^\s+//;
	    s/\s+/ /g;

	    # append to the arguments string
	    my $sep = ' ';
	    $sep = '' if ($arguments =~ m/\($/);
	    $arguments .= $sep . $_;

	    # check for end of arguments
	    if (balanced_parentheses($arguments))
	    {
		$state = 0; # finished
		push(@functions, {
			return_type => $return_type,
			name => $function,
			arguments => $arguments,
			filename => $filename});
	    }
	}
	elsif (m/^GLADE_CALLBACK/)
	{
	    s/^GLADE_CALLBACK\s+//;
	    $return_type = $_;
	    $state = 1;
	    next;
	}
    }

    close FH;
}

sub dump_functions()
{
    print "/* generated by gencallbacks.pl, do not edit */\n";
    print "#include \"ui.h\"\n";

    # generate extern declarations
    foreach my $fn (@functions)
    {
	print "/* $fn->{filename} */\n";
	print "GLADE_CALLBACK $fn->{return_type} $fn->{name}$fn->{arguments};\n";
    }

    # generate namelist data structure
    print "ui_named_callback_t ui_callbacks[] =\n{\n";
    foreach my $fn (@functions)
    {
	print "    {\"$fn->{name}\", (GCallback) $fn->{name}},\n";
    }
    print "    {0,0}\n";
    print "};\n";
}

foreach my $filename (@ARGV)
{
    if ($filename =~ m/\.[Cc]$/)
    {
	print STDERR "Scanning $filename\n" if ($debug);
	scan_file($filename);
    }
    else
    {
	print STDERR "Skipping $filename\n" if ($debug);
    }
}

@functions = sort { $a->{name} cmp $b->{name} } @functions;

dump_functions();
