-- #hide
--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALC.BasicTypes
-- Copyright   :  (c) Sven Panne 2003-2004
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven.panne@aedion.de
-- Stability   :  provisional
-- Portability :  portable
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALC.BasicTypes (
   ALCenum,
   Device(..), isDevice, fromDevice,
   withALString, peekALString,
   ContextAttribute(..), marshalContextAttribute, unmarshalContextAttribute
) where

import Foreign
import Foreign.C.String ( withCString, peekCString )
import Sound.OpenAL.AL.BasicTypes ( ALubyte, ALint )
import Sound.OpenAL.AL.ALboolean ( marshalALboolean, unmarshalALboolean )
import Sound.OpenAL.AL.Buffer ( Frequency )

--------------------------------------------------------------------------------

#include "HsOpenALConfig.h"

type ALCenum = HTYPE_ALCENUM

--------------------------------------------------------------------------------

type ALCdevice = ()

newtype Device = Device (Ptr ALCdevice)
   deriving ( Eq, Ord, Show )

isDevice :: Device -> Bool
isDevice (Device dev) = dev /= nullPtr

fromDevice :: Maybe Device -> Device
fromDevice = maybe (Device nullPtr) id

--------------------------------------------------------------------------------

-- AL uses "Ptr ALubyte" instead of "CString" for strings, so some wrappers
-- are quite handy.

withALString :: String -> (Ptr ALubyte -> IO a) -> IO a
withALString str action = withCString str (action . castPtr)

peekALString :: Ptr ALubyte -> IO String
peekALString = peekCString . castPtr

--------------------------------------------------------------------------------

data ContextAttribute =
     Frequency Frequency
   | Refresh ALint
   | Sync Bool
   | Sources ALint
   | Buffers ALint
   deriving ( Eq, Ord, Show )

marshalContextAttribute :: ContextAttribute -> (ALint,ALint)
marshalContextAttribute a = case a of
   Frequency f -> (CONST_ALC_FREQUENCY, fromIntegral f)
   Refresh r -> (CONST_ALC_REFRESH, r)
   Sync s -> (CONST_ALC_SYNC, fromIntegral (marshalALboolean s))
   Sources s -> (0x200000, s)
   Buffers b -> (0x200001, b)

unmarshalContextAttribute :: (ALint,ALint) -> ContextAttribute
unmarshalContextAttribute a@(x,y)
   | x == CONST_ALC_FREQUENCY = Frequency (fromIntegral y)
   | x == CONST_ALC_REFRESH = Refresh y
   | x == CONST_ALC_SYNC = Sync (unmarshalALboolean (fromIntegral y))
   | x == 0x200000 = Sources y
   | x == 0x200001 = Buffers y
   | otherwise = error ("unmarshalContextAttribute: illegal value " ++ show a)
