with Ada.Text_IO; use Ada.Text_IO;
with GNAT.Table;
with GNAT.OS_Lib;
with Errorout; use Errorout;
with Scan;
with Iirs_Utils;
with Parse;
with Back_End;
with Name_Table; use Name_Table;
with Str_Table;
with Sem_Scopes;
with Tokens;
with Files_Map;
with Flags;
with Std_Names;
with Std_Package;
with Iir_Chains;

package body Libraries is
   --  Chain of known libraries.  This is also the top node of all iir node.
   Libraries_Chain : Iir_Library_Declaration := Null_Iir;
   Libraries_Chain_Last : Iir_Library_Declaration := Null_Iir;

   --  A location for any implicit declarations (such as library WORK).
   Implicit_Location: Location_Type;

   --  Table of library pathes.
   package Pathes is new GNAT.Table
     (Table_Index_Type => Integer,
      Table_Component_Type => Name_Id,
      Table_Low_Bound => 1,
      Table_Initial => 4,
      Table_Increment => 100);

   --  Initialize pathes table.
   --  Set the local path.
   procedure Init_Pathes
   is
      Id : Name_Id;
   begin
      Id := Get_Identifier ("");
      Pathes.Append (Id);
      Local_Directory := Id;
      Work_Directory := Id;
   end Init_Pathes;

   function Path_To_Id (Path : String) return Name_Id is
   begin
      if Path (Path'Last) /= GNAT.OS_Lib.Directory_Separator then
         return Get_Identifier (Path & GNAT.OS_Lib.Directory_Separator);
      else
         return Get_Identifier (Path);
      end if;
   end Path_To_Id;

   procedure Add_Library_Path (Path : String)
   is
   begin
      if Path'Length = 0 then
         return;
      end if;
      Pathes.Increment_Last;
      Pathes.Table (Pathes.Last) := Path_To_Id (Path);
   end Add_Library_Path;

   function Get_Nbr_Pathes return Natural is
   begin
      return Pathes.Last;
   end Get_Nbr_Pathes;

   function Get_Path (N : Natural) return Name_Id is
   begin
      if N > Pathes.Last or N < Pathes.First then
         raise Constraint_Error;
      end if;
      return Pathes.Table (N);
   end Get_Path;

   --  Set PATH as the path of the work library.
   procedure Set_Work_Library_Path (Path : String) is
   begin
      Work_Directory := Path_To_Id (Path);
      if not GNAT.OS_Lib.Is_Directory (Get_Address (Work_Directory)) then
         Warning_Msg
           ("directory '" & Path & "' set by --workdir= does not exist");
      end if;
   end Set_Work_Library_Path;

   --  Open LIBRARY map file, return TRUE if successful.
   function Set_Library_File_Name (Dir : Name_Id;
                                   Library: Iir_Library_Declaration)
     return Boolean
   is
      File_Name : String := Back_End.Library_To_File_Name (Library);
      Fe : Source_File_Entry;
   begin
      Fe := Files_Map.Load_Source_File (Dir, Get_Identifier (File_Name));
      if Fe = No_Source_File_Entry then
         return False;
      end if;
      Scan.Set_File (Fe);
      return True;
   end Set_Library_File_Name;

   -- Load the contents of a library from a map file.
   -- The format of this file, used by save_library and load_library is
   -- as follow:
   --
   -- file_format ::= header { design_file_format }
   -- header ::= V 1
   -- design_file_format ::=
   --      filename_format { design_unit_format  }
   -- filename_format ::= FILE "FILENAME" :
   -- design_unit_format ::= entity_format
   --                        | architecture_format
   --                        | package_format
   --                        | package_body_format
   --                        | configuration_format
   -- position_format ::= LINE(POS) + OFF on DATE
   -- entity_format ::=
   --      ENTITY identifier AT position_format ;
   --        [ dependence_list ]
   -- architecture_format ::=
   --      ARCHITECTURE identifier of name AT position_format ;
   --        [ dependence_list ]
   --      [ WITH CONFIGURATION :
   --        [ dependence_list ] ]
   -- package_format ::=
   --      PACKAGE identifier AT position_format [BODY] ;
   --        [ dependence_list ]
   -- package_body_format ::=
   --      PACKAGE BODY identifier AT position_format ;
   --        [ dependence_list ]
   -- configuration_format ::=
   --      CONFIGURATION identifier AT position_format ;
   --        [ dependence_list ]
   -- dependence_list ::= ( dependence { , dependence } )
   -- dependence ::= library_identifier . primary_unit_identifier
   --                [ ( secondary_unit_identifier ) ]
   --
   -- The position_format meaning is:
   --       LINE is the line number (first line is number 1),
   --       POS is the offset of this line number, as a source_ptr value,
   --       OFF is the offset in the line, starting with 0.
   --       DATE is the symbolic date of analysis (order).
   --
   -- Return TRUE if the library was found.
   function Load_Library (Library: Iir_Library_Declaration)
     return Boolean
   is
      use Scan;
      use Tokens;
      use Iirs_Utils;

      File : Source_File_Entry;

      procedure Bad_Library_Format is
      begin
         Error_Msg (Image (Files_Map.Get_File_Name (File)) &
                    ": bad library format");
      end Bad_Library_Format;

      procedure Scan_Expect (Tok: Token_Type) is
      begin
         Scan.Scan;
         if Current_Token /= Tok then
            Bad_Library_Format;
            raise Compilation_Error;
         end if;
      end Scan_Expect;

      function Current_Time_Stamp return Time_Stamp_Id is
      begin
         if Current_String_Length /= Time_Stamp_String'Length then
            Bad_Library_Format;
            raise Compilation_Error;
         end if;
         return Time_Stamp_Id (Current_String_Id);
      end Current_Time_Stamp;

      function String_To_Name_Id return Name_Id
      is
         Len : Natural;
         Ptr : String_Fat_Acc;
      begin
         Len := Natural (Current_String_Length);
         Ptr := Str_Table.Get_String_Fat_Acc (Current_String_Id);
         for I in 1 .. Len loop
            Name_Table.Name_Buffer (I) := Ptr (I);
         end loop;
         Name_Table.Name_Length := Len;
         --  FIXME: should remove last string.
         return Get_Identifier;
      end String_To_Name_Id;

      Design_Unit, Last_Design_Unit : Iir_Design_Unit;
      Lib_Ident : Name_Id;

      function Scan_Unit_List return Iir_List
      is
         List : Iir_List;
         Aspect : Iir_Entity_Aspect_Entity;
         Selected_Name: Iir_Selected_Name;
         Id : Name_Id;
      begin
         if Current_Token = Tok_Left_Paren then
            List := Create_Iir_List;
            Scan_Expect (Tok_Identifier);
            loop
               Selected_Name := Create_Iir (Iir_Kind_Selected_Name);
               Id := Current_Identifier;
               if Id = Lib_Ident then
                  Set_Prefix (Selected_Name, Library);
               else
                  Set_Prefix (Selected_Name, Current_Text);
               end if;
               Scan_Expect (Tok_Dot);
               Scan_Expect (Tok_Identifier);
               Set_Suffix_Identifier (Selected_Name, Current_Identifier);
               Scan.Scan;
               if Current_Token = Tok_Left_Paren then
                  --  This is an architecture.
                  Aspect := Create_Iir (Iir_Kind_Entity_Aspect_Entity);
                  Set_Entity (Aspect, Selected_Name);
                  Scan_Expect (Tok_Identifier);
                  Set_Architecture (Aspect, Current_Text);
                  Scan_Expect (Tok_Right_Paren);
                  Scan.Scan;
                  Append_Element (List, Aspect);
               else
                  Append_Element (List, Selected_Name);
               end if;
               exit when Current_Token /= Tok_Comma;
               Scan.Scan;
            end loop;
            Scan.Scan;
            return List;
         else
            return Null_Iir_List;
         end if;
      end Scan_Unit_List;

      Design_File: Iir_Design_File;
      Library_Unit: Iir;
      Line, Col: Natural;
      Pos: Source_Ptr;
      Date: Date_Type;
      Max_Date: Date_Type := Date_Valid'First;
      Dir : Name_Id;
      use Iir_Chains.Design_Unit_Chain_Handling;
   begin
      Lib_Ident := Get_Identifier (Library);

      -- Check the library was not already loaded.
      if Get_Design_File_Chain (Library) /= Null_Iir then
         raise Internal_Error;
      end if;

      -- Try to open the library file map.
      Dir := Get_Library_Directory (Library);
      if Dir = Null_Identifier then
         --  Search in the library path.
         declare
            File_Name : String := Back_End.Library_To_File_Name (Library);
            L : Natural;
         begin
            for I in Pathes.First .. Pathes.Last loop
               Image (Pathes.Table (I));
               L := Name_Length + File_Name'Length;
               Name_Buffer (Name_Length + 1 .. L) := File_Name;
               Name_Buffer (L + 1) := Character'Val (0);
               if GNAT.OS_Lib.Is_Regular_File (Name_Buffer'Address) then
                  Dir := Pathes.Table (I);
                  Set_Library_Directory (Library, Dir);
                  exit;
               end if;
            end loop;
         end;
      end if;
      if Dir = Null_Identifier
        or else not Set_Library_File_Name (Dir, Library)
      then
         --  Not found.
         Set_Date (Library, Date_Valid'First);
         return False;
      end if;
      File := Get_Current_Source_File;

      --  Parse header.
      Scan.Scan;
      if Current_Token /= Tok_Identifier
        or else Name_Length /= 1 or else Name_Buffer (1) /= 'v'
      then
         Bad_Library_Format;
         raise Compilation_Error;
      end if;
      Scan_Expect (Tok_Integer);
      if Current_Iir_Int64 /= 1 then
         Bad_Library_Format;
         raise Compilation_Error;
      end if;
      Scan.Scan;

      Build_Init (Last_Design_Unit);
      while Current_Token /= Tok_Eof loop
         if Current_Token = Tok_File then
            -- This is a new design file.
            Design_File := Create_Iir (Iir_Kind_Design_File);

            Scan_Expect (Tok_String);
            if Name_Length = 0 then
               --  The directory may be an empty identifier (when the
               --   filename is an absolute filename).
               --  Recognize this now.
               Set_Design_File_Directory (Design_File, Null_Identifier);
            else
               Set_Design_File_Directory (Design_File, String_To_Name_Id);
            end if;

            Scan_Expect (Tok_String);
            Set_Design_File_Filename (Design_File, String_To_Name_Id);

            -- FIXME: check the file name is uniq.

            Set_Parent (Design_File, Library);

            --  Prepend.
            Set_Chain (Design_File, Get_Design_File_Chain (Library));
            Set_Design_File_Chain (Library, Design_File);

            Scan_Expect (Tok_String);
            Set_File_Time_Stamp (Design_File, Current_Time_Stamp);

            Scan_Expect (Tok_String);
            Set_Analysis_Time_Stamp (Design_File, Current_Time_Stamp);

            Scan_Expect (Tok_Colon);
            Scan.Scan;
            Build_Init (Last_Design_Unit);
         else
            -- This is a new design unit.
            Design_Unit := Create_Iir (Iir_Kind_Design_Unit);
            Set_Design_File (Design_Unit, Design_File);
            case Current_Token is
               when Tok_Entity =>
                  Library_Unit := Create_Iir (Iir_Kind_Entity_Declaration);
                  Scan.Scan;
               when Tok_Architecture =>
                  Library_Unit :=
                    Create_Iir (Iir_Kind_Architecture_Declaration);
                  Scan.Scan;
               when Tok_Configuration =>
                  Library_Unit :=
                    Create_Iir (Iir_Kind_Configuration_Declaration);
                  Scan.Scan;
               when Tok_Package =>
                  Scan.Scan;
                  if Current_Token = Tok_Body then
                     Library_Unit := Create_Iir (Iir_Kind_Package_Body);
                     Scan.Scan;
                  else
                     Library_Unit := Create_Iir (Iir_Kind_Package_Declaration);
                  end if;
               when Tok_With =>
                  if Library_Unit = Null_Iir
                    or else
                    Get_Kind (Library_Unit)
                    /= Iir_Kind_Architecture_Declaration
                  then
                     Put_Line ("load_library: invalid use of 'with'");
                     raise Internal_Error;
                  end if;
                  Set_Default_Configuration_Declaration
                    (Library_Unit, Design_Unit);
                  Scan_Expect (Tok_Configuration);
                  Scan_Expect (Tok_Colon);
                  Scan.Scan;
                  Set_Dependence_List (Design_Unit, Scan_Unit_List);
                  goto Next_Line;
               when others =>
                  Put_Line
                    ("load_library: line must start with " &
                     "'architecture', 'entity', 'package' or 'configuration'");
                  raise Internal_Error;
            end case;

            if Current_Token /= Tok_Identifier then
               raise Internal_Error;
            end if;
            Set_Identifier (Library_Unit, Current_Identifier);
            Set_Identifier (Design_Unit, Current_Identifier);
            Set_Visible_Flag (Design_Unit, True);

            if Get_Kind (Library_Unit) = Iir_Kind_Architecture_Declaration then
               Scan_Expect (Tok_Of);
               Scan_Expect (Tok_Identifier);
               Set_Entity (Library_Unit, Current_Text);
            end if;

            -- Scan position.
            Scan_Expect (Tok_Identifier); -- at
            Scan_Expect (Tok_Integer);
            Line := Natural (Current_Iir_Int64);
            Scan_Expect (Tok_Left_Paren);
            Scan_Expect (Tok_Integer);
            Pos := Source_Ptr (Current_Iir_Int64);
            Scan_Expect (Tok_Right_Paren);
            Scan_Expect (Tok_Plus);
            Scan_Expect (Tok_Integer);
            Col := Natural (Current_Iir_Int64);
            Scan_Expect (Tok_On);
            Scan_Expect (Tok_Integer);
            Date := Date_Type (Current_Iir_Int64);

            Scan.Scan;
            if Get_Kind (Library_Unit) = Iir_Kind_Package_Declaration
              and then Current_Token = Tok_Body
            then
               Set_Need_Body (Library_Unit, True);
               Scan.Scan;
            end if;
            if Current_Token /= Tok_Semi_Colon then
               raise Internal_Error;
            end if;
            Scan.Scan;

            if False then
               Put_Line ("line:" & Natural'Image (Line)
                         & ", pos:" & Source_Ptr'Image (Pos));
            end if;

            -- Scan dependence list.
            Set_Dependence_List (Design_Unit, Scan_Unit_List);

            -- Keep the position of the design unit.
            --Set_Location (Design_Unit, Location_Type (File));
            --Set_Location (Library_Unit, Location_Type (File));
            Set_Pos_Line_Off (Design_Unit, Pos, Line, Col);
            Set_Date (Design_Unit, Date);
            if Date > Max_Date then
               Max_Date := Date;
            end if;
            Set_Library_Unit (Design_Unit, Library_Unit);
            Set_Design_Unit (Library_Unit, Design_Unit);

            Append (Last_Design_Unit, Design_File, Design_Unit);
         end if;
         << Next_Line >> null;
      end loop;
      Set_Date (Library, Max_Date);
      Close_File;
      return True;
   end Load_Library;

   procedure Create_Virtual_Locations
   is
      use Files_Map;
      use Name_Table;
      Error_Source_File : Source_File_Entry;
      Implicit_Source_File : Source_File_Entry;
      Command_Source_File : Source_File_Entry;
   begin
      Error_Source_File := Create_Virtual_Source_File
        (Get_Identifier ("*unknown*"));
      if Error_Source_File /= No_Source_File_Entry + 1 then
         raise Internal_Error;
      end if;
      Implicit_Source_File := Create_Virtual_Source_File
        (Get_Identifier ("*implicit*"));
      Command_Source_File := Create_Virtual_Source_File
        (Get_Identifier ("*command line*"));
      Implicit_Location := Source_File_To_Location (Implicit_Source_File);
   end Create_Virtual_Locations;

   -- Note: the scanner shouldn't be in use, since this procedure uses it.
   procedure Load_Std_Library
   is
      Dir : Name_Id;
   begin
      if Libraries_Chain /= Null_Iir then
         --  This procedure must not be called twice.
         raise Internal_Error;
      end if;
      Flags.Create_Flag_String;
      Create_Virtual_Locations;
      Std_Library := Create_Iir (Iir_Kind_Library_Declaration);
      Set_Identifier (Std_Library, Std_Names.Name_Std);
      Set_Location (Std_Library, Implicit_Location);
      --Set_Visible_Flag (Std_Library, True);
      if Flags.Bootstrap
        and then Work_Library_Name = Std_Names.Name_Std
      then
         Dir := Work_Directory;
      else
         Dir := Null_Identifier;
      end if;
      Set_Library_Directory (Std_Library, Dir);
      if Load_Library (Std_Library) = False
        and then not Flags.Bootstrap
      then
         Error_Msg_Option ("cannot find 'std' library");
      end if;
      Libraries_Chain := Std_Library;
      Libraries_Chain_Last := Std_Library;
      Set_Visible_Flag (Std_Library, True);
   end Load_Std_Library;

   procedure Load_Work_Library (Empty : Boolean := False)
   is
      use Std_Names;
   begin
      if Work_Library_Name = Name_Std then
         if not Flags.Bootstrap then
            Error_Msg_Option ("the WORK library cannot be STD");
            return;
         end if;
         Work_Library := Std_Library;
      else
         Work_Library := Create_Iir (Iir_Kind_Library_Declaration);
         Set_Location (Work_Library, Implicit_Location);
         --Set_Visible_Flag (Work_Library, True);
         Set_Library_Directory (Work_Library, Work_Directory);

         Set_Identifier (Work_Library, Work_Library_Name);

         if not Empty then
            if Load_Library (Work_Library) = False then
               null;
            end if;
         end if;

         --  Add it to the list of libraries.
         Set_Chain (Libraries_Chain_Last, Work_Library);
         Libraries_Chain_Last := Work_Library;
      end if;
      Set_Visible_Flag (Work_Library, True);
   end Load_Work_Library;

--    procedure Unload_Library (Library : Iir_Library_Declaration)
--    is
--       File : Iir_Design_File;
--       Unit : Iir_Design_Unit;
--    begin
--       loop
--          File := Get_Design_File_Chain (Library);
--          exit when File = Null_Iir;
--          Set_Design_File_Chain (Library, Get_Chain (File));

--          loop
--             Unit := Get_Design_Unit_Chain (File);
--             exit when Unit = Null_Iir;
--             Set_Design_Unit_Chain (File, Get_Chain (Unit));

--             --  Units should not be loaded.
--             if Get_Loaded_Flag (Unit) then
--                raise Internal_Error;
--             end if;

--             --  Free dependences list.
--          end loop;
--       end loop;
--    end Unload_Library;

--    procedure Unload_All_Libraries
--    is
--       Library : Iir_Library_Declaration;
--    begin
--       if Get_Identifier (Std_Library) /= Name_Std then
--          raise Internal_Error;
--       end if;
--       if Std_Library /= Libraries_Chain then
--          raise Internal_Error;
--       end if;
--       loop
--          Library := Get_Chain (Libraries_Chain);
--          exit when Library = Null_Iir;
--          Set_Chain (Libraries_Chain, Get_Chain (Libraries_Chain));
--          Unload_Library (Library);
--       end loop;
--    end Unload_All_Libraries;

   -- Get or create a library from an identifier.
   function Get_Library (Ident: Name_Id) return Iir_Library_Declaration
   is
      Library: Iir_Library_Declaration;
   begin
      -- library work is a little bit special.
      if Ident = Std_Names.Name_Work or else Ident = Work_Library_Name then
         if Work_Library = Null_Iir then
            --  load_work_library must have been called before.
            raise Internal_Error;
         end if;
         return Work_Library;
      end if;

      --  Check if the library has already been loaded.
      Library := Iirs_Utils.Find_Name_In_Chain (Libraries_Chain, Ident);
      if Library /= Null_Iir then
         return Library;
      end if;

      --  This is a new library.
      if Ident = Std_Names.Name_Std then
         --  Load_std_library must have been called before.
         raise Internal_Error;
      end if;

      Library := Create_Iir (Iir_Kind_Library_Declaration);
      Set_Location (Library, Scan.Get_Token_Location);
      Set_Library_Directory (Library, Null_Identifier);
      Set_Identifier (Library, Ident);
      if Load_Library (Library) = False then
         Error_Msg_Option
           ("cannot find resource library '" & Name_Table.Image (Ident) & "'");
      end if;
      Set_Visible_Flag (Library, True);

      Set_Chain (Libraries_Chain_Last, Library);
      Libraries_Chain_Last := Library;

      return Library;
   end Get_Library;

   -- Return TRUE if LIBRARY_UNIT and UNIT have identifiers for the same
   -- design unit identifier.
   -- eg: 'entity A' and 'package A' returns TRUE.
   function Is_Same_Library_Unit (Library_Unit, Unit: Iir) return Boolean
   is
      Entity_Name1, Entity_Name2: Name_Id;
      Library_Unit_Kind, Unit_Kind : Iir_Kind;
   begin
      if Get_Identifier (Unit) /= Get_Identifier (Library_Unit) then
         return False;
      end if;

      Library_Unit_Kind := Get_Kind (Library_Unit);
      Unit_Kind := Get_Kind (Unit);

      --  Package and package body are never the same library unit.
      if Library_Unit_Kind = Iir_Kind_Package_Declaration
        and then Unit_Kind = Iir_Kind_Package_Body
      then
         return False;
      end if;
      if Unit_Kind = Iir_Kind_Package_Declaration
        and then Library_Unit_Kind = Iir_Kind_Package_Body
      then
         return False;
      end if;

      --  Two architecture declarations are identical only if they also have
      --  the same entity name.
      if Unit_Kind = Iir_Kind_Architecture_Declaration
        and then Library_Unit_Kind = Iir_Kind_Architecture_Declaration
      then
         Entity_Name1 := Get_Identifier (Get_Entity (Unit));
         Entity_Name2 := Get_Identifier (Get_Entity (Library_Unit));
         if Entity_Name1 /= Entity_Name2 then
            return False;
         end if;
      end if;

      --  An architecture declaration never conflits with a library unit that
      --  is not an architecture declaration.
      if (Unit_Kind = Iir_Kind_Architecture_Declaration
          and then Library_Unit_Kind /= Iir_Kind_Architecture_Declaration)
        or else
        (Unit_Kind /= Iir_Kind_Architecture_Declaration
         and then Library_Unit_Kind = Iir_Kind_Architecture_Declaration)
      then
         return False;
      end if;

      return True;
   end Is_Same_Library_Unit;

   procedure Free_Dependence_List (Design : Iir_Design_Unit)
   is
      List : Iir_List;
      El : Iir;
   begin
      List := Get_Dependence_List (Design);
      if List /= Null_Iir_List then
         for I in Natural loop
            El := Get_Nth_Element (List, I);
            exit when El = Null_Iir;
            Iirs_Utils.Free_Recursive (El);
         end loop;
         Destroy_Iir_List (List);
      end if;
   end Free_Dependence_List;

   procedure Free_Design_Unit (Design_Unit : Iir_Design_Unit)
   is
      Lib : Iir;
      Unit : Iir_Design_Unit;
      Dep_List : Iir_List;
   begin
      Dep_List := Get_Dependence_List (Design_Unit);
      Destroy_Iir_List (Dep_List);
      Lib := Get_Library_Unit (Design_Unit);
      if Lib /= Null_Iir
        and then Get_Kind (Lib) = Iir_Kind_Architecture_Declaration
      then
         Unit := Get_Default_Configuration_Declaration (Lib);
         if Unit /= Null_Iir then
            Free_Design_Unit (Unit);
         end if;
      end if;
      Iirs_Utils.Free_Old_Iir (Lib);
   end Free_Design_Unit;

   -- Add or replace a design unit in the working library.
   procedure Add_Design_Unit_Into_Library (Decl: Iir_Design_Unit)
   is
      Design_File: Iir_Design_File;
      Design_Unit, Prev_Design_Unit : Iir_Design_Unit;
      Library_Unit: Iir;
      New_Library_Unit: Iir;
      Done: Boolean := False;
      Date: Date_Type;
      New_Lib_Time_Stamp : Time_Stamp_Id;

      --  File name and dir name of DECL.
      File_Name : Name_Id;
      Dir_Name : Name_Id;
   begin
      pragma Assert (Get_Chain (Decl) = Null_Iir);

      --  Mark this design unit as being loaded.
      Set_Loaded_Flag (Decl, True);
      New_Library_Unit := Get_Library_Unit (Decl);

      --  Set the date of the design unit as the most recently analyzed
      --  design unit.
      case Get_Date (Decl) is
         when Date_Parsed =>
            null;
         when Date_Analyzed =>
            Date := Get_Date (Work_Library) + 1;
            Set_Date (Decl, Date);
            Set_Date (Work_Library, Date);
         when Date_Valid =>
            raise Internal_Error;
         when others =>
            raise Internal_Error;
      end case;

      --  Set file time stamp.
      declare
         File : Source_File_Entry;
         Pos : Source_Ptr;
      begin
         Files_Map.Location_To_File_Pos (Get_Location (New_Library_Unit),
                                         File, Pos);
         New_Lib_Time_Stamp := Files_Map.Get_File_Time_Stamp (File);
         File_Name := Files_Map.Get_File_Name (File);
         Image (File_Name);
         if Files_Map.Is_Absolute_Pathname (Name_Buffer (1 .. Name_Length))
         then
            Dir_Name := Null_Identifier;
         else
            Dir_Name := Files_Map.Get_Home_Directory;
         end if;
      end;

      Design_File := Get_Design_File_Chain (Work_Library);
      while Design_File /= Null_Iir loop

         --  Try to find a design unit with the same name in the work library.
         Prev_Design_Unit := Null_Iir;
         Design_Unit := Get_Design_Unit_Chain (Design_File);
         loop
            << Again >> exit when Design_Unit = Null_Iir;

            Library_Unit := Get_Library_Unit (Design_Unit);

            if Is_Same_Library_Unit (New_Library_Unit, Library_Unit) then
               -- LIBRARY_UNIT and UNIT designate the same design unit.
               -- UNIT *must* replace library_unit if they don't belong
               -- to the same file.
               Set_Date (Design_Unit, Date_Obsolete);
               if Get_Design_File_Filename (Design_File) = File_Name
                 and then Get_Design_File_Directory (Design_File) = Dir_Name
               then
                  declare
                     Next_Design : Iir;
                  begin
                     Next_Design := Get_Chain (Design_Unit);

                     --  Remove DESIGN_UNIT.
                     if Prev_Design_Unit = Null_Iir then
                        Set_Design_Unit_Chain (Design_File, Next_Design);
                     else
                        Set_Chain (Prev_Design_Unit, Next_Design);
                     end if;

                     if Get_Loaded_Flag (Design_Unit) then
                        if Flags.Warn_Library then
                           Warning_Msg_Sem
                             ("redefinition of a library unit in "
                              & "same design file:", Decl);
                           Warning_Msg_Sem
                             (Disp_Node (Library_Unit) & " defined at "
                              & Disp_Location (Library_Unit) & " is now "
                              & Disp_Node (New_Library_Unit), Decl);
                        end if;
                     else
                        --  Free the stub.
                        Free_Design_Unit (Design_Unit);
                     end if;
                     --  Note: the current design unit should not be freed if
                     --  in use; unfortunatly, this is not obvious to check.
                     Design_Unit := Next_Design;
                     goto Again;
                  end;
               else
                  if Flags.Warn_Library then
                     if Get_Kind (Library_Unit) /= Get_Kind (New_Library_Unit)
                     then
                        Warning_Msg ("changing definition of a library unit:");
                        Warning_Msg (Disp_Node (Library_Unit) & " is now "
                                     & Disp_Node (New_Library_Unit));
                     end if;
                     Warning_Msg
                       ("library unit '"
                        & Iirs_Utils.Image_Identifier (Library_Unit)
                        & "' was also defined in file '"
                        & Image (Get_Design_File_Filename (Design_File))
                        & ''');
                  end if;

                  --  Remove DESIGN_UNIT.
                  Design_Unit := Get_Chain (Design_Unit);
                  if Prev_Design_Unit = Null_Iir then
                     Set_Design_Unit_Chain (Design_File, Design_Unit);
                  else
                     Set_Chain (Prev_Design_Unit, Design_Unit);
                  end if;
                  goto Again;
               end if;
            end if;
            Prev_Design_Unit := Design_Unit;
            Design_Unit := Get_Chain (Design_Unit);
         end loop;

         --  Maybe the design_file already exists.
         if Get_Design_File_Filename (Design_File) = File_Name
           and then Get_Design_File_Directory (Design_File) = Dir_Name
         then
            --  The new design unit belongs to this design file.
            if not Done then
               --  Add it if not already replaced.
               if Prev_Design_Unit = Null_Iir then
                  Set_Design_Unit_Chain (Design_File, Decl);
               else
                  Set_Chain (Prev_Design_Unit, Decl);
               end if;
               Set_Design_File (Decl, Design_File);
               Done := True;
            end if;
            if not Files_Map.Is_Eq (New_Lib_Time_Stamp,
                                    Get_File_Time_Stamp (Design_File))
            then
               -- FIXME: this test is not enough: what about reanalyzing
               --  unmodified files (this works only because the order is not
               --  changed).
               -- Design file is updated.
               -- Outdate all other units, overwrite the design_file.
               Set_File_Time_Stamp (Design_File, New_Lib_Time_Stamp);
               Design_Unit := Get_Design_Unit_Chain (Design_File);
               while Design_Unit /= Null_Iir loop
                  if Design_Unit /= Decl then
                     --  Mark other design unit as obsolete.
                     Set_Date (Design_Unit, Date_Obsolete);
                  end if;
                  Design_Unit := Get_Chain (Design_Unit);
               end loop;
               Set_Design_Unit_Chain (Design_File, Decl);
               Set_Chain (Decl, Null_Iir);
            end if;
         end if;
         Design_File := Get_Chain (Design_File);
      end loop;

      if not Done then
         -- This is the first apparition of the design file.
         Design_File := Create_Iir (Iir_Kind_Design_File);
         Location_Copy (Design_File, Decl);

         Set_Design_File_Filename (Design_File, File_Name);
         Set_Design_File_Directory (Design_File, Dir_Name);

         Set_File_Time_Stamp (Design_File, New_Lib_Time_Stamp);
         Set_Parent (Design_File, Work_Library);
         Set_Chain (Design_File, Get_Design_File_Chain (Work_Library));
         Set_Design_File_Chain (Work_Library, Design_File);
         Set_Design_Unit_Chain (Design_File, Decl);
         Set_Design_File (Decl, Design_File);
      else
         Design_File := Get_Design_File (Decl);
      end if;
      --  Update the analyzed time stamp.
      Set_Analysis_Time_Stamp (Design_File, Files_Map.Get_Os_Time_Stamp);
   end Add_Design_Unit_Into_Library;

   -- Save the file map of library LIBRARY.
   procedure Save_Library (Library: Iir_Library_Declaration) is
      File: File_Type;

      procedure Write_Dependence_List (Unit : Iir_Design_Unit)
      is
         use Iirs_Utils;

         procedure Write_Selected_Name (N : Iir_Selected_Name)
         is
         begin
            Put (File, Image_Identifier (Get_Prefix (N)));
            Put (File, '.');
            Put (File, Name_Table.Image (Get_Suffix_Identifier (N)));
         end Write_Selected_Name;

         procedure Write_Secondary_Name (N : Iir)
         is
         begin
            Put (File, '(');
            Put (File, Image_Identifier (N));
            Put (File, ')');
         end Write_Secondary_Name;

         procedure Write_Design_Unit (Design : Iir_Design_Unit)
         is
            Lib_Unit : Iir;
         begin
            Put (File,
                 Image_Identifier (Get_Parent (Get_Design_File (Design))));
            Put (File, '.');
            Lib_Unit := Get_Library_Unit (Design);
            case Get_Kind (Lib_Unit) is
               when Iir_Kind_Architecture_Declaration =>
                  Put (File, Image_Identifier (Get_Entity (Lib_Unit)));
                  Write_Secondary_Name (Lib_Unit);
               when Iir_Kind_Entity_Declaration
                 | Iir_Kind_Package_Declaration
                 | Iir_Kind_Configuration_Declaration =>
                  Put (File, Image_Identifier (Lib_Unit));
               when others =>
                  raise Internal_Error;
            end case;
         end Write_Design_Unit;

         procedure Write_Depend_Element (Depend : Iir)
         is
         begin
            case Get_Kind (Depend) is
               when Iir_Kind_Design_Unit =>
                  Write_Design_Unit (Depend);
               when Iir_Kind_Selected_Name =>
                  Write_Selected_Name (Depend);
               when Iir_Kind_Entity_Aspect_Entity =>
                  Write_Depend_Element (Get_Entity (Depend));
                  Write_Secondary_Name (Get_Architecture (Depend));
               when others =>
                  raise Internal_Error;
            end case;
         end Write_Depend_Element;

         Depend_List: Iir_List;
         Nbr: Natural;
      begin
         Depend_List := Get_Dependence_List (Unit);
         if Depend_List = Null_Iir_List then
            return;
         end if;
         Nbr := Get_Nbr_Elements (Depend_List);
         for I in 0 .. Nbr - 1 loop
            if I = 0 then
               Put (File, "   (");
            else
               Put (File, ", ");
            end if;
            Write_Depend_Element (Get_Nth_Element (Depend_List, I));
         end loop;
         if Nbr /= 0 then
            Put_Line (File, ")");
         end if;
      end Write_Dependence_List;

      Design_File: Iir_Design_File;
      Design_Unit: Iir_Design_Unit;
      Library_Unit: Iir;
      --Ident: Iir_Acc;

      Off, Line: Natural;
      Pos: Source_Ptr;
      Source_File : Source_File_Entry;
   begin
      -- FIXME: directory
      declare
         use Files_Map;
         File_Name: String := Image (Work_Directory)
           & Back_End.Library_To_File_Name (Library);
      begin
         Create (File, Out_File, File_Name);
      exception
         when Use_Error =>
            Open (File, Out_File, File_Name);
      end;

      --  Header: version.
      Put_Line (File, "v 1");

      Design_File := Get_Design_File_Chain (Library);
      while Design_File /= Null_Iir loop
         if Design_File = Std_Package.Std_Standard_File then
            goto Continue;
         end if;
         Design_Unit := Get_Design_Unit_Chain (Design_File);

         if Design_Unit /= Null_Iir then
            Put (File, "file """);
            Image (Get_Design_File_Directory (Design_File));
            Put (File, Name_Buffer (1 .. Name_Length));
            Put (File, """ """);
            Image (Get_Design_File_Filename (Design_File));
            Put (File, Name_Buffer (1 .. Name_Length));
            Put (File, """ """);
            Put (File, Files_Map.Get_Time_Stamp_String
                 (Get_File_Time_Stamp (Design_File)));
            Put (File, """ """);
            Put (File, Files_Map.Get_Time_Stamp_String
                 (Get_Analysis_Time_Stamp (Design_File)));
            Put_Line (File, """:");
         end if;

         while Design_Unit /= Null_Iir loop
            Library_Unit := Get_Library_Unit (Design_Unit);

            Put (File, "  ");
            case Get_Kind (Library_Unit) is
               when Iir_Kind_Entity_Declaration =>
                  Put (File, "entity ");
                  Put (File, Iirs_Utils.Image_Identifier (Library_Unit));
               when Iir_Kind_Architecture_Declaration =>
                  Put (File, "architecture ");
                  Put (File, Iirs_Utils.Image_Identifier (Library_Unit));
                  Put (File, " of ");
                  Put (File, Iirs_Utils.Image_Identifier
                       (Get_Entity (Library_Unit)));
               when Iir_Kind_Package_Declaration =>
                  Put (File, "package ");
                  Put (File, Iirs_Utils.Image_Identifier (Library_Unit));
               when Iir_Kind_Package_Body =>
                  Put (File, "package body ");
                  Put (File, Iirs_Utils.Image_Identifier (Library_Unit));
               when Iir_Kind_Configuration_Declaration =>
                  Put (File, "configuration ");
                  Put (File, Iirs_Utils.Image_Identifier (Library_Unit));
               when others =>
                  Error_Kind ("save_library", Library_Unit);
            end case;

            if not Get_Loaded_Flag (Design_Unit) then
               Get_Pos_Line_Off (Design_Unit, Pos, Line, Off);
            else
               Files_Map.Location_To_Coord (Get_Location (Design_Unit),
                                            Source_File, Pos, Line, Off);
            end if;

            Put (File, " at");
            Put (File, Natural'Image (Line));
            Put (File, "(");
            Put (File, Source_Ptr'Image (Pos));
            Put (File, ") +");
            Put (File, Natural'Image (Off));
            Put (File, " on");
            case Get_Date (Design_Unit) is
               when Date_Valid
                 | Date_Analyzed
                 | Date_Parsed =>
                  Put (File, Date_Type'Image (Get_Date (Design_Unit)));
               when others =>
                  raise Internal_Error;
            end case;
            if Get_Kind (Library_Unit) = Iir_Kind_Package_Declaration
              and then Get_Need_Body (Library_Unit)
            then
               Put (File, " body");
            end if;
            Put_Line (File, ";");

            Write_Dependence_List (Design_Unit);

            if Get_Kind (Library_Unit) = Iir_Kind_Architecture_Declaration then
               declare
                  Config : Iir_Design_Unit;
               begin
                  Config := Get_Default_Configuration_Declaration
                    (Library_Unit);
                  if Config /= Null_Iir then
                     Put_Line (File, "  with configuration:");
                     Write_Dependence_List (Config);
                  end if;
               end;
            end if;
            Design_Unit := Get_Chain (Design_Unit);
         end loop;
         << Continue >> null;
         Design_File := Get_Chain (Design_File);
      end loop;

      Close (File);
   end Save_Library;

   -- Save the map of the work library.
   procedure Save_Work_Library is
   begin
      Save_Library (Work_Library);
   end Save_Work_Library;

   -- Return the name of the latest architecture analysed for an entity.
   function Get_Latest_Architecture (Entity: Iir_Entity_Declaration)
     return Iir_Architecture_Declaration
   is
      Entity_Id : Name_Id;
      Lib : Iir_Library_Declaration;
      Design_File: Iir_Design_File;
      Design_Unit: Iir_Design_Unit;
      Library_Unit: Iir;
      Res: Iir_Design_Unit;
   begin
      Entity_Id := Get_Identifier (Entity);
      Lib := Get_Parent (Get_Design_File (Get_Design_Unit (Entity)));
      Design_File := Get_Design_File_Chain (Lib);
      Res := Null_Iir;
      while Design_File /= Null_Iir loop
         Design_Unit := Get_Design_Unit_Chain (Design_File);
         while Design_Unit /= Null_Iir loop
            Library_Unit := Get_Library_Unit (Design_Unit);

            if Get_Kind (Library_Unit) = Iir_Kind_Architecture_Declaration
              and then Get_Identifier (Get_Entity (Library_Unit)) = Entity_Id
            then
               if Res = Null_Iir then
                  Res := Design_Unit;
               elsif Get_Date (Design_Unit) > Get_Date (Res) then
                  Res := Design_Unit;
               end if;
            end if;
            Design_Unit := Get_Chain (Design_Unit);
         end loop;
         Design_File := Get_Chain (Design_File);
      end loop;
      if Res = Null_Iir then
         return Null_Iir;
      else
         return Get_Library_Unit (Res);
      end if;
   end Get_Latest_Architecture;

   function Load_File (File : Source_File_Entry) return Iir_Design_File
   is
      Res : Iir_Design_File;
   begin
      Scan.Set_File (File);
      Res := Parse.Parse_Design_File;
      Scan.Close_File;
      if Res /= Null_Iir then
         Set_Parent (Res, Work_Library);
         Set_Design_File_Filename (Res, Files_Map.Get_File_Name (File));
      end if;
      return Res;
   end Load_File;

   -- parse a file.
   -- Return a design_file without putting it into the library
   -- (because it was not semantized).
   function Load_File (File_Name: Name_Id) return Iir_Design_File
   is
      Fe : Source_File_Entry;
   begin
      Fe := Files_Map.Load_Source_File (Local_Directory, File_Name);
      if Fe = No_Source_File_Entry then
         Error_Msg_Option ("cannot open " & Image (File_Name));
         return Null_Iir;
      end if;
      return Load_File (Fe);
   end Load_File;

   function Find_Design_Unit (Unit : Iir) return Iir_Design_Unit is
   begin
      case Get_Kind (Unit) is
         when Iir_Kind_Design_Unit =>
            return Unit;
         when Iir_Kind_Selected_Name =>
            declare
               Lib : Iir_Library_Declaration;
            begin
               Lib := Get_Library (Get_Identifier (Get_Prefix (Unit)));
               return Find_Primary_Unit (Lib, Get_Suffix_Identifier (Unit));
            end;
         when Iir_Kind_Entity_Aspect_Entity =>
            declare
               Prim : Iir_Design_Unit;
            begin
               Prim := Find_Design_Unit (Get_Entity (Unit));
               if Prim = Null_Iir then
                  return Null_Iir;
               end if;
               return Find_Secondary_Unit
                 (Prim, Get_Identifier (Get_Architecture (Unit)));
            end;
         when others =>
            Error_Kind ("find_design_unit", Unit);
      end case;
   end Find_Design_Unit;

   function Is_Obsolete (Design_Unit : Iir_Design_Unit; Loc : Iir)
     return Boolean
   is
      procedure Error_Obsolete (Msg : String) is
      begin
         if not Flags.Flag_Elaborate_With_Outdated then
            Error_Msg_Sem (Msg, Loc);
         end if;
      end Error_Obsolete;

      List : Iir_List;
      El : Iir;
      Unit : Iir_Design_Unit;
      U_Ts : Time_Stamp_Id;
      Du_Ts : Time_Stamp_Id;
   begin
      if Get_Date (Design_Unit) = Date_Obsolete then
         Error_Obsolete (Disp_Node (Design_Unit) & " is obsolete");
         return True;
      end if;
      List := Get_Dependence_List (Design_Unit);
      if List = Null_Iir_List then
         return False;
      end if;
      Du_Ts := Get_Analysis_Time_Stamp (Get_Design_File (Design_Unit));
      for I in Natural loop
         El := Get_Nth_Element (List, I);
         exit when El = Null_Iir;
         Unit := Find_Design_Unit (El);
         if Unit /= Null_Iir then
            U_Ts := Get_Analysis_Time_Stamp (Get_Design_File (Unit));
            if Files_Map.Is_Gt (U_Ts, Du_Ts) then
               Error_Obsolete
                 (Disp_Node (Design_Unit) & " is obsoleted by " &
                  Disp_Node (Unit));
               return True;
            elsif Is_Obsolete (Unit, Loc) then
               Error_Obsolete
                 (Disp_Node (Design_Unit) & " depends on obsolete unit");
               return True;
            end if;
         end if;
      end loop;
      return False;
   end Is_Obsolete;

   -- Load, parse, semantize, back-end a design_unit if necessary.
   procedure Load_Design_Unit (Design_Unit: Iir_Design_Unit; Loc : Iir)
   is
      use Scan;
      Line, Off: Natural;
      Pos: Source_Ptr;
      Res: Iir;
      Library : Iir_Library_Declaration;
      Design_File : Iir_Design_File;
      Fe : Source_File_Entry;
   begin
      if Get_Date (Design_Unit) in Date_Valid
        and then Is_Obsolete (Design_Unit, Loc)
      then
         --  Messages are displayed in the IS_OBSOLETE function.
         null;
      end if;

      -- If the primary unit is already loaded and semantized, return.
      if Get_Loaded_Flag (Design_Unit) then
         case Get_Date (Design_Unit) is
            when Date_Valid
              | Date_Analyzing
              | Date_Analyzed =>
               return;
            when Date_Parsed =>
               null;
            when Date_Obsolete =>
               return;
            when others =>
               raise Internal_Error;
         end case;
      else
         -- Load the unit.
         Design_File := Get_Design_File (Design_Unit);
         Library := Get_Parent (Design_File);
         Fe := Files_Map.Load_Source_File
           (Get_Design_File_Directory (Design_File),
            Get_Design_File_Filename (Design_File));
         if Fe = No_Source_File_Entry then
            Error_Msg
              ("cannot load " & Disp_Node (Get_Library_Unit (Design_Unit)));
            raise Compilation_Error;
         end if;
         Set_File (Fe);

         if not Files_Map.Is_Eq
           (Files_Map.Get_File_Time_Stamp (Get_Current_Source_File),
            Get_File_Time_Stamp (Design_File))
         then
            Error_Msg_Sem
              ("file " & Image (Get_Design_File_Filename (Design_File))
               & " has changed and must be reanalysed", Loc);
            raise Compilation_Error;
         elsif Get_Date (Design_Unit) = Date_Obsolete then
            Error_Msg_Sem
              (''' & Disp_Node (Get_Library_Unit (Design_Unit))
               & "' is not anymore in the file",
               Design_Unit);
            raise Compilation_Error;
         end if;
         Get_Pos_Line_Off (Design_Unit, Pos, Line, Off);
         Files_Map.File_Add_Line_Number (Get_Current_Source_File, Line, Pos);
         Set_Current_Position (Pos + Source_Ptr (Off));
         Set_Loaded_Flag (Design_Unit, True);
         Res := Parse.Parse_Design_Unit;
         Close_File;
         if Res = Null_Iir then
            raise Compilation_Error;
         end if;
         if Get_Date (Design_Unit) not in Date_Valid then
            Set_Date (Design_Unit, Get_Date (Res));
         end if;
         --  FIXME: check the library unit read is the one expected.
         --  Copy node.
         Iirs_Utils.Free_Recursive (Get_Library_Unit (Design_Unit));
         Set_Library_Unit (Design_Unit, Get_Library_Unit (Res));
         Set_Design_Unit (Get_Library_Unit (Res), Design_Unit);
         Set_Parent (Get_Library_Unit (Res), Design_Unit);
         Set_Context_Items (Design_Unit, Get_Context_Items (Res));
         Location_Copy (Design_Unit, Res);
         Free_Dependence_List (Design_Unit);
         Set_Dependence_List (Design_Unit, Get_Dependence_List (Res));
         Set_Dependence_List (Res, Null_Iir_List);
         Free_Iir (Res);
      end if;

      --  Analyze it.
      Sem_Scopes.Push_Interpretations;
      Back_End.Finish_Compilation (Design_Unit);
      Sem_Scopes.Pop_Interpretations;
   end Load_Design_Unit;

   procedure Load_Design_Unit_And_Dependences
     (Design_Unit: Iir_Design_Unit; Loc : Iir)
   is
      List : Iir_List;
      El : Iir;
      Unit : Iir;
   begin
      --  First dependences.
      List := Get_Dependence_List (Design_Unit);
      if List /= Null_Iir_List then
         for I in Natural loop
            El := Get_Nth_Element (List, I);
            exit when El = Null_Iir;
            Unit := Find_Design_Unit (El);
            if Unit = Null_Iir then
               Error_Msg_Sem (Disp_Node (El) & " was never analyzed", Loc);
               return;
            end if;
            case Get_Kind (Unit) is
               when Iir_Kind_Design_Unit =>
                  Load_Design_Unit (Unit, Loc);
               when others =>
                  Error_Kind ("load_design_unit_and_dependences", Unit);
            end case;
         end loop;
      end if;

      --  Then design unit.
      Load_Design_Unit (Design_Unit, Loc);
   end Load_Design_Unit_And_Dependences;

   --  Return the declaration of primary unit NAME of LIBRARY.
   function Find_Primary_Unit
     (Library: Iir_Library_Declaration; Name: Name_Id)
      return Iir_Design_Unit
   is
      Design_File: Iir_Design_File;
      Design_Unit: Iir_Design_Unit;
      Library_Unit: Iir;
   begin
      Design_File := Get_Design_File_Chain (Library);
      while Design_File /= Null_Iir loop
         Design_Unit := Get_Design_Unit_Chain (Design_File);
         while Design_Unit /= Null_Iir loop
            Library_Unit := Get_Library_Unit (Design_Unit);

            case Get_Kind (Library_Unit) is
               when Iir_Kind_Entity_Declaration
                 | Iir_Kind_Package_Declaration
                 | Iir_Kind_Configuration_Declaration =>
                  if Name = Get_Identifier (Library_Unit) then
                     return Design_Unit;
                  end if;
               when others =>
                  null;
            end case;
            Design_Unit := Get_Chain (Design_Unit);
         end loop;
         Design_File := Get_Chain (Design_File);
      end loop;

      -- The primary unit is not in the library, return null.
      return Null_Iir;
   end Find_Primary_Unit;

   function Load_Primary_Unit
     (Library: Iir_Library_Declaration; Name: Name_Id; Loc : Iir)
      return Iir_Design_Unit
   is
      Design_Unit: Iir_Design_Unit;
   begin
      Design_Unit := Find_Primary_Unit (Library, Name);
      if Design_Unit /= Null_Iir then
         Load_Design_Unit (Design_Unit, Loc);
      end if;
      return Design_Unit;
   end Load_Primary_Unit;

   -- Return the declaration of secondary unit NAME for PRIMARY, or null if
   -- not found.
   function Find_Secondary_Unit (Primary: Iir_Design_Unit; Name: Name_Id)
      return Iir_Design_Unit
   is
      Design_File: Iir_Design_File;
      Design_Unit: Iir_Design_Unit;
      Library_Unit: Iir;
      Primary_Ident: Name_Id;
      Ident: Name_Id;
   begin
      Design_File :=
        Get_Design_File_Chain (Get_Parent (Get_Design_File (Primary)));
      Primary_Ident := Get_Identifier (Get_Library_Unit (Primary));
      while Design_File /= Null_Iir loop
         Design_Unit := Get_Design_Unit_Chain (Design_File);
         while Design_Unit /= Null_Iir loop
            Library_Unit := Get_Library_Unit (Design_Unit);

            -- Set design_unit to null iff this is not the correct
            -- design unit.
            case Get_Kind (Library_Unit) is
               when Iir_Kind_Architecture_Declaration =>
                  -- The entity field can be either an identifier (if the
                  -- library unit was not loaded) or an access to the entity
                  -- unit.
                  Ident := Get_Identifier (Get_Entity (Library_Unit));
                  if Ident = Primary_Ident
                    and then Get_Identifier (Library_Unit) = Name
                  then
                     return Design_Unit;
                  end if;
               when Iir_Kind_Package_Body =>
                  if Get_Identifier (Library_Unit) = Primary_Ident then
                     return Design_Unit;
                  end if;
               when others =>
                  null;
            end case;
            Design_Unit := Get_Chain (Design_Unit);
         end loop;
         Design_File := Get_Chain (Design_File);
      end loop;

      -- The architecture is not in the library, return null.
      return Null_Iir;
   end Find_Secondary_Unit;

   -- Load an secondary unit and analyse it.
   function Load_Secondary_Unit
     (Primary: Iir_Design_Unit; Name: Name_Id; Loc : Iir)
      return Iir_Design_Unit
   is
      Design_Unit: Iir_Design_Unit;
   begin
      Design_Unit := Find_Secondary_Unit (Primary, Name);
      if Design_Unit /= Null_Iir then
         Load_Design_Unit (Design_Unit, Loc);
      end if;
      return Design_Unit;
   end Load_Secondary_Unit;

end Libraries;
