with GNAT.Table;
with GNAT.Heap_Sort_A;
with Flags;
with Std_Package;
with Errorout; use Errorout;

package body Xrefs is
   type Xref_Type is record
      Loc : Location_Type;
      Ref : Iir;
      Kind : Xref_Kind;
   end record;

   package Xref_Table is new GNAT.Table
     (Table_Index_Type => Natural,
      Table_Component_Type => Xref_Type,
      Table_Low_Bound => 0,
      Table_Initial => 128,
      Table_Increment => 100);

   procedure Init is
   begin
      Xref_Table.Set_Last (Bad_Xref);
   end Init;

   procedure Add_Xref (Loc : Location_Type; Ref : Iir; Kind : Xref_Kind) is
   begin
      Xref_Table.Append (Xref_Type'(Loc => Loc,
                                    Ref => Ref,
                                    Kind => Kind));
   end Add_Xref;

   procedure Xref_Decl (Decl : Iir) is
   begin
      if Flags.Flag_Xref then
         Add_Xref (Get_Location (Decl), Decl, Xref_Decl);
      end if;
   end Xref_Decl;

   procedure Xref_Ref (Name : Iir; Decl : Iir) is
   begin
      if Flags.Flag_Xref then
         Add_Xref (Get_Location (Name), Decl, Xref_Ref);
      end if;
   end Xref_Ref;

   procedure Xref_Body (Bod : Iir; Spec : Iir) is
   begin
      if Flags.Flag_Xref then
         Add_Xref (Get_Location (Bod), Spec, Xref_Body);
      end if;
   end Xref_Body;

   procedure Xref_End (Loc : Location_Type; Decl : Iir) is
   begin
      if Flags.Flag_Xref then
         Add_Xref (Loc, Decl, Xref_End);
      end if;
   end Xref_End;

   procedure Xref_Name_1 (Name : Iir)
   is
      Res : Iir;
   begin
      case Get_Kind (Name) is
         when Iir_Kind_Simple_Name
           | Iir_Kind_Selected_Name
           | Iir_Kind_Operator_Symbol =>
            Res := Get_Named_Entity (Name);
            if Res = Std_Package.Error_Mark then
               return;
            end if;
            Add_Xref (Get_Location (Name), Res, Xref_Ref);
         when Iir_Kind_Parenthesis_Name
           | Iir_Kind_Selected_By_All_Name
           | Iir_Kind_Slice_Name =>
            null;
         when Iir_Kind_Attribute_Name =>
            --  FIXME: user defined attributes.
            null;
         when others =>
            Error_Kind ("xref_name_1", Name);
      end case;
      case Get_Kind (Name) is
         when Iir_Kind_Simple_Name
           | Iir_Kind_Operator_Symbol =>
            null;
         when Iir_Kind_Selected_Name
           | Iir_Kind_Parenthesis_Name
           | Iir_Kind_Attribute_Name
           | Iir_Kind_Slice_Name
           | Iir_Kind_Selected_By_All_Name =>
            Xref_Name_1 (Get_Prefix (Name));
         when others =>
            Error_Kind ("xref_name_1", Name);
      end case;
   end Xref_Name_1;

   procedure Xref_Name (Name : Iir) is
   begin
      if Flags.Flag_Xref and Name /= Null_Iir then
         Xref_Name_1 (Name);
      end if;
   end Xref_Name;

   procedure Move (From : Natural; To : Natural)
   is
      Tmp : Xref_Type;
   begin
      Tmp := Xref_Table.Table (To);
      Xref_Table.Table (To) := Xref_Table.Table (From);
      Xref_Table.Table (From) := Tmp;
   end Move;

   function Lt (Op1, Op2 : Natural) return Boolean is
   begin
      return Xref_Table.Table (Op1).Loc < Xref_Table.Table (Op2).Loc;
   end Lt;

   procedure Sort is
   begin
      GNAT.Heap_Sort_A.Sort (Xref_Table.Last, Move'Access, Lt'Access);
   end Sort;

   function Find (Loc : Location_Type) return Xref
   is
      Low : Xref;
      High : Xref;
      Mid : Xref;
      Mid_Loc : Location_Type;
   begin
      Low := First_Xref;
      High := Xref_Table.Last;
      loop
         Mid := (Low + High + 1) / 2;
         Mid_Loc := Xref_Table.Table (Mid).Loc;
         if Loc = Mid_Loc then
            return Mid;
         end if;
         if Mid = Low then
            return Bad_Xref;
         end if;
         if Loc > Mid_Loc then
            Low := Mid + 1;
         else
            High := Mid - 1;
         end if;
      end loop;
   end Find;

   function Get_Last_Xref return Xref is
   begin
      return Xref_Table.Last;
   end Get_Last_Xref;

   function Get_Xref_Location (N : Xref) return Location_Type is
   begin
      return Xref_Table.Table (N).Loc;
   end Get_Xref_Location;

   function Get_Xref_Kind (N : Xref) return Xref_Kind is
   begin
      return Xref_Table.Table (N).Kind;
   end Get_Xref_Kind;

   function Get_Xref_Node (N : Xref) return Iir is
   begin
      return Xref_Table.Table (N).Ref;
   end Get_Xref_Node;

end Xrefs;

