// MM1UTIL.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1util.h"
#include "mm1mdl.h"

#include <fstream>
#include <iomanip>
#include <algorithm>
using namespace std;

/*################################################################################################*/

mm1_cr::mm1_cr(void)
{
	atmr = NULL;
	bndr = NULL;
}

mm1_cr::mm1_cr(mm1_atom * p1, mm1_bond * p2)
{
	atmr = p1;
	bndr = p2;
}

mm1_cr::~mm1_cr(void)
{
}

bool mm1_cr::operator==(const mm1_cr & p1) const
{
	return (bndr == p1.bndr);
}

/*################################################################################################*/

mm1_atom::mm1_atom(void)
{
}

mm1_atom::mm1_atom(element p1, fGL * p2, i32u p3)
{
	el = p1; atmtp = NOT_DEFINED;
	
	fGL_a3 data = { 0.0, 0.0, 0.0 };
	
	if (p2 != NULL)
	{
		data[0] = p2[0];
		data[1] = p2[1];
		data[2] = p2[2];
	}
	
	for (i32u n1 = 0;n1 < p3;n1++) crd_vector.push_back(data);
	
	topology = NOT_DEFINED; res_id = NOT_DEFINED;
	for (i32u i1 = 0;i1 < 4;i1++) id[i1] = NOT_DEFINED;
	
	index = NOT_DEFINED; selected = false; charge = 0.0;
}

mm1_atom::mm1_atom(const mm1_atom & p1)
{
	el = p1.el; atmtp = p1.atmtp;
	
	cr_list = p1.cr_list;
	crd_vector = p1.crd_vector;
	
	topology = p1.topology; res_id = p1.res_id;
	for (i32u n1 = 0;n1 < 4;n1++) id[n1] = p1.id[n1];
	
	index = p1.index;
	selected = p1.selected;
	charge = p1.charge;
}

mm1_atom::~mm1_atom(void)
{
}

// mm1_atoms will be sorted according to their molecule, chain and residue numbers.
// sorting the container will then arrange molecules/chains/residues in continuous blocks.

bool mm1_atom::operator<(const mm1_atom & p1) const
{
	for (i32s n1 = 0;n1 < 3;n1++)
	{
		if (id[n1] != p1.id[n1]) return (id[n1] < p1.id[n1]);
	}
	
	return false;
}

// for mm1_atoms, equality is tested using pointers -> you can find the
// iterator (using STL's find-function) if the pointer is known.

bool mm1_atom::operator==(const mm1_atom & p1) const
{
	return (this == (& p1));
}

/*################################################################################################*/

mm1_bond::mm1_bond(void)
{
	atmr[0] = atmr[1] = NULL;
	for (i32s n1 = 0;n1 < NFLAGS;n1++) flags.push_back(false);
}

mm1_bond::mm1_bond(mm1_atom * p1, mm1_atom * p2, bondtype p3)
{
	atmr[0] = p1; atmr[1] = p2; bt = p3;
	for (i32s n1 = 0;n1 < NFLAGS;n1++) flags.push_back(false);
}

mm1_bond::mm1_bond(const mm1_bond & p1)
{
	atmr[0] = p1.atmr[0]; atmr[1] = p1.atmr[1]; bt = p1.bt;
	for (i32s n1 = 0;n1 < NFLAGS;n1++) flags.push_back(p1.flags[n1]);
}

mm1_bond::~mm1_bond(void)
{
}

// for bonds, equality is tested using atom pointers. if you need to find a certain bond,
// just make a temporary mm1_bond and use that to find the original one (using STL's find-function).

bool mm1_bond::operator==(const mm1_bond & p1) const
{
	if (atmr[0] == p1.atmr[0] && atmr[1] == p1.atmr[1]) return true;
	if (atmr[0] == p1.atmr[1] && atmr[1] == p1.atmr[0]) return true;
	
	return false;
}

/*################################################################################################*/

mm1_chn_info::mm1_chn_info(void)
{
	description = NULL;
	
	type = not_defined;
	length = NOT_DEFINED;
	
	sequence = NULL;
	state = NULL;
}

mm1_chn_info::mm1_chn_info(chn_type p1, i32s p2, bool p3)
{
	description = NULL;
	
	type = p1;
	length = p2;
	
	sequence = new char[length];
	
	if (p3) state = new char[length];
	else state = NULL;
}

mm1_chn_info::mm1_chn_info(const mm1_chn_info & p1)
{
	if (p1.description != NULL)
	{
		description = new char[strlen(p1.description) + 1];
		strcpy(description, p1.description);
	}
	else description = NULL;
	
	type = p1.type;
	length = p1.length;
	
	if (p1.sequence != NULL)
	{
		sequence = new char[length];
		for (i32s n1 = 0;n1 < length;n1++) sequence[n1] = p1.sequence[n1];
	}
	
	if (p1.state != NULL)
	{
		state = new char[length];
		for (i32s n1 = 0;n1 < length;n1++) state[n1] = p1.state[n1];
	}
}

mm1_chn_info::~mm1_chn_info(void)
{
	if (description != NULL) delete[] description;
	
	if (sequence != NULL) delete[] sequence;
	if (state != NULL) delete[] state;
}

/*################################################################################################*/

mm1_typerule::mm1_typerule(void)
{
	first = NOT_DEFINED;
}

mm1_typerule::mm1_typerule(istream * istr, ostream * ostr)
{
	first = ReadSubRule(istr, ostr);
}

mm1_typerule::mm1_typerule(const mm1_typerule & p1)
{
	first = p1.first;
	sr_vector = p1.sr_vector;
	
	ring_vector.resize(p1.ring_vector.size());
	for (i32u n1 = 0;n1 < ring_vector.size();n1++)
	{
		ring_vector[n1] = new signed char[strlen((const char *) p1.ring_vector[n1]) + 1];
		strcpy((char *) ring_vector[n1], (const char *) p1.ring_vector[n1]);
	}
}

mm1_typerule::~mm1_typerule(void)
{
	for (i32u n1 = 0;n1 < ring_vector.size();n1++) delete[] ring_vector[n1];
}

bool mm1_typerule::Check(mm1_mdl * mdl, mm1_atom * atom, i32s flag)
{
	if (!sr_vector.size()) return true;		// an empty rule always fits...
	return CheckRules(mdl, atom, flag, first);
}

ostream & operator<<(ostream & p1, const mm1_typerule & p2)
{
	if (p2.sr_vector.size() != 0) p2.PrintSubRules(p1, p2.first);
	return p1;
}

i32s mm1_typerule::ReadSubRule(istream * istr, ostream * ostr)
{
	vector<i32s> index_vector;
	if (istr->peek() == '(') istr->get();
	else return NOT_DEFINED;
	
	while (true)
	{
		if (istr->peek() == ')') break;
		else if (istr->peek() == ',') istr->get();
		else
		{
			mm1_tr_subrule newsr;
			newsr.data = NOT_DEFINED;
			newsr.next = NOT_DEFINED;
			newsr.sub = NOT_DEFINED;
			
			if (istr->peek() == 'b')
			{
				istr->get();
				switch (istr->peek())
				{
					case '1': newsr.type = mm1_tr_subrule::NumAllBonds; break;
					case '2': newsr.type = mm1_tr_subrule::NumHABonds; break;
					default: return NOT_DEFINED;
				}
				
				istr->getline(buffer, sizeof(buffer), '=');
				(* istr) >> newsr.data;
			}
			else if (istr->peek() == 'r')
			{
				newsr.type = mm1_tr_subrule::RingSize;
				istr->getline(buffer, sizeof(buffer), '=');
				(* istr) >> newsr.data;
			}
			else if (istr->peek() == '[')
			{
				newsr.type = mm1_tr_subrule::Ring; istr->get();
				for (i32u n1 = 0;n1 < sizeof(buffer);n1++) buffer[n1] = 0;
				istr->getline(buffer, sizeof(buffer), ']');
				
				i32s tmp1[2] = { strlen(buffer), 0 };
				for (i32s n1 = 0;n1 < tmp1[0];n1++)
				{
					if (buffer[n1] == '-') tmp1[1]++;
					if (buffer[n1] == '~') tmp1[1]++;
					if (buffer[n1] == '=') tmp1[1]++;
					if (buffer[n1] == '#') tmp1[1]++;
					if (buffer[n1] == '?') tmp1[1]++;	// wildcard
				} tmp1[1] += (tmp1[1] - 1);
				
				signed char * newring = new signed char[tmp1[1] + 1];
				newring[tmp1[1]] = 0;
				
				tmp1[0] = 0; char tmp2[3] = { 0, 0, 0 };
				for (i32s n1 = 0;n1 < tmp1[1];n1++)
				{
					if (!(n1 % 2)) newring[n1] = buffer[tmp1[0]++];
					else
					{
						tmp2[0] = buffer[tmp1[0]++];
						if (buffer[tmp1[0]] < 'a' || buffer[tmp1[0]] > 'z') tmp2[1] = 0;
						else tmp2[1] = buffer[tmp1[0]++]; element el = element(tmp2);
						
						bool problem = (el.GetAtomicNumber() == NOT_DEFINED && tmp2[0] != '*');
						if (problem && ostr != NULL)
						{
							(* ostr) << "WARNING !!! Unknown element " << tmp2;
							(* ostr) << " -> using wildcard instead..." << endl;
						}
						
						newring[n1] = (char) el.GetAtomicNumber();
					}
				}
				
				newsr.data = ring_vector.size();
				ring_vector.push_back(newring);
			}
			else
			{
				newsr.type = mm1_tr_subrule::BondedTo;
				char tmp1 = (char) istr->get(); newsr.bt = bondtype(tmp1);
				
				bool problem1 = (newsr.bt.GetValue() == NOT_DEFINED && tmp1 != '?');
				if (problem1 && ostr != NULL)
				{
					(* ostr) << "WARNING !!! Unknown bondtype " << tmp1;
					(* ostr) << " -> using wildcard instead..." << endl;
				}
				
				char tmp2[3] = { 0, 0, 0 }; tmp2[0] = (char) istr->get();
				if (istr->peek() >= 'a' && istr->peek() <= 'z') tmp2[1] = (char) istr->get();
				newsr.el = element(tmp2);
				
				bool problem2 = (newsr.el.GetAtomicNumber() == NOT_DEFINED && tmp2[0] != '*');
				if (problem2 && ostr != NULL)
				{
					(* ostr) << "WARNING !!! Unknown element " << tmp2;
					(* ostr) << " -> using wildcard instead..." << endl;
				}
				
				newsr.sub = ReadSubRule(istr, ostr);
			}
			
			index_vector.push_back(sr_vector.size());
			sr_vector.push_back(newsr);
		}
	}
	
	istr->get(); if (!index_vector.size()) return NOT_DEFINED;
	for (i32s n1 = 0;n1 < ((i32s) index_vector.size()) - 1;n1++) sr_vector[index_vector[n1]].next = index_vector[n1 + 1];
	return index_vector[0];
}

// here we must use flags "flag" and "flag + 1" because the general typerule testing and
// testing for ring rules must be independent of each other!!!!!

bool mm1_typerule::CheckRules(mm1_mdl * mdl, mm1_atom * atom, i32s flag, i32s rule)
{
	i32s n1; iter_mm1cl it1;
	switch (sr_vector[rule].type)
	{
		case mm1_tr_subrule::BondedTo:
		for (n1 = false, it1 = atom->cr_list.begin();!n1 && it1 != atom->cr_list.end();it1++)
		{
			if ((* it1).bndr->flags[flag]) continue;
			if (sr_vector[rule].el.GetAtomicNumber() != NOT_DEFINED &&
				sr_vector[rule].el.GetAtomicNumber() != (* it1).atmr->el.GetAtomicNumber()) continue;
			if (sr_vector[rule].bt.GetValue() != NOT_DEFINED &&
				sr_vector[rule].bt.GetValue() != (* it1).bndr->bt.GetValue()) continue;
			
			(* it1).bndr->flags[flag] = true; n1 = true;
			if (n1 && (sr_vector[rule].sub != NOT_DEFINED)) n1 = CheckRules(mdl, (* it1).atmr, flag, sr_vector[rule].sub);
			if (n1 && (sr_vector[rule].next != NOT_DEFINED)) n1 = CheckRules(mdl, atom, flag, sr_vector[rule].next);
			(* it1).bndr->flags[flag] = false;
		}
		return n1;
		
		case mm1_tr_subrule::NumAllBonds:
		return (atom->cr_list.size() == (i32u) sr_vector[rule].data);
		
		case mm1_tr_subrule::NumHABonds:
		n1 = 0;it1 = atom->cr_list.begin();
		while (it1 != atom->cr_list.end())
		{
			bool index = ((* it1).bndr->atmr[0] == atom);
			if ((* it1++).bndr->atmr[index]->el.GetAtomicNumber() != 1) n1++;
		}
		return (n1 == sr_vector[rule].data);
		
		case mm1_tr_subrule::RingSize:
		return mdl->FindRing(atom, atom, NULL, sr_vector[rule].data, flag + 1);
		
		case mm1_tr_subrule::Ring:
		n1 = strlen((const char *) ring_vector[sr_vector[rule].data]); n1 = (n1 + 1) / 2;
		return mdl->FindRing(atom, atom, ring_vector[sr_vector[rule].data], n1, flag + 1);
		
		default:
		return false;
	}
}

void mm1_typerule::PrintSubRules(ostream & p1, i32s p2) const
{
	switch (sr_vector[p2].type)
	{
		case mm1_tr_subrule::BondedTo:
		if (sr_vector[p2].bt.GetValue() == NOT_DEFINED) p1 << "?";
		else p1 << sr_vector[p2].bt.GetSymbol2();
		if (sr_vector[p2].el.GetAtomicNumber() == NOT_DEFINED) p1 << "*";
		else p1 << sr_vector[p2].el.GetSymbol();
		if (sr_vector[p2].sub == NOT_DEFINED) break;
		p1 << "("; PrintSubRules(p1, sr_vector[p2].sub); p1 << ")";
		break;
		
		case mm1_tr_subrule::NumAllBonds:
		p1 << "b1=" << sr_vector[p2].data;
		break;
		
		case mm1_tr_subrule::NumHABonds:
		p1 << "b2=" << sr_vector[p2].data;
		break;
		
		case mm1_tr_subrule::RingSize:
		p1 << "rs=" << sr_vector[p2].data;
		break;
		
		case mm1_tr_subrule::Ring:
		p1 << "["; PrintRing(p1, ring_vector[sr_vector[p2].data]); p1 << "]";
		break;
	}
	
	if (sr_vector[p2].next == NOT_DEFINED) return;
	p1 << ","; PrintSubRules(p1, sr_vector[p2].next);
}

void mm1_typerule::PrintRing(ostream & p1, signed char * p2) const
{
	for (i32u n1 = 0;n1 < strlen((const char *) p2);n1++)
	{
		if (!(n1 % 2)) p1 << (char) p2[n1];
		else
		{
			i32s n2 = p2[n1];
			if (n2 == NOT_DEFINED) p1 << "*";
			else
			{
				element el = element(n2);
				p1 << el.GetSymbol();
			}
		}
	}
}

/*################################################################################################*/

mm1_sb_atm::mm1_sb_atm(void)
{
	typerule = NULL;
}

mm1_sb_atm::mm1_sb_atm(const mm1_sb_atm & p1)
{
	el = p1.el; ic2 = p1.ic2;
	bt[0] = p1.bt[0]; bt[1] = p1.bt[1];
	id[0] = p1.id[0]; id[1] = p1.id[1];
	
	if (p1.typerule != NULL) typerule = new mm1_typerule(* p1.typerule);
	else typerule = NULL;
	
	for (i32s n1 = 0;n1 < 3;n1++)
	{
		prev[n1] = p1.prev[n1];
		ic1[n1] = p1.ic1[n1];
	}
}

mm1_sb_atm::~mm1_sb_atm(void)
{
	if (typerule != NULL) delete typerule;
}

istream & operator>>(istream & p1, mm1_sb_atm & p2)
{
	char buffer[256];
	while (p1.get() != 'M');
	p1 >> p2.id[0]; while (p1.get() != ':');
	p1 >> buffer; p2.el = element(buffer);
	p1 >> p2.prev[0] >> p2.prev[1] >> p2.prev[2];
	p1 >> p2.ic1[0] >> p2.ic1[1] >> p2.ic2 >> p2.ic1[2];
	
	p2.ic1[1] = M_PI * p2.ic1[1] / 180.0;
	p2.ic1[2] = M_PI * p2.ic1[2] / 180.0;
	
	p1 >> buffer;
	p2.bt[0] = bondtype(buffer[0]);
	
	p1 >> p2.id[1] >> buffer;
	p2.bt[1] = bondtype(buffer[0]);
	
	return p1;
}

ostream & operator<<(ostream & p1, mm1_sb_atm & p2)
{
	p1 << hex << "0x" << setw(4) << setfill('0') << p2.id << dec;
	p1 << " " << p2.el.GetSymbol() << " " << p2.bt[0].GetSymbol1();
	if (p2.typerule != NULL) p1 << " " << (* p2.typerule); p1 << " ";
	
	p1 << p2.ic1[0] << " " << p2.ic1[1] << " " << p2.ic2 << " " << p2.ic1[2] << " ";
	p1 << hex << "0x" << setw(4) << setfill('0') << p2.prev[0] << dec << " ";
	p1 << hex << "0x" << setw(4) << setfill('0') << p2.prev[1] << dec << " ";
	p1 << hex << "0x" << setw(4) << setfill('0') << p2.prev[2] << dec;
	
	return p1;
}

/*################################################################################################*/

mm1_sb_bnd::mm1_sb_bnd(void)
{
}

mm1_sb_bnd::~mm1_sb_bnd(void)
{
}

istream & operator>>(istream & p1, mm1_sb_bnd & p2)
{
	char buffer[256];
	while (p1.get() != ':');
	p1 >> p2.atm[0] >> p2.atm[1];
	p1 >> buffer; p2.bt = bondtype(buffer[0]);
	return p1;
}

/*################################################################################################*/

mm1_sb_res::mm1_sb_res(void)
{
	description = NULL;
}

mm1_sb_res::mm1_sb_res(const mm1_sb_res & p1)
{
	id = p1.id;
	symbol = p1.symbol;
	
	if (p1.description != NULL)
	{
		description = new char[strlen(p1.description) + 1];
		strcpy(description, p1.description);
	}
	else description = NULL;
	
	atm_vector = p1.atm_vector;
	bnd_vector = p1.bnd_vector;
}

mm1_sb_res::~mm1_sb_res(void)
{
	if (description != NULL) delete[] description;
}

void mm1_sb_res::ReadModification(istream & p1)
{
	char buffer[256];
	while (p1.peek() != 'E')
	{
		if (p1.peek() == 'A')
		{
			mm1_sb_atm newatm; p1 >> newatm;
			p1.getline(buffer, sizeof(buffer));
			atm_vector.push_back(newatm);
			continue;
		}
		
		if (p1.peek() == 'B')
		{
			mm1_sb_bnd newbnd; p1 >> newbnd;
			p1.getline(buffer, sizeof(buffer));
			bnd_vector.push_back(newbnd);
			continue;
		}
		
		p1.getline(buffer, sizeof(buffer));
	}
	
	p1.getline(buffer, sizeof(buffer));
}

istream & operator>>(istream & p1, mm1_sb_res & p2)
{
	char buffer[256];
	while (p1.get() != 'S'); p1 >> p2.id;
	while (p1.get() != ':'); p1 >> p2.symbol;
	
	while (p1.get() != '"');
	p1.getline(buffer, sizeof(buffer), '"');
	
	p2.description = new char[strlen(buffer) + 1];
	strcpy(p2.description, buffer);
	
	p1.getline(buffer, sizeof(buffer));
	
	while (p1.peek() != 'E')
	{
		if (p1.peek() == 'A')
		{
			mm1_sb_atm newatm; p1 >> newatm;
			p1.getline(buffer, sizeof(buffer));
			p2.atm_vector.push_back(newatm);
			continue;
		}
		
		if (p1.peek() == 'B')
		{
			mm1_sb_bnd newbnd; p1 >> newbnd;
			p1.getline(buffer, sizeof(buffer));
			p2.bnd_vector.push_back(newbnd);
			continue;
		}
		
		p1.getline(buffer, sizeof(buffer));
	}
	
	p1.getline(buffer, sizeof(buffer));
	return p1;
}

/*################################################################################################*/

// here we make some direct output to cout -> not good in TARGET3 ?!?!?!

mm1_sequencebuilder::mm1_sequencebuilder(mm1_chn_info::chn_type ct, char * filename)
{
	type = ct;
	
	ifstream file;
	file.unsetf(ios::dec | ios::oct | ios::hex);
	
	model_simple::OpenParameterFile(file, ios::in, filename);
	
	// read information about the main chain...
	
	while (file.peek() != 'M') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		mm1_sb_atm newatm; file >> newatm;
		while (file.peek() != '(') file.get();
		newatm.typerule = new mm1_typerule(& file, & cout);
		file.getline(buffer, sizeof(buffer));
		main_vector.push_back(newatm);
	}
	
	// read descriptions about the chain initiation...
	
	while (file.peek() != 'H') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		mm1_typerule newrule = mm1_typerule(& file, & cout);
		file.getline(buffer, sizeof(buffer));
		head_vector.push_back(newrule);
	}
	
	// read descriptions about the chain termination...
	
	while (file.peek() != 'T') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		mm1_typerule newrule = mm1_typerule(& file, & cout);
		file.getline(buffer, sizeof(buffer));
		tail_vector.push_back(newrule);
	}
	
	// read the default modifications for residues...
	
	while (file.peek() != 'B') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer)); mod[0] = new mm1_sb_res; mod[0]->ReadModification(file);
	
	while (file.peek() != 'H') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer)); mod[1] = new mm1_sb_res; mod[1]->ReadModification(file);
	
	while (file.peek() != 'T') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer)); mod[2] = new mm1_sb_res; mod[2]->ReadModification(file);
	
	// read descriptions for the residues...
	
	while (file.peek() != 'E')
	{
		if (file.peek() == 'R')
		{
			mm1_sb_res newres; file >> newres;
			residue_vector.push_back(newres);
		}
		else file.getline(buffer, sizeof(buffer));
	}
	
	// ready!!!
	
	file.close();
}

mm1_sequencebuilder::~mm1_sequencebuilder(void)
{
	delete mod[0];
	delete mod[1];
	delete mod[2];
}

// here we use only the first coordinate set, no matter how many there are...

void mm1_sequencebuilder::Build(mm1_mdl * mdl, char * sequence, f64 * tor)
{
	fGL b1crd[3] = { 1.0, 1.0, 0.0 };	// these should be const...
	fGL b2crd[3] = { 1.0, 0.0, 0.0 };
	fGL b3crd[3] = { 0.0, 0.0, 0.0 };
	
	mm1_atom base[3] =
	{
		mm1_atom(element(), b1crd, mdl->cs_vector.size()),	// these should be const...
		mm1_atom(element(), b2crd, mdl->cs_vector.size()),
		mm1_atom(element(), b3crd, mdl->cs_vector.size())
	};
	
	id_vector.resize(0); ref_vector.resize(0);
	id_vector.push_back(main_vector[0].prev[0]); ref_vector.push_back(& base[0]);
	id_vector.push_back(main_vector[0].prev[1]); ref_vector.push_back(& base[1]);
	id_vector.push_back(main_vector[0].prev[2]); ref_vector.push_back(& base[2]);
	
	for (i32u n1 = 0;n1 < strlen(sequence);n1++)
	{
		for (i32u n2 = 0;n2 < main_vector.size();n2++)
		{
			mm1_atom * prev[3];
			f64 ic[3]; fGL crd[3];
			for (i32s n3 = 0;n3 < 3;n3++)
			{
				i32s n4 = 0; while (id_vector[n4] != main_vector[n2].prev[n3]) n4++;
				prev[n3] = ref_vector[n4]; ic[n3] = main_vector[n2].ic1[n3];
			}
			
			if (main_vector[n2].ic2 > -1) ic[2] += tor[main_vector[n2].ic2];
			Convert(prev, ic, crd);
			
			id_vector.push_back(main_vector[n2].id[0]);
			mm1_atom newatom(main_vector[n2].el, crd, mdl->cs_vector.size());
			mdl->AddAtom(newatom);
			
			ref_vector.push_back(& mdl->atom_list.back());
			
			if (n1 || n2)
			{
				mm1_bond newbond(prev[0], & mdl->atom_list.back(), main_vector[n2].bt[0]);
				mdl->AddBond(newbond);
			}
		}
		
		if (!n1) Build(mdl, mod[1], tor);
		else if (n1 == (strlen(sequence) - 1)) Build(mdl, mod[2], tor);
		else Build(mdl, mod[0], tor);
		
		i32u res = 0;
		while (residue_vector[res].symbol != sequence[n1])
		{
			if (res != residue_vector.size()) res++;
			else break;
		}
		
		if (res != residue_vector.size()) Build(mdl, & residue_vector[res], tor);
		else if (mdl->ostr != NULL) (* mdl->ostr) << "unknown residue " << sequence[n1] << endl;
		
		vector<i32s> tmpv1; atmr_vector tmpv2;
		for (i32u n2 = 0;n2 < id_vector.size();n2++)
		{
			if (id_vector[n2] & 0xFF00) continue;
			tmpv1.push_back(id_vector[n2]); tmpv2.push_back(ref_vector[n2]);
		}
		
		id_vector.resize(0); ref_vector.resize(0);
		for (i32u n2 = 0;n2 < tmpv1.size();n2++)
		{
			id_vector.push_back(tmpv1[n2] + 0xFF00);
			ref_vector.push_back(tmpv2[n2]);
		}
	}
	
	mdl->CenterCRDSet(0);
}

// here we will identify the sequences (if there is any), write down the sequences
// and update the /mol/chn/res-numbering...

// are there problems if we identify amino/nucleic acids separately???

void mm1_sequencebuilder::Identify(mm1_mdl * mdl)
{
	mdl->GatherGroups();
	cout << "nmol = " << mdl->nmol << endl;
	
	// here we will find all possible chains and identify them.
	
	for (i32s n1 = 0;n1 < mdl->nmol;n1++)
	{
		iter_mm1al range[2];
		mdl->GetRange(0, n1, range);
		
		vector<mm1_atom *> nterm;	// nucleic acids??? a misleading name...
		vector<mm1_atom *> cterm;	// nucleic acids??? a misleading name...
		
		for (iter_mm1al it1 = range[0];it1 != range[1];it1++)
		{
			i32s tmp1 = (* it1).el.GetAtomicNumber();
			
			if (tmp1 == main_vector.front().el.GetAtomicNumber())
			{
				for (i32u n2 = 0;n2 < head_vector.size();n2++)
				{
					bool flag = head_vector[n2].Check(mdl, & (* it1), 0);
					if (!flag) continue; nterm.push_back(& (* it1)); break;
				}
			}
			
			if (tmp1 == main_vector.back().el.GetAtomicNumber())
			{
				for (i32u n2 = 0;n2 < tail_vector.size();n2++)
				{
					bool flag = tail_vector[n2].Check(mdl, & (* it1), 0);
					if (!flag) continue; cterm.push_back(& (* it1)); break;
				}
			}
		}
		
		// now we have all possible head/tail atoms. next we have to check
		// all possible paths between them to find all possible main chains...
		
		path_vector.resize(0);
		for (i32u n2 = 0;n2 < nterm.size();n2++)
		{
			for (i32u n3 = 0;n3 < cterm.size();n3++)
			{
				FindPath(mdl, nterm[n2], cterm[n3]);
			}
		}
		
		if (path_vector.size()) cout << path_vector.size() << " chains:" << endl;
		
		for (i32s n2 = 0;n2 < (i32s) path_vector.size();n2++)
		{
			for (i32s n3 = 0;n3 < ((i32s) path_vector[n2].size()) - 1;n3++)
			{
				iter_mm1cl it1 = path_vector[n2][n3]->cr_list.begin();
				while ((* it1).atmr != path_vector[n2][n3 + 1]) it1++;
				(* it1).bndr->flags[0] = true;
			}
			
			vector<char> sequence;
			bool head_blocked = !head_vector[0].Check(mdl, path_vector[n2].front(), 1);
			bool tail_blocked = !tail_vector[0].Check(mdl, path_vector[n2].back(), 1);
			i32s last_residue = path_vector[n2].size() / main_vector.size() - 1;
			
			i32s rcount = 0;
			for (i32s n3 = 0;n3 < (i32s) path_vector[n2].size();n3 += main_vector.size())
			{
				i32s residue = n3 / main_vector.size();		// redundant with rcount?!?!?!
				bool is_first = !head_blocked && !residue;
				bool is_last = !tail_blocked && (residue == last_residue);
				
				// now we start identifying the residues.
				
				// we make a simple template for each of them, compare those to
				// the residue descriptions we have, and pick the closest match.
				// there may not be any missing atoms, and the closest match is
				// the one with largest number of correct atoms identified.
				
				// some problems with the blocked head/tail residues???
				
				i32s tmp1[2] = { NOT_DEFINED, NOT_DEFINED };
				for (i32s n4 = 0;n4 < (i32s) residue_vector.size();n4++)
				{
					vector<mm1_sb_tdata> tdata;
					BuildTemplate(tdata, n4, is_first, is_last);
					
					for (i32s n5 = 0;n5 < (i32s) main_vector.size();n5++)
					{
						tdata[n5].ref = path_vector[n2][n3 + n5];
					}
					
					bool result = CheckTemplate(tdata, 0);
					if (result && (tmp1[0] == NOT_DEFINED || ((i32s) tdata.size()) > tmp1[1]))
					{
						tmp1[0] = n4;
						tmp1[1] = (i32s) tdata.size();
					}
				}
				
				if (tmp1[0] == NOT_DEFINED)
				{
					cout << "WARNING : residue " << rcount << " was of unknown type!!!" << endl;
					
					sequence.push_back('?');				// handle the sequence...
					
					for (i32u n4 = 0;n4 < main_vector.size();n4++)		// handle the numbers...
					{
						path_vector[n2][n3 + n4]->res_id = NOT_DEFINED;		// WHAT HERE???
						
						path_vector[n2][n3 + n4]->id[1] = mdl->ref_civ->size();
						path_vector[n2][n3 + n4]->id[2] = rcount;
					}
				}
				else
				{
					sequence.push_back(residue_vector[tmp1[0]].symbol);
					
					// rebuild the best matching template and use that to
					// set up the atom ID numbers...
					
					vector<mm1_sb_tdata> tdata;
					BuildTemplate(tdata, tmp1[0], is_first, is_last);
					
					for (i32s n4 = 0;n4 < (i32s) main_vector.size();n4++)
					{
						tdata[n4].ref = path_vector[n2][n3 + n4];
					}
					
					CheckTemplate(tdata, 0);
					for (i32s n4 = 0;n4 < (i32s) tdata.size();n4++)
					{
						tdata[n4].ref->res_id = (residue_vector[tmp1[0]].id << 8) + tdata[n4].id[0];
						
						tdata[n4].ref->id[1] = mdl->ref_civ->size();
						tdata[n4].ref->id[2] = rcount;
					}
				}
				
				rcount++;
			}
			
			// next we will write down the sequence...
			
			mm1_chn_info newinfo(type, sequence.size(), false);
			for (i32u n3 = 0;n3 < sequence.size();n3++) newinfo.sequence[n3] = sequence[n3];
			mdl->ref_civ->push_back(newinfo);
			
			// and make some output...
			
			if (mdl->ostr != NULL)
			{
				for (i32u n3 = 0;n3 < sequence.size();n3++) (* mdl->ostr) << sequence[n3];
				(* mdl->ostr) << endl;
			}
			
			// clean the (main-chain???) flags...
			
			for (i32s n3 = 0;n3 < ((i32s) path_vector[n2].size()) - 1;n3++)
			{
				iter_mm1cl it1 = path_vector[n2][n3]->cr_list.begin();
				while ((* it1).atmr != path_vector[n2][n3 + 1]) it1++;
				(* it1).bndr->flags[0] = false;
			}
		}
	}
	
	// finally sort the atom list.
	
	// this might be a foolish arrangement if we have to use first peptide/protein builder to
	// identify those sequences, and after that nucleic acid builder. I have not really tried
	// how this combined system would work...
	
	mdl->atom_list.sort();
}

void mm1_sequencebuilder::Build(mm1_mdl * mdl, mm1_sb_res * res, f64 * tor)
{
	for (i32u n1 = 0;n1 < res->atm_vector.size();n1++)
	{
		mm1_atom * prev[3];
		f64 ic[3]; fGL crd[3];
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			i32s n3 = 0; while (id_vector[n3] != res->atm_vector[n1].prev[n2]) n3++;
			prev[n2] = ref_vector[n3]; ic[n2] = res->atm_vector[n1].ic1[n2];
		}
		
		if (res->atm_vector[n1].ic2 > -1) ic[2] += tor[res->atm_vector[n1].ic2];
		Convert(prev, ic, crd);
		
		id_vector.push_back(res->atm_vector[n1].id[0]);
		mm1_atom newatom(res->atm_vector[n1].el, crd, mdl->cs_vector.size());
		mdl->AddAtom(newatom);
		
		ref_vector.push_back(& mdl->atom_list.back());
		
		mm1_bond newbond(prev[0], & mdl->atom_list.back(), res->atm_vector[n1].bt[0]);
		mdl->AddBond(newbond);
	}
	
	for (i32u n1 = 0;n1 < res->bnd_vector.size();n1++)
	{
		mm1_atom * tmp1[2]; i32s tmp2;
		tmp2 = 0; while (id_vector[tmp2] != res->bnd_vector[n1].atm[0]) tmp2++; tmp1[0] = ref_vector[tmp2];
		tmp2 = 0; while (id_vector[tmp2] != res->bnd_vector[n1].atm[1]) tmp2++; tmp1[1] = ref_vector[tmp2];
		mm1_bond newbond(tmp1[0], tmp1[1], res->bnd_vector[n1].bt);
		mdl->AddBond(newbond);
	}
}

void mm1_sequencebuilder::Convert(mm1_atom * prev[], f64 * ic, fGL * crd)
{
	v3d<fGL> v1 = v3d<fGL>(prev[0]->crd_vector[0].data, prev[1]->crd_vector[0].data);
	
	v3d<fGL> tmp1 = v3d<fGL>(prev[1]->crd_vector[0].data, prev[2]->crd_vector[0].data);
	v3d<fGL> tmp2 = v1 * (tmp1.spr(v1) / v1.spr(v1));
	v3d<fGL> v2 = tmp1 - tmp2;
	
	v3d<fGL> v3 = v2.vpr(v1);
	
	v1 = v1 * (ic[0] * cos(ic[1]) / v1.len());
	v2 = v2 * (ic[0] * sin(ic[1]) * cos(ic[2]) / v2.len());
	v3 = v3 * (ic[0] * sin(ic[1]) * sin(ic[2]) / v3.len());
	
	v3d<fGL> tmp3 = v3d<fGL>(prev[0]->crd_vector[0].data); tmp3 = tmp3 + (v1 + (v2 + v3));
	for (i32s n1 = 0;n1 < 3;n1++) crd[n1] = tmp3.data[n1];
}

// idea is that this path-searching function should be used only inside a molecule.
// this way we are sure to find always at least one path. we collect references to
// atoms for all paths and collect this way all possible paths.

// different flags should be used in type checking and in this path-searching...

void mm1_sequencebuilder::FindPath(mm1_mdl * mdl, mm1_atom * ref1, mm1_atom * ref2)
{
	i32s index = temporary_vector.size() % main_vector.size();
	if (ref1->el.GetAtomicNumber() != main_vector[index].el.GetAtomicNumber()) return;
	if (!main_vector[index].typerule->Check(mdl, ref1, 0)) return;
	
	temporary_vector.push_back(ref1);
	if (ref1 == ref2) path_vector.push_back(temporary_vector);
	else
	{
		iter_mm1cl it1;
		for (it1 = ref1->cr_list.begin();it1 != ref1->cr_list.end();it1++)
		{
			if ((* it1).bndr->flags[1]) continue;
			
			(* it1).bndr->flags[1] = true;
			FindPath(mdl, (* it1).atmr, ref2);
			(* it1).bndr->flags[1] = false;
		}
	}
	
	temporary_vector.pop_back();
}

void mm1_sequencebuilder::BuildTemplate(vector<mm1_sb_tdata> & tdata, i32s res, bool is_first, bool is_last)
{
	BuildTemplate(tdata, main_vector);
	
	if (is_first) BuildTemplate(tdata, mod[1]->atm_vector);
	else if (is_last) BuildTemplate(tdata, mod[2]->atm_vector);
	else BuildTemplate(tdata, mod[0]->atm_vector);
	
	BuildTemplate(tdata, residue_vector[res].atm_vector);
}

void mm1_sequencebuilder::BuildTemplate(vector<mm1_sb_tdata> & tdata, vector<mm1_sb_atm> & adata)
{
	for (i32u n1 = 0;n1 < adata.size();n1++)
	{
		mm1_sb_tdata newdata;
		newdata.id[0] = adata[n1].id[0];
		
		if (adata[n1].prev[0] & 0xFF00) newdata.id[1] = NOT_DEFINED;
		else newdata.id[1] = adata[n1].prev[0];
		
		newdata.el = adata[n1].el;
		newdata.bt = adata[n1].bt[1];
		newdata.ref = NULL;
		
		tdata.push_back(newdata);
	}
}

bool mm1_sequencebuilder::CheckTemplate(vector<mm1_sb_tdata> & tdata, i32s flag)
{
	vector<i32s> tmpv1;
	
	// look for a suitable starting point to start matching in the template.
	// if there is no such point we assume the template fits. however, the template
	// could match since it is too small (for exaple, the template for glycine will
	// match for all residues). therefore we have to test all templates and choose
	// the biggest matching one!!!
	
	i32u index = 0;
	while (index < tdata.size())
	{
		if (tdata[index].ref != NULL)
		{
			tmpv1.resize(0);
			for (i32u n1 = 0;n1 < tdata.size();n1++)
			{
				bool test1 = (tdata[n1].ref == NULL);
				bool test2 = (tdata[n1].id[1] == tdata[index].id[0]);
				if (test1 && test2) tmpv1.push_back(n1);
			}
			
			if (tmpv1.size()) break;
		}
		
		index++;
	}
	
	// if there is no suitable starting point, we have tested all atoms and
	// none of them is missing -> template fits -> TRUE!!!
	
	if (index == tdata.size()) return true;
	
	// find all possible bonds that we can use to continue testing...
	
	vector<mm1_cr> tmpv2; iter_mm1cl it1;
	for (it1 = tdata[index].ref->cr_list.begin();it1 != tdata[index].ref->cr_list.end();it1++)
	{
		bool test1 = (* it1).bndr->flags[flag];
		bool test2 = ((* it1).atmr->el.GetAtomicNumber() != 1);
		if (!test1 && test2) tmpv2.push_back((* it1));
	}
	
	// if there is less bond that we need, the template can't match -> FALSE!!!
	
	if (tmpv2.size() < tmpv1.size()) return false;
	
	vector<i32s> tmpv3; tmpv3.resize(tmpv2.size());
	for (i32u n1 = 0;n1 < tmpv3.size();n1++) tmpv3[n1] = n1;
	
	// then we will check all bonds against the template in all possible permutations.
	// if some combination is a match then leave the ID-numbers untouched, clean the
	// bond flags and leave -> TRUE!!!
	
	while (true)
	{
		bool test = true;
		for (i32u n1 = 0;n1 < tmpv1.size();n1++)
		{
			i32s el = tdata[tmpv1[n1]].el.GetAtomicNumber();
			if (el != NOT_DEFINED && el != tmpv2[tmpv3[n1]].atmr->el.GetAtomicNumber()) test = false;
			
			i32s bt = tdata[tmpv1[n1]].bt.GetValue();
			if (bt != NOT_DEFINED && bt != tmpv2[tmpv3[n1]].bndr->bt.GetValue()) test = false;
		}
		
		if (test)
		{
			for (i32u n1 = 0;n1 < tmpv1.size();n1++)
			{
				tmpv2[tmpv3[n1]].bndr->flags[flag] = true;
				tdata[tmpv1[n1]].ref = tmpv2[tmpv3[n1]].atmr;
			}
			
			bool result = CheckTemplate(tdata, flag);
			
			for (i32u n1 = 0;n1 < tmpv1.size();n1++)
			{
				tmpv2[tmpv3[n1]].bndr->flags[flag] = false;
				if (!result) tdata[tmpv1[n1]].ref = NULL;
			}
			
			if (result) return true;
		}
		
		if (!next_permutation(tmpv3.begin(), tmpv3.end())) break;
//		if (!next_permutation(tmpv3.begin(), tmpv3.end(), less<i32s>())) break;	// for WIN32 ???
	}
	
	// none of those combinations matched -> the template doesn't fit -> FALSE!!!
	
	return false;
}

/*################################################################################################*/

#define TRESHOLD -1.0	// h-bond energy treshold kcal/mol (K&S used -0.5 kcal/mol???)

void DefineSecondaryStructure(mm1_mdl * mdl)
{
	vector<mm1_chn_info> & ci_vector = (* mdl->ref_civ);
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		ci_vector[n1].state = new char[ci_vector[n1].length + 1];
		f64 * energy = new f64[ci_vector[n1].length];
		
		ci_vector[n1].state[ci_vector[n1].length] = 0;
		for (i32s n2 = 0;n2 < ci_vector[n1].length;n2++)
		{
			ci_vector[n1].state[n2] = '.';
			energy[n2] = TRESHOLD;
		}
		
		for (i32s n2 = 0;n2 < ci_vector[n1].length - 3;n2++)		// 3-turns...
		{
			i32s tmp1[2] = { n1, n2 };
			i32s tmp2[2] = { n1, n2 + 3 };
			
			f64 tmp3 = HBondEnergy(mdl, tmp1, tmp2);
			if (tmp3 < energy[n2])
			{
				ci_vector[n1].state[n2] = '3';
				energy[n2] = tmp3;
			}
		}
		
		for (i32s n2 = 0;n2 < ci_vector[n1].length - 4;n2++)		// 4-turns...
		{
			i32s tmp1[2] = { n1, n2 };
			i32s tmp2[2] = { n1, n2 + 4 };
			
			f64 tmp3 = HBondEnergy(mdl, tmp1, tmp2);
			if (tmp3 < energy[n2])
			{
				ci_vector[n1].state[n2] = '4';
				energy[n2] = tmp3;
			}
		}
		
		for (i32s n2 = 0;n2 < ci_vector[n1].length - 5;n2++)		// 5-turns...
		{
			i32s tmp1[2] = { n1, n2 };
			i32s tmp2[2] = { n1, n2 + 5 };
			
			f64 tmp3 = HBondEnergy(mdl, tmp1, tmp2);
			if (tmp3 < energy[n2])
			{
				ci_vector[n1].state[n2] = '5';
				energy[n2] = tmp3;
			}
		}
		
		delete[] energy;
		
		if (mdl->ostr) (* mdl->ostr) << "looking for intrachain strands for chain " << (n1 + 1) << ":" << endl;
		
		for (i32s n2 = 1;n2 < ci_vector[n1].length - 4;n2++)
		{
			if (mdl->ostr) (* mdl->ostr) << "." << flush;
			
			for (i32s n3 = n2 + 3;n3 < ci_vector[n1].length - 1;n3++)
			{
				i32s t1a[2] = { n1, n2 - 1 }; i32s t1b[2] = { n1, n3 };
				f64 t1c = HBondEnergy(mdl, t1a, t1b);
				
				i32s t2a[2] = { n1, n3 }; i32s t2b[2] = { n1, n2 + 1 };
				f64 t2c = HBondEnergy(mdl, t2a, t2b);
				
				i32s t3a[2] = { n1, n3 - 1 }; i32s t3b[2] = { n1, n2 };
				f64 t3c = HBondEnergy(mdl, t3a, t3b);
				
				i32s t4a[2] = { n1, n2 }; i32s t4b[2] = { n1, n3 + 1 };
				f64 t4c = HBondEnergy(mdl, t4a, t4b);
				
				i32s t5a[2] = { n1, n2 }; i32s t5b[2] = { n1, n3 };
				f64 t5c = HBondEnergy(mdl, t5a, t5b);
				
				i32s t6a[2] = { n1, n3 }; i32s t6b[2] = { n1, n2 };
				f64 t6c = HBondEnergy(mdl, t6a, t6b);
				
				i32s t7a[2] = { n1, n2 - 1 }; i32s t7b[2] = { n1, n3 + 1 };
				f64 t7c = HBondEnergy(mdl, t7a, t7b);
				
				i32s t8a[2] = { n1, n3 - 1 }; i32s t8b[2] = { n1, n2 + 1 };
				f64 t8c = HBondEnergy(mdl, t8a, t8b);
				
				bool bridge = false;
				if (t1c < TRESHOLD && t2c < TRESHOLD) bridge = true;
				if (t3c < TRESHOLD && t4c < TRESHOLD) bridge = true;
				if (t5c < TRESHOLD && t6c < TRESHOLD) bridge = true;
				if (t7c < TRESHOLD && t8c < TRESHOLD) bridge = true;
				
				if (bridge)
				{
					ci_vector[n1].state[n2] = 'S';
					ci_vector[n1].state[n3] = 'S';
				}
			}
		}
		
		if (mdl->ostr) (* mdl->ostr) << endl;
	}
	
	if (mdl->ostr) (* mdl->ostr) << "looking for interchain strands";
	
	for (i32s n1 = 0;n1 < ((i32s) ci_vector.size()) - 1;n1++)
	{
		for (i32s n2 = n1 + 1;n2 < (i32s) ci_vector.size();n2++)
		{
			if (mdl->ostr) (* mdl->ostr) << "." << flush;
			
			for (i32s n3 = 1;n3 < ci_vector[n1].length - 1;n3++)
			{
				for (i32s n4 = 1;n4 < ci_vector[n2].length - 1;n4++)
				{
					i32s t1a[2] = { n1, n3 - 1 }; i32s t1b[2] = { n2, n4 };
					f64 t1c = HBondEnergy(mdl, t1a, t1b);
					
					i32s t2a[2] = { n2, n4 }; i32s t2b[2] = { n1, n3 + 1 };
					f64 t2c = HBondEnergy(mdl, t2a, t2b);
					
					i32s t3a[2] = { n2, n4 - 1 }; i32s t3b[2] = { n1, n3 };
					f64 t3c = HBondEnergy(mdl, t3a, t3b);
					
					i32s t4a[2] = { n1, n3 }; i32s t4b[2] = { n2, n4 + 1 };
					f64 t4c = HBondEnergy(mdl, t4a, t4b);
					
					i32s t5a[2] = { n1, n3 }; i32s t5b[2] = { n2, n4 };
					f64 t5c = HBondEnergy(mdl, t5a, t5b);
					
					i32s t6a[2] = { n2, n4 }; i32s t6b[2] = { n1, n3 };
					f64 t6c = HBondEnergy(mdl, t6a, t6b);
					
					i32s t7a[2] = { n1, n3 - 1 }; i32s t7b[2] = { n2, n4 + 1 };
					f64 t7c = HBondEnergy(mdl, t7a, t7b);
					
					i32s t8a[2] = { n2, n4 - 1 }; i32s t8b[2] = { n1, n3 + 1 };
					f64 t8c = HBondEnergy(mdl, t8a, t8b);
					
					bool bridge = false;
					if (t1c < TRESHOLD && t2c < TRESHOLD) bridge = true;
					if (t3c < TRESHOLD && t4c < TRESHOLD) bridge = true;
					if (t5c < TRESHOLD && t6c < TRESHOLD) bridge = true;
					if (t7c < TRESHOLD && t8c < TRESHOLD) bridge = true;
					
					if (bridge)
					{
						ci_vector[n1].state[n3] = 'S';
						ci_vector[n2].state[n4] = 'S';
					}
				}
			}
		}
	}
	
	if (mdl->ostr) (* mdl->ostr) << endl;
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		cout << "chain " << n1 << ":" << endl;
		cout << ci_vector[n1].state << endl << endl;
	}
}

// here can be some problems with the carboxyl terminus where
// we can accidentally pick "wrong" oxygen...

f64 HBondEnergy(mm1_mdl * mdl, i32s * res1, i32s * res2)
{
	vector<mm1_chn_info> & ci_vector = (* mdl->ref_civ);
	if (ci_vector[res2[0]].sequence[res2[1]] == 'P') return 0.0;
	
	iter_mm1al r1a[2]; mdl->GetRange(1, res1[0], r1a);
	iter_mm1al r1b[2]; mdl->GetRange(2, r1a, res1[1], r1b);
	
	iter_mm1al r2a[2]; mdl->GetRange(1, res2[0], r2a);
	iter_mm1al r2b[2]; mdl->GetRange(2, r2a, res2[1], r2b);
	
	iter_mm1al it_o = r1b[0];
	while (it_o != r1b[1] && ((* it_o).res_id & 0xFF) != 0x10) it_o++;
	if (it_o == r1b[1]) return 0.0;
	
	iter_mm1al it_n = r2b[0];
	while (it_n != r2b[1] && ((* it_n).res_id & 0xFF) != 0x00) it_n++;
	if (it_n == r2b[1]) return 0.0;
	
	// if the distance between the residues is large enough we can be sure that there
	// is no h-bond and skip this time-consumig process. according to K&S this treshold
	// NO-distance is about 6.0 , but let's be careful and use a bit larger value...
	
	v3d<fGL> sv = v3d<fGL>((* it_o).crd_vector[0].data, (* it_n).crd_vector[0].data);
	if (sv.len() > 0.75) return 0.0;
	
// the above check is not enough to make this fast. problem is that the check is done in
// too late stage to be efficient. calculate first distances between all residues and
// use those results for sanity checking in earlier stage???

	iter_mm1al it_c = r1b[0];
	while (it_c != r1b[1] && ((* it_c).res_id & 0xFF) != 0x02) it_c++;
	if (it_c == r1b[1]) return 0.0;
	
	mm1_atom * ref_h = NULL;
	list<mm1_cr>::iterator it1 = (* it_n).cr_list.begin();
	while (it1 != (* it_n).cr_list.end() && ref_h == NULL)
	{
		if ((* it1).atmr->el.GetAtomicNumber() != 1) it1++;
		else ref_h = (* it1).atmr;
	}
	
	// the sequence builder currently ignores hydrogen atoms, so those
	// are not always available... also the proline case is problematic!!!
	
	// therefore we don't use the real hydrogens here at all, but we "reconstruct"
	// the amide hydrogen using the main-chain heavy atoms...
	
it1 = (* it_n).cr_list.end();	// we couldn't find the amide hydrogen...
	if (it1 == (* it_n).cr_list.end())
	{
		if (!res2[1]) return 0.0;	// skip if N-terminal residue...
		iter_mm1al r2c[2]; mdl->GetRange(2, r2a, res2[1] - 1, r2c);
		
		iter_mm1al it_c1 = r2b[0];
		while (it_c1 != r2b[1] && ((* it_c1).res_id & 0xFF) != 0x01) it_c1++;
		if (it_c1 == r2b[1]) return 0.0;
		
		iter_mm1al it_c2 = r2c[0];
		while (it_c2 != r2c[1] && ((* it_c2).res_id & 0xFF) != 0x02) it_c2++;
		if (it_c2 == r2c[1]) return 0.0;
		
		v3d<fGL> v1 = v3d<fGL>((* it_n).crd_vector[0].data, (* it_c1).crd_vector[0].data);
		v3d<fGL> v2 = v3d<fGL>((* it_n).crd_vector[0].data, (* it_c2).crd_vector[0].data);
		v3d<fGL> v3 = v1.vpr(v2); v3d<fGL> v4 = v1.vpr(v3);
		
		const fGL len = 0.1024;
		const fGL ang = M_PI * 121.0 / 180.0;
		v1 = v1 * (len * cos(ang) / v1.len());
		v4 = v4 * (len * sin(ang) / v4.len());
		v3d<fGL> v5 = v3d<fGL>((* it_n).crd_vector[0].data) + (v1 + v4);
		ref_h = new mm1_atom(element(1), v5.data, 1);
		
//		v3d<fGL> tv1 = v3d<fGL>((* it_n).crd[0], v5.comp);
//		cout << "the H bond length = " << tv1.len() << endl;
//		cout << "the H bond angle = " << (tv1.ang(v2) * 180.0 / M_PI) << endl;
//		i32s zzz; cin >> zzz;
	}
	
	v3d<fGL> on_v = v3d<fGL>((* it_o).crd_vector[0].data, (* it_n).crd_vector[0].data);
	v3d<fGL> ch_v = v3d<fGL>((* it_c).crd_vector[0].data, ref_h->crd_vector[0].data);
	
	v3d<fGL> oh_v = v3d<fGL>((* it_o).crd_vector[0].data, ref_h->crd_vector[0].data);
	v3d<fGL> cn_v = v3d<fGL>((* it_c).crd_vector[0].data, (* it_n).crd_vector[0].data);
	
	if (it1 == (* it_n).cr_list.end()) delete ref_h;
	
	f64 tmp1 = 1.0 / on_v.len() + 1.0 / ch_v.len();
	f64 tmp2 = 1.0 / oh_v.len() + 1.0 / cn_v.len();
	f64 tmp3 = 0.42 * 0.20 * (tmp1 - tmp2);
	return 0.1 * 332.0 * tmp3;
}

/*################################################################################################*/

// eof
