// VIEWS.CPP

// Copyright (C) 1998 Tommi Hassinen, Jarno Huuskonen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "views.h"	// config.h is here -> we get ENABLE-macros here...

#ifdef ENABLE_GRAPHICS
#include "docview.h"
#include "camera.h"
#include "color.h"

#include "mm1docv.h"	// plotting...

#include <strstream>
using namespace std;

/*################################################################################################*/

view::view(void)
{
}

view::~view(void)
{
}

/*################################################################################################*/

mouse_tool::state ogl_view::state = mouse_tool::Unknown;
mouse_tool::button ogl_view::button = mouse_tool::None;

bool ogl_view::shift_down = false;
bool ogl_view::ctrl_down = false;

draw_tool ogl_view::tool_draw;
erase_tool ogl_view::tool_erase;

select_tool ogl_view::tool_select;
select_tool::select_mode ogl_view::select_mode = select_tool::Atom;
zoom_tool ogl_view::tool_zoom;

translate_xy_tool ogl_view::tool_translate_xy;
translate_z_tool ogl_view::tool_translate_z;

orbit_xy_tool ogl_view::tool_orbit_xy;
orbit_z_tool ogl_view::tool_orbit_z;

rotate_xy_tool ogl_view::tool_rotate_xy;
rotate_z_tool ogl_view::tool_rotate_z;

measure_tool ogl_view::tool_measure;
measure_tool::measure_mode ogl_view::measure_mode = measure_tool::Bond;

// tool_orbit_xy is the default tool...
// tool_orbit_xy is the default tool...
// tool_orbit_xy is the default tool...

mouse_tool * ogl_view::current_tool = & ogl_view::tool_orbit_xy;

ogl_view::ogl_view() : view()
{
}

ogl_view::~ogl_view(void)
{
}

// later, before stereo views were added, glViewport() was called here...
// but now, this is no longer the practice; now call glViewport() when rendering!!!

void ogl_view::SetSize(i32s p1, i32s p2)
{
	size[0] = p1; size[1] = p2;
	if (!size[0] || !size[1]) size[0] = NOT_DEFINED;
	Update(false);
}

// 2001-12-28 : tried to add support for both bitmap and line-based fonts but it's not ready yet.
// the font scaling and even orientation seems to be very different... there seems to be in use
// some versions of GLUT that lack the bitmap fonts and related functions. hopefully there remains
// at least some sane way to render text using OpenGL/GLUT; at the moment the support for line
// fonts here isveeery broken...  :(

// 2001-12-29 : the line fonts are now more or less readable everywhere, but 3D text still is
// not correctly aligned...

const fGL stroke_font_scale = 0.1;

i32s ogl_view::GetGlutStringWidth(const char * str, void * font)
{
//font = NULL;	// uncomment this to try the line fonts...
	bool use_bitmap_font = (font != NULL ? true : false);
	
	i32u count = 0; i32s width = 0;
	while (count < strlen(str))
	{
		if (use_bitmap_font) width += glutBitmapWidth(font, str[count++]);
		else width += glutStrokeWidth(GLUT_STROKE_ROMAN, str[count++]);
	}
	
	if (!use_bitmap_font) width = (i32s) (width * stroke_font_scale);
	return width;
}

void ogl_view::WriteGlutString2D(const char * str, fGL x, fGL y, void * font)
{
	glPushMatrix();
	glLoadIdentity();
	
	glMatrixMode(GL_PROJECTION);
	glPushMatrix(); glLoadIdentity();
	gluOrtho2D(0, size[0], 0, size[1]);
	
	WriteGlutString3D(str, x, y, 0.0, NULL, font);
	
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	
	glPopMatrix();
}

void ogl_view::WriteGlutString3D(const char * str, fGL x, fGL y, fGL z, const obj_loc_data * eye, void * font)
{
//font = NULL;	// uncomment this to try the line fonts...
	bool use_bitmap_font = (font != NULL ? true : false);
	
	// text centering?!?!?! add a new flag???
	// text centering?!?!?! add a new flag???
	// text centering?!?!?! add a new flag???
	
	if (use_bitmap_font)
	{
		glDisable(GL_DEPTH_TEST);
		
		glRasterPos3f(x, y, z);
		
		i32u count = 0;
		while (count < strlen(str))
		{
			glutBitmapCharacter(font, str[count++]);
		}
		
		glEnable(GL_DEPTH_TEST);
	}
	else
	{
		glDisable(GL_DEPTH_TEST);
		
		glPushMatrix();
		glTranslatef(x, y, z);
		
	// here the text should be oriented to the "eye" coordinates if they are given...
	// it's possible but tricky; something similar than in SetModelView() at objects.cpp
		
		fGL font_scale = stroke_font_scale;		// the 2D and 3D texts must be scaled very
		if (eye != NULL) font_scale /= 500.0;		// differently; detect 3D cases by "eye"...
		glScalef(font_scale, font_scale, font_scale);	// GetGlutStringWidth() is not scaled!!!
		
		i32u count = 0;
		while (count < strlen(str))
		{
			glutStrokeCharacter(GLUT_STROKE_ROMAN, str[count++]);
		}
		
		glPopMatrix();
		glEnable(GL_DEPTH_TEST);
	}
}

/*################################################################################################*/

bool graphics_view::quick_update = false;
bool graphics_view::draw_info = true;

graphics_view::graphics_view(camera * p1) : ogl_view()
{
	cam = p1;
	
//	render = RENDER_WIREFRAME;		// make this model-dependent like the coloring...
	render = RENDER_BALL_AND_STICK;		// make this model-dependent like the coloring...
	label = LABEL_NOTHING;
	
	colormode = cam->docv->GetDefaultColorMode();
	
	enable_fog = false;
	accumulate = false;
}

graphics_view::~graphics_view(void)
{
}

void graphics_view::GetCRD(i32s * p1, fGL * p2)
{
	v3d<fGL> xdir = (cam->GetLocData()->ydir).vpr(cam->GetLocData()->zdir); xdir = xdir / xdir.len();
	v3d<fGL> tmpv = v3d<fGL>(cam->GetLocData()->crd); tmpv = tmpv + (cam->GetLocData()->zdir * cam->focus);
	tmpv = tmpv + xdir * (2.0 * range[0] * (fGL) (size[0] / 2 - p1[0]) / (fGL) size[0]);
	tmpv = tmpv + cam->GetLocData()->ydir * (2.0 * range[1] * (fGL) (size[1] / 2 - p1[1]) / (fGL) size[1]);
	for (i32s n1 = 0;n1 < 3;n1++) p2[n1] = tmpv[n1];
}

void graphics_view::InitGL(void)
{
	SetCurrent();
	GetDV()->InitGL();
}

void graphics_view::Render(void)
{
	SetCurrent();
	cam->RenderWindow(this);
}

/*################################################################################################*/

// the "udata" handling functions (for creating and using the data) are here. allocate memory with malloc!!!
// the "udata" handling functions (for creating and using the data) are here. allocate memory with malloc!!!
// the "udata" handling functions (for creating and using the data) are here. allocate memory with malloc!!!

void * mm1_convert_cset_to_plotting_udata(mm1_mdl * mdl, i32s cset)
{
	void * udata = malloc(sizeof(i32s) + sizeof(fGL) * mdl->GetAtomCount() * 3);
	
	// first, store the number of atoms in the model, as a simple way to test data validity...
	
	i32s * ptr1 = (i32s *) udata;
	(* ptr1) = mdl->GetAtomCount();
	
	// then just store the coordinates...
	
	fGL * ptr2 = (fGL *) ((i32u) udata + sizeof(i32s)); i32s counter = 0;
	for (iter_mm1al it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		ptr2[counter++] = (* it1).crd_vector[cset].data[0];
		ptr2[counter++] = (* it1).crd_vector[cset].data[1];
		ptr2[counter++] = (* it1).crd_vector[cset].data[2];
	}
	
	return udata;
}

void mm1_apply_udata_as_cset(mm1_docv * docv, void * udata)
{
	i32s * ptr1 = (i32s *) udata;
	if (docv->GetAtomCount() != (* ptr1))
	{
		cout << "oops!!! the atom counts are different." << endl;
		return;
	}
	
	fGL * ptr2 = (fGL *) ((i32u) udata + sizeof(i32s)); i32s counter = 0;
	for (iter_mm1al it1 = docv->GetAtomsBegin();it1 != docv->GetAtomsEnd();it1++)
	{
		const i32s cset = 0;	// how to set this?!?!?!
		(* it1).crd_vector[cset].data[0] = ptr2[counter++];
		(* it1).crd_vector[cset].data[1] = ptr2[counter++];
		(* it1).crd_vector[cset].data[2] = ptr2[counter++];
	}
	
	docv->UpdateAllGraphicsViews();
}

// the base class...
// the base class...
// the base class...

plotting_view::plotting_view(model_simple * p1, i32s p2, i32s p3)
{
	mdl = p1;
	plot_userdata1 = p2;
	plot_userdata2 = p3;
	
	select_buffer = new iGLu[SB_SIZE];
}

plotting_view::~plotting_view(void)
{
	delete[] select_buffer;
}

void plotting_view::InitGL(void)
{
	SetCurrent();
	
	prefs * model_prefs = mdl->model_prefs;		// needed elsewhere???

	const fGL default_background[4] = { 0.0, 0.0, 0.0, 1.0};
	fGL * background = model_prefs->ColorRGBA("Graphics/BkgndColor", default_background);
	glClearColor(background[0], background[1], background[2], background[3]); delete[] background;
	glDepthFunc(GL_LESS); glEnable(GL_DEPTH_TEST);
	
	glMateriali(GL_FRONT_AND_BACK, GL_SHININESS, 64);
	const fGL default_reflectance[4] = { 0.9, 0.9, 0.9, 1.0 };
	fGL * specular_reflectance = model_prefs->ColorRGBA("Graphics/SpecularReflectance", default_reflectance);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular_reflectance); delete[] specular_reflectance;
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glEnable(GL_COLOR_MATERIAL);
	
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	glSelectBuffer(SB_SIZE, select_buffer);
}

/*################################################################################################*/

plot1d_view::plot1d_view(model_simple * p1, i32s p2, i32s p3) : plotting_view(p1, p2, p3)
{
}

plot1d_view::~plot1d_view(void)
{
	cout << "mem leak!!! free() the non-NULL udata blocks for all data points!!!" << endl;
}

void plot1d_view::AddData(f64 c1, f64 v, void * udata)
{
	plot1d_data newdata;
	newdata.c1 = c1; newdata.v = v; newdata.udata = udata;
	
	dv.push_back(newdata);
}

void plot1d_view::SetCenterAndScale(void)
{
	if (dv.empty()) return;
	
	min1 = max1 = dv[0].c1;
	minv = maxv = dv[0].v;
	
	for (i32u n1 = 1;n1 < dv.size();n1++)
	{
		if (dv[n1].c1 < min1) min1 = dv[n1].c1;
		if (dv[n1].c1 > max1) max1 = dv[n1].c1;
		
		if (dv[n1].v < minv) minv = dv[n1].v;
		if (dv[n1].v > maxv) maxv = dv[n1].v;
	}
}

void plot1d_view::Render(void)
{
	if (size[0] < 0) return;			// skip rendering if invalid window!!!
//	prefs * model_prefs = mdl->model_prefs;		// needed elsewhere???

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	// set projection and call gluLookAt() to set up our "eye".
	
	i32s width = size[0]; i32s height = size[1]; glViewport(0, 0, width, height);
	glOrtho(-0.1, +1.1, -0.1, +1.1, -0.5, +1.5);
	glMatrixMode(GL_MODELVIEW); glLoadIdentity();
	
	gluLookAt(0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
	
//	const fGL default_label_color[3] = { 1.0, 1.0, 1.0 };
//	fGL * label_color = model_prefs->ColorRGB("Graphics/LabelColor", default_label_color);

	glInitNames();
	
	for (i32s n1 = 0;n1 < (i32s) dv.size() - 1;n1++)
	{
		glColor3f(0.0, 1.0, 0.0);	// green
		glBegin(GL_LINES); fGL x; fGL y;
		
		x = (dv[n1].c1 - min1) / (max1 - min1);
		y = (dv[n1].v - minv) / (maxv - minv);
		glVertex3f(x, y, 0.1);
		
		x = (dv[n1 + 1].c1 - min1) / (max1 - min1);
		y = (dv[n1 + 1].v - minv) / (maxv - minv);
		glVertex3f(x, y, 0.1);
		
		glEnd();
	}
}

void plot1d_view::ZoomEvent(i32s)
{
}

void plot1d_view::TransEvent(i32s)
{
}

void plot1d_view::UserEvent(i32s x, i32s)
{
	if (!(size[0] > 1)) return;
	fGL sx = 1.2 * (fGL) x / (fGL) size[0] - 0.1;
	
	if (sx < 0.0) return;
	if (sx > 1.0) return;
	
	i32s index = (i32s) (sx * dv.size());
	if (index >= (i32s) dv.size()) return;
	
	if (!dv[index].udata) return;
	
	// ok, we have a valid selection...
	
// TODO : check the plot_userdata1 and plot_userdata2...
// TODO : check the plot_userdata1 and plot_userdata2...
// TODO : check the plot_userdata1 and plot_userdata2...

	mm1_docv * mm1dv = dynamic_cast<mm1_docv *>(mdl);
	if (mm1dv != NULL)
	{
		char mbuff1[256];
		ostrstream str1(mbuff1, sizeof(mbuff1));
		str1 << "tor = " << dv[index].c1 << " energy = " << dv[index].v << " kJ/mol" << endl << ends;
		mdl->PrintToLog(mbuff1);
		
		mm1_apply_udata_as_cset(mm1dv, dv[index].udata);
	}
}

/*################################################################################################*/

plot2d_view::plot2d_view(model_simple * p1, i32s p2, i32s p3) : plotting_view(p1, p2, p3)
{
}

plot2d_view::~plot2d_view(void)
{
	cout << "mem leak!!! free() the non-NULL udata blocks for all data points!!!" << endl;
}

void plot2d_view::AddData(f64 c1, f64 c2, f64 v, void * udata)
{
	plot2d_data newdata;
	newdata.c1 = c1; newdata.c2 = c2; newdata.v = v; newdata.udata = udata;
	
	dv.push_back(newdata);
}

void plot2d_view::SetCenterAndScale(void)
{
	if (dv.empty()) return;
	
	min1 = max1 = dv[0].c1;
	min2 = max2 = dv[0].c2;
	minv = maxv = dv[0].v;
	
	for (i32u n1 = 1;n1 < dv.size();n1++)
	{
		if (dv[n1].c1 < min1) min1 = dv[n1].c1;
		if (dv[n1].c1 > max1) max1 = dv[n1].c1;
		
		if (dv[n1].c2 < min2) min2 = dv[n1].c2;
		if (dv[n1].c2 > max2) max2 = dv[n1].c2;
		
		if (dv[n1].v < minv) minv = dv[n1].v;
		
		// for "maxv" we actually calculate the average value instead of maximum value.
		// the max value can be very high especially if geomopt is skipped -> unreasonable scaling.
		
		maxv += dv[n1].v;	// average!
	}
	
	maxv /= dv.size();	// average!
}

void plot2d_view::Render(void)
{
	if (size[0] < 0) return;			// skip rendering if invalid window!!!
//	prefs * model_prefs = mdl->model_prefs;		// needed elsewhere???

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	// set projection and call gluLookAt() to set up our "eye".
	
	i32s width = size[0]; i32s height = size[1]; glViewport(0, 0, width, height);
	glOrtho(-0.1, +1.1, -0.1, +1.1, -0.5, +1.5);
	glMatrixMode(GL_MODELVIEW); glLoadIdentity();
	
	gluLookAt(0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
	
//	const fGL default_label_color[3] = { 1.0, 1.0, 1.0 };
//	fGL * label_color = model_prefs->ColorRGB("Graphics/LabelColor", default_label_color);

	glInitNames();
	
	i32s zzz = (i32s) sqrt((double) dv.size());	// support square datasets only!!!
	for (i32s n1 = 0;n1 < zzz - 1;n1++)
	{
		for (i32s n2 = 0;n2 < zzz - 1;n2++)
		{
			glBegin(GL_QUADS);
			
			fGL x1 = (dv[(n1 + 0) * zzz + (n2 + 0)].c1 - min1) / (max1 - min1);
			fGL y1 = (dv[(n1 + 0) * zzz + (n2 + 0)].c2 - min2) / (max2 - min2);
			fGL x2 = (dv[(n1 + 1) * zzz + (n2 + 1)].c1 - min1) / (max1 - min1);
			fGL y2 = (dv[(n1 + 1) * zzz + (n2 + 1)].c2 - min2) / (max2 - min2);
			
			SetColor(dv[(n1 + 0) * zzz + (n2 + 0)].v); glVertex3f(x1, y1, 0.1);
			SetColor(dv[(n1 + 1) * zzz + (n2 + 0)].v); glVertex3f(x2, y1, 0.1);
			SetColor(dv[(n1 + 1) * zzz + (n2 + 1)].v); glVertex3f(x2, y2, 0.1);
			SetColor(dv[(n1 + 0) * zzz + (n2 + 1)].v); glVertex3f(x1, y2, 0.1);
			
			glEnd();
		}
	}
}

void plot2d_view::SetColor(f64 v)
{
	fGL color[4];
	f64 cv = (v - minv) / (maxv - minv);
	GetRBRange1(cv * cv, 1.0, color);
	glColor3fv(color);
}

void plot2d_view::ZoomEvent(i32s dy)
{
	f64 zzz = maxv - minv;
	maxv += mouse_tool::dist_sensitivity * zzz * (fGL) dy / (fGL) size[1];
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "now scaling the colours to range " << (maxv - minv) << " kJ/mol" << endl << ends;
	mdl->PrintToLog(mbuff1);
	
	Update();
}

void plot2d_view::TransEvent(i32s)
{
}

void plot2d_view::UserEvent(i32s x, i32s y)
{
	if (!(size[0] > 1)) return;
	fGL sx = 1.2 * (fGL) x / (fGL) size[0] - 0.1;
	
	if (sx < 0.0) return;
	if (sx > 1.0) return;
	
	if (!(size[1] > 1)) return;
	fGL sy = 1.2 * (fGL) y / (fGL) size[1] - 0.1;
	
	if (sy < 0.0) return;
	if (sy > 1.0) return;
	
	i32s zzz = (i32s) sqrt((double) dv.size());	// support square datasets only!!!
	i32s index = (i32s) (((i32s) (sx * zzz)) * zzz + sy * zzz);
	if (index >= (i32s) dv.size()) return;
	
	if (!dv[index].udata) return;
	
	// ok, we have a valid selection...
	
// TODO : check the plot_userdata1 and plot_userdata2...
// TODO : check the plot_userdata1 and plot_userdata2...
// TODO : check the plot_userdata1 and plot_userdata2...
	
	mm1_docv * mm1dv = dynamic_cast<mm1_docv *>(mdl);
	if (mm1dv != NULL)
	{
		char mbuff1[256];
		ostrstream str1(mbuff1, sizeof(mbuff1));
		str1 << "tor1 = " << dv[index].c1 << " tor2 = " << dv[index].c2 << " energy = " << dv[index].v << " kJ/mol" << endl << ends;
		mdl->PrintToLog(mbuff1);
		
		mm1_apply_udata_as_cset(mm1dv, dv[index].udata);
	}
}

/*################################################################################################*/

enlevdiag_view::enlevdiag_view(qm1_mdl * p1) : plotting_view(p1, PLOT_USERDATA_IGNORE, 0)
{
	mdl = p1;
	SetCenterAndScale();
}

enlevdiag_view::~enlevdiag_view(void)
{
}

void enlevdiag_view::SetCenterAndScale(void)
{
	if (mdl->current_eng != NULL)
	{
		f64 mine = mdl->current_eng->GetOrbitalEnergy(0);
		f64 maxe = mdl->current_eng->GetOrbitalEnergy(mdl->current_eng->GetOrbitalCount() - 1);
		
		// above we assumed that the first orbital has lowest and the last orbital has highest energy...
		
		center = (mine + maxe) / 2.0;
		scale = (maxe - mine) * 1.5;
	}
	else
	{
		center = 0.0;
		scale = 1.0;
	}
}

void enlevdiag_view::Render(void)
{
	if (size[0] < 0) return;			// skip rendering if invalid window!!!
	prefs * model_prefs = mdl->model_prefs;		// needed elsewhere???

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	// set projection and call gluLookAt() to set up our "eye".
	
// PREFS: making this value bigger will make the left margin (for displaying text) bigger.
	fGL diagram_left_margin = 0.2 + 0.5;
	
	i32s width = size[0]; i32s height = size[1]; glViewport(0, 0, width, height);
	glOrtho(-diagram_left_margin, +1.2, center - scale * 0.5, center + scale * 0.5, -0.5, +1.5);
	glMatrixMode(GL_MODELVIEW); glLoadIdentity();
	
	gluLookAt(0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
	
// render...
// render...
// render...

	const fGL default_label_color[3] = { 1.0, 1.0, 1.0 };
	fGL *label_color = model_prefs->ColorRGB("Graphics/LabelColor", default_label_color);

	glInitNames();
	if (!mdl->current_eng)		// print out a message if we have no data...
	{
		glColor3f(label_color[0], label_color[1], label_color[2]);
		
		fGL xpos; fGL ypos;
		void * font = GLUT_BITMAP_TIMES_ROMAN_24;
		
		const char * txt1 = "No data available,";
		xpos = (size[0] - GetGlutStringWidth(txt1, font)) / 2.0;
		ypos = (size[1] - 24) / 2.0 + 14;
		WriteGlutString2D(txt1, xpos, ypos, font);
		
		const char * txt2 = "please calculate energy!";
		xpos = (size[0] - GetGlutStringWidth(txt2, font)) / 2.0;
		ypos = (size[1] - 24) / 2.0 - 14;
		WriteGlutString2D(txt2, xpos, ypos, font);
	}
	else	// ...or otherwise render the diagram.
	{
		fGL one_pixel_height = scale / (fGL) height;
		
// PREFS: making this value bigger will make lines thicker (the value is in pixels).
		fGL diagram_line_thickness = model_prefs->Double("ELGraphics/LineWidth", 2.0);
// PREFS: making this value bigger will make triangles taller (the value is in pixels).
		fGL diagram_triangle_height = model_prefs->Double("ELGraphics/TriangleHeight", 30.0);
// PREFS: making this value bigger will make lines wider.
		fGL diagram_triangle_width = model_prefs->Double("ELGraphics/TriangleWidth", 0.1);
		
// the functions:
// GetOrbitalCount()
// GetOrbitalEnergy()
// need to separate between alpha and beta electrons.
// also a simple way to determine occupation (and whether is alpha/beta) is needed.

		// gather the degenerate energy levels (del) into groups...
		
		const fGL del_tolerance = 0.001;
		i32s * del_first = new i32s[mdl->current_eng->GetOrbitalCount()];
		
		del_first[0] = 0; i32s dd = 1;
		while (dd < mdl->current_eng->GetOrbitalCount())
		{
			i32s ind_first = del_first[dd - 1];
			
			fGL ene_first = mdl->current_eng->GetOrbitalEnergy(ind_first);
			fGL ene_current = mdl->current_eng->GetOrbitalEnergy(dd);
			
			if (ene_current - ene_first < del_tolerance)
			{
				del_first[dd] = ind_first;	// belongs to the previous set.
			}
			else
			{
				del_first[dd] = dd;		// is the first of a new set.
			}
			
			dd++;
		}
		
		i32s * del_count = new i32s[mdl->current_eng->GetOrbitalCount()];
		for (dd = 0;dd < mdl->current_eng->GetOrbitalCount();dd++) del_count[dd] = 0;
		
		dd = 0; i32s first = 0;
		while (dd < mdl->current_eng->GetOrbitalCount())	// calculate the group sizes...
		{
			if (del_first[dd] != del_first[first]) first = dd;
			del_count[first]++;
			dd++;
		}
		
		dd = 0; first = 0;
		while (dd < mdl->current_eng->GetOrbitalCount())	// ...and copy them to all group members.
		{
			if (del_first[dd] != del_first[first]) first = dd;
			if (first != dd) del_count[dd] = del_count[first];
			dd++;
		}
		
		// render...
		
		for (i32s n1 = 0;n1 < mdl->current_eng->GetOrbitalCount();n1++)
		{
			fGL energy = mdl->current_eng->GetOrbitalEnergy(n1);
			
		//fGL left = 0.0; fGL right = 1.0; fGL width = right - left;	// OBSOLETE...
		
		fGL width = 1.0 / (fGL) del_count[n1];
		fGL left = 0.0 + (fGL) (n1 - del_first[n1]) * width;
		fGL right = 0.0 + (fGL) (n1 - del_first[n1] + 1) * width;
		
			// draw the line...
			// draw the line...
			// draw the line...
			
			glColor3f(0.0, 1.0, 0.0);	// green
			glBegin(GL_QUADS);
			
			glVertex3f(right - 0.05 * width, energy, 0.1);
			glVertex3f(left + 0.05 * width, energy, 0.1);
			glVertex3f(left + 0.05 * width, energy + one_pixel_height * diagram_line_thickness, 0.1);
			glVertex3f(right - 0.05 * width, energy + one_pixel_height * diagram_line_thickness, 0.1);
			
			glEnd();
			
			// draw the electrons...
			// draw the electrons...
			// draw the electrons...
			
			if (n1 < mdl->current_eng->GetElectronCount() / 2)
			{
				glColor3f(1.0, 1.0, 0.0);	// yellow
				glBegin(GL_TRIANGLES);
				
glVertex3f(left + 0.3 * width - diagram_triangle_width, energy - one_pixel_height * diagram_triangle_height / 3.0, 0.0);
glVertex3f(left + 0.3 * width, energy + one_pixel_height * diagram_triangle_height / 1.5, 0.0);
glVertex3f(left + 0.3 * width + diagram_triangle_width, energy - one_pixel_height * diagram_triangle_height / 3.0, 0.0);

glVertex3f(right - 0.3 * width - diagram_triangle_width, energy + one_pixel_height * diagram_triangle_height / 3.0, 0.0);
glVertex3f(right - 0.3 * width, energy - one_pixel_height * diagram_triangle_height / 1.5, 0.0);
glVertex3f(right - 0.3 * width + diagram_triangle_width, energy + one_pixel_height * diagram_triangle_height / 3.0, 0.0);

				glEnd();
			}
			
			// print out the text...
			// print out the text...
			// print out the text...
			
			glColor3f(label_color[0], label_color[1], label_color[2]);
			
			fGL xpos; fGL ypos;
			void * font = GLUT_BITMAP_TIMES_ROMAN_10;
			
			char txt[256];
			ostrstream str(txt, sizeof(txt));
			str << "i = " << n1 << " e = " << energy << ends;
			
			xpos = 10.0;
			ypos = ((energy - center) / scale + 0.5) * size[1] - 5.0;	// 5.0 comes from font height 10...
			ypos += (fGL) (n1 - del_first[n1]) * 10.0;			// 10.0 comes from font height 10...
			
			WriteGlutString2D(txt, xpos, ypos, font);
		}
		
		delete[] del_first;
		delete[] del_count;
	}
	
	delete[] label_color;
}

void enlevdiag_view::ZoomEvent(i32s dy)
{
	scale += mouse_tool::dist_sensitivity * scale * (fGL) dy / (fGL) size[1];
	Update();
}

void enlevdiag_view::TransEvent(i32s dy)
{
	center -= 0.5 * scale * (fGL) dy / (fGL) size[1];
	Update();
}

// figure out the selections using coordinates instead of OpenGL selection mode?!?!?! might be simpler in this 2-dimensional case...
// figure out the selections using coordinates instead of OpenGL selection mode?!?!?! might be simpler in this 2-dimensional case...
// figure out the selections using coordinates instead of OpenGL selection mode?!?!?! might be simpler in this 2-dimensional case...

void enlevdiag_view::UserEvent(i32s, i32s)
{
	cout << "UserEvent() was called!" << endl;
}

/*################################################################################################*/

#ifdef ENABLE_TREELIST_VIEW

treelist_view::treelist_view() : view()
{
}

treelist_view::~treelist_view(void)
{
}

#endif	// ENABLE_TREELIST_VIEW

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS

// eof
