/*
 * $Id: gt_protocol.c,v 1.94 2003/12/23 10:28:15 hipnod Exp $
 *
 * Copyright (C) 2001-2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

#include "sha1.h"
#include "xml.h"

#include "gt_node.h"
#include "gt_node_cache.h"
#include "gt_node_list.h"
#include "gt_netorg.h"

#include "gt_connect.h"

#include "gt_packet.h"
#include "gt_protocol.h"

#include "gt_search.h"
#include "gt_search_exec.h"

#include "gt_query_route.h"
#include "gt_stats.h"

#include "gt_share.h"
#include "gt_share_file.h"

#include "gt_utils.h"
#include "gt_xfer.h"

#include "gt_accept.h"
#include "gt_ban.h"

#include "gt_urn.h"

#include "io/rx_stack.h"               /* gt_rx_stack_new */
#include "io/tx_stack.h"               /* gt_tx_stack_new */

#include <libgift/mime.h>

/*****************************************************************************/

typedef void (*ProtocolHandler) (TCPC *c, GtPacket *packet);

#define GTLA_HANDLER(func) \
	static void gt_##func (TCPC *c, GtPacket *packet)

GTLA_HANDLER (ping_request);
GTLA_HANDLER (ping_response);

GTLA_HANDLER (bye_request);
GTLA_HANDLER (push_request);
GTLA_HANDLER (query_route);
GTLA_HANDLER (query_request);
GTLA_HANDLER (query_response);

/*
 * Whether to attach the number of hops each result travelled as a metadata
 * field "Hops". Interesting to see but probably not too useful for the
 * average user, and so disabled by default.
 */
#define HOPS_AS_META         gt_config_get_int("search/hops_as_meta=0")

/*****************************************************************************/

static struct _handler_table
{
	uint8_t command;
	ProtocolHandler func;
}
protocol_handler[] =
{
	{ GT_PING_REQUEST,     gt_ping_request    },
	{ GT_PING_RESPONSE,    gt_ping_response   },
	{ GT_BYE_REQUEST,      gt_bye_request     },
	{ GT_QUERY_ROUTE,      gt_query_route     },
	{ GT_PUSH_REQUEST,     gt_push_request    },
	{ GT_QUERY_REQUEST,    gt_query_request   },
	{ GT_QUERY_RESPONSE,   gt_query_response  },
	{ 0x00,                NULL               }
};

struct _search_reply
{
	uint8_t     ttl;
	uint8_t     results;  /* number of results to on the current packet */
	GtPacket   *packet;   /* the current packet to stack results on */
	gt_guid_t  *guid;
	int         dontfree; /* workaround queue_add() calling the free func */
};

/*****************************************************************************/

static int protocol_handle_command (TCPC *c, GtPacket *packet)
{
	struct _handler_table *handler;
	unsigned char command;

	if (!packet)
		return FALSE;

	command = gt_packet_command (packet);

	/* locate the handler */
	for (handler = protocol_handler; handler->func; handler++)
	{
		if (command == handler->command)
		{
			handler->func (c, packet);
			return TRUE;
		}
	}

	GIFT_ERROR (("[%s] found no handler for cmd %hx, payload %hx",
				 net_ip_str (GT_NODE(c)->ip), command,
				 gt_packet_payload_len (packet)));

	return FALSE;
}

static void cleanup_node_rx (GtNode *node)
{
	TCPC    *c = GT_CONN(node);

	assert (GT_NODE(c) == node);
	gt_node_disconnect (c);
}

/* TODO: make this the same type as cleanup_node_rx */
static void cleanup_node_tx (GtTxStack *stack, GtNode *node)
{
	TCPC *c = GT_CONN(node);

	assert (GT_NODE(c) == node);
	gt_node_disconnect (c);
}

static void recv_packet (GtNode *node, GtPacket *packet)
{
	assert (packet != NULL);

	gt_packet_log (packet, GT_CONN(node), FALSE);
	(void)protocol_handle_command (node->c, packet);
}

/* Find out what our IP is */
static in_addr_t get_self_ip (TCPC *c)
{
	in_addr_t our_ip;
	char     *ip_str;

	if ((ip_str = dataset_lookupstr (GT_NODE(c)->hdr, "remote-ip")))
	{
		/* 
		 * Since we may be firewalled, we may not know what our ip is.  So set
		 * the ip from what the other node thinks it is.
		 *
		 * Doing this allows you to setup port forwarding on a firewall
		 * and accept incoming connections. 
		 */
		our_ip = net_ip (ip_str);
	}
	else
	{
		struct sockaddr_in saddr;
		int    len = sizeof (saddr);

		if (getsockname (c->fd, (struct sockaddr *)&saddr, &len) == 0)
			our_ip = saddr.sin_addr.s_addr;
		else
			our_ip = net_ip ("127.0.0.1");
	}

	return our_ip;
}

/*
 * Begin a node connection with the peer on the specified TCPC.
 *
 * We arrive here from either an incoming or outgoing connection.
 * This is the entrance point to the main packet-reading loop.
 *
 * After setting up the connection, we send the node a ping.
 * If it doesn't respond after a timeout, we will destroy the
 * connection.
 */
void gnutella_start_connection (int fd, input_id id, TCPC *c)
{
	GtPacket    *ping;
	GtNode      *node;

	node = GT_NODE(c);
	assert (GT_CONN(node) == c);

	/* remove the old input handler first -- need to before sending data */
	input_remove (id);

	if (net_sock_error (c->fd))
	{
		if (HANDSHAKE_DEBUG)
			gt_node_error (c, NULL);

		gt_node_disconnect (c);
		return;
	}

	/* if this is the crawler, disconnect */
	if (dataset_lookupstr (GT_NODE(c)->hdr, "crawler"))
	{
		if (HANDSHAKE_DEBUG)
			GT->DBGSOCK (GT, c, "closing crawler connection");

		gt_node_disconnect (c);
		return;
	}

	if (!(node->rx_stack = gt_rx_stack_new (node, c, node->rx_inflated)))
	{
		if (HANDSHAKE_DEBUG)
			GT->DBGSOCK (GT, c, "error allocating rx_stack");

		gt_node_disconnect (c);
		return;
	}

	if (!(node->tx_stack = gt_tx_stack_new (c, node->tx_deflated)))
	{
		if (HANDSHAKE_DEBUG)
			GT->DBGSOCK (GT, c, "error allocating tx stack");

		gt_node_disconnect (c);
		return;
	}

	/* determine the other node's opinion of our IP address */
	node->my_ip = get_self_ip (c);

	/* determine the other ends port */
	peer_addr (c->fd, NULL, &node->peer_port);
	
	if (HANDSHAKE_DEBUG)
	{
		GT->DBGSOCK (GT, c, "self IP=[%s]", net_ip_str (node->my_ip));
		GT->DBGSOCK (GT, c, "peer port=%hu", node->peer_port);
	}

	if (!(ping = gt_packet_new (GT_PING_REQUEST, 1, NULL)))
	{
		gt_node_disconnect (c);
		return;
	}

	/* set the state as intermediately connecting and mark the node connected
	 * only when it replies to our ping */
	gt_node_state_set (node, GT_NODE_CONNECTING_2);

	/* give the connection some more time */
	gnutella_set_handshake_timeout (c, TIMEOUT_3 * SECONDS);

	/*
	 * Setup our packet handlers, for both receiving and sending packets.
	 */
	gt_rx_stack_set_handler (node->rx_stack,
	                         (GtRxStackHandler)recv_packet,
	                         (GtRxStackCleanup)cleanup_node_rx,
	                         node);

	gt_tx_stack_set_handler (node->tx_stack,
	                         (GtTxStackCleanup)cleanup_node_tx,
	                         node);

	/* send first ping */
	gt_packet_send (c, ping);
	gt_packet_free (ping);
}

/******************************************************************************/
/* MESSAGE HANDLING */

static BOOL is_pow2 (uint32_t num)
{
	return BOOL_EXPR (num > 0 && (num & (num-1)) == 0);
}

static uint32_t get_shared_size (unsigned long size_mb)
{
	uint32_t size_kb;

	size_kb = size_mb * 1024;

	if (GT_SELF->klass & GT_NODE_ULTRA)
		/* TODO: round up to nearest power of two >= 8 here */;
	else if (is_pow2 (size_kb))
		size_kb += 5; /* unmakes all powers of two, including 1 */

	return size_kb;
}

/* reply to a ping packet */
static void ping_reply_self (GtPacket *packet, TCPC *c)
{
	unsigned long  files, size_kb;
	double         size_mb;
	GtPacket      *reply;

	share_index (&files, &size_mb);
	size_kb = get_shared_size (size_mb);

	if (!(reply = gt_packet_reply (packet, GT_PING_RESPONSE)))
		return;

	gt_packet_put_port   (reply, GT_SELF->gt_port);
	gt_packet_put_ip     (reply, GT_NODE(c)->my_ip);
	gt_packet_put_uint32 (reply, (uint32_t)files);
	gt_packet_put_uint32 (reply, (uint32_t)size_kb);

	if (gt_packet_error (reply))
	{
		gt_packet_free (reply);
		return;
	}

	gt_packet_send (c, reply);
	gt_packet_free (reply);
}

/* send info about node dst to node c */
static TCPC *send_status (TCPC *c, GtNode *node, void **data)
{
	GtPacket   *pkt = (GtPacket *)   data[0];
	TCPC       *dst = (TCPC *)       data[1];
	GtPacket   *reply;

	/* don't send a ping for the node itself */
	if (c == dst)
		return NULL;

	if (!(reply = gt_packet_reply (pkt, GT_PING_RESPONSE)))
		return NULL;

	gt_packet_put_port   (reply, node->gt_port);
	gt_packet_put_ip     (reply, node->ip);
	gt_packet_put_uint32 (reply, node->files);
	gt_packet_put_uint32 (reply, node->size_kb);

	/* set the number of hops travelled to 1 */
	gt_packet_set_hops (reply, 1);

	if (gt_packet_error (reply))
	{
		gt_packet_free (reply);
		return NULL;
	}

	gt_packet_send (dst, reply);
	gt_packet_free (reply);

	return NULL;
}

static void handle_crawler_ping (GtPacket *packet, TCPC *c)
{
	void *data[2];

	data[0] = packet;
	data[1] = c;

	/* reply ourselves */
	ping_reply_self (packet, c);

	/* send pings from connected hosts */
	gt_conn_foreach (GT_CONN_FOREACH(send_status), data,
	                 GT_NODE_NONE, GT_NODE_CONNECTED, 0);
}

static int need_connections (void)
{
	BOOL am_ultrapeer;

	am_ultrapeer = GT_SELF->klass & GT_NODE_ULTRA;

	/* send a pong if we need connections, but do this
	 * only if this is a search node: leaves shouldnt send pongs */
	if (gt_conn_need_connections (GT_NODE_ULTRA) > 0 && am_ultrapeer)
		return TRUE;

	/* pretend we need connections temporarily even if we don't in order to
	 * figure out whether we are firewalled or not */
	if (gt_uptime () < 10 * EMINUTES && GT_SELF->firewalled)
		return TRUE;

	return FALSE;
}

GTLA_HANDLER (ping_request)
{
	time_t   last_ping_time, now;
	uint8_t  ttl, hops;

	now = time (NULL);

	ttl  = gt_packet_ttl (packet);
	hops = gt_packet_hops (packet);

	last_ping_time = GT_NODE(c)->last_ping_time;
	GT_NODE(c)->last_ping_time = now;

	if ((ttl == 1 && (hops == 0 || hops == 1))    /* tests if host is up */
	 || GT_NODE(c)->state == GT_NODE_CONNECTING_2 /* need to reply */
	 || need_connections ())                      /* we need connections */
	{
		ping_reply_self (packet, c);

		if (ttl == 1)
			return;
	}
	else if (ttl == 2 && hops == 0)
	{
		/* crawler ping: respond with all connected nodes */
		handle_crawler_ping (packet, c);
		return;
	}

	/* dont re-broadcast pings from search nodes if we are not one */
	if ((GT_NODE(c)->klass & GT_NODE_ULTRA) && !(GT_SELF->klass & GT_NODE_ULTRA))
	   return;

#if 0
	/* notify this host when the pong cache gets full */
	pong_cache_waiter_add (c, packet);
#endif

	/* dont accept pings too often */
	if (now - last_ping_time < 30 * ESECONDS)
		return;

#if 0
	if (!pong_cache_reply (c, packet))
	{
		/* refill the pong cache */
		pong_cache_refill ();
		return;
	}

	pong_cache_waiter_remove (c);
#endif
}

/*****************************************************************************/

GTLA_HANDLER (ping_response)
{
	uint16_t     port;
	uint32_t     ip;
	uint32_t     files;
	uint32_t     size_kb;
	GtNodeClass  klass;

	port    = gt_packet_get_port   (packet);
	ip      = gt_packet_get_ip     (packet);
	files   = gt_packet_get_uint32 (packet);
	size_kb = gt_packet_get_uint32 (packet);

	/* update stats and port */
	if (gt_packet_ttl (packet) == 1 && gt_packet_hops (packet) == 0)
	{
		/* check if this is the first ping response on this connection */
		if (GT_NODE(c)->state == GT_NODE_CONNECTING_2)
		{
			/* mark this node as now connected */
			gt_node_state_set (GT_NODE(c), GT_NODE_CONNECTED);

			/* submit the routing table */
			if ((GT_NODE(c)->klass & GT_NODE_ULTRA) &&
			    !(GT_SELF->klass & GT_NODE_ULTRA))
			{
				query_route_table_submit (c);
			}

			/* submit unfinished searches soon */
			gt_searches_submit (c, 30 * SECONDS);
		}

		if (ip == GT_NODE(c)->ip)
		{
			if (GT_NODE(c)->gt_port != port || !GT_NODE(c)->verified)
			{
				/* update the port */
				GT_NODE(c)->gt_port = port;

				/* verify the port like OpenFT does */
				gt_connect_test (GT_NODE(c), GT_NODE(c)->gt_port);
			}

			/* update stats information */
			GT_NODE(c)->size_kb = size_kb;
			GT_NODE(c)->files   = files;
		}

		return;
	}

	/*
	 * Add this node to the cache
	 */
	klass = GT_NODE_NONE;

	/* LimeWire marks ultrapeer pongs by making files size a power of two */
	if (size_kb >= 8 && is_pow2 (size_kb))
		klass = GT_NODE_ULTRA;

	/* don't register this node if its local and the peer isnt */
	if (gt_is_local_ip (ip, GT_NODE(c)->ip))
		return;

	/* keep track of stats from pongs */
	gt_stats_accumulate (ip, port, GT_NODE(c)->ip, files, size_kb);

	/* TODO: check uptime GGEP extension and add it here */
	gt_node_cache_add_ipv4 (ip, port, klass, time (NULL), 0, GT_NODE(c)->ip);
	gt_node_cache_trace ();
}

/* sent upon connection-close by some nodes */
GTLA_HANDLER (bye_request)
{
	uint16_t  code;
	char     *reason;

	code   = gt_packet_get_uint16 (packet);
	reason = gt_packet_get_str    (packet);

	/* log the message and code and be done with it */
	if (MSG_DEBUG)
	{
		GT->DBGFN (GT, "%s:%hu sent bye packet: code %hu, reason: %s",
		           net_ip_str (GT_NODE(c)->ip), GT_NODE(c)->gt_port,
		           code, reason);
	}

	/* we incur the TIME_WAIT penalty instead of the remote node if we
	 * close before they do */
	gt_node_disconnect (c);
}

/*****************************************************************************/

/* create a table for routing queries from a child node
 * disabled for now because ultrapeer mode doesnt work yet */
GTLA_HANDLER (query_route)
{
#if 0
	uint8_t   type;
	uint32_t len;
	uint8_t   largest_val;
	uint8_t   seq_no;
	uint8_t   seq_size;
	uint8_t   compressed;
	uint8_t   bits;
	size_t    size;

	GT->DBGFN (GT, "entered");

	type = gt_packet_get_uint8 (packet);

	/* TODO: rate-limit clients calling query_route; timeouts */

	switch (type)
	{
	 case 0: /* reset table */
		len         = gt_packet_get_uint32 (packet);
		largest_val	= gt_packet_get_uint8  (packet);

		if (GT_NODE(c)->query_router)
			query_router_free (GT_NODE(c)->query_router);

		GT_NODE(c)->query_router = query_router_new (len, largest_val);

		GT->DBGFN (GT, "reset table: len = %u, largest val = %u",
		           len, largest_val);
		break;

	 case 1: /* patch table */
		seq_no     = gt_packet_get_uint8 (packet);
		seq_size   = gt_packet_get_uint8 (packet);
		compressed = gt_packet_get_uint8 (packet);
		bits       = gt_packet_get_uint8 (packet);

		GT->DBGFN (GT, "patch table: seq_no=%i seq_size=%i compressed=%i bits=%i",
		           seq_no, seq_size, compressed, bits);

		/* size of the patch is the packet length minus len of patch header */
		size = gt_packet_payload_len (packet) - 5;

		GT->DBGFN (GT, "size = %u, packet->offset = %u", size, packet->offset);
		query_router_update (GT_NODE(c)->query_router, seq_no, seq_size,
		                     compressed, bits, &packet->data[packet->offset],
		                     size);
		break;

	 default:
		GT->DBGFN (GT, "unknown query-route message type: %d", type);
		break;
	}
#endif
}

static void push_connect (int fd, input_id id, TCPC *c)
{
	uint32_t  index;
	char     *str;

	if (MSG_DEBUG)
		GT->DBGFN (GT, "entered");

	if (net_sock_error (fd))
	{
		if (MSG_DEBUG)
			GT->DBGFN (GT, "error connecting back: %s", GIFT_NETERROR ());

		tcp_close (c);
		return;
	}

	/* restore the index */
	memcpy (&index, &c->udata, MIN (sizeof (c->udata), sizeof (index)));
	c->udata = NULL;

	str = stringf ("GIV %u:%s/\n\n", index, gt_guid_str (GT_SELF_GUID));

	if (MSG_DEBUG)
		GT->DBGSOCK (GT, c, "sending GIV response: %s", str);

	if (tcp_send (c, str, strlen (str)) <= 0)
	{
		if (MSG_DEBUG)
			GT->DBGFN (GT, "error sending: %s", GIFT_NETERROR ());

		tcp_close (c);
		return;
	}

	/* use this connection for something */
	input_remove (id);
	input_add (c->fd, c, INPUT_READ,
	           (InputCallback)gnutella_determine_method, TIMEOUT_DEF);
}

static void gt_giv_request (GtNode *src, uint32_t index, in_addr_t ip,
                            in_port_t port, uint8_t hops)
{
	TCPC  *c;

	if (MSG_DEBUG)
		GT->DBGFN (GT, "entered");

	/* skip request if we are busy, and they could have connected anyway */
	if (upload_availability () == 0 && !GT_SELF->firewalled)
		return;

	/* if the pushed IP address is local, forget about it */
	if (gt_is_local_ip (ip, src->ip))
		return;

	/* special case: if the node we got the push from is local
	 * and the push is from them (hops=0), don't connect to the
	 * external address but the internal */
	if (hops == 0 && gt_is_local_ip (src->ip, ip))
		ip = src->ip;

	if (!(c = tcp_open (ip, port, FALSE)))
		return;

	/* store the index directly. this way no memory is necessary */
	memcpy (&c->udata, &index, MIN (sizeof (index), sizeof (c->udata)));

	input_add (c->fd, c, INPUT_WRITE,
	           (InputCallback)push_connect, TIMEOUT_DEF);
}

GTLA_HANDLER (push_request)
{
	gt_guid_t  *client_guid;
	uint32_t    index;
	uint32_t    ip;
	uint16_t    port;
	uint8_t     hops;

	if (MSG_DEBUG)
		GT->DBGFN (GT, "entered");

	client_guid = gt_packet_get_ustr   (packet, 16);
	index       = gt_packet_get_uint32 (packet);
	ip          = gt_packet_get_ip     (packet);
	port        = gt_packet_get_port   (packet);

	hops = gt_packet_hops (packet);

	if (MSG_DEBUG)
	{
		GT->DBGSOCK (GT, c, "client_guid=%s index=%d ip=%s port=%hu",
		             gt_guid_str (client_guid), index, net_ip_str (ip), port);
	}

	if (gt_guid_cmp (client_guid, GT_SELF_GUID) == 0)
	{
		/* TODO: we should not respond if we get a lot of these */
		gt_giv_request (GT_NODE(c), index, ip, port, hops);
		return;
	}

#if 0
	if ((dst_c = push_cache_lookup (client->guid)))
		gt_route_forward_packet (dst_c, packet);
#endif
}

/*****************************************************************************/

static BOOL is_printable (const char *s)
{
	while (*s)
	{
		if (!isprint (*s))
			return FALSE;

		s++;
	}

	return TRUE;
}

static void parse_text_meta (const char *data, Dataset **meta)
{
	int      rate, freq, min, sec;
	int      n;
	char    *lower;

	if (!data)
		return;

	/* only ASCII strings are plaintext metadata */
	if (!is_printable (data))
		return;

	/* skip strings that start with "urn:", we know what those are */
	if (!strncasecmp (data, "urn:", 4))
		return;

	if (!(lower = STRDUP (data)))
		return;

	string_lower (lower);
	n = sscanf (lower, "%d kbps %d khz %d:%d", &rate, &freq, &min, &sec);

	/* try again with a slightly different format if it failed */
	if (n != 4)
		n = sscanf (lower, "%d kbps(vbr) %d khz %d:%d", &rate, &freq, &min, &sec);

	free (lower);

	if (n != 4)
	{
#if 0
		static int warned = 0;

		if (warned++ < 4)
			GT->DBGFN (GT, "unknown plaintext metadata?: %s", data);
#endif

		return;
	}

	/* XXX: actually this should be META_DEBUG */
	if (XML_DEBUG)
		GT->DBGFN (GT, "parsed %d kbps %d khz %d:%d", rate, freq, min, sec);

	dataset_insertstr (meta, "bitrate",   stringf ("%li", rate * 1000));
	dataset_insertstr (meta, "frequency", stringf ("%u", freq * 1000));
	dataset_insertstr (meta, "duration",  stringf ("%i", min * 60 + sec));
}

static void parse_extended_data (char *ext_block, gt_urn_t **r_urn,
                                 Dataset **r_meta)
{
	gt_urn_t  *urn = NULL;
	char      *ext;

	if (r_urn)
		*r_urn = NULL;
	if (r_meta)
		*r_meta = NULL;

	if (!ext_block)
		return;

	/*
	 * 0x1c is the separator character for so-called "GEM" 
	 * (Gnutella-Extension Mechanism) extensions.
	 */
	while ((ext = string_sep (&ext_block, "\x1c")))
	{
		if (string_isempty (ext))
			break;

		if (r_urn && (urn = gt_urn_parse (ext)))
		{
			free (*r_urn);
			*r_urn = urn;
		}

		if (r_meta)
		{
			parse_text_meta (ext, r_meta);
			gt_xml_parse (ext, r_meta);
		}
	}
}

static int append_result (GtPacket *packet, FileShare *file)
{
	GtShare    *share;
	Hash       *hash;

	if (!(share = share_get_udata (file, GT->name)))
		return FALSE;

	/* search results
	 * format: <index#> <file size> <file name> <extra data(include hash)> */
	gt_packet_put_uint32 (packet, share->index);
	gt_packet_put_uint32 (packet, file->size);
	gt_packet_put_str    (packet, share->filename);

	/*
	 * This is the information that goes "between the nulls" in a
	 * query hit. The first null comes after the filename.
	 *
	 * This is a bit specific and icky. It should be abstracted away.
	 */
	if ((hash = share_get_hash (file, "SHA1")))
	{
		char *sha1;

		assert (hash->len == SHA1_BINSIZE);

		if ((sha1 = sha1_string (hash->data)))
		{
			char  buf[128];
			int   len;

			/* make the hash be uppercase */
			string_upper (sha1);

			len = strlen (sha1);
			assert (len == SHA1_STRLEN);

			snprintf (buf, sizeof (buf) - 1, "urn:sha1:%s", sha1);
			len += strlen ("urn:sha1:");

			gt_packet_put_ustr (packet, buf, len);
			free (sha1);
		}
	}

	/* put the second null there */
	gt_packet_put_uint8 (packet, 0);

	if (gt_packet_error (packet))
	{
		gt_packet_free (packet);
		return FALSE;
	}

	return TRUE;
}

/* add a trailer to the packets */
static void transmit_results (TCPC *c, GtPacket *packet, uint8_t hits)
{
	GtExtendedQHD1 eqhd1 = 0;
	GtExtendedQHD2 eqhd2 = 0;

	/* set the push bit as significant */
	eqhd2 |= EQHD2_HAS_PUSH;
	/* set the busy bit as significant */
 	eqhd1 |= EQHD1_HAS_BUSY;

	/*
	 * We shouldnt mark ourselves firewalled if the destination is
	 * a local ip address and ttl == 1. However, for greater TTLs,
	 * there's no knowing if we should mark it or not...
	 */
	if (GT_SELF->firewalled)
		eqhd1 |= EQHD1_PUSH_FLAG;

	if (upload_availability () == 0)
		eqhd2 |= EQHD2_BUSY_FLAG;

	/* add the query hit descriptor
	 * <vendor id> <length> <qhd_data1> <qhd_data2> <private_data> */
	gt_packet_put_ustr   (packet, (unsigned char *) "GIFT", 4);
	gt_packet_put_uint8  (packet, 2);
	gt_packet_put_uint8  (packet, eqhd1);
	gt_packet_put_uint8  (packet, eqhd2);

	/* client identifier */
	gt_packet_put_ustr (packet, GT_SELF_GUID, 16);

	if (gt_packet_error (packet))
	{
		gt_packet_free (packet);
		return;
	}

#if 0
	GT->DBGFN (GT, "packet before twiddling result number: (will twiddle %i)", hits);
	TRACE_MEM (packet->data, packet->len);
#endif

	/* rewind the packet to the search hit count and replace the hitcount
	 * it is the first byte after the header
	 * XXX: this should use a facility of gt_packet */
	packet->data[GNUTELLA_HDR_LEN] = hits;

#if 0
	GT->DBGFN (GT, "packet after twiddling:");
	TRACE_MEM (packet->data, packet->len);
#endif

	if (LOG_RESULT_PACKETS)
		GT->dbg (GT, "transmitting %i", hits);

	/* send the reply along the path to the node that queried us */
	gt_packet_send (c, packet);
	gt_packet_free (packet);
}

static int query_request_result (TCPC *c, FileShare *file,
                                 struct _search_reply *reply)
{
	GtPacket *packet;

	/* this tells controls whether query_request_result_free() frees
	 * unrefs the FileShare */
	reply->dontfree = FALSE;

	if (!file)
	{
		/* send any remaining data */
		if (reply->packet)
			transmit_results (c, reply->packet, reply->results);

		return FALSE;
	}

	packet = reply->packet;

	if (packet)
	{
		/* send the packet if the max results per packet is reached
		 * or the size of the packet is large */
		if (reply->results == 255 || gt_packet_payload_len (packet) > 2000)
		{
			transmit_results (c, packet, reply->results);

			reply->packet  = NULL;
			reply->results = 0;

			/* handle this item again */
			reply->dontfree = TRUE;
			return TRUE;
		}

		if (append_result (packet, file))
			reply->results++;

		return FALSE;
	}

	/* allocate a packet */
	if (!(packet = gt_packet_new (GT_QUERY_RESPONSE, reply->ttl, reply->guid)))
	{
		GIFT_ERROR (("mem failure?"));
		return FALSE;
	}

	/* setup the search header */
	gt_packet_put_uint8  (packet, 0);
	gt_packet_put_port   (packet, GT_SELF->gt_port);
	gt_packet_put_ip     (packet, GT_NODE(c)->my_ip);
	gt_packet_put_uint32 (packet, 0); /* speed (kbits) */

	if (gt_packet_error (packet))
	{
		GIFT_ERROR (("failed seting up search result packet"));
		gt_packet_free (packet);
		return FALSE;
	}

	reply->packet = packet;

	/* handle this item again */
	reply->dontfree = TRUE;
	return TRUE;
#if 0
	GtPacket *reply;

	if (!(guid = gt_guid_new (guid)))
		return;

	if (!(reply = gt_packet_reply_new (packet, GT_QUERY_RESPONSE)))
	{
		free (guid)
		return;
	}

	/* host info */
	gt_packet_put_uint8  (reply, 1);                      /* number of hits */
	gt_packet_put_port   (reply, GT_SELF->gt_port);
	gt_packet_put_ip     (reply, GT_NODE(c)->my_ip);
	gt_packet_put_uint32 (reply, 1000 * 1024);            /* speed (kbits) */

	/* search results */
	gt_packet_put_uint32 (reply, 1);                      /* index */
	gt_packet_put_uint32 (reply, 10);                     /* file size */
	gt_packet_put_str    (reply, "hello.mp3");            /* file name */
	gt_packet_put_str    (reply, "urn:sha1:ABCDEFGHIJKLMNOPQRSTUVWXYZ234567");

	/* query hit descriptor */
	gt_packet_put_ustr   (reply, (unsigned char *) "GIFT", 4); /* vendor code */
	gt_packet_put_uint8  (reply, 4);                     /* public sect len */
	gt_packet_put_uint8  (reply, 0x1c);                  /* status bits */
	gt_packet_put_uint8  (reply, 0x19);                  /* more status bits */
	gt_packet_put_uint16 (reply, 0);                     /* private data */

	/* client identifier */
	gt_packet_put_ustr   (reply, guid, 16);

	/* send the reply along the path to the node that queried us */
	gt_packet_send (c, reply);
	gt_packet_free (reply);
#endif
}

static int query_request_result_free (TCPC *c, FileShare *file,
                                      struct _search_reply *reply)
{
	GtShare *share;

	if (!file)
	{
		free (reply->guid);
		free (reply);
		return FALSE;
	}

	/* the queue system calls the free func even if the queue func
	 * asked for a repeat, so we need to prevent freeing more than once.. */
	if (reply->dontfree)
		return FALSE;

	/* just a sanity check */
	if (file && !(share = share_get_udata (file, GT->name)))
		return FALSE;

	return FALSE;
}

/* This emulates the old queue interface */
static int send_result (FileShare *file, void **args)
{
	TCPC                 *c     = args[0];
	struct _search_reply *reply = args[1];

	while (query_request_result (c, file, reply))
		;

	query_request_result_free (c, file, reply);
	return TRUE;
}

static void send_results (TCPC *c, List *results,
                          struct _search_reply *reply)
{
	void *args[] = { c, reply };

	results = list_foreach_remove (results, (ListForeachFunc)send_result, args);
	assert (results == NULL);

	query_request_result (c, NULL, reply);
	query_request_result_free (c, NULL, reply);
}

static int flush_old (ds_data_t *key, ds_data_t *value, time_t *now)
{
	time_t *timestamp = value->data;

	if (*now - *timestamp >= 10 * EMINUTES)
		return DS_CONTINUE | DS_REMOVE;

	return DS_CONTINUE;
}

static BOOL flush_qcache (Dataset *cache)
{
	time_t now = time (NULL);

	dataset_foreach_ex (cache, DS_FOREACH_EX(flush_old), &now);
	return TRUE;
}

/* TODO: need to break up this file soon to isolate these things */
static BOOL query_cache_lookup (gt_guid_t *guid)
{
	static Dataset *cache = NULL;
	static timer_id timer = 0;
	time_t          now;

	if (dataset_lookup (cache, guid, GT_GUID_LEN))
		return TRUE;

	/* limit the maximum length the query cache can grow */
	if (dataset_length (cache) >= 2000)
		return FALSE;

	/*
	 * Add the guid for catching duplicates next time
	 */
	now = time (NULL);
	dataset_insert (&cache, guid, GT_GUID_LEN, &now, sizeof (now));

	if (!timer)
	{
		timer = timer_add (5 * MINUTES, (TimerCallback)flush_qcache,
		                   cache);
	}

	return FALSE;
}

GTLA_HANDLER (query_request)
{
	char         *query;
	char         *extended;
	gt_guid_t    *guid;
	gt_urn_t     *urn;
	List         *ptr;
	uint8_t       ttl;
	uint8_t       hops;
	unsigned char        *hash;
	GtQueryFlags          flags;
	GtSearchType          type;
	struct _search_reply *reply;

	flags     = gt_packet_get_uint16 (packet);
	query     = gt_packet_get_str    (packet);
	extended  = gt_packet_get_str    (packet);

	guid = gt_packet_guid (packet);

	/* don't reply if the host is firewalled and we are too */
	if ((flags & QF_HAS_FLAGS) && (flags & QF_ONLY_NON_FW) &&
	    GT_SELF->firewalled)
	{
		if (MSG_DEBUG)
			GT->dbg (GT, "not searching, flags=%04x", flags);

		return;
	}

	/* don't reply if this is our own search -- TODO: substitute a
	 * full-fledged routing table */
	if (gt_search_find (guid))
	{
		if (MSG_DEBUG)
		{
			GT->dbg (GT, "not searching, own search (guid %s)",
			         gt_guid_str (guid));
		}

		return;
	}

	/* check if we've handled this search already */
	if (query_cache_lookup (guid))
	{
		if (MSG_DEBUG)
			GT->DBGSOCK (GT, c, "duplicate search (%s)", gt_guid_str (guid));

		return;
	}

	parse_extended_data (extended, &urn, NULL);

	/* WARNING: this assumes sha1 */
	hash = gt_urn_data (urn);

	if (hash)
		type = GT_SEARCH_HASH;
	else
		type = GT_SEARCH_KEYWORD;

#if 0
	GT->DBGFN (GT, "min_speed = %hu, query = '%s', extended data = '%s'",
	           min_speed, query, extended);
#endif

	ttl  = gt_packet_ttl  (packet);
	hops = gt_packet_hops (packet);

	ptr = gt_search_exec (query, type, urn, ttl, hops);
	free (urn);

	if (!ptr)
		return;

	if (!(reply = malloc (sizeof (struct _search_reply))))
	{
		list_free (ptr);
		return;
	}

	memset (reply, 0, sizeof (struct _search_reply));

	/* set the ttl of the reply to be +1 the hops the request travelled */
	reply->ttl = gt_packet_hops (packet) + 1;

	/* use the guid of the packet in replying to results */
	reply->guid = gt_guid_dup (guid);

	send_results (c, ptr, reply);
}

/*****************************************************************************/

static void add_meta (ds_data_t *key, ds_data_t *value, FileShare *file)
{
	char *keystr   = key->data;
	char *valuestr = value->data;

	share_set_meta (file, keystr, valuestr);
}

/* parse XML data right before the client guid */
static void parse_xml_block (GtPacket *packet, size_t xml_bin_len,
                             Share **results, size_t hits)
{
	int    old_offset;
	char  *xml;

	/* if the length is one there is only the null terminator */
	if (xml_bin_len < 1)
		return;

	/* 
	 * Look for the XML before the client guid. Subtract the length of the
	 * client guid and the size of the XML.
	 */
	old_offset = gt_packet_seek (packet, packet->len - 16 - xml_bin_len);
	
	/* hopefully, if xml_bin_len is bogus this will fail */
	if (old_offset < 0)
		return;

	xml = gt_packet_get_ustr (packet, xml_bin_len);

	if (!xml)
		return;

	/* don't parse it at all if they didn't nul-terminate it */
	if (xml[xml_bin_len - 1] != 0)
		return;

	if (XML_DEBUG)
		GT->dbg (GT, "xmldata=%s", xml);

	/* 
	 * The XML before the client guid that we are parsing has a special
	 * property "index" indicating which share the XML property corresponds
	 * to, and so needs to be handled differently from XML appearing in
	 * each individual hit.
	 */
	gt_xml_parse_indexed (xml, xml_bin_len, results, hits);
}

/*
 * Attach the travelled hops as a metadata field.
 */
static void attach_hops (Share *share, int hops)
{
	char buf[12];

	if (!HOPS_AS_META)
		return;

	snprintf (buf, sizeof (buf) - 1, "%u", hops);
	share_set_meta (share, "Hops", buf);
}

void gt_query_hits_parse (GtPacket *packet, GtSearch *search,
                          TCPC *c, gt_guid_t *client_guid)
{
	uint8_t      count;
	in_port_t    port;
	in_addr_t    host;
	uint32_t     speed;
	Share       *results[255];
	uint16_t     xml_len         = 0;
	int          i, availability = 1;
	BOOL         firewalled      = FALSE;
	int          total;

	count = gt_packet_get_uint8  (packet);
	port  = gt_packet_get_port   (packet);
	host  = gt_packet_get_ip     (packet);
	speed = gt_packet_get_uint32 (packet);

	/* check if this host is banned */
	if (gt_ban_ipv4_is_banned (host))
	{
		GT->dbg (GT, "discarding search results from %s [address banned]",
		         net_ip_str (host));
		return;
	}

	for (i = 0; i < count; i++)
	{
		uint32_t       index;
		uint32_t       size;
		char          *fname, *data;
		gt_urn_t      *urn  = NULL;
		Dataset       *meta = NULL;
		Share         *file;

		index = gt_packet_get_uint32 (packet);
		size  = gt_packet_get_uint32 (packet);
		fname = gt_packet_get_str    (packet);
		data  = gt_packet_get_str    (packet);

		/* If there was an error parsing the packet (not enough results),
		 * stop parsing */
		if (gt_packet_error (packet))
			break;

		if (!fname || string_isempty (fname))
		{
			results[i] = NULL;
			continue;
		}

		parse_extended_data (data, &urn, &meta);

		/*
		 * WARNING: calling gt_urn_data here assumes sha1.
		 *
		 * TODO: this is a potential bug if gt_share_new() makes assumptions
		 * about the hash data's size and gt_urn_t changes to be multiple
		 * sizes later.
		 */
		if (!(file = gt_share_new (fname, index, size, gt_urn_data (urn))))
		{
			GIFT_ERROR (("error making fileshare, why?"));

			dataset_clear (meta);
			free (urn);

			/* make sure we find out about it if we're missing results ;) */
			assert (0);

			results[i] = NULL;
			continue;
		}

		/* HACK: set the mimetype from the file extension */
		share_set_mime (file, mime_type (fname));

		dataset_foreach (meta, DS_FOREACH(add_meta), file);

		/* Attach the hops the search travelled as a metadata field */
		attach_hops (file, gt_packet_hops (packet));

		dataset_clear (meta);
		free (urn);

		results[i] = file;
	}

	total = i;

	/* look for the query hit descriptor */
	if (!gt_packet_error (packet) &&
	    packet->len - packet->offset >= 16 + 7 /* min qhd len */)
	{
		unsigned char *vendor;
		uint8_t        eqhd_len;
		uint8_t        eqhd[2];

		vendor   = gt_packet_get_ustr  (packet, 4);
		eqhd_len = gt_packet_get_uint8 (packet);
		eqhd[0]  = gt_packet_get_uint8 (packet);
		eqhd[1]  = gt_packet_get_uint8 (packet);

		/* set availability to 0 or 1 depending on the busy flag */
		availability = ((eqhd[0] & EQHD1_HAS_BUSY) &&
		                !(eqhd[1] & EQHD2_BUSY_FLAG)) ? 1 : 0;

		/* set firewalled status based on the PUSH flag */
		firewalled   = BOOL_EXPR ((eqhd[0] & EQHD1_PUSH_FLAG) && 
		                          (eqhd[1] & EQHD2_HAS_PUSH));

		/* 
		 * Check for an XML metadata block, that is usually present
		 * when the size of the "public area" is 4
		 */
		if (eqhd_len >= 4)
			xml_len = gt_packet_get_uint16 (packet);

		if (xml_len > 0)
		{
			if (XML_DEBUG)
			{
				char str[5] = { 0 };

				if (vendor)
					memcpy (str, vendor, 4);

				GT->dbg (GT, "(%s) xml_len=%d", str, xml_len);
			}

			parse_xml_block (packet, xml_len, results, total);
		}

#if 0
		if (MSG_DEBUG)
		{
			GT->DBGFN (GT, "vendor = %s, qhd_len = %u, qhd_0 = %x, qhd_1 = %x,"
			           " availability = %i, firewalled = %i",
			           make_str (vendor, 4), qhd_len, qhd[0], qhd[1],
			           availability, firewalled);
		}
#endif
	}

	/* send the results to the interface protocol */
	for (i = 0; i < total; i++)
	{
		if (results[i])
		{
			gt_search_reply (search, c, host, port, client_guid, availability,
			                 firewalled, results[i]);

			gt_share_unref (results[i]);
		}
	}
}

/* should split this up into two routines */
GTLA_HANDLER (query_response)
{
	GtSearch   *search;
	int         save_offset;
	gt_guid_t  *client_guid;

	/* Each client has a unique identifier at the end of the
	 * packet.  Grab that first. */
	if (packet->len < 16)
	{
		if (MSG_DEBUG)
			GT->DBGSOCK (GT, c, "illegal query response packet, < 16 bytes");

		return;
	}

	/* hack the offset in the packet */
	save_offset = packet->offset;
	packet->offset = packet->len - 16;

	client_guid = gt_packet_get_ustr (packet, 16);

	/* put the offset back */
	packet->offset = save_offset;

	if (!(search = gt_search_find (gt_packet_guid (packet)))
		/*&& query_cache_lookup (packet->guid)*/)
	{
		/* TODO: support forwarding of query responses by
		 * looking up their destinations in the guid cache */

		/*gt_route_forward_packet (packet, c);*/

		/* add the client GUID to the push cache: in case of a
		 * push request we know where to send it */
		/*push_cache_add (client_guid, c);*/

		return;
	}

	gt_query_hits_parse (packet, search, c, client_guid);
}
