/*

Copyright 2010 Sebastian Kraft

This file is part of GimpLensfun.

GimpLensfun is free software: you can redistribute it and/or 
modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, either version 
3 of the License, or (at your option) any later version.

GimpLensfun is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public 
License along with GimpLensfun. If not, see 
http://www.gnu.org/licenses/.

*/

#include "lensfun.h"
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <stdio.h>
#include <math.h>
#include <iostream>
#include <algorithm>
#include <string>
#include <vector>

#include <exiv2/image.hpp>
#include <exiv2/exif.hpp>

static void query (void);
static void run   (const gchar      *name,
                   gint              nparams,
                   const GimpParam  *param,
                   gint             *nreturn_vals,
                   GimpParam       **return_vals);

static gboolean lensfun_dialog (GimpDrawable *drawable);


GtkWidget *camera_combo, *maker_combo, *lens_combo;
lfDatabase *ldb;
bool ComboBoxLock = false;

char const *CameraMakers[] = {
    "Canon",
    "Casio",
    "Fujifilm",
    "Kodak",
    "Konica",
    "Leica",
    "Nikon",
    "Olympus",
    "Panasonic",
    "Pentax",
    "Ricoh",
    "Samsung",
    "Sigma",
    "Sony",
    "NULL"
};

//##############################
// struct for storing camera/lens info 
// and parameters
//##############################
typedef struct
{
    int ModifyFlags;
    bool Inverse;
    const char *Camera;
    char *CamMaker;
    const char *Lens;
    //const char *LensMaker;
    float Scale;
    float Crop;
    float Focal;
    float Aperture;
    float Distance;
    lfLensType TargetGeom;
} MyLensfunOpts;

static MyLensfunOpts opts =
{
    LF_MODIFY_DISTORTION,
    false,
    "",
    "",
    "",
    1.0,
    0,
    0,
    0,
    1.0,
    LF_RECTILINEAR
};

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,
  NULL,
  query,
  run
};

MAIN()

static void
query (void)
{
  static GimpParamDef args[] =
  {
    {
      GIMP_PDB_INT32,
      "run-mode",
      "Run mode"
    },
    {
      GIMP_PDB_IMAGE,
      "image",
      "Input image"
    },
    {
      GIMP_PDB_DRAWABLE,
      "drawable",
      "Input drawable"
    }
  };

  gimp_install_procedure (
    "plug-in-lensfun",
    "Correct lens distortion with lensfun",
    "Correct lens distortion with lensfun",
    "Sebastian Kraft",
    "Copyright Sebastian Kraft",
    "2010",
    "_GimpLensfun...",
    "RGB",
    GIMP_PLUGIN,
    G_N_ELEMENTS (args), 0,
    args, NULL);

  gimp_plugin_menu_register ("plug-in-lensfun",
                             "<Image>/Filters/Enhance");
}






// Round float to integer value
int roundfloat2int(float d) 
{ 
  return d<0?d-.5:d+.5; 
}


//
void StrReplace(std::string& str, const std::string& old, const std::string& newstr)
{
  size_t pos = 0;
  while((pos = str.find(old, pos)) != std::string::npos)
  {
     str.replace(pos, old.length(), newstr);
     pos += newstr.length();
  }
}

//#################################
// Helper functions for printing debug output
//#################################
static void PrintMount (const lfMount *mount)
{
    g_print ("Mount: %s\n", lf_mlstr_get (mount->Name));
    if (mount->Compat)
        for (int j = 0; mount->Compat [j]; j++)
            g_print ("\tCompat: %s\n", mount->Compat [j]);
}

static void PrintCamera (const lfCamera *camera)
{
    g_print ("Camera: %s / %s %s%s%s\n",
             lf_mlstr_get (camera->Maker),
             lf_mlstr_get (camera->Model),
             camera->Variant ? "(" : "",
             camera->Variant ? lf_mlstr_get (camera->Variant) : "",
             camera->Variant ? ")" : "");
    g_print ("\tMount: %s\n", lf_db_mount_name (ldb, camera->Mount));
    g_print ("\tCrop factor: %g\n", camera->CropFactor);
}

static void PrintLens (const lfLens *lens)
{
    g_print ("Lens: %s / %s\n",
             lf_mlstr_get (lens->Maker),
             lf_mlstr_get (lens->Model));
    g_print ("\tCrop factor: %g\n", lens->CropFactor);
    g_print ("\tFocal: %g-%g\n", lens->MinFocal, lens->MaxFocal);
    g_print ("\tAperture: %g-%g\n", lens->MinAperture, lens->MaxAperture);
    g_print ("\tCenter: %g,%g\n", lens->CenterX, lens->CenterY);
    g_print ("\tCCI: %g/%g/%g\n", lens->RedCCI, lens->GreenCCI, lens->BlueCCI);
    if (lens->Mounts)
        for (int j = 0; lens->Mounts [j]; j++)
            g_print ("\tMount: %s\n", lf_db_mount_name (ldb, lens->Mounts [j]));
}

static void PrintCameras (const lfCamera **cameras)
{
    if (cameras)
        for (int i = 0; cameras [i]; i++)
        {
            g_print ("--- camera %d: ---\n", i + 1);
            PrintCamera (cameras [i]);
        }
    else
        g_print ("\t- failed\n");
}

static void PrintLenses (const lfLens **lenses)
{
    if (lenses)
        for (int i = 0; lenses [i]; i++)
        {
            g_print ("--- lens %d, score %d: ---\n", i + 1, lenses [i]->Score);
            PrintLens (lenses [i]);
        }
    else
        g_print ("\t- failed\n");
}

//######################################################################

static void dialog_set_cboxes(const char *maker, const char *camera, const char *lens) {
   
   const char *maker_cb;  
   const lfCamera *selected_cam = NULL; 

   for (int i = 0; CameraMakers[i]!="NULL"; i++)
   {
      if (strcasestr(maker, CameraMakers[i])) {
	maker_cb = CameraMakers[i];
	
	gtk_combo_box_set_active(GTK_COMBO_BOX(maker_combo), i);
      }
   }  
   
   GtkTreeModel *store = gtk_combo_box_get_model( GTK_COMBO_BOX(camera_combo) ); 
   gtk_list_store_clear( GTK_LIST_STORE( store ) );
   
   const lfCamera **cameras = ldb->FindCamerasExt (maker_cb, NULL, LF_SEARCH_LOOSE );   

   for (int i = 0; cameras [i]; i++)
   {
      gtk_combo_box_append_text( GTK_COMBO_BOX( camera_combo ), cameras[i]->Model);
      if (strcasecmp(camera, cameras[i]->Model)==0) {
	gtk_combo_box_set_active(GTK_COMBO_BOX(camera_combo), i);
	opts.CamMaker = cameras[i]->Maker;
	opts.Camera = cameras[i]->Model;
	opts.Crop = cameras[i]->CropFactor;
	selected_cam = cameras[i]; 
      }
   }     

   store = gtk_combo_box_get_model( GTK_COMBO_BOX(lens_combo) ); 
   gtk_list_store_clear( GTK_LIST_STORE( store ) );

   if (selected_cam!=NULL) {
     const lfLens **lenses = ldb->FindLenses (selected_cam, NULL, NULL);   
   
     std::vector<std::string> lensvec;
     std::vector<std::string>::const_iterator lensiter;
     for (int i = 0; lenses [i]; i++)
     {
       lensvec.push_back(lenses[i]->Model);
     }
   
     std::sort(lensvec.begin(), lensvec.end());
     for (int i = 0; i<lensvec.size(); i++)
     {
        gtk_combo_box_append_text( GTK_COMBO_BOX( lens_combo ), (lensvec[i]).c_str());
        if (strcasecmp(lens, (lensvec[i]).c_str())==0) {
	  gtk_combo_box_set_active(GTK_COMBO_BOX(lens_combo), i);
	  //opts.LensMaker = lenses[i]->Maker;
	  opts.Lens = lens;
        }
      }
   }   
}


static void
maker_cb_changed( GtkComboBox *combo,
            gpointer     data )
{
  if (!ComboBoxLock) {
    ComboBoxLock = true;
    dialog_set_cboxes(gtk_combo_box_get_active_text(GTK_COMBO_BOX(maker_combo)), 
		      "", 
		      "");
    ComboBoxLock = false;
  }
}

static void
camera_cb_changed( GtkComboBox *combo,
            gpointer     data )
{
  if (!ComboBoxLock) {
    ComboBoxLock = true;  
    dialog_set_cboxes(gtk_combo_box_get_active_text(GTK_COMBO_BOX(maker_combo)), 
		      gtk_combo_box_get_active_text(GTK_COMBO_BOX(camera_combo)), 
		      "");
    ComboBoxLock = false;
  }    
}

static void
lens_cb_changed( GtkComboBox *combo,
            gpointer     data )
{
    if (!ComboBoxLock) {
        ComboBoxLock = true;  
        dialog_set_cboxes(gtk_combo_box_get_active_text(GTK_COMBO_BOX(maker_combo)), 
		      gtk_combo_box_get_active_text(GTK_COMBO_BOX(camera_combo)), 
		      gtk_combo_box_get_active_text(GTK_COMBO_BOX(lens_combo)));
        ComboBoxLock = false;
  }    	
}

static void
focal_changed( GtkComboBox *combo,
            gpointer     data )
{
    opts.Focal = gtk_adjustment_get_value(GTK_ADJUSTMENT(data));
}

static void
scalecheck_changed( GtkCheckButton *togglebutn,
            gpointer     data )
{
    opts.Scale = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutn));   
}

static gboolean lensfun_dialog (GimpDrawable *drawable)
{
   GtkWidget *dialog;
   GtkWidget *main_vbox;
   GtkWidget *frame;
   GtkWidget *camera_label, *lens_label, *maker_label;
   GtkWidget *focal_label;
   GtkWidget *scalecheck;

   GtkWidget *spinbutton;
   GtkObject *spinbutton_adj;
   GtkWidget *frame_label;
   GtkWidget *table;
   gboolean   run;
   gimp_ui_init ("mylensfun", FALSE);

  dialog = gimp_dialog_new ("Lensfun", "mylensfun",
					   NULL, GTK_DIALOG_MODAL ,
                            gimp_standard_help_func, "plug-in-lensfun",
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK,     GTK_RESPONSE_OK,
                            NULL);
   
   main_vbox = gtk_vbox_new (FALSE, 6);
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), main_vbox);
   gtk_widget_show (main_vbox);
  
   frame = gtk_frame_new (NULL);
   gtk_widget_show (frame);
   gtk_box_pack_start (GTK_BOX (main_vbox), frame, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (frame), 6);
 
   frame_label = gtk_label_new ("Camera/Lens Parameters");
   gtk_widget_show (frame_label);
   gtk_frame_set_label_widget (GTK_FRAME (frame), frame_label);
   gtk_label_set_use_markup (GTK_LABEL (frame_label), TRUE);
 
   table = gtk_table_new(5, 2, TRUE);
   gtk_table_set_homogeneous(GTK_TABLE(table), false);
   gtk_table_set_row_spacings(GTK_TABLE(table), 2);
   gtk_table_set_col_spacings(GTK_TABLE(table), 2);

   maker_label = gtk_label_new ("Maker:");
   gtk_misc_set_alignment(GTK_MISC(maker_label),0.0,0.5);
   gtk_widget_show (maker_label);
   gtk_table_attach(GTK_TABLE(table), maker_label, 0,1,0,1, GTK_FILL, GTK_FILL, 0,0 );
 
   maker_combo = gtk_combo_box_new_text();
   gtk_widget_show (maker_combo);

   for (int i = 0; CameraMakers[i]!="NULL"; i++)
   {
      gtk_combo_box_append_text( GTK_COMBO_BOX( maker_combo ), CameraMakers[i]);
   }     
   
   gtk_table_attach_defaults(GTK_TABLE(table), maker_combo, 1,2,0,1 ); 
   
   camera_label = gtk_label_new ("Camera:");
   gtk_misc_set_alignment(GTK_MISC(camera_label),0.0,0.5);
   gtk_widget_show (camera_label);
   gtk_table_attach(GTK_TABLE(table), camera_label, 0,1,1,2, GTK_FILL, GTK_FILL, 0,0 );
   
   camera_combo = gtk_combo_box_new_text();
   gtk_widget_show (camera_combo);
   

   gtk_table_attach_defaults(GTK_TABLE(table), camera_combo, 1,2,1,2 );   

   lens_label = gtk_label_new ("Lens:");
   gtk_misc_set_alignment(GTK_MISC(lens_label),0.0,0.5);
   gtk_widget_show (lens_label);
   gtk_table_attach_defaults(GTK_TABLE(table), lens_label, 0,1,2,3 );
   
   lens_combo = gtk_combo_box_new_text();
   gtk_widget_show (lens_combo);


   gtk_table_attach_defaults(GTK_TABLE(table), lens_combo, 1,2,2,3 );     
    
   focal_label = gtk_label_new("Focal length (mm):");
   gtk_misc_set_alignment(GTK_MISC(focal_label),0.0,0.5);
   gtk_widget_show (focal_label);
   gtk_table_attach_defaults(GTK_TABLE(table), focal_label, 0,1,3,4 );   
   
   spinbutton_adj = gtk_adjustment_new (opts.Focal, 0, 2000, 1, 0, 0);
   spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton_adj), 1, 0);
   gtk_misc_set_alignment(GTK_MISC(spinbutton),0.0,0.5);
   gtk_widget_show (spinbutton);
   gtk_table_attach_defaults(GTK_TABLE(table), spinbutton, 1,2,3,4 );   
   
   gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinbutton), TRUE);   

   scalecheck = gtk_check_button_new_with_label("Scale to fit");
 //gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), FALSE);
   gtk_widget_show (scalecheck);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scalecheck), !opts.Scale);
   gtk_table_attach_defaults(GTK_TABLE(table), scalecheck, 1,2,5,6 );   
   
   
   gtk_container_add (GTK_CONTAINER (frame), table);
   //gtk_box_pack_start (GTK_BOX (frame), table, FALSE, FALSE, 6);
   
   gtk_widget_show_all(table);
   
   dialog_set_cboxes(opts.CamMaker, opts.Camera, opts.Lens);

   g_signal_connect( G_OBJECT( maker_combo ), "changed",
                      G_CALLBACK( maker_cb_changed ), NULL );     
   g_signal_connect( G_OBJECT( camera_combo ), "changed",
                      G_CALLBACK( camera_cb_changed ), NULL );  
   g_signal_connect( G_OBJECT( lens_combo ), "changed",
                      G_CALLBACK( lens_cb_changed ), NULL ); 
   g_signal_connect (spinbutton_adj, "value_changed",
		      G_CALLBACK (focal_changed), spinbutton_adj);
   
   g_signal_connect( G_OBJECT( scalecheck ), "toggled",
                      G_CALLBACK( scalecheck_changed ), NULL );    
   gtk_widget_show (dialog);

   run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);

   gtk_widget_destroy (dialog);
   return run;

  
}


static void process_image (GimpDrawable *drawable) {
  gint         i, j, channels;
  gint         x1, y1, x2, y2, imgwidth, imgheight;

  GimpPixelRgn rgn_in, rgn_out;
  guchar *ImgBuffer;
  guchar *ImgBufferOut;
  
  gimp_drawable_mask_bounds (drawable->drawable_id,
                                   &x1, &y1,
                                   &x2, &y2);
  imgwidth = x2-x1;
  imgheight = y2-y1;
  
  channels = gimp_drawable_bpp (drawable->drawable_id);

  gimp_pixel_rgn_init (&rgn_in,
                             drawable,
                             x1, y1,
                             imgwidth, imgheight, 
                             FALSE, FALSE);
  gimp_pixel_rgn_init (&rgn_out,
                             drawable,
                             x1, y1,
                             imgwidth, imgheight,
                             TRUE, TRUE);

  //Init input and output buffer
  ImgBuffer = g_new (guchar, channels * imgwidth * imgheight);  
  ImgBufferOut = g_new (guchar, channels * imgwidth * imgheight);
  
  //Read pixel data from GIMP
  gimp_pixel_rgn_get_rect (&rgn_in, ImgBuffer, x1, y1, imgwidth, imgheight);

  printf("Camera: %s, %s\n", opts.CamMaker, opts.Camera);
  printf("Lens: %s, %s\n", "", opts.Lens);
  printf("Focal Length: %f\n", opts.Focal);
  printf("F-Stop: %f\n", opts.Aperture);
  printf("Crop Factor: %f\n", opts.Crop);  
  printf("Scale: %f\n", opts.Scale);
  
  if (opts.Scale<1) {
    opts.ModifyFlags |= LF_MODIFY_SCALE;
  }
  
  const lfCamera **cameras = ldb->FindCameras (opts.CamMaker, opts.Camera);
  const lfCamera *camera = cameras[0];  
  
  const lfLens **lenses = ldb->FindLenses (camera, NULL, opts.Lens); 
  const lfLens *lens = lenses[0];


  
  lf_free(lenses);
  
  //init lensfun modifier
  lfModifier *mod = lfModifier::Create (lens, opts.Crop, imgwidth, imgheight);
  int modflags = mod->Initialize (
            lens, LF_PF_U8, opts.Focal,
            opts.Aperture, opts.Distance, opts.Scale, opts.TargetGeom,
            opts.ModifyFlags, opts.Inverse);   
 
  float *pos;
  pos = g_new (float, (imgwidth)*2*3);
  
  //main loop for processing
  for (i = 0; i < imgheight; i++)
  { //iterate through rows   
     float *src = pos;
     
     mod->ApplySubpixelGeometryDistortion (0, i, imgwidth, 1, pos);
     
     int newx, newy;

     for (j = 0; j < imgwidth*channels; j += channels)
     { //iterate through pixels in one row
        newx = roundfloat2int(src [0]);
        newy = roundfloat2int(src [1]);
        
        if ((newx>0) && (newx<imgwidth) && (newy>0) && (newy<imgheight))  {
                ImgBufferOut[(channels*imgwidth*i) + (j)  ] = ImgBuffer[ (channels*imgwidth*newy) + (newx*channels) ];
                ImgBufferOut[(channels*imgwidth*i) + (j)+1] = ImgBuffer[ (channels*imgwidth*newy) + (newx*channels) + 1 ];
                ImgBufferOut[(channels*imgwidth*i) + (j)+2] = ImgBuffer[ (channels*imgwidth*newy) + (newx*channels) + 2 ];
                //printf (" R: %d G:%d B:%d \n", ImgBuffer[ (3*imgwidth*newy) + newx ], ImgBuffer[ (3*imgwidth*newy) + newx +2 ], ImgBuffer[ (3*imgwidth*newy) + newx +2 ]);
	   } else {
                ImgBufferOut[(channels*imgwidth*i) + (j)  ] = 0;
                ImgBufferOut[(channels*imgwidth*i) + (j)+1] = 0;
                ImgBufferOut[(channels*imgwidth*i) + (j)+2] = 0;
	   }
        src += 2 * 3;     
     }
     if (i % 10 == 0) gimp_progress_update ((gdouble) (i - y1) / (gdouble) (imgheight));     
  }
  //printf("DONE\n");
  
  //write data back to gimp
  gimp_pixel_rgn_set_rect (&rgn_out, ImgBufferOut, x1, y1, imgwidth, imgheight);

  g_free(ImgBufferOut);
  g_free(ImgBuffer);

  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id,
                              x1, y1,
                              imgwidth, imgheight);    
  gimp_displays_flush ();
  gimp_drawable_detach (drawable);    
}

static void read_opts_from_exif(const char *filename) {
  
  Exiv2::Image::AutoPtr Exiv2image = Exiv2::ImageFactory::open(filename);
  Exiv2image.get();
  Exiv2image->readMetadata();
  Exiv2::ExifData &exifData = Exiv2image->exifData();

  //Search for camera
  const lfCamera **cameras = ldb->FindCameras (exifData["Exif.Image.Make"].toString().c_str(), exifData["Exif.Image.Model"].toString().c_str());
  const lfCamera *camera;
  if (cameras) {
    camera = cameras [0];
    opts.Crop = camera->CropFactor;
    opts.Camera = camera->Model;
    opts.CamMaker = camera->Maker;
  }  else {
    opts.CamMaker = new char[exifData["Exif.Image.Make"].toString().length()+1];
    strcpy(opts.CamMaker, exifData["Exif.Image.Make"].toString().c_str());
  }
  //PrintCameras(cameras, ldb);
  lf_free (cameras);  

  //Get lensID
  std::string CamMaker = exifData["Exif.Image.Make"].toString();
  std::transform(CamMaker.begin(), CamMaker.end(),CamMaker.begin(), ::tolower);
  std::string MakerNoteKey;
  
  //Select special MakerNote Tag for lens depending on Maker
  if ((CamMaker.find("pentax"))!=std::string::npos) {
    MakerNoteKey = "Exif.Pentax.LensType";
  }
  if ((CamMaker.find("canon"))!=std::string::npos) {
    MakerNoteKey = "Exif.CanonCs.LensType";
  }
  if ((CamMaker.find("minolta"))!=std::string::npos) {
    MakerNoteKey = "Exif.Minolta.LensID";
  }
  if ((CamMaker.find("nikon"))!=std::string::npos) {
    MakerNoteKey = "Exif.NikonLd3.LensIDNumber";
    if (exifData[MakerNoteKey].toString().size()==0) {
      MakerNoteKey = "Exif.NikonLd2.LensIDNumber";
    }
    if (exifData[MakerNoteKey].toString().size()==0) {
      MakerNoteKey = "Exif.NikonLd1.LensIDNumber";
    }
  }
  if ((CamMaker.find("olympus"))!=std::string::npos) {
    MakerNoteKey = "Exif.OlympusEq.LensType";
  }
  if ((CamMaker.find("sony"))!=std::string::npos) {
    //MakerNoteKey = "Exif.Sony1.LensID";
  }  
  //std::cout << MakerNoteKey << "\n";
  //std::cout << exifData[MakerNoteKey].toString() << "\n";

  const lfLens **lenses;
  const lfLens *lens;
  std::string LensNameMN;

  //Decode Lens ID
  if ((MakerNoteKey.size()>0) && (exifData[MakerNoteKey].toString().size()>0))  {
      Exiv2::ExifKey ek(MakerNoteKey);    
      Exiv2::ExifData::const_iterator md = exifData.findKey(ek);
      if (md != exifData.end()) {
	  LensNameMN = md->print(&exifData);
	  //std::cout << LensNameMN << "\n";
	  //Modify some lens names for better searching in lfDatabase
	  if ((CamMaker.find("nikon"))!=std::string::npos) {
	      StrReplace(LensNameMN, "Nikon", "");
	      StrReplace(LensNameMN, "Zoom-Nikkor", "");
	  }
      }
  }
  
  if (camera) {  
     if (LensNameMN.size()>8) {	// only take lens names with significant length
       lenses = ldb->FindLenses (camera, NULL, LensNameMN.c_str());  
     } else {
       lenses = ldb->FindLenses (camera, NULL, NULL);
     }
     if (lenses) {
       lens = lenses[0];
       opts.Lens = lens->Model;
 //    opts.LensMaker = lens->Maker;  
     }
     lf_free (lenses);
  }
  
  opts.Focal = exifData["Exif.Photo.FocalLength"].toFloat();
  opts.Aperture = exifData["Exif.Photo.ApertureValue"].toFloat();  
}  
  
static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam  values[1];
  GimpPDBStatusType status = GIMP_PDB_SUCCESS;
  gint32 imageID;
  GimpRunMode       run_mode;
  GimpDrawable     *drawable;  
  
  /* Setting mandatory output values */
  *nreturn_vals = 1;
  *return_vals  = values;

  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;

  drawable = gimp_drawable_get (param[2].data.d_drawable);
  
  gimp_progress_init ("Lensfun correction..."); 
  
  imageID = param[1].data.d_drawable;
  
  //Load lensfun database
  ldb = lf_db_new ();
  ldb->Load ();

  read_opts_from_exif(gimp_image_get_filename(imageID));
  
  /* Getting run_mode - we won't display a dialog if 
   * we are in NONINTERACTIVE mode */
  run_mode = GimpRunMode(param[0].data.d_int32);
  
  switch (run_mode)
          {
          case GIMP_RUN_INTERACTIVE:
            /* Get options last values if needed */
            //gimp_get_data ("plug-in-myblur", &bvals);

            /* Display the dialog */
            if (! lensfun_dialog (drawable))
              return;
            break;

          case GIMP_RUN_NONINTERACTIVE:
            if (nparams != 4)
              status = GIMP_PDB_CALLING_ERROR;
            if (status == GIMP_PDB_SUCCESS)
             // bvals.radius = param[3].data.d_int32;
            break;

          case GIMP_RUN_WITH_LAST_VALS:
            /*  Get options last values if needed  */
            //gimp_get_data ("plug-in-myblur", &bvals);
            break;

          default:
            break;
  }
  
  process_image(drawable);
  
}