#!/bin/bash

# Predefinitions.
EXECUTABLE="gip"
PACKAGENAME="Gip"
MAJORVERSION="1"
MINORVERSION="4.0"
RELEASE="1"
TAGVERSION=""

# Requirements.
REQUIRED_PROGS='getopt basename mkdir find pkg-config g++ intltool-update'
REQUIRED_PROGS="$REQUIRED_PROGS xgettext"
REQUIRED_LIBS='gtkmm-2.4 sigc++-2.0'

# Program files.
PROGFILES=`find src/ -name "*.cc" -o -name "*.c"`

# Installation paths. (Plugins will be installed in the LIBDIR)
INST_IN_LIBDIR=`find . -name "*.glade"`
INST_IN_PIXMAPDIR=`find src/ -name "*.png"`
INST_IN_DOCDIR=''
INST_IN_GNOMEDOCDIR=`find doc/ -name "*.xml" -o -path "doc/figures*" -maxdepth 1`

###############################################################################
#
PATH="$PATH:./installer"
ARGS=$@

## Print the usage informations out.
##
print_usage() {
  echo "Syntax: $0 [options] files" >&2
  echo -e "Options:" >&2
  echo -e "--prefix prefix  Where to install. Default is /usr/local." >&2
  echo -e "-r, --rebuild    Delete and rebuild all files." >&2
  echo -e "--install        If set, the program will be built but not" >&2
  echo -e "                 installed." >&2
}


## Builds all plugins mentioned in $PLUGINDIRS.
##
build_plugins_all() {
  PLUGINDIRS="$@"
  for PLUGINDIR in $PLUGINDIRS; do          # Compile all plugins.
    PLUGINNAME=`basename $PLUGINDIR`
    echo "####################################################################";
    echo "# Building plugin: $PLUGINNAME";
    echo "####################################################################";
    PLUGINNAMEUC=`echo $PLUGINNAME | tr [:lower:] [:upper:]`
    PLUGINFILES=`find "$PLUGINDIR" -name "*.c" -o -name "*.cc"`
    PLUGINVARNAME="PLUGIN_$PLUGINNAMEUC"
    PLUGINFILES="$PLUGINFILES ${!PLUGINVARNAME}"
    OUTFILE="$PLUGINOUTDIR/$PLUGINNAME.so"
    build_files.sh -s -c "$CFLAGS" -l "$LFLAGS" -o $OUTFILE $PLUGINFILES
    if [ "$?" -ne "0" ]; then
      echo Build failed: $PLUGINNAME plugin. >&2
      return 1
    fi
    echo
  done
  return 0
}


build_tarball() {
  ./clean.sh
  TARBASENAME="$EXECUTABLE-$MAJORVERSION.$MINORVERSION-$RELEASE"
  mkdir $TARBASENAME
  find . -type d -a ! -path "*$TARBASENAME*" -a ! -path "*CVS*" -a ! -path "." \
  | while read DIR; do
    mkdir "$TARBASENAME/$DIR" || return 1
  done
  find . ! -type d -a ! -path "*$TARBASENAME*" -a ! -path "*CVS*" \
  | while read FILE; do
    echo Copying $FILE
    cp -r "$FILE" "$TARBASENAME/$FILE" || return 1
  done
  tar czf $TARBASENAME.tar.gz $TARBASENAME
  rm -R $TARBASENAME
  return 0
}


###############################################################################
# Check prerequirements.
###############################################################################
echo "####################################################################";
echo "# Checking prerequired programs.";
echo "####################################################################";
check_programs.sh $REQUIRED_PROGS || exit 1
echo
echo "####################################################################";
echo "# Checking prerequired libraries.";
echo "####################################################################";
check_libs.sh $REQUIRED_LIBS      || exit 1
echo


###############################################################################
# Use getopt to determine the allowed options.
###############################################################################
getopt -l "prefix:,install,rebuild,dist" -- "r" $* 1>/dev/null
if [ $? != 0 ]; then
  print_usage
  exit 1
fi


###############################################################################
# This for-loop parses the command line for options and sets a variable
# accordingly.
###############################################################################
# Default options.
REBUILD=0
INSTALL=0
MAKEDIST=0
INST_PREFIX='/usr/local'

for i in "$@"; do
  case $i in
    "-r" | "--rebuild")
      REBUILD=1
      shift;;
    "--prefix")
      INST_PREFIX="$2"
      shift 2;;
    "--install")
      INSTALL=1
      shift;;
    "--dist")
      MAKEDIST=1
      shift;;
    # End of options, just command arguments left.
    "--")
      shift
      break;;
  esac
done

INST_BINDIR="$INST_PREFIX/bin/"
INST_LIBDIR="$INST_PREFIX/lib/$EXECUTABLE"
INST_LOCALEDIR="$INST_PREFIX/share/locale"
INST_PIXMAPDIR="$INST_PREFIX/lib/$EXECUTABLE"
INST_DOCDIR="$INST_PREFIX/doc/$EXECUTABLE"
INST_GNOMEDOCDIR="$INST_PREFIX/share/gnome/help/$EXECUTABLE/C"

# Preprocessor macros:
MACROS="PACKAGE_LIB_DIR=$INST_LIBDIR/\
:PACKAGE_PIXMAPS_DIR=$INST_PIXMAPDIR/\
:PACKAGE_LOCALE_DIR=$INST_LOCALEDIR/\
:PACKAGE_DOC_DIR=$INST_DOCDIR/\
:VERSION=$MAJORVERSION.$MINORVERSION-$RELEASE\
:SUBVERSION=$TAGVERSION\
:GETTEXTPACKAGE=$EXECUTABLE\
:ENABLE_NLS=1\
:ENABLE_OGG=1"

PROGRAMNAME="$PACKAGENAME $MAJORVERSION.$MINORVERSION-$RELEASE $TAGVERSION"


###############################################################################
# Build locales.
###############################################################################
echo "####################################################################";
echo "# Locales.";
echo "####################################################################";
cd po/
echo Generating locales...
./gen.sh || exit 1
cd ..
echo
INST_IN_LOCALEDIR=`find po/ -name "*.mo"`

###############################################################################
# Build a tarball distribution.
###############################################################################
if [ "$MAKEDIST" -eq 1 ]; then
  echo "####################################################################";
  echo "# Building tarball distribution.";
  echo "####################################################################";
  build_tarball || exit 1
  echo Tarball build finished.
  exit 0
fi


###############################################################################
# Cleanup.
###############################################################################
# If specified, clean up first.
if [ "$REBUILD" -eq "1" ]; then
  echo "####################################################################";
  echo "# Cleaning up, as you requested a rebuild."
  echo "####################################################################";
  ./clean.sh || exit 1
  echo
fi


###############################################################################
# Re-generate config.h.
###############################################################################
echo "####################################################################";
echo "# Defining Macros.";
echo "####################################################################";
make_config_h.sh $MACROS | tee config.h || exit 1
echo "config.h successfully generated."
echo


###############################################################################
# Build.
###############################################################################
# Compiler/Linker options.
CFLAGS="-DHAVE_CONFIG_H -I. -I.. `pkg-config $REQUIRED_LIBS --cflags`"
LFLAGS=`pkg-config $REQUIRED_LIBS --libs`

build_plugins_all $PLUGINDIRS || exit 1
INST_IN_LIBDIR="$INST_IN_LIBDIR `find "$PLUGINOUTDIR" -name "*.so"`"

echo "####################################################################";
echo "# Building the program core.";
echo "####################################################################";
build_files.sh -o "$EXECUTABLE" -c "$CFLAGS" -l "$LFLAGS" $PROGFILES || exit 1
echo

echo $PROGRAMNAME compilation finished successfully!
echo


###############################################################################
# Install.
###############################################################################
if [ "$INSTALL" -ne "1" ]; then
  echo For program installation run $0 $ARGS --install
  exit 0
fi

echo "####################################################################";
echo "# Installation in $INST_PREFIX";
echo "####################################################################";
if [ ! -d "$INST_PREFIX" ]; then
  echo "Installation prefix directory $INST_PREFIX does not exist!" >&2
  echo "Use \"--prefix prefix\" to specify an existing directory." >&2
  echo "Installation aborted." >&2
  exit 1
fi
echo Creating directories...
CREATEDIRS="$INST_BINDIR $INST_LIBDIR $INST_PIXMAPDIR $INST_DOCDIR"
CREATEDIRS="$CREATEDIRS $INST_LOCALEDIR $INST_GNOMEDOCDIR"
make_installdirs.sh $CREATEDIRS                       || exit 1

echo Installing $EXECUTABLE...
cp $EXECUTABLE $INST_BINDIR                           || exit 1

echo Installing libraries in $INST_LIBDIR...
if [ "$INST_IN_LIBDIR" != "" ]; then
  cp $INST_IN_LIBDIR $INST_LIBDIR                     || exit 1
fi

echo Installing pixmaps in $INST_PIXMAPDIR...
if [ "$INST_IN_PIXMAPDIR" != "" ]; then
  cp $INST_IN_PIXMAPDIR $INST_PIXMAPDIR               || exit 1
fi

echo Installing documentation in $INST_DOCDIR...
if [ "$INST_IN_DOCDIR" != "" ]; then
  cp $INST_IN_DOCDIR $INST_DOCDIR                     || exit 1
fi

echo Installing GNOME documentation in $INST_GNOMEDOCDIR...
if [ -d "$INST_GNOMEDOCDIR" -a -w "$INST_GNOMEDOCDIR" ]; then
  cp -r "$INST_IN_GNOMEDOCDIR" "$INST_GNOMEDOCDIR"
else
  echo "WARNING: GNOME documentation directory not found or it is not"  >&2
  echo "         writeable! Skipping GNOME documentation installation." >&2
fi

echo Installing localisation files in $INST_LOCALEDIR...
if [ "$INST_IN_LOCALEDIR" != "" ]; then
  install_locales.sh -p $INST_LOCALEDIR -d $EXECUTABLE $INST_IN_LOCALEDIR \
   || exit 1
fi

echo
echo $PROGRAMNAME has been installed successfully!
echo If you find any bugs, please report to the mailinglist at
echo "<gip-devel gna org>".
echo

exit 0
