#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <getopt.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>

#include "gkdial.h"
#include "defines.h"
#include "details.h"
#include "prefs.h"
#include "peers_list.h"
#include "connedit.h"
#include "main_loop.h"
#include "images.h"
#include "util.h"
#include "tray.h"

struct option long_opts[] = {
  /*
   * { name  has_arg  *flag  val } 
   */
  {"help", no_argument, NULL, 'h'},
  {"edit", required_argument, NULL, 'e'},
  {"add", no_argument, NULL, 'a'},
  {"remove", required_argument, NULL, 'r'},
  {0, 0, 0, 0}
};

void
gkdial_show_about (GtkWidget * widget, gpointer data)
{
  gchar *translator_credits = g_strdup (_("translator_credits"));

  gk_dialog (GTK_MESSAGE_INFO,
	     _("GkDial %s\n\n"
	       "A PPP setup and dialing tool for GNOME.\n\n"
	       "Author:\n\n"
	       "\tGustavo Noronha Silva <kov@debian.org>\n\n"
	       "Thanks to:\n\n"
	       "\tLeandro A. F. Pereira <leandro@linuxmag.com.br>\n\n"
	       "%s"), PACKAGE_VERSION, translator_credits);

  g_free (translator_credits);
}

int
gkdial_defaults_init (GkDial * gkdial)
{
  /*
   * Connection status and info stuff 
   */
  gkdial->first_time = TRUE;	/* used on the check_connection loop */
  gkdial->was_gkdial = FALSE;	/* true if pppd is started by other means */
  gkdial->lock_file = g_strdup (LOCK_DEFAULT);
  gkdial->state = DISCONNECTED;
  gkdial->state_locked = FALSE;

  gkdial->num_retries = 0;

  gkdial->conf_client = gconf_client_get_default ();
  gconf_client_add_dir (gkdial->conf_client, "/apps/gkdial",
			GCONF_CLIENT_PRELOAD_RECURSIVE, NULL);
  gconf_client_notify_add (gkdial->conf_client, "/apps/gkdial",
			   gkdial_conf_changed, gkdial, NULL, NULL);

  /*
   * open ip socket to get ppp info 
   */
  if ((gkdial->ip_socket = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
    {
      gk_dialog (GTK_MESSAGE_ERROR, _("could not open an ip socket"));
      return 1;
    }

  /*
   * tray icon stuff 
   */
  gkdial->menu_item_connect = NULL;
  gkdial->menu_item_disconnect = NULL;
  gkdial->menu_item_prefs = NULL;

  /*
   * load the icons for the tray 
   */
  gkdial->pixbuf_ppp_none = pixbuf_from_file ("ppp_none_16x16.png");
  gkdial->pixbuf_ppp_both = pixbuf_from_file ("ppp_both_16x16.png");
  gkdial->pixbuf_ppp_in = pixbuf_from_file ("ppp_in_16x16.png");
  gkdial->pixbuf_ppp_out = pixbuf_from_file ("ppp_out_16x16.png");
  gkdial->pixbuf_ppp_off = pixbuf_from_file ("ppp_off_16x16.png");
  gkdial->pixbuf_ppp_trying = pixbuf_from_file ("ppp_trying_16x16.png");

  /*
   * GUI stuff 
   */
  gkdial->main_window = glade_xml_get_widget (gkdial->gui, "main_window");
  gkdial->main_combo = glade_xml_get_widget (gkdial->gui, "main_combo");

  gkdial->menu_new = glade_xml_get_widget (gkdial->gui, "menu_new");
  gkdial->menu_quit = glade_xml_get_widget (gkdial->gui, "menu_quit");

  gkdial->menu_properties = glade_xml_get_widget (gkdial->gui,
						  "menu_properties");
  gkdial->menu_preferences = glade_xml_get_widget (gkdial->gui,
						   "menu_preferences");

  gkdial->menu_about = glade_xml_get_widget (gkdial->gui, "menu_about");

  /*
   * connection handling buttons 
   */
  gkdial->button_connect = glade_xml_get_widget (gkdial->gui,
						 "button_connect");
  gkdial->button_disconnect = glade_xml_get_widget (gkdial->gui,
						    "button_disconnect");

  gkdial->label_set_time = glade_xml_get_widget (gkdial->gui,
						 "label_set_time");
  gkdial->label_set_in = glade_xml_get_widget (gkdial->gui, "label_set_in");
  gkdial->label_set_out = glade_xml_get_widget (gkdial->gui, "label_set_out");

  /*
   * details window 
   */
  gkdial->details_window = glade_xml_get_widget (gkdial->gui,
						 "details_window");
  gkdial->image_details_ppp = glade_xml_get_widget (gkdial->gui,
						    "image_details_ppp");
  gkdial->details_statusbar = glade_xml_get_widget (gkdial->gui,
						    "details_statusbar");
  /*
   * statusbar ids 
   */
  gkdial->status_state_id =
    gtk_statusbar_get_context_id (GTK_STATUSBAR
				  (gkdial->details_statusbar),
				  "current state of the connection");

  /*
   * load the icons for the details window 
   */
  gkdial->pixbuf_ppp_none_64 = pixbuf_from_file ("ppp_none_64x64.png");
  gkdial->pixbuf_ppp_both_64 = pixbuf_from_file ("ppp_both_64x64.png");
  gkdial->pixbuf_ppp_in_64 = pixbuf_from_file ("ppp_in_64x64.png");
  gkdial->pixbuf_ppp_out_64 = pixbuf_from_file ("ppp_out_64x64.png");
  gkdial->pixbuf_ppp_off_64 = pixbuf_from_file ("ppp_off_64x64.png");
  gkdial->pixbuf_ppp_trying_64 = pixbuf_from_file ("ppp_trying_64x64.png");
  gtk_image_set_from_pixbuf (GTK_IMAGE (gkdial->image_details_ppp),
			     gkdial->pixbuf_ppp_off_64);

  /*
   * set image for the main window 
   */
  gkdial->main_image = glade_xml_get_widget (gkdial->gui, "main_image");
  gtk_image_set_from_pixbuf (GTK_IMAGE (gkdial->main_image),
			     gkdial->pixbuf_ppp_both_64);

  /*
   * preferences window 
   */
  gkdial->prefs_window = glade_xml_get_widget (gkdial->gui, "prefs_window");

  gkdial->conf_locked = FALSE;

  /*
   * accounts 
   */
  gkdial->prefs_peers_list =
    GTK_TREE_VIEW (glade_xml_get_widget (gkdial->gui, "prefs_peers_list"));
  gkdial->button_peer_add =
    glade_xml_get_widget (gkdial->gui, "button_peer_add");
  gkdial->button_peer_edit =
    glade_xml_get_widget (gkdial->gui, "button_peer_edit");
  gkdial->button_peer_remove =
    glade_xml_get_widget (gkdial->gui, "button_peer_remove");

  /*
   * connection properties 
   */
  gkdial->connedit_window = glade_xml_get_widget (gkdial->gui,
						  "connedit_window");

  gkdial->cp_button_cancel = glade_xml_get_widget (gkdial->gui,
						   "cp_button_cancel");
  gkdial->cp_button_save = glade_xml_get_widget (gkdial->gui,
						 "cp_button_save");

  gkdial->cp_entry_conn_name = glade_xml_get_widget (gkdial->gui,
						     "cp_entry_conn_name");
  g_object_set_data (G_OBJECT (gkdial->cp_entry_conn_name),
		     "name", "conn_name");
  gkdial->cp_entry_phone = glade_xml_get_widget (gkdial->gui,
						 "cp_entry_phone");
  g_object_set_data (G_OBJECT (gkdial->cp_entry_phone), "name", "phone");
  gkdial->cp_entry_user = glade_xml_get_widget (gkdial->gui, "cp_entry_user");
  g_object_set_data (G_OBJECT (gkdial->cp_entry_user), "name", "user");
  gkdial->cp_entry_password = glade_xml_get_widget (gkdial->gui,
						    "cp_entry_password");
  g_object_set_data (G_OBJECT (gkdial->cp_entry_password),
		     "name", "password");
  gkdial->cp_entry_auth = glade_xml_get_widget (gkdial->gui, "cp_entry_auth");
  g_object_set_data (G_OBJECT (gkdial->cp_entry_auth), "name", "auth");

  /*
   * GkConnection 
   */
  gkdial->connection = g_new (GkConnection, 1);
  gkdial->connection_backup = g_new (GkConnection, 1);

  void nullify_connection (GkConnection * conn)
  {
    conn->name = NULL;
    conn->phone = NULL;
    conn->user = NULL;
    conn->password = NULL;
    conn->auth_type = NULL;
  }
  nullify_connection (gkdial->connection);
  nullify_connection (gkdial->connection_backup);

  /*
   * general 
   */
  /*
   * dialing 
   */
  gkdial->check_retry_when_dialing =
    GTK_TOGGLE_BUTTON (glade_xml_get_widget (gkdial->gui,
					     "check_retry_when_dialing"));
  g_object_set_data (G_OBJECT (gkdial->check_retry_when_dialing),
		     "check", "retry_when_dialing");

  gkdial->spinb_times_to_retry =
    glade_xml_get_widget (gkdial->gui, "spinb_times_to_retry");

  gkdial->check_reconn_if_lost =
    GTK_TOGGLE_BUTTON (glade_xml_get_widget (gkdial->gui,
					     "check_reconn_if_lost"));
  g_object_set_data (G_OBJECT (gkdial->check_reconn_if_lost),
		     "check", "reconn_if_lost");

  /*
   * main window 
   */
  gkdial->check_mw_starts_hiden =
    GTK_TOGGLE_BUTTON (glade_xml_get_widget (gkdial->gui,
					     "check_mw_starts_hiden"));
  g_object_set_data (G_OBJECT (gkdial->check_mw_starts_hiden),
		     "check", "mw_starts_hiden");

	gkdial->check_quit_on_off =
		GTK_TOGGLE_BUTTON (glade_xml_get_widget (gkdial->gui,
							"check_quit_on_off"));
	g_object_set_data (G_OBJECT (gkdial->check_quit_on_off),
				"check", "quit_on_off");

	/*
	 *	details window
	 */
	gkdial->check_hide_dw_on_conn =
		GTK_TOGGLE_BUTTON (glade_xml_get_widget (gkdial->gui,
			"check_hide_dw_on_conn"));
	g_object_set_data (G_OBJECT (gkdial->check_hide_dw_on_conn),
		"check", "hide_dw_on_conn");

  /*
   * tray 
   */
  gkdial->check_always_tray =
    GTK_TOGGLE_BUTTON (glade_xml_get_widget (gkdial->gui,
					     "check_always_tray"));
  g_object_set_data (G_OBJECT (gkdial->check_always_tray),
		     "check", "always_tray");


  /*
   * close prefs window 
   */
  gkdial->button_prefs_close = glade_xml_get_widget (gkdial->gui,
						     "button_prefs_close");

  /*
   * connecting stuff 
   */
  gkdial->connecting_thread = NULL;
  gkdial->connected_to = NULL;
  gkdial->command = NULL;
  gkdial->return_code = -1;

  return 0;
}

void
gkdial_connect_signals (GkDial * gkdial)
{
  /*
   * main windows 
   */
  g_signal_connect (G_OBJECT (gkdial->main_window), "delete_event",
		    G_CALLBACK (gtk_main_quit), NULL);

  g_signal_connect (G_OBJECT (gkdial->menu_new), "activate",
		    G_CALLBACK (gkdial_connadd_show), gkdial);
  g_signal_connect (G_OBJECT (gkdial->menu_quit), "activate",
		    G_CALLBACK (gtk_main_quit), NULL);

  /*
   * FIXME!: needs to select the connection on the peers_list, based on
   * the combo selection
   * 
   * g_signal_connect (G_OBJECT(gkdial->menu_properties), "activate",
   * G_CALLBACK(gkdial_connedit_menu_show), gkdial); 
   */
  g_signal_connect (G_OBJECT (gkdial->menu_preferences), "activate",
		    G_CALLBACK (gkdial_prefs_show), gkdial);

  g_signal_connect (G_OBJECT (gkdial->menu_about), "activate",
		    G_CALLBACK (gkdial_show_about), gkdial);

  /*
   * connection handling buttons 
   */
  g_signal_connect (G_OBJECT (gkdial->button_connect), "clicked",
		    G_CALLBACK (gk_connect), gkdial);
  g_signal_connect (G_OBJECT (gkdial->button_disconnect), "clicked",
		    G_CALLBACK (gk_disconnect), gkdial);

  /*
   * details window 
   */
  g_signal_connect (G_OBJECT (gkdial->details_window), "delete-event",
		    G_CALLBACK (gkdial_details_delete), gkdial);

  /*
   * preferences window 
   */
  g_signal_connect (G_OBJECT (gkdial->prefs_window), "delete-event",
		    G_CALLBACK (gkdial_prefs_delete), gkdial);

  g_signal_connect (G_OBJECT (gkdial->button_prefs_close), "clicked",
		    G_CALLBACK (gkdial_prefs_close), gkdial);

  /*
   * accounts 
   */
  g_signal_connect (G_OBJECT (gkdial->button_peer_add), "clicked",
		    G_CALLBACK (gkdial_connadd_show), gkdial);
  g_signal_connect (G_OBJECT (gkdial->button_peer_edit), "clicked",
		    G_CALLBACK (gkdial_connedit_show), gkdial);
  g_signal_connect (G_OBJECT (gkdial->button_peer_remove), "clicked",
		    G_CALLBACK (gkdial_connection_remove), gkdial);

  /*
   * connection 
   */
  g_signal_connect (G_OBJECT (gkdial->connedit_window), "delete-event",
		    G_CALLBACK (gkdial_connedit_delete), gkdial);

  g_signal_connect (G_OBJECT (gkdial->cp_button_cancel), "clicked",
		    G_CALLBACK (gkdial_connedit_close), gkdial);
  g_signal_connect (G_OBJECT (gkdial->cp_button_save), "clicked",
		    G_CALLBACK (gkdial_connedit_save), gkdial);

  /*
   * conn properties 
   */
  g_signal_connect (G_OBJECT (gkdial->cp_entry_conn_name), "changed",
		    G_CALLBACK (cp_entry_changed_cb), gkdial);
  g_signal_connect (G_OBJECT (gkdial->cp_entry_phone), "changed",
		    G_CALLBACK (cp_entry_changed_cb), gkdial);
  g_signal_connect (G_OBJECT (gkdial->cp_entry_user), "changed",
		    G_CALLBACK (cp_entry_changed_cb), gkdial);
  g_signal_connect (G_OBJECT (gkdial->cp_entry_password), "changed",
		    G_CALLBACK (cp_entry_changed_cb), gkdial);
  g_signal_connect (G_OBJECT (gkdial->cp_entry_auth), "changed",
		    G_CALLBACK (cp_entry_changed_cb), gkdial);

  /*
   * general 
   */
  g_signal_connect (G_OBJECT (gkdial->check_retry_when_dialing),
		    "toggled", G_CALLBACK (check_changed_cb), gkdial);
  g_signal_connect (G_OBJECT (gkdial->spinb_times_to_retry),
		    "value-changed", G_CALLBACK (spinb_changed_cb), gkdial);
  g_signal_connect (G_OBJECT (gkdial->check_reconn_if_lost), "toggled",
		    G_CALLBACK (check_changed_cb), gkdial);

  g_signal_connect (G_OBJECT (gkdial->check_mw_starts_hiden),
		    "toggled", G_CALLBACK (check_changed_cb), gkdial);
	g_signal_connect (G_OBJECT (gkdial->check_quit_on_off),
				"toggled", G_CALLBACK (check_changed_cb), gkdial);

	g_signal_connect (G_OBJECT (gkdial->check_hide_dw_on_conn),
				"toggled", G_CALLBACK (check_changed_cb), gkdial);

  g_signal_connect (G_OBJECT (gkdial->check_always_tray),
		    "toggled", G_CALLBACK (check_changed_cb), gkdial);
}

int
main (int argc, char *argv[])
{
  GkDial gkdial;

  /*
   * edit a connection only, if TRUE 
   */
  gkdial.add_only = FALSE;
  gkdial.edit_only = FALSE;
  gkdial.remove_only = FALSE;
  char c;

  /*
   * it seems like poff kills gkdial together with it's children, so
   * here'a signal handling to make it work 
   */
  signal (SIGTERM, SIG_IGN);

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_set_locale ();

  gnome_program_init (PACKAGE, VERSION,
		      LIBGNOMEUI_MODULE, argc, argv, GNOME_PARAM_NONE);

  glade_init ();

  gkdial.gui =
    glade_xml_new (PACKAGE_DATA_DIR "/gkdial/glade/gkdial.glade", NULL, NULL);

  if (gkdial_defaults_init (&gkdial))
    return 1;

  /*
   * inits the basic needed stuff to start working 
   */
  gkdial_connect_signals (&gkdial);
  gkdial_conf_init (&gkdial);
  gkdial_peers_list_init (&gkdial);

  while ((c = getopt_long (argc, argv, "?he:ar:", long_opts, NULL)) != EOF)
    {
      switch (c)
	{
	case 'h':
	  /*
	   * FIXME: should show some useful help? 
	   */
	  exit (0);
	  break;
	case 'e':
	  gkdial.edit_only = TRUE;
	  gkdial.connection->name = conv_to_utf8 (optarg);
	  break;
	case 'a':
	  gkdial.add_only = TRUE;
	  break;
	case 'r':
	  gkdial.remove_only = TRUE;
	  gkdial.connection->name = conv_to_utf8 (optarg);
	  break;
	}
    }

  if (gkdial.add_only)
    {
      gkdial_connadd_show (NULL, &gkdial);
      gtk_main ();
      return 0;
    }

  if (gkdial.edit_only || gkdial.remove_only)
    {
      /*
       * gkdial is being called just to edit or remove a connection. It 
       * knows what peer it's going to edit/remove by looking at the
       * tree's selection 
       */
      GtkTreeIter *iter;

      iter = prefs_peers_list_find (&gkdial, gkdial.connection->name);
      prefs_peers_list_select (&gkdial, iter);
      gtk_tree_iter_free (iter);

      if (gkdial.edit_only)
	{
	  gkdial_connedit_show (NULL, &gkdial);
	  gtk_main ();
	}
      else
	gkdial_connection_remove (NULL, &gkdial);

      return 0;
    }

  gkdial_tray_init (&gkdial);

  if (!gkdial.mw_starts_hiden && !GTK_WIDGET_VISIBLE (gkdial.main_window))
    gtk_widget_show (gkdial.main_window);
  else
    {
      if (gkdial.tray)
	gtk_widget_show_all (gkdial.tray);
      else
	gtk_widget_show (gkdial.main_window);
    }

  gkdial.check_timeout = gtk_timeout_add (CHK_TIMEOUT, check_connection,
					  &gkdial);

  gtk_main ();
  return 0;
}
