#include <stdio.h>

#include <gtk/gtk.h>

#include "gkdial.h"
#include "defines.h"
#include "prefs.h"
#include "peers_list.h"

void
gkdial_prefs_show (GtkWidget * w, gpointer data)
{
  GkDial *gkdial = (GkDial *) data;

  gtk_widget_show (gkdial->prefs_window);

  if (gkdial->tray)
    gtk_widget_set_sensitive (gkdial->menu_item_prefs, FALSE);
  gtk_widget_set_sensitive (gkdial->menu_preferences, FALSE);
}

void
gkdial_prefs_delete (GtkWidget * w, GdkEventAny * event, gpointer data)
{
  gkdial_prefs_close (w, data);
}

void
gkdial_prefs_close (GtkWidget * w, gpointer data)
{
  GkDial *gkdial = (GkDial *) data;

  if (gkdial->tray)
    gtk_widget_set_sensitive (gkdial->menu_item_prefs, TRUE);
  gtk_widget_set_sensitive (gkdial->menu_preferences, TRUE);
  gtk_widget_hide (gkdial->prefs_window);
}

void
gkdial_prefs_buttons_set_sensitive (GkDial * gkdial, gboolean sensitive)
{
  gtk_widget_set_sensitive (gkdial->button_peer_add, sensitive);
  gtk_widget_set_sensitive (gkdial->menu_new, sensitive);
  gtk_widget_set_sensitive (gkdial->button_peer_edit, sensitive);
  gtk_widget_set_sensitive (gkdial->menu_properties, sensitive);
  gtk_widget_set_sensitive (gkdial->button_peer_remove, sensitive);
}

void
gkdial_conf_init (GkDial * gkdial)
{
  /*
   * retry_when_dialing / reconn_if_lost 
   */
  {
    gboolean value;

	value = gkdial_conf_get_bool (gkdial, "preferences/retry_when_dialing");
	gtk_toggle_button_set_active (gkdial->check_retry_when_dialing, 
		value);

    gkdial->max_retries = 0;
    gkdial->max_retries = gkdial_conf_get_float (gkdial,
						 "preferences/max_retries");
    if (gkdial->max_retries)
      gtk_spin_button_set_value (GTK_SPIN_BUTTON
				 (gkdial->spinb_times_to_retry),
				 gkdial->max_retries);

	value = gkdial_conf_get_bool (gkdial, "preferences/reconn_if_lost");
	gtk_toggle_button_set_active (gkdial->check_reconn_if_lost, value);

   gkdial->mw_starts_hiden =
     gkdial_conf_get_bool (gkdial, "preferences/mw_starts_hiden");
   gtk_toggle_button_set_active (gkdial->check_mw_starts_hiden, 
		gkdial->mw_starts_hiden);

	gkdial->quit_on_off =
		gkdial_conf_get_bool (gkdial, "preferences/quit_on_off");
	gtk_toggle_button_set_active (gkdial->check_quit_on_off, 
		gkdial->quit_on_off);

	gkdial->hide_dw_on_conn =
		gkdial_conf_get_bool (gkdial, "preferences/hide_dw_on_conn");
	gtk_toggle_button_set_active (gkdial->check_hide_dw_on_conn,
		gkdial->hide_dw_on_conn);	
	
	gkdial->always_tray =
		gkdial_conf_get_bool (gkdial, "preferences/always_tray");
	gtk_toggle_button_set_active (gkdial->check_always_tray, 
		gkdial->always_tray);
  }

  /*
   * checks for consistency 
   */

  /*
   * if mw is not going to be shown, tray must be there! 
   */
  if (gkdial->mw_starts_hiden && !gkdial->always_tray)
    gkdial_conf_set_bool (gkdial, "preferences/always_tray", TRUE);
}

void
gkdial_conf_changed (GConfClient * client, guint cnxn_id,
		     GConfEntry * entry, gpointer data)
{
  GkDial *gkdial = (GkDial *) data;
  const gchar *key;
  GConfValue *value;

  g_return_if_fail (client != NULL);

  key = gconf_entry_get_key (entry);
  value = gconf_entry_get_value (entry);

#ifdef DEBUG
  printf ("(gkdial_conf_changed): %s\n", key);
#endif

  if (!strcmp (key, "/apps/gkdial/preferences/default_peer"))
    {
      GtkTreeIter *iter;
      const gchar *value_str;

#ifdef DEBUG
      printf ("(gkdial_conf_changed): default_peer changed!\n");
      fflush (stdout);
#endif
      value_str = gconf_value_get_string (value);
      iter = prefs_peers_list_find (gkdial, value_str);

      if (iter)
	{

	  default_toggle_set_peers_list (gkdial, iter);
	  gtk_tree_iter_free (iter);
	}
    }
  else if (!strcmp (key, "/apps/gkdial/preferences/retry_when_dialing"))
    {
      gboolean value_bool;

      value_bool = gkdial_conf_get_bool (gkdial,
					 "preferences/retry_when_dialing");
      gtk_toggle_button_set_active (gkdial->check_retry_when_dialing,
				    value_bool);

#ifdef DEBUG
      printf ("(gkdial_conf_changed): retry_when_dialing changed"
	      " to %d!\n", value_bool);
      fflush (stdout);
#endif
    }
  else if (!strcmp (key, "/apps/gkdial/preferences/max_retries"))
    {
      gkdial->max_retries = 0;
      gkdial->max_retries = gkdial_conf_get_float (gkdial,
						   "preferences/max_retries");

      gtk_spin_button_set_value (GTK_SPIN_BUTTON
				 (gkdial->spinb_times_to_retry),
				 gkdial->max_retries);
    }
  else if (!strcmp (key, "/apps/gkdial/preferences/retry_when_dialing"))
    {
      gboolean value_bool;

      value_bool = gkdial_conf_get_bool (gkdial,
					 "preferences/reconn_if_lost");
      gtk_toggle_button_set_active (gkdial->check_reconn_if_lost, value_bool);

#ifdef DEBUG
      printf ("(gkdial_conf_changed): reconn_if_lost changed"
	      " to %d!\n", value_bool);
      fflush (stdout);
#endif
    }
  else if (!strcmp (key, "/apps/gkdial/preferences/mw_starts_hiden"))
    {
      gkdial->mw_starts_hiden =
	gkdial_conf_get_bool (gkdial, "preferences/mw_starts_hiden");

      gtk_toggle_button_set_active (gkdial->check_mw_starts_hiden,
				    gkdial->mw_starts_hiden);

      if (gkdial->mw_starts_hiden && !gkdial->always_tray)
	{
	  gk_dialog (GTK_MESSAGE_INFO,
		     _("This option can only be activated "
		       "if <i>Always show tray icon</i> is checked.\n\n"
		       "I'm going to check it for you."));

	  gkdial_conf_set_bool (gkdial, "preferences/always_tray", TRUE);
	}
    }
   else if (!strcmp (key, "/apps/gkdial/preferences/quit_on_off"))
   {
   	gkdial->quit_on_off =
   		gkdial_conf_get_bool (gkdial, "preferences/quit_on_off");
   		
   	gtk_toggle_button_set_active (gkdial->check_quit_on_off,
   		gkdial->quit_on_off);
   }
   else if (!strcmp (key, "/apps/gkdial/preferences/hide_dw_on_conn"))
   {
   	gkdial->hide_dw_on_conn =
   		gkdial_conf_get_bool (gkdial, "preferences/hide_dw_on_conn");
   		
   	gtk_toggle_button_set_active (gkdial->check_hide_dw_on_conn,
   		gkdial->hide_dw_on_conn);
   }
  else if (!strcmp (key, "/apps/gkdial/preferences/always_tray"))
    {
      gkdial->always_tray =
	gkdial_conf_get_bool (gkdial, "preferences/always_tray");

      gtk_toggle_button_set_active (gkdial->check_always_tray,
				    gkdial->always_tray);

      if (!gkdial->always_tray && gkdial->state != CONNECTED)
	{
	  if (gkdial->mw_starts_hiden)
	    {
	      gk_dialog (GTK_MESSAGE_INFO,
			 _("This option needs to be on for the <i>Main "
			   "window is hidden when the program starts</i> "
			   "to work.\n\n"
			   "I'm going to uncheck that option."));

	      gkdial_conf_set_bool (gkdial,
				    "preferences/mw_starts_hiden", FALSE);
	    }
	  gtk_widget_hide (gkdial->tray);

	  if (gkdial->state == CONNECTING)
	    gtk_widget_show (gkdial->details_window);
	  else
	    gtk_widget_show (gkdial->main_window);
	}
      else
	gtk_widget_show_all (gkdial->tray);
    }
}

gchar *
gkdial_conf_get_string (GkDial * gkdial, const gchar * key)
{
  gchar *value;
  gchar *real_key = g_strdup_printf ("/apps/gkdial/%s",
				     key);

  value = gconf_client_get_string (gkdial->conf_client, real_key, NULL);
  g_free (real_key);

  return value;
}

void
gkdial_conf_set_string (GkDial * gkdial, const gchar * key,
			const gchar * value)
{
  gchar *real_key = g_strdup_printf ("/apps/gkdial/%s",
				     key);

  gconf_client_set_string (gkdial->conf_client, real_key, value, NULL);

  g_free (real_key);
}

gboolean
gkdial_conf_get_bool (GkDial * gkdial, const gchar * key)
{
  GError *err = NULL;
  gchar *real_key;
  gboolean value;

  real_key = g_strdup_printf ("/apps/gkdial/%s", key);
  value = gconf_client_get_bool (gkdial->conf_client, real_key, &err);
  if (err)
    {
      printf ("(get_bool)GError: %s\n", err->message);
      value = FALSE;
      g_error_free (err);
    }
  g_free (real_key);

  return value;
}

void
gkdial_conf_set_bool (GkDial * gkdial, const gchar * key,
		      const gboolean value)
{
  GError *err = NULL;
  gchar *real_key = g_strdup_printf ("/apps/gkdial/%s",
				     key);

  gconf_client_set_bool (gkdial->conf_client, real_key, value, &err);

  if (err)
    {
      printf ("(set_bool)GError: %s\n", err->message);
      g_error_free (err);
    }

  g_free (real_key);
}

gdouble
gkdial_conf_get_float (GkDial * gkdial, const gchar * key)
{
  gdouble value;
  gchar *real_key = g_strdup_printf ("/apps/gkdial/%s",
				     key);

  value = gconf_client_get_float (gkdial->conf_client, real_key, NULL);
  g_free (real_key);

  return value;
}

void
gkdial_conf_set_float (GkDial * gkdial, const gchar * key,
		       const gdouble value)
{
  GError *err = NULL;
  gchar *real_key = g_strdup_printf ("/apps/gkdial/%s",
				     key);

  gconf_client_set_float (gkdial->conf_client, real_key, value, &err);

  if (err)
    {
      printf ("(set_float)GError: %s\n", err->message);
      g_error_free (err);
    }

  g_free (real_key);
}

void
spinb_changed_cb (GtkWidget * spinb, gpointer data)
{
  GkDial *gkdial = (GkDial *) data;

  gkdial->max_retries = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spinb));

  gkdial_conf_set_float (gkdial, "preferences/max_retries",
			 gkdial->max_retries);
}

void
check_changed_cb (GtkWidget * w, gpointer data)
{
  GkDial *gkdial = (GkDial *) data;
  gboolean value, conf_value;
  gchar *check_name = NULL, *pref_name = NULL;

  check_name = (gchar *) g_object_get_data (G_OBJECT (w), "check");
  g_return_if_fail (check_name != NULL);

  pref_name = g_strdup_printf ("preferences/%s", check_name);

  g_return_if_fail (pref_name != NULL);

  value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
  conf_value = gkdial_conf_get_bool (gkdial, pref_name);

#ifdef DEBUG
  printf ("(check_changed_cb): pn: %s v: %d cv: %d!\n",
	  pref_name, value, conf_value);
  fflush (stdout);
#endif

  if (value == conf_value)
    return;

  if (value)
    gkdial_conf_set_bool (gkdial, pref_name, TRUE);
  else
    gkdial_conf_set_bool (gkdial, pref_name, FALSE);

  g_free (pref_name);

#ifdef DEBUG
  printf ("(check_changed_cb): Bye!\n");
  fflush (stdout);
#endif
}
