/* rcfile.c
 * 
 * Copyright (C) 2001-2002 Claudio Girardi
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>		/* for time() and ctime() */
#include "util.h"
#include "rcfile.h"
#include "glfer.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_DMALLOC_H
#include <dmalloc.h>		/* dmalloc.h should be included last */
#endif


extern opt_t opt;

/* rc filename; it is set after the first call to rc_file_parse() */
char *rc_filename = NULL;

/* parse rcfile */
int parse_rcfile(void)
{
  char *env_home;
  char *rc_name = PACKAGE "rc";
  char *filename = NULL;
  int result;

  env_home = getenv("HOME");
  filename = (char *) malloc(strlen(env_home) + strlen(rc_name) + 3);
  sprintf(filename, "%s/.%s", env_home, rc_name);
  /* load preferences */
  result = rc_file_parse(filename);
  free(filename);
  return result;
}


/* skip_equals, skips to the first non-whitespace character after equals sign
 */
static int skip_equals(char **buf)
{
  int i;

  for (i = 0; i < 200; i++) {
    if (**buf == '=') {
      (*buf)++;
      while (**buf == ' ')
	(*buf)++;
      return (TRUE);
    }
    (*buf)++;
  }

  fprintf(stderr, "error in rcfile, expected equals\n");
  return (FALSE);
}


int rc_file_parse(const char *filename)
{
  FILE *fp;
  char buf[256], *i;

  if (rc_filename)
    free(rc_filename);

  /* save the filename for when we write back the preferences */
  rc_filename = strdup(filename);

  fp = fopen(filename, "r");

  if (!fp)			/* problems opening file */
    return -1;

  while (fgets(buf, 256, fp)) {
    i = &buf[0];
    /* skip spaces */
    while ((*i == ' ') || (*i == '\t'))
      i++;
    /* don't process empty lines */
    if ((*i == '#') || (*i == '\n'))
      continue;

    /* work out what the setting is */
    if (!strncasecmp(i, "mode", 4)) {
      if (skip_equals(&i)) {
	opt.mode = atoi(i);
      }
    } else if (!strncasecmp(i, "scale_type", 10)) {
      if (skip_equals(&i)) {
	opt.scale_type = atoi(i);
      }
    } else if (!strncasecmp(i, "fft_n", 5)) {
      if (skip_equals(&i)) {
	opt.fft_n = atoi(i);
      }
    } else if (!strncasecmp(i, "fft_overlap", 11)) {
      if (skip_equals(&i)) {
	opt.fft_overlap = atof(i);
      }
    } else if (!strncasecmp(i, "mtm_n", 5)) {
      if (skip_equals(&i)) {
	opt.mtm_n = atoi(i);
      }
    } else if (!strncasecmp(i, "mtm_overlap", 11)) {
      if (skip_equals(&i)) {
	opt.mtm_overlap = atof(i);
      }
    } else if (!strncasecmp(i, "mtm_w", 5)) {
      if (skip_equals(&i)) {
	opt.mtm_w = atof(i);
      }
    } else if (!strncasecmp(i, "mtm_k", 5)) {
      if (skip_equals(&i)) {
	opt.mtm_k = atoi(i);
      }
    } else if (!strncasecmp(i, "hparma_n", 8)) {
      if (skip_equals(&i)) {
	opt.hparma_n = atoi(i);
      }
    } else if (!strncasecmp(i, "hparma_overlap", 14)) {
      if (skip_equals(&i)) {
	opt.hparma_overlap = atof(i);
      }
    } else if (!strncasecmp(i, "hparma_t", 8)) {
      if (skip_equals(&i)) {
	opt.hparma_t = atof(i);
      }
    } else if (!strncasecmp(i, "hparma_p_e", 10)) {
      if (skip_equals(&i)) {
	opt.hparma_p_e = atof(i);
      }
    } else if (!strncasecmp(i, "window_type", 11)) {
      if (skip_equals(&i)) {
	opt.window_type = atof(i);
      }
    } else if (!strncasecmp(i, "sample_rate", 11)) {
      if (skip_equals(&i)) {
	opt.sample_rate = atoi(i);
      }
    } else if (!strncasecmp(i, "offset_freq", 11)) {
      if (skip_equals(&i)) {
	opt.offset_freq = atof(i);
      }
    } else if (!strncasecmp(i, "dot_time", 8)) {
      if (skip_equals(&i)) {
	opt.dot_time = atof(i);
      }
    } else if (!strncasecmp(i, "dfcw_gap_time", 13)) {
      if (skip_equals(&i)) {
	opt.dfcw_gap_time = atof(i);
      }
    } else if (!strncasecmp(i, "tx_mode", 7)) {
      if (skip_equals(&i)) {
	if (!strncasecmp(i, "QRSS", 4))
	  opt.tx_mode = QRSS;
	else if (!strncasecmp(i, "DFCW", 4))
	  opt.tx_mode = DFCW;
	else
	  fprintf(stderr, "error in rcfile %s: %s\n", filename, buf);
      }
    } else if (!strncasecmp(i, "dash_dot_ratio", 14)) {
      if (skip_equals(&i)) {
	opt.dash_dot_ratio = atof(i);
      }
    } else if (!strncasecmp(i, "ptt_delay", 9)) {
      if (skip_equals(&i)) {
	opt.ptt_delay = atof(i);
      }
    } else if (!strncasecmp(i, "sidetone_freq", 13)) {
      if (skip_equals(&i)) {
	opt.sidetone_freq = atof(i);
      }
    } else if (!strncasecmp(i, "sidetone", 8)) {
      if (skip_equals(&i)) {
	if (!strncasecmp(i, "ON", 2))
	  opt.sidetone = TRUE;
	else if (!strncasecmp(i, "OFF", 3))
	  opt.sidetone = FALSE;
	else
	  fprintf(stderr, "error in rcfile %s: %s\n", filename, buf);
      }
    } else if (!strncasecmp(i, "dfcw_dot_freq", 13)) {
      if (skip_equals(&i)) {
	opt.dfcw_dot_freq = atof(i);
      }
    } else if (!strncasecmp(i, "dfcw_dash_freq", 14)) {
      if (skip_equals(&i)) {
	opt.dfcw_dash_freq = atof(i);
      }
    } else if (!strncasecmp(i, "ctrl_device", 11)) {
      if (skip_equals(&i)) {
	char *iend;
	/* Get the ending position.  */
	for (iend = i; *iend && *iend != '\n'; iend++);
	opt.ctrl_device = (char *) malloc(iend - i + 1);
	memcpy(opt.ctrl_device, i, iend - i);
	opt.ctrl_device[iend - i] = '\0';
      }
    } else if (!strncasecmp(i, "device_type", 11)) {
      if (skip_equals(&i)) {
	if (!strncasecmp(i, "DEV_SERIAL", 10))
	  opt.device_type = DEV_SERIAL;
	else if (!strncasecmp(i, "DEV_PARALLEL", 12))
	  opt.device_type = DEV_PARALLEL;
	else
	  fprintf(stderr, "error in rcfile %s: %s\n", filename, buf);
      }
    } else if (!strncasecmp(i, "audio_device", 11)) {
      if (skip_equals(&i)) {
	char *iend;
	/* Get the ending position.  */
	for (iend = i; *iend && *iend != '\n'; iend++);
	opt.audio_device = (char *) malloc(iend - i + 1);
	memcpy(opt.audio_device, i, iend - i);
	opt.audio_device[iend - i] = '\0';
      }
    } else
      fprintf(stderr, "error in rcfile %s: %s\n", filename, buf);
  }

  fclose(fp);
  /* everything went ok */
  return 0;
}


/* rc_file_write, writes the user's preferences back to file */
int rc_file_write(void)
{
  FILE *fp;
  time_t date;

  time(&date);

  fp = fopen(rc_filename, "w+");

  if (!fp)
    return (-1);

  /* write header */
  fprintf(fp, "# This is the startup file for " PACKAGE
#ifdef VERSION
	  "-" VERSION
#endif /* #ifdef VERSION */
	  "\n");
  fprintf(fp, "# automatically generated on %s", ctime(&date));
  fprintf(fp, "\n");
  fprintf(fp, "# Lines starting with \'#\' are ignored\n");
  fprintf(fp, "\n");
  fprintf(fp, "mode = %i\n", opt.mode);
  fprintf(fp, "scale_type = %i\n", opt.scale_type);

  fprintf(fp, "fft_n = %i\n", opt.fft_n);
  fprintf(fp, "fft_overlap = %f\n", opt.fft_overlap);

  fprintf(fp, "mtm_n = %i\n", opt.mtm_n);
  fprintf(fp, "mtm_overlap = %f\n", opt.mtm_overlap);
  fprintf(fp, "mtm_w = %f\n", opt.mtm_w);
  fprintf(fp, "mtm_kw = %i\n", opt.mtm_k);

  fprintf(fp, "hparma_n = %i\n", opt.hparma_n);
  fprintf(fp, "hparma_overlap = %f\n", opt.hparma_overlap);
  fprintf(fp, "hparma_t = %i\n", opt.hparma_t);
  fprintf(fp, "hparma_p_e = %i\n", opt.hparma_p_e);

  fprintf(fp, "window_type = %i\n", opt.window_type);
  fprintf(fp, "sample_rate = %i\n", opt.sample_rate);
  fprintf(fp, "offset_freq = %f\n", opt.offset_freq);
  fprintf(fp, "dot_time = %f\n", opt.dot_time);
  fprintf(fp, "dfcw_gap_time = %f\n", opt.dfcw_gap_time);
  fprintf(fp, "tx_mode = %s\n", opt.tx_mode == QRSS ? "QRSS" : "DFCW");
  fprintf(fp, "dash_dot_ratio = %f\n", opt.dash_dot_ratio);
  fprintf(fp, "ptt_delay = %f\n", opt.ptt_delay);
  fprintf(fp, "sidetone_freq = %f\n", opt.sidetone_freq);
  fprintf(fp, "sidetone = %s\n", opt.sidetone == TRUE ? "ON" : "OFF");
  fprintf(fp, "dfcw_dot_freq = %f\n", opt.dfcw_dot_freq);
  fprintf(fp, "dfcw_dash_freq = %f\n", opt.dfcw_dash_freq);
  fprintf(fp, "ctrl_device = %s\n", opt.ctrl_device);
  fprintf(fp, "device_type = %s\n", opt.device_type == DEV_SERIAL ? "DEV_SERIAL" : "DEV_PARALLEL");
  fprintf(fp, "audio_device = %s\n", opt.audio_device);
  fclose(fp);

  return (0);
}


void rc_file_done(void)
{
  FREE_MAYBE(rc_filename);
}
