/*  Glimmer - gnome-vfs-remote-dialog.c
 *
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-file-info.h>

#include "gnome-vfs-remote-dialog.h"

/* function declarations */
static void gnome_vfs_remote_dialog_class_init (GnomeVFSRemoteDialogClass *
                                                klass);
static void gnome_vfs_remote_dialog_init (GnomeVFSRemoteDialog * dialog);
static void gnome_vfs_remote_dialog_destroy (GtkObject * object);
static gboolean check_glist_for_multiple_entries (GList * list,
                                                  gchar * new_entry);
static void kill_combo_history (GList * list);
static void check_enable_button (GtkWidget * entry, GtkWidget * button);
static GtkWidget *build_history_menu (GnomeVFSRemoteDialog * dialog,
                                      GList * list);
static void history_menu_item_cb (GtkWidget * widget,
                                  GnomeVFSRemoteDialog * dialog);

/* end function declarations */

static GtkWindowClass *parent_class = NULL;

GtkType
gnome_vfs_remote_dialog_get_type (void)
{
    static GtkType vfs_remote_dialog_type = 0;

    if (!vfs_remote_dialog_type) {
        static const GtkTypeInfo vfs_remote_dialog_info = {
            "GnomeVFSRemoteDialog",
            sizeof (GnomeVFSRemoteDialog),
            sizeof (GnomeVFSRemoteDialogClass),
            (GtkClassInitFunc) gnome_vfs_remote_dialog_class_init,
            (GtkObjectInitFunc) gnome_vfs_remote_dialog_init,
            NULL,
            NULL,
            (GtkClassInitFunc) NULL,
        };

        vfs_remote_dialog_type =
            gtk_type_unique (GTK_TYPE_WINDOW, &vfs_remote_dialog_info);
    }
    return (vfs_remote_dialog_type);
}

static void
gnome_vfs_remote_dialog_class_init (GnomeVFSRemoteDialogClass * klass)
{
    GtkObjectClass *object_class;

    object_class = (GtkObjectClass *) klass;
    parent_class = gtk_type_class (GTK_TYPE_WINDOW);
    object_class->destroy = gnome_vfs_remote_dialog_destroy;
}

static void
gnome_vfs_remote_dialog_init (GnomeVFSRemoteDialog * dialog)
{
    dialog->history_list = NULL;
}

GtkWidget *
gnome_vfs_remote_dialog_new (gchar * title, GList * history)
{
    GnomeVFSRemoteDialog *dialog = NULL;
    GtkWidget *main_box = NULL;
    GtkWidget *frame = NULL;
    GtkWidget *util_box = NULL;
    GtkWidget *inside_box = NULL;
    GtkWidget *label = NULL;
    GtkWidget *hsep = NULL;

    dialog =
        (GnomeVFSRemoteDialog *) gtk_type_new (GNOME_TYPE_VFS_REMOTE_DIALOG);
    dialog->history_list = history;
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
    gtk_window_set_title (GTK_WINDOW (dialog), title);

    main_box = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (dialog), main_box);
    gtk_widget_show (main_box);

    dialog->history = gtk_option_menu_new ();
    gtk_option_menu_set_menu (GTK_OPTION_MENU (dialog->history),
                              build_history_menu (dialog,
                                                  dialog->history_list));
    gtk_option_menu_set_history (GTK_OPTION_MENU (dialog->history), 0);
    gtk_box_pack_start (GTK_BOX (main_box), dialog->history, TRUE, TRUE, 0);
    gtk_widget_show (dialog->history);

    frame = gtk_frame_new ("Remote Host");
    gtk_box_pack_start (GTK_BOX (main_box), frame, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
    gtk_widget_show (frame);

    util_box = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (frame), util_box);
    gtk_widget_show (util_box);

    inside_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (util_box), inside_box, TRUE, TRUE, 0);
    gtk_widget_show (inside_box);

    label = gtk_label_new ("Hostname:");
    gtk_box_pack_start (GTK_BOX (inside_box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);

    dialog->host_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (inside_box), dialog->host_entry, TRUE, TRUE,
                        5);
    gtk_widget_show (dialog->host_entry);

    label = gtk_label_new ("  Port:");
    gtk_box_pack_start (GTK_BOX (inside_box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);

    dialog->port_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (inside_box), dialog->port_entry, TRUE, TRUE,
                        5);
    gtk_widget_show (dialog->port_entry);

    inside_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (util_box), inside_box, TRUE, TRUE, 5);
    gtk_widget_show (inside_box);

    label = gtk_label_new ("Path:");
    gtk_box_pack_start (GTK_BOX (inside_box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);

    dialog->path_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (inside_box), dialog->path_entry, TRUE, TRUE,
                        5);
    gtk_widget_show (dialog->path_entry);

    frame = gtk_frame_new ("Authentication");
    gtk_box_pack_start (GTK_BOX (main_box), frame, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
    gtk_widget_show (frame);

    util_box = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (frame), util_box);
    gtk_widget_show (util_box);

    inside_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (util_box), inside_box, TRUE, TRUE, 0);
    gtk_widget_show (inside_box);

    label = gtk_label_new ("Username:");
    gtk_box_pack_start (GTK_BOX (inside_box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);

    dialog->username_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (inside_box), dialog->username_entry, TRUE,
                        TRUE, 5);
    gtk_widget_show (dialog->username_entry);

    label = gtk_label_new ("  Password:");
    gtk_box_pack_start (GTK_BOX (inside_box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);

    dialog->password_entry = gtk_entry_new ();
    gtk_entry_set_visibility (GTK_ENTRY (dialog->password_entry), FALSE);
    gtk_box_pack_start (GTK_BOX (inside_box), dialog->password_entry, TRUE,
                        TRUE, 5);
    gtk_widget_show (dialog->password_entry);

    inside_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (util_box), inside_box, TRUE, TRUE, 5);
    gtk_widget_show (inside_box);

    dialog->save_pass = gtk_check_button_new_with_label ("Save Password?");
    gtk_box_pack_end (GTK_BOX (inside_box), dialog->save_pass, FALSE, FALSE, 5);
    gtk_widget_show (dialog->save_pass);

    hsep = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (main_box), hsep, TRUE, TRUE, 5);
    gtk_widget_show (hsep);

    util_box = gtk_hbutton_box_new ();
    gtk_box_pack_start (GTK_BOX (main_box), util_box, TRUE, TRUE, 0);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (util_box),
                               gnome_preferences_get_button_layout ());
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (util_box), GNOME_PAD);
    gtk_widget_show (util_box);

    dialog->ok_button = gnome_stock_button (GNOME_STOCK_BUTTON_OK);
    gtk_box_pack_end (GTK_BOX (util_box), dialog->ok_button, FALSE, FALSE, 5);
    gtk_widget_show (dialog->ok_button);
    gtk_widget_set_sensitive (dialog->ok_button, FALSE);
    GTK_WIDGET_SET_FLAGS (dialog->ok_button, GTK_CAN_DEFAULT);
    gtk_window_set_default (GTK_WINDOW (dialog), dialog->ok_button);

    dialog->cancel_button = gnome_stock_button (GNOME_STOCK_BUTTON_CANCEL);
    gtk_box_pack_end (GTK_BOX (util_box), dialog->cancel_button, FALSE, FALSE,
                      5);
    GTK_WIDGET_SET_FLAGS (dialog->cancel_button, GTK_CAN_DEFAULT);
    gtk_widget_show (dialog->cancel_button);

    gtk_signal_connect (GTK_OBJECT (dialog->path_entry), "changed",
                        GTK_SIGNAL_FUNC (check_enable_button),
                        (gpointer) dialog->ok_button);
    return GTK_WIDGET (dialog);
}

static void
gnome_vfs_remote_dialog_destroy (GtkObject * object)
{
    GtkWidget *dialog;

    g_return_if_fail (object != NULL);
    g_return_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (object));
    dialog = GTK_WIDGET (object);
    if (GNOME_VFS_REMOTE_DIALOG (dialog)->history_list)
        kill_combo_history (GNOME_VFS_REMOTE_DIALOG (dialog)->history_list);
    GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

gboolean
gnome_vfs_remote_dialog_add_history_string (GnomeVFSRemoteDialog * dialog,
                                            gchar * string)
{
    gchar *temp;
    GtkWidget *menuitem;

    g_return_val_if_fail (dialog != NULL, FALSE);
    g_return_val_if_fail (string != NULL, FALSE);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), FALSE);
    if (check_glist_for_multiple_entries (dialog->history_list, string))
        return (FALSE);
    temp = g_strdup (string);
    dialog->history_list =
        g_list_append (dialog->history_list, (gpointer) temp);
    menuitem = gtk_menu_item_new_with_label (temp);
    gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                        GTK_SIGNAL_FUNC (history_menu_item_cb), dialog);
    gtk_object_set_data (GTK_OBJECT (menuitem), "uri-string", temp);
    gtk_menu_append (GTK_MENU
                     (gtk_option_menu_get_menu
                      (GTK_OPTION_MENU (dialog->history))), menuitem);
    gtk_widget_show (menuitem);
    return (TRUE);
}

gchar *
gnome_vfs_remote_dialog_get_host (GnomeVFSRemoteDialog * dialog)
{
    gchar *text;

    g_return_val_if_fail (dialog != NULL, NULL);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), NULL);
    text = gtk_entry_get_text (GTK_ENTRY (dialog->host_entry));
    return (text);
}

gint
gnome_vfs_remote_dialog_get_port (GnomeVFSRemoteDialog * dialog)
{
    gchar *text;
    gint port = 0;

    g_return_val_if_fail (dialog != NULL, 0);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), 0);
    text = gtk_entry_get_text (GTK_ENTRY (dialog->port_entry));
    sscanf (text, "%d", &port);
    return (port);
}

gchar *
gnome_vfs_remote_dialog_get_path (GnomeVFSRemoteDialog * dialog)
{
    gchar *text;

    g_return_val_if_fail (dialog != NULL, NULL);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), NULL);
    text = gtk_entry_get_text (GTK_ENTRY (dialog->path_entry));
    return (text);
}

gchar *
gnome_vfs_remote_dialog_get_username (GnomeVFSRemoteDialog * dialog)
{
    gchar *text;

    g_return_val_if_fail (dialog != NULL, NULL);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), NULL);
    text = gtk_entry_get_text (GTK_ENTRY (dialog->username_entry));
    return (text);
}

gchar *
gnome_vfs_remote_dialog_get_pasword (GnomeVFSRemoteDialog * dialog)
{
    gchar *text;

    g_return_val_if_fail (dialog != NULL, NULL);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), NULL);
    text = gtk_entry_get_text (GTK_ENTRY (dialog->password_entry));
    return (text);
}

GnomeVFSURI *
gnome_vfs_remote_dialog_get_uri (GnomeVFSRemoteDialog * dialog)
{
    GnomeVFSURI *uri = NULL;
    GnomeVFSURI *full_uri = NULL;
    gchar *host;
    gchar *path;
    gchar *username;
    gchar *password;
    gint port;

    g_return_val_if_fail (dialog != NULL, NULL);
    g_return_val_if_fail (GNOME_IS_VFS_REMOTE_DIALOG (dialog), NULL);
    host = gnome_vfs_remote_dialog_get_host (dialog);
    path = gnome_vfs_remote_dialog_get_path (dialog);
    username = gnome_vfs_remote_dialog_get_username (dialog);
    password = gnome_vfs_remote_dialog_get_pasword (dialog);
    port = gnome_vfs_remote_dialog_get_port (dialog);
    uri = gnome_vfs_uri_new (host);
    if (uri) {
        full_uri = gnome_vfs_uri_append_string (uri, path);
        gnome_vfs_uri_unref (uri);
        uri = full_uri;
        if (username && strlen (username)) {
            gnome_vfs_uri_set_user_name (uri, username);
            if (password && strlen (password)) {
                gnome_vfs_uri_set_password (uri, password);
            }
        }
        if (port)
            gnome_vfs_uri_set_host_port (uri, (guint) port);
    }
    return (uri);
}

gboolean
gnome_vfs_remote_dialog_save_pass (GnomeVFSRemoteDialog * dialog)
{
    return (GTK_TOGGLE_BUTTON (dialog->save_pass)->active);
}

static gboolean
check_glist_for_multiple_entries (GList * list, gchar * new_entry)
{
    GList *temp;
    gchar *cmp_text;

    if (list == NULL)
        return (FALSE);
    temp = list;
    while (temp != NULL) {
        cmp_text = temp->data;
        if (!strcmp (cmp_text, new_entry)) {
            return (TRUE);
        }
        temp = g_list_next (temp);
    }
    return (FALSE);
}

static void
kill_combo_history (GList * list)
{
    GList *temp;

    g_return_if_fail (list != NULL);
    for (temp = list; temp; temp = temp->next)
        g_free (temp->data);
    g_list_free (list);
}

static void
check_enable_button (GtkWidget * entry, GtkWidget * button)
{
    gchar *text;
    gboolean sensitive = FALSE;

    text = gtk_entry_get_text (GTK_ENTRY (entry));
    if (text && strlen (text))
        sensitive = TRUE;
    gtk_widget_set_sensitive (button, sensitive);
}

static GtkWidget *
build_history_menu (GnomeVFSRemoteDialog * dialog, GList * list)
{
    GtkWidget *menu;
    GtkWidget *menuitem;
    GList *iter;
    gchar *text;

    menu = gtk_menu_new ();
    list = g_list_prepend (list, g_strdup ("None"));
    gtk_widget_show (menu);
    for (iter = list; iter; iter = iter->next) {
        text = (gchar *) iter->data;
        menuitem = gtk_menu_item_new_with_label (text);
        gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                            GTK_SIGNAL_FUNC (history_menu_item_cb), dialog);
        gtk_object_set_data (GTK_OBJECT (menuitem), "uri-string", text);
        gtk_menu_append (GTK_MENU (menu), menuitem);
        gtk_widget_show (menuitem);
    }
    return (menu);
}

static void
history_menu_item_cb (GtkWidget * widget, GnomeVFSRemoteDialog * dialog)
{
    GnomeVFSURI *uri = NULL;
    gchar *text = NULL;
    gchar *scheme = NULL;
    gchar *host = NULL;
    gchar *path = NULL;
    gchar *username = NULL;
    gchar *password = NULL;
    gchar *pstring = NULL;
    guint port = 0;
    gboolean save_pass = FALSE;

    text = (gchar *) gtk_object_get_data (GTK_OBJECT (widget), "uri-string");
    if ((uri = gnome_vfs_uri_new (text))) {
        scheme = (gchar *) gnome_vfs_uri_get_scheme (uri);
        host = (gchar *) gnome_vfs_uri_get_host_name (uri);
        path =
            gnome_vfs_uri_to_string (uri,
                                     GNOME_VFS_URI_HIDE_USER_NAME |
                                     GNOME_VFS_URI_HIDE_PASSWORD |
                                     GNOME_VFS_URI_HIDE_HOST_NAME |
                                     GNOME_VFS_URI_HIDE_HOST_PORT |
                                     GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD |
                                     GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER);
        username = (gchar *) gnome_vfs_uri_get_user_name (uri);
        password = (gchar *) gnome_vfs_uri_get_password (uri);
        port = gnome_vfs_uri_get_host_port (uri);
        if (port)
            pstring = g_strdup_printf ("%d", port);
        gtk_entry_set_text (GTK_ENTRY (dialog->host_entry),
                            scheme ? scheme : "");
        gtk_entry_append_text (GTK_ENTRY (dialog->host_entry), scheme
                               && strlen (scheme) ? "://" : "");
        gtk_entry_append_text (GTK_ENTRY (dialog->host_entry),
                               host ? host : "");
        gtk_entry_set_text (GTK_ENTRY (dialog->path_entry), path ? path : "");
        gtk_entry_set_text (GTK_ENTRY (dialog->username_entry),
                            username ? username : "");
        gtk_entry_set_text (GTK_ENTRY (dialog->password_entry),
                            password ? password : "");
        if (password && strlen (password))
            save_pass = TRUE;
        gtk_entry_set_text (GTK_ENTRY (dialog->port_entry),
                            pstring ? pstring : "");
        gnome_vfs_uri_unref (uri);
    }
    if (pstring)
        g_free (pstring);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->save_pass),
                                  save_pass);
}
