/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <guichaz@yahoo.fr>
 */

/*****************************************
 * A simplified ftw(3), specific to GLiv *
 *****************************************/

#include <sys/types.h>          /* dev_t, ino_t */
#include <sys/stat.h>           /* struct stat, stat() */
#include <dirent.h>             /* struct dirent, *dir() */
#include <stdio.h>              /* perror() */

#include "gliv.h"
#include "foreach_file.h"

/* Used to detect insertions of already present elements. */
static gboolean destroyed_node = FALSE;

/*
 * Directories informations saved into
 * the AVL to avoid infinite loops.
 */
typedef struct {
    dev_t dev;
    ino_t ino;
} dir_info;

/* Called when we insert an element that was already there. */
static void value_destroy_notify(gpointer data)
{
    g_free(data);
    destroyed_node = TRUE;
}

/* To build and search the AVL. */
static gint cmp_func(const dir_info * dir1, const dir_info * dir2)
{
    if (dir1->ino < dir2->ino)
        return -1;

    if (dir1->ino > dir2->ino)
        return 1;

    if (dir1->dev < dir2->dev)
        return -1;

    return dir1->dev > dir2->dev;
}

/* Push directories and handle files. */
static GSList *process_dir(GSList * stack, const gchar * dirname,
                           foreach_file_func func, gint * res)
{
    DIR *dir;
    struct dirent *dir_ent;
    struct stat st;
    gchar *name, *fullname;
    gboolean is_dir = FALSE, is_dir_known;

    dir = opendir(dirname);
    if (dir == NULL) {
        perror(dirname);
        return stack;
    }

    for (dir_ent = readdir(dir); dir_ent != NULL; dir_ent = readdir(dir)) {
        name = dir_ent->d_name;

        if (name[0] == '.' &&
            (name[1] == '\0' || (name[1] == '.' && name[2] == '\0')))
            /* Skip "." and "..". */
            continue;

        fullname = g_build_filename(dirname, name, NULL);
        is_dir_known = FALSE;

#ifdef _DIRENT_HAVE_D_TYPE
        switch (dir_ent->d_type) {
        case DT_UNKNOWN:
            break;

        case DT_DIR:
            is_dir = TRUE;
            is_dir_known = TRUE;
            break;

        default:
            is_dir = FALSE;
            is_dir_known = TRUE;
            break;
        }
#endif

        if (is_dir_known == FALSE && !stat(fullname, &st)) {
            is_dir = S_ISDIR(st.st_mode);
            is_dir_known = TRUE;
        }

        if (is_dir_known) {
            if (is_dir)
                /* A directory. */
                stack = g_slist_prepend(stack, fullname);
            else {
                /* A file. */
                *res += (*func) (fullname);
                g_free(fullname);
            }
        }
    }

    closedir(dir);
    return stack;
}

/* Return TRUE if the dir has been correctly inserted. */
static gboolean tree_insert_dir(GTree * tree, struct stat *st)
{
    dir_info *dir;

    dir = g_new(dir_info, 1);

    dir->dev = st->st_dev;
    dir->ino = st->st_ino;

    g_tree_insert(tree, dir, NULL);

    if (destroyed_node) {
        /* It was already there. */
        destroyed_node = FALSE;
        return FALSE;
    }

    return TRUE;
}

/*
 * We know that func is add_file_to_list and
 * we return the number of inserted files.
 */
gint foreach_file(const gchar * path, foreach_file_func func)
{
    GSList *stack = NULL;
    GTree *tree;
    gchar *current;
    struct stat st;
    gint res = 0;

    if (stat(path, &st) < 0) {
        /* The path is not usable. */
        perror(path);
        return 0;
    }

    if (S_ISDIR(st.st_mode) == FALSE)
        /* The path is a file, not a directory. */
        return (*func) (path);

    /* The path is a valid directory. */

    stack = g_slist_prepend(stack, g_strdup(path));     /* push */

    tree = g_tree_new_full((GCompareDataFunc) cmp_func, NULL,
                           g_free, value_destroy_notify);

    while (stack) {
        current = stack->data;
        stack = g_slist_remove_link(stack, stack);      /* pop */

        if (!stat(current, &st) && tree_insert_dir(tree, &st))
            /* Not already scanned. */
            stack = process_dir(stack, current, func, &res);

        g_free(current);
    }

    g_tree_destroy(tree);
    destroyed_node = FALSE;

    return res;
}
