/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/***************************************************************************
 
  Transform	RTP
  Module:	transform_rtp.c

  Documentation:  http://www-scf.usc.edu/~soonwook/nexus/rtp/html

***************************************************************************/
#include "internal.h"
#include "nexus_dc.h"

#include <sys/socket.h>      /* struct sockaddr in host2ip() */
#include <netdb.h>           /* gethostbyname() in host2ip() */
#include <netinet/in.h>      /* sockaddr_in  in host2ip() */
#include <arpa/inet.h>       /* inet_addr() in host2ip() */
#include <rpcsvc/ypclnt.h>   /* YP */
#include <fcntl.h>

#include <sys/time.h>	     /* gettimeofday */

#include "rtp.h"

#define NEXUS_RTP_EXAMPLE 5
#define NEXUS_PAYLOAD_TYPE 110

#define TRANSFORM_BLOCKSIZE	8
#define LARGEST_POSSIBLE_SIZEOF_ULONG 8

#define RTCP_FREQUENCE 5	/* frequence of control data pe  RTP data */
#define MAXMSEG  1024
#define rtp_fatal globus_fatal  /* currently, do nothing */
	
#define RTCP_DEFAULT  0		/* default */
#define RTCP_COUNT    1		/* send SR using the count of data sent */
#define RTCP_SECOND   2		/* send SR using periodic time */

/********************************/
/* Global Variable              */
/********************************/

static fd_set read_fds;
static fd_set write_fds;

static max_fds;	

static send_sr_type; 		/* RTCP_DEFAULT or RTCP_COUNT or RTCP_SECOND */
static int sr_frequency;	/* SR rate is 1/sr_frequnecy */
static struct timeval delta, nexttime;
static struct timeval rr_delta, rr_nexttime;

/********************************/
/* Externally Visible Functions */
/********************************/

void *_nx_transform_rtp_info(void);


/******************************/
/* Transform Table Functions */
/*****************************/

static int transform_rtp_transform_id();
static void transform_rtp_init(
			    nexus_bool_t   *modifies_data,
			    unsigned long  *header_size,
			    unsigned long  *trailer_size);
static void transform_rtp_init_endpoint_state(nexus_transformattr_t *attr, 
					    nexus_transformstate_t **ep_state);
static void transform_rtp_destroy_endpoint_state(
					    nexus_transformstate_t *ep_state);
static void transform_rtp_update_endpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t *sp_state);
static void transform_rtp_init_startpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t **sp_state,
					    nexus_bool_t  *copy_sp_locally,
					    nexus_bool_t  *destroy_sp_locally);
static void transform_rtp_destroy_startpoint_state(
					    nexus_transformstate_t *sp_state);
static void transform_rtp_copy_startpoint_state(
					nexus_transformstate_t *sp_state,
					nexus_transformstate_t **sp_state_copy);
static int transform_rtp_sizeof_startpoint_state(nexus_transformstate_t *s);
static void transform_rtp_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state);
static void transform_rtp_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state);
static int transform_rtp_transform(nexus_transformstate_t *startpoint_state,
                                        nexus_byte_t *storage_start,
                                        unsigned long storage_size,
                                        nexus_byte_t *data_start,
                                        unsigned long *data_size,
                                        nexus_bool_t must_alloc_new_buffer,
                                        nexus_byte_t *transform_info_start,
                                        nexus_byte_t **out_storage_start,
                                        unsigned long *out_storage_size,
                                        nexus_byte_t **out_data_start,
                                        unsigned long *out_data_size);

static int transform_rtp_untransform(nexus_transformstate_t *endpoint_state,
				     nexus_byte_t *data_start,
				     unsigned long *data_size,
				     nexus_byte_t *transform_info_start,
				     int format,
				     nexus_byte_t **destination_start,
				     unsigned long *destination_size);


static void transform_rtp_transformstate_get_info(
						nexus_transformstate_t *state,
						void *info,
						int flag);


static void rtp_write_data(nexus_byte_t *data_start, u_int32 seq,
						     u_int32 src_id);

static int rtp_write_traffic(char *buffer, 
				 nexus_transformstate_t *state,
				 int flag);

struct in_addr * gethostip(void);

void fd_set_nonblocking(int fd);

u_int32 timeval_ntp32(struct timeval *tv);

void gettimeofday_ntp(u_int32 *t);

static void rtp_read_control(nexus_transformstate_t *state, char *mesg);

static void rtp_read_traffic(nexus_transformstate_t *state, rtcp_t *r);

int size_transform_info_except_rtphdr(void);

void timevaladd(struct timeval *a, struct timeval *b, struct timeval *sum);

int timevalless(struct timeval *a, struct timeval *b);

int time_to_send_SR(nexus_transformstate_t *sp_state);

int time_to_send_RR(nexus_transformstate_t *ep_state);

double rtcp_period(int members, int senders, double bw, int we_sent,
			int packet_size, int *avg_rtcp_size, int initial);

/*******************/
/* Data Structures */
/*******************/

typedef struct _nexus_session_t
{
    struct _nexus_session_t *next;

    /* number of the message last received */
    u_int32 sequence;

    /* this session node id */
    u_int32 this_endnode_id;

    /* sender id */
    u_int32 sender_id;

    /* listening fd for control data */
    int listen_fd;

    /* information for data and control information */
    struct timeval last_arrival[2]; /* last arrival (data , SR) */
    struct sockaddr nsrc;           /* network source address */

    /* outgoing packet */
    int senders;		   /* # of active senders */
    struct {
	double bw;		   /* bandwidth, bytes/sec */
	int avg;		   /* smoothed size */
	int last;		   /* last_size */
    } control;
    int members;		   /* # of members */

    /* this endpoint get SR report so that it can know his address */
    int know_hisaddr;

    /* activity timestamps */
    u_int32 lsr_sec;               /* last sender report */
    u_int32 lsr_frac;              /* last sender report */

    /* statistics */
    u_int32 msg_lost;		/* count of message lost */
    u_int32 received;		/* packets received */
    u_int32 expected;		/* packets expected */
    u_int32 prior_ts;		/* previous timestamp */
    int  jitter;		/* current jitter estimated */
    
    /* saved from last reporting interval */
    struct {
	u_int32 received;
	u_int32 expected;
    } last;

} nexus_session_t;


/* 
 * singly linked list of sequence nodes, one for each startpoint
 * bound to me
 */

typedef struct _endpoint_transform_state_t
{
    nexus_session_t *session_hdr;  	/* point to header */
    nexus_session_t *current_session;   /* point to currrent session node */

} endpoint_transform_state_t; 



typedef struct _startpoint_transform_state_t
{
    /* address of my sequence node on endpoint side */
    nexus_byte_t ep_sequence_liba[LARGEST_POSSIBLE_SIZEOF_ULONG];

    /* source id */
    u_int32 src_id;		   /* sender identifier */

    /* number of the message last sent */
    u_int32 sequence;  

    /* destination  address */
    u_int32 dest_ip;		   /* ip address */
    u_int16 dest_port; 	           /* destination rtcp port */
    struct sockaddr_in *his_addr;  /* socket address */

    /* rtcp control sockek description */
    int rtcp_sockfd;

    /* outgoing packet */
    u_int32 packets[2];		   /* number of data/control packets sent */
    u_int32 bytes;		   /* bytes sent */
    int senders;		   /* # of active senders */
    struct {
	double bw;		   /* bandwidth, bytes/sec */
	int avg;		   /* smoothed size */
	int last;		   /* last_size  */
    } control;

    int members;		   /* # of members */

    /* informationfor data and control information */
    struct timeval last;	   /* last arrival( RR) */

    /* statistics */
    struct timeval rtt; 	   /* round trip time */
    u_int32 jitter_estimated;  	   /* jitter */

} startpoint_transform_state_t;


/**************************/
/* Local Global Variables */
/**************************/

static nexus_transform_funcs_t transform_rtp_funcs =
{
    transform_rtp_transform_id,
    transform_rtp_init,
    NULL, /* transform_rtp_shutdown */
    NULL, /* transform_rtp_transformattr_init */
    NULL, /* transform_rtp_transformattr_destroy */
    NULL, /* transform_rtp_transformattr_get_info */
    transform_rtp_init_endpoint_state,
    transform_rtp_destroy_endpoint_state,
    transform_rtp_update_endpoint_state,
    transform_rtp_init_startpoint_state,
    NULL, /* transform_rtp_copy_startpoint_state,                           */
	  /* do not provide copy_sp routine when copy_locally = NEXUS_FALSE */
    transform_rtp_destroy_startpoint_state,
    transform_rtp_sizeof_startpoint_state,
    transform_rtp_put_startpoint_state,
    transform_rtp_get_startpoint_state,
    transform_rtp_transform,
    transform_rtp_untransform,
    transform_rtp_transformstate_get_info
};

/********************************/
/* Externally Visible Functions */
/********************************/


/*
 * _nx_transform_rtp_info()
 *
 * Return the function table for this module.
 */
void *_nx_transform_rtp_info(void)
{
    return ((void *) (&transform_rtp_funcs));
}

/*******************/
/* Local Functions */
/*******************/

/*
 * transform_rtp_transform_id()
 */

static int transform_rtp_transform_id()
{
    return(NEXUS_RTP_EXAMPLE);
} /* transform_rtp_transform_id() */


/*
 * transform_rtp_init()
 */
static void transform_rtp_init(
                                nexus_bool_t   *modifies_data,
                                unsigned long  *header_size,   
                                unsigned long  *trailer_size)
{
    char *arg;
    int temp_int;
    double period;
    struct timeval now;

    NexusAssert2((modifies_data), 
	("transform_rtp_init(): rcvd NULL modifies_data\n"));
    NexusAssert2((header_size), 
	("transform_rtp_init(): rcvd NULL header_size\n"));
    NexusAssert2((trailer_size), 
	("transform_rtp_init(): rcvd NULL trailer_size\n"));

    NexusAssert2((sizeof(unsigned long) <= LARGEST_POSSIBLE_SIZEOF_ULONG), 
("transform_rtp_init(): detected sizeof(unsigned long) > LARGEST_POSSIBLE_SIZEOF_ULONG\n"));


    /* initilaze random seed */
    srand48(time(0));

    /* initialize max_fds to 0 */
    max_fds = 0;

    /* adds 1 to every byte of msg data */
    *modifies_data = NEXUS_FALSE;

    /* ep_seq_liba and sequence and rtp header*/
    *header_size = 2 * LARGEST_POSSIBLE_SIZEOF_ULONG
			+ sizeof(rtp_hdr_t);

    /* 
     * would like to have enough extra bytes to pad to 
     * TRANSFORM_BLOCKSIZE-byte boundary without allocating 
     * a new buffer 
     */

     *trailer_size = TRANSFORM_BLOCKSIZE - 1;

    /*
     * determine the type of sending SR, which is among RTCP_DEFAULT, 
     * RTCP_COUNT or RTCP_SECOND, and initialize the corresponding
     * global variable
     */

     send_sr_type = RTCP_DEFAULT;	
     period = rtcp_period(0,0,0,0,0,&temp_int,1); 
     period *= (drand48() + 0.5);

     delta.tv_sec = (unsigned long)period;
     delta.tv_usec = ( period - delta.tv_sec ) * 1e6;

     gettimeofday(&now, 0);
     timevaladd(&now, &delta, &nexttime);

     period = rtcp_period(0,0,0,0,0,&temp_int,1);
     period *= (drand48() + 0.5);

     rr_delta.tv_sec = (unsigned long)period;
     rr_delta.tv_usec = ( period - rr_delta.tv_sec ) * 1e6;

     timevaladd(&now, &rr_delta, &rr_nexttime);

     if ((arg = globus_nexus_option_find("rtp_cnt")) != GLOBUS_NULL)
     {
	printf("transform_rtp_init(): argumient = rtp_cnt\n");

	send_sr_type =  RTCP_COUNT;
	if((sr_frequency = atoi(arg)) < 0)
	{		
	    globus_fatal("Cannot use -rtp_cnt with negative value\n");
	}
	printf("transform_rtp_init(): sr_frequency = %d\n", sr_frequency);
     }
     if ((arg = globus_nexus_option_find("rtp_sec")) != GLOBUS_NULL)
     {
	double temp_double;

	printf("transform_rtp_init(): argument = rtp_sec\n");
	send_sr_type = RTCP_SECOND;
	if((temp_double = atof(arg)) < 0)
	{
	    globus_fatal("Cannot use -rtp_sec with negative value\n");
	}
	printf("transform_rtp_init():temp_double=%f\n", temp_double);

	delta.tv_sec = (unsigned long)temp_double; /* truncate */
	delta.tv_usec = ( temp_double - delta.tv_sec ) * 1e6;

	gettimeofday(&now, 0);
	timevaladd(&now, &delta, &nexttime);
     }
} /* transform_rtp_init() */


/*
 * transform_rtp_init_endpoint_state()
 */
static void transform_rtp_init_endpoint_state(nexus_transformattr_t *attr,
				           nexus_transformstate_t **ep_state)
{
    nexus_debug_printf(2, ("transform_rtp_init_endpoint_state\n"));   

    /* NOTE ignoring attr for EXAMPLE */
    NexusAssert2((ep_state), 
	("transform_rtp_init_endpoint_state(): rcvd NULL ep_state\n"));


    *ep_state = (endpoint_transform_state_t *) 
		    nexus_malloc(sizeof(endpoint_transform_state_t));

    ((endpoint_transform_state_t *) (*ep_state))->session_hdr  
					= (nexus_session_t *) NULL;
    ((endpoint_transform_state_t *) (*ep_state))->current_session
					= (nexus_session_t *) NULL;

} /* transform_rtp_init_endpoint_state() */


/*
 * transform_rtp_destroy_endpoint_state()
 */

static void transform_rtp_destroy_endpoint_state(
					    nexus_transformstate_t *ep_state)
{
    nexus_session_t *session_node;

    nexus_debug_printf(2, ("transform_rtp_destroy_endpoint_state\n"));

    NexusAssert2((ep_state), 
	("transform_rtp_destroy_endpoint_state(): rcvd NULL ep_state\n"));

    while (((endpoint_transform_state_t *) (ep_state))->session_hdr)
    {
       session_node = ((endpoint_transform_state_t *) (ep_state))->session_hdr;
       ((endpoint_transform_state_t *) (ep_state))->session_hdr 
						= session_node->next;
	nexus_free(session_node);
    } /* endwhile */

    nexus_free(ep_state);

} /* transform_rtp_destroy_endpoint_state() */


/*
 * transform_rtp_update_endpoint_state()
 */

static void transform_rtp_update_endpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t *sp_state)
{
    nexus_debug_printf(2,("transform_rtp_update_endpoint_state\n"));

    NexusAssert2((ep_state), 
	("transform_rtp_update_endpoint_state(): rcvd NULL ep_state\n"));
    NexusAssert2((sp_state), 
	("transform_rtp_update_endpoint_state(): rcvd NULL sp_state\n"));

    nexus_debug_printf(2, ("***********not implemented()\n"));

} /* end transform_rtp_update_endpoint_state() */


/*
 * transform_rtp_init_startpoint_state()
 */

static void transform_rtp_init_startpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t **sp_state,
					    nexus_bool_t  *copy_sp_locally,
					    nexus_bool_t  *destroy_sp_locally)
{
    int fd;
    int save_error;
    nexus_session_t *session_node;
    unsigned long temp_ulong;
    struct in_addr *in;
    struct sockaddr_in my_addr;
    int len, port, one = 1;
    
    nexus_debug_printf(2, ("transform_rtp_init_startpoint_state\n"));

    NexusAssert2((ep_state), 
	("transform_rtp_init_startpoint_state(): rcvd NULL ep_state\n"));
    NexusAssert2((sp_state), 
	("transform_rtp_init_startpoint_state(): rcvd NULL sp_state\n"));
    NexusAssert2((copy_sp_locally), 
	("transform_rtp_init_startpoint_state(): rcvd NULL copy_sp_locally\n"));
    NexusAssert2((destroy_sp_locally), 
    ("transform_rtp_init_startpoint_state(): rcvd NULL destroy_sp_locally\n"));

    /* instantiate endpoint's sequence node for for this startpoint state */
    session_node = (nexus_session_t *) nexus_malloc(sizeof(nexus_session_t));
    temp_ulong = (unsigned long) session_node;

    /* instantiate startpoint state */
    *sp_state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));

    /* bind startpoint to endpoint */
    memcpy((void *)((startpoint_transform_state_t *) (*sp_state))->ep_sequence_liba,
	    (void *) &temp_ulong,
	    sizeof(unsigned long));

    /* initialize endpoint's session node and this startpoint state */
    session_node->sequence = 0;
    session_node->msg_lost = 0;
    ((startpoint_transform_state_t *)(*sp_state))->sequence = 0;

    /* inserting new sequence node into ep linked list */
    session_node->next =((endpoint_transform_state_t *) ep_state)->session_hdr;
    ((endpoint_transform_state_t *) ep_state)->session_hdr = session_node;

    /* get host ip address */
    in = (struct in_addr *)gethostip();  
    ((startpoint_transform_state_t *)(*sp_state))->dest_ip = in->s_addr;

    /* udp connection */
    if ( (fd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
    { 
        save_error = errno;
        rtp_fatal("transform_rtp_init_startpoint_state(): socket call \
		failed: %s\n", globus_libc_system_error_string(errno));
    }

    /* 
     * SO_REUSEADDR allows two sessions to bind same port
     * For unicast, only the last to bind will receive packets, but
     * earlier ones will pick up after the last arrival leaves.
     * For multicast, this should be irrelevant (both get all packets).
     * This is also the vat behavior.
     */
    if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR,
	      			(char *)&one, sizeof(one)) < 0) 
	rtp_fatal("transform_rtp_init_startpoint_state(): socket call \
		failed: %s\n", globus_libc_system_error_string(errno));


    /* update max_fds */
    if(max_fds < fd ) max_fds = fd;

    /* set fd with nonblocking mode */
    fd_set_nonblocking(fd);

    bzero((char*) &my_addr, sizeof(my_addr));
    my_addr.sin_family = AF_INET;
    my_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    my_addr.sin_port = htons(0);
   
    /* setsockopt(u->in, SOL_SOCKET, SO_REUSEADDR,
		(char *)&one, sizeof(one)) < 0);  */

    if( bind(fd, (struct sockaddr*) &my_addr, sizeof(my_addr)) < 0)
    {
        save_error = errno;
        rtp_fatal("transform_rtp_init_startpoint_state(): bind call \
		failed: %s\n", globus_libc_system_error_string(errno));
    }

    /* to do : rtp_set_socket_size */

    /* figure out the local port number */
    len = sizeof(my_addr);
    if( (getsockname( fd, (struct sockaddr *)&my_addr, &len)) < 0)
    {
	save_error = errno;
	rtp_fatal("udp_get_my_mi_proto(): getsockname call2 failed: %s\n",
			globus_libc_system_error_string(save_error));
    }
    port = (unsigned short) my_addr.sin_port;
    ((startpoint_transform_state_t *)(*sp_state))->dest_port = port;
     
    session_node->sender_id = lrand48();
    ((startpoint_transform_state_t *)(*sp_state))->src_id 
					= session_node->sender_id;

    /*nexus_printf("transform_rtp_init_startpoint_state(): hostname = %s,\
			port = %d\n", in->s_addr, port); */

    FD_SET(fd, &read_fds);
    nexus_debug_printf(2, ("FD_SET(): fd = %d, port =%d\n", fd, port));
    if (max_fds < fd) max_fds = fd; 

    session_node->listen_fd = fd;

    /* initialize endpoint session node */
    session_node->this_endnode_id = lrand48();

    session_node->lsr_sec = 0;
    session_node->lsr_frac = 0;
    session_node->received = 0;
    session_node->expected = 0;
    session_node->last.received = 0;
    session_node->last.expected = 0;
    
    session_node->jitter = 0;
    session_node->prior_ts = 0;

    session_node->members = 2;	  /* default is 2 */
				  /* sender and self */
    session_node->senders = 1;
    session_node->control.bw = 8000 * 0.05; /* default is 400 */
    session_node->control.avg = 0;
    session_node->control.last = 0;

    session_node->know_hisaddr = 0;  /* at start, we don't know his address */
    
    *copy_sp_locally = NEXUS_FALSE;
    *destroy_sp_locally = NEXUS_FALSE;
} /* transform_rtp_init_startpoint_state() */


/*
 * transform_rtp_destroy_startpoint_state()
 */

static void transform_rtp_destroy_startpoint_state(
					    nexus_transformstate_t *sp_state)
{
    nexus_debug_printf(2, ("transform_rtp_destroy_startpoint_state\n"));

    NexusAssert2((sp_state), 
	("transform_rtp_destroy_startpoint_state(): rcvd NULL sp_state\n"));

    nexus_free(sp_state); 
} /* transform_rtp_destroy_startpoint_state() */


/*
 * transform_rtp_sizeof_startpoint_state()
 */

static int transform_rtp_sizeof_startpoint_state(nexus_transformstate_t *s)
{
    NexusAssert2((s), 
	("transform_rtp_sizeof_startpoint_state(): rcvd NULL s\n"));
    
    /* ep_sequence_liba and sequence will be transferred */
    return (5 * nexus_sizeof_byte(LARGEST_POSSIBLE_SIZEOF_ULONG));

} /* transform_rtp_sizeof_startpoint_state() */


/*
 * transform_rtp_put_startpoint_state()
 */

static void transform_rtp_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state)
{
    nexus_debug_printf(2, ("transform_rtp_put_startpoint_state\n"));

    NexusAssert2((buffer), 
	("transform_rtp_put_startpoint_state(): rcvd NULL buffer\n"));
    NexusAssert2((state), 
	("transform_rtp_put_startpoint_state(): rcvd NULL state\n"));

    /* need to place ep_sequence_liba and sequence into buffer */
    nexus_dc_put_byte(buffer, 
		    ((startpoint_transform_state_t *) state)->ep_sequence_liba, 
		    LARGEST_POSSIBLE_SIZEOF_ULONG);
    nexus_dc_put_byte(buffer, 
		    &(((startpoint_transform_state_t *) state)->sequence), 
		    LARGEST_POSSIBLE_SIZEOF_ULONG);
    nexus_dc_put_byte(buffer,
		    &(((startpoint_transform_state_t *) state)->dest_ip),
		    LARGEST_POSSIBLE_SIZEOF_ULONG);
    nexus_dc_put_byte(buffer,
		    &(((startpoint_transform_state_t *) state)->dest_port),
		    LARGEST_POSSIBLE_SIZEOF_ULONG);

    nexus_dc_put_byte(buffer,
		    &(((startpoint_transform_state_t *) state)->src_id),
		    LARGEST_POSSIBLE_SIZEOF_ULONG);

} /* transform_rtp_put_startpoint_state() */


/*
 * transform_rtp_get_startpoint_state()
 */

static void transform_rtp_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state)
{
    int sockfd, save_error, one = 1;
    struct sockaddr_in my_addr, *his_addr;

    nexus_debug_printf(2, ("transform_rtp_get_startpoint_state\n"));

    NexusAssert2((buffer), 
	("transform_rtp_get_startpoint_state(): rcvd NULL buffer\n"));
    NexusAssert2((state), 
	("transform_rtp_get_startpoint_state(): rcvd NULL state\n"));

    *state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));

    nexus_dc_get_byte(buffer, 
	(unsigned char *) ((startpoint_transform_state_t *) (*state))->ep_sequence_liba,
			LARGEST_POSSIBLE_SIZEOF_ULONG, 
			format); 
    nexus_dc_get_byte(buffer, 
	(unsigned char *) &(((startpoint_transform_state_t *) (*state))->sequence),
			LARGEST_POSSIBLE_SIZEOF_ULONG, 
			format); 

    nexus_dc_get_byte(buffer,
	(unsigned char *) &(((startpoint_transform_state_t *) (*state))->dest_ip),
			LARGEST_POSSIBLE_SIZEOF_ULONG,
			format);

    nexus_dc_get_byte(buffer,
       (unsigned char *) &(((startpoint_transform_state_t *) (*state))->dest_port),
			 LARGEST_POSSIBLE_SIZEOF_ULONG,
			 format);

    nexus_dc_get_byte(buffer,
	(unsigned char *) &(((startpoint_transform_state_t *) (*state))->src_id),
			 LARGEST_POSSIBLE_SIZEOF_ULONG,
			 format);

    /* initialize rtcp socket description */
    if((sockfd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
    {
       save_error = errno;
       rtp_fatal("transform_rtp_init_startpoint_state(): socket call \
			failed: %s\n", globus_libc_system_error_string(errno));
    }

    /*
     * SO_REUSEADDR allows two sessions to bind same port
     * For unicast, only the last to bind will receive packets, but
     * earlier ones will pick up after the last arrival leaves.
     * For multicast, this should be irrelevant (both get all packets).
     * This is also the vat behavior.
     */
     if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR,
			      (char *)&one, sizeof(one)) < 0)
		rtp_fatal("transform_rtp_init_startpoint_state(): socket\
			call failed: %s\n", globus_libc_system_error_string(errno));

    if(max_fds < sockfd) max_fds = sockfd;

    /* set socket descriptor as nonblocking mode */
    fd_set_nonblocking(sockfd);

    bzero((char*) &my_addr, sizeof(my_addr));
    my_addr.sin_family = AF_INET;
    my_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    my_addr.sin_port = htons(0);

    ((startpoint_transform_state_t *) (*state))->rtcp_sockfd = sockfd;

    if( bind(((startpoint_transform_state_t *) (*state))->rtcp_sockfd, 
			(struct sockaddr*) &my_addr, sizeof(my_addr)) < 0 )
    {
        save_error = errno;
  	rtp_fatal("transform_rtp_init_startpoint_state(): bind call failed:\
			%s\n", globus_libc_system_error_string(errno));
    }

    /* initialize his_addr */
    his_addr = (struct sockaddr_in *)calloc(1, sizeof(struct sockaddr_in));
    his_addr->sin_family = AF_INET;
    his_addr->sin_addr.s_addr 
		= ((startpoint_transform_state_t *)(*state))->dest_ip;
    his_addr->sin_port = ((startpoint_transform_state_t *)(*state))->dest_port;

    ((startpoint_transform_state_t *)(*state))->his_addr 
				= (struct sockaddr_in *)his_addr;

    /* initialize startpoibt_transform_state */
    ((startpoint_transform_state_t *)(*state))->packets[0] = 0;
    ((startpoint_transform_state_t *)(*state))->packets[1] = 0;
    ((startpoint_transform_state_t *)(*state))->bytes = 0;
    ((startpoint_transform_state_t *)(*state))->members=2;/* default is 2 */
							  /* sender, receiver*/
    ((startpoint_transform_state_t *)(*state))->senders=1;
    ((startpoint_transform_state_t *)(*state))->control.bw 
					= 8000 * 0.05; /* default is 400 */
    ((startpoint_transform_state_t *)(*state))->control.avg = 0;
    ((startpoint_transform_state_t *)(*state))->control.last = 0;

    nexus_debug_printf(2, ("transform_rtp_get_startpoint_state(): dest-ip=%d, dest_port=%d\n", ((startpoint_transform_state_t *) (*state))->dest_ip, ((startpoint_transform_state_t *) (*state))->dest_port));

} /* transform_rtp_get_startpoint_state() */
 

/*
 * transform_rtp_transform()
 */

static int transform_rtp_transform(nexus_transformstate_t *startpoint_state,
                                        nexus_byte_t *storage_start,
                                        unsigned long storage_size,
                                        nexus_byte_t *data_start,
                                        unsigned long *data_size,
                                        nexus_bool_t must_alloc_new_buffer,
                                        nexus_byte_t *transform_info_start,
                                        nexus_byte_t **out_storage_start,
                                        unsigned long *out_storage_size,
                                        nexus_byte_t **out_data_start,
                                        unsigned long *out_data_size)
/*
 * *storage_start
 * |
 * |         *data_start
 * |              |
 * v              v
 * +--------------+---------------------+--------------+
 * |    unused	  |	user data	|   unused     |
 * | 		  |			|	       |
 * +--------------+---------------------+--------------+
 *
 *          	  |<--- *data_size ---->|
 *
 * |<------------------- storage_size ---------------->|
 *
 */
{
    int i,
        transform_size,
	needed_unused_size,
	current_unused_size;
    unsigned long magic;
    char *source,
	 *dest;

    static m = 0;

    struct sockaddr_in my_addr;
    int len, n_ready, n_read;
    char mseg[MAXMSEG];
    struct timeval timeout;
    int rtcp_len; 

    rtcp_t *r;
    rtcp_rr_t *rr;

    int sockfd = ((startpoint_transform_state_t *)(startpoint_state))->rtcp_sockfd;
    struct sockaddr_in *his_addr = ((startpoint_transform_state_t *)(startpoint_state))->his_addr;

    nexus_debug_printf(2, ("transform_rtp_transform(): sockfd = %d\n", \
								     sockfd));

    NexusAssert2((startpoint_state),
        ("transform_ecb_transform(): rcvd NULL startpoint_state\n"));   
    NexusAssert2((storage_start), ("transform_ecb_transform(): rcvd NULL storage_start\n"));
    NexusAssert2((data_start),
        ("transform_ecb_transform(): rcvd NULL data_start\n"));
    NexusAssert2((data_size),
        ("transform_ecb_transform(): rcvd NULL data_size\n"));
    NexusAssert2((transform_info_start),
        ("transform_ecb_transform(): rcvd NULL transform_info_start\n"));
    NexusAssert2((out_storage_start),
        ("transform_ecb_transform(): rcvd NULL out_storage_start\n"));
    NexusAssert2((out_storage_size),
        ("transform_ecb_transform(): rcvd NULL out_storage_size\n"));
    NexusAssert2((out_data_start),
        ("transform_ecb_transform(): rcvd NULL out_data_start\n"));
    NexusAssert2((out_data_size),
        ("transform_ecb_transform(): rcvd NULL out_data_size\n"));

    /* increment sequence */
    ((startpoint_transform_state_t *)(startpoint_state))->sequence += 1;
    nexus_debug_printf(2, ("transform_rtp_transform(): sp_sequence = %d\n",\
	((startpoint_transform_state_t *)(startpoint_state))->sequence));

    /* increase the number of rtp packets sent */
    ((startpoint_transform_state_t *)(startpoint_state))->packets[0] += 1;

    /* placing ep_sequence_liba and sequence into transform_info */
    memcpy((void *) transform_info_start,
	(void *) ((startpoint_transform_state_t *) (startpoint_state))->ep_sequence_liba,
	LARGEST_POSSIBLE_SIZEOF_ULONG);

/* *(transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG) = (byte)(nexus_dc_format());*/
/*
    nexus_put_ulong((transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG),  
			&(((startpoint_transform_state_t *)(startpoint_state))->sequence),
			1);
*/
    memcpy((void *) (transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG),
	(void *) &(((startpoint_transform_state_t *) (startpoint_state))->sequence),
	LARGEST_POSSIBLE_SIZEOF_ULONG);

/*    magic = 12344321;
    memcpy((void *) (transform_info_start + 2*LARGEST_POSSIBLE_SIZEOF_ULONG),
					&magic, LARGEST_POSSIBLE_SIZEOF_ULONG);
*/

    rtp_write_data(transform_info_start + 2 * LARGEST_POSSIBLE_SIZEOF_ULONG,
	((startpoint_transform_state_t *)(startpoint_state))->sequence,
	((startpoint_transform_state_t *)(startpoint_state))->src_id);

    /* make transform_size a multiple of TRANSFORM_BLOCKSIZE */
    transform_size = *data_size;

    ((startpoint_transform_state_t *)(startpoint_state))->bytes +=
						  	transform_size;
					
    ((startpoint_transform_state_t *)(startpoint_state))->control.last =
							transform_size;
							
    if ((needed_unused_size = TRANSFORM_BLOCKSIZE - (transform_size % TRANSFORM_BLOCKSIZE)) == TRANSFORM_BLOCKSIZE) 
    {
	needed_unused_size = 0;
    }
    transform_size += needed_unused_size;
    current_unused_size = storage_size 
				- (data_start - storage_start + *data_size);

    /* determine if a new buffer is needed */
    if (must_alloc_new_buffer || (needed_unused_size > current_unused_size))
    {
    nexus_debug_printf(3, ("transform_rtp_transform(): allocating new\
								buffer\n"));

	/* must allocate new buffer */
	*out_storage_size = data_start - storage_start + transform_size;
	*out_storage_start = (nexus_byte_t *) nexus_malloc(*out_storage_size);
	*out_data_start = *out_storage_start + (data_start - storage_start);
	*out_data_size = transform_size;

	/* transform from old buffer to new */
	source = (char *) data_start;
	dest = (char *) *out_data_start;
    }
    else
    {
    nexus_debug_printf( 3, ("transform_rtp_transform(): NOT allocating \
							     new buffer\n")); 

	/* no need to allocate new buffer */
	*out_storage_start = NULL;
	*data_size = transform_size;

	/* transform in place */
	source = dest = (char *) data_start;

    } /* endif */

    /* transform user-data ... perhaps in place */
    for (i = 0; i < transform_size; i++, source++, dest++)
    {
	/* *dest = *source + 1; */
    } 

    if(time_to_send_SR(startpoint_state)) {
/*    if(time_to_send_SR(((startpoint_transform_state_t *)(startpoint_state))->sequence)) {
    if(((startpoint_transform_state_t *)(startpoint_state))->sequence % RTCP_FREQUENCE == 0)    {  

 	sockfd = socket( AF_INET, SOCK_DGRAM, 0 );

	fd_set_nonblocking(sockfd);

	bzero((char*) &my_addr, sizeof(my_addr));
	my_addr.sin_family = AF_INET;
	my_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	my_addr.sin_port = htons(0);

	bind(sockfd, (struct sockaddr*) &my_addr, sizeof(my_addr));


	his_addr.sin_family = AF_INET;
	his_addr.sin_addr.s_addr = ((startpoint_transform_state_t *)(startpoint_state))->dest_ip;
	his_addr.sin_port = ((startpoint_transform_state_t *)(startpoint_state))->dest_port;

	mseg[0] = '\0';
	strcat(mseg, "THIS IS SAMPLE DATA");
*/
	rtcp_len = rtp_write_traffic(mseg, 
			(nexus_transformstate_t *)startpoint_state, 0);

	len = sizeof(struct sockaddr);
        sendto(sockfd, mseg, rtcp_len, 0, (struct sockaddr *)his_addr, len);
	
	((startpoint_transform_state_t *)(startpoint_state))->control.last =
								rtcp_len;
	/* increase the number of control data sent */
	((startpoint_transform_state_t *)(startpoint_state))->packets[1] += 1;

	nexus_debug_printf(2, ("transform_rtp_transform(): sendto: \
		his_addr = %d, his_port=%d\n", his_addr->sin_addr.s_addr, \
				his_addr->sin_port)); 
   }


    /* check control data received */
    timeout.tv_sec = 0L;
    timeout.tv_usec = 0L;

    if(sockfd < 0 || sockfd > max_fds) sockfd = 0; /* set sockfd, otherwise,*/
						   /* there may be SEGV     */ 
    FD_ZERO(&read_fds);
    FD_SET(sockfd, &read_fds);
    
    n_ready = select(max_fds+1 ,(fd_set *)&read_fds, NULL, NULL, &timeout);

    nexus_debug_printf(2, (" transform_rtp_transform() : pass select \
				statement and n_ready = %d\n ", n_ready ));

    if(n_ready > 0) {
	 if(FD_ISSET(sockfd, &read_fds)) {
	       len = sizeof(struct sockaddr_in);
	       n_read = recvfrom(sockfd, mseg, MAXMSEG, 0, 0, &len);
	       r = (rtcp_t *)mseg;
	       rr = r->r.rr.rr;

	       n_ready = 0; 	/* clean n_ready */

#ifdef BUILD_DEBUG
	nexus_debug_printf(1,("transform_RTP_transform(): received message : \
					num = %d, len = %d\n", ++m, n_read));
	nexus_debug_printf(1,("transform_RTP_transform(): v=%d p=%d cc=%d \
		      pt=%d len=%lu", r->common.version, r->common.p, \
		      r->common.count, r->common.pt, r->common.length));

	nexus_debug_printf(1,("ssrc =%lu, ssrc=%lu fraction=%lu, lost=%iu, \
		   last_seq=%lu, jitter=%lu, lsr=%lu, dlsr=%lu\n", \
		   r->r.rr.ssrc, rr->ssrc, rr->fraction, rr->lost, \
		   rr->last_seq, rr->jitter, rr->lsr, rr->dlsr));
#endif

	       /*
	        * process RTCP packet arrived, 
		* Here, we only can get receiver report(RR) packets, because
		* we assume that a nexus application can not be a receiver
		* as well as a sender.
		*/

	       rtp_read_control((nexus_transformstate_t *)startpoint_state,
					mseg);

         }
    }    

    nexus_debug_printf(2, ("exit transform_rtp_transform()\n")); 
    return(NEXUS_SUCCESS);

} /* transform_rtp_transform() */


/*
 * transform_rtp_untransform()
 */
static int transform_rtp_untransform(nexus_transformstate_t *endpoint_state,
                                     nexus_byte_t *data_start,
                                     unsigned long *data_size,
                                     nexus_byte_t *transform_info_start,
				     int format,
                                     nexus_byte_t **destination_start,
                                     unsigned long *destination_size)
{
    int i,
	n_ready,
        data_format;
    unsigned long temp_ulong,
		  magic,
		  seq,
      		  sp_sequence;
    rtp_hdr_t h;
    nexus_session_t *session_node;
    char *source,
	 *dest;

    static int k=0;
    static struct sockaddr_in hisaddr;

    int sockfd, n_read, 
	transit, prior_transit, d;	/* for computing jitter */
    char mesg[MAXMSEG];			/* buffer for sending/receiving */
    struct timeval timeout, now;
    int rtcp_len;

    rtcp_t *r;

    unsigned int tempint; /* added for debugging jitter computing */

    nexus_debug_printf(1, ("transform_rtp_untransform()\n"));

    NexusAssert2((endpoint_state),
        ("transform_ecb_untransform(): rcvd NULL endpoint_state\n"));
    NexusAssert2((data_start),
        ("transform_ecb_untransform(): rcvd NULL data_start\n"));
    NexusAssert2((data_size),
        ("transform_ecb_untransform(): rcvd NULL data_size\n"));
    NexusAssert2((transform_info_start),
        ("transform_ecb_untransform(): rcvd NULL transform_info_start\n"));
    NexusAssert2((destination_start),
        ("transform_ecb_untransform(): rcvd NULL destination_start\n"));
    NexusAssert2((destination_size),
        ("transform_ecb_untransform(): rcvd NULL destination_size\n"));

/*
 * *storage_start
 * |
 * |         *data_start
 * |              |
 * v              v
 * +--------------+---------------------+--------------+
 * |    unused	  |	user data	|   unused     |
 * | 		  |			|	       |
 * +--------------+---------------------+--------------+
 *
 *          	  |<--- *data_size ---->|
 *
 * |<------------------- storage_size ---------------->|
 *
 */

 
    /* extracting ep_sequence_liba and sequence from transform_info */
    memcpy((void *) &temp_ulong, 
	    (void *) transform_info_start, 
	    sizeof(unsigned long));
    session_node = (nexus_session_t *) temp_ulong;

    /* set current session node */
    ((endpoint_transform_state_t *) endpoint_state)->current_session 
						=  session_node;

/*
    nexus_get_ulong((transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG),
			&sp_sequence, 1); 
*/    

    memcpy((void *) &sp_sequence, 
	    (void *) (transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG), 
	    sizeof(unsigned long));

/*    memcpy((void *) &magic,
	     (void *) (transform_info_start + 2*LARGEST_POSSIBLE_SIZEOF_ULONG),
	     sizeof(unsigned long));

*/

     nexus_debug_printf(2, ("transform_rtp_untransform(): received sp\
					sequence = %d\n", sp_sequence));

     memcpy((void *) &h,
	     (void *) (transform_info_start + 2*LARGEST_POSSIBLE_SIZEOF_ULONG),
	     sizeof(rtp_hdr_t));

#ifdef BUILD_DEBUG

    seq = (u_int32)ntohs(h.seq);
    nexus_debug_printf(1, ("transform_RTP_untransform(): v=%d p=%d x=%d \
         cc=%d m=%d  pt=%d seq=%lu ts=%lu ssrc=%lu\n", h.version, h.p, h.x, \
	 h.cc, h.m, h.pt, seq, ntohl(h.ts), ntohl(h.ssrc)));

#endif

    /* update nexus_session status using this rtp packet arrived */
    
    /* usleep(srandom((u_int32)random()) % 100000); */

    gettimeofday(&now, 0);

    if(session_node->received++) { /* compute jitter */
	tempint = ((now.tv_sec << 16 ) | (now.tv_usec >> 16));

        transit = tempint  -  ntohl(h.ts);
	/*fprintf(stderr, "now=%lu\nh.ts=%lu\n", tempint, h.ts);
	fprintf(stderr, "now.tv_sec=%lu\nnow.tv_usec=%lu\n", (now.tv_sec << 16 ) , (now.tv_usec >> 16)); */
	prior_transit = ((session_node->last_arrival[0].tv_sec << 16) 
		| (session_node->last_arrival[0].tv_usec >> 16)) 
					- session_node->prior_ts;

 	d = transit - prior_transit;
	if( d < 0) d = -d;

	/*fprintf(stderr, "transit=%lu\n prior_transit=%lu\n  d=%lu\n", transit, prior_transit, d);*/

	nexus_debug_printf(2, ("JITTER : transit =%d, prior_transit=%d \
			d = %d", transit, prior_transit, d));
	nexus_debug_printf(2, (" now = %lu\n", ((now.tv_sec << 16 ) | (now.tv_usec >> 16))));
	/*fprintf(stderr, "\n\nsesseion_node->jitter = %lu", session_node->jitter); */

	session_node->jitter += (d - session_node->jitter) / 16 ;
	/*fprintf(stderr, "\nnew session_node->jitter = %lu", session_node->jitter);*/

    }

    session_node->prior_ts =  ntohl(h.ts);
    session_node->last_arrival[0] = now;

    NexusAssert2((session_node), 
    ("transform_rtp_untransform(): extracted NULL sequence node pointer from buffer\n"));

#ifdef BUILD_DEBUG
{
    /*
     * verifying that session_node is still in this endpoints 
     * sequence_node list 
     */

    nexus_session_t *temp_session_node;

    if (NexusDebug(1))
    {
        nexus_printf("transform_rtp_untransform(): SEARCHING verifying \
		sequence %x is in endpoint's sequence list\n", session_node);
    } /* endif */

    for (temp_session_node = ((endpoint_transform_state_t *) endpoint_state)->session_hdr;
	temp_session_node && temp_session_node != session_node;
	    temp_session_node = temp_session_node->next)
	/* do nothing */ ;

    if (!temp_session_node)
    {
    	globus_fatal("transform_rtp_untransform() could not find sequence \
							node in ep's list\n");
    } 
    else if (NexusDebug(1))
    {
	nexus_printf("transform_rtp_untransform(): FOUND sequence node %x \
					in endpoint's list\n", session_node);
    } /* endif */
}
#endif

    nexus_debug_printf(2, ("transform_rtp_untransform(): endpoint sequence \
			node's sequence = %d\n", session_node->sequence));

    /* take care of sequence and lost message information */
    if (sp_sequence == (session_node->sequence + 1))
    {
	/* received the message number we expected */
	session_node->sequence = sp_sequence;
    }
    else if (sp_sequence > (session_node->sequence + 1))
    {
	/* 
	 * received a message number greater than we expected -- some 
	 * messages lost 
	 */
	session_node->msg_lost = sp_sequence - (session_node-> sequence + 1);
	session_node->sequence = sp_sequence;
    }
    else
    {
	/* received a message we earlier counted as lost */
	session_node->msg_lost -= 1;
    }
    nexus_debug_printf(2, ("transform_rtp_untransform(): endpoint sequence \
			node's msg_lost = %d\n", session_node->msg_lost));

    /* making sure data to transform is a multiple of TRANSFORM_BLOCKSIZE  */
    if (*data_size % TRANSFORM_BLOCKSIZE != 0) 
    {
	globus_fatal("transform_rtp_untransform(): data_size not multiple \
						of TRANSFORM_BLOCKSIZE\n");
    }

    /* determining if we can decrypt in place */
    if (!(*destination_start))
    {
    nexus_debug_printf(2, ("transform_rtp_untransform(): decrypting in \
								place\n")); 

	/* dec in place */
	source = dest = (char *) data_start;
    }
    else
    {
	if (*destination_size == *data_size)
 	{
        nexus_debug_printf(2, ("transform_rtp_untransform(): decrypting to \
						      destination_start\n")); 

	    /* dec to new buffer */
	    source = (char *) data_start;
	    dest = (char *) *destination_start;
	}
	else
	{
  	nexus_debug_printf(2, ("transform_rtp_untransform(): not \
						        decrypting\n")); 
 	nexus_debug_printf(2, ("exit transform_rtp_untransform()\n")); 

	    /* do nothing */
	    return (!NEXUS_SUCCESS);
	}
    } /* endif */

    /* untransform user-data ... perhaps in place */ 
    for (i = 0; i < *data_size; i++, source++, dest++)
    {
	 /* *dest = *source - 1; */
    } 

    if(session_node->know_hisaddr && time_to_send_RR(endpoint_state)) {
	int len = sizeof(struct sockaddr);
	/*
	 * generate receiver report.
	 * the third argument "1" means that this packet generated is
	 *  receiver report packet
	 */
	rtcp_len = rtp_write_traffic(mesg,
		(nexus_transformstate_t *)session_node, 1); /* RR report */

	/*
	 * send a receiver report created above routine to a sender
	 */
 	sendto(session_node->listen_fd, mesg, /*n_read,*/ rtcp_len,
			0, (struct sockaddr *)&hisaddr, len);
	session_node->control.last = rtcp_len;

    }

    /* check control data arriving */
    timeout.tv_sec = 0L;
    timeout.tv_usec = 0L;

    /*if(session_node->listen_fd < 0 || session_node->listen_fd > max_fds )
		rtp_fatal */

    FD_ZERO(&read_fds);
    FD_SET(session_node->listen_fd, &read_fds);

    n_ready = select(max_fds+1 ,(fd_set *)&read_fds, NULL, NULL, &timeout);

    nexus_debug_printf(2, ("transform_rtp_untransform(): listen_fd = %d\n", \
					session_node->listen_fd));

    nexus_debug_printf(2, (" transform_rtp_untransform() : pass select \
			statement and n_ready = %d\n ", n_ready ));

    if(n_ready > 0) {
        if(FD_ISSET(session_node->listen_fd, &read_fds)) {  
	     int len = sizeof(struct sockaddr_in);
	     n_read = recvfrom(session_node->listen_fd, mesg, MAXMSEG, 0, 
					(struct sockaddr *)&hisaddr, &len);

#ifdef BUILD_DEBUG

	r = (rtcp_t *)mesg;
	nexus_debug_printf(1,("transform_RTP_untransform(): received message : \
				num = %d len = %d", ++k, n_read));
	nexus_debug_printf(1,("transform_RTP_untransform(): v=%d p=%d cc=%d \
		pt=%d len=%lu", r->common.version, r->common.p,\
		r->common.count, r->common.pt, r->common.length));
	nexus_debug_printf(1,("ssrc=%lu, ntp_sec=%lu, ntp_usec=%lu, rtp_ts=%lu,\
		psent=%lu,osent=%lu\n", r->r.sr.ssrc, r->r.sr.ntp_sec, \
		r->r.sr.ntp_frac, r->r.sr.rtp_ts, r->r.sr.psent, \
		r->r.sr.osent));

#endif

	     n_ready = 0;		/* clear n_ready */
		
	     /*
	      * MUST get a sender report packet from a nexus sender,
	      * and process it
	      */
	     rtp_read_control((nexus_transformstate_t *)session_node, mesg);

	     session_node->know_hisaddr = 1;	/* know the sender's address */
	     session_node->nsrc = *(struct sockaddr*)&hisaddr;
	     
#if 0	     /*
	      * generate receiver report.
	      * the third argument "1" means that this packet generated is
	      * a receiver report packet
	      */
	     rtcp_len = rtp_write_traffic(mesg, 
		   (nexus_transformstate_t *)session_node, 1); /* RR report */
	     
	     /*
	      * send a receiver report created above routine to a sender
	      */
	     sendto(session_node->listen_fd, mesg, /*n_read,*/ rtcp_len,  
				0, (struct sockaddr *)&hisaddr, len);
#endif 

	}
    } 

	nexus_debug_printf(2, ("exit transform_rtp_untransform()\n")); 

    return(NEXUS_SUCCESS);

} /* transform_rtp_untransform */


/*
 * retrieve the current state of startpoint or endpoint. if (flag = 0) then
 * retrieve startpoint information, otherwise, retrieve endpoint infomation.
 */

static void transform_rtp_transformstate_get_info(
						nexus_transformstate_t *state,
						void *info, int flag)
{
	if(flag == 0) { /* get info. of startpoint state */
		/* *info = (startpoint_transform_state_t *)
			malloc(1, sizeof(startpoint_transform_state_t)); */
		memcpy(info, 
			(startpoint_transform_state_t *)state, 
			sizeof(startpoint_transform_state_t));
	}  else { /* get info. of endpoint state */
		/* *info = (nexus_session_t *)
			calloc(1, sizeof(nexus_session_t)); */
		memcpy(info, 
		       ((endpoint_transform_state_t *)state)->current_session, 
		        sizeof(nexus_session_t));
	}
}


/*
 * Create a RTP packets
 */

static void rtp_write_data(nexus_byte_t *data_start, u_int32 seq, u_int32 src)
{

	rtp_hdr_t h;
	struct timeval time;

	nexus_debug_printf(2, ("rtp_write_data(): enter\n"));

	gettimeofday(&time, 0);

	h.version= RTP_VERSION;  /* RTP version */
	h.p = 0;                 /* no pad */
	h.x = 0;                 /* no extension */
	h.cc = 0;                /* number of CSRC = 0 */
	h.m = 0;                 /* no mark */
	h.pt= NEXUS_PAYLOAD_TYPE;/* 110 */
	h.seq = htons((u_int16)seq);
	h.ts = htonl((time.tv_sec << 16) | (time.tv_usec >> 16));
	h.ssrc = htonl(src);
	h.csrc[0] = 0;

	memcpy((void *)data_start, &h, sizeof(rtp_hdr_t));
	nexus_debug_printf(2, ("rtp_write_data(): exit\n"));

} /* rtp_write_data */


/*
* Return IP address
*/
struct in_addr*  gethostip()
{
  struct in_addr *in_ap;
  register struct hostent *hep;
  char hostname[MAXHOSTNAMELEN]; 

  globus_libc_gethostname(hostname, MAXHOSTNAMELEN);

  in_ap = (struct in_addr *)calloc(1, sizeof(struct in_addr));
  if ((hep = gethostbyname(hostname))) {
	*in_ap = *(struct in_addr *)(hep->h_addr_list[0]);
  }else {/* try YP */
    static char *domain = 0;  /* YP domain */
    char *value;              /* key value */
    int value_len;            /* length of returned value */

    if (!domain) yp_get_default_domain(&domain);
    if (yp_match(domain, "hosts.byname", hostname, strlen(hostname), &value, &value_len)== 0) {
      in_ap->s_addr = inet_addr(value);
    }
  }
  return ((struct in_addr *)in_ap);
} /* gethostip */



/*
 * set file descriptor as nonblocking 
 */

void fd_set_nonblocking(int fd)
{
    int flags;
    int save_errno;

    if ((flags = fcntl(fd, F_GETFL, 0)) < 0)
    {
        save_errno = errno;
	/* error routine */
    }

#ifdef TARGET_ARCH_HPUX
    flags |= O_NONBLOCK;
#else
    flags |= O_NDELAY;
#endif

    if (fcntl(fd, F_SETFL, flags) == -1)
    {
	save_errno = errno;
	/* error routine */
    }
}


/*
 * Create SR or RR packet. if flag == 0, then create SR packet, otherwise,
 * create RR packet.
 */
static int rtp_write_traffic(char *buffer,
			            nexus_transformstate_t *state,
			            int flag)
{

	rtcp_t *r;
	rtcp_rr_t * rr;
	struct timeval now;
	u_int32 now_ntp32;
	int len = 1;

	nexus_debug_printf(2, ("rtp_write_traffic(): enter\n"));

	r = (rtcp_t *)buffer;

	r->common.version = RTP_VERSION;
	r->common.p = 0;               /* no padding */

	r->common.count = 0;
	rr = r->r.rr.rr;
		
	gettimeofday(&now, 0);
	now_ntp32 = timeval_ntp32(&now);

	if(flag == 0) { /* sender report */
	   r->common.pt = RTCP_SR;
	   r->r.rr.ssrc = ((startpoint_transform_state_t *)(state))->src_id;
	   r->r.sr.rtp_ts = htonl((now.tv_sec << 16) | (now.tv_usec >> 16));
	   gettimeofday_ntp(&r->r.sr.ntp_sec);
	   r->r.sr.ntp_sec = htonl(r->r.sr.ntp_sec);
	   r->r.sr.ntp_frac  = htonl(r->r.sr.ntp_frac);
	   r->r.sr.psent     = htonl(((startpoint_transform_state_t *)(state))
					->packets[0]);
	   r->r.sr.osent     = htonl(((startpoint_transform_state_t *)(state))
					->bytes);
	   rr = r->r.sr.rr;
	   len += 5;
	} else { /* receiver report */
	   int newly_expected, newly_received;

	   r->common.pt = RTCP_RR;
	   r->r.rr.ssrc = ((nexus_session_t *)(state))->this_endnode_id;
	   
	   /* updates statistics */

	   /* 
	    * Assuming cycles = 0, init_seq = 1			    
	    * expected = cycles + max_seq - init_seq + 1            
	    * so, expected == max_seq				    
	    */
	   ((nexus_session_t *)(state))->expected 
			= ((nexus_session_t *)(state))->sequence;
	   newly_expected = ((nexus_session_t *)(state))->expected 
				- ((nexus_session_t *)(state))->last.expected;
	   newly_received = ((nexus_session_t *)(state))->received 
				- ((nexus_session_t *)(state))->last.received;

	   rr->ssrc = ((nexus_session_t *)(state))->sender_id;
	   if(newly_expected)
		rr->fraction = (double)(newly_expected - newly_received) 
							/ newly_expected * 256;
	   else
		rr->fraction = 0;
	   
	   rr->last_seq = ((nexus_session_t *)(state))->sequence;
	   rr->lost = ((nexus_session_t *)(state))->expected 
				- ((nexus_session_t *)(state))->received;
	   rr->jitter = ((nexus_session_t *)(state))->jitter;
	   rr->lsr = htonl( (((nexus_session_t *)(state))->lsr_sec << 16)
			| (((nexus_session_t *)(state))->lsr_frac >> 16));
	   
	   if (  ((nexus_session_t *)(state))->last_arrival[1].tv_sec )
		rr->dlsr = htonl(now_ntp32 - timeval_ntp32(&(((nexus_session_t *)(state))->last_arrival[1])));
 	   else rr->dlsr = 0;
	   rr++;
	   r->common.count++;
	   len +=6;

	   ((nexus_session_t *)(state))->last.expected 
			= ((nexus_session_t *)(state))->expected;
	   ((nexus_session_t *)(state))->last.received 
			= ((nexus_session_t *)(state))->received;
	}
	r->common.length = len;
	return((r->common.length + 1) << 2 );
}


/*
 * Convert timeval to NTP-32 timestamp
 */

u_int32 timeval_ntp32(struct timeval *tv)
{
  return ((tv->tv_sec  + 2208988800U) << 16) |
	   ((u_long)(tv->tv_usec * 4294.967296) >> 16);
} /* timeval_ntp32 */


/*
* Return current time as NTP timestamp.
*/

void gettimeofday_ntp(u_int32 *t)
{
  struct timeval tv;

  gettimeofday(&tv, 0);
  t[0] = tv.tv_sec  + 2208988800U;
  t[1] = tv.tv_usec * 4294.967296;
} /* gettimeofday_ntp */



/*
 * Process RTCP packets
 */

static void rtp_read_control(nexus_transformstate_t *state, char *mesg)
{

	long len;
	rtcp_t *r= (rtcp_t *)mesg;

	len = r->common.length + 1;
	while( len > 0) {
	   len -= r->common.length + 1;

	   if(len < 0 || r->common.version != RTP_VERSION) {
		rtp_fatal("rtp_read_control(): rtp bad header: len = %d \
				version=%d\n", len, r->common.version);
	   }

	   switch(r->common.pt) {
	   case RTCP_BYE: break;    /*  no RTCP_BYE */
	   case RTCP_SDES: break;   /*  no  RTCP_SDES */
	   case RTCP_SR:
	   case RTCP_RR:
		rtp_read_traffic(state, r);
		break;

		   default:
		break;
	   } /* end switch */
	   r = (rtcp_t *)((u_int32 *)r + r->common.length + 1);
	}
}


/*
 * Decode incoming SR/RR report 
 */

static void rtp_read_traffic(nexus_transformstate_t *state, rtcp_t *r)
{

	rtcp_rr_t *rr;
	int i;

	nexus_session_t *session_node;
	startpoint_transform_state_t *startpoint_state;
	
	/* sender report */
	if(r->common.pt == RTCP_SR) { /* sender report */
		session_node = (nexus_session_t *)state;

		session_node->lsr_sec = ntohl(r->r.sr.ntp_sec);
		session_node->lsr_frac = ntohl(r->r.sr.ntp_frac);
		gettimeofday(&(session_node->last_arrival[1]), 0);
		rr = r->r.sr.rr;
	} else { /* receiver report */
		u_int32 rtt32;

		startpoint_state = (startpoint_transform_state_t *)state;

		/* last arrival (RR) */
		gettimeofday(&(startpoint_state->last), 0); 
		rr = r->r.rr.rr;

		/* compute rtt */
		rtt32 =  timeval_ntp32(&(startpoint_state->last))
						- rr->dlsr - rr->lsr;

		startpoint_state->rtt.tv_sec = rtt32 / ( 1 << 16);
		startpoint_state->rtt.tv_usec = rtt32 % (1 << 16);

		nexus_debug_printf(2, ("rtp_read_traffic(): last = %d rtt =  \
		     %d\n", timeval_ntp32(&(startpoint_state->last)), rtt32));
		nexus_debug_printf(2, ("rtp_read_traffic(): rtt time is %d \
		    sec %d useci\n", startpoint_state->rtt.tv_sec , \
		    startpoint_state->rtt.tv_usec));
	}

	/* process receiver reports */
	for(i = 0; i < r->common.count; i++) {
		/* process receiver reports */
		startpoint_state->jitter_estimated = rr->jitter;
		rr++;
	}
} /* rtp_read_traffic */

#if 0
int size_transform_info_except_rtphd()
{
	/* ep_seq_liba and sequence */
	return( 2 * LARGEST_POSSIBLE_SIZEOF_ULONG );
}
#endif


/*
 * add the first two timeval and result in the third timeval
 */

void timevaladd(struct timeval *a, struct timeval *b,
					struct timeval *sum)
{
  sum->tv_usec = a->tv_usec + b->tv_usec;
  if (sum->tv_usec > 1000000L) {
    sum->tv_sec = a->tv_sec + b->tv_sec + 1;
    sum->tv_usec -= 1000000L;
  }
  else {
    sum->tv_sec = a->tv_sec + b->tv_sec;
  }
} /* timevaladd */



/*
 * Return 1 if a < b, 0 otherwise.
 */

int timevalless(struct timeval *a, struct timeval *b)
{
 if (a->tv_sec < b->tv_sec ||
     (a->tv_sec == b->tv_sec && a->tv_usec < b->tv_usec)) return 1;
 return 0;
} /* timevalless */


/*
 * return 1 if it's time to send a sender report, 0 otherwise
 * and compute time interval to send a next packet
 */
 
int time_to_send_SR(nexus_transformstate_t *sp_state)
{
   struct timeval now;
   startpoint_transform_state_t *s = (startpoint_transform_state_t *)sp_state;

   switch(send_sr_type) {
   case RTCP_COUNT:
     if((s->sequence % sr_frequency) == 0)
	return 1;
     else 
	return 0;
   case RTCP_DEFAULT:
   case RTCP_SECOND:
     gettimeofday(&now, 0);
     if(timevalless(&now, &nexttime))
	return 0;
     else{
	if(send_sr_type ==  RTCP_DEFAULT) {
	    double period;

	    period = rtcp_period(s->members, s->senders, 
			s->control.bw, 1, s->control.last, &s->control.avg,
			s->control.avg == 0.);
 	    period *= (drand48() + 0.5);
	    nexus_debug_printf(1,("time_to_send_SR(): period = %f\n", period));

	    delta.tv_sec = (unsigned long)period;
	    delta.tv_usec = ( period - delta.tv_sec ) * 1e6;
	  }	
	timevaladd(&now, &delta, &nexttime);
	return 1;
     }
   default:
	return 0;
   } /* switch */
}


/*
 * return 1 if it's time to send a receiver report, 0 otherwise
 * and compute time interval to send a next packet
 */

int time_to_send_RR(nexus_transformstate_t *ep_state)
{
   struct timeval now;
   nexus_session_t *node 
		= ((endpoint_transform_state_t *)ep_state)->current_session;
   
   gettimeofday(&now, 0);
  
   if(timevalless(&now, &rr_nexttime))
	return 0;
   else {
	double period;

	period = rtcp_period(node->members, node->senders,
			node->control.bw, 0, node->control.last, 
			&node->control.avg, node->control.avg == 0.);
     	period *= (drand48() + 0.5);
	nexus_debug_printf(1,("time_to_send_RR(): period = %f\n", period));

	rr_delta.tv_sec = (unsigned long)period;
	rr_delta.tv_usec = ( period - delta.tv_sec ) * 1e6;

	timevaladd(&now, &rr_delta, &rr_nexttime);

	return 1;
   }
}/* time_to_send_RR */

	
/*
* Compute control packet period.
* members:        number of session members
* senders:        senders among members
* bw:             total session bandwidth
* we_sent:        we sent recently
* packet_size:    current packet size
* avg_rtcp_size:  smoothed average
* initial:        initial call
*/
double rtcp_period(int members, int senders, double bw, int we_sent,
  int packet_size, int *avg_rtcp_size, int initial)
{
  /*
   * Minimum time between control packets from this site (in seconds).
   * This time prevents the reports from 'clumping' when sessions are
   * small and the law of large numbers isn't helping to smooth
   * out the traffic. It also keeps the report interval from becoming
   * ridiculously small during transient outages like a network
   * partition.
   */
  double const RTCP_MIN_TIME = 5.;

  /*
   * Fraction of the control bandwidth to be shared among active
   * senders. (This fraction was chosen so that in a typical session
   * with one or two active senders, the computed report time would be
   * roughly equal to the minimum report time so that we don't
   * unnecessarily slow down receiver reports.) The receiver fraction
   * must be 1 - the sender fraction.
   */
  double const RTCP_SENDER_BW_FRACTION = 0.25;
  double const RTCP_RECEIVER_BW_FRACTION = (1 - RTCP_SENDER_BW_FRACTION);

  /*
   * Gain (smoothing constant) for the low pass filter that estimates
   * the average RTCP packet size (see Cadzow reference).
   */
  double const RTCP_SIZE_GAIN = (1./16.);


  double t = 0.;   /* interval */
  double rtcp_min_time = RTCP_MIN_TIME;
  int n;      /* number of members for computation */

  /*
   * Very first call at application start-up uses half the minimum
   * delay for quicker notification while still allowing some time before
   * reporting for randomization and to learn about other sources.
   * The avg. RTCP size is initialized to 128 bytes which is
   * conservative (it assumes everyone else is generating SRs instead
   * of RRs).
   */
  if (initial) {
    rtcp_min_time /= 2;
    *avg_rtcp_size = 128;
  }

  /*
   * If there were active senders, split the RTCP bandwidth between
   * senders and receivers. Otherwise, the receivers share all the
   * RTCP bandwidth.
   */
  n = members;
  if (senders > 0 && senders < members * RTCP_SENDER_BW_FRACTION) {
    if (we_sent) {
       bw *= RTCP_SENDER_BW_FRACTION;
       n   = senders;
    }
    else {
      bw *= RTCP_RECEIVER_BW_FRACTION;
      n  -= senders;
    }
  }

  /*
   * Update the average size estimate by the size of the report
   * packet we just sent.
   */
  *avg_rtcp_size += (packet_size - *avg_rtcp_size) * RTCP_SIZE_GAIN;

  /*
   * The effective number of sites times the average packet size is
   * the total number of bytes sent when each site sends a report.
   * Dividing this by the effective bandwidth gives the time interval
   * over which those packets must be sent in order to meet the
   * bandwidth target. In that time interval we send one report so
   * this time is also our average time between reports.
   */
  if (bw > 0)
    t = (*avg_rtcp_size) * n / bw;

  /*  Enforce minimum spacing. */
  if (t < rtcp_min_time) t = rtcp_min_time;

  /*
   * To avoid traffic bursts from unintended synchronization with
   * other sites, we then pick our actual next report interval as a
   * random number uniformly distributed between 0.5*t and 1.5*t.
   * We do this in the calling routine so that we can use the period
   * for timing out sources.
   */
  return t;
} /* rtcp_perid */

