/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "script.h"
#include "window.h"

#ifdef SCRIPT

#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <stdlib.h>

#include "perlscript.h"
#include "tclscript.h"
#include "settings.h"
#include "world.h"
#include "notebook.h"

#include "gtkgmo.h"

typedef struct _reg_macro     reg_macro;
typedef struct _reg_simple    reg_simple;

struct _reg_macro {
    int   script_type;
    char *macro_name;
    char *function_name;
    char *description;
};

struct _reg_simple {
    int   script_type;
    char *function_name;
};

GList *registered_macros = NULL;
GList *registered_starts = NULL;
GList *registered_ends   = NULL;
GList *registered_opens   = NULL;
GList *registered_closes   = NULL;
GList *registered_connects   = NULL;
GList *registered_disconnects   = NULL;

GList *get_file_names(const char *global_dir, const char *user_dir);

void handle_start_end(reg_simple *se) {
#ifdef PERL
    if(se->script_type == PERL_TYPE) {
	gm_perl_exec(se->function_name, NULL);
    }
#endif
#ifdef TCL
    if(se->script_type == TCL_TYPE) {
	gm_tcl_exec(se->function_name);
    }
#endif
}

void gm_script_init(const char *programname) {
    char *global_dir = SHAREDIR "/gmoo/scripts";
    char *user_dir   = gm_settings_get_scripts_dir();
    GList *files;
    GList *l;

    if(debug)
        printf("Initing scripts...\n\tglobal dir: \"%s\"\n"
               "\tuser   dir: \"%s\"\n", global_dir, user_dir);

    /* Start interpreters */
#ifdef PERL
    gm_perl_init(programname);
#endif
#ifdef TCL
    gm_tcl_init(programname);
#endif

    /* Load all files in the interpreters */
    files = get_file_names(global_dir, user_dir);
    for(l = files; l; l = g_list_next(l)) {
#ifdef PERL
        if(gm_perl_is_perl_file((char *) l->data)) {
            gm_perl_load_file((char *) l->data);
            continue;
        }
#endif
#ifdef TCL
        if(gm_tcl_is_tcl_file((char *) l->data)) {
            gm_tcl_load_file((char *) l->data);
            continue;
        }
#endif	
    }
    /* Clean up */
    g_list_foreach(files, (GFunc) g_free, NULL);
    g_list_free(files);

    g_list_foreach(registered_starts, (GFunc) handle_start_end, NULL);
}

void free_simple(GList **list) {
    GList *l;
    for(l = *list; l; l = g_list_next(l)) {
	g_free(((reg_simple *) (l->data))->function_name);
	g_free(l->data);
    }
    g_list_free(*list);
    *list = NULL;
}

void gm_script_exit() {
    GList *l;

    g_list_foreach(registered_ends, (GFunc) handle_start_end, NULL);

#ifdef PERL
    gm_perl_exit();
#endif
#ifdef TCL
    gm_tcl_exit();
#endif
    for(l = registered_macros; l; l = g_list_next(l)) {
	g_free(((reg_macro *) l->data)->macro_name);
	g_free(((reg_macro *) l->data)->function_name);
	g_free(((reg_macro *) l->data)->description);
	g_free(l->data);
    }
    g_list_free(registered_macros);
    registered_macros = NULL;

    free_simple(&registered_starts);
    free_simple(&registered_ends);
    free_simple(&registered_opens);
    free_simple(&registered_closes);
    free_simple(&registered_connects);
    free_simple(&registered_disconnects);
}

void parse_macro(const char *line, char **macro_name, char **argstr) {
    char *s;
    *macro_name = g_strdup(line);
    if((s = strchr(*macro_name, ' '))) {
	s[0] = 0;
	*argstr = g_strdup(s + 1);
    } else {
	*argstr = g_strdup("");
    }
}

reg_macro *find_macro(const char *name) {
    reg_macro *m;
    GList *l;
    for(l = registered_macros; l; l = g_list_next(l)) {
	m = l->data;
	if(strcmp(m->macro_name, name) == 0) {
	    return m;
	}
    }
    return NULL;
}

int gm_script_handle_macro(world *w, const char *line) {
    reg_macro *m;
    char *macro_name, *argstr;

    if(line[0] != '/') return FALSE;
    parse_macro(line + 1, &macro_name, &argstr);

    if((m = find_macro(macro_name))) {
#ifdef PERL
	if(m->script_type == PERL_TYPE) {
	    gm_perl_do_macro(w->id, m->function_name, argstr);
	}
#endif
#ifdef TCL
	if(m->script_type == TCL_TYPE) {
	    gm_tcl_do_macro(w->id, m->function_name, argstr);
	}
#endif
    }
    g_free(macro_name);
    g_free(argstr);

    return m ? TRUE : FALSE;
}

void do_simple_world_event(reg_simple *reg, world *w) {
    char *s;
#ifdef PERL
    if(reg->script_type == PERL_TYPE) {
	s = g_strdup_printf("%d", w->id);
	gm_perl_exec(reg->function_name, s);
	g_free(s);
    }
#endif
#ifdef TCL
    if(reg->script_type == TCL_TYPE) {
	s = g_strdup_printf("%s %d", reg->function_name, w->id);
	gm_tcl_exec(s);
	g_free(s);
    }
#endif
}

void gm_script_world_opened(world *w) {
    g_list_foreach(registered_opens, (GFunc) do_simple_world_event, w);
}

void gm_script_world_closed(world *w) {
    g_list_foreach(registered_closes, (GFunc) do_simple_world_event, w);
}

void gm_script_world_connected(world *w) {
    g_list_foreach(registered_connects, (GFunc) do_simple_world_event, w);
}

void gm_script_world_disconnected(world *w) {
    g_list_foreach(registered_disconnects, (GFunc) do_simple_world_event, w);
}

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

GList *get_file_names(const char *global_dir, const char *user_dir) {
    GList *files = NULL;
    GList *filenames = NULL;
    GList *l;
    DIR *dir;
    struct dirent *ent;

    /* add all files in the userdir */
    if((dir = opendir(user_dir))) {
        while((ent = readdir(dir))) {
	    if(ent->d_name[0] != '.') {
		filenames = g_list_append(filenames, g_strdup(ent->d_name));
		files = g_list_append(files,
				      g_strconcat(user_dir, "/",
						  ent->d_name, NULL));
	    }
	}
        closedir(dir);
    }

    /* add only new files in the global dir */
    if((dir = opendir(global_dir))) {
        while((ent = readdir(dir))) {
	    if(ent->d_name[0] != '.') {
		for(l = filenames; l; l = g_list_next(l)) {
		    if(strcmp((char *) l->data, ent->d_name) == 0)
			goto dont_add;
		}
		files = g_list_append(files,
				      g_strconcat(global_dir, "/",
						  ent->d_name, NULL));
	    dont_add:
	    }
	}
        closedir(dir);
    }
    /* clean up */
    g_list_foreach(filenames, (GFunc) g_free, NULL);
    g_list_free(filenames);

    return files;
}


/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

int gm_script_do_print(int id, const char *text) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    gm_world_print_buffered(w, text);
    return TRUE;
}

int gm_script_do_println(int id, const char *text) {
    char *s;
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    s = g_strconcat(text, "\n", NULL);
    gm_world_println(w, s);
    g_free(s);
    return TRUE;
}

int gm_script_do_msg(int id, const char *text) {
    char *s;
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    s = g_strconcat("% ", text, "\n", NULL);
    gm_world_print_buffered(w, s);
    g_free(s);
    return TRUE;
}

int gm_script_do_write(int id, const char *text) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    gm_world_write_buffered(w, text);
    return TRUE;
}

int gm_script_do_writeln(int id, const char *text) {
    char *s;
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    s = g_strconcat(text, "\n", NULL);
    gm_world_write_buffered(w, s);
    g_free(s);
    return TRUE;
}

int gm_script_do_name(int id, char **retval) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    *retval = w->p->name;
    return TRUE;
}

int gm_script_do_hostname(int id, char **retval) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    *retval = w->p->hostname;
    return TRUE;
}

int gm_script_do_port(int id, int *retval) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    *retval = w->p->port;
    return TRUE;
}

int gm_script_do_connected(int id, int *retval) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    *retval = gm_world_is_connected(w);
    return TRUE;
}

int gm_script_do_width(int id, int *retval) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    *retval = GTK_GMO(w->output)->width;
    return TRUE;
}

int gm_script_do_height(int id, int *retval) {
    world *w = gm_notebook_get_world_by_id(id);
    if(!w) return FALSE;
    *retval = GTK_GMO(w->output)->height;
    return TRUE;
}

int gm_script_do_register_open(int script_type, const char *name) {
    reg_simple *s;
    if(!name || !name[0]) return FALSE;
    s = g_malloc(sizeof(reg_simple));
    s->script_type = script_type;
    s->function_name = g_strdup(name);
    registered_opens = g_list_append(registered_opens, s);
    if(debug) printf("\t\tRegistered open %s\n", s->function_name);
    return TRUE;
}

int gm_script_do_register_close(int script_type, const char *name) {
    reg_simple *s;
    if(!name || !name[0]) return FALSE;
    s = g_malloc(sizeof(reg_simple));
    s->script_type = script_type;
    s->function_name = g_strdup(name);
    registered_opens = g_list_append(registered_opens, s);
    if(debug) printf("\t\tRegistered open %s\n", s->function_name);
    return TRUE;
}

int gm_script_do_register_connect(int script_type, const char *name) {
    reg_simple *s;
    if(!name || !name[0]) return FALSE;
    s = g_malloc(sizeof(reg_simple));
    s->script_type = script_type;
    s->function_name = g_strdup(name);
    registered_connects = g_list_append(registered_connects, s);
    if(debug) printf("\t\tRegistered connect %s\n", s->function_name);
    return TRUE;
}

int gm_script_do_register_disconnect(int script_type, const char *name) {
    reg_simple *s;
    if(!name || !name[0]) return FALSE;
    s = g_malloc(sizeof(reg_simple));
    s->script_type = script_type;
    s->function_name = g_strdup(name);
    registered_disconnects = g_list_append(registered_disconnects, s);
    if(debug) printf("\t\tRegistered disconnect %s\n", s->function_name);
    return TRUE;
}


int gm_script_do_register_macro(int script_type,
				const char *macroname,
				const char *funcname,
				const char *description) {
    reg_macro *m;
    if(!macroname[0] || !funcname[0] || find_macro(macroname)) {
	return FALSE;
    }

    m = g_malloc(sizeof(reg_macro));
    m->script_type    = script_type;
    m->macro_name     = g_strdup(macroname);
    m->function_name  = g_strdup(funcname);
    m->description    = g_strdup(description);
    registered_macros = g_list_append(registered_macros, m);
    if(debug) printf("\t\tRegistered macro %s\n", m->macro_name);
    return TRUE;
}

int gm_script_do_register_start(int script_type, const char *name) {
    reg_simple *s;
    if(!name || !name[0]) return FALSE;
    s = g_malloc(sizeof(reg_simple));
    s->script_type = script_type;
    s->function_name = g_strdup(name);
    registered_starts = g_list_append(registered_starts, s);
    if(debug) printf("\t\tRegistered start %s\n", s->function_name);
    return TRUE;
}

int gm_script_do_register_end(int script_type, const char *name) {
    reg_simple *e;
    if(!name || !name[0]) return FALSE;
    e = g_malloc(sizeof(reg_simple));
    e->script_type = script_type;
    e->function_name = g_strdup(name);
    registered_ends = g_list_append(registered_ends, e);
    if(debug) printf("\t\tRegistered end %s\n", e->function_name);
    return TRUE;
}

int gm_script_do_version(char **retval) {
    *retval = VERSION;
    return TRUE;
}

#define s_equals(a, b) (g_strcasecmp(a, b) == 0)
#define s_equals2(a, b, c) (s_equals(a, b) || s_equals(a, c))

#define NONE        1
#define STOP        2
#define QUESTION    3
#define WARNING     4
#define INFORMATION 5
int parse_type(const char *type) {
    if(s_equals2(type, "error", "stop")) return STOP;
    else if(s_equals(type, "question"))   return QUESTION;
    else if(s_equals2(type, "warning", "exclamation")) return WARNING;
    else if(s_equals2(type, "information", "asterix")) return INFORMATION;
    else if(s_equals(type, "none")) return NONE;
    return FALSE;
}

#define ABORT  (1 << 1)
#define RETRY  (1 << 2)
#define IGNORE (1 << 3)
#define OK     (1 << 4)
#define YES    (1 << 5)
#define NO     (1 << 6)
#define CANCEL (1 << 7)
int parse_buttons(const char *buttons, int *b, int *b_c) {
    if(s_equals(buttons, "ok")) {
	*b = OK; *b_c = 1; return TRUE;
    } else if(s_equals(buttons, "okcancel")) {
	*b = OK | CANCEL; *b_c = 2; return TRUE;
    } else if(s_equals(buttons, "retrycancel")) {
	*b = RETRY | CANCEL; *b_c = 2; return TRUE;
    } else if(s_equals(buttons, "yesno")) {
	*b = YES | NO; *b_c = 2; return TRUE;
    } else if(s_equals(buttons, "yesnocancel")) {
	*b = YES | NO | CANCEL; *b_c = 3; return TRUE;
    } else if(s_equals(buttons, "abortretryignore")) {
	*b = ABORT | RETRY | IGNORE; *b_c = 3; return TRUE;
    }
    return FALSE;
}

void dialog_buttons_pressed(GtkWidget *button, int value) {
    int *pressed = gtk_object_get_data(GTK_OBJECT(button), "pressed_int");
    *pressed = value;
}

int gm_script_do_msgbox(const char *title, const char *type_s,
			const char *buttons_s, const char *text,
			int default_button, int *retval) {
    int type, buttons, buttons_count, i, pressed = -1, delete;
    GtkWidget *buttons_w[3];
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *hbb;
    GtkWidget *tmp;

    if(!(type = parse_type(type_s)) ||
       !(parse_buttons(buttons_s, &buttons, &buttons_count))) {
	return FALSE;
    }

    window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(window),
				 GTK_WINDOW(gm_window_get_window()));
    gtk_window_set_title(GTK_WINDOW(window), title);
    gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
    gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, TRUE);
    gtk_window_set_modal(GTK_WINDOW(window), TRUE);
    delete = gtk_signal_connect(GTK_OBJECT(window), "delete_event",
				GTK_SIGNAL_FUNC(gtk_true), NULL);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add(GTK_CONTAINER(window), vbox);

    /* TODO: Add supprt for icons on the dialog window */
    tmp = gtk_label_new(text);
    gtk_misc_set_padding(GTK_MISC(tmp), 5, 5);
    gtk_widget_show(tmp);
    gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);

    tmp = gtk_hseparator_new ();
    gtk_widget_show (tmp);
    gtk_box_pack_start (GTK_BOX (vbox), tmp, TRUE, TRUE, 0);

    hbb = gtk_hbutton_box_new ();
    gtk_widget_show (hbb);
    gtk_box_pack_start (GTK_BOX (vbox), hbb, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbb), 5);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbb), 0);
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbb), 3, 0);

    i = 0;
    if(buttons & ABORT)  buttons_w[i++]=gtk_button_new_with_label(_("Abort"));
    if(buttons & RETRY)  buttons_w[i++]=gtk_button_new_with_label(_("Retry"));
    if(buttons & IGNORE) buttons_w[i++]=gtk_button_new_with_label(_("Ignore"));
    if(buttons & OK)     buttons_w[i++]=gtk_button_new_with_label(_("Ok"));
    if(buttons & RETRY)  buttons_w[i++]=gtk_button_new_with_label(_("Retry"));
    if(buttons & YES)    buttons_w[i++]=gtk_button_new_with_label(_("Yes"));
    if(buttons & NO)     buttons_w[i++]=gtk_button_new_with_label(_("No"));
    if(buttons & CANCEL) buttons_w[i++]=gtk_button_new_with_label(_("Cancel"));

    for(i = 0; i < buttons_count; i++) {
	gtk_widget_show(buttons_w[i]);
	gtk_container_add(GTK_CONTAINER(hbb), buttons_w[i]);
	GTK_WIDGET_SET_FLAGS(buttons_w[i], GTK_CAN_DEFAULT);
	gtk_object_set_data(GTK_OBJECT(buttons_w[i]), "pressed_int", &pressed);
	gtk_signal_connect(GTK_OBJECT(buttons_w[i]), "clicked",
			   GTK_SIGNAL_FUNC(dialog_buttons_pressed), (void *)i);
    }
    tmp = default_button >= 0 ? 
	buttons_w[CLAMP(default_button, 0, buttons_count - 1)] :
	buttons_w[buttons_count - 1];
    gtk_widget_grab_default(tmp);
    gtk_widget_grab_focus(tmp);

    gtk_widget_show(window);

    while(pressed < 0) {
	gtk_main_iteration_do(TRUE);
    }

    *retval = pressed;

    gtk_signal_disconnect(GTK_OBJECT(window), delete);
    gtk_widget_destroy(window);

    return TRUE;
}


int parse_type_input(const char *type) {
    if(s_equals(type, "string")) {
	return TYPE_STR;
    } else if(s_equals2(type, "int", "integer")) {
	return TYPE_INT;
    }
    return FALSE;
}

int gm_script_do_input(const char *title, const char *description,
		       const char *type_str, int *type_int,
		       const char *def,
		       char **ret_str, int *ret_int) {
    int pressed = 0;
    int delete;
    int func_return;
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *vbox2;
    GtkWidget *label;
    GtkObject *spinbutton_adj;
    GtkWidget *value_w;
    GtkWidget *hseparator;
    GtkWidget *hbb;
    GtkWidget *ok_but;
    GtkWidget *cancel_but;

    if(!(*type_int = parse_type_input(type_str))) {
	return FALSE;
    }

    window = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(window),
				 GTK_WINDOW(gm_window_get_window()));
    gtk_window_set_title (GTK_WINDOW (window), title);
    gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_window_set_policy (GTK_WINDOW (window), FALSE, FALSE, TRUE);
    delete = gtk_signal_connect(GTK_OBJECT(window), "delete_event",
				GTK_SIGNAL_FUNC(gtk_true), NULL);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    vbox2 = gtk_vbox_new (FALSE, 5);
    gtk_widget_show (vbox2);
    gtk_box_pack_start (GTK_BOX (vbox), vbox2, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox2), 5);

    label = gtk_label_new (description);
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

    if(*type_int == TYPE_INT) {
	spinbutton_adj = gtk_adjustment_new(atoi(def), G_MININT, G_MAXINT,
					    1, 10, 10);
	value_w = gtk_spin_button_new(GTK_ADJUSTMENT(spinbutton_adj), 1, 0);
	gtk_widget_show (value_w);
	gtk_box_pack_start (GTK_BOX (vbox2), value_w, FALSE, FALSE, 0);
    } else { /* *type_int == TYPE_STR */
	value_w = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(value_w), def);
	gtk_widget_show(value_w);
	gtk_box_pack_start (GTK_BOX (vbox2), value_w, FALSE, FALSE, 0);
    }

    hseparator = gtk_hseparator_new ();
    gtk_widget_show (hseparator);
    gtk_box_pack_start (GTK_BOX (vbox), hseparator, FALSE, TRUE, 5);

    hbb = gtk_hbutton_box_new ();
    gtk_widget_show (hbb);
    gtk_box_pack_start (GTK_BOX (vbox), hbb, FALSE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbb), 5);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbb), GTK_BUTTONBOX_END);

    ok_but = gtk_button_new_with_label (_("Ok"));
    gtk_widget_show (ok_but);
    gtk_container_add (GTK_CONTAINER (hbb), ok_but);
    GTK_WIDGET_SET_FLAGS (ok_but, GTK_CAN_DEFAULT);
    gtk_object_set_data(GTK_OBJECT(ok_but), "pressed_int", &pressed);
    gtk_signal_connect(GTK_OBJECT(ok_but), "clicked",
		       GTK_SIGNAL_FUNC(dialog_buttons_pressed), (void *) OK);

    cancel_but = gtk_button_new_with_label (_("Cancel"));
    gtk_widget_show (cancel_but);
    gtk_container_add (GTK_CONTAINER (hbb), cancel_but);
    GTK_WIDGET_SET_FLAGS (cancel_but, GTK_CAN_DEFAULT);
    gtk_object_set_data(GTK_OBJECT(cancel_but), "pressed_int", &pressed);
    gtk_signal_connect(GTK_OBJECT(cancel_but), "clicked",
		       GTK_SIGNAL_FUNC(dialog_buttons_pressed),(void *)CANCEL);

    gtk_widget_show(window);

    while(!pressed) {
	gtk_main_iteration_do(TRUE);
    }

    if(pressed == OK) {
	if(*type_int == TYPE_INT) {
	    *ret_int =
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(value_w));
	} else { /* *type_int == TYPE_STR */
	    *ret_str = g_strdup(gtk_entry_get_text(GTK_ENTRY(value_w)));
	}
	func_return = TRUE;
    } else { /* pressed == CANCEL */
	func_return = FALSE;
    }

    gtk_signal_disconnect(GTK_OBJECT(window), delete);
    gtk_widget_destroy(window);

    return func_return;
}

#endif /* SCRIPT */
