/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "config.h"

#include "configdb.h"
#include "settings.h"
#include "rcfile.h"
#include "dialog.h"
#include "world.h"
#include "notebook.h"
#include "toolbar.h"
#ifdef ZVT
#   include <zvt/zvtterm.h>
#endif

#include "gtkgmo.h"

char *config_dir = NULL;
char *config_main_file = NULL;
char *config_worlds_file = NULL;
char *config_editor_dir = NULL;
char *config_scripts_dir = NULL;
char *config_worlds_dir = NULL;
char *config_log_dir = NULL;
char *config_pixmap_dir = NULL;
char *config_trigger_file = NULL;

extern GtkWidget *main_window;


#define matches(a,b)    (strcmp(a,b) == 0)
#define matches2(a,b,c) (strcmp(a,b) == 0 || strcmp(a, c) == 0)

void init_colors_and_styles();

char *gm_settings_get_config_dir() {
    return config_dir;
}

char *gm_settings_get_log_dir() {
    return config_log_dir;
}

char *gm_settings_get_editor_dir() {
    return config_editor_dir;
}

char *gm_settings_get_scripts_dir() {
    return config_scripts_dir;
}

char *gm_settings_get_worlds_dir() {
    return config_worlds_dir;
}

char *gm_settings_get_pixmap_dir() {
    return config_pixmap_dir;
}

char *gm_settings_get_trigger_file() {
    return config_trigger_file;
}

void print_help_message() {
    printf("Usage: gmoo [Options]\n"
           "\n"
           "Where Options:\n"
           "  -h, --help                  Show this help message\n"
           "  -v, --version               Show the version of gmoo\n"
           "  -d, --debug                 Output all kinds of debug output\n"
           "\n"
           "And of cource all the GTK+ options (not listed here)\n"
           "\n");

}

void parse_args(int argc, char *argv[]) {
    int i;
    int do_exit = FALSE;
    for(i = 1; i < argc; i++) {
        if(matches2(argv[i], "-d", "--debug"))
            debug = TRUE;
        else if(matches2(argv[i], "-v", "--version")) {
            printf(_("gmoo version %s\n"), VERSION);
            do_exit = TRUE;
        } else if(matches2(argv[i], "-h", "--help")) {
            print_help_message();
            do_exit = TRUE;
        } else {
            printf(_("gmoo: Error, Invalid argument: %s\n"), argv[i]);
            exit(1);
        }
    }
    if(do_exit)
        exit(0);

#if 0
    if(argc == 1) {
        return;
    } else if(argc == 2) {
        if(matches2(argv[1], "-h", "--help")) {
            print_help_message();
        } else if(matches2(argv[1], "-v", "--version")) {
            printf("gmoo version "VERSION"\n");
        } else {
            printf("gmoo: Error, Invalid argument\n");
            exit(1);
        }
    } else {
        printf("gmoo: Error, Invalid number of argumens\n");
        exit(1);
    }
    exit(0);
#endif
}

/* Returns TRUE upong successfull creation, FALSE otherwise */
int checkout_dir(const char *dirname) {
    if(mkdir(dirname, 00755) == 0) {
	return TRUE;
    } else if(errno != EEXIST) {
	printf("gmoo: Error creating dir: %s\n", strerror(errno));
    }
    return FALSE;
}

void gm_settings_init(int argc, char *argv[]) {
    if(debug) printf("Initing settings\n");

    parse_args(argc, argv);

    config_dir         = g_strconcat(g_get_home_dir(), "/.gmoo",   NULL);
    config_editor_dir  = g_strconcat(config_dir,       "/editor",  NULL);
    config_scripts_dir = g_strconcat(config_dir,       "/scripts", NULL);
    config_worlds_dir  = g_strconcat(config_dir,       "/worlds",  NULL);
    config_log_dir     = g_strconcat(config_dir,       "/logs",    NULL);
    config_pixmap_dir  = g_strconcat(config_dir,       "/pixmaps", NULL);
    config_trigger_file = g_strconcat(config_dir, "/triggers", NULL);

    config_main_file   = g_strconcat(config_dir,       "/gmoorc",  NULL);

    checkout_dir(config_dir);
    checkout_dir(config_editor_dir);
    checkout_dir(config_scripts_dir);
    checkout_dir(config_worlds_dir);
    checkout_dir(config_log_dir);
    checkout_dir(config_pixmap_dir);

    gm_world_create_template();

    chmod(config_log_dir,    00700);
    chmod(config_worlds_dir, 00700);

    settings = gm_settings_alloc();

    gm_settings_load(settings);
}

#include "color_scheme.h"
void settings_get_rgbs(gushort *reds, gushort *greens, gushort *blues) {
    int i;
    if(settings->color_palette == COLOR_CUSTOM) {
        for(i = 0; i < 16; i++) {
            reds[i]   = settings->reds[i];
            greens[i] = settings->greens[i];
            blues[i]  = settings->blues[i];
        }
    } else {
        for(i = 0; i < N_COLORS; i++) {
            reds[i]   = color_palettes[settings->color_palette - 1][0][i];
            greens[i] = color_palettes[settings->color_palette - 1][1][i];
            blues[i]  = color_palettes[settings->color_palette - 1][2][i];
        }
    }
    if(settings->fgbg_palette == COLOR_CUSTOM) {
        for(i = 16; i < 18; i++) {
            reds[i]   = settings->reds[i];
            greens[i] = settings->greens[i];
            blues[i]  = settings->blues[i];
        }
    } else {
        for(i = 16; i < 18; i++) {
            reds[i]   = fgbg_palettes[settings->fgbg_palette - 1][0][i - 16];
            greens[i] = fgbg_palettes[settings->fgbg_palette - 1][1][i - 16];
            blues[i]  = fgbg_palettes[settings->fgbg_palette - 1][2][i - 16];
        }
    }
    reds[FG_SELECT] = select_fg[0];
    greens[FG_SELECT] = select_fg[1];
    blues[FG_SELECT] = select_fg[2];
    reds[BG_SELECT] = select_bg[0];
    greens[BG_SELECT] = select_bg[1];
    blues[BG_SELECT] = select_bg[2];
}

extern GtkWidget *main_window;
void set_color(GdkColor *c, gushort r, gushort g, gushort b) {
    if(c->pixel) {
        if(c->red == r && c->green == g && c->blue == b) {
            if(debug) printf("Reallocating not nessecary, "
                             "color stays the same.\n");
            return;
        }
        gdk_colormap_free_colors(gtk_widget_get_colormap(main_window), c, 1);
        if(debug) printf("Reallocating color from %x/%x/%x to %x/%x/%x\n",
                         c->red, c->green, c->blue, r, g, b);
    } else if(debug) {
        printf("Allocating color %x/%x/%x\n", r, g, b);
    }

    c->red = r;
    c->green = g;
    c->blue = b;
    c->pixel = (gulong) ((r & 0xff00) * 256 +
                         (g & 0xff00) +
                         (b &0xff00) / 256);

    /*  if(!gdk_color_alloc(gtk_widget_get_colormap(main_window), c)) { */
    if(!gdk_colormap_alloc_color(gtk_widget_get_colormap(main_window),
                                 c, FALSE, TRUE)) {
        fputs("gmoo.error: cannot alloc color :( !\n", stderr);
    }
}

#define copy_color(dest, source) memcpy(dest, source, sizeof(GdkColor))
void update_colors_and_styles() {
    gushort r[N_COLORS];
    gushort g[N_COLORS];
    gushort b[N_COLORS];
    int i;

    settings_get_rgbs(r, g, b);
    for(i = 0; i < N_COLORS; i++) {
        set_color(&colors[i], r[i], g[i], b[i]);
    }
    set_color(&redtab_style->fg[GTK_STATE_NORMAL], 0xffff, 0, 0);
    set_color(&bluetab_style->fg[GTK_STATE_NORMAL], 0, 0, 0xffff);

    copy_color(&input_style->text[GTK_STATE_NORMAL], &colors[FG_COLOR]);
    copy_color(&input_style->base[GTK_STATE_NORMAL], &colors[BG_COLOR]);
    gdk_font_unref(input_style->font);
    input_style->font = gdk_font_load(settings->fontname);
}

void init_colors_and_styles() {
    int i;
    static int done = FALSE;

    if(done) return;
    
    for(i = 0; i < N_COLORS; i++) {
        colors[i].pixel = 0;
        colors[i].red   = 0;
        colors[i].green = 0;
        colors[i].blue  = 0;
    }
    redtab_style  = gtk_style_new();
    redtab_style->font = gtk_widget_get_style(main_window)->font;
    bluetab_style = gtk_style_new();
    bluetab_style->font = gtk_widget_get_style(main_window)->font;
    input_style   = gtk_style_new();
    
    redtab_style->fg[GTK_STATE_NORMAL].pixel = 0;
    bluetab_style->fg[GTK_STATE_NORMAL].pixel = 0;
    redtab_style->fg[GTK_STATE_NORMAL].pixel = 0;
}

void gm_settings_update() {
    init_colors_and_styles();
    update_colors_and_styles();
}

void set_hotkey_mask(settings_t *s) {
    if (s->hotkey == HOTKEY_EITHER)
        s->hotkey_mask = GDK_MOD1_MASK | GDK_CONTROL_MASK;
    else if(s->hotkey == HOTKEY_ALT)
        s->hotkey_mask = GDK_MOD1_MASK;
    else
        s->hotkey_mask = GDK_CONTROL_MASK;
}

#define do_get_color(a,b,c) \
configdb_get_rgb(db, b, c, &(s->reds[a]), \
		 &(s->greens[a]), \
		 &(s->blues[a]))

void gm_settings_load(settings_t *s) {
    ConfigDB *db;
    if(debug) printf("Loading config file data\n");

    db = configdb_load(config_main_file, FALSE);

    do_get_color(BLACK,   COLOR_BLACK,   _COLOR_BLACK);
    do_get_color(RED,     COLOR_RED,     _COLOR_RED);
    do_get_color(GREEN,   COLOR_GREEN,   _COLOR_GREEN);
    do_get_color(YELLOW,  COLOR_YELLOW,  _COLOR_YELLOW);
    do_get_color(BLUE,    COLOR_BLUE,    _COLOR_BLUE);
    do_get_color(MAGENTA, COLOR_MAGENTA, _COLOR_MAGENTA);
    do_get_color(CYAN,    COLOR_CYAN,    _COLOR_CYAN);
    do_get_color(WHITE,   COLOR_WHITE,   _COLOR_WHITE);

    do_get_color(BRIGHT+BLACK,   COLOR_BBLACK,   _COLOR_BBLACK);
    do_get_color(BRIGHT+RED,     COLOR_BRED,     _COLOR_BRED);
    do_get_color(BRIGHT+GREEN,   COLOR_BGREEN,   _COLOR_BGREEN);
    do_get_color(BRIGHT+YELLOW,  COLOR_BYELLOW,  _COLOR_BYELLOW);
    do_get_color(BRIGHT+BLUE,    COLOR_BBLUE,    _COLOR_BBLUE);
    do_get_color(BRIGHT+MAGENTA, COLOR_BMAGENTA, _COLOR_BMAGENTA);
    do_get_color(BRIGHT+CYAN,    COLOR_BCYAN,    _COLOR_BCYAN);
    do_get_color(BRIGHT+WHITE,   COLOR_BWHITE,   _COLOR_BWHITE);

    do_get_color(FG_COLOR, COLOR_FORE, _COLOR_FORE);
    do_get_color(BG_COLOR, COLOR_BACK, _COLOR_BACK);

    s->color_palette = configdb_get_int(db, COLOR_PALETTE, _COLOR_PALETTE);
    s->fgbg_palette  = configdb_get_int(db, FGBG_PALETTE,  _FGBG_PALETTE);

    /* Window */
    s->save_window_size = configdb_get_int(db, SAVE_WINDOW_SIZE,
                                           _SAVE_WINDOW_SIZE);
    s->window_height = configdb_get_int(db, WINDOW_HEIGHT, _WINDOW_HEIGHT);
    s->window_width  = configdb_get_int(db, WINDOW_WIDTH,  _WINDOW_WIDTH);

    /* Fonts */
    s->fontname   = configdb_get_string(db, FONT, _FONT);

    /* notebook */
    s->tab_position            = configdb_get_int(db, TAB_POSITION,
                                                  _TAB_POSITION);
    s->tab_limit_length        = configdb_get_int(db, TAB_LIMIT_LENGTH,
                                                  _TAB_LIMIT_LENGTH);
    s->tab_max_length          = configdb_get_int(db, TAB_MAX_LENGTH,
                                                  _TAB_MAX_LENGTH);
    s->homo_tabs               = configdb_get_int(db, HOMO_TABS, _HOMO_TABS);
    s->tabs_show_lines_waiting = configdb_get_int(db, TABS_SHOW_LINES_WAITING,
                                                  _TABS_SHOW_LINES_WAITING);
    s->hotkey                  = configdb_get_int(db, HOTKEY, _HOTKEY);
    s->hotkey_arrow            = configdb_get_int(db, HOTKEY_ARROW,
                                                  _HOTKEY_ARROW);

    s->toolbar                 = configdb_get_int(db, TOOLBAR,
                                                  _TOOLBAR);

    /* World showing */
    s->save_input_height       = configdb_get_int(db, SAVE_INPUT_HEIGHT,
                                                  _SAVE_INPUT_HEIGHT);

    s->editor_external  = configdb_get_int(db, EDITOR_EXTERNAL,
                                           _EDITOR_EXTERNAL);
    s->editor_use_env  = configdb_get_int(db, EDITOR_USE_ENV,
                                          _EDITOR_USE_ENV);
    s->editor_needs_term  = configdb_get_int(db, EDITOR_NEEDS_TERM,
                                             _EDITOR_NEEDS_TERM);
    s->editor_command  = configdb_get_string(db, EDITOR_COMMAND,
                                             _EDITOR_COMMAND);
    s->editor_delete_file = configdb_get_int(db, EDITOR_DELETE_FILE,
                                             _EDITOR_DELETE_FILE);

    s->userlist_pos  = configdb_get_int(db, USERLIST_POS,
                                        _USERLIST_POS);
    s->userlist_resizeable  = configdb_get_int(db, USERLIST_RESIZEABLE,
                                               _USERLIST_RESIZEABLE);
    s->userlist_totals  = configdb_get_int(db, USERLIST_TOTALS,
                                        _USERLIST_TOTALS);
    s->userlist_headers  = configdb_get_int(db, USERLIST_HEADERS,
                                        _USERLIST_HEADERS);
    s->userlist_objects  = configdb_get_int(db, USERLIST_OBJECTS,
                                        _USERLIST_OBJECTS);
    s->userlist_friends_color  = configdb_get_int(db, USERLIST_FRIENDS_COLOR,
                                                  _USERLIST_FRIENDS_COLOR);
    s->userlist_auto_open  = configdb_get_int(db, USERLIST_AUTO_OPEN,
                                        _USERLIST_AUTO_OPEN);

    s->tab_pos_on_tab = configdb_get_int(db, TAB_POS_ON_TAB, _TAB_POS_ON_TAB);

    s->restore_open_worlds = configdb_get_int(db, RESTORE_OPEN_WORLDS,
                                              _RESTORE_OPEN_WORLDS);
    s->enable_run_menu = configdb_get_int(db, ENABLE_RUN_MENU,
                                          _ENABLE_RUN_MENU);

    s->history = configdb_get_int(db, HISTORY, _HISTORY);
    s->history_size = configdb_get_int(db, HISTORY_SIZE, _HISTORY_SIZE);
    s->history_selective = configdb_get_int(db, HISTORY_SELECTIVE,
                                            _HISTORY_SELECTIVE);
    s->history_sensitive = configdb_get_int(db, HISTORY_SENSITIVE,
                                            _HISTORY_SENSITIVE);
    s->history_store = configdb_get_int(db, HISTORY_STORE, _HISTORY_STORE);

    s->beep = configdb_get_int(db, BEEP, _BEEP);
    s->scroll_on_enter = configdb_get_int(db, SCROLL_ON_ENTER, _SCROLL_ON_ENTER);
    s->scroll_on_text = configdb_get_int(db, SCROLL_ON_TEXT, _SCROLL_ON_TEXT);

    s->show_own_input = configdb_get_int(db, SHOW_OWN_INPUT, _SHOW_OWN_INPUT);
    s->show_own_input_fgcolor = configdb_get_int(db, SHOW_OWN_INPUT_FGCOLOR,
                                                 _SHOW_OWN_INPUT_FGCOLOR);
    s->show_own_input_bgcolor = configdb_get_int(db, SHOW_OWN_INPUT_BGCOLOR,
                                                 _SHOW_OWN_INPUT_BGCOLOR);
    s->show_own_input_bold = configdb_get_int(db, SHOW_OWN_INPUT_BOLD,
                                              _SHOW_OWN_INPUT_BOLD);
    s->show_own_input_underline = configdb_get_int(db, SHOW_OWN_INPUT_UNDERLINE,
                                                   _SHOW_OWN_INPUT_UNDERLINE);

    s->reconnect = configdb_get_int(db, RECONNECT, _RECONNECT);
    s->buffer_size = configdb_get_int(db, BUFFER_SIZE, _BUFFER_SIZE);

    s->timestamp = configdb_get_string(db, TIMESTAMP, _TIMESTAMP);

    s->bg_col = configdb_get_int(db, BG_COL, _BG_COL);
    s->bg_trans = configdb_get_int(db, BG_TRANS, _BG_TRANS);
    s->bg_file = configdb_get_int(db, BG_FILE, _BG_FILE);
    s->bg_filename = configdb_get_string(db, BG_FILENAME, _BG_FILENAME);
    s->bg_tinted = configdb_get_int(db, BG_TINTED, _BG_TINTED);
    s->bg_tint_r = configdb_get_int(db, BG_TINT_R, _BG_TINT_R);
    s->bg_tint_g = configdb_get_int(db, BG_TINT_G, _BG_TINT_G);
    s->bg_tint_b = configdb_get_int(db, BG_TINT_B, _BG_TINT_B);
    s->bg_tile = configdb_get_int(db, BG_TILE, _BG_TILE);

    set_hotkey_mask(s);

    configdb_free(db);
}

#define do_set_color(a,b)				\
db = configdb_set_rgb(db, b, (s->reds[a]),	\
                             (s->greens[a]),	\
                             (s->blues[a]))

void gm_settings_store(settings_t *s) {
    ConfigDB *db;
    if(debug) printf("Saving config file data\n");

    db = configdb_load(config_main_file, FALSE);

    do_set_color(BLACK,   COLOR_BLACK);
    do_set_color(RED,     COLOR_RED);
    do_set_color(GREEN,   COLOR_GREEN);
    do_set_color(YELLOW,  COLOR_YELLOW);
    do_set_color(BLUE,    COLOR_BLUE);
    do_set_color(MAGENTA, COLOR_MAGENTA);
    do_set_color(CYAN,    COLOR_CYAN);
    do_set_color(WHITE,   COLOR_WHITE);

    do_set_color(BRIGHT+BLACK,   COLOR_BBLACK);
    do_set_color(BRIGHT+RED,     COLOR_BRED);
    do_set_color(BRIGHT+GREEN,   COLOR_BGREEN);
    do_set_color(BRIGHT+YELLOW,  COLOR_BYELLOW);
    do_set_color(BRIGHT+BLUE,    COLOR_BBLUE);
    do_set_color(BRIGHT+MAGENTA, COLOR_BMAGENTA);
    do_set_color(BRIGHT+CYAN,    COLOR_BCYAN);
    do_set_color(BRIGHT+WHITE,   COLOR_BWHITE);

    do_set_color(FG_COLOR, COLOR_FORE);
    do_set_color(BG_COLOR, COLOR_BACK);

    db = configdb_set_int(db, COLOR_PALETTE, s->color_palette);
    db = configdb_set_int(db, FGBG_PALETTE,  s->fgbg_palette);

    /* fonts */
    db = configdb_set_string(db, FONT,   s->fontname);

    /* Window */
    db = configdb_set_int(db, SAVE_WINDOW_SIZE, s->save_window_size);
    db = configdb_set_int(db, WINDOW_HEIGHT, s->window_height);
    db = configdb_set_int(db, WINDOW_WIDTH,  s->window_width);

    /* notebook */
    db = configdb_set_int(db, TAB_POSITION, s->tab_position);
    db = configdb_set_int(db, TAB_LIMIT_LENGTH, s->tab_limit_length);
    db = configdb_set_int(db, TAB_MAX_LENGTH, s->tab_max_length);
    db = configdb_set_int(db, HOMO_TABS,    s->homo_tabs);
    db = configdb_set_int(db, TABS_SHOW_LINES_WAITING,
                          s->tabs_show_lines_waiting);
    db = configdb_set_int(db, HOTKEY, s->hotkey);
    db = configdb_set_int(db, HOTKEY_ARROW, s->hotkey_arrow);

    db = configdb_set_int(db, TOOLBAR, s->toolbar);

    /* World showing */
    db = configdb_set_int(db, SAVE_INPUT_HEIGHT, s->save_input_height);

    db = configdb_set_int(db, EDITOR_EXTERNAL, s->editor_external);
    db = configdb_set_string(db, EDITOR_COMMAND, s->editor_command);
    db = configdb_set_int(db, EDITOR_USE_ENV, s->editor_use_env);
    db = configdb_set_int(db, EDITOR_NEEDS_TERM, s->editor_needs_term);
    db = configdb_set_int(db, EDITOR_DELETE_FILE, s->editor_delete_file);

    db = configdb_set_int(db, USERLIST_POS, s->userlist_pos);
    db = configdb_set_int(db, USERLIST_RESIZEABLE, s->userlist_resizeable);
    db = configdb_set_int(db, USERLIST_TOTALS, s->userlist_totals);
    db = configdb_set_int(db, USERLIST_HEADERS, s->userlist_headers);
    db = configdb_set_int(db, USERLIST_OBJECTS, s->userlist_objects);
    db = configdb_set_int(db, USERLIST_FRIENDS_COLOR, s->userlist_friends_color);
    db = configdb_set_int(db, USERLIST_AUTO_OPEN, s->userlist_auto_open);

    db = configdb_set_int(db, TAB_POS_ON_TAB, s->tab_pos_on_tab);

    db = configdb_set_int(db, RESTORE_OPEN_WORLDS, s->restore_open_worlds);
    db = configdb_set_int(db, ENABLE_RUN_MENU, s->enable_run_menu);

    db = configdb_set_int(db, HISTORY, s->history);
    db = configdb_set_int(db, HISTORY_SIZE, s->history_size);
    db = configdb_set_int(db, HISTORY_SELECTIVE, s->history_selective);
    db = configdb_set_int(db, HISTORY_SENSITIVE, s->history_sensitive);
    db = configdb_set_int(db, HISTORY_STORE, s->history_store);

    db = configdb_set_int(db, BEEP, s->beep);
    db = configdb_set_int(db, SCROLL_ON_ENTER, s->scroll_on_enter);
    db = configdb_set_int(db, SCROLL_ON_TEXT, s->scroll_on_text);

    db = configdb_set_int(db, SHOW_OWN_INPUT, s->show_own_input);
    db = configdb_set_int(db, SHOW_OWN_INPUT_FGCOLOR, s->show_own_input_fgcolor);
    db = configdb_set_int(db, SHOW_OWN_INPUT_BGCOLOR, s->show_own_input_bgcolor);
    db = configdb_set_int(db, SHOW_OWN_INPUT_BOLD, s->show_own_input_bold);
    db = configdb_set_int(db, SHOW_OWN_INPUT_UNDERLINE,
                          s->show_own_input_underline);

    db = configdb_set_int(db, RECONNECT, s->reconnect);
    db = configdb_set_int(db, BUFFER_SIZE, s->buffer_size);

    db = configdb_set_string(db, TIMESTAMP, s->timestamp);

    db = configdb_set_int(db, BG_COL, s->bg_col);
    db = configdb_set_int(db, BG_TRANS, s->bg_trans);
    db = configdb_set_int(db, BG_FILE, s->bg_file);
    db = configdb_set_string(db, BG_FILENAME, s->bg_filename);
    db = configdb_set_int(db, BG_TINTED, s->bg_tinted);
    db = configdb_set_int(db, BG_TINT_R, s->bg_tint_r);
    db = configdb_set_int(db, BG_TINT_G, s->bg_tint_g);
    db = configdb_set_int(db, BG_TINT_B, s->bg_tint_b);
    db = configdb_set_int(db, BG_TILE, s->bg_tile);
    
    configdb_store(db, config_main_file, FALSE);

    configdb_free(db);
}

/* NOTE: these are not the default settings, but empty settings */
settings_t *gm_settings_alloc() {
    settings_t *s = g_malloc(sizeof(settings_t));

    /* Fonts */
    s->fontname  = NULL;

    s->timestamp = NULL;
    
    s->editor_command = NULL;

    s->bg_filename = NULL;

    return s;
}

settings_t *gm_settings_copy(settings_t *settings) {
    int i;
    settings_t *s = gm_settings_alloc();

    if(debug) printf("copying settings\n");

    for(i = 0; i < N_COLORS; i++) {
        s->reds[i] = settings->reds[i];
        s->greens[i] = settings->greens[i];
        s->blues[i] = settings->blues[i];
    }

    s->color_palette = settings->color_palette;
    s->fgbg_palette = settings->fgbg_palette;

    /* Fonts */
    s->fontname  = g_strdup(settings->fontname);

    s->save_window_size = settings->save_window_size;
    s->window_height = settings->window_height;
    s->window_width  = settings->window_width;

    /* notebook */
    s->tab_position     = settings->tab_position;
    s->tab_limit_length = settings->tab_limit_length;
    s->tab_max_length   = settings->tab_max_length;
    s->homo_tabs        = settings->homo_tabs;
    s->tabs_show_lines_waiting = settings->tabs_show_lines_waiting;
    s->hotkey           = settings->hotkey;
    s->hotkey_arrow     = settings->hotkey_arrow;

    s->toolbar = settings->toolbar;

    /* World showing */
    s->save_input_height = settings->save_input_height;

    s->editor_external    = settings->editor_external;
    s->editor_use_env     = settings->editor_use_env;
    s->editor_command     = g_strdup(settings->editor_command);
    s->editor_needs_term  = settings->editor_needs_term;
    s->editor_delete_file = settings->editor_delete_file;

    s->userlist_pos = settings->userlist_pos;
    s->userlist_resizeable = settings->userlist_resizeable;
    s->userlist_totals = settings->userlist_totals;
    s->userlist_headers = settings->userlist_headers;
    s->userlist_objects = settings->userlist_objects;
    s->userlist_friends_color = settings->userlist_friends_color;
    s->userlist_auto_open = settings->userlist_auto_open;

    s->restore_open_worlds = settings->restore_open_worlds;
    s->tab_pos_on_tab = settings->tab_pos_on_tab;

    s->enable_run_menu = settings->enable_run_menu;

    s->history = settings->history;
    s->history_size = settings->history_size;
    s->history_selective = settings->history_selective;
    s->history_sensitive = settings->history_sensitive;
    s->history_store = settings->history_store;
    
    s->beep = settings->beep;
    s->scroll_on_enter = settings->scroll_on_enter;
    s->scroll_on_text = settings->scroll_on_text;
    
    s->show_own_input = settings->show_own_input;
    s->show_own_input_fgcolor = settings->show_own_input_fgcolor;
    s->show_own_input_bgcolor = settings->show_own_input_bgcolor;
    s->show_own_input_bold = settings->show_own_input_bold;
    s->show_own_input_underline = settings->show_own_input_underline;

    s->reconnect = settings->reconnect;
    s->buffer_size = settings->buffer_size;

    s->timestamp = g_strdup(settings->timestamp);
    
    s->bg_col = settings->bg_col;
    s->bg_trans = settings->bg_trans;
    s->bg_file = settings->bg_file;
    s->bg_filename = g_strdup(settings->bg_filename);
    s->bg_tinted = settings->bg_tinted;
    s->bg_tint_r = settings->bg_tint_r;
    s->bg_tint_g = settings->bg_tint_g;
    s->bg_tint_b = settings->bg_tint_b;
    s->bg_tile = settings->bg_tile;
    
    return s;
}

void gm_settings_free(settings_t *s) {
    if(debug) printf("freeing settings\n");

    g_free(s->fontname);

    g_free(s->editor_command);

    g_free(s->timestamp);

    g_free(s->bg_filename);
    
    g_free(s);
}

#define changed(sett) old->sett != new->sett

int palettes_changed(settings_t *old, settings_t *new) {
    int i;
    if(changed(color_palette) || changed(fgbg_palette))
	return TRUE;
    for(i = 0; i < N_COLORS; i++) {
	if(old->reds[i]   != new->reds[i] ||
	   old->greens[i] != new->greens[i] ||
	   old->blues[i]  != new->blues[i])
	    return TRUE;
    }
    return FALSE;
}

void apply_changes(settings_t *old) {
    GList *worlds = gm_notebook_get_worlds();
    GList *node;
    world *w;
    settings_t *new = settings;
    int pchanged = palettes_changed(old, new);
    
    if(debug) printf("Applying changes to the configuration\n");

    if(changed(tab_position)) {
        gm_notebook_set_tab_position(new->tab_position);
    }
    if(changed(homo_tabs)) {
        gm_notebook_set_homogeneous_tabs(new->homo_tabs);
    }
    if(changed(tab_pos_on_tab) || changed(tab_limit_length) ||
       changed(tab_max_length)) {
        gm_notebook_changed();
    }
    if(changed(toolbar)) {
        gm_toolbar_update_visibility();
    }
    set_hotkey_mask(new);

    if(changed(tabs_show_lines_waiting)) {
        gm_notebook_set_tabs_show_lines_waiting(new->tabs_show_lines_waiting);
    }
    gm_notebook_update_input_styles();

    for(node = worlds; node; node = g_list_next(node)) {
        w = (world *) node->data;
        if(changed(userlist_pos)) {
            gm_world_update_userlist_pos(w);
        }
        if(changed(userlist_resizeable)) {
            gm_world_update_userlist_resizeable(w);
        }
        if(changed(userlist_totals)) {
            gm_world_update_userlist_totals(w);
        }
        if(changed(userlist_objects)) {
            gm_world_update_userlist_objects(w);
        }
        if(changed(userlist_headers)) {
            gm_world_update_userlist_headers(w);
        }
        if(changed(userlist_friends_color)) {
            gm_world_update_userlist_friends_color(w);
        }
        if(strcmp(new->fontname, old->fontname) != 0) {
            gtk_gmo_set_font(GTK_GMO(w->output), NULL, new->fontname);
        }
        if(pchanged) {
            gtk_gmo_set_palette(GTK_GMO(w->output), colors);
	}
        if(changed(beep)) {
            gtk_gmo_set_beep(GTK_GMO(w->output), new->beep);
        }
        if(changed(scroll_on_text)) {
            gtk_gmo_set_scroll_on_text(GTK_GMO(w->output), new->scroll_on_text);
        }
        if(changed(buffer_size)) {
            gtk_gmo_set_max_lines(GTK_GMO(w->output), new->buffer_size);
        }
        if(strcmp(new->timestamp, old->timestamp)) {
            gtk_gmo_set_timestamp(GTK_GMO(w->output), new->timestamp);
        }
	if(changed(bg_col) || changed(bg_trans) || changed(bg_file) ||
	   strcmp(new->bg_filename, old->bg_filename) ||
	   changed(bg_tinted) || changed(bg_tint_r) || changed(bg_tint_g) ||
	   changed(bg_tint_b) || changed(bg_tile)) {
	    gtk_gmo_set_background(GTK_GMO(w->output),
				   (new->bg_file &&
				    new->bg_filename &&
				    new->bg_filename[0]) ? new->bg_filename : NULL,
				   new->bg_tile,
				   new->bg_trans,
				   new->bg_tinted,
				   new->bg_tint_r,
				   new->bg_tint_g,
				   new->bg_tint_b);
	    gtk_widget_queue_draw(w->output);
	}
    }
    g_list_free(worlds);
}

settings_t *gm_settings_replace(settings_t *new_settings) {
    settings_t *old_settings = settings;
    settings = gm_settings_copy(new_settings);

    gm_settings_update();
    gm_settings_store(settings);

    apply_changes(old_settings);

    gm_settings_free(old_settings);
    return new_settings;
}

char *gm_settings_get_color_name(int color) {
    static char *colornames[8] = {
        "Black",
        "Red",
        "Green",
        "Yellow",
        "Blue",
        "Magenta",
        "Cyan",
        "White"};
    return _(colornames[CLAMP(color, 0, 7)]);
}
